/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.finspace;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.finspace.internal.FinspaceServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.finspace.model.AccessDeniedException;
import software.amazon.awssdk.services.finspace.model.ConflictException;
import software.amazon.awssdk.services.finspace.model.CreateEnvironmentRequest;
import software.amazon.awssdk.services.finspace.model.CreateEnvironmentResponse;
import software.amazon.awssdk.services.finspace.model.CreateKxChangesetRequest;
import software.amazon.awssdk.services.finspace.model.CreateKxChangesetResponse;
import software.amazon.awssdk.services.finspace.model.CreateKxClusterRequest;
import software.amazon.awssdk.services.finspace.model.CreateKxClusterResponse;
import software.amazon.awssdk.services.finspace.model.CreateKxDatabaseRequest;
import software.amazon.awssdk.services.finspace.model.CreateKxDatabaseResponse;
import software.amazon.awssdk.services.finspace.model.CreateKxEnvironmentRequest;
import software.amazon.awssdk.services.finspace.model.CreateKxEnvironmentResponse;
import software.amazon.awssdk.services.finspace.model.CreateKxUserRequest;
import software.amazon.awssdk.services.finspace.model.CreateKxUserResponse;
import software.amazon.awssdk.services.finspace.model.DeleteEnvironmentRequest;
import software.amazon.awssdk.services.finspace.model.DeleteEnvironmentResponse;
import software.amazon.awssdk.services.finspace.model.DeleteKxClusterRequest;
import software.amazon.awssdk.services.finspace.model.DeleteKxClusterResponse;
import software.amazon.awssdk.services.finspace.model.DeleteKxDatabaseRequest;
import software.amazon.awssdk.services.finspace.model.DeleteKxDatabaseResponse;
import software.amazon.awssdk.services.finspace.model.DeleteKxEnvironmentRequest;
import software.amazon.awssdk.services.finspace.model.DeleteKxEnvironmentResponse;
import software.amazon.awssdk.services.finspace.model.DeleteKxUserRequest;
import software.amazon.awssdk.services.finspace.model.DeleteKxUserResponse;
import software.amazon.awssdk.services.finspace.model.FinspaceException;
import software.amazon.awssdk.services.finspace.model.GetEnvironmentRequest;
import software.amazon.awssdk.services.finspace.model.GetEnvironmentResponse;
import software.amazon.awssdk.services.finspace.model.GetKxChangesetRequest;
import software.amazon.awssdk.services.finspace.model.GetKxChangesetResponse;
import software.amazon.awssdk.services.finspace.model.GetKxClusterRequest;
import software.amazon.awssdk.services.finspace.model.GetKxClusterResponse;
import software.amazon.awssdk.services.finspace.model.GetKxConnectionStringRequest;
import software.amazon.awssdk.services.finspace.model.GetKxConnectionStringResponse;
import software.amazon.awssdk.services.finspace.model.GetKxDatabaseRequest;
import software.amazon.awssdk.services.finspace.model.GetKxDatabaseResponse;
import software.amazon.awssdk.services.finspace.model.GetKxEnvironmentRequest;
import software.amazon.awssdk.services.finspace.model.GetKxEnvironmentResponse;
import software.amazon.awssdk.services.finspace.model.GetKxUserRequest;
import software.amazon.awssdk.services.finspace.model.GetKxUserResponse;
import software.amazon.awssdk.services.finspace.model.InternalServerException;
import software.amazon.awssdk.services.finspace.model.InvalidRequestException;
import software.amazon.awssdk.services.finspace.model.LimitExceededException;
import software.amazon.awssdk.services.finspace.model.ListEnvironmentsRequest;
import software.amazon.awssdk.services.finspace.model.ListEnvironmentsResponse;
import software.amazon.awssdk.services.finspace.model.ListKxChangesetsRequest;
import software.amazon.awssdk.services.finspace.model.ListKxChangesetsResponse;
import software.amazon.awssdk.services.finspace.model.ListKxClusterNodesRequest;
import software.amazon.awssdk.services.finspace.model.ListKxClusterNodesResponse;
import software.amazon.awssdk.services.finspace.model.ListKxClustersRequest;
import software.amazon.awssdk.services.finspace.model.ListKxClustersResponse;
import software.amazon.awssdk.services.finspace.model.ListKxDatabasesRequest;
import software.amazon.awssdk.services.finspace.model.ListKxDatabasesResponse;
import software.amazon.awssdk.services.finspace.model.ListKxEnvironmentsRequest;
import software.amazon.awssdk.services.finspace.model.ListKxEnvironmentsResponse;
import software.amazon.awssdk.services.finspace.model.ListKxUsersRequest;
import software.amazon.awssdk.services.finspace.model.ListKxUsersResponse;
import software.amazon.awssdk.services.finspace.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.finspace.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.finspace.model.ResourceAlreadyExistsException;
import software.amazon.awssdk.services.finspace.model.ResourceNotFoundException;
import software.amazon.awssdk.services.finspace.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.finspace.model.TagResourceRequest;
import software.amazon.awssdk.services.finspace.model.TagResourceResponse;
import software.amazon.awssdk.services.finspace.model.ThrottlingException;
import software.amazon.awssdk.services.finspace.model.UntagResourceRequest;
import software.amazon.awssdk.services.finspace.model.UntagResourceResponse;
import software.amazon.awssdk.services.finspace.model.UpdateEnvironmentRequest;
import software.amazon.awssdk.services.finspace.model.UpdateEnvironmentResponse;
import software.amazon.awssdk.services.finspace.model.UpdateKxClusterCodeConfigurationRequest;
import software.amazon.awssdk.services.finspace.model.UpdateKxClusterCodeConfigurationResponse;
import software.amazon.awssdk.services.finspace.model.UpdateKxClusterDatabasesRequest;
import software.amazon.awssdk.services.finspace.model.UpdateKxClusterDatabasesResponse;
import software.amazon.awssdk.services.finspace.model.UpdateKxDatabaseRequest;
import software.amazon.awssdk.services.finspace.model.UpdateKxDatabaseResponse;
import software.amazon.awssdk.services.finspace.model.UpdateKxEnvironmentNetworkRequest;
import software.amazon.awssdk.services.finspace.model.UpdateKxEnvironmentNetworkResponse;
import software.amazon.awssdk.services.finspace.model.UpdateKxEnvironmentRequest;
import software.amazon.awssdk.services.finspace.model.UpdateKxEnvironmentResponse;
import software.amazon.awssdk.services.finspace.model.UpdateKxUserRequest;
import software.amazon.awssdk.services.finspace.model.UpdateKxUserResponse;
import software.amazon.awssdk.services.finspace.model.ValidationException;
import software.amazon.awssdk.services.finspace.transform.CreateEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.CreateKxChangesetRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.CreateKxClusterRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.CreateKxDatabaseRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.CreateKxEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.CreateKxUserRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.DeleteEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.DeleteKxClusterRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.DeleteKxDatabaseRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.DeleteKxEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.DeleteKxUserRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.GetEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.GetKxChangesetRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.GetKxClusterRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.GetKxConnectionStringRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.GetKxDatabaseRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.GetKxEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.GetKxUserRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.ListEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.ListKxChangesetsRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.ListKxClusterNodesRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.ListKxClustersRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.ListKxDatabasesRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.ListKxEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.ListKxUsersRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.UpdateEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.UpdateKxClusterCodeConfigurationRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.UpdateKxClusterDatabasesRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.UpdateKxDatabaseRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.UpdateKxEnvironmentNetworkRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.UpdateKxEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.finspace.transform.UpdateKxUserRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link FinspaceAsyncClient}.
 *
 * @see FinspaceAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultFinspaceAsyncClient implements FinspaceAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultFinspaceAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final FinspaceServiceClientConfiguration serviceClientConfiguration;

    protected DefaultFinspaceAsyncClient(FinspaceServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Create a new FinSpace environment.
     * </p>
     *
     * @param createEnvironmentRequest
     * @return A Java Future containing the result of the CreateEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ServiceQuotaExceededException You have exceeded your service quota. To perform the requested action,
     *         remove some of the relevant resources, or use Service Quotas to request a service quota increase.</li>
     *         <li>LimitExceededException A service limit or quota is exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.CreateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/CreateEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateEnvironmentResponse> createEnvironment(CreateEnvironmentRequest createEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateEnvironmentRequest, CreateEnvironmentResponse>()
                            .withOperationName("CreateEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createEnvironmentRequest));
            CompletableFuture<CreateEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a changeset for a kdb database. A changeset allows you to add and delete existing files by using an
     * ordered list of change requests.
     * </p>
     *
     * @param createKxChangesetRequest
     * @return A Java Future containing the result of the CreateKxChangeset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ConflictException There was a conflict with this action, and it could not be completed.</li>
     *         <li>LimitExceededException A service limit or quota is exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.CreateKxChangeset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/CreateKxChangeset" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateKxChangesetResponse> createKxChangeset(CreateKxChangesetRequest createKxChangesetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createKxChangesetRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createKxChangesetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateKxChangeset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateKxChangesetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateKxChangesetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateKxChangesetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateKxChangesetRequest, CreateKxChangesetResponse>()
                            .withOperationName("CreateKxChangeset").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateKxChangesetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createKxChangesetRequest));
            CompletableFuture<CreateKxChangesetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new kdb cluster.
     * </p>
     *
     * @param createKxClusterRequest
     * @return A Java Future containing the result of the CreateKxCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>LimitExceededException A service limit or quota is exceeded.</li>
     *         <li>ConflictException There was a conflict with this action, and it could not be completed.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.CreateKxCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/CreateKxCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateKxClusterResponse> createKxCluster(CreateKxClusterRequest createKxClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createKxClusterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createKxClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateKxCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateKxClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateKxClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateKxClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateKxClusterRequest, CreateKxClusterResponse>()
                            .withOperationName("CreateKxCluster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateKxClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createKxClusterRequest));
            CompletableFuture<CreateKxClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new kdb database in the environment.
     * </p>
     *
     * @param createKxDatabaseRequest
     * @return A Java Future containing the result of the CreateKxDatabase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ConflictException There was a conflict with this action, and it could not be completed.</li>
     *         <li>ResourceAlreadyExistsException The specified resource group already exists.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>LimitExceededException A service limit or quota is exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.CreateKxDatabase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/CreateKxDatabase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateKxDatabaseResponse> createKxDatabase(CreateKxDatabaseRequest createKxDatabaseRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createKxDatabaseRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createKxDatabaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateKxDatabase");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateKxDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateKxDatabaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateKxDatabaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateKxDatabaseRequest, CreateKxDatabaseResponse>()
                            .withOperationName("CreateKxDatabase").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateKxDatabaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createKxDatabaseRequest));
            CompletableFuture<CreateKxDatabaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a managed kdb environment for the account.
     * </p>
     *
     * @param createKxEnvironmentRequest
     * @return A Java Future containing the result of the CreateKxEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ServiceQuotaExceededException You have exceeded your service quota. To perform the requested action,
     *         remove some of the relevant resources, or use Service Quotas to request a service quota increase.</li>
     *         <li>LimitExceededException A service limit or quota is exceeded.</li>
     *         <li>ConflictException There was a conflict with this action, and it could not be completed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.CreateKxEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/CreateKxEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateKxEnvironmentResponse> createKxEnvironment(
            CreateKxEnvironmentRequest createKxEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createKxEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createKxEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateKxEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateKxEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateKxEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateKxEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateKxEnvironmentRequest, CreateKxEnvironmentResponse>()
                            .withOperationName("CreateKxEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateKxEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createKxEnvironmentRequest));
            CompletableFuture<CreateKxEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a user in FinSpace kdb environment with an associated IAM role.
     * </p>
     *
     * @param createKxUserRequest
     * @return A Java Future containing the result of the CreateKxUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceAlreadyExistsException The specified resource group already exists.</li>
     *         <li>LimitExceededException A service limit or quota is exceeded.</li>
     *         <li>ConflictException There was a conflict with this action, and it could not be completed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.CreateKxUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/CreateKxUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateKxUserResponse> createKxUser(CreateKxUserRequest createKxUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createKxUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createKxUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateKxUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateKxUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateKxUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateKxUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateKxUserRequest, CreateKxUserResponse>()
                            .withOperationName("CreateKxUser").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateKxUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createKxUserRequest));
            CompletableFuture<CreateKxUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Delete an FinSpace environment.
     * </p>
     *
     * @param deleteEnvironmentRequest
     * @return A Java Future containing the result of the DeleteEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.DeleteEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/DeleteEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEnvironmentResponse> deleteEnvironment(DeleteEnvironmentRequest deleteEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEnvironmentRequest, DeleteEnvironmentResponse>()
                            .withOperationName("DeleteEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteEnvironmentRequest));
            CompletableFuture<DeleteEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a kdb cluster.
     * </p>
     *
     * @param deleteKxClusterRequest
     * @return A Java Future containing the result of the DeleteKxCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>LimitExceededException A service limit or quota is exceeded.</li>
     *         <li>ConflictException There was a conflict with this action, and it could not be completed.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.DeleteKxCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/DeleteKxCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteKxClusterResponse> deleteKxCluster(DeleteKxClusterRequest deleteKxClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteKxClusterRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteKxClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteKxCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteKxClusterResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteKxClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteKxClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteKxClusterRequest, DeleteKxClusterResponse>()
                            .withOperationName("DeleteKxCluster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteKxClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteKxClusterRequest));
            CompletableFuture<DeleteKxClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified database and all of its associated data. This action is irreversible. You must copy any
     * data out of the database before deleting it if the data is to be retained.
     * </p>
     *
     * @param deleteKxDatabaseRequest
     * @return A Java Future containing the result of the DeleteKxDatabase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ConflictException There was a conflict with this action, and it could not be completed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.DeleteKxDatabase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/DeleteKxDatabase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteKxDatabaseResponse> deleteKxDatabase(DeleteKxDatabaseRequest deleteKxDatabaseRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteKxDatabaseRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteKxDatabaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteKxDatabase");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteKxDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteKxDatabaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteKxDatabaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteKxDatabaseRequest, DeleteKxDatabaseResponse>()
                            .withOperationName("DeleteKxDatabase").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteKxDatabaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteKxDatabaseRequest));
            CompletableFuture<DeleteKxDatabaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the kdb environment. This action is irreversible. Deleting a kdb environment will remove all the
     * associated data and any services running in it.
     * </p>
     *
     * @param deleteKxEnvironmentRequest
     * @return A Java Future containing the result of the DeleteKxEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.DeleteKxEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/DeleteKxEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteKxEnvironmentResponse> deleteKxEnvironment(
            DeleteKxEnvironmentRequest deleteKxEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteKxEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteKxEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteKxEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteKxEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteKxEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteKxEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteKxEnvironmentRequest, DeleteKxEnvironmentResponse>()
                            .withOperationName("DeleteKxEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteKxEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteKxEnvironmentRequest));
            CompletableFuture<DeleteKxEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a user in the specified kdb environment.
     * </p>
     *
     * @param deleteKxUserRequest
     * @return A Java Future containing the result of the DeleteKxUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.DeleteKxUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/DeleteKxUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteKxUserResponse> deleteKxUser(DeleteKxUserRequest deleteKxUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteKxUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteKxUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteKxUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteKxUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteKxUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteKxUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteKxUserRequest, DeleteKxUserResponse>()
                            .withOperationName("DeleteKxUser").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteKxUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteKxUserRequest));
            CompletableFuture<DeleteKxUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the FinSpace environment object.
     * </p>
     *
     * @param getEnvironmentRequest
     * @return A Java Future containing the result of the GetEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.GetEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/GetEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetEnvironmentResponse> getEnvironment(GetEnvironmentRequest getEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEnvironmentRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEnvironmentRequest, GetEnvironmentResponse>()
                            .withOperationName("GetEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEnvironmentRequest));
            CompletableFuture<GetEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a kdb changeset.
     * </p>
     *
     * @param getKxChangesetRequest
     * @return A Java Future containing the result of the GetKxChangeset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.GetKxChangeset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/GetKxChangeset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetKxChangesetResponse> getKxChangeset(GetKxChangesetRequest getKxChangesetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getKxChangesetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKxChangesetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKxChangeset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetKxChangesetResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetKxChangesetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetKxChangesetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetKxChangesetRequest, GetKxChangesetResponse>()
                            .withOperationName("GetKxChangeset").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetKxChangesetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getKxChangesetRequest));
            CompletableFuture<GetKxChangesetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about a kdb cluster.
     * </p>
     *
     * @param getKxClusterRequest
     * @return A Java Future containing the result of the GetKxCluster operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>LimitExceededException A service limit or quota is exceeded.</li>
     *         <li>ConflictException There was a conflict with this action, and it could not be completed.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.GetKxCluster
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/GetKxCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetKxClusterResponse> getKxCluster(GetKxClusterRequest getKxClusterRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getKxClusterRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKxClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKxCluster");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetKxClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetKxClusterResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetKxClusterResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetKxClusterRequest, GetKxClusterResponse>()
                            .withOperationName("GetKxCluster").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetKxClusterRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getKxClusterRequest));
            CompletableFuture<GetKxClusterResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a connection string for a user to connect to a kdb cluster. You must call this API using the same role
     * that you have defined while creating a user.
     * </p>
     *
     * @param getKxConnectionStringRequest
     * @return A Java Future containing the result of the GetKxConnectionString operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.GetKxConnectionString
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/GetKxConnectionString"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetKxConnectionStringResponse> getKxConnectionString(
            GetKxConnectionStringRequest getKxConnectionStringRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getKxConnectionStringRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKxConnectionStringRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKxConnectionString");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetKxConnectionStringResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetKxConnectionStringResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetKxConnectionStringResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetKxConnectionStringRequest, GetKxConnectionStringResponse>()
                            .withOperationName("GetKxConnectionString").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetKxConnectionStringRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getKxConnectionStringRequest));
            CompletableFuture<GetKxConnectionStringResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns database information for the specified environment ID.
     * </p>
     *
     * @param getKxDatabaseRequest
     * @return A Java Future containing the result of the GetKxDatabase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.GetKxDatabase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/GetKxDatabase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetKxDatabaseResponse> getKxDatabase(GetKxDatabaseRequest getKxDatabaseRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getKxDatabaseRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKxDatabaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKxDatabase");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetKxDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetKxDatabaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetKxDatabaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetKxDatabaseRequest, GetKxDatabaseResponse>()
                            .withOperationName("GetKxDatabase").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetKxDatabaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getKxDatabaseRequest));
            CompletableFuture<GetKxDatabaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all the information for the specified kdb environment.
     * </p>
     *
     * @param getKxEnvironmentRequest
     * @return A Java Future containing the result of the GetKxEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.GetKxEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/GetKxEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetKxEnvironmentResponse> getKxEnvironment(GetKxEnvironmentRequest getKxEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getKxEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKxEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKxEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetKxEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetKxEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetKxEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetKxEnvironmentRequest, GetKxEnvironmentResponse>()
                            .withOperationName("GetKxEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetKxEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getKxEnvironmentRequest));
            CompletableFuture<GetKxEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves information about the specified kdb user.
     * </p>
     *
     * @param getKxUserRequest
     * @return A Java Future containing the result of the GetKxUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.GetKxUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/GetKxUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetKxUserResponse> getKxUser(GetKxUserRequest getKxUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getKxUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getKxUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetKxUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetKxUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetKxUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetKxUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetKxUserRequest, GetKxUserResponse>().withOperationName("GetKxUser")
                            .withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetKxUserRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getKxUserRequest));
            CompletableFuture<GetKxUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A list of all of your FinSpace environments.
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return A Java Future containing the result of the ListEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/ListEnvironments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListEnvironmentsResponse> listEnvironments(ListEnvironmentsRequest listEnvironmentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listEnvironmentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEnvironmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEnvironments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListEnvironmentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEnvironmentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEnvironmentsRequest, ListEnvironmentsResponse>()
                            .withOperationName("ListEnvironments").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListEnvironmentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listEnvironmentsRequest));
            CompletableFuture<ListEnvironmentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all the changesets for a database.
     * </p>
     *
     * @param listKxChangesetsRequest
     * @return A Java Future containing the result of the ListKxChangesets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.ListKxChangesets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/ListKxChangesets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListKxChangesetsResponse> listKxChangesets(ListKxChangesetsRequest listKxChangesetsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listKxChangesetsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKxChangesetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKxChangesets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListKxChangesetsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListKxChangesetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListKxChangesetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListKxChangesetsRequest, ListKxChangesetsResponse>()
                            .withOperationName("ListKxChangesets").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListKxChangesetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listKxChangesetsRequest));
            CompletableFuture<ListKxChangesetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the nodes in a kdb cluster.
     * </p>
     *
     * @param listKxClusterNodesRequest
     * @return A Java Future containing the result of the ListKxClusterNodes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>LimitExceededException A service limit or quota is exceeded.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.ListKxClusterNodes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/ListKxClusterNodes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListKxClusterNodesResponse> listKxClusterNodes(ListKxClusterNodesRequest listKxClusterNodesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listKxClusterNodesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKxClusterNodesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKxClusterNodes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListKxClusterNodesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListKxClusterNodesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListKxClusterNodesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListKxClusterNodesRequest, ListKxClusterNodesResponse>()
                            .withOperationName("ListKxClusterNodes").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListKxClusterNodesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listKxClusterNodesRequest));
            CompletableFuture<ListKxClusterNodesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of clusters.
     * </p>
     *
     * @param listKxClustersRequest
     * @return A Java Future containing the result of the ListKxClusters operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>LimitExceededException A service limit or quota is exceeded.</li>
     *         <li>ConflictException There was a conflict with this action, and it could not be completed.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.ListKxClusters
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/ListKxClusters" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListKxClustersResponse> listKxClusters(ListKxClustersRequest listKxClustersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listKxClustersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKxClustersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKxClusters");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListKxClustersResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListKxClustersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListKxClustersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListKxClustersRequest, ListKxClustersResponse>()
                            .withOperationName("ListKxClusters").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListKxClustersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listKxClustersRequest));
            CompletableFuture<ListKxClustersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all the databases in the kdb environment.
     * </p>
     *
     * @param listKxDatabasesRequest
     * @return A Java Future containing the result of the ListKxDatabases operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.ListKxDatabases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/ListKxDatabases" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListKxDatabasesResponse> listKxDatabases(ListKxDatabasesRequest listKxDatabasesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listKxDatabasesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKxDatabasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKxDatabases");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListKxDatabasesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListKxDatabasesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListKxDatabasesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListKxDatabasesRequest, ListKxDatabasesResponse>()
                            .withOperationName("ListKxDatabases").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListKxDatabasesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listKxDatabasesRequest));
            CompletableFuture<ListKxDatabasesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of kdb environments created in an account.
     * </p>
     *
     * @param listKxEnvironmentsRequest
     * @return A Java Future containing the result of the ListKxEnvironments operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.ListKxEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/ListKxEnvironments" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListKxEnvironmentsResponse> listKxEnvironments(ListKxEnvironmentsRequest listKxEnvironmentsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listKxEnvironmentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKxEnvironmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKxEnvironments");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListKxEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListKxEnvironmentsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListKxEnvironmentsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListKxEnvironmentsRequest, ListKxEnvironmentsResponse>()
                            .withOperationName("ListKxEnvironments").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListKxEnvironmentsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listKxEnvironmentsRequest));
            CompletableFuture<ListKxEnvironmentsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all the users in a kdb environment.
     * </p>
     *
     * @param listKxUsersRequest
     * @return A Java Future containing the result of the ListKxUsers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.ListKxUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/ListKxUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListKxUsersResponse> listKxUsers(ListKxUsersRequest listKxUsersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listKxUsersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listKxUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListKxUsers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListKxUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListKxUsersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListKxUsersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListKxUsersRequest, ListKxUsersResponse>()
                            .withOperationName("ListKxUsers").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListKxUsersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listKxUsersRequest));
            CompletableFuture<ListKxUsersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A list of all tags for a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>InvalidRequestException The request is invalid. Something is wrong with the input to the request.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds metadata tags to a FinSpace resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>InvalidRequestException The request is invalid. Something is wrong with the input to the request.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes metadata tags from a FinSpace resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>InvalidRequestException The request is invalid. Something is wrong with the input to the request.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Update your FinSpace environment.
     * </p>
     *
     * @param updateEnvironmentRequest
     * @return A Java Future containing the result of the UpdateEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.UpdateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/UpdateEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEnvironmentResponse> updateEnvironment(UpdateEnvironmentRequest updateEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEnvironmentRequest, UpdateEnvironmentResponse>()
                            .withOperationName("UpdateEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateEnvironmentRequest));
            CompletableFuture<UpdateEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows you to update code configuration on a running cluster. By using this API you can update the code, the
     * initialization script path, and the command line arguments for a specific cluster. The configuration that you
     * want to update will override any existing configurations on the cluster.
     * </p>
     *
     * @param updateKxClusterCodeConfigurationRequest
     * @return A Java Future containing the result of the UpdateKxClusterCodeConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>LimitExceededException A service limit or quota is exceeded.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ConflictException There was a conflict with this action, and it could not be completed.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.UpdateKxClusterCodeConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/UpdateKxClusterCodeConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateKxClusterCodeConfigurationResponse> updateKxClusterCodeConfiguration(
            UpdateKxClusterCodeConfigurationRequest updateKxClusterCodeConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateKxClusterCodeConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateKxClusterCodeConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateKxClusterCodeConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateKxClusterCodeConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateKxClusterCodeConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateKxClusterCodeConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateKxClusterCodeConfigurationRequest, UpdateKxClusterCodeConfigurationResponse>()
                            .withOperationName("UpdateKxClusterCodeConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateKxClusterCodeConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateKxClusterCodeConfigurationRequest));
            CompletableFuture<UpdateKxClusterCodeConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the databases mounted on a kdb cluster, which includes the <code>changesetId</code> and all the dbPaths
     * to be cached. This API does not allow you to change a database name or add a database if you created a cluster
     * without one.
     * </p>
     * <p>
     * Using this API you can point a cluster to a different changeset and modify a list of partitions being cached.
     * </p>
     *
     * @param updateKxClusterDatabasesRequest
     * @return A Java Future containing the result of the UpdateKxClusterDatabases operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>LimitExceededException A service limit or quota is exceeded.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ConflictException There was a conflict with this action, and it could not be completed.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.UpdateKxClusterDatabases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/UpdateKxClusterDatabases"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateKxClusterDatabasesResponse> updateKxClusterDatabases(
            UpdateKxClusterDatabasesRequest updateKxClusterDatabasesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateKxClusterDatabasesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateKxClusterDatabasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateKxClusterDatabases");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateKxClusterDatabasesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateKxClusterDatabasesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateKxClusterDatabasesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateKxClusterDatabasesRequest, UpdateKxClusterDatabasesResponse>()
                            .withOperationName("UpdateKxClusterDatabases").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateKxClusterDatabasesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateKxClusterDatabasesRequest));
            CompletableFuture<UpdateKxClusterDatabasesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates information for the given kdb database.
     * </p>
     *
     * @param updateKxDatabaseRequest
     * @return A Java Future containing the result of the UpdateKxDatabase operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ConflictException There was a conflict with this action, and it could not be completed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.UpdateKxDatabase
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/UpdateKxDatabase" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateKxDatabaseResponse> updateKxDatabase(UpdateKxDatabaseRequest updateKxDatabaseRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateKxDatabaseRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateKxDatabaseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateKxDatabase");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateKxDatabaseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateKxDatabaseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateKxDatabaseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateKxDatabaseRequest, UpdateKxDatabaseResponse>()
                            .withOperationName("UpdateKxDatabase").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateKxDatabaseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateKxDatabaseRequest));
            CompletableFuture<UpdateKxDatabaseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates information for the given kdb environment.
     * </p>
     *
     * @param updateKxEnvironmentRequest
     * @return A Java Future containing the result of the UpdateKxEnvironment operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ConflictException There was a conflict with this action, and it could not be completed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.UpdateKxEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/UpdateKxEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateKxEnvironmentResponse> updateKxEnvironment(
            UpdateKxEnvironmentRequest updateKxEnvironmentRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateKxEnvironmentRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateKxEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateKxEnvironment");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateKxEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateKxEnvironmentResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateKxEnvironmentResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateKxEnvironmentRequest, UpdateKxEnvironmentResponse>()
                            .withOperationName("UpdateKxEnvironment").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateKxEnvironmentRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateKxEnvironmentRequest));
            CompletableFuture<UpdateKxEnvironmentResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates environment network to connect to your internal network by using a transit gateway. This API supports
     * request to create a transit gateway attachment from FinSpace VPC to your transit gateway ID and create a custom
     * Route-53 outbound resolvers.
     * </p>
     * <p>
     * Once you send a request to update a network, you cannot change it again. Network update might require termination
     * of any clusters that are running in the existing network.
     * </p>
     *
     * @param updateKxEnvironmentNetworkRequest
     * @return A Java Future containing the result of the UpdateKxEnvironmentNetwork operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>ConflictException There was a conflict with this action, and it could not be completed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.UpdateKxEnvironmentNetwork
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/UpdateKxEnvironmentNetwork"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateKxEnvironmentNetworkResponse> updateKxEnvironmentNetwork(
            UpdateKxEnvironmentNetworkRequest updateKxEnvironmentNetworkRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateKxEnvironmentNetworkRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateKxEnvironmentNetworkRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateKxEnvironmentNetwork");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateKxEnvironmentNetworkResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateKxEnvironmentNetworkResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateKxEnvironmentNetworkResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateKxEnvironmentNetworkRequest, UpdateKxEnvironmentNetworkResponse>()
                            .withOperationName("UpdateKxEnvironmentNetwork").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateKxEnvironmentNetworkRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateKxEnvironmentNetworkRequest));
            CompletableFuture<UpdateKxEnvironmentNetworkResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the user details. You can only update the IAM role associated with a user.
     * </p>
     *
     * @param updateKxUserRequest
     * @return A Java Future containing the result of the UpdateKxUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException The request processing has failed because of an unknown error, exception or
     *         failure.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>ResourceNotFoundException One or more resources can't be found.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>LimitExceededException A service limit or quota is exceeded.</li>
     *         <li>ConflictException There was a conflict with this action, and it could not be completed.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FinspaceException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FinspaceAsyncClient.UpdateKxUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/UpdateKxUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateKxUserResponse> updateKxUser(UpdateKxUserRequest updateKxUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateKxUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateKxUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "finspace");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateKxUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateKxUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateKxUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateKxUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateKxUserRequest, UpdateKxUserResponse>()
                            .withOperationName("UpdateKxUser").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateKxUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateKxUserRequest));
            CompletableFuture<UpdateKxUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final FinspaceServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(FinspaceException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceAlreadyExistsException")
                                .exceptionBuilderSupplier(ResourceAlreadyExistsException::builder).httpStatusCode(409).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        FinspaceServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = FinspaceServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
