/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A description of a unique event within a stream.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Record implements SdkPojo, Serializable, ToCopyableBuilder<Record.Builder, Record> {
    private static final SdkField<String> EVENT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Record::eventID)).setter(setter(Builder::eventID))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventID").build()).build();

    private static final SdkField<String> EVENT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Record::eventNameAsString)).setter(setter(Builder::eventName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventName").build()).build();

    private static final SdkField<String> EVENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Record::eventVersion)).setter(setter(Builder::eventVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventVersion").build()).build();

    private static final SdkField<String> EVENT_SOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Record::eventSource)).setter(setter(Builder::eventSource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("eventSource").build()).build();

    private static final SdkField<String> AWS_REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Record::awsRegion)).setter(setter(Builder::awsRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("awsRegion").build()).build();

    private static final SdkField<StreamRecord> DYNAMODB_FIELD = SdkField.<StreamRecord> builder(MarshallingType.SDK_POJO)
            .getter(getter(Record::dynamodb)).setter(setter(Builder::dynamodb)).constructor(StreamRecord::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dynamodb").build()).build();

    private static final SdkField<Identity> USER_IDENTITY_FIELD = SdkField.<Identity> builder(MarshallingType.SDK_POJO)
            .getter(getter(Record::userIdentity)).setter(setter(Builder::userIdentity)).constructor(Identity::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("userIdentity").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVENT_ID_FIELD,
            EVENT_NAME_FIELD, EVENT_VERSION_FIELD, EVENT_SOURCE_FIELD, AWS_REGION_FIELD, DYNAMODB_FIELD, USER_IDENTITY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String eventID;

    private final String eventName;

    private final String eventVersion;

    private final String eventSource;

    private final String awsRegion;

    private final StreamRecord dynamodb;

    private final Identity userIdentity;

    private Record(BuilderImpl builder) {
        this.eventID = builder.eventID;
        this.eventName = builder.eventName;
        this.eventVersion = builder.eventVersion;
        this.eventSource = builder.eventSource;
        this.awsRegion = builder.awsRegion;
        this.dynamodb = builder.dynamodb;
        this.userIdentity = builder.userIdentity;
    }

    /**
     * <p>
     * A globally unique identifier for the event that was recorded in this stream record.
     * </p>
     * 
     * @return A globally unique identifier for the event that was recorded in this stream record.
     */
    public String eventID() {
        return eventID;
    }

    /**
     * <p>
     * The type of data modification that was performed on the DynamoDB table:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>INSERT</code> - a new item was added to the table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MODIFY</code> - one or more of an existing item's attributes were modified.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REMOVE</code> - the item was deleted from the table
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventName} will
     * return {@link OperationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #eventNameAsString}.
     * </p>
     * 
     * @return The type of data modification that was performed on the DynamoDB table:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>INSERT</code> - a new item was added to the table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MODIFY</code> - one or more of an existing item's attributes were modified.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REMOVE</code> - the item was deleted from the table
     *         </p>
     *         </li>
     * @see OperationType
     */
    public OperationType eventName() {
        return OperationType.fromValue(eventName);
    }

    /**
     * <p>
     * The type of data modification that was performed on the DynamoDB table:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>INSERT</code> - a new item was added to the table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MODIFY</code> - one or more of an existing item's attributes were modified.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REMOVE</code> - the item was deleted from the table
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventName} will
     * return {@link OperationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #eventNameAsString}.
     * </p>
     * 
     * @return The type of data modification that was performed on the DynamoDB table:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>INSERT</code> - a new item was added to the table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MODIFY</code> - one or more of an existing item's attributes were modified.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REMOVE</code> - the item was deleted from the table
     *         </p>
     *         </li>
     * @see OperationType
     */
    public String eventNameAsString() {
        return eventName;
    }

    /**
     * <p>
     * The version number of the stream record format. This number is updated whenever the structure of
     * <code>Record</code> is modified.
     * </p>
     * <p>
     * Client applications must not assume that <code>eventVersion</code> will remain at a particular value, as this
     * number is subject to change at any time. In general, <code>eventVersion</code> will only increase as the
     * low-level DynamoDB Streams API evolves.
     * </p>
     * 
     * @return The version number of the stream record format. This number is updated whenever the structure of
     *         <code>Record</code> is modified.</p>
     *         <p>
     *         Client applications must not assume that <code>eventVersion</code> will remain at a particular value, as
     *         this number is subject to change at any time. In general, <code>eventVersion</code> will only increase as
     *         the low-level DynamoDB Streams API evolves.
     */
    public String eventVersion() {
        return eventVersion;
    }

    /**
     * <p>
     * The AWS service from which the stream record originated. For DynamoDB Streams, this is <code>aws:dynamodb</code>.
     * </p>
     * 
     * @return The AWS service from which the stream record originated. For DynamoDB Streams, this is
     *         <code>aws:dynamodb</code>.
     */
    public String eventSource() {
        return eventSource;
    }

    /**
     * <p>
     * The region in which the <code>GetRecords</code> request was received.
     * </p>
     * 
     * @return The region in which the <code>GetRecords</code> request was received.
     */
    public String awsRegion() {
        return awsRegion;
    }

    /**
     * <p>
     * The main body of the stream record, containing all of the DynamoDB-specific fields.
     * </p>
     * 
     * @return The main body of the stream record, containing all of the DynamoDB-specific fields.
     */
    public StreamRecord dynamodb() {
        return dynamodb;
    }

    /**
     * <p>
     * Items that are deleted by the Time to Live process after expiration have the following fields:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Records[].userIdentity.type
     * </p>
     * <p>
     * "Service"
     * </p>
     * </li>
     * <li>
     * <p>
     * Records[].userIdentity.principalId
     * </p>
     * <p>
     * "dynamodb.amazonaws.com"
     * </p>
     * </li>
     * </ul>
     * 
     * @return Items that are deleted by the Time to Live process after expiration have the following fields: </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Records[].userIdentity.type
     *         </p>
     *         <p>
     *         "Service"
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Records[].userIdentity.principalId
     *         </p>
     *         <p>
     *         "dynamodb.amazonaws.com"
     *         </p>
     *         </li>
     */
    public Identity userIdentity() {
        return userIdentity;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(eventID());
        hashCode = 31 * hashCode + Objects.hashCode(eventNameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(eventVersion());
        hashCode = 31 * hashCode + Objects.hashCode(eventSource());
        hashCode = 31 * hashCode + Objects.hashCode(awsRegion());
        hashCode = 31 * hashCode + Objects.hashCode(dynamodb());
        hashCode = 31 * hashCode + Objects.hashCode(userIdentity());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Record)) {
            return false;
        }
        Record other = (Record) obj;
        return Objects.equals(eventID(), other.eventID()) && Objects.equals(eventNameAsString(), other.eventNameAsString())
                && Objects.equals(eventVersion(), other.eventVersion()) && Objects.equals(eventSource(), other.eventSource())
                && Objects.equals(awsRegion(), other.awsRegion()) && Objects.equals(dynamodb(), other.dynamodb())
                && Objects.equals(userIdentity(), other.userIdentity());
    }

    @Override
    public String toString() {
        return ToString.builder("Record").add("EventID", eventID()).add("EventName", eventNameAsString())
                .add("EventVersion", eventVersion()).add("EventSource", eventSource()).add("AwsRegion", awsRegion())
                .add("Dynamodb", dynamodb()).add("UserIdentity", userIdentity()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "eventID":
            return Optional.ofNullable(clazz.cast(eventID()));
        case "eventName":
            return Optional.ofNullable(clazz.cast(eventNameAsString()));
        case "eventVersion":
            return Optional.ofNullable(clazz.cast(eventVersion()));
        case "eventSource":
            return Optional.ofNullable(clazz.cast(eventSource()));
        case "awsRegion":
            return Optional.ofNullable(clazz.cast(awsRegion()));
        case "dynamodb":
            return Optional.ofNullable(clazz.cast(dynamodb()));
        case "userIdentity":
            return Optional.ofNullable(clazz.cast(userIdentity()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Record, T> g) {
        return obj -> g.apply((Record) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Record> {
        /**
         * <p>
         * A globally unique identifier for the event that was recorded in this stream record.
         * </p>
         * 
         * @param eventID
         *        A globally unique identifier for the event that was recorded in this stream record.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventID(String eventID);

        /**
         * <p>
         * The type of data modification that was performed on the DynamoDB table:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>INSERT</code> - a new item was added to the table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MODIFY</code> - one or more of an existing item's attributes were modified.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REMOVE</code> - the item was deleted from the table
         * </p>
         * </li>
         * </ul>
         * 
         * @param eventName
         *        The type of data modification that was performed on the DynamoDB table:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>INSERT</code> - a new item was added to the table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MODIFY</code> - one or more of an existing item's attributes were modified.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REMOVE</code> - the item was deleted from the table
         *        </p>
         *        </li>
         * @see OperationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperationType
         */
        Builder eventName(String eventName);

        /**
         * <p>
         * The type of data modification that was performed on the DynamoDB table:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>INSERT</code> - a new item was added to the table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MODIFY</code> - one or more of an existing item's attributes were modified.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REMOVE</code> - the item was deleted from the table
         * </p>
         * </li>
         * </ul>
         * 
         * @param eventName
         *        The type of data modification that was performed on the DynamoDB table:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>INSERT</code> - a new item was added to the table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MODIFY</code> - one or more of an existing item's attributes were modified.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REMOVE</code> - the item was deleted from the table
         *        </p>
         *        </li>
         * @see OperationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OperationType
         */
        Builder eventName(OperationType eventName);

        /**
         * <p>
         * The version number of the stream record format. This number is updated whenever the structure of
         * <code>Record</code> is modified.
         * </p>
         * <p>
         * Client applications must not assume that <code>eventVersion</code> will remain at a particular value, as this
         * number is subject to change at any time. In general, <code>eventVersion</code> will only increase as the
         * low-level DynamoDB Streams API evolves.
         * </p>
         * 
         * @param eventVersion
         *        The version number of the stream record format. This number is updated whenever the structure of
         *        <code>Record</code> is modified.</p>
         *        <p>
         *        Client applications must not assume that <code>eventVersion</code> will remain at a particular value,
         *        as this number is subject to change at any time. In general, <code>eventVersion</code> will only
         *        increase as the low-level DynamoDB Streams API evolves.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventVersion(String eventVersion);

        /**
         * <p>
         * The AWS service from which the stream record originated. For DynamoDB Streams, this is
         * <code>aws:dynamodb</code>.
         * </p>
         * 
         * @param eventSource
         *        The AWS service from which the stream record originated. For DynamoDB Streams, this is
         *        <code>aws:dynamodb</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eventSource(String eventSource);

        /**
         * <p>
         * The region in which the <code>GetRecords</code> request was received.
         * </p>
         * 
         * @param awsRegion
         *        The region in which the <code>GetRecords</code> request was received.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsRegion(String awsRegion);

        /**
         * <p>
         * The main body of the stream record, containing all of the DynamoDB-specific fields.
         * </p>
         * 
         * @param dynamodb
         *        The main body of the stream record, containing all of the DynamoDB-specific fields.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dynamodb(StreamRecord dynamodb);

        /**
         * <p>
         * The main body of the stream record, containing all of the DynamoDB-specific fields.
         * </p>
         * This is a convenience that creates an instance of the {@link StreamRecord.Builder} avoiding the need to
         * create one manually via {@link StreamRecord#builder()}.
         *
         * When the {@link Consumer} completes, {@link StreamRecord.Builder#build()} is called immediately and its
         * result is passed to {@link #dynamodb(StreamRecord)}.
         * 
         * @param dynamodb
         *        a consumer that will call methods on {@link StreamRecord.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dynamodb(StreamRecord)
         */
        default Builder dynamodb(Consumer<StreamRecord.Builder> dynamodb) {
            return dynamodb(StreamRecord.builder().applyMutation(dynamodb).build());
        }

        /**
         * <p>
         * Items that are deleted by the Time to Live process after expiration have the following fields:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Records[].userIdentity.type
         * </p>
         * <p>
         * "Service"
         * </p>
         * </li>
         * <li>
         * <p>
         * Records[].userIdentity.principalId
         * </p>
         * <p>
         * "dynamodb.amazonaws.com"
         * </p>
         * </li>
         * </ul>
         * 
         * @param userIdentity
         *        Items that are deleted by the Time to Live process after expiration have the following fields: </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Records[].userIdentity.type
         *        </p>
         *        <p>
         *        "Service"
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Records[].userIdentity.principalId
         *        </p>
         *        <p>
         *        "dynamodb.amazonaws.com"
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userIdentity(Identity userIdentity);

        /**
         * <p>
         * Items that are deleted by the Time to Live process after expiration have the following fields:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Records[].userIdentity.type
         * </p>
         * <p>
         * "Service"
         * </p>
         * </li>
         * <li>
         * <p>
         * Records[].userIdentity.principalId
         * </p>
         * <p>
         * "dynamodb.amazonaws.com"
         * </p>
         * </li>
         * </ul>
         * This is a convenience that creates an instance of the {@link Identity.Builder} avoiding the need to create
         * one manually via {@link Identity#builder()}.
         *
         * When the {@link Consumer} completes, {@link Identity.Builder#build()} is called immediately and its result is
         * passed to {@link #userIdentity(Identity)}.
         * 
         * @param userIdentity
         *        a consumer that will call methods on {@link Identity.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userIdentity(Identity)
         */
        default Builder userIdentity(Consumer<Identity.Builder> userIdentity) {
            return userIdentity(Identity.builder().applyMutation(userIdentity).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String eventID;

        private String eventName;

        private String eventVersion;

        private String eventSource;

        private String awsRegion;

        private StreamRecord dynamodb;

        private Identity userIdentity;

        private BuilderImpl() {
        }

        private BuilderImpl(Record model) {
            eventID(model.eventID);
            eventName(model.eventName);
            eventVersion(model.eventVersion);
            eventSource(model.eventSource);
            awsRegion(model.awsRegion);
            dynamodb(model.dynamodb);
            userIdentity(model.userIdentity);
        }

        public final String getEventID() {
            return eventID;
        }

        @Override
        public final Builder eventID(String eventID) {
            this.eventID = eventID;
            return this;
        }

        public final void setEventID(String eventID) {
            this.eventID = eventID;
        }

        public final String getEventName() {
            return eventName;
        }

        @Override
        public final Builder eventName(String eventName) {
            this.eventName = eventName;
            return this;
        }

        @Override
        public final Builder eventName(OperationType eventName) {
            this.eventName(eventName.toString());
            return this;
        }

        public final void setEventName(String eventName) {
            this.eventName = eventName;
        }

        public final String getEventVersion() {
            return eventVersion;
        }

        @Override
        public final Builder eventVersion(String eventVersion) {
            this.eventVersion = eventVersion;
            return this;
        }

        public final void setEventVersion(String eventVersion) {
            this.eventVersion = eventVersion;
        }

        public final String getEventSource() {
            return eventSource;
        }

        @Override
        public final Builder eventSource(String eventSource) {
            this.eventSource = eventSource;
            return this;
        }

        public final void setEventSource(String eventSource) {
            this.eventSource = eventSource;
        }

        public final String getAwsRegion() {
            return awsRegion;
        }

        @Override
        public final Builder awsRegion(String awsRegion) {
            this.awsRegion = awsRegion;
            return this;
        }

        public final void setAwsRegion(String awsRegion) {
            this.awsRegion = awsRegion;
        }

        public final StreamRecord.Builder getDynamodb() {
            return dynamodb != null ? dynamodb.toBuilder() : null;
        }

        @Override
        public final Builder dynamodb(StreamRecord dynamodb) {
            this.dynamodb = dynamodb;
            return this;
        }

        public final void setDynamodb(StreamRecord.BuilderImpl dynamodb) {
            this.dynamodb = dynamodb != null ? dynamodb.build() : null;
        }

        public final Identity.Builder getUserIdentity() {
            return userIdentity != null ? userIdentity.toBuilder() : null;
        }

        @Override
        public final Builder userIdentity(Identity userIdentity) {
            this.userIdentity = userIdentity;
            return this;
        }

        public final void setUserIdentity(Identity.BuilderImpl userIdentity) {
            this.userIdentity = userIdentity != null ? userIdentity.build() : null;
        }

        @Override
        public Record build() {
            return new Record(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
