/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.devicefarm.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a request to the create device pool operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateDevicePoolRequest extends DeviceFarmRequest implements
        ToCopyableBuilder<CreateDevicePoolRequest.Builder, CreateDevicePoolRequest> {
    private static final SdkField<String> PROJECT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateDevicePoolRequest::projectArn)).setter(setter(Builder::projectArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("projectArn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateDevicePoolRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateDevicePoolRequest::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<List<Rule>> RULES_FIELD = SdkField
            .<List<Rule>> builder(MarshallingType.LIST)
            .getter(getter(CreateDevicePoolRequest::rules))
            .setter(setter(Builder::rules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Rule> builder(MarshallingType.SDK_POJO)
                                            .constructor(Rule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> MAX_DEVICES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(CreateDevicePoolRequest::maxDevices)).setter(setter(Builder::maxDevices))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxDevices").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PROJECT_ARN_FIELD, NAME_FIELD,
            DESCRIPTION_FIELD, RULES_FIELD, MAX_DEVICES_FIELD));

    private final String projectArn;

    private final String name;

    private final String description;

    private final List<Rule> rules;

    private final Integer maxDevices;

    private CreateDevicePoolRequest(BuilderImpl builder) {
        super(builder);
        this.projectArn = builder.projectArn;
        this.name = builder.name;
        this.description = builder.description;
        this.rules = builder.rules;
        this.maxDevices = builder.maxDevices;
    }

    /**
     * <p>
     * The ARN of the project for the device pool.
     * </p>
     * 
     * @return The ARN of the project for the device pool.
     */
    public String projectArn() {
        return projectArn;
    }

    /**
     * <p>
     * The device pool's name.
     * </p>
     * 
     * @return The device pool's name.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The device pool's description.
     * </p>
     * 
     * @return The device pool's description.
     */
    public String description() {
        return description;
    }

    /**
     * Returns true if the Rules property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasRules() {
        return rules != null && !(rules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The device pool's rules.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRules()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The device pool's rules.
     */
    public List<Rule> rules() {
        return rules;
    }

    /**
     * <p>
     * The number of devices that Device Farm can add to your device pool. Device Farm adds devices that are available
     * and meet the criteria that you assign for the <code>rules</code> parameter. Depending on how many devices meet
     * these constraints, your device pool might contain fewer devices than the value for this parameter.
     * </p>
     * <p>
     * By specifying the maximum number of devices, you can control the costs that you incur by running tests.
     * </p>
     * 
     * @return The number of devices that Device Farm can add to your device pool. Device Farm adds devices that are
     *         available and meet the criteria that you assign for the <code>rules</code> parameter. Depending on how
     *         many devices meet these constraints, your device pool might contain fewer devices than the value for this
     *         parameter.</p>
     *         <p>
     *         By specifying the maximum number of devices, you can control the costs that you incur by running tests.
     */
    public Integer maxDevices() {
        return maxDevices;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(projectArn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(rules());
        hashCode = 31 * hashCode + Objects.hashCode(maxDevices());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateDevicePoolRequest)) {
            return false;
        }
        CreateDevicePoolRequest other = (CreateDevicePoolRequest) obj;
        return Objects.equals(projectArn(), other.projectArn()) && Objects.equals(name(), other.name())
                && Objects.equals(description(), other.description()) && Objects.equals(rules(), other.rules())
                && Objects.equals(maxDevices(), other.maxDevices());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateDevicePoolRequest").add("ProjectArn", projectArn()).add("Name", name())
                .add("Description", description()).add("Rules", rules()).add("MaxDevices", maxDevices()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "projectArn":
            return Optional.ofNullable(clazz.cast(projectArn()));
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "rules":
            return Optional.ofNullable(clazz.cast(rules()));
        case "maxDevices":
            return Optional.ofNullable(clazz.cast(maxDevices()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateDevicePoolRequest, T> g) {
        return obj -> g.apply((CreateDevicePoolRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DeviceFarmRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateDevicePoolRequest> {
        /**
         * <p>
         * The ARN of the project for the device pool.
         * </p>
         * 
         * @param projectArn
         *        The ARN of the project for the device pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projectArn(String projectArn);

        /**
         * <p>
         * The device pool's name.
         * </p>
         * 
         * @param name
         *        The device pool's name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The device pool's description.
         * </p>
         * 
         * @param description
         *        The device pool's description.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The device pool's rules.
         * </p>
         * 
         * @param rules
         *        The device pool's rules.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rules(Collection<Rule> rules);

        /**
         * <p>
         * The device pool's rules.
         * </p>
         * 
         * @param rules
         *        The device pool's rules.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rules(Rule... rules);

        /**
         * <p>
         * The device pool's rules.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Rule>.Builder} avoiding the need to create
         * one manually via {@link List<Rule>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Rule>.Builder#build()} is called immediately and its result
         * is passed to {@link #rules(List<Rule>)}.
         * 
         * @param rules
         *        a consumer that will call methods on {@link List<Rule>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rules(List<Rule>)
         */
        Builder rules(Consumer<Rule.Builder>... rules);

        /**
         * <p>
         * The number of devices that Device Farm can add to your device pool. Device Farm adds devices that are
         * available and meet the criteria that you assign for the <code>rules</code> parameter. Depending on how many
         * devices meet these constraints, your device pool might contain fewer devices than the value for this
         * parameter.
         * </p>
         * <p>
         * By specifying the maximum number of devices, you can control the costs that you incur by running tests.
         * </p>
         * 
         * @param maxDevices
         *        The number of devices that Device Farm can add to your device pool. Device Farm adds devices that are
         *        available and meet the criteria that you assign for the <code>rules</code> parameter. Depending on how
         *        many devices meet these constraints, your device pool might contain fewer devices than the value for
         *        this parameter.</p>
         *        <p>
         *        By specifying the maximum number of devices, you can control the costs that you incur by running
         *        tests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxDevices(Integer maxDevices);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DeviceFarmRequest.BuilderImpl implements Builder {
        private String projectArn;

        private String name;

        private String description;

        private List<Rule> rules = DefaultSdkAutoConstructList.getInstance();

        private Integer maxDevices;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateDevicePoolRequest model) {
            super(model);
            projectArn(model.projectArn);
            name(model.name);
            description(model.description);
            rules(model.rules);
            maxDevices(model.maxDevices);
        }

        public final String getProjectArn() {
            return projectArn;
        }

        @Override
        public final Builder projectArn(String projectArn) {
            this.projectArn = projectArn;
            return this;
        }

        public final void setProjectArn(String projectArn) {
            this.projectArn = projectArn;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<Rule.Builder> getRules() {
            return rules != null ? rules.stream().map(Rule::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder rules(Collection<Rule> rules) {
            this.rules = RulesCopier.copy(rules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder rules(Rule... rules) {
            rules(Arrays.asList(rules));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder rules(Consumer<Rule.Builder>... rules) {
            rules(Stream.of(rules).map(c -> Rule.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setRules(Collection<Rule.BuilderImpl> rules) {
            this.rules = RulesCopier.copyFromBuilder(rules);
        }

        public final Integer getMaxDevices() {
            return maxDevices;
        }

        @Override
        public final Builder maxDevices(Integer maxDevices) {
            this.maxDevices = maxDevices;
            return this;
        }

        public final void setMaxDevices(Integer maxDevices) {
            this.maxDevices = maxDevices;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateDevicePoolRequest build() {
            return new CreateDevicePoolRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
