/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The details about an Amazon Web Services storage service that DataSync Discovery recommends for a resource in your
 * on-premises storage system.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/datasync/latest/userguide/discovery-understand-recommendations.html"
 * >Recommendations provided by DataSync Discovery</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Recommendation implements SdkPojo, Serializable, ToCopyableBuilder<Recommendation.Builder, Recommendation> {
    private static final SdkField<String> STORAGE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StorageType").getter(getter(Recommendation::storageType)).setter(setter(Builder::storageType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageType").build()).build();

    private static final SdkField<Map<String, String>> STORAGE_CONFIGURATION_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("StorageConfiguration")
            .getter(getter(Recommendation::storageConfiguration))
            .setter(setter(Builder::storageConfiguration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageConfiguration").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> ESTIMATED_MONTHLY_STORAGE_COST_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EstimatedMonthlyStorageCost")
            .getter(getter(Recommendation::estimatedMonthlyStorageCost))
            .setter(setter(Builder::estimatedMonthlyStorageCost))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EstimatedMonthlyStorageCost")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STORAGE_TYPE_FIELD,
            STORAGE_CONFIGURATION_FIELD, ESTIMATED_MONTHLY_STORAGE_COST_FIELD));

    private static final long serialVersionUID = 1L;

    private final String storageType;

    private final Map<String, String> storageConfiguration;

    private final String estimatedMonthlyStorageCost;

    private Recommendation(BuilderImpl builder) {
        this.storageType = builder.storageType;
        this.storageConfiguration = builder.storageConfiguration;
        this.estimatedMonthlyStorageCost = builder.estimatedMonthlyStorageCost;
    }

    /**
     * <p>
     * A recommended Amazon Web Services storage service that you can migrate data to based on information that DataSync
     * Discovery collects about your on-premises storage system.
     * </p>
     * 
     * @return A recommended Amazon Web Services storage service that you can migrate data to based on information that
     *         DataSync Discovery collects about your on-premises storage system.
     */
    public final String storageType() {
        return storageType;
    }

    /**
     * For responses, this returns true if the service returned a value for the StorageConfiguration property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStorageConfiguration() {
        return storageConfiguration != null && !(storageConfiguration instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Information about how you can set up a recommended Amazon Web Services storage service.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStorageConfiguration} method.
     * </p>
     * 
     * @return Information about how you can set up a recommended Amazon Web Services storage service.
     */
    public final Map<String, String> storageConfiguration() {
        return storageConfiguration;
    }

    /**
     * <p>
     * The estimated monthly cost of the recommended Amazon Web Services storage service.
     * </p>
     * 
     * @return The estimated monthly cost of the recommended Amazon Web Services storage service.
     */
    public final String estimatedMonthlyStorageCost() {
        return estimatedMonthlyStorageCost;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(storageType());
        hashCode = 31 * hashCode + Objects.hashCode(hasStorageConfiguration() ? storageConfiguration() : null);
        hashCode = 31 * hashCode + Objects.hashCode(estimatedMonthlyStorageCost());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Recommendation)) {
            return false;
        }
        Recommendation other = (Recommendation) obj;
        return Objects.equals(storageType(), other.storageType()) && hasStorageConfiguration() == other.hasStorageConfiguration()
                && Objects.equals(storageConfiguration(), other.storageConfiguration())
                && Objects.equals(estimatedMonthlyStorageCost(), other.estimatedMonthlyStorageCost());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Recommendation").add("StorageType", storageType())
                .add("StorageConfiguration", hasStorageConfiguration() ? storageConfiguration() : null)
                .add("EstimatedMonthlyStorageCost", estimatedMonthlyStorageCost()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StorageType":
            return Optional.ofNullable(clazz.cast(storageType()));
        case "StorageConfiguration":
            return Optional.ofNullable(clazz.cast(storageConfiguration()));
        case "EstimatedMonthlyStorageCost":
            return Optional.ofNullable(clazz.cast(estimatedMonthlyStorageCost()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Recommendation, T> g) {
        return obj -> g.apply((Recommendation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Recommendation> {
        /**
         * <p>
         * A recommended Amazon Web Services storage service that you can migrate data to based on information that
         * DataSync Discovery collects about your on-premises storage system.
         * </p>
         * 
         * @param storageType
         *        A recommended Amazon Web Services storage service that you can migrate data to based on information
         *        that DataSync Discovery collects about your on-premises storage system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageType(String storageType);

        /**
         * <p>
         * Information about how you can set up a recommended Amazon Web Services storage service.
         * </p>
         * 
         * @param storageConfiguration
         *        Information about how you can set up a recommended Amazon Web Services storage service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageConfiguration(Map<String, String> storageConfiguration);

        /**
         * <p>
         * The estimated monthly cost of the recommended Amazon Web Services storage service.
         * </p>
         * 
         * @param estimatedMonthlyStorageCost
         *        The estimated monthly cost of the recommended Amazon Web Services storage service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder estimatedMonthlyStorageCost(String estimatedMonthlyStorageCost);
    }

    static final class BuilderImpl implements Builder {
        private String storageType;

        private Map<String, String> storageConfiguration = DefaultSdkAutoConstructMap.getInstance();

        private String estimatedMonthlyStorageCost;

        private BuilderImpl() {
        }

        private BuilderImpl(Recommendation model) {
            storageType(model.storageType);
            storageConfiguration(model.storageConfiguration);
            estimatedMonthlyStorageCost(model.estimatedMonthlyStorageCost);
        }

        public final String getStorageType() {
            return storageType;
        }

        public final void setStorageType(String storageType) {
            this.storageType = storageType;
        }

        @Override
        public final Builder storageType(String storageType) {
            this.storageType = storageType;
            return this;
        }

        public final Map<String, String> getStorageConfiguration() {
            if (storageConfiguration instanceof SdkAutoConstructMap) {
                return null;
            }
            return storageConfiguration;
        }

        public final void setStorageConfiguration(Map<String, String> storageConfiguration) {
            this.storageConfiguration = RecommendationsConfigMapCopier.copy(storageConfiguration);
        }

        @Override
        public final Builder storageConfiguration(Map<String, String> storageConfiguration) {
            this.storageConfiguration = RecommendationsConfigMapCopier.copy(storageConfiguration);
            return this;
        }

        public final String getEstimatedMonthlyStorageCost() {
            return estimatedMonthlyStorageCost;
        }

        public final void setEstimatedMonthlyStorageCost(String estimatedMonthlyStorageCost) {
            this.estimatedMonthlyStorageCost = estimatedMonthlyStorageCost;
        }

        @Override
        public final Builder estimatedMonthlyStorageCost(String estimatedMonthlyStorageCost) {
            this.estimatedMonthlyStorageCost = estimatedMonthlyStorageCost;
            return this;
        }

        @Override
        public Recommendation build() {
            return new Recommendation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
