/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * CreateLocationObjectStorageRequest
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateLocationObjectStorageRequest extends DataSyncRequest implements
        ToCopyableBuilder<CreateLocationObjectStorageRequest.Builder, CreateLocationObjectStorageRequest> {
    private static final SdkField<String> SERVER_HOSTNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerHostname").getter(getter(CreateLocationObjectStorageRequest::serverHostname))
            .setter(setter(Builder::serverHostname))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerHostname").build()).build();

    private static final SdkField<Integer> SERVER_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ServerPort").getter(getter(CreateLocationObjectStorageRequest::serverPort))
            .setter(setter(Builder::serverPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerPort").build()).build();

    private static final SdkField<String> SERVER_PROTOCOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerProtocol").getter(getter(CreateLocationObjectStorageRequest::serverProtocolAsString))
            .setter(setter(Builder::serverProtocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerProtocol").build()).build();

    private static final SdkField<String> SUBDIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Subdirectory").getter(getter(CreateLocationObjectStorageRequest::subdirectory))
            .setter(setter(Builder::subdirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subdirectory").build()).build();

    private static final SdkField<String> BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BucketName").getter(getter(CreateLocationObjectStorageRequest::bucketName))
            .setter(setter(Builder::bucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BucketName").build()).build();

    private static final SdkField<String> ACCESS_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccessKey").getter(getter(CreateLocationObjectStorageRequest::accessKey))
            .setter(setter(Builder::accessKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessKey").build()).build();

    private static final SdkField<String> SECRET_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretKey").getter(getter(CreateLocationObjectStorageRequest::secretKey))
            .setter(setter(Builder::secretKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretKey").build()).build();

    private static final SdkField<List<String>> AGENT_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AgentArns")
            .getter(getter(CreateLocationObjectStorageRequest::agentArns))
            .setter(setter(Builder::agentArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AgentArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<TagListEntry>> TAGS_FIELD = SdkField
            .<List<TagListEntry>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CreateLocationObjectStorageRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TagListEntry> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagListEntry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<SdkBytes> SERVER_CERTIFICATE_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("ServerCertificate").getter(getter(CreateLocationObjectStorageRequest::serverCertificate))
            .setter(setter(Builder::serverCertificate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerCertificate").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SERVER_HOSTNAME_FIELD,
            SERVER_PORT_FIELD, SERVER_PROTOCOL_FIELD, SUBDIRECTORY_FIELD, BUCKET_NAME_FIELD, ACCESS_KEY_FIELD, SECRET_KEY_FIELD,
            AGENT_ARNS_FIELD, TAGS_FIELD, SERVER_CERTIFICATE_FIELD));

    private final String serverHostname;

    private final Integer serverPort;

    private final String serverProtocol;

    private final String subdirectory;

    private final String bucketName;

    private final String accessKey;

    private final String secretKey;

    private final List<String> agentArns;

    private final List<TagListEntry> tags;

    private final SdkBytes serverCertificate;

    private CreateLocationObjectStorageRequest(BuilderImpl builder) {
        super(builder);
        this.serverHostname = builder.serverHostname;
        this.serverPort = builder.serverPort;
        this.serverProtocol = builder.serverProtocol;
        this.subdirectory = builder.subdirectory;
        this.bucketName = builder.bucketName;
        this.accessKey = builder.accessKey;
        this.secretKey = builder.secretKey;
        this.agentArns = builder.agentArns;
        this.tags = builder.tags;
        this.serverCertificate = builder.serverCertificate;
    }

    /**
     * <p>
     * Specifies the domain name or IP address of the object storage server. A DataSync agent uses this hostname to
     * mount the object storage server in a network.
     * </p>
     * 
     * @return Specifies the domain name or IP address of the object storage server. A DataSync agent uses this hostname
     *         to mount the object storage server in a network.
     */
    public final String serverHostname() {
        return serverHostname;
    }

    /**
     * <p>
     * Specifies the port that your object storage server accepts inbound network traffic on (for example, port 443).
     * </p>
     * 
     * @return Specifies the port that your object storage server accepts inbound network traffic on (for example, port
     *         443).
     */
    public final Integer serverPort() {
        return serverPort;
    }

    /**
     * <p>
     * Specifies the protocol that your object storage server uses to communicate.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #serverProtocol}
     * will return {@link ObjectStorageServerProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #serverProtocolAsString}.
     * </p>
     * 
     * @return Specifies the protocol that your object storage server uses to communicate.
     * @see ObjectStorageServerProtocol
     */
    public final ObjectStorageServerProtocol serverProtocol() {
        return ObjectStorageServerProtocol.fromValue(serverProtocol);
    }

    /**
     * <p>
     * Specifies the protocol that your object storage server uses to communicate.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #serverProtocol}
     * will return {@link ObjectStorageServerProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #serverProtocolAsString}.
     * </p>
     * 
     * @return Specifies the protocol that your object storage server uses to communicate.
     * @see ObjectStorageServerProtocol
     */
    public final String serverProtocolAsString() {
        return serverProtocol;
    }

    /**
     * <p>
     * Specifies the object prefix for your object storage server. If this is a source location, DataSync only copies
     * objects with this prefix. If this is a destination location, DataSync writes all objects with this prefix.
     * </p>
     * 
     * @return Specifies the object prefix for your object storage server. If this is a source location, DataSync only
     *         copies objects with this prefix. If this is a destination location, DataSync writes all objects with this
     *         prefix.
     */
    public final String subdirectory() {
        return subdirectory;
    }

    /**
     * <p>
     * Specifies the name of the object storage bucket involved in the transfer.
     * </p>
     * 
     * @return Specifies the name of the object storage bucket involved in the transfer.
     */
    public final String bucketName() {
        return bucketName;
    }

    /**
     * <p>
     * Specifies the access key (for example, a user name) if credentials are required to authenticate with the object
     * storage server.
     * </p>
     * 
     * @return Specifies the access key (for example, a user name) if credentials are required to authenticate with the
     *         object storage server.
     */
    public final String accessKey() {
        return accessKey;
    }

    /**
     * <p>
     * Specifies the secret key (for example, a password) if credentials are required to authenticate with the object
     * storage server.
     * </p>
     * 
     * @return Specifies the secret key (for example, a password) if credentials are required to authenticate with the
     *         object storage server.
     */
    public final String secretKey() {
        return secretKey;
    }

    /**
     * For responses, this returns true if the service returned a value for the AgentArns property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAgentArns() {
        return agentArns != null && !(agentArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the Amazon Resource Names (ARNs) of the DataSync agents that can securely connect with your location.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAgentArns} method.
     * </p>
     * 
     * @return Specifies the Amazon Resource Names (ARNs) of the DataSync agents that can securely connect with your
     *         location.
     */
    public final List<String> agentArns() {
        return agentArns;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the key-value pair that represents a tag that you want to add to the resource. Tags can help you
     * manage, filter, and search for your resources. We recommend creating a name tag for your location.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Specifies the key-value pair that represents a tag that you want to add to the resource. Tags can help
     *         you manage, filter, and search for your resources. We recommend creating a name tag for your location.
     */
    public final List<TagListEntry> tags() {
        return tags;
    }

    /**
     * <p>
     * Specifies a file with the certificates that are used to sign the object storage server's certificate (for
     * example, <code>file:///home/user/.ssh/storage_sys_certificate.pem</code>). The file you specify must include the
     * following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The certificate of the signing certificate authority (CA)
     * </p>
     * </li>
     * <li>
     * <p>
     * Any intermediate certificates
     * </p>
     * </li>
     * <li>
     * <p>
     * base64 encoding
     * </p>
     * </li>
     * <li>
     * <p>
     * A <code>.pem</code> extension
     * </p>
     * </li>
     * </ul>
     * <p>
     * The file can be up to 32768 bytes (before base64 encoding).
     * </p>
     * <p>
     * To use this parameter, configure <code>ServerProtocol</code> to <code>HTTPS</code>.
     * </p>
     * 
     * @return Specifies a file with the certificates that are used to sign the object storage server's certificate (for
     *         example, <code>file:///home/user/.ssh/storage_sys_certificate.pem</code>). The file you specify must
     *         include the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The certificate of the signing certificate authority (CA)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Any intermediate certificates
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         base64 encoding
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A <code>.pem</code> extension
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The file can be up to 32768 bytes (before base64 encoding).
     *         </p>
     *         <p>
     *         To use this parameter, configure <code>ServerProtocol</code> to <code>HTTPS</code>.
     */
    public final SdkBytes serverCertificate() {
        return serverCertificate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(serverHostname());
        hashCode = 31 * hashCode + Objects.hashCode(serverPort());
        hashCode = 31 * hashCode + Objects.hashCode(serverProtocolAsString());
        hashCode = 31 * hashCode + Objects.hashCode(subdirectory());
        hashCode = 31 * hashCode + Objects.hashCode(bucketName());
        hashCode = 31 * hashCode + Objects.hashCode(accessKey());
        hashCode = 31 * hashCode + Objects.hashCode(secretKey());
        hashCode = 31 * hashCode + Objects.hashCode(hasAgentArns() ? agentArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(serverCertificate());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateLocationObjectStorageRequest)) {
            return false;
        }
        CreateLocationObjectStorageRequest other = (CreateLocationObjectStorageRequest) obj;
        return Objects.equals(serverHostname(), other.serverHostname()) && Objects.equals(serverPort(), other.serverPort())
                && Objects.equals(serverProtocolAsString(), other.serverProtocolAsString())
                && Objects.equals(subdirectory(), other.subdirectory()) && Objects.equals(bucketName(), other.bucketName())
                && Objects.equals(accessKey(), other.accessKey()) && Objects.equals(secretKey(), other.secretKey())
                && hasAgentArns() == other.hasAgentArns() && Objects.equals(agentArns(), other.agentArns())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(serverCertificate(), other.serverCertificate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateLocationObjectStorageRequest").add("ServerHostname", serverHostname())
                .add("ServerPort", serverPort()).add("ServerProtocol", serverProtocolAsString())
                .add("Subdirectory", subdirectory()).add("BucketName", bucketName()).add("AccessKey", accessKey())
                .add("SecretKey", secretKey() == null ? null : "*** Sensitive Data Redacted ***")
                .add("AgentArns", hasAgentArns() ? agentArns() : null).add("Tags", hasTags() ? tags() : null)
                .add("ServerCertificate", serverCertificate()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ServerHostname":
            return Optional.ofNullable(clazz.cast(serverHostname()));
        case "ServerPort":
            return Optional.ofNullable(clazz.cast(serverPort()));
        case "ServerProtocol":
            return Optional.ofNullable(clazz.cast(serverProtocolAsString()));
        case "Subdirectory":
            return Optional.ofNullable(clazz.cast(subdirectory()));
        case "BucketName":
            return Optional.ofNullable(clazz.cast(bucketName()));
        case "AccessKey":
            return Optional.ofNullable(clazz.cast(accessKey()));
        case "SecretKey":
            return Optional.ofNullable(clazz.cast(secretKey()));
        case "AgentArns":
            return Optional.ofNullable(clazz.cast(agentArns()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "ServerCertificate":
            return Optional.ofNullable(clazz.cast(serverCertificate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateLocationObjectStorageRequest, T> g) {
        return obj -> g.apply((CreateLocationObjectStorageRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DataSyncRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, CreateLocationObjectStorageRequest> {
        /**
         * <p>
         * Specifies the domain name or IP address of the object storage server. A DataSync agent uses this hostname to
         * mount the object storage server in a network.
         * </p>
         * 
         * @param serverHostname
         *        Specifies the domain name or IP address of the object storage server. A DataSync agent uses this
         *        hostname to mount the object storage server in a network.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverHostname(String serverHostname);

        /**
         * <p>
         * Specifies the port that your object storage server accepts inbound network traffic on (for example, port
         * 443).
         * </p>
         * 
         * @param serverPort
         *        Specifies the port that your object storage server accepts inbound network traffic on (for example,
         *        port 443).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverPort(Integer serverPort);

        /**
         * <p>
         * Specifies the protocol that your object storage server uses to communicate.
         * </p>
         * 
         * @param serverProtocol
         *        Specifies the protocol that your object storage server uses to communicate.
         * @see ObjectStorageServerProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ObjectStorageServerProtocol
         */
        Builder serverProtocol(String serverProtocol);

        /**
         * <p>
         * Specifies the protocol that your object storage server uses to communicate.
         * </p>
         * 
         * @param serverProtocol
         *        Specifies the protocol that your object storage server uses to communicate.
         * @see ObjectStorageServerProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ObjectStorageServerProtocol
         */
        Builder serverProtocol(ObjectStorageServerProtocol serverProtocol);

        /**
         * <p>
         * Specifies the object prefix for your object storage server. If this is a source location, DataSync only
         * copies objects with this prefix. If this is a destination location, DataSync writes all objects with this
         * prefix.
         * </p>
         * 
         * @param subdirectory
         *        Specifies the object prefix for your object storage server. If this is a source location, DataSync
         *        only copies objects with this prefix. If this is a destination location, DataSync writes all objects
         *        with this prefix.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subdirectory(String subdirectory);

        /**
         * <p>
         * Specifies the name of the object storage bucket involved in the transfer.
         * </p>
         * 
         * @param bucketName
         *        Specifies the name of the object storage bucket involved in the transfer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketName(String bucketName);

        /**
         * <p>
         * Specifies the access key (for example, a user name) if credentials are required to authenticate with the
         * object storage server.
         * </p>
         * 
         * @param accessKey
         *        Specifies the access key (for example, a user name) if credentials are required to authenticate with
         *        the object storage server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessKey(String accessKey);

        /**
         * <p>
         * Specifies the secret key (for example, a password) if credentials are required to authenticate with the
         * object storage server.
         * </p>
         * 
         * @param secretKey
         *        Specifies the secret key (for example, a password) if credentials are required to authenticate with
         *        the object storage server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretKey(String secretKey);

        /**
         * <p>
         * Specifies the Amazon Resource Names (ARNs) of the DataSync agents that can securely connect with your
         * location.
         * </p>
         * 
         * @param agentArns
         *        Specifies the Amazon Resource Names (ARNs) of the DataSync agents that can securely connect with your
         *        location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(Collection<String> agentArns);

        /**
         * <p>
         * Specifies the Amazon Resource Names (ARNs) of the DataSync agents that can securely connect with your
         * location.
         * </p>
         * 
         * @param agentArns
         *        Specifies the Amazon Resource Names (ARNs) of the DataSync agents that can securely connect with your
         *        location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(String... agentArns);

        /**
         * <p>
         * Specifies the key-value pair that represents a tag that you want to add to the resource. Tags can help you
         * manage, filter, and search for your resources. We recommend creating a name tag for your location.
         * </p>
         * 
         * @param tags
         *        Specifies the key-value pair that represents a tag that you want to add to the resource. Tags can help
         *        you manage, filter, and search for your resources. We recommend creating a name tag for your location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<TagListEntry> tags);

        /**
         * <p>
         * Specifies the key-value pair that represents a tag that you want to add to the resource. Tags can help you
         * manage, filter, and search for your resources. We recommend creating a name tag for your location.
         * </p>
         * 
         * @param tags
         *        Specifies the key-value pair that represents a tag that you want to add to the resource. Tags can help
         *        you manage, filter, and search for your resources. We recommend creating a name tag for your location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(TagListEntry... tags);

        /**
         * <p>
         * Specifies the key-value pair that represents a tag that you want to add to the resource. Tags can help you
         * manage, filter, and search for your resources. We recommend creating a name tag for your location.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.datasync.model.TagListEntry.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.datasync.model.TagListEntry#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.datasync.model.TagListEntry.Builder#build()} is called immediately and
         * its result is passed to {@link #tags(List<TagListEntry>)}.
         * 
         * @param tags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.datasync.model.TagListEntry.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<TagListEntry>)
         */
        Builder tags(Consumer<TagListEntry.Builder>... tags);

        /**
         * <p>
         * Specifies a file with the certificates that are used to sign the object storage server's certificate (for
         * example, <code>file:///home/user/.ssh/storage_sys_certificate.pem</code>). The file you specify must include
         * the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The certificate of the signing certificate authority (CA)
         * </p>
         * </li>
         * <li>
         * <p>
         * Any intermediate certificates
         * </p>
         * </li>
         * <li>
         * <p>
         * base64 encoding
         * </p>
         * </li>
         * <li>
         * <p>
         * A <code>.pem</code> extension
         * </p>
         * </li>
         * </ul>
         * <p>
         * The file can be up to 32768 bytes (before base64 encoding).
         * </p>
         * <p>
         * To use this parameter, configure <code>ServerProtocol</code> to <code>HTTPS</code>.
         * </p>
         * 
         * @param serverCertificate
         *        Specifies a file with the certificates that are used to sign the object storage server's certificate
         *        (for example, <code>file:///home/user/.ssh/storage_sys_certificate.pem</code>). The file you specify
         *        must include the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The certificate of the signing certificate authority (CA)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Any intermediate certificates
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        base64 encoding
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A <code>.pem</code> extension
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The file can be up to 32768 bytes (before base64 encoding).
         *        </p>
         *        <p>
         *        To use this parameter, configure <code>ServerProtocol</code> to <code>HTTPS</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverCertificate(SdkBytes serverCertificate);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DataSyncRequest.BuilderImpl implements Builder {
        private String serverHostname;

        private Integer serverPort;

        private String serverProtocol;

        private String subdirectory;

        private String bucketName;

        private String accessKey;

        private String secretKey;

        private List<String> agentArns = DefaultSdkAutoConstructList.getInstance();

        private List<TagListEntry> tags = DefaultSdkAutoConstructList.getInstance();

        private SdkBytes serverCertificate;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateLocationObjectStorageRequest model) {
            super(model);
            serverHostname(model.serverHostname);
            serverPort(model.serverPort);
            serverProtocol(model.serverProtocol);
            subdirectory(model.subdirectory);
            bucketName(model.bucketName);
            accessKey(model.accessKey);
            secretKey(model.secretKey);
            agentArns(model.agentArns);
            tags(model.tags);
            serverCertificate(model.serverCertificate);
        }

        public final String getServerHostname() {
            return serverHostname;
        }

        public final void setServerHostname(String serverHostname) {
            this.serverHostname = serverHostname;
        }

        @Override
        public final Builder serverHostname(String serverHostname) {
            this.serverHostname = serverHostname;
            return this;
        }

        public final Integer getServerPort() {
            return serverPort;
        }

        public final void setServerPort(Integer serverPort) {
            this.serverPort = serverPort;
        }

        @Override
        public final Builder serverPort(Integer serverPort) {
            this.serverPort = serverPort;
            return this;
        }

        public final String getServerProtocol() {
            return serverProtocol;
        }

        public final void setServerProtocol(String serverProtocol) {
            this.serverProtocol = serverProtocol;
        }

        @Override
        public final Builder serverProtocol(String serverProtocol) {
            this.serverProtocol = serverProtocol;
            return this;
        }

        @Override
        public final Builder serverProtocol(ObjectStorageServerProtocol serverProtocol) {
            this.serverProtocol(serverProtocol == null ? null : serverProtocol.toString());
            return this;
        }

        public final String getSubdirectory() {
            return subdirectory;
        }

        public final void setSubdirectory(String subdirectory) {
            this.subdirectory = subdirectory;
        }

        @Override
        public final Builder subdirectory(String subdirectory) {
            this.subdirectory = subdirectory;
            return this;
        }

        public final String getBucketName() {
            return bucketName;
        }

        public final void setBucketName(String bucketName) {
            this.bucketName = bucketName;
        }

        @Override
        public final Builder bucketName(String bucketName) {
            this.bucketName = bucketName;
            return this;
        }

        public final String getAccessKey() {
            return accessKey;
        }

        public final void setAccessKey(String accessKey) {
            this.accessKey = accessKey;
        }

        @Override
        public final Builder accessKey(String accessKey) {
            this.accessKey = accessKey;
            return this;
        }

        public final String getSecretKey() {
            return secretKey;
        }

        public final void setSecretKey(String secretKey) {
            this.secretKey = secretKey;
        }

        @Override
        public final Builder secretKey(String secretKey) {
            this.secretKey = secretKey;
            return this;
        }

        public final Collection<String> getAgentArns() {
            if (agentArns instanceof SdkAutoConstructList) {
                return null;
            }
            return agentArns;
        }

        public final void setAgentArns(Collection<String> agentArns) {
            this.agentArns = AgentArnListCopier.copy(agentArns);
        }

        @Override
        public final Builder agentArns(Collection<String> agentArns) {
            this.agentArns = AgentArnListCopier.copy(agentArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder agentArns(String... agentArns) {
            agentArns(Arrays.asList(agentArns));
            return this;
        }

        public final List<TagListEntry.Builder> getTags() {
            List<TagListEntry.Builder> result = InputTagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<TagListEntry.BuilderImpl> tags) {
            this.tags = InputTagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<TagListEntry> tags) {
            this.tags = InputTagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(TagListEntry... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<TagListEntry.Builder>... tags) {
            tags(Stream.of(tags).map(c -> TagListEntry.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final ByteBuffer getServerCertificate() {
            return serverCertificate == null ? null : serverCertificate.asByteBuffer();
        }

        public final void setServerCertificate(ByteBuffer serverCertificate) {
            serverCertificate(serverCertificate == null ? null : SdkBytes.fromByteBuffer(serverCertificate));
        }

        @Override
        public final Builder serverCertificate(SdkBytes serverCertificate) {
            this.serverCertificate = serverCertificate;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateLocationObjectStorageRequest build() {
            return new CreateLocationObjectStorageRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
