/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.computeoptimizer.internal.ComputeOptimizerServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.computeoptimizer.model.AccessDeniedException;
import software.amazon.awssdk.services.computeoptimizer.model.ComputeOptimizerException;
import software.amazon.awssdk.services.computeoptimizer.model.DeleteRecommendationPreferencesRequest;
import software.amazon.awssdk.services.computeoptimizer.model.DeleteRecommendationPreferencesResponse;
import software.amazon.awssdk.services.computeoptimizer.model.DescribeRecommendationExportJobsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.DescribeRecommendationExportJobsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.ExportAutoScalingGroupRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.ExportAutoScalingGroupRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.ExportEbsVolumeRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.ExportEbsVolumeRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.ExportEc2InstanceRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.ExportEc2InstanceRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.ExportEcsServiceRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.ExportEcsServiceRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.ExportLambdaFunctionRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.ExportLambdaFunctionRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.ExportLicenseRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.ExportLicenseRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetAutoScalingGroupRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetAutoScalingGroupRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEbsVolumeRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEbsVolumeRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEc2InstanceRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEc2InstanceRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEc2RecommendationProjectedMetricsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEc2RecommendationProjectedMetricsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEcsServiceRecommendationProjectedMetricsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEcsServiceRecommendationProjectedMetricsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEcsServiceRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEcsServiceRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEffectiveRecommendationPreferencesRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEffectiveRecommendationPreferencesResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEnrollmentStatusRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEnrollmentStatusResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEnrollmentStatusesForOrganizationRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEnrollmentStatusesForOrganizationResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetLambdaFunctionRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetLambdaFunctionRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetLicenseRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetLicenseRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetRecommendationPreferencesRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetRecommendationPreferencesResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetRecommendationSummariesRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetRecommendationSummariesResponse;
import software.amazon.awssdk.services.computeoptimizer.model.InternalServerException;
import software.amazon.awssdk.services.computeoptimizer.model.InvalidParameterValueException;
import software.amazon.awssdk.services.computeoptimizer.model.LimitExceededException;
import software.amazon.awssdk.services.computeoptimizer.model.MissingAuthenticationTokenException;
import software.amazon.awssdk.services.computeoptimizer.model.OptInRequiredException;
import software.amazon.awssdk.services.computeoptimizer.model.PutRecommendationPreferencesRequest;
import software.amazon.awssdk.services.computeoptimizer.model.PutRecommendationPreferencesResponse;
import software.amazon.awssdk.services.computeoptimizer.model.ResourceNotFoundException;
import software.amazon.awssdk.services.computeoptimizer.model.ServiceUnavailableException;
import software.amazon.awssdk.services.computeoptimizer.model.ThrottlingException;
import software.amazon.awssdk.services.computeoptimizer.model.UpdateEnrollmentStatusRequest;
import software.amazon.awssdk.services.computeoptimizer.model.UpdateEnrollmentStatusResponse;
import software.amazon.awssdk.services.computeoptimizer.transform.DeleteRecommendationPreferencesRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.DescribeRecommendationExportJobsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.ExportAutoScalingGroupRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.ExportEbsVolumeRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.ExportEc2InstanceRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.ExportEcsServiceRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.ExportLambdaFunctionRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.ExportLicenseRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetAutoScalingGroupRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEbsVolumeRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEc2InstanceRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEc2RecommendationProjectedMetricsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEcsServiceRecommendationProjectedMetricsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEcsServiceRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEffectiveRecommendationPreferencesRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEnrollmentStatusRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEnrollmentStatusesForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetLambdaFunctionRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetLicenseRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetRecommendationPreferencesRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetRecommendationSummariesRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.PutRecommendationPreferencesRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.UpdateEnrollmentStatusRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ComputeOptimizerAsyncClient}.
 *
 * @see ComputeOptimizerAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultComputeOptimizerAsyncClient implements ComputeOptimizerAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultComputeOptimizerAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ComputeOptimizerServiceClientConfiguration serviceClientConfiguration;

    protected DefaultComputeOptimizerAsyncClient(ComputeOptimizerServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Deletes a recommendation preference, such as enhanced infrastructure metrics.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html">Activating
     * enhanced infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     *
     * @param deleteRecommendationPreferencesRequest
     * @return A Java Future containing the result of the DeleteRecommendationPreferences operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action doesn't exist.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.DeleteRecommendationPreferences
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/DeleteRecommendationPreferences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRecommendationPreferencesResponse> deleteRecommendationPreferences(
            DeleteRecommendationPreferencesRequest deleteRecommendationPreferencesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRecommendationPreferencesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteRecommendationPreferencesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRecommendationPreferences");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRecommendationPreferencesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRecommendationPreferencesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRecommendationPreferencesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRecommendationPreferencesRequest, DeleteRecommendationPreferencesResponse>()
                            .withOperationName("DeleteRecommendationPreferences")
                            .withMarshaller(new DeleteRecommendationPreferencesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteRecommendationPreferencesRequest));
            CompletableFuture<DeleteRecommendationPreferencesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes recommendation export jobs created in the last seven days.
     * </p>
     * <p>
     * Use the <a>ExportAutoScalingGroupRecommendations</a> or <a>ExportEC2InstanceRecommendations</a> actions to
     * request an export of your recommendations. Then use the <a>DescribeRecommendationExportJobs</a> action to view
     * your export jobs.
     * </p>
     *
     * @param describeRecommendationExportJobsRequest
     * @return A Java Future containing the result of the DescribeRecommendationExportJobs operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action doesn't exist.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.DescribeRecommendationExportJobs
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/DescribeRecommendationExportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRecommendationExportJobsResponse> describeRecommendationExportJobs(
            DescribeRecommendationExportJobsRequest describeRecommendationExportJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeRecommendationExportJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeRecommendationExportJobsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRecommendationExportJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRecommendationExportJobsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeRecommendationExportJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRecommendationExportJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRecommendationExportJobsRequest, DescribeRecommendationExportJobsResponse>()
                            .withOperationName("DescribeRecommendationExportJobs")
                            .withMarshaller(new DescribeRecommendationExportJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeRecommendationExportJobsRequest));
            CompletableFuture<DescribeRecommendationExportJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports optimization recommendations for Auto Scaling groups.
     * </p>
     * <p>
     * Recommendations are exported in a comma-separated values (.csv) file, and its metadata in a JavaScript Object
     * Notation (JSON) (.json) file, to an existing Amazon Simple Storage Service (Amazon S3) bucket that you specify.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/exporting-recommendations.html">Exporting
     * Recommendations</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     * <p>
     * You can have only one Auto Scaling group export job in progress per Amazon Web Services Region.
     * </p>
     *
     * @param exportAutoScalingGroupRecommendationsRequest
     * @return A Java Future containing the result of the ExportAutoScalingGroupRecommendations operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The request exceeds a limit of the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.ExportAutoScalingGroupRecommendations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/ExportAutoScalingGroupRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportAutoScalingGroupRecommendationsResponse> exportAutoScalingGroupRecommendations(
            ExportAutoScalingGroupRecommendationsRequest exportAutoScalingGroupRecommendationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportAutoScalingGroupRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                exportAutoScalingGroupRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportAutoScalingGroupRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportAutoScalingGroupRecommendationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ExportAutoScalingGroupRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportAutoScalingGroupRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportAutoScalingGroupRecommendationsRequest, ExportAutoScalingGroupRecommendationsResponse>()
                            .withOperationName("ExportAutoScalingGroupRecommendations")
                            .withMarshaller(new ExportAutoScalingGroupRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(exportAutoScalingGroupRecommendationsRequest));
            CompletableFuture<ExportAutoScalingGroupRecommendationsResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports optimization recommendations for Amazon EBS volumes.
     * </p>
     * <p>
     * Recommendations are exported in a comma-separated values (.csv) file, and its metadata in a JavaScript Object
     * Notation (JSON) (.json) file, to an existing Amazon Simple Storage Service (Amazon S3) bucket that you specify.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/exporting-recommendations.html">Exporting
     * Recommendations</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     * <p>
     * You can have only one Amazon EBS volume export job in progress per Amazon Web Services Region.
     * </p>
     *
     * @param exportEbsVolumeRecommendationsRequest
     * @return A Java Future containing the result of the ExportEBSVolumeRecommendations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The request exceeds a limit of the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.ExportEBSVolumeRecommendations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/ExportEBSVolumeRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportEbsVolumeRecommendationsResponse> exportEBSVolumeRecommendations(
            ExportEbsVolumeRecommendationsRequest exportEbsVolumeRecommendationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportEbsVolumeRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                exportEbsVolumeRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportEBSVolumeRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportEbsVolumeRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExportEbsVolumeRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportEbsVolumeRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportEbsVolumeRecommendationsRequest, ExportEbsVolumeRecommendationsResponse>()
                            .withOperationName("ExportEBSVolumeRecommendations")
                            .withMarshaller(new ExportEbsVolumeRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(exportEbsVolumeRecommendationsRequest));
            CompletableFuture<ExportEbsVolumeRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports optimization recommendations for Amazon EC2 instances.
     * </p>
     * <p>
     * Recommendations are exported in a comma-separated values (.csv) file, and its metadata in a JavaScript Object
     * Notation (JSON) (.json) file, to an existing Amazon Simple Storage Service (Amazon S3) bucket that you specify.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/exporting-recommendations.html">Exporting
     * Recommendations</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     * <p>
     * You can have only one Amazon EC2 instance export job in progress per Amazon Web Services Region.
     * </p>
     *
     * @param exportEc2InstanceRecommendationsRequest
     * @return A Java Future containing the result of the ExportEC2InstanceRecommendations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The request exceeds a limit of the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.ExportEC2InstanceRecommendations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/ExportEC2InstanceRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportEc2InstanceRecommendationsResponse> exportEC2InstanceRecommendations(
            ExportEc2InstanceRecommendationsRequest exportEc2InstanceRecommendationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportEc2InstanceRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                exportEc2InstanceRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportEC2InstanceRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportEc2InstanceRecommendationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ExportEc2InstanceRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportEc2InstanceRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportEc2InstanceRecommendationsRequest, ExportEc2InstanceRecommendationsResponse>()
                            .withOperationName("ExportEC2InstanceRecommendations")
                            .withMarshaller(new ExportEc2InstanceRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(exportEc2InstanceRecommendationsRequest));
            CompletableFuture<ExportEc2InstanceRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports optimization recommendations for Amazon ECS services on Fargate.
     * </p>
     * <p>
     * Recommendations are exported in a CSV file, and its metadata in a JSON file, to an existing Amazon Simple Storage
     * Service (Amazon S3) bucket that you specify. For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/exporting-recommendations.html">Exporting
     * Recommendations</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     * <p>
     * You can only have one Amazon ECS service export job in progress per Amazon Web Services Region.
     * </p>
     *
     * @param exportEcsServiceRecommendationsRequest
     * @return A Java Future containing the result of the ExportECSServiceRecommendations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The request exceeds a limit of the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.ExportECSServiceRecommendations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/ExportECSServiceRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportEcsServiceRecommendationsResponse> exportECSServiceRecommendations(
            ExportEcsServiceRecommendationsRequest exportEcsServiceRecommendationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportEcsServiceRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                exportEcsServiceRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportECSServiceRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportEcsServiceRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExportEcsServiceRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportEcsServiceRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportEcsServiceRecommendationsRequest, ExportEcsServiceRecommendationsResponse>()
                            .withOperationName("ExportECSServiceRecommendations")
                            .withMarshaller(new ExportEcsServiceRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(exportEcsServiceRecommendationsRequest));
            CompletableFuture<ExportEcsServiceRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports optimization recommendations for Lambda functions.
     * </p>
     * <p>
     * Recommendations are exported in a comma-separated values (.csv) file, and its metadata in a JavaScript Object
     * Notation (JSON) (.json) file, to an existing Amazon Simple Storage Service (Amazon S3) bucket that you specify.
     * For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/exporting-recommendations.html">Exporting
     * Recommendations</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     * <p>
     * You can have only one Lambda function export job in progress per Amazon Web Services Region.
     * </p>
     *
     * @param exportLambdaFunctionRecommendationsRequest
     * @return A Java Future containing the result of the ExportLambdaFunctionRecommendations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The request exceeds a limit of the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.ExportLambdaFunctionRecommendations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/ExportLambdaFunctionRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportLambdaFunctionRecommendationsResponse> exportLambdaFunctionRecommendations(
            ExportLambdaFunctionRecommendationsRequest exportLambdaFunctionRecommendationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportLambdaFunctionRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                exportLambdaFunctionRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportLambdaFunctionRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportLambdaFunctionRecommendationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ExportLambdaFunctionRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportLambdaFunctionRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportLambdaFunctionRecommendationsRequest, ExportLambdaFunctionRecommendationsResponse>()
                            .withOperationName("ExportLambdaFunctionRecommendations")
                            .withMarshaller(new ExportLambdaFunctionRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(exportLambdaFunctionRecommendationsRequest));
            CompletableFuture<ExportLambdaFunctionRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Export optimization recommendations for your licenses.
     * </p>
     * <p>
     * Recommendations are exported in a comma-separated values (CSV) file, and its metadata in a JavaScript Object
     * Notation (JSON) file, to an existing Amazon Simple Storage Service (Amazon S3) bucket that you specify. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/exporting-recommendations.html">Exporting
     * Recommendations</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     * <p>
     * You can have only one license export job in progress per Amazon Web Services Region.
     * </p>
     *
     * @param exportLicenseRecommendationsRequest
     * @return A Java Future containing the result of the ExportLicenseRecommendations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The request exceeds a limit of the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.ExportLicenseRecommendations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/ExportLicenseRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportLicenseRecommendationsResponse> exportLicenseRecommendations(
            ExportLicenseRecommendationsRequest exportLicenseRecommendationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportLicenseRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportLicenseRecommendationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportLicenseRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportLicenseRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExportLicenseRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportLicenseRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportLicenseRecommendationsRequest, ExportLicenseRecommendationsResponse>()
                            .withOperationName("ExportLicenseRecommendations")
                            .withMarshaller(new ExportLicenseRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(exportLicenseRecommendationsRequest));
            CompletableFuture<ExportLicenseRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns Auto Scaling group recommendations.
     * </p>
     * <p>
     * Compute Optimizer generates recommendations for Amazon EC2 Auto Scaling groups that meet a specific set of
     * requirements. For more information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     * requirements</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     *
     * @param getAutoScalingGroupRecommendationsRequest
     * @return A Java Future containing the result of the GetAutoScalingGroupRecommendations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action doesn't exist.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetAutoScalingGroupRecommendations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetAutoScalingGroupRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAutoScalingGroupRecommendationsResponse> getAutoScalingGroupRecommendations(
            GetAutoScalingGroupRecommendationsRequest getAutoScalingGroupRecommendationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAutoScalingGroupRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getAutoScalingGroupRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAutoScalingGroupRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAutoScalingGroupRecommendationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetAutoScalingGroupRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAutoScalingGroupRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAutoScalingGroupRecommendationsRequest, GetAutoScalingGroupRecommendationsResponse>()
                            .withOperationName("GetAutoScalingGroupRecommendations")
                            .withMarshaller(new GetAutoScalingGroupRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAutoScalingGroupRecommendationsRequest));
            CompletableFuture<GetAutoScalingGroupRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns Amazon Elastic Block Store (Amazon EBS) volume recommendations.
     * </p>
     * <p>
     * Compute Optimizer generates recommendations for Amazon EBS volumes that meet a specific set of requirements. For
     * more information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     * requirements</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     *
     * @param getEbsVolumeRecommendationsRequest
     * @return A Java Future containing the result of the GetEBSVolumeRecommendations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action doesn't exist.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetEBSVolumeRecommendations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetEBSVolumeRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEbsVolumeRecommendationsResponse> getEBSVolumeRecommendations(
            GetEbsVolumeRecommendationsRequest getEbsVolumeRecommendationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEbsVolumeRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEbsVolumeRecommendationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEBSVolumeRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEbsVolumeRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEbsVolumeRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEbsVolumeRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEbsVolumeRecommendationsRequest, GetEbsVolumeRecommendationsResponse>()
                            .withOperationName("GetEBSVolumeRecommendations")
                            .withMarshaller(new GetEbsVolumeRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEbsVolumeRecommendationsRequest));
            CompletableFuture<GetEbsVolumeRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns Amazon EC2 instance recommendations.
     * </p>
     * <p>
     * Compute Optimizer generates recommendations for Amazon Elastic Compute Cloud (Amazon EC2) instances that meet a
     * specific set of requirements. For more information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     * requirements</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     *
     * @param getEc2InstanceRecommendationsRequest
     * @return A Java Future containing the result of the GetEC2InstanceRecommendations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action doesn't exist.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetEC2InstanceRecommendations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetEC2InstanceRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEc2InstanceRecommendationsResponse> getEC2InstanceRecommendations(
            GetEc2InstanceRecommendationsRequest getEc2InstanceRecommendationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEc2InstanceRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getEc2InstanceRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEC2InstanceRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEc2InstanceRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEc2InstanceRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEc2InstanceRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEc2InstanceRecommendationsRequest, GetEc2InstanceRecommendationsResponse>()
                            .withOperationName("GetEC2InstanceRecommendations")
                            .withMarshaller(new GetEc2InstanceRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEc2InstanceRecommendationsRequest));
            CompletableFuture<GetEc2InstanceRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the projected utilization metrics of Amazon EC2 instance recommendations.
     * </p>
     * <note>
     * <p>
     * The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics returned when you
     * run this action. Additionally, the <code>Memory</code> metric is returned only for resources that have the
     * unified CloudWatch agent installed on them. For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory Utilization
     * with the CloudWatch Agent</a>.
     * </p>
     * </note>
     *
     * @param getEc2RecommendationProjectedMetricsRequest
     * @return A Java Future containing the result of the GetEC2RecommendationProjectedMetrics operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action doesn't exist.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetEC2RecommendationProjectedMetrics
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetEC2RecommendationProjectedMetrics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEc2RecommendationProjectedMetricsResponse> getEC2RecommendationProjectedMetrics(
            GetEc2RecommendationProjectedMetricsRequest getEc2RecommendationProjectedMetricsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEc2RecommendationProjectedMetricsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getEc2RecommendationProjectedMetricsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEC2RecommendationProjectedMetrics");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEc2RecommendationProjectedMetricsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetEc2RecommendationProjectedMetricsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEc2RecommendationProjectedMetricsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEc2RecommendationProjectedMetricsRequest, GetEc2RecommendationProjectedMetricsResponse>()
                            .withOperationName("GetEC2RecommendationProjectedMetrics")
                            .withMarshaller(new GetEc2RecommendationProjectedMetricsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEc2RecommendationProjectedMetricsRequest));
            CompletableFuture<GetEc2RecommendationProjectedMetricsResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the projected metrics of Amazon ECS service recommendations.
     * </p>
     *
     * @param getEcsServiceRecommendationProjectedMetricsRequest
     * @return A Java Future containing the result of the GetECSServiceRecommendationProjectedMetrics operation returned
     *         by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action doesn't exist.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetECSServiceRecommendationProjectedMetrics
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetECSServiceRecommendationProjectedMetrics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEcsServiceRecommendationProjectedMetricsResponse> getECSServiceRecommendationProjectedMetrics(
            GetEcsServiceRecommendationProjectedMetricsRequest getEcsServiceRecommendationProjectedMetricsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                getEcsServiceRecommendationProjectedMetricsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getEcsServiceRecommendationProjectedMetricsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetECSServiceRecommendationProjectedMetrics");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEcsServiceRecommendationProjectedMetricsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetEcsServiceRecommendationProjectedMetricsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEcsServiceRecommendationProjectedMetricsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEcsServiceRecommendationProjectedMetricsRequest, GetEcsServiceRecommendationProjectedMetricsResponse>()
                            .withOperationName("GetECSServiceRecommendationProjectedMetrics")
                            .withMarshaller(new GetEcsServiceRecommendationProjectedMetricsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEcsServiceRecommendationProjectedMetricsRequest));
            CompletableFuture<GetEcsServiceRecommendationProjectedMetricsResponse> whenCompleted = executeFuture.whenComplete((r,
                    e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns Amazon ECS service recommendations.
     * </p>
     * <p>
     * Compute Optimizer generates recommendations for Amazon ECS services on Fargate that meet a specific set of
     * requirements. For more information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     * requirements</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     *
     * @param getEcsServiceRecommendationsRequest
     * @return A Java Future containing the result of the GetECSServiceRecommendations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action doesn't exist.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetECSServiceRecommendations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetECSServiceRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEcsServiceRecommendationsResponse> getECSServiceRecommendations(
            GetEcsServiceRecommendationsRequest getEcsServiceRecommendationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEcsServiceRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEcsServiceRecommendationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetECSServiceRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEcsServiceRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEcsServiceRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEcsServiceRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEcsServiceRecommendationsRequest, GetEcsServiceRecommendationsResponse>()
                            .withOperationName("GetECSServiceRecommendations")
                            .withMarshaller(new GetEcsServiceRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEcsServiceRecommendationsRequest));
            CompletableFuture<GetEcsServiceRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the recommendation preferences that are in effect for a given resource, such as enhanced infrastructure
     * metrics. Considers all applicable preferences that you might have set at the resource, account, and organization
     * level.
     * </p>
     * <p>
     * When you create a recommendation preference, you can set its status to <code>Active</code> or
     * <code>Inactive</code>. Use this action to view the recommendation preferences that are in effect, or
     * <code>Active</code>.
     * </p>
     *
     * @param getEffectiveRecommendationPreferencesRequest
     * @return A Java Future containing the result of the GetEffectiveRecommendationPreferences operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action doesn't exist.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetEffectiveRecommendationPreferences
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetEffectiveRecommendationPreferences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEffectiveRecommendationPreferencesResponse> getEffectiveRecommendationPreferences(
            GetEffectiveRecommendationPreferencesRequest getEffectiveRecommendationPreferencesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEffectiveRecommendationPreferencesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getEffectiveRecommendationPreferencesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEffectiveRecommendationPreferences");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEffectiveRecommendationPreferencesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetEffectiveRecommendationPreferencesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEffectiveRecommendationPreferencesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEffectiveRecommendationPreferencesRequest, GetEffectiveRecommendationPreferencesResponse>()
                            .withOperationName("GetEffectiveRecommendationPreferences")
                            .withMarshaller(new GetEffectiveRecommendationPreferencesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEffectiveRecommendationPreferencesRequest));
            CompletableFuture<GetEffectiveRecommendationPreferencesResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the enrollment (opt in) status of an account to the Compute Optimizer service.
     * </p>
     * <p>
     * If the account is the management account of an organization, this action also confirms the enrollment status of
     * member accounts of the organization. Use the <a>GetEnrollmentStatusesForOrganization</a> action to get detailed
     * information about the enrollment status of member accounts of an organization.
     * </p>
     *
     * @param getEnrollmentStatusRequest
     * @return A Java Future containing the result of the GetEnrollmentStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetEnrollmentStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetEnrollmentStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEnrollmentStatusResponse> getEnrollmentStatus(
            GetEnrollmentStatusRequest getEnrollmentStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEnrollmentStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEnrollmentStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEnrollmentStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEnrollmentStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEnrollmentStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEnrollmentStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEnrollmentStatusRequest, GetEnrollmentStatusResponse>()
                            .withOperationName("GetEnrollmentStatus")
                            .withMarshaller(new GetEnrollmentStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEnrollmentStatusRequest));
            CompletableFuture<GetEnrollmentStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the Compute Optimizer enrollment (opt-in) status of organization member accounts, if your account is an
     * organization management account.
     * </p>
     * <p>
     * To get the enrollment status of standalone accounts, use the <a>GetEnrollmentStatus</a> action.
     * </p>
     *
     * @param getEnrollmentStatusesForOrganizationRequest
     * @return A Java Future containing the result of the GetEnrollmentStatusesForOrganization operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetEnrollmentStatusesForOrganization
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetEnrollmentStatusesForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEnrollmentStatusesForOrganizationResponse> getEnrollmentStatusesForOrganization(
            GetEnrollmentStatusesForOrganizationRequest getEnrollmentStatusesForOrganizationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEnrollmentStatusesForOrganizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getEnrollmentStatusesForOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEnrollmentStatusesForOrganization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEnrollmentStatusesForOrganizationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetEnrollmentStatusesForOrganizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEnrollmentStatusesForOrganizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEnrollmentStatusesForOrganizationRequest, GetEnrollmentStatusesForOrganizationResponse>()
                            .withOperationName("GetEnrollmentStatusesForOrganization")
                            .withMarshaller(new GetEnrollmentStatusesForOrganizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getEnrollmentStatusesForOrganizationRequest));
            CompletableFuture<GetEnrollmentStatusesForOrganizationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns Lambda function recommendations.
     * </p>
     * <p>
     * Compute Optimizer generates recommendations for functions that meet a specific set of requirements. For more
     * information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     * requirements</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     *
     * @param getLambdaFunctionRecommendationsRequest
     * @return A Java Future containing the result of the GetLambdaFunctionRecommendations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The request exceeds a limit of the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetLambdaFunctionRecommendations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetLambdaFunctionRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLambdaFunctionRecommendationsResponse> getLambdaFunctionRecommendations(
            GetLambdaFunctionRecommendationsRequest getLambdaFunctionRecommendationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLambdaFunctionRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getLambdaFunctionRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLambdaFunctionRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLambdaFunctionRecommendationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetLambdaFunctionRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLambdaFunctionRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLambdaFunctionRecommendationsRequest, GetLambdaFunctionRecommendationsResponse>()
                            .withOperationName("GetLambdaFunctionRecommendations")
                            .withMarshaller(new GetLambdaFunctionRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLambdaFunctionRecommendationsRequest));
            CompletableFuture<GetLambdaFunctionRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns license recommendations for Amazon EC2 instances that run on a specific license.
     * </p>
     * <p>
     * Compute Optimizer generates recommendations for licenses that meet a specific set of requirements. For more
     * information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     * requirements</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     *
     * @param getLicenseRecommendationsRequest
     * @return A Java Future containing the result of the GetLicenseRecommendations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action doesn't exist.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetLicenseRecommendations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetLicenseRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLicenseRecommendationsResponse> getLicenseRecommendations(
            GetLicenseRecommendationsRequest getLicenseRecommendationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLicenseRecommendationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLicenseRecommendationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLicenseRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLicenseRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLicenseRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLicenseRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLicenseRecommendationsRequest, GetLicenseRecommendationsResponse>()
                            .withOperationName("GetLicenseRecommendations")
                            .withMarshaller(new GetLicenseRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLicenseRecommendationsRequest));
            CompletableFuture<GetLicenseRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns existing recommendation preferences, such as enhanced infrastructure metrics.
     * </p>
     * <p>
     * Use the <code>scope</code> parameter to specify which preferences to return. You can specify to return
     * preferences for an organization, a specific account ID, or a specific EC2 instance or Auto Scaling group Amazon
     * Resource Name (ARN).
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html">Activating
     * enhanced infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     *
     * @param getRecommendationPreferencesRequest
     * @return A Java Future containing the result of the GetRecommendationPreferences operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action doesn't exist.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetRecommendationPreferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetRecommendationPreferences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRecommendationPreferencesResponse> getRecommendationPreferences(
            GetRecommendationPreferencesRequest getRecommendationPreferencesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRecommendationPreferencesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRecommendationPreferencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecommendationPreferences");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRecommendationPreferencesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRecommendationPreferencesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRecommendationPreferencesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRecommendationPreferencesRequest, GetRecommendationPreferencesResponse>()
                            .withOperationName("GetRecommendationPreferences")
                            .withMarshaller(new GetRecommendationPreferencesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRecommendationPreferencesRequest));
            CompletableFuture<GetRecommendationPreferencesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the optimization findings for an account.
     * </p>
     * <p>
     * It returns the number of:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon EC2 instances in an account that are <code>Underprovisioned</code>, <code>Overprovisioned</code>, or
     * <code>Optimized</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Auto Scaling groups in an account that are <code>NotOptimized</code>, or <code>Optimized</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon EBS volumes in an account that are <code>NotOptimized</code>, or <code>Optimized</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lambda functions in an account that are <code>NotOptimized</code>, or <code>Optimized</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon ECS services in an account that are <code>Underprovisioned</code>, <code>Overprovisioned</code>, or
     * <code>Optimized</code>.
     * </p>
     * </li>
     * </ul>
     *
     * @param getRecommendationSummariesRequest
     * @return A Java Future containing the result of the GetRecommendationSummaries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetRecommendationSummaries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetRecommendationSummaries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRecommendationSummariesResponse> getRecommendationSummaries(
            GetRecommendationSummariesRequest getRecommendationSummariesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRecommendationSummariesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRecommendationSummariesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecommendationSummaries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRecommendationSummariesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRecommendationSummariesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRecommendationSummariesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRecommendationSummariesRequest, GetRecommendationSummariesResponse>()
                            .withOperationName("GetRecommendationSummaries")
                            .withMarshaller(new GetRecommendationSummariesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getRecommendationSummariesRequest));
            CompletableFuture<GetRecommendationSummariesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new recommendation preference or updates an existing recommendation preference, such as enhanced
     * infrastructure metrics.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/enhanced-infrastructure-metrics.html">Activating
     * enhanced infrastructure metrics</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     *
     * @param putRecommendationPreferencesRequest
     * @return A Java Future containing the result of the PutRecommendationPreferences operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>ResourceNotFoundException A resource that is required for the action doesn't exist.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.PutRecommendationPreferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/PutRecommendationPreferences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<PutRecommendationPreferencesResponse> putRecommendationPreferences(
            PutRecommendationPreferencesRequest putRecommendationPreferencesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRecommendationPreferencesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRecommendationPreferencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRecommendationPreferences");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutRecommendationPreferencesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutRecommendationPreferencesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutRecommendationPreferencesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutRecommendationPreferencesRequest, PutRecommendationPreferencesResponse>()
                            .withOperationName("PutRecommendationPreferences")
                            .withMarshaller(new PutRecommendationPreferencesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(putRecommendationPreferencesRequest));
            CompletableFuture<PutRecommendationPreferencesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the enrollment (opt in and opt out) status of an account to the Compute Optimizer service.
     * </p>
     * <p>
     * If the account is a management account of an organization, this action can also be used to enroll member accounts
     * of the organization.
     * </p>
     * <p>
     * You must have the appropriate permissions to opt in to Compute Optimizer, to view its recommendations, and to opt
     * out. For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/security-iam.html">Controlling access with Amazon
     * Web Services Identity and Access Management</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     * <p>
     * When you opt in, Compute Optimizer automatically creates a service-linked role in your account to access its
     * data. For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/using-service-linked-roles.html">Using
     * Service-Linked Roles for Compute Optimizer</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     *
     * @param updateEnrollmentStatusRequest
     * @return A Java Future containing the result of the UpdateEnrollmentStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException The value supplied for the input parameter is out of range or not
     *         valid.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) Amazon Web
     *         Services access key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.UpdateEnrollmentStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/UpdateEnrollmentStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEnrollmentStatusResponse> updateEnrollmentStatus(
            UpdateEnrollmentStatusRequest updateEnrollmentStatusRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateEnrollmentStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEnrollmentStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEnrollmentStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEnrollmentStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEnrollmentStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEnrollmentStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEnrollmentStatusRequest, UpdateEnrollmentStatusResponse>()
                            .withOperationName("UpdateEnrollmentStatus")
                            .withMarshaller(new UpdateEnrollmentStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateEnrollmentStatusRequest));
            CompletableFuture<UpdateEnrollmentStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final ComputeOptimizerServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ComputeOptimizerException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OptInRequiredException")
                                .exceptionBuilderSupplier(OptInRequiredException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingAuthenticationToken")
                                .exceptionBuilderSupplier(MissingAuthenticationTokenException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        ComputeOptimizerServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = ComputeOptimizerServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
