/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Auto Scaling group recommendation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutoScalingGroupRecommendation implements SdkPojo, Serializable,
        ToCopyableBuilder<AutoScalingGroupRecommendation.Builder, AutoScalingGroupRecommendation> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accountId").getter(getter(AutoScalingGroupRecommendation::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<String> AUTO_SCALING_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("autoScalingGroupArn").getter(getter(AutoScalingGroupRecommendation::autoScalingGroupArn))
            .setter(setter(Builder::autoScalingGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autoScalingGroupArn").build())
            .build();

    private static final SdkField<String> AUTO_SCALING_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("autoScalingGroupName").getter(getter(AutoScalingGroupRecommendation::autoScalingGroupName))
            .setter(setter(Builder::autoScalingGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("autoScalingGroupName").build())
            .build();

    private static final SdkField<String> FINDING_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("finding")
            .getter(getter(AutoScalingGroupRecommendation::findingAsString)).setter(setter(Builder::finding))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("finding").build()).build();

    private static final SdkField<List<UtilizationMetric>> UTILIZATION_METRICS_FIELD = SdkField
            .<List<UtilizationMetric>> builder(MarshallingType.LIST)
            .memberName("utilizationMetrics")
            .getter(getter(AutoScalingGroupRecommendation::utilizationMetrics))
            .setter(setter(Builder::utilizationMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("utilizationMetrics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<UtilizationMetric> builder(MarshallingType.SDK_POJO)
                                            .constructor(UtilizationMetric::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Double> LOOK_BACK_PERIOD_IN_DAYS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("lookBackPeriodInDays").getter(getter(AutoScalingGroupRecommendation::lookBackPeriodInDays))
            .setter(setter(Builder::lookBackPeriodInDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lookBackPeriodInDays").build())
            .build();

    private static final SdkField<AutoScalingGroupConfiguration> CURRENT_CONFIGURATION_FIELD = SdkField
            .<AutoScalingGroupConfiguration> builder(MarshallingType.SDK_POJO).memberName("currentConfiguration")
            .getter(getter(AutoScalingGroupRecommendation::currentConfiguration)).setter(setter(Builder::currentConfiguration))
            .constructor(AutoScalingGroupConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentConfiguration").build())
            .build();

    private static final SdkField<List<AutoScalingGroupRecommendationOption>> RECOMMENDATION_OPTIONS_FIELD = SdkField
            .<List<AutoScalingGroupRecommendationOption>> builder(MarshallingType.LIST)
            .memberName("recommendationOptions")
            .getter(getter(AutoScalingGroupRecommendation::recommendationOptions))
            .setter(setter(Builder::recommendationOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recommendationOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AutoScalingGroupRecommendationOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(AutoScalingGroupRecommendationOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> LAST_REFRESH_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastRefreshTimestamp").getter(getter(AutoScalingGroupRecommendation::lastRefreshTimestamp))
            .setter(setter(Builder::lastRefreshTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastRefreshTimestamp").build())
            .build();

    private static final SdkField<String> CURRENT_PERFORMANCE_RISK_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("currentPerformanceRisk").getter(getter(AutoScalingGroupRecommendation::currentPerformanceRiskAsString))
            .setter(setter(Builder::currentPerformanceRisk))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentPerformanceRisk").build())
            .build();

    private static final SdkField<EffectiveRecommendationPreferences> EFFECTIVE_RECOMMENDATION_PREFERENCES_FIELD = SdkField
            .<EffectiveRecommendationPreferences> builder(MarshallingType.SDK_POJO)
            .memberName("effectiveRecommendationPreferences")
            .getter(getter(AutoScalingGroupRecommendation::effectiveRecommendationPreferences))
            .setter(setter(Builder::effectiveRecommendationPreferences))
            .constructor(EffectiveRecommendationPreferences::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("effectiveRecommendationPreferences")
                    .build()).build();

    private static final SdkField<List<String>> INFERRED_WORKLOAD_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("inferredWorkloadTypes")
            .getter(getter(AutoScalingGroupRecommendation::inferredWorkloadTypesAsStrings))
            .setter(setter(Builder::inferredWorkloadTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inferredWorkloadTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<GpuInfo> CURRENT_INSTANCE_GPU_INFO_FIELD = SdkField.<GpuInfo> builder(MarshallingType.SDK_POJO)
            .memberName("currentInstanceGpuInfo").getter(getter(AutoScalingGroupRecommendation::currentInstanceGpuInfo))
            .setter(setter(Builder::currentInstanceGpuInfo)).constructor(GpuInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentInstanceGpuInfo").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            AUTO_SCALING_GROUP_ARN_FIELD, AUTO_SCALING_GROUP_NAME_FIELD, FINDING_FIELD, UTILIZATION_METRICS_FIELD,
            LOOK_BACK_PERIOD_IN_DAYS_FIELD, CURRENT_CONFIGURATION_FIELD, RECOMMENDATION_OPTIONS_FIELD,
            LAST_REFRESH_TIMESTAMP_FIELD, CURRENT_PERFORMANCE_RISK_FIELD, EFFECTIVE_RECOMMENDATION_PREFERENCES_FIELD,
            INFERRED_WORKLOAD_TYPES_FIELD, CURRENT_INSTANCE_GPU_INFO_FIELD));

    private static final long serialVersionUID = 1L;

    private final String accountId;

    private final String autoScalingGroupArn;

    private final String autoScalingGroupName;

    private final String finding;

    private final List<UtilizationMetric> utilizationMetrics;

    private final Double lookBackPeriodInDays;

    private final AutoScalingGroupConfiguration currentConfiguration;

    private final List<AutoScalingGroupRecommendationOption> recommendationOptions;

    private final Instant lastRefreshTimestamp;

    private final String currentPerformanceRisk;

    private final EffectiveRecommendationPreferences effectiveRecommendationPreferences;

    private final List<String> inferredWorkloadTypes;

    private final GpuInfo currentInstanceGpuInfo;

    private AutoScalingGroupRecommendation(BuilderImpl builder) {
        this.accountId = builder.accountId;
        this.autoScalingGroupArn = builder.autoScalingGroupArn;
        this.autoScalingGroupName = builder.autoScalingGroupName;
        this.finding = builder.finding;
        this.utilizationMetrics = builder.utilizationMetrics;
        this.lookBackPeriodInDays = builder.lookBackPeriodInDays;
        this.currentConfiguration = builder.currentConfiguration;
        this.recommendationOptions = builder.recommendationOptions;
        this.lastRefreshTimestamp = builder.lastRefreshTimestamp;
        this.currentPerformanceRisk = builder.currentPerformanceRisk;
        this.effectiveRecommendationPreferences = builder.effectiveRecommendationPreferences;
        this.inferredWorkloadTypes = builder.inferredWorkloadTypes;
        this.currentInstanceGpuInfo = builder.currentInstanceGpuInfo;
    }

    /**
     * <p>
     * The Amazon Web Services account ID of the Auto Scaling group.
     * </p>
     * 
     * @return The Amazon Web Services account ID of the Auto Scaling group.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Auto Scaling group.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Auto Scaling group.
     */
    public final String autoScalingGroupArn() {
        return autoScalingGroupArn;
    }

    /**
     * <p>
     * The name of the Auto Scaling group.
     * </p>
     * 
     * @return The name of the Auto Scaling group.
     */
    public final String autoScalingGroupName() {
        return autoScalingGroupName;
    }

    /**
     * <p>
     * The finding classification of the Auto Scaling group.
     * </p>
     * <p>
     * Findings for Auto Scaling groups include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>NotOptimized</code> </b>—An Auto Scaling group is considered not optimized when Compute Optimizer
     * identifies a recommendation that can provide better performance for your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b>—An Auto Scaling group is considered optimized when Compute Optimizer determines
     * that the group is correctly provisioned to run your workload based on the chosen instance type. For optimized
     * resources, Compute Optimizer might recommend a new generation instance type.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link Finding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification of the Auto Scaling group.</p>
     *         <p>
     *         Findings for Auto Scaling groups include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>NotOptimized</code> </b>—An Auto Scaling group is considered not optimized when Compute
     *         Optimizer identifies a recommendation that can provide better performance for your workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b>—An Auto Scaling group is considered optimized when Compute Optimizer
     *         determines that the group is correctly provisioned to run your workload based on the chosen instance
     *         type. For optimized resources, Compute Optimizer might recommend a new generation instance type.
     *         </p>
     *         </li>
     * @see Finding
     */
    public final Finding finding() {
        return Finding.fromValue(finding);
    }

    /**
     * <p>
     * The finding classification of the Auto Scaling group.
     * </p>
     * <p>
     * Findings for Auto Scaling groups include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>NotOptimized</code> </b>—An Auto Scaling group is considered not optimized when Compute Optimizer
     * identifies a recommendation that can provide better performance for your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b>—An Auto Scaling group is considered optimized when Compute Optimizer determines
     * that the group is correctly provisioned to run your workload based on the chosen instance type. For optimized
     * resources, Compute Optimizer might recommend a new generation instance type.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link Finding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification of the Auto Scaling group.</p>
     *         <p>
     *         Findings for Auto Scaling groups include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>NotOptimized</code> </b>—An Auto Scaling group is considered not optimized when Compute
     *         Optimizer identifies a recommendation that can provide better performance for your workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b>—An Auto Scaling group is considered optimized when Compute Optimizer
     *         determines that the group is correctly provisioned to run your workload based on the chosen instance
     *         type. For optimized resources, Compute Optimizer might recommend a new generation instance type.
     *         </p>
     *         </li>
     * @see Finding
     */
    public final String findingAsString() {
        return finding;
    }

    /**
     * For responses, this returns true if the service returned a value for the UtilizationMetrics property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasUtilizationMetrics() {
        return utilizationMetrics != null && !(utilizationMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the utilization metrics of the Auto Scaling group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUtilizationMetrics} method.
     * </p>
     * 
     * @return An array of objects that describe the utilization metrics of the Auto Scaling group.
     */
    public final List<UtilizationMetric> utilizationMetrics() {
        return utilizationMetrics;
    }

    /**
     * <p>
     * The number of days for which utilization metrics were analyzed for the Auto Scaling group.
     * </p>
     * 
     * @return The number of days for which utilization metrics were analyzed for the Auto Scaling group.
     */
    public final Double lookBackPeriodInDays() {
        return lookBackPeriodInDays;
    }

    /**
     * <p>
     * An array of objects that describe the current configuration of the Auto Scaling group.
     * </p>
     * 
     * @return An array of objects that describe the current configuration of the Auto Scaling group.
     */
    public final AutoScalingGroupConfiguration currentConfiguration() {
        return currentConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the RecommendationOptions property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasRecommendationOptions() {
        return recommendationOptions != null && !(recommendationOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the recommendation options for the Auto Scaling group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRecommendationOptions} method.
     * </p>
     * 
     * @return An array of objects that describe the recommendation options for the Auto Scaling group.
     */
    public final List<AutoScalingGroupRecommendationOption> recommendationOptions() {
        return recommendationOptions;
    }

    /**
     * <p>
     * The timestamp of when the Auto Scaling group recommendation was last generated.
     * </p>
     * 
     * @return The timestamp of when the Auto Scaling group recommendation was last generated.
     */
    public final Instant lastRefreshTimestamp() {
        return lastRefreshTimestamp;
    }

    /**
     * <p>
     * The risk of the current Auto Scaling group not meeting the performance needs of its workloads. The higher the
     * risk, the more likely the current Auto Scaling group configuration has insufficient capacity and cannot meet
     * workload requirements.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #currentPerformanceRisk} will return {@link CurrentPerformanceRisk#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #currentPerformanceRiskAsString}.
     * </p>
     * 
     * @return The risk of the current Auto Scaling group not meeting the performance needs of its workloads. The higher
     *         the risk, the more likely the current Auto Scaling group configuration has insufficient capacity and
     *         cannot meet workload requirements.
     * @see CurrentPerformanceRisk
     */
    public final CurrentPerformanceRisk currentPerformanceRisk() {
        return CurrentPerformanceRisk.fromValue(currentPerformanceRisk);
    }

    /**
     * <p>
     * The risk of the current Auto Scaling group not meeting the performance needs of its workloads. The higher the
     * risk, the more likely the current Auto Scaling group configuration has insufficient capacity and cannot meet
     * workload requirements.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #currentPerformanceRisk} will return {@link CurrentPerformanceRisk#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #currentPerformanceRiskAsString}.
     * </p>
     * 
     * @return The risk of the current Auto Scaling group not meeting the performance needs of its workloads. The higher
     *         the risk, the more likely the current Auto Scaling group configuration has insufficient capacity and
     *         cannot meet workload requirements.
     * @see CurrentPerformanceRisk
     */
    public final String currentPerformanceRiskAsString() {
        return currentPerformanceRisk;
    }

    /**
     * <p>
     * An object that describes the effective recommendation preferences for the Auto Scaling group.
     * </p>
     * 
     * @return An object that describes the effective recommendation preferences for the Auto Scaling group.
     */
    public final EffectiveRecommendationPreferences effectiveRecommendationPreferences() {
        return effectiveRecommendationPreferences;
    }

    /**
     * <p>
     * The applications that might be running on the instances in the Auto Scaling group as inferred by Compute
     * Optimizer.
     * </p>
     * <p>
     * Compute Optimizer can infer if one of the following applications might be running on the instances:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Memcached</code> - Infers that Memcached might be running on the instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NGINX</code> - Infers that NGINX might be running on the instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Redis</code> - Infers that Redis might be running on the instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Kafka</code> - Infers that Kafka might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInferredWorkloadTypes} method.
     * </p>
     * 
     * @return The applications that might be running on the instances in the Auto Scaling group as inferred by Compute
     *         Optimizer.</p>
     *         <p>
     *         Compute Optimizer can infer if one of the following applications might be running on the instances:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Memcached</code> - Infers that Memcached might be running on the instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NGINX</code> - Infers that NGINX might be running on the instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Redis</code> - Infers that Redis might be running on the instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Kafka</code> - Infers that Kafka might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
     *         </p>
     *         </li>
     */
    public final List<InferredWorkloadType> inferredWorkloadTypes() {
        return InferredWorkloadTypesCopier.copyStringToEnum(inferredWorkloadTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the InferredWorkloadTypes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasInferredWorkloadTypes() {
        return inferredWorkloadTypes != null && !(inferredWorkloadTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The applications that might be running on the instances in the Auto Scaling group as inferred by Compute
     * Optimizer.
     * </p>
     * <p>
     * Compute Optimizer can infer if one of the following applications might be running on the instances:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Memcached</code> - Infers that Memcached might be running on the instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NGINX</code> - Infers that NGINX might be running on the instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Redis</code> - Infers that Redis might be running on the instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Kafka</code> - Infers that Kafka might be running on the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInferredWorkloadTypes} method.
     * </p>
     * 
     * @return The applications that might be running on the instances in the Auto Scaling group as inferred by Compute
     *         Optimizer.</p>
     *         <p>
     *         Compute Optimizer can infer if one of the following applications might be running on the instances:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Memcached</code> - Infers that Memcached might be running on the instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NGINX</code> - Infers that NGINX might be running on the instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Redis</code> - Infers that Redis might be running on the instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Kafka</code> - Infers that Kafka might be running on the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
     *         </p>
     *         </li>
     */
    public final List<String> inferredWorkloadTypesAsStrings() {
        return inferredWorkloadTypes;
    }

    /**
     * <p>
     * Describes the GPU accelerator settings for the current instance type of the Auto Scaling group.
     * </p>
     * 
     * @return Describes the GPU accelerator settings for the current instance type of the Auto Scaling group.
     */
    public final GpuInfo currentInstanceGpuInfo() {
        return currentInstanceGpuInfo;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingGroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(findingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasUtilizationMetrics() ? utilizationMetrics() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lookBackPeriodInDays());
        hashCode = 31 * hashCode + Objects.hashCode(currentConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasRecommendationOptions() ? recommendationOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lastRefreshTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(currentPerformanceRiskAsString());
        hashCode = 31 * hashCode + Objects.hashCode(effectiveRecommendationPreferences());
        hashCode = 31 * hashCode + Objects.hashCode(hasInferredWorkloadTypes() ? inferredWorkloadTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(currentInstanceGpuInfo());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutoScalingGroupRecommendation)) {
            return false;
        }
        AutoScalingGroupRecommendation other = (AutoScalingGroupRecommendation) obj;
        return Objects.equals(accountId(), other.accountId())
                && Objects.equals(autoScalingGroupArn(), other.autoScalingGroupArn())
                && Objects.equals(autoScalingGroupName(), other.autoScalingGroupName())
                && Objects.equals(findingAsString(), other.findingAsString())
                && hasUtilizationMetrics() == other.hasUtilizationMetrics()
                && Objects.equals(utilizationMetrics(), other.utilizationMetrics())
                && Objects.equals(lookBackPeriodInDays(), other.lookBackPeriodInDays())
                && Objects.equals(currentConfiguration(), other.currentConfiguration())
                && hasRecommendationOptions() == other.hasRecommendationOptions()
                && Objects.equals(recommendationOptions(), other.recommendationOptions())
                && Objects.equals(lastRefreshTimestamp(), other.lastRefreshTimestamp())
                && Objects.equals(currentPerformanceRiskAsString(), other.currentPerformanceRiskAsString())
                && Objects.equals(effectiveRecommendationPreferences(), other.effectiveRecommendationPreferences())
                && hasInferredWorkloadTypes() == other.hasInferredWorkloadTypes()
                && Objects.equals(inferredWorkloadTypesAsStrings(), other.inferredWorkloadTypesAsStrings())
                && Objects.equals(currentInstanceGpuInfo(), other.currentInstanceGpuInfo());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AutoScalingGroupRecommendation").add("AccountId", accountId())
                .add("AutoScalingGroupArn", autoScalingGroupArn()).add("AutoScalingGroupName", autoScalingGroupName())
                .add("Finding", findingAsString())
                .add("UtilizationMetrics", hasUtilizationMetrics() ? utilizationMetrics() : null)
                .add("LookBackPeriodInDays", lookBackPeriodInDays()).add("CurrentConfiguration", currentConfiguration())
                .add("RecommendationOptions", hasRecommendationOptions() ? recommendationOptions() : null)
                .add("LastRefreshTimestamp", lastRefreshTimestamp())
                .add("CurrentPerformanceRisk", currentPerformanceRiskAsString())
                .add("EffectiveRecommendationPreferences", effectiveRecommendationPreferences())
                .add("InferredWorkloadTypes", hasInferredWorkloadTypes() ? inferredWorkloadTypesAsStrings() : null)
                .add("CurrentInstanceGpuInfo", currentInstanceGpuInfo()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "autoScalingGroupArn":
            return Optional.ofNullable(clazz.cast(autoScalingGroupArn()));
        case "autoScalingGroupName":
            return Optional.ofNullable(clazz.cast(autoScalingGroupName()));
        case "finding":
            return Optional.ofNullable(clazz.cast(findingAsString()));
        case "utilizationMetrics":
            return Optional.ofNullable(clazz.cast(utilizationMetrics()));
        case "lookBackPeriodInDays":
            return Optional.ofNullable(clazz.cast(lookBackPeriodInDays()));
        case "currentConfiguration":
            return Optional.ofNullable(clazz.cast(currentConfiguration()));
        case "recommendationOptions":
            return Optional.ofNullable(clazz.cast(recommendationOptions()));
        case "lastRefreshTimestamp":
            return Optional.ofNullable(clazz.cast(lastRefreshTimestamp()));
        case "currentPerformanceRisk":
            return Optional.ofNullable(clazz.cast(currentPerformanceRiskAsString()));
        case "effectiveRecommendationPreferences":
            return Optional.ofNullable(clazz.cast(effectiveRecommendationPreferences()));
        case "inferredWorkloadTypes":
            return Optional.ofNullable(clazz.cast(inferredWorkloadTypesAsStrings()));
        case "currentInstanceGpuInfo":
            return Optional.ofNullable(clazz.cast(currentInstanceGpuInfo()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AutoScalingGroupRecommendation, T> g) {
        return obj -> g.apply((AutoScalingGroupRecommendation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AutoScalingGroupRecommendation> {
        /**
         * <p>
         * The Amazon Web Services account ID of the Auto Scaling group.
         * </p>
         * 
         * @param accountId
         *        The Amazon Web Services account ID of the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Auto Scaling group.
         * </p>
         * 
         * @param autoScalingGroupArn
         *        The Amazon Resource Name (ARN) of the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingGroupArn(String autoScalingGroupArn);

        /**
         * <p>
         * The name of the Auto Scaling group.
         * </p>
         * 
         * @param autoScalingGroupName
         *        The name of the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingGroupName(String autoScalingGroupName);

        /**
         * <p>
         * The finding classification of the Auto Scaling group.
         * </p>
         * <p>
         * Findings for Auto Scaling groups include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>NotOptimized</code> </b>—An Auto Scaling group is considered not optimized when Compute Optimizer
         * identifies a recommendation that can provide better performance for your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b>—An Auto Scaling group is considered optimized when Compute Optimizer
         * determines that the group is correctly provisioned to run your workload based on the chosen instance type.
         * For optimized resources, Compute Optimizer might recommend a new generation instance type.
         * </p>
         * </li>
         * </ul>
         * 
         * @param finding
         *        The finding classification of the Auto Scaling group.</p>
         *        <p>
         *        Findings for Auto Scaling groups include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>NotOptimized</code> </b>—An Auto Scaling group is considered not optimized when Compute
         *        Optimizer identifies a recommendation that can provide better performance for your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b>—An Auto Scaling group is considered optimized when Compute Optimizer
         *        determines that the group is correctly provisioned to run your workload based on the chosen instance
         *        type. For optimized resources, Compute Optimizer might recommend a new generation instance type.
         *        </p>
         *        </li>
         * @see Finding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Finding
         */
        Builder finding(String finding);

        /**
         * <p>
         * The finding classification of the Auto Scaling group.
         * </p>
         * <p>
         * Findings for Auto Scaling groups include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>NotOptimized</code> </b>—An Auto Scaling group is considered not optimized when Compute Optimizer
         * identifies a recommendation that can provide better performance for your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b>—An Auto Scaling group is considered optimized when Compute Optimizer
         * determines that the group is correctly provisioned to run your workload based on the chosen instance type.
         * For optimized resources, Compute Optimizer might recommend a new generation instance type.
         * </p>
         * </li>
         * </ul>
         * 
         * @param finding
         *        The finding classification of the Auto Scaling group.</p>
         *        <p>
         *        Findings for Auto Scaling groups include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>NotOptimized</code> </b>—An Auto Scaling group is considered not optimized when Compute
         *        Optimizer identifies a recommendation that can provide better performance for your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b>—An Auto Scaling group is considered optimized when Compute Optimizer
         *        determines that the group is correctly provisioned to run your workload based on the chosen instance
         *        type. For optimized resources, Compute Optimizer might recommend a new generation instance type.
         *        </p>
         *        </li>
         * @see Finding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Finding
         */
        Builder finding(Finding finding);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the Auto Scaling group.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(Collection<UtilizationMetric> utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the Auto Scaling group.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(UtilizationMetric... utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the Auto Scaling group.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationMetric.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationMetric#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationMetric.Builder#build()} is called
         * immediately and its result is passed to {@link #utilizationMetrics(List<UtilizationMetric>)}.
         * 
         * @param utilizationMetrics
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.UtilizationMetric.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #utilizationMetrics(java.util.Collection<UtilizationMetric>)
         */
        Builder utilizationMetrics(Consumer<UtilizationMetric.Builder>... utilizationMetrics);

        /**
         * <p>
         * The number of days for which utilization metrics were analyzed for the Auto Scaling group.
         * </p>
         * 
         * @param lookBackPeriodInDays
         *        The number of days for which utilization metrics were analyzed for the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lookBackPeriodInDays(Double lookBackPeriodInDays);

        /**
         * <p>
         * An array of objects that describe the current configuration of the Auto Scaling group.
         * </p>
         * 
         * @param currentConfiguration
         *        An array of objects that describe the current configuration of the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentConfiguration(AutoScalingGroupConfiguration currentConfiguration);

        /**
         * <p>
         * An array of objects that describe the current configuration of the Auto Scaling group.
         * </p>
         * This is a convenience method that creates an instance of the {@link AutoScalingGroupConfiguration.Builder}
         * avoiding the need to create one manually via {@link AutoScalingGroupConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AutoScalingGroupConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #currentConfiguration(AutoScalingGroupConfiguration)}.
         * 
         * @param currentConfiguration
         *        a consumer that will call methods on {@link AutoScalingGroupConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #currentConfiguration(AutoScalingGroupConfiguration)
         */
        default Builder currentConfiguration(Consumer<AutoScalingGroupConfiguration.Builder> currentConfiguration) {
            return currentConfiguration(AutoScalingGroupConfiguration.builder().applyMutation(currentConfiguration).build());
        }

        /**
         * <p>
         * An array of objects that describe the recommendation options for the Auto Scaling group.
         * </p>
         * 
         * @param recommendationOptions
         *        An array of objects that describe the recommendation options for the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationOptions(Collection<AutoScalingGroupRecommendationOption> recommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the Auto Scaling group.
         * </p>
         * 
         * @param recommendationOptions
         *        An array of objects that describe the recommendation options for the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationOptions(AutoScalingGroupRecommendationOption... recommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the Auto Scaling group.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.computeoptimizer.model.AutoScalingGroupRecommendationOption.Builder}
         * avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.computeoptimizer.model.AutoScalingGroupRecommendationOption#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.computeoptimizer.model.AutoScalingGroupRecommendationOption.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #recommendationOptions(List<AutoScalingGroupRecommendationOption>)}.
         * 
         * @param recommendationOptions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.computeoptimizer.model.AutoScalingGroupRecommendationOption.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recommendationOptions(java.util.Collection<AutoScalingGroupRecommendationOption>)
         */
        Builder recommendationOptions(Consumer<AutoScalingGroupRecommendationOption.Builder>... recommendationOptions);

        /**
         * <p>
         * The timestamp of when the Auto Scaling group recommendation was last generated.
         * </p>
         * 
         * @param lastRefreshTimestamp
         *        The timestamp of when the Auto Scaling group recommendation was last generated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastRefreshTimestamp(Instant lastRefreshTimestamp);

        /**
         * <p>
         * The risk of the current Auto Scaling group not meeting the performance needs of its workloads. The higher the
         * risk, the more likely the current Auto Scaling group configuration has insufficient capacity and cannot meet
         * workload requirements.
         * </p>
         * 
         * @param currentPerformanceRisk
         *        The risk of the current Auto Scaling group not meeting the performance needs of its workloads. The
         *        higher the risk, the more likely the current Auto Scaling group configuration has insufficient
         *        capacity and cannot meet workload requirements.
         * @see CurrentPerformanceRisk
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CurrentPerformanceRisk
         */
        Builder currentPerformanceRisk(String currentPerformanceRisk);

        /**
         * <p>
         * The risk of the current Auto Scaling group not meeting the performance needs of its workloads. The higher the
         * risk, the more likely the current Auto Scaling group configuration has insufficient capacity and cannot meet
         * workload requirements.
         * </p>
         * 
         * @param currentPerformanceRisk
         *        The risk of the current Auto Scaling group not meeting the performance needs of its workloads. The
         *        higher the risk, the more likely the current Auto Scaling group configuration has insufficient
         *        capacity and cannot meet workload requirements.
         * @see CurrentPerformanceRisk
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CurrentPerformanceRisk
         */
        Builder currentPerformanceRisk(CurrentPerformanceRisk currentPerformanceRisk);

        /**
         * <p>
         * An object that describes the effective recommendation preferences for the Auto Scaling group.
         * </p>
         * 
         * @param effectiveRecommendationPreferences
         *        An object that describes the effective recommendation preferences for the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder effectiveRecommendationPreferences(EffectiveRecommendationPreferences effectiveRecommendationPreferences);

        /**
         * <p>
         * An object that describes the effective recommendation preferences for the Auto Scaling group.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link EffectiveRecommendationPreferences.Builder} avoiding the need to create one manually via
         * {@link EffectiveRecommendationPreferences#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EffectiveRecommendationPreferences.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #effectiveRecommendationPreferences(EffectiveRecommendationPreferences)}.
         * 
         * @param effectiveRecommendationPreferences
         *        a consumer that will call methods on {@link EffectiveRecommendationPreferences.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #effectiveRecommendationPreferences(EffectiveRecommendationPreferences)
         */
        default Builder effectiveRecommendationPreferences(
                Consumer<EffectiveRecommendationPreferences.Builder> effectiveRecommendationPreferences) {
            return effectiveRecommendationPreferences(EffectiveRecommendationPreferences.builder()
                    .applyMutation(effectiveRecommendationPreferences).build());
        }

        /**
         * <p>
         * The applications that might be running on the instances in the Auto Scaling group as inferred by Compute
         * Optimizer.
         * </p>
         * <p>
         * Compute Optimizer can infer if one of the following applications might be running on the instances:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Memcached</code> - Infers that Memcached might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NGINX</code> - Infers that NGINX might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Redis</code> - Infers that Redis might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Kafka</code> - Infers that Kafka might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         * </p>
         * </li>
         * </ul>
         * 
         * @param inferredWorkloadTypes
         *        The applications that might be running on the instances in the Auto Scaling group as inferred by
         *        Compute Optimizer.</p>
         *        <p>
         *        Compute Optimizer can infer if one of the following applications might be running on the instances:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Memcached</code> - Infers that Memcached might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NGINX</code> - Infers that NGINX might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Redis</code> - Infers that Redis might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Kafka</code> - Infers that Kafka might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferredWorkloadTypesWithStrings(Collection<String> inferredWorkloadTypes);

        /**
         * <p>
         * The applications that might be running on the instances in the Auto Scaling group as inferred by Compute
         * Optimizer.
         * </p>
         * <p>
         * Compute Optimizer can infer if one of the following applications might be running on the instances:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Memcached</code> - Infers that Memcached might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NGINX</code> - Infers that NGINX might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Redis</code> - Infers that Redis might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Kafka</code> - Infers that Kafka might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         * </p>
         * </li>
         * </ul>
         * 
         * @param inferredWorkloadTypes
         *        The applications that might be running on the instances in the Auto Scaling group as inferred by
         *        Compute Optimizer.</p>
         *        <p>
         *        Compute Optimizer can infer if one of the following applications might be running on the instances:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Memcached</code> - Infers that Memcached might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NGINX</code> - Infers that NGINX might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Redis</code> - Infers that Redis might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Kafka</code> - Infers that Kafka might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferredWorkloadTypesWithStrings(String... inferredWorkloadTypes);

        /**
         * <p>
         * The applications that might be running on the instances in the Auto Scaling group as inferred by Compute
         * Optimizer.
         * </p>
         * <p>
         * Compute Optimizer can infer if one of the following applications might be running on the instances:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Memcached</code> - Infers that Memcached might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NGINX</code> - Infers that NGINX might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Redis</code> - Infers that Redis might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Kafka</code> - Infers that Kafka might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         * </p>
         * </li>
         * </ul>
         * 
         * @param inferredWorkloadTypes
         *        The applications that might be running on the instances in the Auto Scaling group as inferred by
         *        Compute Optimizer.</p>
         *        <p>
         *        Compute Optimizer can infer if one of the following applications might be running on the instances:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Memcached</code> - Infers that Memcached might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NGINX</code> - Infers that NGINX might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Redis</code> - Infers that Redis might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Kafka</code> - Infers that Kafka might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferredWorkloadTypes(Collection<InferredWorkloadType> inferredWorkloadTypes);

        /**
         * <p>
         * The applications that might be running on the instances in the Auto Scaling group as inferred by Compute
         * Optimizer.
         * </p>
         * <p>
         * Compute Optimizer can infer if one of the following applications might be running on the instances:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Memcached</code> - Infers that Memcached might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NGINX</code> - Infers that NGINX might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Redis</code> - Infers that Redis might be running on the instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Kafka</code> - Infers that Kafka might be running on the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         * </p>
         * </li>
         * </ul>
         * 
         * @param inferredWorkloadTypes
         *        The applications that might be running on the instances in the Auto Scaling group as inferred by
         *        Compute Optimizer.</p>
         *        <p>
         *        Compute Optimizer can infer if one of the following applications might be running on the instances:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AmazonEmr</code> - Infers that Amazon EMR might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheCassandra</code> - Infers that Apache Cassandra might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ApacheHadoop</code> - Infers that Apache Hadoop might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Memcached</code> - Infers that Memcached might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NGINX</code> - Infers that NGINX might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PostgreSql</code> - Infers that PostgreSQL might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Redis</code> - Infers that Redis might be running on the instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Kafka</code> - Infers that Kafka might be running on the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SQLServer</code> - Infers that SQLServer might be running on the instance.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferredWorkloadTypes(InferredWorkloadType... inferredWorkloadTypes);

        /**
         * <p>
         * Describes the GPU accelerator settings for the current instance type of the Auto Scaling group.
         * </p>
         * 
         * @param currentInstanceGpuInfo
         *        Describes the GPU accelerator settings for the current instance type of the Auto Scaling group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentInstanceGpuInfo(GpuInfo currentInstanceGpuInfo);

        /**
         * <p>
         * Describes the GPU accelerator settings for the current instance type of the Auto Scaling group.
         * </p>
         * This is a convenience method that creates an instance of the {@link GpuInfo.Builder} avoiding the need to
         * create one manually via {@link GpuInfo#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link GpuInfo.Builder#build()} is called immediately and its result is
         * passed to {@link #currentInstanceGpuInfo(GpuInfo)}.
         * 
         * @param currentInstanceGpuInfo
         *        a consumer that will call methods on {@link GpuInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #currentInstanceGpuInfo(GpuInfo)
         */
        default Builder currentInstanceGpuInfo(Consumer<GpuInfo.Builder> currentInstanceGpuInfo) {
            return currentInstanceGpuInfo(GpuInfo.builder().applyMutation(currentInstanceGpuInfo).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String accountId;

        private String autoScalingGroupArn;

        private String autoScalingGroupName;

        private String finding;

        private List<UtilizationMetric> utilizationMetrics = DefaultSdkAutoConstructList.getInstance();

        private Double lookBackPeriodInDays;

        private AutoScalingGroupConfiguration currentConfiguration;

        private List<AutoScalingGroupRecommendationOption> recommendationOptions = DefaultSdkAutoConstructList.getInstance();

        private Instant lastRefreshTimestamp;

        private String currentPerformanceRisk;

        private EffectiveRecommendationPreferences effectiveRecommendationPreferences;

        private List<String> inferredWorkloadTypes = DefaultSdkAutoConstructList.getInstance();

        private GpuInfo currentInstanceGpuInfo;

        private BuilderImpl() {
        }

        private BuilderImpl(AutoScalingGroupRecommendation model) {
            accountId(model.accountId);
            autoScalingGroupArn(model.autoScalingGroupArn);
            autoScalingGroupName(model.autoScalingGroupName);
            finding(model.finding);
            utilizationMetrics(model.utilizationMetrics);
            lookBackPeriodInDays(model.lookBackPeriodInDays);
            currentConfiguration(model.currentConfiguration);
            recommendationOptions(model.recommendationOptions);
            lastRefreshTimestamp(model.lastRefreshTimestamp);
            currentPerformanceRisk(model.currentPerformanceRisk);
            effectiveRecommendationPreferences(model.effectiveRecommendationPreferences);
            inferredWorkloadTypesWithStrings(model.inferredWorkloadTypes);
            currentInstanceGpuInfo(model.currentInstanceGpuInfo);
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getAutoScalingGroupArn() {
            return autoScalingGroupArn;
        }

        public final void setAutoScalingGroupArn(String autoScalingGroupArn) {
            this.autoScalingGroupArn = autoScalingGroupArn;
        }

        @Override
        public final Builder autoScalingGroupArn(String autoScalingGroupArn) {
            this.autoScalingGroupArn = autoScalingGroupArn;
            return this;
        }

        public final String getAutoScalingGroupName() {
            return autoScalingGroupName;
        }

        public final void setAutoScalingGroupName(String autoScalingGroupName) {
            this.autoScalingGroupName = autoScalingGroupName;
        }

        @Override
        public final Builder autoScalingGroupName(String autoScalingGroupName) {
            this.autoScalingGroupName = autoScalingGroupName;
            return this;
        }

        public final String getFinding() {
            return finding;
        }

        public final void setFinding(String finding) {
            this.finding = finding;
        }

        @Override
        public final Builder finding(String finding) {
            this.finding = finding;
            return this;
        }

        @Override
        public final Builder finding(Finding finding) {
            this.finding(finding == null ? null : finding.toString());
            return this;
        }

        public final List<UtilizationMetric.Builder> getUtilizationMetrics() {
            List<UtilizationMetric.Builder> result = UtilizationMetricsCopier.copyToBuilder(this.utilizationMetrics);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUtilizationMetrics(Collection<UtilizationMetric.BuilderImpl> utilizationMetrics) {
            this.utilizationMetrics = UtilizationMetricsCopier.copyFromBuilder(utilizationMetrics);
        }

        @Override
        public final Builder utilizationMetrics(Collection<UtilizationMetric> utilizationMetrics) {
            this.utilizationMetrics = UtilizationMetricsCopier.copy(utilizationMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(UtilizationMetric... utilizationMetrics) {
            utilizationMetrics(Arrays.asList(utilizationMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(Consumer<UtilizationMetric.Builder>... utilizationMetrics) {
            utilizationMetrics(Stream.of(utilizationMetrics).map(c -> UtilizationMetric.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Double getLookBackPeriodInDays() {
            return lookBackPeriodInDays;
        }

        public final void setLookBackPeriodInDays(Double lookBackPeriodInDays) {
            this.lookBackPeriodInDays = lookBackPeriodInDays;
        }

        @Override
        public final Builder lookBackPeriodInDays(Double lookBackPeriodInDays) {
            this.lookBackPeriodInDays = lookBackPeriodInDays;
            return this;
        }

        public final AutoScalingGroupConfiguration.Builder getCurrentConfiguration() {
            return currentConfiguration != null ? currentConfiguration.toBuilder() : null;
        }

        public final void setCurrentConfiguration(AutoScalingGroupConfiguration.BuilderImpl currentConfiguration) {
            this.currentConfiguration = currentConfiguration != null ? currentConfiguration.build() : null;
        }

        @Override
        public final Builder currentConfiguration(AutoScalingGroupConfiguration currentConfiguration) {
            this.currentConfiguration = currentConfiguration;
            return this;
        }

        public final List<AutoScalingGroupRecommendationOption.Builder> getRecommendationOptions() {
            List<AutoScalingGroupRecommendationOption.Builder> result = AutoScalingGroupRecommendationOptionsCopier
                    .copyToBuilder(this.recommendationOptions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRecommendationOptions(
                Collection<AutoScalingGroupRecommendationOption.BuilderImpl> recommendationOptions) {
            this.recommendationOptions = AutoScalingGroupRecommendationOptionsCopier.copyFromBuilder(recommendationOptions);
        }

        @Override
        public final Builder recommendationOptions(Collection<AutoScalingGroupRecommendationOption> recommendationOptions) {
            this.recommendationOptions = AutoScalingGroupRecommendationOptionsCopier.copy(recommendationOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendationOptions(AutoScalingGroupRecommendationOption... recommendationOptions) {
            recommendationOptions(Arrays.asList(recommendationOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendationOptions(
                Consumer<AutoScalingGroupRecommendationOption.Builder>... recommendationOptions) {
            recommendationOptions(Stream.of(recommendationOptions)
                    .map(c -> AutoScalingGroupRecommendationOption.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Instant getLastRefreshTimestamp() {
            return lastRefreshTimestamp;
        }

        public final void setLastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
        }

        @Override
        public final Builder lastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
            return this;
        }

        public final String getCurrentPerformanceRisk() {
            return currentPerformanceRisk;
        }

        public final void setCurrentPerformanceRisk(String currentPerformanceRisk) {
            this.currentPerformanceRisk = currentPerformanceRisk;
        }

        @Override
        public final Builder currentPerformanceRisk(String currentPerformanceRisk) {
            this.currentPerformanceRisk = currentPerformanceRisk;
            return this;
        }

        @Override
        public final Builder currentPerformanceRisk(CurrentPerformanceRisk currentPerformanceRisk) {
            this.currentPerformanceRisk(currentPerformanceRisk == null ? null : currentPerformanceRisk.toString());
            return this;
        }

        public final EffectiveRecommendationPreferences.Builder getEffectiveRecommendationPreferences() {
            return effectiveRecommendationPreferences != null ? effectiveRecommendationPreferences.toBuilder() : null;
        }

        public final void setEffectiveRecommendationPreferences(
                EffectiveRecommendationPreferences.BuilderImpl effectiveRecommendationPreferences) {
            this.effectiveRecommendationPreferences = effectiveRecommendationPreferences != null ? effectiveRecommendationPreferences
                    .build() : null;
        }

        @Override
        public final Builder effectiveRecommendationPreferences(
                EffectiveRecommendationPreferences effectiveRecommendationPreferences) {
            this.effectiveRecommendationPreferences = effectiveRecommendationPreferences;
            return this;
        }

        public final Collection<String> getInferredWorkloadTypes() {
            if (inferredWorkloadTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return inferredWorkloadTypes;
        }

        public final void setInferredWorkloadTypes(Collection<String> inferredWorkloadTypes) {
            this.inferredWorkloadTypes = InferredWorkloadTypesCopier.copy(inferredWorkloadTypes);
        }

        @Override
        public final Builder inferredWorkloadTypesWithStrings(Collection<String> inferredWorkloadTypes) {
            this.inferredWorkloadTypes = InferredWorkloadTypesCopier.copy(inferredWorkloadTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inferredWorkloadTypesWithStrings(String... inferredWorkloadTypes) {
            inferredWorkloadTypesWithStrings(Arrays.asList(inferredWorkloadTypes));
            return this;
        }

        @Override
        public final Builder inferredWorkloadTypes(Collection<InferredWorkloadType> inferredWorkloadTypes) {
            this.inferredWorkloadTypes = InferredWorkloadTypesCopier.copyEnumToString(inferredWorkloadTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder inferredWorkloadTypes(InferredWorkloadType... inferredWorkloadTypes) {
            inferredWorkloadTypes(Arrays.asList(inferredWorkloadTypes));
            return this;
        }

        public final GpuInfo.Builder getCurrentInstanceGpuInfo() {
            return currentInstanceGpuInfo != null ? currentInstanceGpuInfo.toBuilder() : null;
        }

        public final void setCurrentInstanceGpuInfo(GpuInfo.BuilderImpl currentInstanceGpuInfo) {
            this.currentInstanceGpuInfo = currentInstanceGpuInfo != null ? currentInstanceGpuInfo.build() : null;
        }

        @Override
        public final Builder currentInstanceGpuInfo(GpuInfo currentInstanceGpuInfo) {
            this.currentInstanceGpuInfo = currentInstanceGpuInfo;
            return this;
        }

        @Override
        public AutoScalingGroupRecommendation build() {
            return new AutoScalingGroupRecommendation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
