/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.comprehend.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information about a document classifier.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DocumentClassifierProperties implements SdkPojo, Serializable,
        ToCopyableBuilder<DocumentClassifierProperties.Builder, DocumentClassifierProperties> {
    private static final SdkField<String> DOCUMENT_CLASSIFIER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DocumentClassifierArn").getter(getter(DocumentClassifierProperties::documentClassifierArn))
            .setter(setter(Builder::documentClassifierArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DocumentClassifierArn").build())
            .build();

    private static final SdkField<String> LANGUAGE_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LanguageCode").getter(getter(DocumentClassifierProperties::languageCodeAsString))
            .setter(setter(Builder::languageCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LanguageCode").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(DocumentClassifierProperties::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Message")
            .getter(getter(DocumentClassifierProperties::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Message").build()).build();

    private static final SdkField<Instant> SUBMIT_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("SubmitTime").getter(getter(DocumentClassifierProperties::submitTime))
            .setter(setter(Builder::submitTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubmitTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTime").getter(getter(DocumentClassifierProperties::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build()).build();

    private static final SdkField<Instant> TRAINING_START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("TrainingStartTime").getter(getter(DocumentClassifierProperties::trainingStartTime))
            .setter(setter(Builder::trainingStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingStartTime").build()).build();

    private static final SdkField<Instant> TRAINING_END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("TrainingEndTime").getter(getter(DocumentClassifierProperties::trainingEndTime))
            .setter(setter(Builder::trainingEndTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TrainingEndTime").build()).build();

    private static final SdkField<DocumentClassifierInputDataConfig> INPUT_DATA_CONFIG_FIELD = SdkField
            .<DocumentClassifierInputDataConfig> builder(MarshallingType.SDK_POJO).memberName("InputDataConfig")
            .getter(getter(DocumentClassifierProperties::inputDataConfig)).setter(setter(Builder::inputDataConfig))
            .constructor(DocumentClassifierInputDataConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputDataConfig").build()).build();

    private static final SdkField<DocumentClassifierOutputDataConfig> OUTPUT_DATA_CONFIG_FIELD = SdkField
            .<DocumentClassifierOutputDataConfig> builder(MarshallingType.SDK_POJO).memberName("OutputDataConfig")
            .getter(getter(DocumentClassifierProperties::outputDataConfig)).setter(setter(Builder::outputDataConfig))
            .constructor(DocumentClassifierOutputDataConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutputDataConfig").build()).build();

    private static final SdkField<ClassifierMetadata> CLASSIFIER_METADATA_FIELD = SdkField
            .<ClassifierMetadata> builder(MarshallingType.SDK_POJO).memberName("ClassifierMetadata")
            .getter(getter(DocumentClassifierProperties::classifierMetadata)).setter(setter(Builder::classifierMetadata))
            .constructor(ClassifierMetadata::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClassifierMetadata").build())
            .build();

    private static final SdkField<String> DATA_ACCESS_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataAccessRoleArn").getter(getter(DocumentClassifierProperties::dataAccessRoleArn))
            .setter(setter(Builder::dataAccessRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataAccessRoleArn").build()).build();

    private static final SdkField<String> VOLUME_KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VolumeKmsKeyId").getter(getter(DocumentClassifierProperties::volumeKmsKeyId))
            .setter(setter(Builder::volumeKmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeKmsKeyId").build()).build();

    private static final SdkField<VpcConfig> VPC_CONFIG_FIELD = SdkField.<VpcConfig> builder(MarshallingType.SDK_POJO)
            .memberName("VpcConfig").getter(getter(DocumentClassifierProperties::vpcConfig)).setter(setter(Builder::vpcConfig))
            .constructor(VpcConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcConfig").build()).build();

    private static final SdkField<String> MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Mode")
            .getter(getter(DocumentClassifierProperties::modeAsString)).setter(setter(Builder::mode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Mode").build()).build();

    private static final SdkField<String> MODEL_KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ModelKmsKeyId").getter(getter(DocumentClassifierProperties::modelKmsKeyId))
            .setter(setter(Builder::modelKmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ModelKmsKeyId").build()).build();

    private static final SdkField<String> VERSION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VersionName").getter(getter(DocumentClassifierProperties::versionName))
            .setter(setter(Builder::versionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionName").build()).build();

    private static final SdkField<String> SOURCE_MODEL_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceModelArn").getter(getter(DocumentClassifierProperties::sourceModelArn))
            .setter(setter(Builder::sourceModelArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceModelArn").build()).build();

    private static final SdkField<String> FLYWHEEL_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FlywheelArn").getter(getter(DocumentClassifierProperties::flywheelArn))
            .setter(setter(Builder::flywheelArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FlywheelArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOCUMENT_CLASSIFIER_ARN_FIELD,
            LANGUAGE_CODE_FIELD, STATUS_FIELD, MESSAGE_FIELD, SUBMIT_TIME_FIELD, END_TIME_FIELD, TRAINING_START_TIME_FIELD,
            TRAINING_END_TIME_FIELD, INPUT_DATA_CONFIG_FIELD, OUTPUT_DATA_CONFIG_FIELD, CLASSIFIER_METADATA_FIELD,
            DATA_ACCESS_ROLE_ARN_FIELD, VOLUME_KMS_KEY_ID_FIELD, VPC_CONFIG_FIELD, MODE_FIELD, MODEL_KMS_KEY_ID_FIELD,
            VERSION_NAME_FIELD, SOURCE_MODEL_ARN_FIELD, FLYWHEEL_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String documentClassifierArn;

    private final String languageCode;

    private final String status;

    private final String message;

    private final Instant submitTime;

    private final Instant endTime;

    private final Instant trainingStartTime;

    private final Instant trainingEndTime;

    private final DocumentClassifierInputDataConfig inputDataConfig;

    private final DocumentClassifierOutputDataConfig outputDataConfig;

    private final ClassifierMetadata classifierMetadata;

    private final String dataAccessRoleArn;

    private final String volumeKmsKeyId;

    private final VpcConfig vpcConfig;

    private final String mode;

    private final String modelKmsKeyId;

    private final String versionName;

    private final String sourceModelArn;

    private final String flywheelArn;

    private DocumentClassifierProperties(BuilderImpl builder) {
        this.documentClassifierArn = builder.documentClassifierArn;
        this.languageCode = builder.languageCode;
        this.status = builder.status;
        this.message = builder.message;
        this.submitTime = builder.submitTime;
        this.endTime = builder.endTime;
        this.trainingStartTime = builder.trainingStartTime;
        this.trainingEndTime = builder.trainingEndTime;
        this.inputDataConfig = builder.inputDataConfig;
        this.outputDataConfig = builder.outputDataConfig;
        this.classifierMetadata = builder.classifierMetadata;
        this.dataAccessRoleArn = builder.dataAccessRoleArn;
        this.volumeKmsKeyId = builder.volumeKmsKeyId;
        this.vpcConfig = builder.vpcConfig;
        this.mode = builder.mode;
        this.modelKmsKeyId = builder.modelKmsKeyId;
        this.versionName = builder.versionName;
        this.sourceModelArn = builder.sourceModelArn;
        this.flywheelArn = builder.flywheelArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) that identifies the document classifier.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) that identifies the document classifier.
     */
    public final String documentClassifierArn() {
        return documentClassifierArn;
    }

    /**
     * <p>
     * The language code for the language of the documents that the classifier was trained on.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code for the language of the documents that the classifier was trained on.
     * @see LanguageCode
     */
    public final LanguageCode languageCode() {
        return LanguageCode.fromValue(languageCode);
    }

    /**
     * <p>
     * The language code for the language of the documents that the classifier was trained on.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #languageCode} will
     * return {@link LanguageCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #languageCodeAsString}.
     * </p>
     * 
     * @return The language code for the language of the documents that the classifier was trained on.
     * @see LanguageCode
     */
    public final String languageCodeAsString() {
        return languageCode;
    }

    /**
     * <p>
     * The status of the document classifier. If the status is <code>TRAINED</code> the classifier is ready to use. If
     * the status is <code>TRAINED_WITH_WARNINGS</code> the classifier training succeeded, but you should review the
     * warnings returned in the <code>CreateDocumentClassifier</code> response.
     * </p>
     * <p>
     * If the status is <code>FAILED</code> you can see additional information about why the classifier wasn't trained
     * in the <code>Message</code> field.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ModelStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the document classifier. If the status is <code>TRAINED</code> the classifier is ready to
     *         use. If the status is <code>TRAINED_WITH_WARNINGS</code> the classifier training succeeded, but you
     *         should review the warnings returned in the <code>CreateDocumentClassifier</code> response.</p>
     *         <p>
     *         If the status is <code>FAILED</code> you can see additional information about why the classifier wasn't
     *         trained in the <code>Message</code> field.
     * @see ModelStatus
     */
    public final ModelStatus status() {
        return ModelStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the document classifier. If the status is <code>TRAINED</code> the classifier is ready to use. If
     * the status is <code>TRAINED_WITH_WARNINGS</code> the classifier training succeeded, but you should review the
     * warnings returned in the <code>CreateDocumentClassifier</code> response.
     * </p>
     * <p>
     * If the status is <code>FAILED</code> you can see additional information about why the classifier wasn't trained
     * in the <code>Message</code> field.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ModelStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the document classifier. If the status is <code>TRAINED</code> the classifier is ready to
     *         use. If the status is <code>TRAINED_WITH_WARNINGS</code> the classifier training succeeded, but you
     *         should review the warnings returned in the <code>CreateDocumentClassifier</code> response.</p>
     *         <p>
     *         If the status is <code>FAILED</code> you can see additional information about why the classifier wasn't
     *         trained in the <code>Message</code> field.
     * @see ModelStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Additional information about the status of the classifier.
     * </p>
     * 
     * @return Additional information about the status of the classifier.
     */
    public final String message() {
        return message;
    }

    /**
     * <p>
     * The time that the document classifier was submitted for training.
     * </p>
     * 
     * @return The time that the document classifier was submitted for training.
     */
    public final Instant submitTime() {
        return submitTime;
    }

    /**
     * <p>
     * The time that training the document classifier completed.
     * </p>
     * 
     * @return The time that training the document classifier completed.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * Indicates the time when the training starts on documentation classifiers. You are billed for the time interval
     * between this time and the value of TrainingEndTime.
     * </p>
     * 
     * @return Indicates the time when the training starts on documentation classifiers. You are billed for the time
     *         interval between this time and the value of TrainingEndTime.
     */
    public final Instant trainingStartTime() {
        return trainingStartTime;
    }

    /**
     * <p>
     * The time that training of the document classifier was completed. Indicates the time when the training completes
     * on documentation classifiers. You are billed for the time interval between this time and the value of
     * TrainingStartTime.
     * </p>
     * 
     * @return The time that training of the document classifier was completed. Indicates the time when the training
     *         completes on documentation classifiers. You are billed for the time interval between this time and the
     *         value of TrainingStartTime.
     */
    public final Instant trainingEndTime() {
        return trainingEndTime;
    }

    /**
     * <p>
     * The input data configuration that you supplied when you created the document classifier for training.
     * </p>
     * 
     * @return The input data configuration that you supplied when you created the document classifier for training.
     */
    public final DocumentClassifierInputDataConfig inputDataConfig() {
        return inputDataConfig;
    }

    /**
     * <p>
     * Provides output results configuration parameters for custom classifier jobs.
     * </p>
     * 
     * @return Provides output results configuration parameters for custom classifier jobs.
     */
    public final DocumentClassifierOutputDataConfig outputDataConfig() {
        return outputDataConfig;
    }

    /**
     * <p>
     * Information about the document classifier, including the number of documents used for training the classifier,
     * the number of documents used for test the classifier, and an accuracy rating.
     * </p>
     * 
     * @return Information about the document classifier, including the number of documents used for training the
     *         classifier, the number of documents used for test the classifier, and an accuracy rating.
     */
    public final ClassifierMetadata classifierMetadata() {
        return classifierMetadata;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IAM role that grants Amazon Comprehend read access to your input data.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IAM role that grants Amazon Comprehend read access to your input
     *         data.
     */
    public final String dataAccessRoleArn() {
        return dataAccessRoleArn;
    }

    /**
     * <p>
     * ID for the Amazon Web Services Key Management Service (KMS) key that Amazon Comprehend uses to encrypt data on
     * the storage volume attached to the ML compute instance(s) that process the analysis job. The VolumeKmsKeyId can
     * be either of the following formats:
     * </p>
     * <ul>
     * <li>
     * <p>
     * KMS Key ID: <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Resource Name (ARN) of a KMS Key:
     * <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return ID for the Amazon Web Services Key Management Service (KMS) key that Amazon Comprehend uses to encrypt
     *         data on the storage volume attached to the ML compute instance(s) that process the analysis job. The
     *         VolumeKmsKeyId can be either of the following formats:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         KMS Key ID: <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Resource Name (ARN) of a KMS Key:
     *         <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     *         </p>
     *         </li>
     */
    public final String volumeKmsKeyId() {
        return volumeKmsKeyId;
    }

    /**
     * <p>
     * Configuration parameters for a private Virtual Private Cloud (VPC) containing the resources you are using for
     * your custom classifier. For more information, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/userguide/what-is-amazon-vpc.html">Amazon VPC</a>.
     * </p>
     * 
     * @return Configuration parameters for a private Virtual Private Cloud (VPC) containing the resources you are using
     *         for your custom classifier. For more information, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/userguide/what-is-amazon-vpc.html">Amazon VPC</a>.
     */
    public final VpcConfig vpcConfig() {
        return vpcConfig;
    }

    /**
     * <p>
     * Indicates the mode in which the specific classifier was trained. This also indicates the format of input
     * documents and the format of the confusion matrix. Each classifier can only be trained in one mode and this cannot
     * be changed once the classifier is trained.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link DocumentClassifierMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return Indicates the mode in which the specific classifier was trained. This also indicates the format of input
     *         documents and the format of the confusion matrix. Each classifier can only be trained in one mode and
     *         this cannot be changed once the classifier is trained.
     * @see DocumentClassifierMode
     */
    public final DocumentClassifierMode mode() {
        return DocumentClassifierMode.fromValue(mode);
    }

    /**
     * <p>
     * Indicates the mode in which the specific classifier was trained. This also indicates the format of input
     * documents and the format of the confusion matrix. Each classifier can only be trained in one mode and this cannot
     * be changed once the classifier is trained.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #mode} will return
     * {@link DocumentClassifierMode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #modeAsString}.
     * </p>
     * 
     * @return Indicates the mode in which the specific classifier was trained. This also indicates the format of input
     *         documents and the format of the confusion matrix. Each classifier can only be trained in one mode and
     *         this cannot be changed once the classifier is trained.
     * @see DocumentClassifierMode
     */
    public final String modeAsString() {
        return mode;
    }

    /**
     * <p>
     * ID for the KMS key that Amazon Comprehend uses to encrypt trained custom models. The ModelKmsKeyId can be either
     * of the following formats:
     * </p>
     * <ul>
     * <li>
     * <p>
     * KMS Key ID: <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Resource Name (ARN) of a KMS Key:
     * <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return ID for the KMS key that Amazon Comprehend uses to encrypt trained custom models. The ModelKmsKeyId can be
     *         either of the following formats:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         KMS Key ID: <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Resource Name (ARN) of a KMS Key:
     *         <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
     *         </p>
     *         </li>
     */
    public final String modelKmsKeyId() {
        return modelKmsKeyId;
    }

    /**
     * <p>
     * The version name that you assigned to the document classifier.
     * </p>
     * 
     * @return The version name that you assigned to the document classifier.
     */
    public final String versionName() {
        return versionName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the source model. This model was imported from a different Amazon Web Services
     * account to create the document classifier model in your Amazon Web Services account.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the source model. This model was imported from a different Amazon Web
     *         Services account to create the document classifier model in your Amazon Web Services account.
     */
    public final String sourceModelArn() {
        return sourceModelArn;
    }

    /**
     * <p>
     * The Amazon Resource Number (ARN) of the flywheel
     * </p>
     * 
     * @return The Amazon Resource Number (ARN) of the flywheel
     */
    public final String flywheelArn() {
        return flywheelArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(documentClassifierArn());
        hashCode = 31 * hashCode + Objects.hashCode(languageCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(submitTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(trainingStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(trainingEndTime());
        hashCode = 31 * hashCode + Objects.hashCode(inputDataConfig());
        hashCode = 31 * hashCode + Objects.hashCode(outputDataConfig());
        hashCode = 31 * hashCode + Objects.hashCode(classifierMetadata());
        hashCode = 31 * hashCode + Objects.hashCode(dataAccessRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(volumeKmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(vpcConfig());
        hashCode = 31 * hashCode + Objects.hashCode(modeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(modelKmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(versionName());
        hashCode = 31 * hashCode + Objects.hashCode(sourceModelArn());
        hashCode = 31 * hashCode + Objects.hashCode(flywheelArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DocumentClassifierProperties)) {
            return false;
        }
        DocumentClassifierProperties other = (DocumentClassifierProperties) obj;
        return Objects.equals(documentClassifierArn(), other.documentClassifierArn())
                && Objects.equals(languageCodeAsString(), other.languageCodeAsString())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(message(), other.message())
                && Objects.equals(submitTime(), other.submitTime()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(trainingStartTime(), other.trainingStartTime())
                && Objects.equals(trainingEndTime(), other.trainingEndTime())
                && Objects.equals(inputDataConfig(), other.inputDataConfig())
                && Objects.equals(outputDataConfig(), other.outputDataConfig())
                && Objects.equals(classifierMetadata(), other.classifierMetadata())
                && Objects.equals(dataAccessRoleArn(), other.dataAccessRoleArn())
                && Objects.equals(volumeKmsKeyId(), other.volumeKmsKeyId()) && Objects.equals(vpcConfig(), other.vpcConfig())
                && Objects.equals(modeAsString(), other.modeAsString()) && Objects.equals(modelKmsKeyId(), other.modelKmsKeyId())
                && Objects.equals(versionName(), other.versionName()) && Objects.equals(sourceModelArn(), other.sourceModelArn())
                && Objects.equals(flywheelArn(), other.flywheelArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DocumentClassifierProperties").add("DocumentClassifierArn", documentClassifierArn())
                .add("LanguageCode", languageCodeAsString()).add("Status", statusAsString()).add("Message", message())
                .add("SubmitTime", submitTime()).add("EndTime", endTime()).add("TrainingStartTime", trainingStartTime())
                .add("TrainingEndTime", trainingEndTime()).add("InputDataConfig", inputDataConfig())
                .add("OutputDataConfig", outputDataConfig())
                .add("ClassifierMetadata", classifierMetadata() == null ? null : "*** Sensitive Data Redacted ***")
                .add("DataAccessRoleArn", dataAccessRoleArn()).add("VolumeKmsKeyId", volumeKmsKeyId())
                .add("VpcConfig", vpcConfig()).add("Mode", modeAsString()).add("ModelKmsKeyId", modelKmsKeyId())
                .add("VersionName", versionName()).add("SourceModelArn", sourceModelArn()).add("FlywheelArn", flywheelArn())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DocumentClassifierArn":
            return Optional.ofNullable(clazz.cast(documentClassifierArn()));
        case "LanguageCode":
            return Optional.ofNullable(clazz.cast(languageCodeAsString()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "Message":
            return Optional.ofNullable(clazz.cast(message()));
        case "SubmitTime":
            return Optional.ofNullable(clazz.cast(submitTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "TrainingStartTime":
            return Optional.ofNullable(clazz.cast(trainingStartTime()));
        case "TrainingEndTime":
            return Optional.ofNullable(clazz.cast(trainingEndTime()));
        case "InputDataConfig":
            return Optional.ofNullable(clazz.cast(inputDataConfig()));
        case "OutputDataConfig":
            return Optional.ofNullable(clazz.cast(outputDataConfig()));
        case "ClassifierMetadata":
            return Optional.ofNullable(clazz.cast(classifierMetadata()));
        case "DataAccessRoleArn":
            return Optional.ofNullable(clazz.cast(dataAccessRoleArn()));
        case "VolumeKmsKeyId":
            return Optional.ofNullable(clazz.cast(volumeKmsKeyId()));
        case "VpcConfig":
            return Optional.ofNullable(clazz.cast(vpcConfig()));
        case "Mode":
            return Optional.ofNullable(clazz.cast(modeAsString()));
        case "ModelKmsKeyId":
            return Optional.ofNullable(clazz.cast(modelKmsKeyId()));
        case "VersionName":
            return Optional.ofNullable(clazz.cast(versionName()));
        case "SourceModelArn":
            return Optional.ofNullable(clazz.cast(sourceModelArn()));
        case "FlywheelArn":
            return Optional.ofNullable(clazz.cast(flywheelArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DocumentClassifierProperties, T> g) {
        return obj -> g.apply((DocumentClassifierProperties) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DocumentClassifierProperties> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) that identifies the document classifier.
         * </p>
         * 
         * @param documentClassifierArn
         *        The Amazon Resource Name (ARN) that identifies the document classifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder documentClassifierArn(String documentClassifierArn);

        /**
         * <p>
         * The language code for the language of the documents that the classifier was trained on.
         * </p>
         * 
         * @param languageCode
         *        The language code for the language of the documents that the classifier was trained on.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(String languageCode);

        /**
         * <p>
         * The language code for the language of the documents that the classifier was trained on.
         * </p>
         * 
         * @param languageCode
         *        The language code for the language of the documents that the classifier was trained on.
         * @see LanguageCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LanguageCode
         */
        Builder languageCode(LanguageCode languageCode);

        /**
         * <p>
         * The status of the document classifier. If the status is <code>TRAINED</code> the classifier is ready to use.
         * If the status is <code>TRAINED_WITH_WARNINGS</code> the classifier training succeeded, but you should review
         * the warnings returned in the <code>CreateDocumentClassifier</code> response.
         * </p>
         * <p>
         * If the status is <code>FAILED</code> you can see additional information about why the classifier wasn't
         * trained in the <code>Message</code> field.
         * </p>
         * 
         * @param status
         *        The status of the document classifier. If the status is <code>TRAINED</code> the classifier is ready
         *        to use. If the status is <code>TRAINED_WITH_WARNINGS</code> the classifier training succeeded, but you
         *        should review the warnings returned in the <code>CreateDocumentClassifier</code> response.</p>
         *        <p>
         *        If the status is <code>FAILED</code> you can see additional information about why the classifier
         *        wasn't trained in the <code>Message</code> field.
         * @see ModelStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ModelStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the document classifier. If the status is <code>TRAINED</code> the classifier is ready to use.
         * If the status is <code>TRAINED_WITH_WARNINGS</code> the classifier training succeeded, but you should review
         * the warnings returned in the <code>CreateDocumentClassifier</code> response.
         * </p>
         * <p>
         * If the status is <code>FAILED</code> you can see additional information about why the classifier wasn't
         * trained in the <code>Message</code> field.
         * </p>
         * 
         * @param status
         *        The status of the document classifier. If the status is <code>TRAINED</code> the classifier is ready
         *        to use. If the status is <code>TRAINED_WITH_WARNINGS</code> the classifier training succeeded, but you
         *        should review the warnings returned in the <code>CreateDocumentClassifier</code> response.</p>
         *        <p>
         *        If the status is <code>FAILED</code> you can see additional information about why the classifier
         *        wasn't trained in the <code>Message</code> field.
         * @see ModelStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ModelStatus
         */
        Builder status(ModelStatus status);

        /**
         * <p>
         * Additional information about the status of the classifier.
         * </p>
         * 
         * @param message
         *        Additional information about the status of the classifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * The time that the document classifier was submitted for training.
         * </p>
         * 
         * @param submitTime
         *        The time that the document classifier was submitted for training.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder submitTime(Instant submitTime);

        /**
         * <p>
         * The time that training the document classifier completed.
         * </p>
         * 
         * @param endTime
         *        The time that training the document classifier completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * Indicates the time when the training starts on documentation classifiers. You are billed for the time
         * interval between this time and the value of TrainingEndTime.
         * </p>
         * 
         * @param trainingStartTime
         *        Indicates the time when the training starts on documentation classifiers. You are billed for the time
         *        interval between this time and the value of TrainingEndTime.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingStartTime(Instant trainingStartTime);

        /**
         * <p>
         * The time that training of the document classifier was completed. Indicates the time when the training
         * completes on documentation classifiers. You are billed for the time interval between this time and the value
         * of TrainingStartTime.
         * </p>
         * 
         * @param trainingEndTime
         *        The time that training of the document classifier was completed. Indicates the time when the training
         *        completes on documentation classifiers. You are billed for the time interval between this time and the
         *        value of TrainingStartTime.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder trainingEndTime(Instant trainingEndTime);

        /**
         * <p>
         * The input data configuration that you supplied when you created the document classifier for training.
         * </p>
         * 
         * @param inputDataConfig
         *        The input data configuration that you supplied when you created the document classifier for training.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputDataConfig(DocumentClassifierInputDataConfig inputDataConfig);

        /**
         * <p>
         * The input data configuration that you supplied when you created the document classifier for training.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link DocumentClassifierInputDataConfig.Builder} avoiding the need to create one manually via
         * {@link DocumentClassifierInputDataConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DocumentClassifierInputDataConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #inputDataConfig(DocumentClassifierInputDataConfig)}.
         * 
         * @param inputDataConfig
         *        a consumer that will call methods on {@link DocumentClassifierInputDataConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inputDataConfig(DocumentClassifierInputDataConfig)
         */
        default Builder inputDataConfig(Consumer<DocumentClassifierInputDataConfig.Builder> inputDataConfig) {
            return inputDataConfig(DocumentClassifierInputDataConfig.builder().applyMutation(inputDataConfig).build());
        }

        /**
         * <p>
         * Provides output results configuration parameters for custom classifier jobs.
         * </p>
         * 
         * @param outputDataConfig
         *        Provides output results configuration parameters for custom classifier jobs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outputDataConfig(DocumentClassifierOutputDataConfig outputDataConfig);

        /**
         * <p>
         * Provides output results configuration parameters for custom classifier jobs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link DocumentClassifierOutputDataConfig.Builder} avoiding the need to create one manually via
         * {@link DocumentClassifierOutputDataConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DocumentClassifierOutputDataConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #outputDataConfig(DocumentClassifierOutputDataConfig)}.
         * 
         * @param outputDataConfig
         *        a consumer that will call methods on {@link DocumentClassifierOutputDataConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #outputDataConfig(DocumentClassifierOutputDataConfig)
         */
        default Builder outputDataConfig(Consumer<DocumentClassifierOutputDataConfig.Builder> outputDataConfig) {
            return outputDataConfig(DocumentClassifierOutputDataConfig.builder().applyMutation(outputDataConfig).build());
        }

        /**
         * <p>
         * Information about the document classifier, including the number of documents used for training the
         * classifier, the number of documents used for test the classifier, and an accuracy rating.
         * </p>
         * 
         * @param classifierMetadata
         *        Information about the document classifier, including the number of documents used for training the
         *        classifier, the number of documents used for test the classifier, and an accuracy rating.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder classifierMetadata(ClassifierMetadata classifierMetadata);

        /**
         * <p>
         * Information about the document classifier, including the number of documents used for training the
         * classifier, the number of documents used for test the classifier, and an accuracy rating.
         * </p>
         * This is a convenience method that creates an instance of the {@link ClassifierMetadata.Builder} avoiding the
         * need to create one manually via {@link ClassifierMetadata#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ClassifierMetadata.Builder#build()} is called immediately and its
         * result is passed to {@link #classifierMetadata(ClassifierMetadata)}.
         * 
         * @param classifierMetadata
         *        a consumer that will call methods on {@link ClassifierMetadata.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #classifierMetadata(ClassifierMetadata)
         */
        default Builder classifierMetadata(Consumer<ClassifierMetadata.Builder> classifierMetadata) {
            return classifierMetadata(ClassifierMetadata.builder().applyMutation(classifierMetadata).build());
        }

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IAM role that grants Amazon Comprehend read access to your input data.
         * </p>
         * 
         * @param dataAccessRoleArn
         *        The Amazon Resource Name (ARN) of the IAM role that grants Amazon Comprehend read access to your input
         *        data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataAccessRoleArn(String dataAccessRoleArn);

        /**
         * <p>
         * ID for the Amazon Web Services Key Management Service (KMS) key that Amazon Comprehend uses to encrypt data
         * on the storage volume attached to the ML compute instance(s) that process the analysis job. The
         * VolumeKmsKeyId can be either of the following formats:
         * </p>
         * <ul>
         * <li>
         * <p>
         * KMS Key ID: <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Resource Name (ARN) of a KMS Key:
         * <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param volumeKmsKeyId
         *        ID for the Amazon Web Services Key Management Service (KMS) key that Amazon Comprehend uses to encrypt
         *        data on the storage volume attached to the ML compute instance(s) that process the analysis job. The
         *        VolumeKmsKeyId can be either of the following formats:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        KMS Key ID: <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Resource Name (ARN) of a KMS Key:
         *        <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeKmsKeyId(String volumeKmsKeyId);

        /**
         * <p>
         * Configuration parameters for a private Virtual Private Cloud (VPC) containing the resources you are using for
         * your custom classifier. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/what-is-amazon-vpc.html">Amazon VPC</a>.
         * </p>
         * 
         * @param vpcConfig
         *        Configuration parameters for a private Virtual Private Cloud (VPC) containing the resources you are
         *        using for your custom classifier. For more information, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/userguide/what-is-amazon-vpc.html">Amazon VPC</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConfig(VpcConfig vpcConfig);

        /**
         * <p>
         * Configuration parameters for a private Virtual Private Cloud (VPC) containing the resources you are using for
         * your custom classifier. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/userguide/what-is-amazon-vpc.html">Amazon VPC</a>.
         * </p>
         * This is a convenience method that creates an instance of the {@link VpcConfig.Builder} avoiding the need to
         * create one manually via {@link VpcConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VpcConfig.Builder#build()} is called immediately and its result
         * is passed to {@link #vpcConfig(VpcConfig)}.
         * 
         * @param vpcConfig
         *        a consumer that will call methods on {@link VpcConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcConfig(VpcConfig)
         */
        default Builder vpcConfig(Consumer<VpcConfig.Builder> vpcConfig) {
            return vpcConfig(VpcConfig.builder().applyMutation(vpcConfig).build());
        }

        /**
         * <p>
         * Indicates the mode in which the specific classifier was trained. This also indicates the format of input
         * documents and the format of the confusion matrix. Each classifier can only be trained in one mode and this
         * cannot be changed once the classifier is trained.
         * </p>
         * 
         * @param mode
         *        Indicates the mode in which the specific classifier was trained. This also indicates the format of
         *        input documents and the format of the confusion matrix. Each classifier can only be trained in one
         *        mode and this cannot be changed once the classifier is trained.
         * @see DocumentClassifierMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentClassifierMode
         */
        Builder mode(String mode);

        /**
         * <p>
         * Indicates the mode in which the specific classifier was trained. This also indicates the format of input
         * documents and the format of the confusion matrix. Each classifier can only be trained in one mode and this
         * cannot be changed once the classifier is trained.
         * </p>
         * 
         * @param mode
         *        Indicates the mode in which the specific classifier was trained. This also indicates the format of
         *        input documents and the format of the confusion matrix. Each classifier can only be trained in one
         *        mode and this cannot be changed once the classifier is trained.
         * @see DocumentClassifierMode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DocumentClassifierMode
         */
        Builder mode(DocumentClassifierMode mode);

        /**
         * <p>
         * ID for the KMS key that Amazon Comprehend uses to encrypt trained custom models. The ModelKmsKeyId can be
         * either of the following formats:
         * </p>
         * <ul>
         * <li>
         * <p>
         * KMS Key ID: <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Resource Name (ARN) of a KMS Key:
         * <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param modelKmsKeyId
         *        ID for the KMS key that Amazon Comprehend uses to encrypt trained custom models. The ModelKmsKeyId can
         *        be either of the following formats:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        KMS Key ID: <code>"1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Resource Name (ARN) of a KMS Key:
         *        <code>"arn:aws:kms:us-west-2:111122223333:key/1234abcd-12ab-34cd-56ef-1234567890ab"</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelKmsKeyId(String modelKmsKeyId);

        /**
         * <p>
         * The version name that you assigned to the document classifier.
         * </p>
         * 
         * @param versionName
         *        The version name that you assigned to the document classifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionName(String versionName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the source model. This model was imported from a different Amazon Web
         * Services account to create the document classifier model in your Amazon Web Services account.
         * </p>
         * 
         * @param sourceModelArn
         *        The Amazon Resource Name (ARN) of the source model. This model was imported from a different Amazon
         *        Web Services account to create the document classifier model in your Amazon Web Services account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceModelArn(String sourceModelArn);

        /**
         * <p>
         * The Amazon Resource Number (ARN) of the flywheel
         * </p>
         * 
         * @param flywheelArn
         *        The Amazon Resource Number (ARN) of the flywheel
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder flywheelArn(String flywheelArn);
    }

    static final class BuilderImpl implements Builder {
        private String documentClassifierArn;

        private String languageCode;

        private String status;

        private String message;

        private Instant submitTime;

        private Instant endTime;

        private Instant trainingStartTime;

        private Instant trainingEndTime;

        private DocumentClassifierInputDataConfig inputDataConfig;

        private DocumentClassifierOutputDataConfig outputDataConfig;

        private ClassifierMetadata classifierMetadata;

        private String dataAccessRoleArn;

        private String volumeKmsKeyId;

        private VpcConfig vpcConfig;

        private String mode;

        private String modelKmsKeyId;

        private String versionName;

        private String sourceModelArn;

        private String flywheelArn;

        private BuilderImpl() {
        }

        private BuilderImpl(DocumentClassifierProperties model) {
            documentClassifierArn(model.documentClassifierArn);
            languageCode(model.languageCode);
            status(model.status);
            message(model.message);
            submitTime(model.submitTime);
            endTime(model.endTime);
            trainingStartTime(model.trainingStartTime);
            trainingEndTime(model.trainingEndTime);
            inputDataConfig(model.inputDataConfig);
            outputDataConfig(model.outputDataConfig);
            classifierMetadata(model.classifierMetadata);
            dataAccessRoleArn(model.dataAccessRoleArn);
            volumeKmsKeyId(model.volumeKmsKeyId);
            vpcConfig(model.vpcConfig);
            mode(model.mode);
            modelKmsKeyId(model.modelKmsKeyId);
            versionName(model.versionName);
            sourceModelArn(model.sourceModelArn);
            flywheelArn(model.flywheelArn);
        }

        public final String getDocumentClassifierArn() {
            return documentClassifierArn;
        }

        public final void setDocumentClassifierArn(String documentClassifierArn) {
            this.documentClassifierArn = documentClassifierArn;
        }

        @Override
        public final Builder documentClassifierArn(String documentClassifierArn) {
            this.documentClassifierArn = documentClassifierArn;
            return this;
        }

        public final String getLanguageCode() {
            return languageCode;
        }

        public final void setLanguageCode(String languageCode) {
            this.languageCode = languageCode;
        }

        @Override
        public final Builder languageCode(String languageCode) {
            this.languageCode = languageCode;
            return this;
        }

        @Override
        public final Builder languageCode(LanguageCode languageCode) {
            this.languageCode(languageCode == null ? null : languageCode.toString());
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ModelStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getMessage() {
            return message;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final Instant getSubmitTime() {
            return submitTime;
        }

        public final void setSubmitTime(Instant submitTime) {
            this.submitTime = submitTime;
        }

        @Override
        public final Builder submitTime(Instant submitTime) {
            this.submitTime = submitTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final Instant getTrainingStartTime() {
            return trainingStartTime;
        }

        public final void setTrainingStartTime(Instant trainingStartTime) {
            this.trainingStartTime = trainingStartTime;
        }

        @Override
        public final Builder trainingStartTime(Instant trainingStartTime) {
            this.trainingStartTime = trainingStartTime;
            return this;
        }

        public final Instant getTrainingEndTime() {
            return trainingEndTime;
        }

        public final void setTrainingEndTime(Instant trainingEndTime) {
            this.trainingEndTime = trainingEndTime;
        }

        @Override
        public final Builder trainingEndTime(Instant trainingEndTime) {
            this.trainingEndTime = trainingEndTime;
            return this;
        }

        public final DocumentClassifierInputDataConfig.Builder getInputDataConfig() {
            return inputDataConfig != null ? inputDataConfig.toBuilder() : null;
        }

        public final void setInputDataConfig(DocumentClassifierInputDataConfig.BuilderImpl inputDataConfig) {
            this.inputDataConfig = inputDataConfig != null ? inputDataConfig.build() : null;
        }

        @Override
        public final Builder inputDataConfig(DocumentClassifierInputDataConfig inputDataConfig) {
            this.inputDataConfig = inputDataConfig;
            return this;
        }

        public final DocumentClassifierOutputDataConfig.Builder getOutputDataConfig() {
            return outputDataConfig != null ? outputDataConfig.toBuilder() : null;
        }

        public final void setOutputDataConfig(DocumentClassifierOutputDataConfig.BuilderImpl outputDataConfig) {
            this.outputDataConfig = outputDataConfig != null ? outputDataConfig.build() : null;
        }

        @Override
        public final Builder outputDataConfig(DocumentClassifierOutputDataConfig outputDataConfig) {
            this.outputDataConfig = outputDataConfig;
            return this;
        }

        public final ClassifierMetadata.Builder getClassifierMetadata() {
            return classifierMetadata != null ? classifierMetadata.toBuilder() : null;
        }

        public final void setClassifierMetadata(ClassifierMetadata.BuilderImpl classifierMetadata) {
            this.classifierMetadata = classifierMetadata != null ? classifierMetadata.build() : null;
        }

        @Override
        public final Builder classifierMetadata(ClassifierMetadata classifierMetadata) {
            this.classifierMetadata = classifierMetadata;
            return this;
        }

        public final String getDataAccessRoleArn() {
            return dataAccessRoleArn;
        }

        public final void setDataAccessRoleArn(String dataAccessRoleArn) {
            this.dataAccessRoleArn = dataAccessRoleArn;
        }

        @Override
        public final Builder dataAccessRoleArn(String dataAccessRoleArn) {
            this.dataAccessRoleArn = dataAccessRoleArn;
            return this;
        }

        public final String getVolumeKmsKeyId() {
            return volumeKmsKeyId;
        }

        public final void setVolumeKmsKeyId(String volumeKmsKeyId) {
            this.volumeKmsKeyId = volumeKmsKeyId;
        }

        @Override
        public final Builder volumeKmsKeyId(String volumeKmsKeyId) {
            this.volumeKmsKeyId = volumeKmsKeyId;
            return this;
        }

        public final VpcConfig.Builder getVpcConfig() {
            return vpcConfig != null ? vpcConfig.toBuilder() : null;
        }

        public final void setVpcConfig(VpcConfig.BuilderImpl vpcConfig) {
            this.vpcConfig = vpcConfig != null ? vpcConfig.build() : null;
        }

        @Override
        public final Builder vpcConfig(VpcConfig vpcConfig) {
            this.vpcConfig = vpcConfig;
            return this;
        }

        public final String getMode() {
            return mode;
        }

        public final void setMode(String mode) {
            this.mode = mode;
        }

        @Override
        public final Builder mode(String mode) {
            this.mode = mode;
            return this;
        }

        @Override
        public final Builder mode(DocumentClassifierMode mode) {
            this.mode(mode == null ? null : mode.toString());
            return this;
        }

        public final String getModelKmsKeyId() {
            return modelKmsKeyId;
        }

        public final void setModelKmsKeyId(String modelKmsKeyId) {
            this.modelKmsKeyId = modelKmsKeyId;
        }

        @Override
        public final Builder modelKmsKeyId(String modelKmsKeyId) {
            this.modelKmsKeyId = modelKmsKeyId;
            return this;
        }

        public final String getVersionName() {
            return versionName;
        }

        public final void setVersionName(String versionName) {
            this.versionName = versionName;
        }

        @Override
        public final Builder versionName(String versionName) {
            this.versionName = versionName;
            return this;
        }

        public final String getSourceModelArn() {
            return sourceModelArn;
        }

        public final void setSourceModelArn(String sourceModelArn) {
            this.sourceModelArn = sourceModelArn;
        }

        @Override
        public final Builder sourceModelArn(String sourceModelArn) {
            this.sourceModelArn = sourceModelArn;
            return this;
        }

        public final String getFlywheelArn() {
            return flywheelArn;
        }

        public final void setFlywheelArn(String flywheelArn) {
            this.flywheelArn = flywheelArn;
        }

        @Override
        public final Builder flywheelArn(String flywheelArn) {
            this.flywheelArn = flywheelArn;
            return this;
        }

        @Override
        public DocumentClassifierProperties build() {
            return new DocumentClassifierProperties(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
