/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A key group configuration.
 * </p>
 * <p>
 * A key group contains a list of public keys that you can use with <a
 * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/PrivateContent.html">CloudFront signed URLs
 * and signed cookies</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class KeyGroupConfig implements SdkPojo, Serializable, ToCopyableBuilder<KeyGroupConfig.Builder, KeyGroupConfig> {
    private static final SdkField<String> NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Name")
            .getter(getter(KeyGroupConfig::name))
            .setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name")
                    .unmarshallLocationName("Name").build()).build();

    private static final SdkField<List<String>> ITEMS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Items")
            .getter(getter(KeyGroupConfig::items))
            .setter(setter(Builder::items))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Items")
                    .unmarshallLocationName("Items").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("PublicKey")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("PublicKey").unmarshallLocationName("PublicKey").build())
                                            .build()).build()).build();

    private static final SdkField<String> COMMENT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Comment")
            .getter(getter(KeyGroupConfig::comment))
            .setter(setter(Builder::comment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Comment")
                    .unmarshallLocationName("Comment").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ITEMS_FIELD,
            COMMENT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<String> items;

    private final String comment;

    private KeyGroupConfig(BuilderImpl builder) {
        this.name = builder.name;
        this.items = builder.items;
        this.comment = builder.comment;
    }

    /**
     * <p>
     * A name to identify the key group.
     * </p>
     * 
     * @return A name to identify the key group.
     */
    public final String name() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the Items property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasItems() {
        return items != null && !(items instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the identifiers of the public keys in the key group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasItems} method.
     * </p>
     * 
     * @return A list of the identifiers of the public keys in the key group.
     */
    public final List<String> items() {
        return items;
    }

    /**
     * <p>
     * A comment to describe the key group. The comment cannot be longer than 128 characters.
     * </p>
     * 
     * @return A comment to describe the key group. The comment cannot be longer than 128 characters.
     */
    public final String comment() {
        return comment;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(hasItems() ? items() : null);
        hashCode = 31 * hashCode + Objects.hashCode(comment());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KeyGroupConfig)) {
            return false;
        }
        KeyGroupConfig other = (KeyGroupConfig) obj;
        return Objects.equals(name(), other.name()) && hasItems() == other.hasItems() && Objects.equals(items(), other.items())
                && Objects.equals(comment(), other.comment());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("KeyGroupConfig").add("Name", name()).add("Items", hasItems() ? items() : null)
                .add("Comment", comment()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Items":
            return Optional.ofNullable(clazz.cast(items()));
        case "Comment":
            return Optional.ofNullable(clazz.cast(comment()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<KeyGroupConfig, T> g) {
        return obj -> g.apply((KeyGroupConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, KeyGroupConfig> {
        /**
         * <p>
         * A name to identify the key group.
         * </p>
         * 
         * @param name
         *        A name to identify the key group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A list of the identifiers of the public keys in the key group.
         * </p>
         * 
         * @param items
         *        A list of the identifiers of the public keys in the key group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(Collection<String> items);

        /**
         * <p>
         * A list of the identifiers of the public keys in the key group.
         * </p>
         * 
         * @param items
         *        A list of the identifiers of the public keys in the key group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(String... items);

        /**
         * <p>
         * A comment to describe the key group. The comment cannot be longer than 128 characters.
         * </p>
         * 
         * @param comment
         *        A comment to describe the key group. The comment cannot be longer than 128 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder comment(String comment);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<String> items = DefaultSdkAutoConstructList.getInstance();

        private String comment;

        private BuilderImpl() {
        }

        private BuilderImpl(KeyGroupConfig model) {
            name(model.name);
            items(model.items);
            comment(model.comment);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final Collection<String> getItems() {
            if (items instanceof SdkAutoConstructList) {
                return null;
            }
            return items;
        }

        public final void setItems(Collection<String> items) {
            this.items = PublicKeyIdListCopier.copy(items);
        }

        @Override
        public final Builder items(Collection<String> items) {
            this.items = PublicKeyIdListCopier.copy(items);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder items(String... items) {
            items(Arrays.asList(items));
            return this;
        }

        public final String getComment() {
            return comment;
        }

        public final void setComment(String comment) {
            this.comment = comment;
        }

        @Override
        public final Builder comment(String comment) {
            this.comment = comment;
            return this;
        }

        @Override
        public KeyGroupConfig build() {
            return new KeyGroupConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
