/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudcontrol;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.cloudcontrol.internal.CloudControlServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.cloudcontrol.model.AlreadyExistsException;
import software.amazon.awssdk.services.cloudcontrol.model.CancelResourceRequestRequest;
import software.amazon.awssdk.services.cloudcontrol.model.CancelResourceRequestResponse;
import software.amazon.awssdk.services.cloudcontrol.model.ClientTokenConflictException;
import software.amazon.awssdk.services.cloudcontrol.model.CloudControlException;
import software.amazon.awssdk.services.cloudcontrol.model.ConcurrentModificationException;
import software.amazon.awssdk.services.cloudcontrol.model.ConcurrentOperationException;
import software.amazon.awssdk.services.cloudcontrol.model.CreateResourceRequest;
import software.amazon.awssdk.services.cloudcontrol.model.CreateResourceResponse;
import software.amazon.awssdk.services.cloudcontrol.model.DeleteResourceRequest;
import software.amazon.awssdk.services.cloudcontrol.model.DeleteResourceResponse;
import software.amazon.awssdk.services.cloudcontrol.model.GeneralServiceException;
import software.amazon.awssdk.services.cloudcontrol.model.GetResourceRequest;
import software.amazon.awssdk.services.cloudcontrol.model.GetResourceRequestStatusRequest;
import software.amazon.awssdk.services.cloudcontrol.model.GetResourceRequestStatusResponse;
import software.amazon.awssdk.services.cloudcontrol.model.GetResourceResponse;
import software.amazon.awssdk.services.cloudcontrol.model.HandlerFailureException;
import software.amazon.awssdk.services.cloudcontrol.model.HandlerInternalFailureException;
import software.amazon.awssdk.services.cloudcontrol.model.InvalidCredentialsException;
import software.amazon.awssdk.services.cloudcontrol.model.InvalidRequestException;
import software.amazon.awssdk.services.cloudcontrol.model.ListResourceRequestsRequest;
import software.amazon.awssdk.services.cloudcontrol.model.ListResourceRequestsResponse;
import software.amazon.awssdk.services.cloudcontrol.model.ListResourcesRequest;
import software.amazon.awssdk.services.cloudcontrol.model.ListResourcesResponse;
import software.amazon.awssdk.services.cloudcontrol.model.NetworkFailureException;
import software.amazon.awssdk.services.cloudcontrol.model.NotStabilizedException;
import software.amazon.awssdk.services.cloudcontrol.model.NotUpdatableException;
import software.amazon.awssdk.services.cloudcontrol.model.PrivateTypeException;
import software.amazon.awssdk.services.cloudcontrol.model.RequestTokenNotFoundException;
import software.amazon.awssdk.services.cloudcontrol.model.ResourceConflictException;
import software.amazon.awssdk.services.cloudcontrol.model.ResourceNotFoundException;
import software.amazon.awssdk.services.cloudcontrol.model.ServiceInternalErrorException;
import software.amazon.awssdk.services.cloudcontrol.model.ServiceLimitExceededException;
import software.amazon.awssdk.services.cloudcontrol.model.ThrottlingException;
import software.amazon.awssdk.services.cloudcontrol.model.TypeNotFoundException;
import software.amazon.awssdk.services.cloudcontrol.model.UnsupportedActionException;
import software.amazon.awssdk.services.cloudcontrol.model.UpdateResourceRequest;
import software.amazon.awssdk.services.cloudcontrol.model.UpdateResourceResponse;
import software.amazon.awssdk.services.cloudcontrol.transform.CancelResourceRequestRequestMarshaller;
import software.amazon.awssdk.services.cloudcontrol.transform.CreateResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudcontrol.transform.DeleteResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudcontrol.transform.GetResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudcontrol.transform.GetResourceRequestStatusRequestMarshaller;
import software.amazon.awssdk.services.cloudcontrol.transform.ListResourceRequestsRequestMarshaller;
import software.amazon.awssdk.services.cloudcontrol.transform.ListResourcesRequestMarshaller;
import software.amazon.awssdk.services.cloudcontrol.transform.UpdateResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudcontrol.waiters.CloudControlWaiter;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link CloudControlClient}.
 *
 * @see CloudControlClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudControlClient implements CloudControlClient {
    private static final Logger log = Logger.loggerFor(DefaultCloudControlClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloudControlClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Cancels the specified resource operation request. For more information, see <a href=
     * "https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-manage-requests.html#resource-operations-manage-requests-cancel"
     * >Canceling resource operation requests</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
     * </p>
     * <p>
     * Only resource operations requests with a status of <code>PENDING</code> or <code>IN_PROGRESS</code> can be
     * canceled.
     * </p>
     *
     * @param cancelResourceRequestRequest
     * @return Result of the CancelResourceRequest operation returned by the service.
     * @throws ConcurrentModificationException
     *         The resource is currently being modified by another operation.
     * @throws RequestTokenNotFoundException
     *         A resource operation with the specified request token can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudControlClient.CancelResourceRequest
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/CancelResourceRequest"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CancelResourceRequestResponse cancelResourceRequest(CancelResourceRequestRequest cancelResourceRequestRequest)
            throws ConcurrentModificationException, RequestTokenNotFoundException, AwsServiceException, SdkClientException,
            CloudControlException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelResourceRequestResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CancelResourceRequestResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelResourceRequestRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelResourceRequestRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudControl");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelResourceRequest");

            return clientHandler.execute(new ClientExecutionParams<CancelResourceRequestRequest, CancelResourceRequestResponse>()
                    .withOperationName("CancelResourceRequest").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(cancelResourceRequestRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelResourceRequestRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates the specified resource. For more information, see <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-create.html">Creating a
     * resource</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
     * </p>
     * <p>
     * After you have initiated a resource creation request, you can monitor the progress of your request by calling <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/APIReference/API_GetResourceRequestStatus.html">
     * GetResourceRequestStatus</a> using the <code>RequestToken</code> of the <code>ProgressEvent</code> type returned
     * by <code>CreateResource</code>.
     * </p>
     *
     * @param createResourceRequest
     * @return Result of the CreateResource operation returned by the service.
     * @throws AlreadyExistsException
     *         The resource with the name requested already exists.
     * @throws HandlerInternalFailureException
     *         The resource handler has returned that an unexpected error occurred within the resource handler.
     * @throws GeneralServiceException
     *         The resource handler has returned that the downstream service generated an error that doesn't map to any
     *         other handler error code.
     * @throws NotUpdatableException
     *         One or more properties included in this resource operation are defined as create-only, and therefore
     *         can't be updated.
     * @throws TypeNotFoundException
     *         The specified extension doesn't exist in the CloudFormation registry.
     * @throws ConcurrentOperationException
     *         Another resource operation is currently being performed on this resource.
     * @throws InvalidRequestException
     *         The resource handler has returned that invalid input from the user has generated a generic exception.
     * @throws PrivateTypeException
     *         Cloud Control API hasn't received a valid response from the resource handler, due to a configuration
     *         error. This includes issues such as the resource handler returning an invalid response, or timing out.
     * @throws ResourceNotFoundException
     *         A resource with the specified identifier can't be found.
     * @throws NetworkFailureException
     *         The resource handler has returned that the request couldn't be completed due to networking issues, such
     *         as a failure to receive a response from the server.
     * @throws UnsupportedActionException
     *         The specified resource doesn't support this resource operation.
     * @throws NotStabilizedException
     *         The resource handler has returned that the downstream resource failed to complete all of its ready-state
     *         checks.
     * @throws ServiceInternalErrorException
     *         The resource handler has returned that the downstream service returned an internal error, typically with
     *         a <code>5XX HTTP</code> status code.
     * @throws HandlerFailureException
     *         The resource handler has failed without a returning a more specific error code. This can include
     *         timeouts.
     * @throws ServiceLimitExceededException
     *         The resource handler has returned that a non-transient resource limit was reached on the service side.
     * @throws InvalidCredentialsException
     *         The resource handler has returned that the credentials provided by the user are invalid.
     * @throws ResourceConflictException
     *         The resource is temporarily unavailable to be acted upon. For example, if the resource is currently
     *         undergoing an operation and can't be acted upon until that operation is finished.
     * @throws ClientTokenConflictException
     *         The specified client token has already been used in another resource request.</p>
     *         <p>
     *         It's best practice for client tokens to be unique for each resource operation request. However, client
     *         token expire after 36 hours.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudControlClient.CreateResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/CreateResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateResourceResponse createResource(CreateResourceRequest createResourceRequest) throws AlreadyExistsException,
            HandlerInternalFailureException, GeneralServiceException, NotUpdatableException, TypeNotFoundException,
            ConcurrentOperationException, InvalidRequestException, PrivateTypeException, ResourceNotFoundException,
            NetworkFailureException, UnsupportedActionException, NotStabilizedException, ServiceInternalErrorException,
            HandlerFailureException, ServiceLimitExceededException, InvalidCredentialsException, ResourceConflictException,
            ClientTokenConflictException, ThrottlingException, AwsServiceException, SdkClientException, CloudControlException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudControl");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateResource");

            return clientHandler.execute(new ClientExecutionParams<CreateResourceRequest, CreateResourceResponse>()
                    .withOperationName("CreateResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified resource. For details, see <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-delete.html">Deleting a
     * resource</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
     * </p>
     * <p>
     * After you have initiated a resource deletion request, you can monitor the progress of your request by calling <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/APIReference/API_GetResourceRequestStatus.html">
     * GetResourceRequestStatus</a> using the <code>RequestToken</code> of the <code>ProgressEvent</code> returned by
     * <code>DeleteResource</code>.
     * </p>
     *
     * @param deleteResourceRequest
     * @return Result of the DeleteResource operation returned by the service.
     * @throws AlreadyExistsException
     *         The resource with the name requested already exists.
     * @throws HandlerInternalFailureException
     *         The resource handler has returned that an unexpected error occurred within the resource handler.
     * @throws GeneralServiceException
     *         The resource handler has returned that the downstream service generated an error that doesn't map to any
     *         other handler error code.
     * @throws NotUpdatableException
     *         One or more properties included in this resource operation are defined as create-only, and therefore
     *         can't be updated.
     * @throws TypeNotFoundException
     *         The specified extension doesn't exist in the CloudFormation registry.
     * @throws ConcurrentOperationException
     *         Another resource operation is currently being performed on this resource.
     * @throws InvalidRequestException
     *         The resource handler has returned that invalid input from the user has generated a generic exception.
     * @throws PrivateTypeException
     *         Cloud Control API hasn't received a valid response from the resource handler, due to a configuration
     *         error. This includes issues such as the resource handler returning an invalid response, or timing out.
     * @throws ResourceNotFoundException
     *         A resource with the specified identifier can't be found.
     * @throws NetworkFailureException
     *         The resource handler has returned that the request couldn't be completed due to networking issues, such
     *         as a failure to receive a response from the server.
     * @throws UnsupportedActionException
     *         The specified resource doesn't support this resource operation.
     * @throws NotStabilizedException
     *         The resource handler has returned that the downstream resource failed to complete all of its ready-state
     *         checks.
     * @throws ServiceInternalErrorException
     *         The resource handler has returned that the downstream service returned an internal error, typically with
     *         a <code>5XX HTTP</code> status code.
     * @throws HandlerFailureException
     *         The resource handler has failed without a returning a more specific error code. This can include
     *         timeouts.
     * @throws ServiceLimitExceededException
     *         The resource handler has returned that a non-transient resource limit was reached on the service side.
     * @throws InvalidCredentialsException
     *         The resource handler has returned that the credentials provided by the user are invalid.
     * @throws ResourceConflictException
     *         The resource is temporarily unavailable to be acted upon. For example, if the resource is currently
     *         undergoing an operation and can't be acted upon until that operation is finished.
     * @throws ClientTokenConflictException
     *         The specified client token has already been used in another resource request.</p>
     *         <p>
     *         It's best practice for client tokens to be unique for each resource operation request. However, client
     *         token expire after 36 hours.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudControlClient.DeleteResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/DeleteResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteResourceResponse deleteResource(DeleteResourceRequest deleteResourceRequest) throws AlreadyExistsException,
            HandlerInternalFailureException, GeneralServiceException, NotUpdatableException, TypeNotFoundException,
            ConcurrentOperationException, InvalidRequestException, PrivateTypeException, ResourceNotFoundException,
            NetworkFailureException, UnsupportedActionException, NotStabilizedException, ServiceInternalErrorException,
            HandlerFailureException, ServiceLimitExceededException, InvalidCredentialsException, ResourceConflictException,
            ClientTokenConflictException, ThrottlingException, AwsServiceException, SdkClientException, CloudControlException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudControl");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResource");

            return clientHandler.execute(new ClientExecutionParams<DeleteResourceRequest, DeleteResourceResponse>()
                    .withOperationName("DeleteResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the current state of the specified resource. For details, see <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-read.html">Reading a
     * resource's current state</a>.
     * </p>
     * <p>
     * You can use this action to return information about an existing resource in your account and Amazon Web Services
     * Region, whether those resources were provisioned using Cloud Control API.
     * </p>
     *
     * @param getResourceRequest
     * @return Result of the GetResource operation returned by the service.
     * @throws AlreadyExistsException
     *         The resource with the name requested already exists.
     * @throws HandlerInternalFailureException
     *         The resource handler has returned that an unexpected error occurred within the resource handler.
     * @throws GeneralServiceException
     *         The resource handler has returned that the downstream service generated an error that doesn't map to any
     *         other handler error code.
     * @throws NotUpdatableException
     *         One or more properties included in this resource operation are defined as create-only, and therefore
     *         can't be updated.
     * @throws TypeNotFoundException
     *         The specified extension doesn't exist in the CloudFormation registry.
     * @throws InvalidRequestException
     *         The resource handler has returned that invalid input from the user has generated a generic exception.
     * @throws PrivateTypeException
     *         Cloud Control API hasn't received a valid response from the resource handler, due to a configuration
     *         error. This includes issues such as the resource handler returning an invalid response, or timing out.
     * @throws ResourceNotFoundException
     *         A resource with the specified identifier can't be found.
     * @throws NetworkFailureException
     *         The resource handler has returned that the request couldn't be completed due to networking issues, such
     *         as a failure to receive a response from the server.
     * @throws UnsupportedActionException
     *         The specified resource doesn't support this resource operation.
     * @throws NotStabilizedException
     *         The resource handler has returned that the downstream resource failed to complete all of its ready-state
     *         checks.
     * @throws ServiceInternalErrorException
     *         The resource handler has returned that the downstream service returned an internal error, typically with
     *         a <code>5XX HTTP</code> status code.
     * @throws HandlerFailureException
     *         The resource handler has failed without a returning a more specific error code. This can include
     *         timeouts.
     * @throws ServiceLimitExceededException
     *         The resource handler has returned that a non-transient resource limit was reached on the service side.
     * @throws InvalidCredentialsException
     *         The resource handler has returned that the credentials provided by the user are invalid.
     * @throws ResourceConflictException
     *         The resource is temporarily unavailable to be acted upon. For example, if the resource is currently
     *         undergoing an operation and can't be acted upon until that operation is finished.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudControlClient.GetResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/GetResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetResourceResponse getResource(GetResourceRequest getResourceRequest) throws AlreadyExistsException,
            HandlerInternalFailureException, GeneralServiceException, NotUpdatableException, TypeNotFoundException,
            InvalidRequestException, PrivateTypeException, ResourceNotFoundException, NetworkFailureException,
            UnsupportedActionException, NotStabilizedException, ServiceInternalErrorException, HandlerFailureException,
            ServiceLimitExceededException, InvalidCredentialsException, ResourceConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, CloudControlException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudControl");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResource");

            return clientHandler.execute(new ClientExecutionParams<GetResourceRequest, GetResourceResponse>()
                    .withOperationName("GetResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the current status of a resource operation request. For more information, see <a href=
     * "https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-manage-requests.html#resource-operations-manage-requests-track"
     * >Tracking the progress of resource operation requests</a> in the <i>Amazon Web Services Cloud Control API User
     * Guide</i>.
     * </p>
     *
     * @param getResourceRequestStatusRequest
     * @return Result of the GetResourceRequestStatus operation returned by the service.
     * @throws RequestTokenNotFoundException
     *         A resource operation with the specified request token can't be found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudControlClient.GetResourceRequestStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/GetResourceRequestStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetResourceRequestStatusResponse getResourceRequestStatus(
            GetResourceRequestStatusRequest getResourceRequestStatusRequest) throws RequestTokenNotFoundException,
            AwsServiceException, SdkClientException, CloudControlException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourceRequestStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetResourceRequestStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourceRequestStatusRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceRequestStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudControl");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceRequestStatus");

            return clientHandler
                    .execute(new ClientExecutionParams<GetResourceRequestStatusRequest, GetResourceRequestStatusResponse>()
                            .withOperationName("GetResourceRequestStatus").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getResourceRequestStatusRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetResourceRequestStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns existing resource operation requests. This includes requests of all status types. For more information,
     * see <a href=
     * "https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-manage-requests.html#resource-operations-manage-requests-list"
     * >Listing active resource operation requests</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
     * </p>
     * <note>
     * <p>
     * Resource operation requests expire after 7 days.
     * </p>
     * </note>
     *
     * @param listResourceRequestsRequest
     * @return Result of the ListResourceRequests operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudControlClient.ListResourceRequests
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/ListResourceRequests"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListResourceRequestsResponse listResourceRequests(ListResourceRequestsRequest listResourceRequestsRequest)
            throws AwsServiceException, SdkClientException, CloudControlException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourceRequestsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListResourceRequestsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResourceRequestsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceRequestsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudControl");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceRequests");

            return clientHandler.execute(new ClientExecutionParams<ListResourceRequestsRequest, ListResourceRequestsResponse>()
                    .withOperationName("ListResourceRequests").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listResourceRequestsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListResourceRequestsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the specified resources. For more information, see <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-list.html">Discovering
     * resources</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
     * </p>
     * <p>
     * You can use this action to return information about existing resources in your account and Amazon Web Services
     * Region, whether those resources were provisioned using Cloud Control API.
     * </p>
     *
     * @param listResourcesRequest
     * @return Result of the ListResources operation returned by the service.
     * @throws AlreadyExistsException
     *         The resource with the name requested already exists.
     * @throws HandlerInternalFailureException
     *         The resource handler has returned that an unexpected error occurred within the resource handler.
     * @throws GeneralServiceException
     *         The resource handler has returned that the downstream service generated an error that doesn't map to any
     *         other handler error code.
     * @throws NotUpdatableException
     *         One or more properties included in this resource operation are defined as create-only, and therefore
     *         can't be updated.
     * @throws TypeNotFoundException
     *         The specified extension doesn't exist in the CloudFormation registry.
     * @throws InvalidRequestException
     *         The resource handler has returned that invalid input from the user has generated a generic exception.
     * @throws PrivateTypeException
     *         Cloud Control API hasn't received a valid response from the resource handler, due to a configuration
     *         error. This includes issues such as the resource handler returning an invalid response, or timing out.
     * @throws ResourceNotFoundException
     *         A resource with the specified identifier can't be found.
     * @throws NetworkFailureException
     *         The resource handler has returned that the request couldn't be completed due to networking issues, such
     *         as a failure to receive a response from the server.
     * @throws UnsupportedActionException
     *         The specified resource doesn't support this resource operation.
     * @throws NotStabilizedException
     *         The resource handler has returned that the downstream resource failed to complete all of its ready-state
     *         checks.
     * @throws ServiceInternalErrorException
     *         The resource handler has returned that the downstream service returned an internal error, typically with
     *         a <code>5XX HTTP</code> status code.
     * @throws HandlerFailureException
     *         The resource handler has failed without a returning a more specific error code. This can include
     *         timeouts.
     * @throws ServiceLimitExceededException
     *         The resource handler has returned that a non-transient resource limit was reached on the service side.
     * @throws InvalidCredentialsException
     *         The resource handler has returned that the credentials provided by the user are invalid.
     * @throws ResourceConflictException
     *         The resource is temporarily unavailable to be acted upon. For example, if the resource is currently
     *         undergoing an operation and can't be acted upon until that operation is finished.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudControlClient.ListResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/ListResources" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListResourcesResponse listResources(ListResourcesRequest listResourcesRequest) throws AlreadyExistsException,
            HandlerInternalFailureException, GeneralServiceException, NotUpdatableException, TypeNotFoundException,
            InvalidRequestException, PrivateTypeException, ResourceNotFoundException, NetworkFailureException,
            UnsupportedActionException, NotStabilizedException, ServiceInternalErrorException, HandlerFailureException,
            ServiceLimitExceededException, InvalidCredentialsException, ResourceConflictException, ThrottlingException,
            AwsServiceException, SdkClientException, CloudControlException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListResourcesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResourcesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudControl");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResources");

            return clientHandler.execute(new ClientExecutionParams<ListResourcesRequest, ListResourcesResponse>()
                    .withOperationName("ListResources").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the specified property values in the resource.
     * </p>
     * <p>
     * You specify your resource property updates as a list of patch operations contained in a JSON patch document that
     * adheres to the <a href="https://datatracker.ietf.org/doc/html/rfc6902"> <i>RFC 6902 - JavaScript Object Notation
     * (JSON) Patch</i> </a> standard.
     * </p>
     * <p>
     * For details on how Cloud Control API performs resource update operations, see <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/userguide/resource-operations-update.html">Updating a
     * resource</a> in the <i>Amazon Web Services Cloud Control API User Guide</i>.
     * </p>
     * <p>
     * After you have initiated a resource update request, you can monitor the progress of your request by calling <a
     * href="https://docs.aws.amazon.com/cloudcontrolapi/latest/APIReference/API_GetResourceRequestStatus.html">
     * GetResourceRequestStatus</a> using the <code>RequestToken</code> of the <code>ProgressEvent</code> returned by
     * <code>UpdateResource</code>.
     * </p>
     * <p>
     * For more information about the properties of a specific resource, refer to the related topic for the resource in
     * the <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-template-resource-type-ref.html">
     * Resource and property types reference</a> in the <i>CloudFormation Users Guide</i>.
     * </p>
     *
     * @param updateResourceRequest
     * @return Result of the UpdateResource operation returned by the service.
     * @throws AlreadyExistsException
     *         The resource with the name requested already exists.
     * @throws HandlerInternalFailureException
     *         The resource handler has returned that an unexpected error occurred within the resource handler.
     * @throws GeneralServiceException
     *         The resource handler has returned that the downstream service generated an error that doesn't map to any
     *         other handler error code.
     * @throws NotUpdatableException
     *         One or more properties included in this resource operation are defined as create-only, and therefore
     *         can't be updated.
     * @throws TypeNotFoundException
     *         The specified extension doesn't exist in the CloudFormation registry.
     * @throws ConcurrentOperationException
     *         Another resource operation is currently being performed on this resource.
     * @throws InvalidRequestException
     *         The resource handler has returned that invalid input from the user has generated a generic exception.
     * @throws PrivateTypeException
     *         Cloud Control API hasn't received a valid response from the resource handler, due to a configuration
     *         error. This includes issues such as the resource handler returning an invalid response, or timing out.
     * @throws ResourceNotFoundException
     *         A resource with the specified identifier can't be found.
     * @throws NetworkFailureException
     *         The resource handler has returned that the request couldn't be completed due to networking issues, such
     *         as a failure to receive a response from the server.
     * @throws UnsupportedActionException
     *         The specified resource doesn't support this resource operation.
     * @throws NotStabilizedException
     *         The resource handler has returned that the downstream resource failed to complete all of its ready-state
     *         checks.
     * @throws ServiceInternalErrorException
     *         The resource handler has returned that the downstream service returned an internal error, typically with
     *         a <code>5XX HTTP</code> status code.
     * @throws HandlerFailureException
     *         The resource handler has failed without a returning a more specific error code. This can include
     *         timeouts.
     * @throws ServiceLimitExceededException
     *         The resource handler has returned that a non-transient resource limit was reached on the service side.
     * @throws InvalidCredentialsException
     *         The resource handler has returned that the credentials provided by the user are invalid.
     * @throws ResourceConflictException
     *         The resource is temporarily unavailable to be acted upon. For example, if the resource is currently
     *         undergoing an operation and can't be acted upon until that operation is finished.
     * @throws ClientTokenConflictException
     *         The specified client token has already been used in another resource request.</p>
     *         <p>
     *         It's best practice for client tokens to be unique for each resource operation request. However, client
     *         token expire after 36 hours.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudControlException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudControlClient.UpdateResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cloudcontrol-2021-09-30/UpdateResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateResourceResponse updateResource(UpdateResourceRequest updateResourceRequest) throws AlreadyExistsException,
            HandlerInternalFailureException, GeneralServiceException, NotUpdatableException, TypeNotFoundException,
            ConcurrentOperationException, InvalidRequestException, PrivateTypeException, ResourceNotFoundException,
            NetworkFailureException, UnsupportedActionException, NotStabilizedException, ServiceInternalErrorException,
            HandlerFailureException, ServiceLimitExceededException, InvalidCredentialsException, ResourceConflictException,
            ClientTokenConflictException, ThrottlingException, AwsServiceException, SdkClientException, CloudControlException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudControl");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResource");

            return clientHandler.execute(new ClientExecutionParams<UpdateResourceRequest, UpdateResourceResponse>()
                    .withOperationName("UpdateResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create an instance of {@link CloudControlWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link CloudControlWaiter}
     */
    @Override
    public CloudControlWaiter waiter() {
        return CloudControlWaiter.builder().client(this).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        CloudControlServiceClientConfigurationBuilder serviceConfigBuilder = new CloudControlServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CloudControlException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("HandlerFailureException")
                                .exceptionBuilderSupplier(HandlerFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("HandlerInternalFailureException")
                                .exceptionBuilderSupplier(HandlerInternalFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceLimitExceededException")
                                .exceptionBuilderSupplier(ServiceLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceInternalErrorException")
                                .exceptionBuilderSupplier(ServiceInternalErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PrivateTypeException")
                                .exceptionBuilderSupplier(PrivateTypeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestTokenNotFoundException")
                                .exceptionBuilderSupplier(RequestTokenNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ClientTokenConflictException")
                                .exceptionBuilderSupplier(ClientTokenConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NetworkFailureException")
                                .exceptionBuilderSupplier(NetworkFailureException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AlreadyExistsException")
                                .exceptionBuilderSupplier(AlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentOperationException")
                                .exceptionBuilderSupplier(ConcurrentOperationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedActionException")
                                .exceptionBuilderSupplier(UnsupportedActionException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotStabilizedException")
                                .exceptionBuilderSupplier(NotStabilizedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceConflictException")
                                .exceptionBuilderSupplier(ResourceConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidCredentialsException")
                                .exceptionBuilderSupplier(InvalidCredentialsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("GeneralServiceException")
                                .exceptionBuilderSupplier(GeneralServiceException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TypeNotFoundException")
                                .exceptionBuilderSupplier(TypeNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotUpdatableException")
                                .exceptionBuilderSupplier(NotUpdatableException::builder).httpStatusCode(400).build());
    }

    @Override
    public final CloudControlServiceClientConfiguration serviceClientConfiguration() {
        return new CloudControlServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
