/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cleanrooms;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.cleanrooms.model.AccessDeniedException;
import software.amazon.awssdk.services.cleanrooms.model.BatchGetSchemaRequest;
import software.amazon.awssdk.services.cleanrooms.model.BatchGetSchemaResponse;
import software.amazon.awssdk.services.cleanrooms.model.CleanRoomsException;
import software.amazon.awssdk.services.cleanrooms.model.ConflictException;
import software.amazon.awssdk.services.cleanrooms.model.CreateCollaborationRequest;
import software.amazon.awssdk.services.cleanrooms.model.CreateCollaborationResponse;
import software.amazon.awssdk.services.cleanrooms.model.CreateConfiguredTableAnalysisRuleRequest;
import software.amazon.awssdk.services.cleanrooms.model.CreateConfiguredTableAnalysisRuleResponse;
import software.amazon.awssdk.services.cleanrooms.model.CreateConfiguredTableAssociationRequest;
import software.amazon.awssdk.services.cleanrooms.model.CreateConfiguredTableAssociationResponse;
import software.amazon.awssdk.services.cleanrooms.model.CreateConfiguredTableRequest;
import software.amazon.awssdk.services.cleanrooms.model.CreateConfiguredTableResponse;
import software.amazon.awssdk.services.cleanrooms.model.CreateMembershipRequest;
import software.amazon.awssdk.services.cleanrooms.model.CreateMembershipResponse;
import software.amazon.awssdk.services.cleanrooms.model.DeleteCollaborationRequest;
import software.amazon.awssdk.services.cleanrooms.model.DeleteCollaborationResponse;
import software.amazon.awssdk.services.cleanrooms.model.DeleteConfiguredTableAnalysisRuleRequest;
import software.amazon.awssdk.services.cleanrooms.model.DeleteConfiguredTableAnalysisRuleResponse;
import software.amazon.awssdk.services.cleanrooms.model.DeleteConfiguredTableAssociationRequest;
import software.amazon.awssdk.services.cleanrooms.model.DeleteConfiguredTableAssociationResponse;
import software.amazon.awssdk.services.cleanrooms.model.DeleteConfiguredTableRequest;
import software.amazon.awssdk.services.cleanrooms.model.DeleteConfiguredTableResponse;
import software.amazon.awssdk.services.cleanrooms.model.DeleteMemberRequest;
import software.amazon.awssdk.services.cleanrooms.model.DeleteMemberResponse;
import software.amazon.awssdk.services.cleanrooms.model.DeleteMembershipRequest;
import software.amazon.awssdk.services.cleanrooms.model.DeleteMembershipResponse;
import software.amazon.awssdk.services.cleanrooms.model.GetCollaborationRequest;
import software.amazon.awssdk.services.cleanrooms.model.GetCollaborationResponse;
import software.amazon.awssdk.services.cleanrooms.model.GetConfiguredTableAnalysisRuleRequest;
import software.amazon.awssdk.services.cleanrooms.model.GetConfiguredTableAnalysisRuleResponse;
import software.amazon.awssdk.services.cleanrooms.model.GetConfiguredTableAssociationRequest;
import software.amazon.awssdk.services.cleanrooms.model.GetConfiguredTableAssociationResponse;
import software.amazon.awssdk.services.cleanrooms.model.GetConfiguredTableRequest;
import software.amazon.awssdk.services.cleanrooms.model.GetConfiguredTableResponse;
import software.amazon.awssdk.services.cleanrooms.model.GetMembershipRequest;
import software.amazon.awssdk.services.cleanrooms.model.GetMembershipResponse;
import software.amazon.awssdk.services.cleanrooms.model.GetProtectedQueryRequest;
import software.amazon.awssdk.services.cleanrooms.model.GetProtectedQueryResponse;
import software.amazon.awssdk.services.cleanrooms.model.GetSchemaAnalysisRuleRequest;
import software.amazon.awssdk.services.cleanrooms.model.GetSchemaAnalysisRuleResponse;
import software.amazon.awssdk.services.cleanrooms.model.GetSchemaRequest;
import software.amazon.awssdk.services.cleanrooms.model.GetSchemaResponse;
import software.amazon.awssdk.services.cleanrooms.model.InternalServerException;
import software.amazon.awssdk.services.cleanrooms.model.ListCollaborationsRequest;
import software.amazon.awssdk.services.cleanrooms.model.ListCollaborationsResponse;
import software.amazon.awssdk.services.cleanrooms.model.ListConfiguredTableAssociationsRequest;
import software.amazon.awssdk.services.cleanrooms.model.ListConfiguredTableAssociationsResponse;
import software.amazon.awssdk.services.cleanrooms.model.ListConfiguredTablesRequest;
import software.amazon.awssdk.services.cleanrooms.model.ListConfiguredTablesResponse;
import software.amazon.awssdk.services.cleanrooms.model.ListMembersRequest;
import software.amazon.awssdk.services.cleanrooms.model.ListMembersResponse;
import software.amazon.awssdk.services.cleanrooms.model.ListMembershipsRequest;
import software.amazon.awssdk.services.cleanrooms.model.ListMembershipsResponse;
import software.amazon.awssdk.services.cleanrooms.model.ListProtectedQueriesRequest;
import software.amazon.awssdk.services.cleanrooms.model.ListProtectedQueriesResponse;
import software.amazon.awssdk.services.cleanrooms.model.ListSchemasRequest;
import software.amazon.awssdk.services.cleanrooms.model.ListSchemasResponse;
import software.amazon.awssdk.services.cleanrooms.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.cleanrooms.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.cleanrooms.model.ResourceNotFoundException;
import software.amazon.awssdk.services.cleanrooms.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.cleanrooms.model.StartProtectedQueryRequest;
import software.amazon.awssdk.services.cleanrooms.model.StartProtectedQueryResponse;
import software.amazon.awssdk.services.cleanrooms.model.TagResourceRequest;
import software.amazon.awssdk.services.cleanrooms.model.TagResourceResponse;
import software.amazon.awssdk.services.cleanrooms.model.ThrottlingException;
import software.amazon.awssdk.services.cleanrooms.model.UntagResourceRequest;
import software.amazon.awssdk.services.cleanrooms.model.UntagResourceResponse;
import software.amazon.awssdk.services.cleanrooms.model.UpdateCollaborationRequest;
import software.amazon.awssdk.services.cleanrooms.model.UpdateCollaborationResponse;
import software.amazon.awssdk.services.cleanrooms.model.UpdateConfiguredTableAnalysisRuleRequest;
import software.amazon.awssdk.services.cleanrooms.model.UpdateConfiguredTableAnalysisRuleResponse;
import software.amazon.awssdk.services.cleanrooms.model.UpdateConfiguredTableAssociationRequest;
import software.amazon.awssdk.services.cleanrooms.model.UpdateConfiguredTableAssociationResponse;
import software.amazon.awssdk.services.cleanrooms.model.UpdateConfiguredTableRequest;
import software.amazon.awssdk.services.cleanrooms.model.UpdateConfiguredTableResponse;
import software.amazon.awssdk.services.cleanrooms.model.UpdateMembershipRequest;
import software.amazon.awssdk.services.cleanrooms.model.UpdateMembershipResponse;
import software.amazon.awssdk.services.cleanrooms.model.UpdateProtectedQueryRequest;
import software.amazon.awssdk.services.cleanrooms.model.UpdateProtectedQueryResponse;
import software.amazon.awssdk.services.cleanrooms.model.ValidationException;
import software.amazon.awssdk.services.cleanrooms.transform.BatchGetSchemaRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.CreateCollaborationRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.CreateConfiguredTableAnalysisRuleRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.CreateConfiguredTableAssociationRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.CreateConfiguredTableRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.CreateMembershipRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.DeleteCollaborationRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.DeleteConfiguredTableAnalysisRuleRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.DeleteConfiguredTableAssociationRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.DeleteConfiguredTableRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.DeleteMemberRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.DeleteMembershipRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.GetCollaborationRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.GetConfiguredTableAnalysisRuleRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.GetConfiguredTableAssociationRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.GetConfiguredTableRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.GetMembershipRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.GetProtectedQueryRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.GetSchemaAnalysisRuleRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.GetSchemaRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.ListCollaborationsRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.ListConfiguredTableAssociationsRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.ListConfiguredTablesRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.ListMembersRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.ListMembershipsRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.ListProtectedQueriesRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.ListSchemasRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.StartProtectedQueryRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.UpdateCollaborationRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.UpdateConfiguredTableAnalysisRuleRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.UpdateConfiguredTableAssociationRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.UpdateConfiguredTableRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.UpdateMembershipRequestMarshaller;
import software.amazon.awssdk.services.cleanrooms.transform.UpdateProtectedQueryRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CleanRoomsAsyncClient}.
 *
 * @see CleanRoomsAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCleanRoomsAsyncClient implements CleanRoomsAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCleanRoomsAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final CleanRoomsServiceClientConfiguration serviceClientConfiguration;

    protected DefaultCleanRoomsAsyncClient(CleanRoomsServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Retrieves multiple schemas by their identifiers.
     * </p>
     *
     * @param batchGetSchemaRequest
     * @return A Java Future containing the result of the BatchGetSchema operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.BatchGetSchema
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/BatchGetSchema" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetSchemaResponse> batchGetSchema(BatchGetSchemaRequest batchGetSchemaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetSchemaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetSchema");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetSchemaResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetSchemaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetSchemaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetSchemaRequest, BatchGetSchemaResponse>()
                            .withOperationName("BatchGetSchema")
                            .withMarshaller(new BatchGetSchemaRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(batchGetSchemaRequest));
            CompletableFuture<BatchGetSchemaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new collaboration.
     * </p>
     *
     * @param createCollaborationRequest
     * @return A Java Future containing the result of the CreateCollaboration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request denied because service quota has been exceeded.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.CreateCollaboration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/CreateCollaboration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCollaborationResponse> createCollaboration(
            CreateCollaborationRequest createCollaborationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCollaborationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCollaboration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCollaborationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateCollaborationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCollaborationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCollaborationRequest, CreateCollaborationResponse>()
                            .withOperationName("CreateCollaboration")
                            .withMarshaller(new CreateCollaborationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createCollaborationRequest));
            CompletableFuture<CreateCollaborationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new configured table resource.
     * </p>
     *
     * @param createConfiguredTableRequest
     * @return A Java Future containing the result of the CreateConfiguredTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ServiceQuotaExceededException Request denied because service quota has been exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.CreateConfiguredTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/CreateConfiguredTable"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConfiguredTableResponse> createConfiguredTable(
            CreateConfiguredTableRequest createConfiguredTableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConfiguredTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConfiguredTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConfiguredTableResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateConfiguredTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConfiguredTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConfiguredTableRequest, CreateConfiguredTableResponse>()
                            .withOperationName("CreateConfiguredTable")
                            .withMarshaller(new CreateConfiguredTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createConfiguredTableRequest));
            CompletableFuture<CreateConfiguredTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new analysis rule for a configured table. Currently, only one analysis rule can be created for a given
     * configured table.
     * </p>
     *
     * @param createConfiguredTableAnalysisRuleRequest
     * @return A Java Future containing the result of the CreateConfiguredTableAnalysisRule operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.CreateConfiguredTableAnalysisRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/CreateConfiguredTableAnalysisRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConfiguredTableAnalysisRuleResponse> createConfiguredTableAnalysisRule(
            CreateConfiguredTableAnalysisRuleRequest createConfiguredTableAnalysisRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createConfiguredTableAnalysisRuleRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConfiguredTableAnalysisRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConfiguredTableAnalysisRuleResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateConfiguredTableAnalysisRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConfiguredTableAnalysisRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConfiguredTableAnalysisRuleRequest, CreateConfiguredTableAnalysisRuleResponse>()
                            .withOperationName("CreateConfiguredTableAnalysisRule")
                            .withMarshaller(new CreateConfiguredTableAnalysisRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createConfiguredTableAnalysisRuleRequest));
            CompletableFuture<CreateConfiguredTableAnalysisRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a configured table association. A configured table association links a configured table with a
     * collaboration.
     * </p>
     *
     * @param createConfiguredTableAssociationRequest
     * @return A Java Future containing the result of the CreateConfiguredTableAssociation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ServiceQuotaExceededException Request denied because service quota has been exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.CreateConfiguredTableAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/CreateConfiguredTableAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConfiguredTableAssociationResponse> createConfiguredTableAssociation(
            CreateConfiguredTableAssociationRequest createConfiguredTableAssociationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createConfiguredTableAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConfiguredTableAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConfiguredTableAssociationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateConfiguredTableAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConfiguredTableAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConfiguredTableAssociationRequest, CreateConfiguredTableAssociationResponse>()
                            .withOperationName("CreateConfiguredTableAssociation")
                            .withMarshaller(new CreateConfiguredTableAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createConfiguredTableAssociationRequest));
            CompletableFuture<CreateConfiguredTableAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a membership for a specific collaboration identifier and joins the collaboration.
     * </p>
     *
     * @param createMembershipRequest
     * @return A Java Future containing the result of the CreateMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ServiceQuotaExceededException Request denied because service quota has been exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.CreateMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/CreateMembership" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMembershipResponse> createMembership(CreateMembershipRequest createMembershipRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMembershipRequest, CreateMembershipResponse>()
                            .withOperationName("CreateMembership")
                            .withMarshaller(new CreateMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createMembershipRequest));
            CompletableFuture<CreateMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a collaboration. It can only be called by the collaboration owner.
     * </p>
     *
     * @param deleteCollaborationRequest
     * @return A Java Future containing the result of the DeleteCollaboration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.DeleteCollaboration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/DeleteCollaboration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCollaborationResponse> deleteCollaboration(
            DeleteCollaborationRequest deleteCollaborationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCollaborationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCollaboration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCollaborationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCollaborationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCollaborationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCollaborationRequest, DeleteCollaborationResponse>()
                            .withOperationName("DeleteCollaboration")
                            .withMarshaller(new DeleteCollaborationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteCollaborationRequest));
            CompletableFuture<DeleteCollaborationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a configured table.
     * </p>
     *
     * @param deleteConfiguredTableRequest
     * @return A Java Future containing the result of the DeleteConfiguredTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.DeleteConfiguredTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/DeleteConfiguredTable"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConfiguredTableResponse> deleteConfiguredTable(
            DeleteConfiguredTableRequest deleteConfiguredTableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConfiguredTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConfiguredTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConfiguredTableResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteConfiguredTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConfiguredTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConfiguredTableRequest, DeleteConfiguredTableResponse>()
                            .withOperationName("DeleteConfiguredTable")
                            .withMarshaller(new DeleteConfiguredTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteConfiguredTableRequest));
            CompletableFuture<DeleteConfiguredTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a configured table analysis rule.
     * </p>
     *
     * @param deleteConfiguredTableAnalysisRuleRequest
     * @return A Java Future containing the result of the DeleteConfiguredTableAnalysisRule operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.DeleteConfiguredTableAnalysisRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/DeleteConfiguredTableAnalysisRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConfiguredTableAnalysisRuleResponse> deleteConfiguredTableAnalysisRule(
            DeleteConfiguredTableAnalysisRuleRequest deleteConfiguredTableAnalysisRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteConfiguredTableAnalysisRuleRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConfiguredTableAnalysisRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConfiguredTableAnalysisRuleResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteConfiguredTableAnalysisRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConfiguredTableAnalysisRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConfiguredTableAnalysisRuleRequest, DeleteConfiguredTableAnalysisRuleResponse>()
                            .withOperationName("DeleteConfiguredTableAnalysisRule")
                            .withMarshaller(new DeleteConfiguredTableAnalysisRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteConfiguredTableAnalysisRuleRequest));
            CompletableFuture<DeleteConfiguredTableAnalysisRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a configured table association.
     * </p>
     *
     * @param deleteConfiguredTableAssociationRequest
     * @return A Java Future containing the result of the DeleteConfiguredTableAssociation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.DeleteConfiguredTableAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/DeleteConfiguredTableAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteConfiguredTableAssociationResponse> deleteConfiguredTableAssociation(
            DeleteConfiguredTableAssociationRequest deleteConfiguredTableAssociationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteConfiguredTableAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConfiguredTableAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteConfiguredTableAssociationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteConfiguredTableAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteConfiguredTableAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteConfiguredTableAssociationRequest, DeleteConfiguredTableAssociationResponse>()
                            .withOperationName("DeleteConfiguredTableAssociation")
                            .withMarshaller(new DeleteConfiguredTableAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteConfiguredTableAssociationRequest));
            CompletableFuture<DeleteConfiguredTableAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified member from a collaboration. The removed member is placed in the Removed status and can't
     * interact with the collaboration. The removed member's data is inaccessible to active members of the
     * collaboration.
     * </p>
     *
     * @param deleteMemberRequest
     * @return A Java Future containing the result of the DeleteMember operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.DeleteMember
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/DeleteMember" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMemberResponse> deleteMember(DeleteMemberRequest deleteMemberRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMemberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMember");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMemberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteMemberResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMemberResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMemberRequest, DeleteMemberResponse>()
                            .withOperationName("DeleteMember").withMarshaller(new DeleteMemberRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteMemberRequest));
            CompletableFuture<DeleteMemberResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a specified membership. All resources under a membership must be deleted.
     * </p>
     *
     * @param deleteMembershipRequest
     * @return A Java Future containing the result of the DeleteMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.DeleteMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/DeleteMembership" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMembershipResponse> deleteMembership(DeleteMembershipRequest deleteMembershipRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMembershipRequest, DeleteMembershipResponse>()
                            .withOperationName("DeleteMembership")
                            .withMarshaller(new DeleteMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteMembershipRequest));
            CompletableFuture<DeleteMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns metadata about a collaboration.
     * </p>
     *
     * @param getCollaborationRequest
     * @return A Java Future containing the result of the GetCollaboration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.GetCollaboration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/GetCollaboration" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCollaborationResponse> getCollaboration(GetCollaborationRequest getCollaborationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCollaborationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCollaboration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCollaborationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCollaborationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCollaborationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCollaborationRequest, GetCollaborationResponse>()
                            .withOperationName("GetCollaboration")
                            .withMarshaller(new GetCollaborationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getCollaborationRequest));
            CompletableFuture<GetCollaborationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a configured table.
     * </p>
     *
     * @param getConfiguredTableRequest
     * @return A Java Future containing the result of the GetConfiguredTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.GetConfiguredTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/GetConfiguredTable" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetConfiguredTableResponse> getConfiguredTable(GetConfiguredTableRequest getConfiguredTableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConfiguredTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfiguredTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetConfiguredTableResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetConfiguredTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetConfiguredTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConfiguredTableRequest, GetConfiguredTableResponse>()
                            .withOperationName("GetConfiguredTable")
                            .withMarshaller(new GetConfiguredTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getConfiguredTableRequest));
            CompletableFuture<GetConfiguredTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a configured table analysis rule.
     * </p>
     *
     * @param getConfiguredTableAnalysisRuleRequest
     * @return A Java Future containing the result of the GetConfiguredTableAnalysisRule operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.GetConfiguredTableAnalysisRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/GetConfiguredTableAnalysisRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetConfiguredTableAnalysisRuleResponse> getConfiguredTableAnalysisRule(
            GetConfiguredTableAnalysisRuleRequest getConfiguredTableAnalysisRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getConfiguredTableAnalysisRuleRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfiguredTableAnalysisRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetConfiguredTableAnalysisRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetConfiguredTableAnalysisRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetConfiguredTableAnalysisRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConfiguredTableAnalysisRuleRequest, GetConfiguredTableAnalysisRuleResponse>()
                            .withOperationName("GetConfiguredTableAnalysisRule")
                            .withMarshaller(new GetConfiguredTableAnalysisRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getConfiguredTableAnalysisRuleRequest));
            CompletableFuture<GetConfiguredTableAnalysisRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a configured table association.
     * </p>
     *
     * @param getConfiguredTableAssociationRequest
     * @return A Java Future containing the result of the GetConfiguredTableAssociation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.GetConfiguredTableAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/GetConfiguredTableAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetConfiguredTableAssociationResponse> getConfiguredTableAssociation(
            GetConfiguredTableAssociationRequest getConfiguredTableAssociationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getConfiguredTableAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfiguredTableAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetConfiguredTableAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetConfiguredTableAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetConfiguredTableAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetConfiguredTableAssociationRequest, GetConfiguredTableAssociationResponse>()
                            .withOperationName("GetConfiguredTableAssociation")
                            .withMarshaller(new GetConfiguredTableAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getConfiguredTableAssociationRequest));
            CompletableFuture<GetConfiguredTableAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a specified membership for an identifier.
     * </p>
     *
     * @param getMembershipRequest
     * @return A Java Future containing the result of the GetMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.GetMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/GetMembership" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetMembershipResponse> getMembership(GetMembershipRequest getMembershipRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMembershipResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMembershipRequest, GetMembershipResponse>()
                            .withOperationName("GetMembership")
                            .withMarshaller(new GetMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getMembershipRequest));
            CompletableFuture<GetMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns query processing metadata.
     * </p>
     *
     * @param getProtectedQueryRequest
     * @return A Java Future containing the result of the GetProtectedQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.GetProtectedQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/GetProtectedQuery" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetProtectedQueryResponse> getProtectedQuery(GetProtectedQueryRequest getProtectedQueryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getProtectedQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProtectedQuery");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetProtectedQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetProtectedQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetProtectedQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetProtectedQueryRequest, GetProtectedQueryResponse>()
                            .withOperationName("GetProtectedQuery")
                            .withMarshaller(new GetProtectedQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getProtectedQueryRequest));
            CompletableFuture<GetProtectedQueryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the schema for a relation within a collaboration.
     * </p>
     *
     * @param getSchemaRequest
     * @return A Java Future containing the result of the GetSchema operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.GetSchema
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/GetSchema" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSchemaResponse> getSchema(GetSchemaRequest getSchemaRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSchemaRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSchema");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSchemaResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetSchemaResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSchemaResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSchemaRequest, GetSchemaResponse>().withOperationName("GetSchema")
                            .withMarshaller(new GetSchemaRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getSchemaRequest));
            CompletableFuture<GetSchemaResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a schema analysis rule.
     * </p>
     *
     * @param getSchemaAnalysisRuleRequest
     * @return A Java Future containing the result of the GetSchemaAnalysisRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.GetSchemaAnalysisRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/GetSchemaAnalysisRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSchemaAnalysisRuleResponse> getSchemaAnalysisRule(
            GetSchemaAnalysisRuleRequest getSchemaAnalysisRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getSchemaAnalysisRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetSchemaAnalysisRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSchemaAnalysisRuleResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSchemaAnalysisRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSchemaAnalysisRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSchemaAnalysisRuleRequest, GetSchemaAnalysisRuleResponse>()
                            .withOperationName("GetSchemaAnalysisRule")
                            .withMarshaller(new GetSchemaAnalysisRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getSchemaAnalysisRuleRequest));
            CompletableFuture<GetSchemaAnalysisRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists collaborations the caller owns, is active in, or has been invited to.
     * </p>
     *
     * @param listCollaborationsRequest
     * @return A Java Future containing the result of the ListCollaborations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.ListCollaborations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/ListCollaborations" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCollaborationsResponse> listCollaborations(ListCollaborationsRequest listCollaborationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCollaborationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCollaborations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCollaborationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCollaborationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCollaborationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCollaborationsRequest, ListCollaborationsResponse>()
                            .withOperationName("ListCollaborations")
                            .withMarshaller(new ListCollaborationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listCollaborationsRequest));
            CompletableFuture<ListCollaborationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists configured table associations for a membership.
     * </p>
     *
     * @param listConfiguredTableAssociationsRequest
     * @return A Java Future containing the result of the ListConfiguredTableAssociations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.ListConfiguredTableAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/ListConfiguredTableAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListConfiguredTableAssociationsResponse> listConfiguredTableAssociations(
            ListConfiguredTableAssociationsRequest listConfiguredTableAssociationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listConfiguredTableAssociationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConfiguredTableAssociations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConfiguredTableAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConfiguredTableAssociationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListConfiguredTableAssociationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConfiguredTableAssociationsRequest, ListConfiguredTableAssociationsResponse>()
                            .withOperationName("ListConfiguredTableAssociations")
                            .withMarshaller(new ListConfiguredTableAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listConfiguredTableAssociationsRequest));
            CompletableFuture<ListConfiguredTableAssociationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists configured tables.
     * </p>
     *
     * @param listConfiguredTablesRequest
     * @return A Java Future containing the result of the ListConfiguredTables operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.ListConfiguredTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/ListConfiguredTables"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListConfiguredTablesResponse> listConfiguredTables(
            ListConfiguredTablesRequest listConfiguredTablesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConfiguredTablesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConfiguredTables");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConfiguredTablesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConfiguredTablesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListConfiguredTablesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConfiguredTablesRequest, ListConfiguredTablesResponse>()
                            .withOperationName("ListConfiguredTables")
                            .withMarshaller(new ListConfiguredTablesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listConfiguredTablesRequest));
            CompletableFuture<ListConfiguredTablesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all members within a collaboration.
     * </p>
     *
     * @param listMembersRequest
     * @return A Java Future containing the result of the ListMembers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.ListMembers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/ListMembers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListMembersResponse> listMembers(ListMembersRequest listMembersRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMembersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMembers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMembersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListMembersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMembersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMembersRequest, ListMembersResponse>()
                            .withOperationName("ListMembers").withMarshaller(new ListMembersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listMembersRequest));
            CompletableFuture<ListMembersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all memberships resources within the caller's account.
     * </p>
     *
     * @param listMembershipsRequest
     * @return A Java Future containing the result of the ListMemberships operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.ListMemberships
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/ListMemberships" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMembershipsResponse> listMemberships(ListMembershipsRequest listMembershipsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMembershipsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMemberships");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMembershipsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListMembershipsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListMembershipsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMembershipsRequest, ListMembershipsResponse>()
                            .withOperationName("ListMemberships")
                            .withMarshaller(new ListMembershipsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listMembershipsRequest));
            CompletableFuture<ListMembershipsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists protected queries, sorted by the most recent query.
     * </p>
     *
     * @param listProtectedQueriesRequest
     * @return A Java Future containing the result of the ListProtectedQueries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.ListProtectedQueries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/ListProtectedQueries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListProtectedQueriesResponse> listProtectedQueries(
            ListProtectedQueriesRequest listProtectedQueriesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listProtectedQueriesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProtectedQueries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListProtectedQueriesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListProtectedQueriesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListProtectedQueriesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListProtectedQueriesRequest, ListProtectedQueriesResponse>()
                            .withOperationName("ListProtectedQueries")
                            .withMarshaller(new ListProtectedQueriesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listProtectedQueriesRequest));
            CompletableFuture<ListProtectedQueriesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the schemas for relations within a collaboration.
     * </p>
     *
     * @param listSchemasRequest
     * @return A Java Future containing the result of the ListSchemas operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.ListSchemas
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/ListSchemas" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListSchemasResponse> listSchemas(ListSchemasRequest listSchemasRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listSchemasRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSchemas");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListSchemasResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListSchemasResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListSchemasResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListSchemasRequest, ListSchemasResponse>()
                            .withOperationName("ListSchemas").withMarshaller(new ListSchemasRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listSchemasRequest));
            CompletableFuture<ListSchemasResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the tags that have been added to a resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a protected query that is started by Clean Rooms .
     * </p>
     *
     * @param startProtectedQueryRequest
     * @return A Java Future containing the result of the StartProtectedQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException Request denied because service quota has been exceeded.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.StartProtectedQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/StartProtectedQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<StartProtectedQueryResponse> startProtectedQuery(
            StartProtectedQueryRequest startProtectedQueryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startProtectedQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartProtectedQuery");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<StartProtectedQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, StartProtectedQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<StartProtectedQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<StartProtectedQueryRequest, StartProtectedQueryResponse>()
                            .withOperationName("StartProtectedQuery")
                            .withMarshaller(new StartProtectedQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(startProtectedQueryRequest));
            CompletableFuture<StartProtectedQueryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Tags a resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag or list of tags from a resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates collaboration metadata and can only be called by the collaboration owner.
     * </p>
     *
     * @param updateCollaborationRequest
     * @return A Java Future containing the result of the UpdateCollaboration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.UpdateCollaboration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/UpdateCollaboration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateCollaborationResponse> updateCollaboration(
            UpdateCollaborationRequest updateCollaborationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCollaborationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCollaboration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateCollaborationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateCollaborationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateCollaborationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateCollaborationRequest, UpdateCollaborationResponse>()
                            .withOperationName("UpdateCollaboration")
                            .withMarshaller(new UpdateCollaborationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateCollaborationRequest));
            CompletableFuture<UpdateCollaborationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a configured table.
     * </p>
     *
     * @param updateConfiguredTableRequest
     * @return A Java Future containing the result of the UpdateConfiguredTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.UpdateConfiguredTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/UpdateConfiguredTable"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateConfiguredTableResponse> updateConfiguredTable(
            UpdateConfiguredTableRequest updateConfiguredTableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConfiguredTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConfiguredTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateConfiguredTableResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateConfiguredTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateConfiguredTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateConfiguredTableRequest, UpdateConfiguredTableResponse>()
                            .withOperationName("UpdateConfiguredTable")
                            .withMarshaller(new UpdateConfiguredTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateConfiguredTableRequest));
            CompletableFuture<UpdateConfiguredTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a configured table analysis rule.
     * </p>
     *
     * @param updateConfiguredTableAnalysisRuleRequest
     * @return A Java Future containing the result of the UpdateConfiguredTableAnalysisRule operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.UpdateConfiguredTableAnalysisRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/UpdateConfiguredTableAnalysisRule"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateConfiguredTableAnalysisRuleResponse> updateConfiguredTableAnalysisRule(
            UpdateConfiguredTableAnalysisRuleRequest updateConfiguredTableAnalysisRuleRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateConfiguredTableAnalysisRuleRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConfiguredTableAnalysisRule");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateConfiguredTableAnalysisRuleResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateConfiguredTableAnalysisRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateConfiguredTableAnalysisRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateConfiguredTableAnalysisRuleRequest, UpdateConfiguredTableAnalysisRuleResponse>()
                            .withOperationName("UpdateConfiguredTableAnalysisRule")
                            .withMarshaller(new UpdateConfiguredTableAnalysisRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateConfiguredTableAnalysisRuleRequest));
            CompletableFuture<UpdateConfiguredTableAnalysisRuleResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a configured table association.
     * </p>
     *
     * @param updateConfiguredTableAssociationRequest
     * @return A Java Future containing the result of the UpdateConfiguredTableAssociation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.UpdateConfiguredTableAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/UpdateConfiguredTableAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateConfiguredTableAssociationResponse> updateConfiguredTableAssociation(
            UpdateConfiguredTableAssociationRequest updateConfiguredTableAssociationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateConfiguredTableAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConfiguredTableAssociation");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateConfiguredTableAssociationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateConfiguredTableAssociationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateConfiguredTableAssociationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateConfiguredTableAssociationRequest, UpdateConfiguredTableAssociationResponse>()
                            .withOperationName("UpdateConfiguredTableAssociation")
                            .withMarshaller(new UpdateConfiguredTableAssociationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateConfiguredTableAssociationRequest));
            CompletableFuture<UpdateConfiguredTableAssociationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a membership.
     * </p>
     *
     * @param updateMembershipRequest
     * @return A Java Future containing the result of the UpdateMembership operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.UpdateMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/UpdateMembership" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateMembershipResponse> updateMembership(UpdateMembershipRequest updateMembershipRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMembership");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateMembershipResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateMembershipResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateMembershipRequest, UpdateMembershipResponse>()
                            .withOperationName("UpdateMembership")
                            .withMarshaller(new UpdateMembershipRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateMembershipRequest));
            CompletableFuture<UpdateMembershipResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the processing of a currently running query.
     * </p>
     *
     * @param updateProtectedQueryRequest
     * @return A Java Future containing the result of the UpdateProtectedQuery operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException Updating or deleting a resource can cause an inconsistent state.</li>
     *         <li>ResourceNotFoundException Request references a resource which does not exist.</li>
     *         <li>InternalServerException Unexpected error during processing of request.</li>
     *         <li>ValidationException The input fails to satisfy the specified constraints.</li>
     *         <li>ThrottlingException Request was denied due to request throttling.</li>
     *         <li>AccessDeniedException Caller does not have sufficient access to perform this action.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CleanRoomsException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CleanRoomsAsyncClient.UpdateProtectedQuery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cleanrooms-2022-02-17/UpdateProtectedQuery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateProtectedQueryResponse> updateProtectedQuery(
            UpdateProtectedQueryRequest updateProtectedQueryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateProtectedQueryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CleanRooms");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProtectedQuery");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateProtectedQueryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateProtectedQueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateProtectedQueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateProtectedQueryRequest, UpdateProtectedQueryResponse>()
                            .withOperationName("UpdateProtectedQuery")
                            .withMarshaller(new UpdateProtectedQueryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateProtectedQueryRequest));
            CompletableFuture<UpdateProtectedQueryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final CleanRoomsServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CleanRoomsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
