/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrock;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.bedrock.internal.BedrockServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.bedrock.model.AccessDeniedException;
import software.amazon.awssdk.services.bedrock.model.BedrockException;
import software.amazon.awssdk.services.bedrock.model.ConflictException;
import software.amazon.awssdk.services.bedrock.model.CreateModelCustomizationJobRequest;
import software.amazon.awssdk.services.bedrock.model.CreateModelCustomizationJobResponse;
import software.amazon.awssdk.services.bedrock.model.CreateProvisionedModelThroughputRequest;
import software.amazon.awssdk.services.bedrock.model.CreateProvisionedModelThroughputResponse;
import software.amazon.awssdk.services.bedrock.model.DeleteCustomModelRequest;
import software.amazon.awssdk.services.bedrock.model.DeleteCustomModelResponse;
import software.amazon.awssdk.services.bedrock.model.DeleteModelInvocationLoggingConfigurationRequest;
import software.amazon.awssdk.services.bedrock.model.DeleteModelInvocationLoggingConfigurationResponse;
import software.amazon.awssdk.services.bedrock.model.DeleteProvisionedModelThroughputRequest;
import software.amazon.awssdk.services.bedrock.model.DeleteProvisionedModelThroughputResponse;
import software.amazon.awssdk.services.bedrock.model.GetCustomModelRequest;
import software.amazon.awssdk.services.bedrock.model.GetCustomModelResponse;
import software.amazon.awssdk.services.bedrock.model.GetFoundationModelRequest;
import software.amazon.awssdk.services.bedrock.model.GetFoundationModelResponse;
import software.amazon.awssdk.services.bedrock.model.GetModelCustomizationJobRequest;
import software.amazon.awssdk.services.bedrock.model.GetModelCustomizationJobResponse;
import software.amazon.awssdk.services.bedrock.model.GetModelInvocationLoggingConfigurationRequest;
import software.amazon.awssdk.services.bedrock.model.GetModelInvocationLoggingConfigurationResponse;
import software.amazon.awssdk.services.bedrock.model.GetProvisionedModelThroughputRequest;
import software.amazon.awssdk.services.bedrock.model.GetProvisionedModelThroughputResponse;
import software.amazon.awssdk.services.bedrock.model.InternalServerException;
import software.amazon.awssdk.services.bedrock.model.ListCustomModelsRequest;
import software.amazon.awssdk.services.bedrock.model.ListCustomModelsResponse;
import software.amazon.awssdk.services.bedrock.model.ListFoundationModelsRequest;
import software.amazon.awssdk.services.bedrock.model.ListFoundationModelsResponse;
import software.amazon.awssdk.services.bedrock.model.ListModelCustomizationJobsRequest;
import software.amazon.awssdk.services.bedrock.model.ListModelCustomizationJobsResponse;
import software.amazon.awssdk.services.bedrock.model.ListProvisionedModelThroughputsRequest;
import software.amazon.awssdk.services.bedrock.model.ListProvisionedModelThroughputsResponse;
import software.amazon.awssdk.services.bedrock.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.bedrock.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.bedrock.model.PutModelInvocationLoggingConfigurationRequest;
import software.amazon.awssdk.services.bedrock.model.PutModelInvocationLoggingConfigurationResponse;
import software.amazon.awssdk.services.bedrock.model.ResourceNotFoundException;
import software.amazon.awssdk.services.bedrock.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.bedrock.model.StopModelCustomizationJobRequest;
import software.amazon.awssdk.services.bedrock.model.StopModelCustomizationJobResponse;
import software.amazon.awssdk.services.bedrock.model.TagResourceRequest;
import software.amazon.awssdk.services.bedrock.model.TagResourceResponse;
import software.amazon.awssdk.services.bedrock.model.ThrottlingException;
import software.amazon.awssdk.services.bedrock.model.TooManyTagsException;
import software.amazon.awssdk.services.bedrock.model.UntagResourceRequest;
import software.amazon.awssdk.services.bedrock.model.UntagResourceResponse;
import software.amazon.awssdk.services.bedrock.model.UpdateProvisionedModelThroughputRequest;
import software.amazon.awssdk.services.bedrock.model.UpdateProvisionedModelThroughputResponse;
import software.amazon.awssdk.services.bedrock.model.ValidationException;
import software.amazon.awssdk.services.bedrock.transform.CreateModelCustomizationJobRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.CreateProvisionedModelThroughputRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.DeleteCustomModelRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.DeleteModelInvocationLoggingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.DeleteProvisionedModelThroughputRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.GetCustomModelRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.GetFoundationModelRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.GetModelCustomizationJobRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.GetModelInvocationLoggingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.GetProvisionedModelThroughputRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.ListCustomModelsRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.ListFoundationModelsRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.ListModelCustomizationJobsRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.ListProvisionedModelThroughputsRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.PutModelInvocationLoggingConfigurationRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.StopModelCustomizationJobRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.bedrock.transform.UpdateProvisionedModelThroughputRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link BedrockClient}.
 *
 * @see BedrockClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultBedrockClient implements BedrockClient {
    private static final Logger log = Logger.loggerFor(DefaultBedrockClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final BedrockServiceClientConfiguration serviceClientConfiguration;

    protected DefaultBedrockClient(BedrockServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a fine-tuning job to customize a base model.
     * </p>
     * <p>
     * You specify the base foundation model and the location of the training data. After the model-customization job
     * completes successfully, your custom model resource will be ready to use. Training data contains input and output
     * text for each record in a JSONL format. Optionally, you can specify validation data in the same format as the
     * training data. Bedrock returns validation loss metrics and output generations after the job completes.
     * </p>
     * <p>
     * Model-customization jobs are asynchronous and the completion time depends on the base model and the
     * training/validation data size. To monitor a job, use the <code>GetModelCustomizationJob</code> operation to
     * retrieve the job status.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/custom-models.html">Custom models</a> in the Bedrock
     * User Guide.
     * </p>
     *
     * @param createModelCustomizationJobRequest
     * @return Result of the CreateModelCustomizationJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. Check the ARN and try your request again.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws ConflictException
     *         Error occurred because of a conflict while performing an operation.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws TooManyTagsException
     *         The request contains more tags than can be associated with a resource (50 tags per resource). The maximum
     *         number of tags includes both existing tags and those included in your current request.
     * @throws ServiceQuotaExceededException
     *         The number of requests exceeds the service quota. Resubmit your request later.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.CreateModelCustomizationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/CreateModelCustomizationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateModelCustomizationJobResponse createModelCustomizationJob(
            CreateModelCustomizationJobRequest createModelCustomizationJobRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, ConflictException, InternalServerException, TooManyTagsException,
            ServiceQuotaExceededException, ThrottlingException, AwsServiceException, SdkClientException, BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateModelCustomizationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateModelCustomizationJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createModelCustomizationJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createModelCustomizationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateModelCustomizationJob");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateModelCustomizationJobRequest, CreateModelCustomizationJobResponse>()
                            .withOperationName("CreateModelCustomizationJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createModelCustomizationJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateModelCustomizationJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a provisioned throughput with dedicated capacity for a foundation model or a fine-tuned model.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Provisioned throughput</a> in
     * the Bedrock User Guide.
     * </p>
     *
     * @param createProvisionedModelThroughputRequest
     * @return Result of the CreateProvisionedModelThroughput operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. Check the ARN and try your request again.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws TooManyTagsException
     *         The request contains more tags than can be associated with a resource (50 tags per resource). The maximum
     *         number of tags includes both existing tags and those included in your current request.
     * @throws ServiceQuotaExceededException
     *         The number of requests exceeds the service quota. Resubmit your request later.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.CreateProvisionedModelThroughput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/CreateProvisionedModelThroughput"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateProvisionedModelThroughputResponse createProvisionedModelThroughput(
            CreateProvisionedModelThroughputRequest createProvisionedModelThroughputRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, InternalServerException, TooManyTagsException,
            ServiceQuotaExceededException, ThrottlingException, AwsServiceException, SdkClientException, BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateProvisionedModelThroughputResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateProvisionedModelThroughputResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createProvisionedModelThroughputRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createProvisionedModelThroughputRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateProvisionedModelThroughput");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateProvisionedModelThroughputRequest, CreateProvisionedModelThroughputResponse>()
                            .withOperationName("CreateProvisionedModelThroughput").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createProvisionedModelThroughputRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateProvisionedModelThroughputRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a custom model that you created earlier. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/custom-models.html">Custom models</a> in the Bedrock
     * User Guide.
     * </p>
     *
     * @param deleteCustomModelRequest
     * @return Result of the DeleteCustomModel operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. Check the ARN and try your request again.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws ConflictException
     *         Error occurred because of a conflict while performing an operation.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.DeleteCustomModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/DeleteCustomModel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteCustomModelResponse deleteCustomModel(DeleteCustomModelRequest deleteCustomModelRequest)
            throws ResourceNotFoundException, AccessDeniedException, ValidationException, ConflictException,
            InternalServerException, ThrottlingException, AwsServiceException, SdkClientException, BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteCustomModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteCustomModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCustomModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCustomModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCustomModel");

            return clientHandler.execute(new ClientExecutionParams<DeleteCustomModelRequest, DeleteCustomModelResponse>()
                    .withOperationName("DeleteCustomModel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteCustomModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteCustomModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Delete the invocation logging.
     * </p>
     *
     * @param deleteModelInvocationLoggingConfigurationRequest
     * @return Result of the DeleteModelInvocationLoggingConfiguration operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.DeleteModelInvocationLoggingConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/DeleteModelInvocationLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteModelInvocationLoggingConfigurationResponse deleteModelInvocationLoggingConfiguration(
            DeleteModelInvocationLoggingConfigurationRequest deleteModelInvocationLoggingConfigurationRequest)
            throws AccessDeniedException, InternalServerException, ThrottlingException, AwsServiceException, SdkClientException,
            BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteModelInvocationLoggingConfigurationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DeleteModelInvocationLoggingConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                deleteModelInvocationLoggingConfigurationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteModelInvocationLoggingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteModelInvocationLoggingConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteModelInvocationLoggingConfigurationRequest, DeleteModelInvocationLoggingConfigurationResponse>()
                            .withOperationName("DeleteModelInvocationLoggingConfiguration")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(deleteModelInvocationLoggingConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteModelInvocationLoggingConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a provisioned throughput. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Provisioned throughput</a> in
     * the Bedrock User Guide.
     * </p>
     *
     * @param deleteProvisionedModelThroughputRequest
     * @return Result of the DeleteProvisionedModelThroughput operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. Check the ARN and try your request again.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws ConflictException
     *         Error occurred because of a conflict while performing an operation.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.DeleteProvisionedModelThroughput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/DeleteProvisionedModelThroughput"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteProvisionedModelThroughputResponse deleteProvisionedModelThroughput(
            DeleteProvisionedModelThroughputRequest deleteProvisionedModelThroughputRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, ConflictException, InternalServerException, ThrottlingException,
            AwsServiceException, SdkClientException, BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteProvisionedModelThroughputResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteProvisionedModelThroughputResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteProvisionedModelThroughputRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteProvisionedModelThroughputRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteProvisionedModelThroughput");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteProvisionedModelThroughputRequest, DeleteProvisionedModelThroughputResponse>()
                            .withOperationName("DeleteProvisionedModelThroughput").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteProvisionedModelThroughputRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteProvisionedModelThroughputRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get the properties associated with a Bedrock custom model that you have created.For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/custom-models.html">Custom models</a> in the Bedrock
     * User Guide.
     * </p>
     *
     * @param getCustomModelRequest
     * @return Result of the GetCustomModel operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. Check the ARN and try your request again.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.GetCustomModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/GetCustomModel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetCustomModelResponse getCustomModel(GetCustomModelRequest getCustomModelRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, InternalServerException, ThrottlingException, AwsServiceException,
            SdkClientException, BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCustomModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetCustomModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCustomModelRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCustomModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCustomModel");

            return clientHandler.execute(new ClientExecutionParams<GetCustomModelRequest, GetCustomModelResponse>()
                    .withOperationName("GetCustomModel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getCustomModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetCustomModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get details about a Bedrock foundation model.
     * </p>
     *
     * @param getFoundationModelRequest
     * @return Result of the GetFoundationModel operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. Check the ARN and try your request again.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.GetFoundationModel
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/GetFoundationModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetFoundationModelResponse getFoundationModel(GetFoundationModelRequest getFoundationModelRequest)
            throws ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException,
            ThrottlingException, AwsServiceException, SdkClientException, BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetFoundationModelResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetFoundationModelResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getFoundationModelRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getFoundationModelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetFoundationModel");

            return clientHandler.execute(new ClientExecutionParams<GetFoundationModelRequest, GetFoundationModelResponse>()
                    .withOperationName("GetFoundationModel").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getFoundationModelRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetFoundationModelRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the properties associated with a model-customization job, including the status of the job. For more
     * information, see <a href="https://docs.aws.amazon.com/bedrock/latest/userguide/custom-models.html">Custom
     * models</a> in the Bedrock User Guide.
     * </p>
     *
     * @param getModelCustomizationJobRequest
     * @return Result of the GetModelCustomizationJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. Check the ARN and try your request again.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.GetModelCustomizationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/GetModelCustomizationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetModelCustomizationJobResponse getModelCustomizationJob(
            GetModelCustomizationJobRequest getModelCustomizationJobRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, InternalServerException, ThrottlingException, AwsServiceException,
            SdkClientException, BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetModelCustomizationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetModelCustomizationJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getModelCustomizationJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getModelCustomizationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetModelCustomizationJob");

            return clientHandler
                    .execute(new ClientExecutionParams<GetModelCustomizationJobRequest, GetModelCustomizationJobResponse>()
                            .withOperationName("GetModelCustomizationJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getModelCustomizationJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetModelCustomizationJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get the current configuration values for model invocation logging.
     * </p>
     *
     * @param getModelInvocationLoggingConfigurationRequest
     * @return Result of the GetModelInvocationLoggingConfiguration operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.GetModelInvocationLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/GetModelInvocationLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetModelInvocationLoggingConfigurationResponse getModelInvocationLoggingConfiguration(
            GetModelInvocationLoggingConfigurationRequest getModelInvocationLoggingConfigurationRequest)
            throws AccessDeniedException, InternalServerException, ThrottlingException, AwsServiceException, SdkClientException,
            BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetModelInvocationLoggingConfigurationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, GetModelInvocationLoggingConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getModelInvocationLoggingConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getModelInvocationLoggingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetModelInvocationLoggingConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<GetModelInvocationLoggingConfigurationRequest, GetModelInvocationLoggingConfigurationResponse>()
                            .withOperationName("GetModelInvocationLoggingConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(getModelInvocationLoggingConfigurationRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetModelInvocationLoggingConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get details for a provisioned throughput. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Provisioned throughput</a> in
     * the Bedrock User Guide.
     * </p>
     *
     * @param getProvisionedModelThroughputRequest
     * @return Result of the GetProvisionedModelThroughput operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. Check the ARN and try your request again.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.GetProvisionedModelThroughput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/GetProvisionedModelThroughput"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetProvisionedModelThroughputResponse getProvisionedModelThroughput(
            GetProvisionedModelThroughputRequest getProvisionedModelThroughputRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, InternalServerException, ThrottlingException, AwsServiceException,
            SdkClientException, BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetProvisionedModelThroughputResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetProvisionedModelThroughputResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getProvisionedModelThroughputRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getProvisionedModelThroughputRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetProvisionedModelThroughput");

            return clientHandler
                    .execute(new ClientExecutionParams<GetProvisionedModelThroughputRequest, GetProvisionedModelThroughputResponse>()
                            .withOperationName("GetProvisionedModelThroughput").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getProvisionedModelThroughputRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetProvisionedModelThroughputRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the custom models that you have created with the <code>CreateModelCustomizationJob</code>
     * operation.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/custom-models.html">Custom models</a> in the Bedrock
     * User Guide.
     * </p>
     *
     * @param listCustomModelsRequest
     * @return Result of the ListCustomModels operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.ListCustomModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/ListCustomModels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListCustomModelsResponse listCustomModels(ListCustomModelsRequest listCustomModelsRequest)
            throws AccessDeniedException, ValidationException, InternalServerException, ThrottlingException, AwsServiceException,
            SdkClientException, BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCustomModelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListCustomModelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCustomModelsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCustomModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCustomModels");

            return clientHandler.execute(new ClientExecutionParams<ListCustomModelsRequest, ListCustomModelsResponse>()
                    .withOperationName("ListCustomModels").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listCustomModelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListCustomModelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List of Bedrock foundation models that you can use. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/foundation-models.html">Foundation models</a> in the
     * Bedrock User Guide.
     * </p>
     *
     * @param listFoundationModelsRequest
     * @return Result of the ListFoundationModels operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.ListFoundationModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/ListFoundationModels" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListFoundationModelsResponse listFoundationModels(ListFoundationModelsRequest listFoundationModelsRequest)
            throws AccessDeniedException, ValidationException, InternalServerException, ThrottlingException, AwsServiceException,
            SdkClientException, BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListFoundationModelsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListFoundationModelsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listFoundationModelsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listFoundationModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFoundationModels");

            return clientHandler.execute(new ClientExecutionParams<ListFoundationModelsRequest, ListFoundationModelsResponse>()
                    .withOperationName("ListFoundationModels").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listFoundationModelsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListFoundationModelsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of model customization jobs that you have submitted. You can filter the jobs to return based on
     * one or more criteria.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/custom-models.html">Custom models</a> in the Bedrock
     * User Guide.
     * </p>
     *
     * @param listModelCustomizationJobsRequest
     * @return Result of the ListModelCustomizationJobs operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.ListModelCustomizationJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/ListModelCustomizationJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListModelCustomizationJobsResponse listModelCustomizationJobs(
            ListModelCustomizationJobsRequest listModelCustomizationJobsRequest) throws AccessDeniedException,
            ValidationException, InternalServerException, ThrottlingException, AwsServiceException, SdkClientException,
            BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListModelCustomizationJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListModelCustomizationJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listModelCustomizationJobsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listModelCustomizationJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListModelCustomizationJobs");

            return clientHandler
                    .execute(new ClientExecutionParams<ListModelCustomizationJobsRequest, ListModelCustomizationJobsResponse>()
                            .withOperationName("ListModelCustomizationJobs").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listModelCustomizationJobsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListModelCustomizationJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List the provisioned capacities. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Provisioned throughput</a> in
     * the Bedrock User Guide.
     * </p>
     *
     * @param listProvisionedModelThroughputsRequest
     * @return Result of the ListProvisionedModelThroughputs operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.ListProvisionedModelThroughputs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/ListProvisionedModelThroughputs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListProvisionedModelThroughputsResponse listProvisionedModelThroughputs(
            ListProvisionedModelThroughputsRequest listProvisionedModelThroughputsRequest) throws AccessDeniedException,
            ValidationException, InternalServerException, ThrottlingException, AwsServiceException, SdkClientException,
            BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListProvisionedModelThroughputsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListProvisionedModelThroughputsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listProvisionedModelThroughputsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listProvisionedModelThroughputsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListProvisionedModelThroughputs");

            return clientHandler
                    .execute(new ClientExecutionParams<ListProvisionedModelThroughputsRequest, ListProvisionedModelThroughputsResponse>()
                            .withOperationName("ListProvisionedModelThroughputs").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listProvisionedModelThroughputsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListProvisionedModelThroughputsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List the tags associated with the specified resource.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Tagging resources</a> in the
     * Bedrock User Guide.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. Check the ARN and try your request again.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, AccessDeniedException, ValidationException, InternalServerException,
            ThrottlingException, AwsServiceException, SdkClientException, BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Set the configuration values for model invocation logging.
     * </p>
     *
     * @param putModelInvocationLoggingConfigurationRequest
     * @return Result of the PutModelInvocationLoggingConfiguration operation returned by the service.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.PutModelInvocationLoggingConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/PutModelInvocationLoggingConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public PutModelInvocationLoggingConfigurationResponse putModelInvocationLoggingConfiguration(
            PutModelInvocationLoggingConfigurationRequest putModelInvocationLoggingConfigurationRequest)
            throws AccessDeniedException, ValidationException, InternalServerException, ThrottlingException, AwsServiceException,
            SdkClientException, BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutModelInvocationLoggingConfigurationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, PutModelInvocationLoggingConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putModelInvocationLoggingConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                putModelInvocationLoggingConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutModelInvocationLoggingConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<PutModelInvocationLoggingConfigurationRequest, PutModelInvocationLoggingConfigurationResponse>()
                            .withOperationName("PutModelInvocationLoggingConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(putModelInvocationLoggingConfigurationRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutModelInvocationLoggingConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops an active model customization job. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/custom-models.html">Custom models</a> in the Bedrock
     * User Guide.
     * </p>
     *
     * @param stopModelCustomizationJobRequest
     * @return Result of the StopModelCustomizationJob operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. Check the ARN and try your request again.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws ConflictException
     *         Error occurred because of a conflict while performing an operation.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.StopModelCustomizationJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/StopModelCustomizationJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StopModelCustomizationJobResponse stopModelCustomizationJob(
            StopModelCustomizationJobRequest stopModelCustomizationJobRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, ConflictException, InternalServerException, ThrottlingException,
            AwsServiceException, SdkClientException, BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopModelCustomizationJobResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StopModelCustomizationJobResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(stopModelCustomizationJobRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopModelCustomizationJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopModelCustomizationJob");

            return clientHandler
                    .execute(new ClientExecutionParams<StopModelCustomizationJobRequest, StopModelCustomizationJobResponse>()
                            .withOperationName("StopModelCustomizationJob").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(stopModelCustomizationJobRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StopModelCustomizationJobRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associate tags with a resource. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Tagging resources</a> in the
     * Bedrock User Guide.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. Check the ARN and try your request again.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws TooManyTagsException
     *         The request contains more tags than can be associated with a resource (50 tags per resource). The maximum
     *         number of tags includes both existing tags and those included in your current request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, InternalServerException, TooManyTagsException, ThrottlingException,
            AwsServiceException, SdkClientException, BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Remove one or more tags from a resource. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Tagging resources</a> in the
     * Bedrock User Guide.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. Check the ARN and try your request again.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, InternalServerException, ThrottlingException, AwsServiceException,
            SdkClientException, BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update a provisioned throughput. For more information, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/what-is-service.html">Provisioned throughput</a> in
     * the Bedrock User Guide.
     * </p>
     *
     * @param updateProvisionedModelThroughputRequest
     * @return Result of the UpdateProvisionedModelThroughput operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified resource ARN was not found. Check the ARN and try your request again.
     * @throws AccessDeniedException
     *         The request is denied because of missing access permissions.
     * @throws ValidationException
     *         Input validation failed. Check your request parameters and retry the request.
     * @throws InternalServerException
     *         An internal server error occurred. Retry your request.
     * @throws ThrottlingException
     *         The number of requests exceeds the limit. Resubmit your request later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BedrockException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BedrockClient.UpdateProvisionedModelThroughput
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bedrock-2023-04-20/UpdateProvisionedModelThroughput"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateProvisionedModelThroughputResponse updateProvisionedModelThroughput(
            UpdateProvisionedModelThroughputRequest updateProvisionedModelThroughputRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, InternalServerException, ThrottlingException, AwsServiceException,
            SdkClientException, BedrockException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateProvisionedModelThroughputResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateProvisionedModelThroughputResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateProvisionedModelThroughputRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateProvisionedModelThroughputRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Bedrock");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateProvisionedModelThroughput");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateProvisionedModelThroughputRequest, UpdateProvisionedModelThroughputResponse>()
                            .withOperationName("UpdateProvisionedModelThroughput").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateProvisionedModelThroughputRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateProvisionedModelThroughputRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        BedrockServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = BedrockServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(BedrockException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build());
    }

    @Override
    public final BedrockServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
