/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.auditmanager.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A metadata object that's associated with an assessment in Audit Manager.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AssessmentMetadataItem implements SdkPojo, Serializable,
        ToCopyableBuilder<AssessmentMetadataItem.Builder, AssessmentMetadataItem> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(AssessmentMetadataItem::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(AssessmentMetadataItem::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> COMPLIANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("complianceType").getter(getter(AssessmentMetadataItem::complianceType))
            .setter(setter(Builder::complianceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("complianceType").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(AssessmentMetadataItem::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<List<Role>> ROLES_FIELD = SdkField
            .<List<Role>> builder(MarshallingType.LIST)
            .memberName("roles")
            .getter(getter(AssessmentMetadataItem::roles))
            .setter(setter(Builder::roles))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roles").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Role> builder(MarshallingType.SDK_POJO)
                                            .constructor(Role::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Delegation>> DELEGATIONS_FIELD = SdkField
            .<List<Delegation>> builder(MarshallingType.LIST)
            .memberName("delegations")
            .getter(getter(AssessmentMetadataItem::delegations))
            .setter(setter(Builder::delegations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("delegations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Delegation> builder(MarshallingType.SDK_POJO)
                                            .constructor(Delegation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("creationTime").getter(getter(AssessmentMetadataItem::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("creationTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastUpdated").getter(getter(AssessmentMetadataItem::lastUpdated)).setter(setter(Builder::lastUpdated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastUpdated").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ID_FIELD,
            COMPLIANCE_TYPE_FIELD, STATUS_FIELD, ROLES_FIELD, DELEGATIONS_FIELD, CREATION_TIME_FIELD, LAST_UPDATED_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String id;

    private final String complianceType;

    private final String status;

    private final List<Role> roles;

    private final List<Delegation> delegations;

    private final Instant creationTime;

    private final Instant lastUpdated;

    private AssessmentMetadataItem(BuilderImpl builder) {
        this.name = builder.name;
        this.id = builder.id;
        this.complianceType = builder.complianceType;
        this.status = builder.status;
        this.roles = builder.roles;
        this.delegations = builder.delegations;
        this.creationTime = builder.creationTime;
        this.lastUpdated = builder.lastUpdated;
    }

    /**
     * <p>
     * The name of the assessment.
     * </p>
     * 
     * @return The name of the assessment.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The unique identifier for the assessment.
     * </p>
     * 
     * @return The unique identifier for the assessment.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The name of the compliance standard that's related to the assessment, such as PCI-DSS.
     * </p>
     * 
     * @return The name of the compliance standard that's related to the assessment, such as PCI-DSS.
     */
    public final String complianceType() {
        return complianceType;
    }

    /**
     * <p>
     * The current status of the assessment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AssessmentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the assessment.
     * @see AssessmentStatus
     */
    public final AssessmentStatus status() {
        return AssessmentStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the assessment.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AssessmentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the assessment.
     * @see AssessmentStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * For responses, this returns true if the service returned a value for the Roles property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasRoles() {
        return roles != null && !(roles instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The roles that are associated with the assessment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRoles} method.
     * </p>
     * 
     * @return The roles that are associated with the assessment.
     */
    public final List<Role> roles() {
        return roles;
    }

    /**
     * For responses, this returns true if the service returned a value for the Delegations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasDelegations() {
        return delegations != null && !(delegations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The delegations that are associated with the assessment.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDelegations} method.
     * </p>
     * 
     * @return The delegations that are associated with the assessment.
     */
    public final List<Delegation> delegations() {
        return delegations;
    }

    /**
     * <p>
     * Specifies when the assessment was created.
     * </p>
     * 
     * @return Specifies when the assessment was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The time of the most recent update.
     * </p>
     * 
     * @return The time of the most recent update.
     */
    public final Instant lastUpdated() {
        return lastUpdated;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(complianceType());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasRoles() ? roles() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasDelegations() ? delegations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdated());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AssessmentMetadataItem)) {
            return false;
        }
        AssessmentMetadataItem other = (AssessmentMetadataItem) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(id(), other.id())
                && Objects.equals(complianceType(), other.complianceType())
                && Objects.equals(statusAsString(), other.statusAsString()) && hasRoles() == other.hasRoles()
                && Objects.equals(roles(), other.roles()) && hasDelegations() == other.hasDelegations()
                && Objects.equals(delegations(), other.delegations()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(lastUpdated(), other.lastUpdated());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AssessmentMetadataItem").add("Name", name() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Id", id()).add("ComplianceType", complianceType() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Status", statusAsString()).add("Roles", roles() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Delegations", delegations() == null ? null : "*** Sensitive Data Redacted ***")
                .add("CreationTime", creationTime()).add("LastUpdated", lastUpdated()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "complianceType":
            return Optional.ofNullable(clazz.cast(complianceType()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "roles":
            return Optional.ofNullable(clazz.cast(roles()));
        case "delegations":
            return Optional.ofNullable(clazz.cast(delegations()));
        case "creationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "lastUpdated":
            return Optional.ofNullable(clazz.cast(lastUpdated()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AssessmentMetadataItem, T> g) {
        return obj -> g.apply((AssessmentMetadataItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AssessmentMetadataItem> {
        /**
         * <p>
         * The name of the assessment.
         * </p>
         * 
         * @param name
         *        The name of the assessment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The unique identifier for the assessment.
         * </p>
         * 
         * @param id
         *        The unique identifier for the assessment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The name of the compliance standard that's related to the assessment, such as PCI-DSS.
         * </p>
         * 
         * @param complianceType
         *        The name of the compliance standard that's related to the assessment, such as PCI-DSS.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder complianceType(String complianceType);

        /**
         * <p>
         * The current status of the assessment.
         * </p>
         * 
         * @param status
         *        The current status of the assessment.
         * @see AssessmentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AssessmentStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the assessment.
         * </p>
         * 
         * @param status
         *        The current status of the assessment.
         * @see AssessmentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AssessmentStatus
         */
        Builder status(AssessmentStatus status);

        /**
         * <p>
         * The roles that are associated with the assessment.
         * </p>
         * 
         * @param roles
         *        The roles that are associated with the assessment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roles(Collection<Role> roles);

        /**
         * <p>
         * The roles that are associated with the assessment.
         * </p>
         * 
         * @param roles
         *        The roles that are associated with the assessment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roles(Role... roles);

        /**
         * <p>
         * The roles that are associated with the assessment.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.auditmanager.model.Role.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.auditmanager.model.Role#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.auditmanager.model.Role.Builder#build()} is called immediately and its
         * result is passed to {@link #roles(List<Role>)}.
         * 
         * @param roles
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.auditmanager.model.Role.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #roles(java.util.Collection<Role>)
         */
        Builder roles(Consumer<Role.Builder>... roles);

        /**
         * <p>
         * The delegations that are associated with the assessment.
         * </p>
         * 
         * @param delegations
         *        The delegations that are associated with the assessment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder delegations(Collection<Delegation> delegations);

        /**
         * <p>
         * The delegations that are associated with the assessment.
         * </p>
         * 
         * @param delegations
         *        The delegations that are associated with the assessment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder delegations(Delegation... delegations);

        /**
         * <p>
         * The delegations that are associated with the assessment.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.auditmanager.model.Delegation.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.auditmanager.model.Delegation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.auditmanager.model.Delegation.Builder#build()} is called immediately
         * and its result is passed to {@link #delegations(List<Delegation>)}.
         * 
         * @param delegations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.auditmanager.model.Delegation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #delegations(java.util.Collection<Delegation>)
         */
        Builder delegations(Consumer<Delegation.Builder>... delegations);

        /**
         * <p>
         * Specifies when the assessment was created.
         * </p>
         * 
         * @param creationTime
         *        Specifies when the assessment was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The time of the most recent update.
         * </p>
         * 
         * @param lastUpdated
         *        The time of the most recent update.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdated(Instant lastUpdated);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String id;

        private String complianceType;

        private String status;

        private List<Role> roles = DefaultSdkAutoConstructList.getInstance();

        private List<Delegation> delegations = DefaultSdkAutoConstructList.getInstance();

        private Instant creationTime;

        private Instant lastUpdated;

        private BuilderImpl() {
        }

        private BuilderImpl(AssessmentMetadataItem model) {
            name(model.name);
            id(model.id);
            complianceType(model.complianceType);
            status(model.status);
            roles(model.roles);
            delegations(model.delegations);
            creationTime(model.creationTime);
            lastUpdated(model.lastUpdated);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getComplianceType() {
            return complianceType;
        }

        public final void setComplianceType(String complianceType) {
            this.complianceType = complianceType;
        }

        @Override
        public final Builder complianceType(String complianceType) {
            this.complianceType = complianceType;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AssessmentStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final List<Role.Builder> getRoles() {
            List<Role.Builder> result = RolesCopier.copyToBuilder(this.roles);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRoles(Collection<Role.BuilderImpl> roles) {
            this.roles = RolesCopier.copyFromBuilder(roles);
        }

        @Override
        public final Builder roles(Collection<Role> roles) {
            this.roles = RolesCopier.copy(roles);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder roles(Role... roles) {
            roles(Arrays.asList(roles));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder roles(Consumer<Role.Builder>... roles) {
            roles(Stream.of(roles).map(c -> Role.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<Delegation.Builder> getDelegations() {
            List<Delegation.Builder> result = DelegationsCopier.copyToBuilder(this.delegations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDelegations(Collection<Delegation.BuilderImpl> delegations) {
            this.delegations = DelegationsCopier.copyFromBuilder(delegations);
        }

        @Override
        public final Builder delegations(Collection<Delegation> delegations) {
            this.delegations = DelegationsCopier.copy(delegations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder delegations(Delegation... delegations) {
            delegations(Arrays.asList(delegations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder delegations(Consumer<Delegation.Builder>... delegations) {
            delegations(Stream.of(delegations).map(c -> Delegation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final Instant getLastUpdated() {
            return lastUpdated;
        }

        public final void setLastUpdated(Instant lastUpdated) {
            this.lastUpdated = lastUpdated;
        }

        @Override
        public final Builder lastUpdated(Instant lastUpdated) {
            this.lastUpdated = lastUpdated;
            return this;
        }

        @Override
        public AssessmentMetadataItem build() {
            return new AssessmentMetadataItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
