/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.auditmanager.paginators;

import java.util.Iterator;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.auditmanager.AuditManagerClient;
import software.amazon.awssdk.services.auditmanager.internal.UserAgentUtils;
import software.amazon.awssdk.services.auditmanager.model.ListKeywordsForDataSourceRequest;
import software.amazon.awssdk.services.auditmanager.model.ListKeywordsForDataSourceResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.auditmanager.AuditManagerClient#listKeywordsForDataSourcePaginator(software.amazon.awssdk.services.auditmanager.model.ListKeywordsForDataSourceRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.auditmanager.model.ListKeywordsForDataSourceResponse} that can be used to
 * iterate through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.auditmanager.paginators.ListKeywordsForDataSourceIterable responses = client.listKeywordsForDataSourcePaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.auditmanager.paginators.ListKeywordsForDataSourceIterable responses = client
 *             .listKeywordsForDataSourcePaginator(request);
 *     for (software.amazon.awssdk.services.auditmanager.model.ListKeywordsForDataSourceResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.auditmanager.paginators.ListKeywordsForDataSourceIterable responses = client.listKeywordsForDataSourcePaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the paginator.
 * It only limits the number of results in each page.</b>
 * </p>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listKeywordsForDataSource(software.amazon.awssdk.services.auditmanager.model.ListKeywordsForDataSourceRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListKeywordsForDataSourceIterable implements SdkIterable<ListKeywordsForDataSourceResponse> {
    private final AuditManagerClient client;

    private final ListKeywordsForDataSourceRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public ListKeywordsForDataSourceIterable(AuditManagerClient client, ListKeywordsForDataSourceRequest firstRequest) {
        this.client = client;
        this.firstRequest = UserAgentUtils.applyPaginatorUserAgent(firstRequest);
        this.nextPageFetcher = new ListKeywordsForDataSourceResponseFetcher();
    }

    @Override
    public Iterator<ListKeywordsForDataSourceResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    private class ListKeywordsForDataSourceResponseFetcher implements SyncPageFetcher<ListKeywordsForDataSourceResponse> {
        @Override
        public boolean hasNextPage(ListKeywordsForDataSourceResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public ListKeywordsForDataSourceResponse nextPage(ListKeywordsForDataSourceResponse previousPage) {
            if (previousPage == null) {
                return client.listKeywordsForDataSource(firstRequest);
            }
            return client.listKeywordsForDataSource(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
