/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.arczonalshift;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.arczonalshift.model.AccessDeniedException;
import software.amazon.awssdk.services.arczonalshift.model.ArcZonalShiftException;
import software.amazon.awssdk.services.arczonalshift.model.CancelZonalShiftRequest;
import software.amazon.awssdk.services.arczonalshift.model.CancelZonalShiftResponse;
import software.amazon.awssdk.services.arczonalshift.model.ConflictException;
import software.amazon.awssdk.services.arczonalshift.model.GetManagedResourceRequest;
import software.amazon.awssdk.services.arczonalshift.model.GetManagedResourceResponse;
import software.amazon.awssdk.services.arczonalshift.model.InternalServerException;
import software.amazon.awssdk.services.arczonalshift.model.ListManagedResourcesRequest;
import software.amazon.awssdk.services.arczonalshift.model.ListManagedResourcesResponse;
import software.amazon.awssdk.services.arczonalshift.model.ListZonalShiftsRequest;
import software.amazon.awssdk.services.arczonalshift.model.ListZonalShiftsResponse;
import software.amazon.awssdk.services.arczonalshift.model.ResourceNotFoundException;
import software.amazon.awssdk.services.arczonalshift.model.StartZonalShiftRequest;
import software.amazon.awssdk.services.arczonalshift.model.StartZonalShiftResponse;
import software.amazon.awssdk.services.arczonalshift.model.ThrottlingException;
import software.amazon.awssdk.services.arczonalshift.model.UpdateZonalShiftRequest;
import software.amazon.awssdk.services.arczonalshift.model.UpdateZonalShiftResponse;
import software.amazon.awssdk.services.arczonalshift.model.ValidationException;
import software.amazon.awssdk.services.arczonalshift.paginators.ListManagedResourcesIterable;
import software.amazon.awssdk.services.arczonalshift.paginators.ListZonalShiftsIterable;

/**
 * Service client for accessing AWS ARC - Zonal Shift. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * This is the API Reference Guide for the zonal shift feature of Amazon Route 53 Application Recovery Controller. This
 * guide is for developers who need detailed information about zonal shift API actions, data types, and errors.
 * </p>
 * <p>
 * Zonal shift is in preview release for Amazon Route 53 Application Recovery Controller and is subject to change.
 * </p>
 * <p>
 * Zonal shift in Route 53 ARC enables you to move traffic for a load balancer resource away from an Availability Zone.
 * Starting a zonal shift helps your application recover immediately, for example, from a developer's bad code
 * deployment or from an AWS infrastructure failure in a single Availability Zone, reducing the impact and time lost
 * from an issue in one zone.
 * </p>
 * <p>
 * Supported AWS resources are automatically registered with Route 53 ARC. Resources that are registered for zonal
 * shifts in Route 53 ARC are managed resources in Route 53 ARC. You can start a zonal shift for any managed resource in
 * your account in a Region. At this time, you can only start a zonal shift for Network Load Balancers and Application
 * Load Balancers with cross-zone load balancing turned off.
 * </p>
 * <p>
 * Zonal shifts are temporary. You must specify an expiration when you start a zonal shift, of up to three days
 * initially. If you want to still keep traffic away from an Availability Zone, you can update the zonal shift and set a
 * new expiration. You can also cancel a zonal shift, before it expires, for example, if you're ready to restore traffic
 * to the Availability Zone.
 * </p>
 * <p>
 * For more information about using zonal shift, see the <a
 * href="https://docs.aws.amazon.com/r53recovery/latest/dg/what-is-route53-recovery.html">Amazon Route 53 Application
 * Recovery Controller Developer Guide</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface ArcZonalShiftClient extends AwsClient {
    String SERVICE_NAME = "arc-zonal-shift";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "arc-zonal-shift";

    /**
     * <p>
     * Cancel a zonal shift in Amazon Route 53 Application Recovery Controller that you've started for a resource in
     * your AWS account in an AWS Region.
     * </p>
     *
     * @param cancelZonalShiftRequest
     * @return Result of the CancelZonalShift operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.CancelZonalShift
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/CancelZonalShift"
     *      target="_top">AWS API Documentation</a>
     */
    default CancelZonalShiftResponse cancelZonalShift(CancelZonalShiftRequest cancelZonalShiftRequest)
            throws InternalServerException, ConflictException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, ArcZonalShiftException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Cancel a zonal shift in Amazon Route 53 Application Recovery Controller that you've started for a resource in
     * your AWS account in an AWS Region.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CancelZonalShiftRequest.Builder} avoiding the need
     * to create one manually via {@link CancelZonalShiftRequest#builder()}
     * </p>
     *
     * @param cancelZonalShiftRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arczonalshift.model.CancelZonalShiftRequest.Builder} to create a
     *        request.
     * @return Result of the CancelZonalShift operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.CancelZonalShift
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/CancelZonalShift"
     *      target="_top">AWS API Documentation</a>
     */
    default CancelZonalShiftResponse cancelZonalShift(Consumer<CancelZonalShiftRequest.Builder> cancelZonalShiftRequest)
            throws InternalServerException, ConflictException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, ArcZonalShiftException {
        return cancelZonalShift(CancelZonalShiftRequest.builder().applyMutation(cancelZonalShiftRequest).build());
    }

    /**
     * <p>
     * Get information about a resource that's been registered for zonal shifts with Amazon Route 53 Application
     * Recovery Controller in this AWS Region. Resources that are registered for zonal shifts are managed resources in
     * Route 53 ARC.
     * </p>
     * <p>
     * At this time, you can only start a zonal shift for Network Load Balancers and Application Load Balancers with
     * cross-zone load balancing turned off.
     * </p>
     *
     * @param getManagedResourceRequest
     * @return Result of the GetManagedResource operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.GetManagedResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/GetManagedResource"
     *      target="_top">AWS API Documentation</a>
     */
    default GetManagedResourceResponse getManagedResource(GetManagedResourceRequest getManagedResourceRequest)
            throws InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException,
            AccessDeniedException, AwsServiceException, SdkClientException, ArcZonalShiftException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Get information about a resource that's been registered for zonal shifts with Amazon Route 53 Application
     * Recovery Controller in this AWS Region. Resources that are registered for zonal shifts are managed resources in
     * Route 53 ARC.
     * </p>
     * <p>
     * At this time, you can only start a zonal shift for Network Load Balancers and Application Load Balancers with
     * cross-zone load balancing turned off.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetManagedResourceRequest.Builder} avoiding the
     * need to create one manually via {@link GetManagedResourceRequest#builder()}
     * </p>
     *
     * @param getManagedResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arczonalshift.model.GetManagedResourceRequest.Builder} to create a
     *        request.
     * @return Result of the GetManagedResource operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.GetManagedResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/GetManagedResource"
     *      target="_top">AWS API Documentation</a>
     */
    default GetManagedResourceResponse getManagedResource(Consumer<GetManagedResourceRequest.Builder> getManagedResourceRequest)
            throws InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException,
            AccessDeniedException, AwsServiceException, SdkClientException, ArcZonalShiftException {
        return getManagedResource(GetManagedResourceRequest.builder().applyMutation(getManagedResourceRequest).build());
    }

    /**
     * <p>
     * Lists all the resources in your AWS account in this AWS Region that are managed for zonal shifts in Amazon Route
     * 53 Application Recovery Controller, and information about them. The information includes their Amazon Resource
     * Names (ARNs), the Availability Zones the resources are deployed in, and the resource name.
     * </p>
     *
     * @param listManagedResourcesRequest
     * @return Result of the ListManagedResources operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.ListManagedResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/ListManagedResources"
     *      target="_top">AWS API Documentation</a>
     */
    default ListManagedResourcesResponse listManagedResources(ListManagedResourcesRequest listManagedResourcesRequest)
            throws InternalServerException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, ArcZonalShiftException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all the resources in your AWS account in this AWS Region that are managed for zonal shifts in Amazon Route
     * 53 Application Recovery Controller, and information about them. The information includes their Amazon Resource
     * Names (ARNs), the Availability Zones the resources are deployed in, and the resource name.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListManagedResourcesRequest.Builder} avoiding the
     * need to create one manually via {@link ListManagedResourcesRequest#builder()}
     * </p>
     *
     * @param listManagedResourcesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arczonalshift.model.ListManagedResourcesRequest.Builder} to create
     *        a request.
     * @return Result of the ListManagedResources operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.ListManagedResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/ListManagedResources"
     *      target="_top">AWS API Documentation</a>
     */
    default ListManagedResourcesResponse listManagedResources(
            Consumer<ListManagedResourcesRequest.Builder> listManagedResourcesRequest) throws InternalServerException,
            ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException, SdkClientException,
            ArcZonalShiftException {
        return listManagedResources(ListManagedResourcesRequest.builder().applyMutation(listManagedResourcesRequest).build());
    }

    /**
     * <p>
     * Lists all the resources in your AWS account in this AWS Region that are managed for zonal shifts in Amazon Route
     * 53 Application Recovery Controller, and information about them. The information includes their Amazon Resource
     * Names (ARNs), the Availability Zones the resources are deployed in, and the resource name.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listManagedResources(software.amazon.awssdk.services.arczonalshift.model.ListManagedResourcesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arczonalshift.paginators.ListManagedResourcesIterable responses = client.listManagedResourcesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.arczonalshift.paginators.ListManagedResourcesIterable responses = client
     *             .listManagedResourcesPaginator(request);
     *     for (software.amazon.awssdk.services.arczonalshift.model.ListManagedResourcesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arczonalshift.paginators.ListManagedResourcesIterable responses = client.listManagedResourcesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listManagedResources(software.amazon.awssdk.services.arczonalshift.model.ListManagedResourcesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listManagedResourcesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.ListManagedResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/ListManagedResources"
     *      target="_top">AWS API Documentation</a>
     */
    default ListManagedResourcesIterable listManagedResourcesPaginator(ListManagedResourcesRequest listManagedResourcesRequest)
            throws InternalServerException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, ArcZonalShiftException {
        return new ListManagedResourcesIterable(this, listManagedResourcesRequest);
    }

    /**
     * <p>
     * Lists all the resources in your AWS account in this AWS Region that are managed for zonal shifts in Amazon Route
     * 53 Application Recovery Controller, and information about them. The information includes their Amazon Resource
     * Names (ARNs), the Availability Zones the resources are deployed in, and the resource name.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listManagedResources(software.amazon.awssdk.services.arczonalshift.model.ListManagedResourcesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arczonalshift.paginators.ListManagedResourcesIterable responses = client.listManagedResourcesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.arczonalshift.paginators.ListManagedResourcesIterable responses = client
     *             .listManagedResourcesPaginator(request);
     *     for (software.amazon.awssdk.services.arczonalshift.model.ListManagedResourcesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arczonalshift.paginators.ListManagedResourcesIterable responses = client.listManagedResourcesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listManagedResources(software.amazon.awssdk.services.arczonalshift.model.ListManagedResourcesRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListManagedResourcesRequest.Builder} avoiding the
     * need to create one manually via {@link ListManagedResourcesRequest#builder()}
     * </p>
     *
     * @param listManagedResourcesRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arczonalshift.model.ListManagedResourcesRequest.Builder} to create
     *        a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.ListManagedResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/ListManagedResources"
     *      target="_top">AWS API Documentation</a>
     */
    default ListManagedResourcesIterable listManagedResourcesPaginator(
            Consumer<ListManagedResourcesRequest.Builder> listManagedResourcesRequest) throws InternalServerException,
            ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException, SdkClientException,
            ArcZonalShiftException {
        return listManagedResourcesPaginator(ListManagedResourcesRequest.builder().applyMutation(listManagedResourcesRequest)
                .build());
    }

    /**
     * <p>
     * Lists all the active zonal shifts in Amazon Route 53 Application Recovery Controller in your AWS account in this
     * AWS Region.
     * </p>
     *
     * @param listZonalShiftsRequest
     * @return Result of the ListZonalShifts operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.ListZonalShifts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/ListZonalShifts"
     *      target="_top">AWS API Documentation</a>
     */
    default ListZonalShiftsResponse listZonalShifts(ListZonalShiftsRequest listZonalShiftsRequest)
            throws InternalServerException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, ArcZonalShiftException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all the active zonal shifts in Amazon Route 53 Application Recovery Controller in your AWS account in this
     * AWS Region.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListZonalShiftsRequest.Builder} avoiding the need
     * to create one manually via {@link ListZonalShiftsRequest#builder()}
     * </p>
     *
     * @param listZonalShiftsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arczonalshift.model.ListZonalShiftsRequest.Builder} to create a
     *        request.
     * @return Result of the ListZonalShifts operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.ListZonalShifts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/ListZonalShifts"
     *      target="_top">AWS API Documentation</a>
     */
    default ListZonalShiftsResponse listZonalShifts(Consumer<ListZonalShiftsRequest.Builder> listZonalShiftsRequest)
            throws InternalServerException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, ArcZonalShiftException {
        return listZonalShifts(ListZonalShiftsRequest.builder().applyMutation(listZonalShiftsRequest).build());
    }

    /**
     * <p>
     * Lists all the active zonal shifts in Amazon Route 53 Application Recovery Controller in your AWS account in this
     * AWS Region.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listZonalShifts(software.amazon.awssdk.services.arczonalshift.model.ListZonalShiftsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arczonalshift.paginators.ListZonalShiftsIterable responses = client.listZonalShiftsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.arczonalshift.paginators.ListZonalShiftsIterable responses = client
     *             .listZonalShiftsPaginator(request);
     *     for (software.amazon.awssdk.services.arczonalshift.model.ListZonalShiftsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arczonalshift.paginators.ListZonalShiftsIterable responses = client.listZonalShiftsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listZonalShifts(software.amazon.awssdk.services.arczonalshift.model.ListZonalShiftsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listZonalShiftsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.ListZonalShifts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/ListZonalShifts"
     *      target="_top">AWS API Documentation</a>
     */
    default ListZonalShiftsIterable listZonalShiftsPaginator(ListZonalShiftsRequest listZonalShiftsRequest)
            throws InternalServerException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, ArcZonalShiftException {
        return new ListZonalShiftsIterable(this, listZonalShiftsRequest);
    }

    /**
     * <p>
     * Lists all the active zonal shifts in Amazon Route 53 Application Recovery Controller in your AWS account in this
     * AWS Region.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listZonalShifts(software.amazon.awssdk.services.arczonalshift.model.ListZonalShiftsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arczonalshift.paginators.ListZonalShiftsIterable responses = client.listZonalShiftsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.arczonalshift.paginators.ListZonalShiftsIterable responses = client
     *             .listZonalShiftsPaginator(request);
     *     for (software.amazon.awssdk.services.arczonalshift.model.ListZonalShiftsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.arczonalshift.paginators.ListZonalShiftsIterable responses = client.listZonalShiftsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listZonalShifts(software.amazon.awssdk.services.arczonalshift.model.ListZonalShiftsRequest)}
     * operation.</b>
     * </p>
     * <p>
     * This is a convenience which creates an instance of the {@link ListZonalShiftsRequest.Builder} avoiding the need
     * to create one manually via {@link ListZonalShiftsRequest#builder()}
     * </p>
     *
     * @param listZonalShiftsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arczonalshift.model.ListZonalShiftsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.ListZonalShifts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/ListZonalShifts"
     *      target="_top">AWS API Documentation</a>
     */
    default ListZonalShiftsIterable listZonalShiftsPaginator(Consumer<ListZonalShiftsRequest.Builder> listZonalShiftsRequest)
            throws InternalServerException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, ArcZonalShiftException {
        return listZonalShiftsPaginator(ListZonalShiftsRequest.builder().applyMutation(listZonalShiftsRequest).build());
    }

    /**
     * <p>
     * You start a zonal shift to temporarily move load balancer traffic away from an Availability Zone in a AWS Region,
     * to help your application recover immediately, for example, from a developer's bad code deployment or from an AWS
     * infrastructure failure in a single Availability Zone. You can start a zonal shift in Route 53 ARC only for
     * managed resources in your account in an AWS Region. Resources are automatically registered with Route 53 ARC by
     * AWS services.
     * </p>
     * <p>
     * At this time, you can only start a zonal shift for Network Load Balancers and Application Load Balancers with
     * cross-zone load balancing turned off.
     * </p>
     * <p>
     * When you start a zonal shift, traffic for the resource is no longer routed to the Availability Zone. The zonal
     * shift is created immediately in Route 53 ARC. However, it can take a short time, typically up to a few minutes,
     * for existing, in-progress connections in the Availability Zone to complete.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.html">Zonal
     * shift</a> in the Amazon Route 53 Application Recovery Controller Developer Guide.
     * </p>
     *
     * @param startZonalShiftRequest
     * @return Result of the StartZonalShift operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.StartZonalShift
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/StartZonalShift"
     *      target="_top">AWS API Documentation</a>
     */
    default StartZonalShiftResponse startZonalShift(StartZonalShiftRequest startZonalShiftRequest)
            throws InternalServerException, ConflictException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, ArcZonalShiftException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * You start a zonal shift to temporarily move load balancer traffic away from an Availability Zone in a AWS Region,
     * to help your application recover immediately, for example, from a developer's bad code deployment or from an AWS
     * infrastructure failure in a single Availability Zone. You can start a zonal shift in Route 53 ARC only for
     * managed resources in your account in an AWS Region. Resources are automatically registered with Route 53 ARC by
     * AWS services.
     * </p>
     * <p>
     * At this time, you can only start a zonal shift for Network Load Balancers and Application Load Balancers with
     * cross-zone load balancing turned off.
     * </p>
     * <p>
     * When you start a zonal shift, traffic for the resource is no longer routed to the Availability Zone. The zonal
     * shift is created immediately in Route 53 ARC. However, it can take a short time, typically up to a few minutes,
     * for existing, in-progress connections in the Availability Zone to complete.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.html">Zonal
     * shift</a> in the Amazon Route 53 Application Recovery Controller Developer Guide.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartZonalShiftRequest.Builder} avoiding the need
     * to create one manually via {@link StartZonalShiftRequest#builder()}
     * </p>
     *
     * @param startZonalShiftRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arczonalshift.model.StartZonalShiftRequest.Builder} to create a
     *        request.
     * @return Result of the StartZonalShift operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.StartZonalShift
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/StartZonalShift"
     *      target="_top">AWS API Documentation</a>
     */
    default StartZonalShiftResponse startZonalShift(Consumer<StartZonalShiftRequest.Builder> startZonalShiftRequest)
            throws InternalServerException, ConflictException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, ArcZonalShiftException {
        return startZonalShift(StartZonalShiftRequest.builder().applyMutation(startZonalShiftRequest).build());
    }

    /**
     * <p>
     * Update an active zonal shift in Amazon Route 53 Application Recovery Controller in your AWS account. You can
     * update a zonal shift to set a new expiration, or edit or replace the comment for the zonal shift.
     * </p>
     *
     * @param updateZonalShiftRequest
     * @return Result of the UpdateZonalShift operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.UpdateZonalShift
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/UpdateZonalShift"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdateZonalShiftResponse updateZonalShift(UpdateZonalShiftRequest updateZonalShiftRequest)
            throws InternalServerException, ConflictException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, ArcZonalShiftException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Update an active zonal shift in Amazon Route 53 Application Recovery Controller in your AWS account. You can
     * update a zonal shift to set a new expiration, or edit or replace the comment for the zonal shift.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateZonalShiftRequest.Builder} avoiding the need
     * to create one manually via {@link UpdateZonalShiftRequest#builder()}
     * </p>
     *
     * @param updateZonalShiftRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.arczonalshift.model.UpdateZonalShiftRequest.Builder} to create a
     *        request.
     * @return Result of the UpdateZonalShift operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.UpdateZonalShift
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/UpdateZonalShift"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdateZonalShiftResponse updateZonalShift(Consumer<UpdateZonalShiftRequest.Builder> updateZonalShiftRequest)
            throws InternalServerException, ConflictException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, ArcZonalShiftException {
        return updateZonalShift(UpdateZonalShiftRequest.builder().applyMutation(updateZonalShiftRequest).build());
    }

    /**
     * Create a {@link ArcZonalShiftClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static ArcZonalShiftClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link ArcZonalShiftClient}.
     */
    static ArcZonalShiftClientBuilder builder() {
        return new DefaultArcZonalShiftClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default ArcZonalShiftServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
