/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.arczonalshift;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.arczonalshift.model.AccessDeniedException;
import software.amazon.awssdk.services.arczonalshift.model.ArcZonalShiftException;
import software.amazon.awssdk.services.arczonalshift.model.CancelZonalShiftRequest;
import software.amazon.awssdk.services.arczonalshift.model.CancelZonalShiftResponse;
import software.amazon.awssdk.services.arczonalshift.model.ConflictException;
import software.amazon.awssdk.services.arczonalshift.model.GetManagedResourceRequest;
import software.amazon.awssdk.services.arczonalshift.model.GetManagedResourceResponse;
import software.amazon.awssdk.services.arczonalshift.model.InternalServerException;
import software.amazon.awssdk.services.arczonalshift.model.ListManagedResourcesRequest;
import software.amazon.awssdk.services.arczonalshift.model.ListManagedResourcesResponse;
import software.amazon.awssdk.services.arczonalshift.model.ListZonalShiftsRequest;
import software.amazon.awssdk.services.arczonalshift.model.ListZonalShiftsResponse;
import software.amazon.awssdk.services.arczonalshift.model.ResourceNotFoundException;
import software.amazon.awssdk.services.arczonalshift.model.StartZonalShiftRequest;
import software.amazon.awssdk.services.arczonalshift.model.StartZonalShiftResponse;
import software.amazon.awssdk.services.arczonalshift.model.ThrottlingException;
import software.amazon.awssdk.services.arczonalshift.model.UpdateZonalShiftRequest;
import software.amazon.awssdk.services.arczonalshift.model.UpdateZonalShiftResponse;
import software.amazon.awssdk.services.arczonalshift.model.ValidationException;
import software.amazon.awssdk.services.arczonalshift.transform.CancelZonalShiftRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.GetManagedResourceRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.ListManagedResourcesRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.ListZonalShiftsRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.StartZonalShiftRequestMarshaller;
import software.amazon.awssdk.services.arczonalshift.transform.UpdateZonalShiftRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ArcZonalShiftClient}.
 *
 * @see ArcZonalShiftClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultArcZonalShiftClient implements ArcZonalShiftClient {
    private static final Logger log = Logger.loggerFor(DefaultArcZonalShiftClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final ArcZonalShiftServiceClientConfiguration serviceClientConfiguration;

    protected DefaultArcZonalShiftClient(ArcZonalShiftServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Cancel a zonal shift in Amazon Route 53 Application Recovery Controller that you've started for a resource in
     * your AWS account in an AWS Region.
     * </p>
     *
     * @param cancelZonalShiftRequest
     * @return Result of the CancelZonalShift operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.CancelZonalShift
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/CancelZonalShift"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CancelZonalShiftResponse cancelZonalShift(CancelZonalShiftRequest cancelZonalShiftRequest)
            throws InternalServerException, ConflictException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, ArcZonalShiftException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelZonalShiftResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CancelZonalShiftResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelZonalShiftRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelZonalShift");

            return clientHandler.execute(new ClientExecutionParams<CancelZonalShiftRequest, CancelZonalShiftResponse>()
                    .withOperationName("CancelZonalShift").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(cancelZonalShiftRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CancelZonalShiftRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Get information about a resource that's been registered for zonal shifts with Amazon Route 53 Application
     * Recovery Controller in this AWS Region. Resources that are registered for zonal shifts are managed resources in
     * Route 53 ARC.
     * </p>
     * <p>
     * At this time, you can only start a zonal shift for Network Load Balancers and Application Load Balancers with
     * cross-zone load balancing turned off.
     * </p>
     *
     * @param getManagedResourceRequest
     * @return Result of the GetManagedResource operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.GetManagedResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/GetManagedResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetManagedResourceResponse getManagedResource(GetManagedResourceRequest getManagedResourceRequest)
            throws InternalServerException, ResourceNotFoundException, ThrottlingException, ValidationException,
            AccessDeniedException, AwsServiceException, SdkClientException, ArcZonalShiftException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetManagedResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetManagedResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getManagedResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetManagedResource");

            return clientHandler.execute(new ClientExecutionParams<GetManagedResourceRequest, GetManagedResourceResponse>()
                    .withOperationName("GetManagedResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getManagedResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetManagedResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the resources in your AWS account in this AWS Region that are managed for zonal shifts in Amazon Route
     * 53 Application Recovery Controller, and information about them. The information includes their Amazon Resource
     * Names (ARNs), the Availability Zones the resources are deployed in, and the resource name.
     * </p>
     *
     * @param listManagedResourcesRequest
     * @return Result of the ListManagedResources operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.ListManagedResources
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/ListManagedResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListManagedResourcesResponse listManagedResources(ListManagedResourcesRequest listManagedResourcesRequest)
            throws InternalServerException, ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException,
            SdkClientException, ArcZonalShiftException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListManagedResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListManagedResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listManagedResourcesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListManagedResources");

            return clientHandler.execute(new ClientExecutionParams<ListManagedResourcesRequest, ListManagedResourcesResponse>()
                    .withOperationName("ListManagedResources").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listManagedResourcesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListManagedResourcesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all the active zonal shifts in Amazon Route 53 Application Recovery Controller in your AWS account in this
     * AWS Region.
     * </p>
     *
     * @param listZonalShiftsRequest
     * @return Result of the ListZonalShifts operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.ListZonalShifts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/ListZonalShifts"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListZonalShiftsResponse listZonalShifts(ListZonalShiftsRequest listZonalShiftsRequest) throws InternalServerException,
            ThrottlingException, ValidationException, AccessDeniedException, AwsServiceException, SdkClientException,
            ArcZonalShiftException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListZonalShiftsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListZonalShiftsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listZonalShiftsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListZonalShifts");

            return clientHandler.execute(new ClientExecutionParams<ListZonalShiftsRequest, ListZonalShiftsResponse>()
                    .withOperationName("ListZonalShifts").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listZonalShiftsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListZonalShiftsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * You start a zonal shift to temporarily move load balancer traffic away from an Availability Zone in a AWS Region,
     * to help your application recover immediately, for example, from a developer's bad code deployment or from an AWS
     * infrastructure failure in a single Availability Zone. You can start a zonal shift in Route 53 ARC only for
     * managed resources in your account in an AWS Region. Resources are automatically registered with Route 53 ARC by
     * AWS services.
     * </p>
     * <p>
     * At this time, you can only start a zonal shift for Network Load Balancers and Application Load Balancers with
     * cross-zone load balancing turned off.
     * </p>
     * <p>
     * When you start a zonal shift, traffic for the resource is no longer routed to the Availability Zone. The zonal
     * shift is created immediately in Route 53 ARC. However, it can take a short time, typically up to a few minutes,
     * for existing, in-progress connections in the Availability Zone to complete.
     * </p>
     * <p>
     * For more information, see <a href="https://docs.aws.amazon.com/r53recovery/latest/dg/arc-zonal-shift.html">Zonal
     * shift</a> in the Amazon Route 53 Application Recovery Controller Developer Guide.
     * </p>
     *
     * @param startZonalShiftRequest
     * @return Result of the StartZonalShift operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.StartZonalShift
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/StartZonalShift"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartZonalShiftResponse startZonalShift(StartZonalShiftRequest startZonalShiftRequest) throws InternalServerException,
            ConflictException, ResourceNotFoundException, ThrottlingException, ValidationException, AccessDeniedException,
            AwsServiceException, SdkClientException, ArcZonalShiftException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartZonalShiftResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartZonalShiftResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startZonalShiftRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartZonalShift");

            return clientHandler.execute(new ClientExecutionParams<StartZonalShiftRequest, StartZonalShiftResponse>()
                    .withOperationName("StartZonalShift").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startZonalShiftRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartZonalShiftRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Update an active zonal shift in Amazon Route 53 Application Recovery Controller in your AWS account. You can
     * update a zonal shift to set a new expiration, or edit or replace the comment for the zonal shift.
     * </p>
     *
     * @param updateZonalShiftRequest
     * @return Result of the UpdateZonalShift operation returned by the service.
     * @throws InternalServerException
     *         There was an internal server error.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ResourceNotFoundException
     *         The input requested a resource that was not found.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws ValidationException
     *         The input fails to satisfy the constraints specified by an AWS service.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ArcZonalShiftException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ArcZonalShiftClient.UpdateZonalShift
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/arc-zonal-shift-2022-10-30/UpdateZonalShift"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateZonalShiftResponse updateZonalShift(UpdateZonalShiftRequest updateZonalShiftRequest)
            throws InternalServerException, ConflictException, ResourceNotFoundException, ThrottlingException,
            ValidationException, AccessDeniedException, AwsServiceException, SdkClientException, ArcZonalShiftException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateZonalShiftResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateZonalShiftResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateZonalShiftRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ARC Zonal Shift");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateZonalShift");

            return clientHandler.execute(new ClientExecutionParams<UpdateZonalShiftRequest, UpdateZonalShiftResponse>()
                    .withOperationName("UpdateZonalShift").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateZonalShiftRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateZonalShiftRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ArcZonalShiftException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build());
    }

    @Override
    public final ArcZonalShiftServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
