/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appstream.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an app block builder.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AppBlockBuilder implements SdkPojo, Serializable, ToCopyableBuilder<AppBlockBuilder.Builder, AppBlockBuilder> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(AppBlockBuilder::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(AppBlockBuilder::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DISPLAY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DisplayName").getter(getter(AppBlockBuilder::displayName)).setter(setter(Builder::displayName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisplayName").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(AppBlockBuilder::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> PLATFORM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Platform").getter(getter(AppBlockBuilder::platformAsString)).setter(setter(Builder::platform))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Platform").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceType").getter(getter(AppBlockBuilder::instanceType)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<Boolean> ENABLE_DEFAULT_INTERNET_ACCESS_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnableDefaultInternetAccess")
            .getter(getter(AppBlockBuilder::enableDefaultInternetAccess))
            .setter(setter(Builder::enableDefaultInternetAccess))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableDefaultInternetAccess")
                    .build()).build();

    private static final SdkField<String> IAM_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IamRoleArn").getter(getter(AppBlockBuilder::iamRoleArn)).setter(setter(Builder::iamRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IamRoleArn").build()).build();

    private static final SdkField<VpcConfig> VPC_CONFIG_FIELD = SdkField.<VpcConfig> builder(MarshallingType.SDK_POJO)
            .memberName("VpcConfig").getter(getter(AppBlockBuilder::vpcConfig)).setter(setter(Builder::vpcConfig))
            .constructor(VpcConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcConfig").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(AppBlockBuilder::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTime").getter(getter(AppBlockBuilder::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime").build()).build();

    private static final SdkField<List<ResourceError>> APP_BLOCK_BUILDER_ERRORS_FIELD = SdkField
            .<List<ResourceError>> builder(MarshallingType.LIST)
            .memberName("AppBlockBuilderErrors")
            .getter(getter(AppBlockBuilder::appBlockBuilderErrors))
            .setter(setter(Builder::appBlockBuilderErrors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AppBlockBuilderErrors").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceError> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceError::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<AppBlockBuilderStateChangeReason> STATE_CHANGE_REASON_FIELD = SdkField
            .<AppBlockBuilderStateChangeReason> builder(MarshallingType.SDK_POJO).memberName("StateChangeReason")
            .getter(getter(AppBlockBuilder::stateChangeReason)).setter(setter(Builder::stateChangeReason))
            .constructor(AppBlockBuilderStateChangeReason::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateChangeReason").build()).build();

    private static final SdkField<List<AccessEndpoint>> ACCESS_ENDPOINTS_FIELD = SdkField
            .<List<AccessEndpoint>> builder(MarshallingType.LIST)
            .memberName("AccessEndpoints")
            .getter(getter(AppBlockBuilder::accessEndpoints))
            .setter(setter(Builder::accessEndpoints))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessEndpoints").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AccessEndpoint> builder(MarshallingType.SDK_POJO)
                                            .constructor(AccessEndpoint::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, NAME_FIELD,
            DISPLAY_NAME_FIELD, DESCRIPTION_FIELD, PLATFORM_FIELD, INSTANCE_TYPE_FIELD, ENABLE_DEFAULT_INTERNET_ACCESS_FIELD,
            IAM_ROLE_ARN_FIELD, VPC_CONFIG_FIELD, STATE_FIELD, CREATED_TIME_FIELD, APP_BLOCK_BUILDER_ERRORS_FIELD,
            STATE_CHANGE_REASON_FIELD, ACCESS_ENDPOINTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final String name;

    private final String displayName;

    private final String description;

    private final String platform;

    private final String instanceType;

    private final Boolean enableDefaultInternetAccess;

    private final String iamRoleArn;

    private final VpcConfig vpcConfig;

    private final String state;

    private final Instant createdTime;

    private final List<ResourceError> appBlockBuilderErrors;

    private final AppBlockBuilderStateChangeReason stateChangeReason;

    private final List<AccessEndpoint> accessEndpoints;

    private AppBlockBuilder(BuilderImpl builder) {
        this.arn = builder.arn;
        this.name = builder.name;
        this.displayName = builder.displayName;
        this.description = builder.description;
        this.platform = builder.platform;
        this.instanceType = builder.instanceType;
        this.enableDefaultInternetAccess = builder.enableDefaultInternetAccess;
        this.iamRoleArn = builder.iamRoleArn;
        this.vpcConfig = builder.vpcConfig;
        this.state = builder.state;
        this.createdTime = builder.createdTime;
        this.appBlockBuilderErrors = builder.appBlockBuilderErrors;
        this.stateChangeReason = builder.stateChangeReason;
        this.accessEndpoints = builder.accessEndpoints;
    }

    /**
     * <p>
     * The ARN of the app block builder.
     * </p>
     * 
     * @return The ARN of the app block builder.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The name of the app block builder.
     * </p>
     * 
     * @return The name of the app block builder.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The display name of the app block builder.
     * </p>
     * 
     * @return The display name of the app block builder.
     */
    public final String displayName() {
        return displayName;
    }

    /**
     * <p>
     * The description of the app block builder.
     * </p>
     * 
     * @return The description of the app block builder.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The platform of the app block builder.
     * </p>
     * <p>
     * <code>WINDOWS_SERVER_2019</code> is the only valid value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link AppBlockBuilderPlatformType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #platformAsString}.
     * </p>
     * 
     * @return The platform of the app block builder.</p>
     *         <p>
     *         <code>WINDOWS_SERVER_2019</code> is the only valid value.
     * @see AppBlockBuilderPlatformType
     */
    public final AppBlockBuilderPlatformType platform() {
        return AppBlockBuilderPlatformType.fromValue(platform);
    }

    /**
     * <p>
     * The platform of the app block builder.
     * </p>
     * <p>
     * <code>WINDOWS_SERVER_2019</code> is the only valid value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link AppBlockBuilderPlatformType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #platformAsString}.
     * </p>
     * 
     * @return The platform of the app block builder.</p>
     *         <p>
     *         <code>WINDOWS_SERVER_2019</code> is the only valid value.
     * @see AppBlockBuilderPlatformType
     */
    public final String platformAsString() {
        return platform;
    }

    /**
     * <p>
     * The instance type of the app block builder.
     * </p>
     * 
     * @return The instance type of the app block builder.
     */
    public final String instanceType() {
        return instanceType;
    }

    /**
     * <p>
     * Indicates whether default internet access is enabled for the app block builder.
     * </p>
     * 
     * @return Indicates whether default internet access is enabled for the app block builder.
     */
    public final Boolean enableDefaultInternetAccess() {
        return enableDefaultInternetAccess;
    }

    /**
     * <p>
     * The ARN of the IAM role that is applied to the app block builder.
     * </p>
     * 
     * @return The ARN of the IAM role that is applied to the app block builder.
     */
    public final String iamRoleArn() {
        return iamRoleArn;
    }

    /**
     * <p>
     * The VPC configuration for the app block builder.
     * </p>
     * 
     * @return The VPC configuration for the app block builder.
     */
    public final VpcConfig vpcConfig() {
        return vpcConfig;
    }

    /**
     * <p>
     * The state of the app block builder.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link AppBlockBuilderState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the app block builder.
     * @see AppBlockBuilderState
     */
    public final AppBlockBuilderState state() {
        return AppBlockBuilderState.fromValue(state);
    }

    /**
     * <p>
     * The state of the app block builder.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link AppBlockBuilderState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the app block builder.
     * @see AppBlockBuilderState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The creation time of the app block builder.
     * </p>
     * 
     * @return The creation time of the app block builder.
     */
    public final Instant createdTime() {
        return createdTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the AppBlockBuilderErrors property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAppBlockBuilderErrors() {
        return appBlockBuilderErrors != null && !(appBlockBuilderErrors instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The app block builder errors.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAppBlockBuilderErrors} method.
     * </p>
     * 
     * @return The app block builder errors.
     */
    public final List<ResourceError> appBlockBuilderErrors() {
        return appBlockBuilderErrors;
    }

    /**
     * <p>
     * The state change reason.
     * </p>
     * 
     * @return The state change reason.
     */
    public final AppBlockBuilderStateChangeReason stateChangeReason() {
        return stateChangeReason;
    }

    /**
     * For responses, this returns true if the service returned a value for the AccessEndpoints property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAccessEndpoints() {
        return accessEndpoints != null && !(accessEndpoints instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The list of interface VPC endpoint (interface endpoint) objects. Administrators can connect to the app block
     * builder only through the specified endpoints.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAccessEndpoints} method.
     * </p>
     * 
     * @return The list of interface VPC endpoint (interface endpoint) objects. Administrators can connect to the app
     *         block builder only through the specified endpoints.
     */
    public final List<AccessEndpoint> accessEndpoints() {
        return accessEndpoints;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(displayName());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(platformAsString());
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(enableDefaultInternetAccess());
        hashCode = 31 * hashCode + Objects.hashCode(iamRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(vpcConfig());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasAppBlockBuilderErrors() ? appBlockBuilderErrors() : null);
        hashCode = 31 * hashCode + Objects.hashCode(stateChangeReason());
        hashCode = 31 * hashCode + Objects.hashCode(hasAccessEndpoints() ? accessEndpoints() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AppBlockBuilder)) {
            return false;
        }
        AppBlockBuilder other = (AppBlockBuilder) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(name(), other.name())
                && Objects.equals(displayName(), other.displayName()) && Objects.equals(description(), other.description())
                && Objects.equals(platformAsString(), other.platformAsString())
                && Objects.equals(instanceType(), other.instanceType())
                && Objects.equals(enableDefaultInternetAccess(), other.enableDefaultInternetAccess())
                && Objects.equals(iamRoleArn(), other.iamRoleArn()) && Objects.equals(vpcConfig(), other.vpcConfig())
                && Objects.equals(stateAsString(), other.stateAsString()) && Objects.equals(createdTime(), other.createdTime())
                && hasAppBlockBuilderErrors() == other.hasAppBlockBuilderErrors()
                && Objects.equals(appBlockBuilderErrors(), other.appBlockBuilderErrors())
                && Objects.equals(stateChangeReason(), other.stateChangeReason())
                && hasAccessEndpoints() == other.hasAccessEndpoints()
                && Objects.equals(accessEndpoints(), other.accessEndpoints());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AppBlockBuilder").add("Arn", arn()).add("Name", name()).add("DisplayName", displayName())
                .add("Description", description()).add("Platform", platformAsString()).add("InstanceType", instanceType())
                .add("EnableDefaultInternetAccess", enableDefaultInternetAccess()).add("IamRoleArn", iamRoleArn())
                .add("VpcConfig", vpcConfig()).add("State", stateAsString()).add("CreatedTime", createdTime())
                .add("AppBlockBuilderErrors", hasAppBlockBuilderErrors() ? appBlockBuilderErrors() : null)
                .add("StateChangeReason", stateChangeReason())
                .add("AccessEndpoints", hasAccessEndpoints() ? accessEndpoints() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "DisplayName":
            return Optional.ofNullable(clazz.cast(displayName()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Platform":
            return Optional.ofNullable(clazz.cast(platformAsString()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "EnableDefaultInternetAccess":
            return Optional.ofNullable(clazz.cast(enableDefaultInternetAccess()));
        case "IamRoleArn":
            return Optional.ofNullable(clazz.cast(iamRoleArn()));
        case "VpcConfig":
            return Optional.ofNullable(clazz.cast(vpcConfig()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "AppBlockBuilderErrors":
            return Optional.ofNullable(clazz.cast(appBlockBuilderErrors()));
        case "StateChangeReason":
            return Optional.ofNullable(clazz.cast(stateChangeReason()));
        case "AccessEndpoints":
            return Optional.ofNullable(clazz.cast(accessEndpoints()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AppBlockBuilder, T> g) {
        return obj -> g.apply((AppBlockBuilder) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AppBlockBuilder> {
        /**
         * <p>
         * The ARN of the app block builder.
         * </p>
         * 
         * @param arn
         *        The ARN of the app block builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The name of the app block builder.
         * </p>
         * 
         * @param name
         *        The name of the app block builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The display name of the app block builder.
         * </p>
         * 
         * @param displayName
         *        The display name of the app block builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder displayName(String displayName);

        /**
         * <p>
         * The description of the app block builder.
         * </p>
         * 
         * @param description
         *        The description of the app block builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The platform of the app block builder.
         * </p>
         * <p>
         * <code>WINDOWS_SERVER_2019</code> is the only valid value.
         * </p>
         * 
         * @param platform
         *        The platform of the app block builder.</p>
         *        <p>
         *        <code>WINDOWS_SERVER_2019</code> is the only valid value.
         * @see AppBlockBuilderPlatformType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AppBlockBuilderPlatformType
         */
        Builder platform(String platform);

        /**
         * <p>
         * The platform of the app block builder.
         * </p>
         * <p>
         * <code>WINDOWS_SERVER_2019</code> is the only valid value.
         * </p>
         * 
         * @param platform
         *        The platform of the app block builder.</p>
         *        <p>
         *        <code>WINDOWS_SERVER_2019</code> is the only valid value.
         * @see AppBlockBuilderPlatformType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AppBlockBuilderPlatformType
         */
        Builder platform(AppBlockBuilderPlatformType platform);

        /**
         * <p>
         * The instance type of the app block builder.
         * </p>
         * 
         * @param instanceType
         *        The instance type of the app block builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * Indicates whether default internet access is enabled for the app block builder.
         * </p>
         * 
         * @param enableDefaultInternetAccess
         *        Indicates whether default internet access is enabled for the app block builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableDefaultInternetAccess(Boolean enableDefaultInternetAccess);

        /**
         * <p>
         * The ARN of the IAM role that is applied to the app block builder.
         * </p>
         * 
         * @param iamRoleArn
         *        The ARN of the IAM role that is applied to the app block builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamRoleArn(String iamRoleArn);

        /**
         * <p>
         * The VPC configuration for the app block builder.
         * </p>
         * 
         * @param vpcConfig
         *        The VPC configuration for the app block builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcConfig(VpcConfig vpcConfig);

        /**
         * <p>
         * The VPC configuration for the app block builder.
         * </p>
         * This is a convenience method that creates an instance of the {@link VpcConfig.Builder} avoiding the need to
         * create one manually via {@link VpcConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VpcConfig.Builder#build()} is called immediately and its result
         * is passed to {@link #vpcConfig(VpcConfig)}.
         * 
         * @param vpcConfig
         *        a consumer that will call methods on {@link VpcConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vpcConfig(VpcConfig)
         */
        default Builder vpcConfig(Consumer<VpcConfig.Builder> vpcConfig) {
            return vpcConfig(VpcConfig.builder().applyMutation(vpcConfig).build());
        }

        /**
         * <p>
         * The state of the app block builder.
         * </p>
         * 
         * @param state
         *        The state of the app block builder.
         * @see AppBlockBuilderState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AppBlockBuilderState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the app block builder.
         * </p>
         * 
         * @param state
         *        The state of the app block builder.
         * @see AppBlockBuilderState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AppBlockBuilderState
         */
        Builder state(AppBlockBuilderState state);

        /**
         * <p>
         * The creation time of the app block builder.
         * </p>
         * 
         * @param createdTime
         *        The creation time of the app block builder.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * The app block builder errors.
         * </p>
         * 
         * @param appBlockBuilderErrors
         *        The app block builder errors.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appBlockBuilderErrors(Collection<ResourceError> appBlockBuilderErrors);

        /**
         * <p>
         * The app block builder errors.
         * </p>
         * 
         * @param appBlockBuilderErrors
         *        The app block builder errors.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appBlockBuilderErrors(ResourceError... appBlockBuilderErrors);

        /**
         * <p>
         * The app block builder errors.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.appstream.model.ResourceError.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.appstream.model.ResourceError#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.appstream.model.ResourceError.Builder#build()} is called immediately
         * and its result is passed to {@link #appBlockBuilderErrors(List<ResourceError>)}.
         * 
         * @param appBlockBuilderErrors
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.appstream.model.ResourceError.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #appBlockBuilderErrors(java.util.Collection<ResourceError>)
         */
        Builder appBlockBuilderErrors(Consumer<ResourceError.Builder>... appBlockBuilderErrors);

        /**
         * <p>
         * The state change reason.
         * </p>
         * 
         * @param stateChangeReason
         *        The state change reason.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateChangeReason(AppBlockBuilderStateChangeReason stateChangeReason);

        /**
         * <p>
         * The state change reason.
         * </p>
         * This is a convenience method that creates an instance of the {@link AppBlockBuilderStateChangeReason.Builder}
         * avoiding the need to create one manually via {@link AppBlockBuilderStateChangeReason#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AppBlockBuilderStateChangeReason.Builder#build()} is called
         * immediately and its result is passed to {@link #stateChangeReason(AppBlockBuilderStateChangeReason)}.
         * 
         * @param stateChangeReason
         *        a consumer that will call methods on {@link AppBlockBuilderStateChangeReason.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stateChangeReason(AppBlockBuilderStateChangeReason)
         */
        default Builder stateChangeReason(Consumer<AppBlockBuilderStateChangeReason.Builder> stateChangeReason) {
            return stateChangeReason(AppBlockBuilderStateChangeReason.builder().applyMutation(stateChangeReason).build());
        }

        /**
         * <p>
         * The list of interface VPC endpoint (interface endpoint) objects. Administrators can connect to the app block
         * builder only through the specified endpoints.
         * </p>
         * 
         * @param accessEndpoints
         *        The list of interface VPC endpoint (interface endpoint) objects. Administrators can connect to the app
         *        block builder only through the specified endpoints.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessEndpoints(Collection<AccessEndpoint> accessEndpoints);

        /**
         * <p>
         * The list of interface VPC endpoint (interface endpoint) objects. Administrators can connect to the app block
         * builder only through the specified endpoints.
         * </p>
         * 
         * @param accessEndpoints
         *        The list of interface VPC endpoint (interface endpoint) objects. Administrators can connect to the app
         *        block builder only through the specified endpoints.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessEndpoints(AccessEndpoint... accessEndpoints);

        /**
         * <p>
         * The list of interface VPC endpoint (interface endpoint) objects. Administrators can connect to the app block
         * builder only through the specified endpoints.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.appstream.model.AccessEndpoint.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.appstream.model.AccessEndpoint#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.appstream.model.AccessEndpoint.Builder#build()} is called immediately
         * and its result is passed to {@link #accessEndpoints(List<AccessEndpoint>)}.
         * 
         * @param accessEndpoints
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.appstream.model.AccessEndpoint.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessEndpoints(java.util.Collection<AccessEndpoint>)
         */
        Builder accessEndpoints(Consumer<AccessEndpoint.Builder>... accessEndpoints);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private String name;

        private String displayName;

        private String description;

        private String platform;

        private String instanceType;

        private Boolean enableDefaultInternetAccess;

        private String iamRoleArn;

        private VpcConfig vpcConfig;

        private String state;

        private Instant createdTime;

        private List<ResourceError> appBlockBuilderErrors = DefaultSdkAutoConstructList.getInstance();

        private AppBlockBuilderStateChangeReason stateChangeReason;

        private List<AccessEndpoint> accessEndpoints = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AppBlockBuilder model) {
            arn(model.arn);
            name(model.name);
            displayName(model.displayName);
            description(model.description);
            platform(model.platform);
            instanceType(model.instanceType);
            enableDefaultInternetAccess(model.enableDefaultInternetAccess);
            iamRoleArn(model.iamRoleArn);
            vpcConfig(model.vpcConfig);
            state(model.state);
            createdTime(model.createdTime);
            appBlockBuilderErrors(model.appBlockBuilderErrors);
            stateChangeReason(model.stateChangeReason);
            accessEndpoints(model.accessEndpoints);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getDisplayName() {
            return displayName;
        }

        public final void setDisplayName(String displayName) {
            this.displayName = displayName;
        }

        @Override
        public final Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getPlatform() {
            return platform;
        }

        public final void setPlatform(String platform) {
            this.platform = platform;
        }

        @Override
        public final Builder platform(String platform) {
            this.platform = platform;
            return this;
        }

        @Override
        public final Builder platform(AppBlockBuilderPlatformType platform) {
            this.platform(platform == null ? null : platform.toString());
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final Boolean getEnableDefaultInternetAccess() {
            return enableDefaultInternetAccess;
        }

        public final void setEnableDefaultInternetAccess(Boolean enableDefaultInternetAccess) {
            this.enableDefaultInternetAccess = enableDefaultInternetAccess;
        }

        @Override
        public final Builder enableDefaultInternetAccess(Boolean enableDefaultInternetAccess) {
            this.enableDefaultInternetAccess = enableDefaultInternetAccess;
            return this;
        }

        public final String getIamRoleArn() {
            return iamRoleArn;
        }

        public final void setIamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
        }

        @Override
        public final Builder iamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
            return this;
        }

        public final VpcConfig.Builder getVpcConfig() {
            return vpcConfig != null ? vpcConfig.toBuilder() : null;
        }

        public final void setVpcConfig(VpcConfig.BuilderImpl vpcConfig) {
            this.vpcConfig = vpcConfig != null ? vpcConfig.build() : null;
        }

        @Override
        public final Builder vpcConfig(VpcConfig vpcConfig) {
            this.vpcConfig = vpcConfig;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(AppBlockBuilderState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final List<ResourceError.Builder> getAppBlockBuilderErrors() {
            List<ResourceError.Builder> result = ResourceErrorsCopier.copyToBuilder(this.appBlockBuilderErrors);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAppBlockBuilderErrors(Collection<ResourceError.BuilderImpl> appBlockBuilderErrors) {
            this.appBlockBuilderErrors = ResourceErrorsCopier.copyFromBuilder(appBlockBuilderErrors);
        }

        @Override
        public final Builder appBlockBuilderErrors(Collection<ResourceError> appBlockBuilderErrors) {
            this.appBlockBuilderErrors = ResourceErrorsCopier.copy(appBlockBuilderErrors);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder appBlockBuilderErrors(ResourceError... appBlockBuilderErrors) {
            appBlockBuilderErrors(Arrays.asList(appBlockBuilderErrors));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder appBlockBuilderErrors(Consumer<ResourceError.Builder>... appBlockBuilderErrors) {
            appBlockBuilderErrors(Stream.of(appBlockBuilderErrors).map(c -> ResourceError.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final AppBlockBuilderStateChangeReason.Builder getStateChangeReason() {
            return stateChangeReason != null ? stateChangeReason.toBuilder() : null;
        }

        public final void setStateChangeReason(AppBlockBuilderStateChangeReason.BuilderImpl stateChangeReason) {
            this.stateChangeReason = stateChangeReason != null ? stateChangeReason.build() : null;
        }

        @Override
        public final Builder stateChangeReason(AppBlockBuilderStateChangeReason stateChangeReason) {
            this.stateChangeReason = stateChangeReason;
            return this;
        }

        public final List<AccessEndpoint.Builder> getAccessEndpoints() {
            List<AccessEndpoint.Builder> result = AccessEndpointListCopier.copyToBuilder(this.accessEndpoints);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAccessEndpoints(Collection<AccessEndpoint.BuilderImpl> accessEndpoints) {
            this.accessEndpoints = AccessEndpointListCopier.copyFromBuilder(accessEndpoints);
        }

        @Override
        public final Builder accessEndpoints(Collection<AccessEndpoint> accessEndpoints) {
            this.accessEndpoints = AccessEndpointListCopier.copy(accessEndpoints);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accessEndpoints(AccessEndpoint... accessEndpoints) {
            accessEndpoints(Arrays.asList(accessEndpoints));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder accessEndpoints(Consumer<AccessEndpoint.Builder>... accessEndpoints) {
            accessEndpoints(Stream.of(accessEndpoints).map(c -> AccessEndpoint.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public AppBlockBuilder build() {
            return new AppBlockBuilder(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
