/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appflow.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The properties that are applied when the custom connector is being used as a source.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CustomConnectorSourceProperties implements SdkPojo, Serializable,
        ToCopyableBuilder<CustomConnectorSourceProperties.Builder, CustomConnectorSourceProperties> {
    private static final SdkField<String> ENTITY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("entityName").getter(getter(CustomConnectorSourceProperties::entityName))
            .setter(setter(Builder::entityName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("entityName").build()).build();

    private static final SdkField<Map<String, String>> CUSTOM_PROPERTIES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("customProperties")
            .getter(getter(CustomConnectorSourceProperties::customProperties))
            .setter(setter(Builder::customProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("customProperties").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<DataTransferApi> DATA_TRANSFER_API_FIELD = SdkField
            .<DataTransferApi> builder(MarshallingType.SDK_POJO).memberName("dataTransferApi")
            .getter(getter(CustomConnectorSourceProperties::dataTransferApi)).setter(setter(Builder::dataTransferApi))
            .constructor(DataTransferApi::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("dataTransferApi").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENTITY_NAME_FIELD,
            CUSTOM_PROPERTIES_FIELD, DATA_TRANSFER_API_FIELD));

    private static final long serialVersionUID = 1L;

    private final String entityName;

    private final Map<String, String> customProperties;

    private final DataTransferApi dataTransferApi;

    private CustomConnectorSourceProperties(BuilderImpl builder) {
        this.entityName = builder.entityName;
        this.customProperties = builder.customProperties;
        this.dataTransferApi = builder.dataTransferApi;
    }

    /**
     * <p>
     * The entity specified in the custom connector as a source in the flow.
     * </p>
     * 
     * @return The entity specified in the custom connector as a source in the flow.
     */
    public final String entityName() {
        return entityName;
    }

    /**
     * For responses, this returns true if the service returned a value for the CustomProperties property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCustomProperties() {
        return customProperties != null && !(customProperties instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Custom properties that are required to use the custom connector as a source.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCustomProperties} method.
     * </p>
     * 
     * @return Custom properties that are required to use the custom connector as a source.
     */
    public final Map<String, String> customProperties() {
        return customProperties;
    }

    /**
     * <p>
     * The API of the connector application that Amazon AppFlow uses to transfer your data.
     * </p>
     * 
     * @return The API of the connector application that Amazon AppFlow uses to transfer your data.
     */
    public final DataTransferApi dataTransferApi() {
        return dataTransferApi;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(entityName());
        hashCode = 31 * hashCode + Objects.hashCode(hasCustomProperties() ? customProperties() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dataTransferApi());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CustomConnectorSourceProperties)) {
            return false;
        }
        CustomConnectorSourceProperties other = (CustomConnectorSourceProperties) obj;
        return Objects.equals(entityName(), other.entityName()) && hasCustomProperties() == other.hasCustomProperties()
                && Objects.equals(customProperties(), other.customProperties())
                && Objects.equals(dataTransferApi(), other.dataTransferApi());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CustomConnectorSourceProperties").add("EntityName", entityName())
                .add("CustomProperties", hasCustomProperties() ? customProperties() : null)
                .add("DataTransferApi", dataTransferApi()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "entityName":
            return Optional.ofNullable(clazz.cast(entityName()));
        case "customProperties":
            return Optional.ofNullable(clazz.cast(customProperties()));
        case "dataTransferApi":
            return Optional.ofNullable(clazz.cast(dataTransferApi()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CustomConnectorSourceProperties, T> g) {
        return obj -> g.apply((CustomConnectorSourceProperties) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CustomConnectorSourceProperties> {
        /**
         * <p>
         * The entity specified in the custom connector as a source in the flow.
         * </p>
         * 
         * @param entityName
         *        The entity specified in the custom connector as a source in the flow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder entityName(String entityName);

        /**
         * <p>
         * Custom properties that are required to use the custom connector as a source.
         * </p>
         * 
         * @param customProperties
         *        Custom properties that are required to use the custom connector as a source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customProperties(Map<String, String> customProperties);

        /**
         * <p>
         * The API of the connector application that Amazon AppFlow uses to transfer your data.
         * </p>
         * 
         * @param dataTransferApi
         *        The API of the connector application that Amazon AppFlow uses to transfer your data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataTransferApi(DataTransferApi dataTransferApi);

        /**
         * <p>
         * The API of the connector application that Amazon AppFlow uses to transfer your data.
         * </p>
         * This is a convenience method that creates an instance of the {@link DataTransferApi.Builder} avoiding the
         * need to create one manually via {@link DataTransferApi#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DataTransferApi.Builder#build()} is called immediately and its
         * result is passed to {@link #dataTransferApi(DataTransferApi)}.
         * 
         * @param dataTransferApi
         *        a consumer that will call methods on {@link DataTransferApi.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dataTransferApi(DataTransferApi)
         */
        default Builder dataTransferApi(Consumer<DataTransferApi.Builder> dataTransferApi) {
            return dataTransferApi(DataTransferApi.builder().applyMutation(dataTransferApi).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String entityName;

        private Map<String, String> customProperties = DefaultSdkAutoConstructMap.getInstance();

        private DataTransferApi dataTransferApi;

        private BuilderImpl() {
        }

        private BuilderImpl(CustomConnectorSourceProperties model) {
            entityName(model.entityName);
            customProperties(model.customProperties);
            dataTransferApi(model.dataTransferApi);
        }

        public final String getEntityName() {
            return entityName;
        }

        public final void setEntityName(String entityName) {
            this.entityName = entityName;
        }

        @Override
        public final Builder entityName(String entityName) {
            this.entityName = entityName;
            return this;
        }

        public final Map<String, String> getCustomProperties() {
            if (customProperties instanceof SdkAutoConstructMap) {
                return null;
            }
            return customProperties;
        }

        public final void setCustomProperties(Map<String, String> customProperties) {
            this.customProperties = CustomPropertiesCopier.copy(customProperties);
        }

        @Override
        public final Builder customProperties(Map<String, String> customProperties) {
            this.customProperties = CustomPropertiesCopier.copy(customProperties);
            return this;
        }

        public final DataTransferApi.Builder getDataTransferApi() {
            return dataTransferApi != null ? dataTransferApi.toBuilder() : null;
        }

        public final void setDataTransferApi(DataTransferApi.BuilderImpl dataTransferApi) {
            this.dataTransferApi = dataTransferApi != null ? dataTransferApi.build() : null;
        }

        @Override
        public final Builder dataTransferApi(DataTransferApi dataTransferApi) {
            this.dataTransferApi = dataTransferApi;
            return this;
        }

        @Override
        public CustomConnectorSourceProperties build() {
            return new CustomConnectorSourceProperties(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
