/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigateway.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.PayloadTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A POST request to import an API to API Gateway using an input of an API definition file.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImportRestApiRequest extends ApiGatewayRequest implements
        ToCopyableBuilder<ImportRestApiRequest.Builder, ImportRestApiRequest> {
    private static final SdkField<Boolean> FAIL_ON_WARNINGS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("failOnWarnings").getter(getter(ImportRestApiRequest::failOnWarnings))
            .setter(setter(Builder::failOnWarnings))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("failonwarnings").build())
            .build();

    private static final SdkField<Map<String, String>> PARAMETERS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("parameters")
            .getter(getter(ImportRestApiRequest::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.QUERY_PARAM).locationName("parameters").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<SdkBytes> BODY_FIELD = SdkField
            .<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("body")
            .getter(getter(ImportRestApiRequest::body))
            .setter(setter(Builder::body))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("body").build(),
                    PayloadTrait.create()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FAIL_ON_WARNINGS_FIELD,
            PARAMETERS_FIELD, BODY_FIELD));

    private final Boolean failOnWarnings;

    private final Map<String, String> parameters;

    private final SdkBytes body;

    private ImportRestApiRequest(BuilderImpl builder) {
        super(builder);
        this.failOnWarnings = builder.failOnWarnings;
        this.parameters = builder.parameters;
        this.body = builder.body;
    }

    /**
     * <p>
     * A query parameter to indicate whether to rollback the API creation (<code>true</code>) or not (<code>false</code>
     * ) when a warning is encountered. The default value is <code>false</code>.
     * </p>
     * 
     * @return A query parameter to indicate whether to rollback the API creation (<code>true</code>) or not (
     *         <code>false</code>) when a warning is encountered. The default value is <code>false</code>.
     */
    public final Boolean failOnWarnings() {
        return failOnWarnings;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parameters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A key-value map of context-specific query string parameters specifying the behavior of different API importing
     * operations. The following shows operation-specific parameters and their supported values.
     * </p>
     * <p>
     * To exclude DocumentationParts from the import, set <code>parameters</code> as <code>ignore=documentation</code>.
     * </p>
     * <p>
     * To configure the endpoint type, set <code>parameters</code> as <code>endpointConfigurationTypes=EDGE</code>,
     * <code>endpointConfigurationTypes=REGIONAL</code>, or <code>endpointConfigurationTypes=PRIVATE</code>. The default
     * endpoint type is <code>EDGE</code>.
     * </p>
     * <p>
     * To handle imported <code>basepath</code>, set <code>parameters</code> as <code>basepath=ignore</code>,
     * <code>basepath=prepend</code> or <code>basepath=split</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameters} method.
     * </p>
     * 
     * @return A key-value map of context-specific query string parameters specifying the behavior of different API
     *         importing operations. The following shows operation-specific parameters and their supported values.</p>
     *         <p>
     *         To exclude DocumentationParts from the import, set <code>parameters</code> as
     *         <code>ignore=documentation</code>.
     *         </p>
     *         <p>
     *         To configure the endpoint type, set <code>parameters</code> as
     *         <code>endpointConfigurationTypes=EDGE</code>, <code>endpointConfigurationTypes=REGIONAL</code>, or
     *         <code>endpointConfigurationTypes=PRIVATE</code>. The default endpoint type is <code>EDGE</code>.
     *         </p>
     *         <p>
     *         To handle imported <code>basepath</code>, set <code>parameters</code> as <code>basepath=ignore</code>,
     *         <code>basepath=prepend</code> or <code>basepath=split</code>.
     */
    public final Map<String, String> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The POST request body containing external API definitions. Currently, only OpenAPI definition JSON/YAML files are
     * supported. The maximum size of the API definition file is 6MB.
     * </p>
     * 
     * @return The POST request body containing external API definitions. Currently, only OpenAPI definition JSON/YAML
     *         files are supported. The maximum size of the API definition file is 6MB.
     */
    public final SdkBytes body() {
        return body;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(failOnWarnings());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(body());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImportRestApiRequest)) {
            return false;
        }
        ImportRestApiRequest other = (ImportRestApiRequest) obj;
        return Objects.equals(failOnWarnings(), other.failOnWarnings()) && hasParameters() == other.hasParameters()
                && Objects.equals(parameters(), other.parameters()) && Objects.equals(body(), other.body());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ImportRestApiRequest").add("FailOnWarnings", failOnWarnings())
                .add("Parameters", hasParameters() ? parameters() : null).add("Body", body()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "failOnWarnings":
            return Optional.ofNullable(clazz.cast(failOnWarnings()));
        case "parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "body":
            return Optional.ofNullable(clazz.cast(body()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ImportRestApiRequest, T> g) {
        return obj -> g.apply((ImportRestApiRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ApiGatewayRequest.Builder, SdkPojo, CopyableBuilder<Builder, ImportRestApiRequest> {
        /**
         * <p>
         * A query parameter to indicate whether to rollback the API creation (<code>true</code>) or not (
         * <code>false</code>) when a warning is encountered. The default value is <code>false</code>.
         * </p>
         * 
         * @param failOnWarnings
         *        A query parameter to indicate whether to rollback the API creation (<code>true</code>) or not (
         *        <code>false</code>) when a warning is encountered. The default value is <code>false</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failOnWarnings(Boolean failOnWarnings);

        /**
         * <p>
         * A key-value map of context-specific query string parameters specifying the behavior of different API
         * importing operations. The following shows operation-specific parameters and their supported values.
         * </p>
         * <p>
         * To exclude DocumentationParts from the import, set <code>parameters</code> as
         * <code>ignore=documentation</code>.
         * </p>
         * <p>
         * To configure the endpoint type, set <code>parameters</code> as <code>endpointConfigurationTypes=EDGE</code>,
         * <code>endpointConfigurationTypes=REGIONAL</code>, or <code>endpointConfigurationTypes=PRIVATE</code>. The
         * default endpoint type is <code>EDGE</code>.
         * </p>
         * <p>
         * To handle imported <code>basepath</code>, set <code>parameters</code> as <code>basepath=ignore</code>,
         * <code>basepath=prepend</code> or <code>basepath=split</code>.
         * </p>
         * 
         * @param parameters
         *        A key-value map of context-specific query string parameters specifying the behavior of different API
         *        importing operations. The following shows operation-specific parameters and their supported
         *        values.</p>
         *        <p>
         *        To exclude DocumentationParts from the import, set <code>parameters</code> as
         *        <code>ignore=documentation</code>.
         *        </p>
         *        <p>
         *        To configure the endpoint type, set <code>parameters</code> as
         *        <code>endpointConfigurationTypes=EDGE</code>, <code>endpointConfigurationTypes=REGIONAL</code>, or
         *        <code>endpointConfigurationTypes=PRIVATE</code>. The default endpoint type is <code>EDGE</code>.
         *        </p>
         *        <p>
         *        To handle imported <code>basepath</code>, set <code>parameters</code> as <code>basepath=ignore</code>,
         *        <code>basepath=prepend</code> or <code>basepath=split</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Map<String, String> parameters);

        /**
         * <p>
         * The POST request body containing external API definitions. Currently, only OpenAPI definition JSON/YAML files
         * are supported. The maximum size of the API definition file is 6MB.
         * </p>
         * 
         * @param body
         *        The POST request body containing external API definitions. Currently, only OpenAPI definition
         *        JSON/YAML files are supported. The maximum size of the API definition file is 6MB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder body(SdkBytes body);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ApiGatewayRequest.BuilderImpl implements Builder {
        private Boolean failOnWarnings;

        private Map<String, String> parameters = DefaultSdkAutoConstructMap.getInstance();

        private SdkBytes body;

        private BuilderImpl() {
        }

        private BuilderImpl(ImportRestApiRequest model) {
            super(model);
            failOnWarnings(model.failOnWarnings);
            parameters(model.parameters);
            body(model.body);
        }

        public final Boolean getFailOnWarnings() {
            return failOnWarnings;
        }

        public final void setFailOnWarnings(Boolean failOnWarnings) {
            this.failOnWarnings = failOnWarnings;
        }

        @Override
        public final Builder failOnWarnings(Boolean failOnWarnings) {
            this.failOnWarnings = failOnWarnings;
            return this;
        }

        public final Map<String, String> getParameters() {
            if (parameters instanceof SdkAutoConstructMap) {
                return null;
            }
            return parameters;
        }

        public final void setParameters(Map<String, String> parameters) {
            this.parameters = MapOfStringToStringCopier.copy(parameters);
        }

        @Override
        public final Builder parameters(Map<String, String> parameters) {
            this.parameters = MapOfStringToStringCopier.copy(parameters);
            return this;
        }

        public final ByteBuffer getBody() {
            return body == null ? null : body.asByteBuffer();
        }

        public final void setBody(ByteBuffer body) {
            body(body == null ? null : SdkBytes.fromByteBuffer(body));
        }

        @Override
        public final Builder body(SdkBytes body) {
            this.body = body;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public ImportRestApiRequest build() {
            return new ImportRestApiRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
