/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.amplifybackend;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.amplifybackend.model.AmplifyBackendException;
import software.amazon.awssdk.services.amplifybackend.model.BadRequestException;
import software.amazon.awssdk.services.amplifybackend.model.CloneBackendRequest;
import software.amazon.awssdk.services.amplifybackend.model.CloneBackendResponse;
import software.amazon.awssdk.services.amplifybackend.model.CreateBackendApiRequest;
import software.amazon.awssdk.services.amplifybackend.model.CreateBackendApiResponse;
import software.amazon.awssdk.services.amplifybackend.model.CreateBackendAuthRequest;
import software.amazon.awssdk.services.amplifybackend.model.CreateBackendAuthResponse;
import software.amazon.awssdk.services.amplifybackend.model.CreateBackendConfigRequest;
import software.amazon.awssdk.services.amplifybackend.model.CreateBackendConfigResponse;
import software.amazon.awssdk.services.amplifybackend.model.CreateBackendRequest;
import software.amazon.awssdk.services.amplifybackend.model.CreateBackendResponse;
import software.amazon.awssdk.services.amplifybackend.model.CreateBackendStorageRequest;
import software.amazon.awssdk.services.amplifybackend.model.CreateBackendStorageResponse;
import software.amazon.awssdk.services.amplifybackend.model.CreateTokenRequest;
import software.amazon.awssdk.services.amplifybackend.model.CreateTokenResponse;
import software.amazon.awssdk.services.amplifybackend.model.DeleteBackendApiRequest;
import software.amazon.awssdk.services.amplifybackend.model.DeleteBackendApiResponse;
import software.amazon.awssdk.services.amplifybackend.model.DeleteBackendAuthRequest;
import software.amazon.awssdk.services.amplifybackend.model.DeleteBackendAuthResponse;
import software.amazon.awssdk.services.amplifybackend.model.DeleteBackendRequest;
import software.amazon.awssdk.services.amplifybackend.model.DeleteBackendResponse;
import software.amazon.awssdk.services.amplifybackend.model.DeleteBackendStorageRequest;
import software.amazon.awssdk.services.amplifybackend.model.DeleteBackendStorageResponse;
import software.amazon.awssdk.services.amplifybackend.model.DeleteTokenRequest;
import software.amazon.awssdk.services.amplifybackend.model.DeleteTokenResponse;
import software.amazon.awssdk.services.amplifybackend.model.GatewayTimeoutException;
import software.amazon.awssdk.services.amplifybackend.model.GenerateBackendApiModelsRequest;
import software.amazon.awssdk.services.amplifybackend.model.GenerateBackendApiModelsResponse;
import software.amazon.awssdk.services.amplifybackend.model.GetBackendApiModelsRequest;
import software.amazon.awssdk.services.amplifybackend.model.GetBackendApiModelsResponse;
import software.amazon.awssdk.services.amplifybackend.model.GetBackendApiRequest;
import software.amazon.awssdk.services.amplifybackend.model.GetBackendApiResponse;
import software.amazon.awssdk.services.amplifybackend.model.GetBackendAuthRequest;
import software.amazon.awssdk.services.amplifybackend.model.GetBackendAuthResponse;
import software.amazon.awssdk.services.amplifybackend.model.GetBackendJobRequest;
import software.amazon.awssdk.services.amplifybackend.model.GetBackendJobResponse;
import software.amazon.awssdk.services.amplifybackend.model.GetBackendRequest;
import software.amazon.awssdk.services.amplifybackend.model.GetBackendResponse;
import software.amazon.awssdk.services.amplifybackend.model.GetBackendStorageRequest;
import software.amazon.awssdk.services.amplifybackend.model.GetBackendStorageResponse;
import software.amazon.awssdk.services.amplifybackend.model.GetTokenRequest;
import software.amazon.awssdk.services.amplifybackend.model.GetTokenResponse;
import software.amazon.awssdk.services.amplifybackend.model.ImportBackendAuthRequest;
import software.amazon.awssdk.services.amplifybackend.model.ImportBackendAuthResponse;
import software.amazon.awssdk.services.amplifybackend.model.ImportBackendStorageRequest;
import software.amazon.awssdk.services.amplifybackend.model.ImportBackendStorageResponse;
import software.amazon.awssdk.services.amplifybackend.model.ListBackendJobsRequest;
import software.amazon.awssdk.services.amplifybackend.model.ListBackendJobsResponse;
import software.amazon.awssdk.services.amplifybackend.model.ListS3BucketsRequest;
import software.amazon.awssdk.services.amplifybackend.model.ListS3BucketsResponse;
import software.amazon.awssdk.services.amplifybackend.model.NotFoundException;
import software.amazon.awssdk.services.amplifybackend.model.RemoveAllBackendsRequest;
import software.amazon.awssdk.services.amplifybackend.model.RemoveAllBackendsResponse;
import software.amazon.awssdk.services.amplifybackend.model.RemoveBackendConfigRequest;
import software.amazon.awssdk.services.amplifybackend.model.RemoveBackendConfigResponse;
import software.amazon.awssdk.services.amplifybackend.model.TooManyRequestsException;
import software.amazon.awssdk.services.amplifybackend.model.UpdateBackendApiRequest;
import software.amazon.awssdk.services.amplifybackend.model.UpdateBackendApiResponse;
import software.amazon.awssdk.services.amplifybackend.model.UpdateBackendAuthRequest;
import software.amazon.awssdk.services.amplifybackend.model.UpdateBackendAuthResponse;
import software.amazon.awssdk.services.amplifybackend.model.UpdateBackendConfigRequest;
import software.amazon.awssdk.services.amplifybackend.model.UpdateBackendConfigResponse;
import software.amazon.awssdk.services.amplifybackend.model.UpdateBackendJobRequest;
import software.amazon.awssdk.services.amplifybackend.model.UpdateBackendJobResponse;
import software.amazon.awssdk.services.amplifybackend.model.UpdateBackendStorageRequest;
import software.amazon.awssdk.services.amplifybackend.model.UpdateBackendStorageResponse;
import software.amazon.awssdk.services.amplifybackend.transform.CloneBackendRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.CreateBackendApiRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.CreateBackendAuthRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.CreateBackendConfigRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.CreateBackendRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.CreateBackendStorageRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.CreateTokenRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.DeleteBackendApiRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.DeleteBackendAuthRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.DeleteBackendRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.DeleteBackendStorageRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.DeleteTokenRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.GenerateBackendApiModelsRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.GetBackendApiModelsRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.GetBackendApiRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.GetBackendAuthRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.GetBackendJobRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.GetBackendRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.GetBackendStorageRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.GetTokenRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.ImportBackendAuthRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.ImportBackendStorageRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.ListBackendJobsRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.ListS3BucketsRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.RemoveAllBackendsRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.RemoveBackendConfigRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.UpdateBackendApiRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.UpdateBackendAuthRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.UpdateBackendConfigRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.UpdateBackendJobRequestMarshaller;
import software.amazon.awssdk.services.amplifybackend.transform.UpdateBackendStorageRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AmplifyBackendAsyncClient}.
 *
 * @see AmplifyBackendAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAmplifyBackendAsyncClient implements AmplifyBackendAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAmplifyBackendAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final AmplifyBackendServiceClientConfiguration serviceClientConfiguration;

    protected DefaultAmplifyBackendAsyncClient(AmplifyBackendServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * This operation clones an existing backend.
     * </p>
     *
     * @param cloneBackendRequest
     *        The request body for CloneBackend.
     * @return A Java Future containing the result of the CloneBackend operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.CloneBackend
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/CloneBackend" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CloneBackendResponse> cloneBackend(CloneBackendRequest cloneBackendRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cloneBackendRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CloneBackend");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CloneBackendResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CloneBackendResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CloneBackendResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CloneBackendRequest, CloneBackendResponse>()
                            .withOperationName("CloneBackend").withMarshaller(new CloneBackendRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(cloneBackendRequest));
            CompletableFuture<CloneBackendResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * This operation creates a backend for an Amplify app. Backends are automatically created at the time of app
     * creation.
     * </p>
     *
     * @param createBackendRequest
     *        The request body for CreateBackend.
     * @return A Java Future containing the result of the CreateBackend operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.CreateBackend
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/CreateBackend" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBackendResponse> createBackend(CreateBackendRequest createBackendRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBackendRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBackend");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBackendResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateBackendResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBackendResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBackendRequest, CreateBackendResponse>()
                            .withOperationName("CreateBackend")
                            .withMarshaller(new CreateBackendRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createBackendRequest));
            CompletableFuture<CreateBackendResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new backend API resource.
     * </p>
     *
     * @param createBackendApiRequest
     *        The request body for CreateBackendAPI.
     * @return A Java Future containing the result of the CreateBackendAPI operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.CreateBackendAPI
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/CreateBackendAPI"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBackendApiResponse> createBackendAPI(CreateBackendApiRequest createBackendApiRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBackendApiRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBackendAPI");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBackendApiResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateBackendApiResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBackendApiResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBackendApiRequest, CreateBackendApiResponse>()
                            .withOperationName("CreateBackendAPI")
                            .withMarshaller(new CreateBackendApiRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createBackendApiRequest));
            CompletableFuture<CreateBackendApiResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new backend authentication resource.
     * </p>
     *
     * @param createBackendAuthRequest
     *        The request body for CreateBackendAuth.
     * @return A Java Future containing the result of the CreateBackendAuth operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.CreateBackendAuth
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/CreateBackendAuth"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBackendAuthResponse> createBackendAuth(CreateBackendAuthRequest createBackendAuthRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBackendAuthRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBackendAuth");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBackendAuthResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateBackendAuthResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBackendAuthResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBackendAuthRequest, CreateBackendAuthResponse>()
                            .withOperationName("CreateBackendAuth")
                            .withMarshaller(new CreateBackendAuthRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createBackendAuthRequest));
            CompletableFuture<CreateBackendAuthResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a config object for a backend.
     * </p>
     *
     * @param createBackendConfigRequest
     *        The request body for CreateBackendConfig.
     * @return A Java Future containing the result of the CreateBackendConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.CreateBackendConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/CreateBackendConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBackendConfigResponse> createBackendConfig(
            CreateBackendConfigRequest createBackendConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBackendConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBackendConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBackendConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateBackendConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBackendConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBackendConfigRequest, CreateBackendConfigResponse>()
                            .withOperationName("CreateBackendConfig")
                            .withMarshaller(new CreateBackendConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createBackendConfigRequest));
            CompletableFuture<CreateBackendConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a backend storage resource.
     * </p>
     *
     * @param createBackendStorageRequest
     *        The request body for CreateBackendStorage.
     * @return A Java Future containing the result of the CreateBackendStorage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.CreateBackendStorage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/CreateBackendStorage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBackendStorageResponse> createBackendStorage(
            CreateBackendStorageRequest createBackendStorageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBackendStorageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBackendStorage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBackendStorageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateBackendStorageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBackendStorageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBackendStorageRequest, CreateBackendStorageResponse>()
                            .withOperationName("CreateBackendStorage")
                            .withMarshaller(new CreateBackendStorageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createBackendStorageRequest));
            CompletableFuture<CreateBackendStorageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates a one-time challenge code to authenticate a user into your Amplify Admin UI.
     * </p>
     *
     * @param createTokenRequest
     * @return A Java Future containing the result of the CreateToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.CreateToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/CreateToken" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTokenResponse> createToken(CreateTokenRequest createTokenRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTokenResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTokenRequest, CreateTokenResponse>()
                            .withOperationName("CreateToken").withMarshaller(new CreateTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createTokenRequest));
            CompletableFuture<CreateTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes an existing environment from your Amplify project.
     * </p>
     *
     * @param deleteBackendRequest
     * @return A Java Future containing the result of the DeleteBackend operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.DeleteBackend
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/DeleteBackend" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBackendResponse> deleteBackend(DeleteBackendRequest deleteBackendRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackendRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackend");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBackendResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteBackendResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteBackendResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBackendRequest, DeleteBackendResponse>()
                            .withOperationName("DeleteBackend")
                            .withMarshaller(new DeleteBackendRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteBackendRequest));
            CompletableFuture<DeleteBackendResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing backend API resource.
     * </p>
     *
     * @param deleteBackendApiRequest
     *        The request body for DeleteBackendAPI.
     * @return A Java Future containing the result of the DeleteBackendAPI operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.DeleteBackendAPI
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/DeleteBackendAPI"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBackendApiResponse> deleteBackendAPI(DeleteBackendApiRequest deleteBackendApiRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackendApiRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackendAPI");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBackendApiResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteBackendApiResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteBackendApiResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBackendApiRequest, DeleteBackendApiResponse>()
                            .withOperationName("DeleteBackendAPI")
                            .withMarshaller(new DeleteBackendApiRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteBackendApiRequest));
            CompletableFuture<DeleteBackendApiResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing backend authentication resource.
     * </p>
     *
     * @param deleteBackendAuthRequest
     *        The request body for DeleteBackendAuth.
     * @return A Java Future containing the result of the DeleteBackendAuth operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.DeleteBackendAuth
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/DeleteBackendAuth"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBackendAuthResponse> deleteBackendAuth(DeleteBackendAuthRequest deleteBackendAuthRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackendAuthRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackendAuth");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBackendAuthResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteBackendAuthResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteBackendAuthResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBackendAuthRequest, DeleteBackendAuthResponse>()
                            .withOperationName("DeleteBackendAuth")
                            .withMarshaller(new DeleteBackendAuthRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteBackendAuthRequest));
            CompletableFuture<DeleteBackendAuthResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified backend storage resource.
     * </p>
     *
     * @param deleteBackendStorageRequest
     *        The request body for DeleteBackendStorage.
     * @return A Java Future containing the result of the DeleteBackendStorage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.DeleteBackendStorage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/DeleteBackendStorage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBackendStorageResponse> deleteBackendStorage(
            DeleteBackendStorageRequest deleteBackendStorageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackendStorageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackendStorage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBackendStorageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteBackendStorageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteBackendStorageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBackendStorageRequest, DeleteBackendStorageResponse>()
                            .withOperationName("DeleteBackendStorage")
                            .withMarshaller(new DeleteBackendStorageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteBackendStorageRequest));
            CompletableFuture<DeleteBackendStorageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the challenge token based on the given appId and sessionId.
     * </p>
     *
     * @param deleteTokenRequest
     * @return A Java Future containing the result of the DeleteToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.DeleteToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/DeleteToken" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTokenResponse> deleteToken(DeleteTokenRequest deleteTokenRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTokenResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTokenRequest, DeleteTokenResponse>()
                            .withOperationName("DeleteToken").withMarshaller(new DeleteTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteTokenRequest));
            CompletableFuture<DeleteTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Generates a model schema for an existing backend API resource.
     * </p>
     *
     * @param generateBackendApiModelsRequest
     *        The request body for GenerateBackendAPIModels.
     * @return A Java Future containing the result of the GenerateBackendAPIModels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.GenerateBackendAPIModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/GenerateBackendAPIModels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GenerateBackendApiModelsResponse> generateBackendAPIModels(
            GenerateBackendApiModelsRequest generateBackendApiModelsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, generateBackendApiModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GenerateBackendAPIModels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GenerateBackendApiModelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GenerateBackendApiModelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GenerateBackendApiModelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GenerateBackendApiModelsRequest, GenerateBackendApiModelsResponse>()
                            .withOperationName("GenerateBackendAPIModels")
                            .withMarshaller(new GenerateBackendApiModelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(generateBackendApiModelsRequest));
            CompletableFuture<GenerateBackendApiModelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Provides project-level details for your Amplify UI project.
     * </p>
     *
     * @param getBackendRequest
     *        The request body for GetBackend.
     * @return A Java Future containing the result of the GetBackend operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.GetBackend
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/GetBackend" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetBackendResponse> getBackend(GetBackendRequest getBackendRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBackendRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBackend");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBackendResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetBackendResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBackendResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBackendRequest, GetBackendResponse>().withOperationName("GetBackend")
                            .withMarshaller(new GetBackendRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getBackendRequest));
            CompletableFuture<GetBackendResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the details for a backend API.
     * </p>
     *
     * @param getBackendApiRequest
     *        The request body for GetBackendAPI.
     * @return A Java Future containing the result of the GetBackendAPI operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.GetBackendAPI
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/GetBackendAPI" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetBackendApiResponse> getBackendAPI(GetBackendApiRequest getBackendApiRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBackendApiRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBackendAPI");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBackendApiResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetBackendApiResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBackendApiResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBackendApiRequest, GetBackendApiResponse>()
                            .withOperationName("GetBackendAPI")
                            .withMarshaller(new GetBackendApiRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getBackendApiRequest));
            CompletableFuture<GetBackendApiResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a model introspection schema for an existing backend API resource.
     * </p>
     *
     * @param getBackendApiModelsRequest
     *        The request body for GetBackendAPIModels.
     * @return A Java Future containing the result of the GetBackendAPIModels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.GetBackendAPIModels
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/GetBackendAPIModels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetBackendApiModelsResponse> getBackendAPIModels(
            GetBackendApiModelsRequest getBackendApiModelsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBackendApiModelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBackendAPIModels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBackendApiModelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetBackendApiModelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBackendApiModelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBackendApiModelsRequest, GetBackendApiModelsResponse>()
                            .withOperationName("GetBackendAPIModels")
                            .withMarshaller(new GetBackendApiModelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getBackendApiModelsRequest));
            CompletableFuture<GetBackendApiModelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a backend auth details.
     * </p>
     *
     * @param getBackendAuthRequest
     *        The request body for GetBackendAuth.
     * @return A Java Future containing the result of the GetBackendAuth operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.GetBackendAuth
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/GetBackendAuth" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetBackendAuthResponse> getBackendAuth(GetBackendAuthRequest getBackendAuthRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBackendAuthRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBackendAuth");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBackendAuthResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetBackendAuthResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBackendAuthResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBackendAuthRequest, GetBackendAuthResponse>()
                            .withOperationName("GetBackendAuth")
                            .withMarshaller(new GetBackendAuthRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getBackendAuthRequest));
            CompletableFuture<GetBackendAuthResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about a specific job.
     * </p>
     *
     * @param getBackendJobRequest
     * @return A Java Future containing the result of the GetBackendJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.GetBackendJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/GetBackendJob" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetBackendJobResponse> getBackendJob(GetBackendJobRequest getBackendJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBackendJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBackendJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBackendJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetBackendJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBackendJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBackendJobRequest, GetBackendJobResponse>()
                            .withOperationName("GetBackendJob")
                            .withMarshaller(new GetBackendJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getBackendJobRequest));
            CompletableFuture<GetBackendJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets details for a backend storage resource.
     * </p>
     *
     * @param getBackendStorageRequest
     *        The request body for GetBackendStorage.
     * @return A Java Future containing the result of the GetBackendStorage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.GetBackendStorage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/GetBackendStorage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetBackendStorageResponse> getBackendStorage(GetBackendStorageRequest getBackendStorageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBackendStorageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBackendStorage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetBackendStorageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetBackendStorageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetBackendStorageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetBackendStorageRequest, GetBackendStorageResponse>()
                            .withOperationName("GetBackendStorage")
                            .withMarshaller(new GetBackendStorageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getBackendStorageRequest));
            CompletableFuture<GetBackendStorageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the challenge token based on the given appId and sessionId.
     * </p>
     *
     * @param getTokenRequest
     * @return A Java Future containing the result of the GetToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.GetToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/GetToken" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTokenResponse> getToken(GetTokenRequest getTokenRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTokenResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTokenRequest, GetTokenResponse>().withOperationName("GetToken")
                            .withMarshaller(new GetTokenRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getTokenRequest));
            CompletableFuture<GetTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Imports an existing backend authentication resource.
     * </p>
     *
     * @param importBackendAuthRequest
     *        The request body for ImportBackendAuth.
     * @return A Java Future containing the result of the ImportBackendAuth operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.ImportBackendAuth
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/ImportBackendAuth"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ImportBackendAuthResponse> importBackendAuth(ImportBackendAuthRequest importBackendAuthRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importBackendAuthRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportBackendAuth");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ImportBackendAuthResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ImportBackendAuthResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ImportBackendAuthResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ImportBackendAuthRequest, ImportBackendAuthResponse>()
                            .withOperationName("ImportBackendAuth")
                            .withMarshaller(new ImportBackendAuthRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(importBackendAuthRequest));
            CompletableFuture<ImportBackendAuthResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Imports an existing backend storage resource.
     * </p>
     *
     * @param importBackendStorageRequest
     *        The request body for ImportBackendStorage.
     * @return A Java Future containing the result of the ImportBackendStorage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.ImportBackendStorage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/ImportBackendStorage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ImportBackendStorageResponse> importBackendStorage(
            ImportBackendStorageRequest importBackendStorageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importBackendStorageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportBackendStorage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ImportBackendStorageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ImportBackendStorageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ImportBackendStorageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ImportBackendStorageRequest, ImportBackendStorageResponse>()
                            .withOperationName("ImportBackendStorage")
                            .withMarshaller(new ImportBackendStorageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(importBackendStorageRequest));
            CompletableFuture<ImportBackendStorageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the jobs for the backend of an Amplify app.
     * </p>
     *
     * @param listBackendJobsRequest
     *        The request body for ListBackendJobs.
     * @return A Java Future containing the result of the ListBackendJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.ListBackendJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/ListBackendJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListBackendJobsResponse> listBackendJobs(ListBackendJobsRequest listBackendJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBackendJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBackendJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListBackendJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListBackendJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListBackendJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListBackendJobsRequest, ListBackendJobsResponse>()
                            .withOperationName("ListBackendJobs")
                            .withMarshaller(new ListBackendJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listBackendJobsRequest));
            CompletableFuture<ListBackendJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The list of S3 buckets in your account.
     * </p>
     *
     * @param listS3BucketsRequest
     *        The request body for S3Buckets.
     * @return A Java Future containing the result of the ListS3Buckets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.ListS3Buckets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/ListS3Buckets" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListS3BucketsResponse> listS3Buckets(ListS3BucketsRequest listS3BucketsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listS3BucketsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListS3Buckets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListS3BucketsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListS3BucketsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListS3BucketsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListS3BucketsRequest, ListS3BucketsResponse>()
                            .withOperationName("ListS3Buckets")
                            .withMarshaller(new ListS3BucketsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listS3BucketsRequest));
            CompletableFuture<ListS3BucketsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes all backend environments from your Amplify project.
     * </p>
     *
     * @param removeAllBackendsRequest
     *        The request body for RemoveAllBackends.
     * @return A Java Future containing the result of the RemoveAllBackends operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.RemoveAllBackends
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/RemoveAllBackends"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveAllBackendsResponse> removeAllBackends(RemoveAllBackendsRequest removeAllBackendsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeAllBackendsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveAllBackends");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveAllBackendsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveAllBackendsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveAllBackendsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveAllBackendsRequest, RemoveAllBackendsResponse>()
                            .withOperationName("RemoveAllBackends")
                            .withMarshaller(new RemoveAllBackendsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeAllBackendsRequest));
            CompletableFuture<RemoveAllBackendsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the AWS resources required to access the Amplify Admin UI.
     * </p>
     *
     * @param removeBackendConfigRequest
     * @return A Java Future containing the result of the RemoveBackendConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.RemoveBackendConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/RemoveBackendConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveBackendConfigResponse> removeBackendConfig(
            RemoveBackendConfigRequest removeBackendConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeBackendConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveBackendConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveBackendConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveBackendConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveBackendConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveBackendConfigRequest, RemoveBackendConfigResponse>()
                            .withOperationName("RemoveBackendConfig")
                            .withMarshaller(new RemoveBackendConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeBackendConfigRequest));
            CompletableFuture<RemoveBackendConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing backend API resource.
     * </p>
     *
     * @param updateBackendApiRequest
     *        The request body for UpdateBackendAPI.
     * @return A Java Future containing the result of the UpdateBackendAPI operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.UpdateBackendAPI
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/UpdateBackendAPI"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateBackendApiResponse> updateBackendAPI(UpdateBackendApiRequest updateBackendApiRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBackendApiRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBackendAPI");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateBackendApiResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateBackendApiResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateBackendApiResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateBackendApiRequest, UpdateBackendApiResponse>()
                            .withOperationName("UpdateBackendAPI")
                            .withMarshaller(new UpdateBackendApiRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateBackendApiRequest));
            CompletableFuture<UpdateBackendApiResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing backend authentication resource.
     * </p>
     *
     * @param updateBackendAuthRequest
     *        The request body for UpdateBackendAuth.
     * @return A Java Future containing the result of the UpdateBackendAuth operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.UpdateBackendAuth
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/UpdateBackendAuth"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateBackendAuthResponse> updateBackendAuth(UpdateBackendAuthRequest updateBackendAuthRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBackendAuthRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBackendAuth");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateBackendAuthResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateBackendAuthResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateBackendAuthResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateBackendAuthRequest, UpdateBackendAuthResponse>()
                            .withOperationName("UpdateBackendAuth")
                            .withMarshaller(new UpdateBackendAuthRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateBackendAuthRequest));
            CompletableFuture<UpdateBackendAuthResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the AWS resources required to access the Amplify Admin UI.
     * </p>
     *
     * @param updateBackendConfigRequest
     *        The request body for UpdateBackendConfig.
     * @return A Java Future containing the result of the UpdateBackendConfig operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.UpdateBackendConfig
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/UpdateBackendConfig"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateBackendConfigResponse> updateBackendConfig(
            UpdateBackendConfigRequest updateBackendConfigRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBackendConfigRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBackendConfig");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateBackendConfigResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateBackendConfigResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateBackendConfigResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateBackendConfigRequest, UpdateBackendConfigResponse>()
                            .withOperationName("UpdateBackendConfig")
                            .withMarshaller(new UpdateBackendConfigRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateBackendConfigRequest));
            CompletableFuture<UpdateBackendConfigResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a specific job.
     * </p>
     *
     * @param updateBackendJobRequest
     *        The request body for GetBackendJob.
     * @return A Java Future containing the result of the UpdateBackendJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.UpdateBackendJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/UpdateBackendJob"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateBackendJobResponse> updateBackendJob(UpdateBackendJobRequest updateBackendJobRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBackendJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBackendJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateBackendJobResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateBackendJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateBackendJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateBackendJobRequest, UpdateBackendJobResponse>()
                            .withOperationName("UpdateBackendJob")
                            .withMarshaller(new UpdateBackendJobRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateBackendJobRequest));
            CompletableFuture<UpdateBackendJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an existing backend storage resource.
     * </p>
     *
     * @param updateBackendStorageRequest
     *        The request body for UpdateBackendStorage.
     * @return A Java Future containing the result of the UpdateBackendStorage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException 404 response</li>
     *         <li>GatewayTimeoutException 504 response</li>
     *         <li>TooManyRequestsException 429 response</li>
     *         <li>BadRequestException 400 response</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AmplifyBackendException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample AmplifyBackendAsyncClient.UpdateBackendStorage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/amplifybackend-2020-08-11/UpdateBackendStorage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateBackendStorageResponse> updateBackendStorage(
            UpdateBackendStorageRequest updateBackendStorageRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBackendStorageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AmplifyBackend");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBackendStorage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateBackendStorageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateBackendStorageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateBackendStorageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateBackendStorageRequest, UpdateBackendStorageResponse>()
                            .withOperationName("UpdateBackendStorage")
                            .withMarshaller(new UpdateBackendStorageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateBackendStorageRequest));
            CompletableFuture<UpdateBackendStorageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final AmplifyBackendServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AmplifyBackendException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("GatewayTimeoutException")
                                .exceptionBuilderSupplier(GatewayTimeoutException::builder).httpStatusCode(504).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
