/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.accessanalyzer.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about an access preview.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AccessPreview implements SdkPojo, Serializable, ToCopyableBuilder<AccessPreview.Builder, AccessPreview> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("id")
            .getter(getter(AccessPreview::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> ANALYZER_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("analyzerArn").getter(getter(AccessPreview::analyzerArn)).setter(setter(Builder::analyzerArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("analyzerArn").build()).build();

    private static final SdkField<Map<String, Configuration>> CONFIGURATIONS_FIELD = SdkField
            .<Map<String, Configuration>> builder(MarshallingType.MAP)
            .memberName("configurations")
            .getter(getter(AccessPreview::configurations))
            .setter(setter(Builder::configurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configurations").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Configuration> builder(MarshallingType.SDK_POJO)
                                            .constructor(Configuration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(AccessPreview::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(AccessPreview::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<AccessPreviewStatusReason> STATUS_REASON_FIELD = SdkField
            .<AccessPreviewStatusReason> builder(MarshallingType.SDK_POJO).memberName("statusReason")
            .getter(getter(AccessPreview::statusReason)).setter(setter(Builder::statusReason))
            .constructor(AccessPreviewStatusReason::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusReason").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, ANALYZER_ARN_FIELD,
            CONFIGURATIONS_FIELD, CREATED_AT_FIELD, STATUS_FIELD, STATUS_REASON_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String analyzerArn;

    private final Map<String, Configuration> configurations;

    private final Instant createdAt;

    private final String status;

    private final AccessPreviewStatusReason statusReason;

    private AccessPreview(BuilderImpl builder) {
        this.id = builder.id;
        this.analyzerArn = builder.analyzerArn;
        this.configurations = builder.configurations;
        this.createdAt = builder.createdAt;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
    }

    /**
     * <p>
     * The unique ID for the access preview.
     * </p>
     * 
     * @return The unique ID for the access preview.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The ARN of the analyzer used to generate the access preview.
     * </p>
     * 
     * @return The ARN of the analyzer used to generate the access preview.
     */
    public final String analyzerArn() {
        return analyzerArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Configurations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasConfigurations() {
        return configurations != null && !(configurations instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A map of resource ARNs for the proposed resource configuration.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasConfigurations} method.
     * </p>
     * 
     * @return A map of resource ARNs for the proposed resource configuration.
     */
    public final Map<String, Configuration> configurations() {
        return configurations;
    }

    /**
     * <p>
     * The time at which the access preview was created.
     * </p>
     * 
     * @return The time at which the access preview was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The status of the access preview.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Creating</code> - The access preview creation is in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Completed</code> - The access preview is complete. You can preview findings for external access to the
     * resource.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Failed</code> - The access preview creation has failed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AccessPreviewStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the access preview.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Creating</code> - The access preview creation is in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Completed</code> - The access preview is complete. You can preview findings for external access to
     *         the resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Failed</code> - The access preview creation has failed.
     *         </p>
     *         </li>
     * @see AccessPreviewStatus
     */
    public final AccessPreviewStatus status() {
        return AccessPreviewStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the access preview.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Creating</code> - The access preview creation is in progress.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Completed</code> - The access preview is complete. You can preview findings for external access to the
     * resource.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Failed</code> - The access preview creation has failed.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AccessPreviewStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the access preview.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Creating</code> - The access preview creation is in progress.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Completed</code> - The access preview is complete. You can preview findings for external access to
     *         the resource.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Failed</code> - The access preview creation has failed.
     *         </p>
     *         </li>
     * @see AccessPreviewStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Provides more details about the current status of the access preview.
     * </p>
     * <p>
     * For example, if the creation of the access preview fails, a <code>Failed</code> status is returned. This failure
     * can be due to an internal issue with the analysis or due to an invalid resource configuration.
     * </p>
     * 
     * @return Provides more details about the current status of the access preview.</p>
     *         <p>
     *         For example, if the creation of the access preview fails, a <code>Failed</code> status is returned. This
     *         failure can be due to an internal issue with the analysis or due to an invalid resource configuration.
     */
    public final AccessPreviewStatusReason statusReason() {
        return statusReason;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(analyzerArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasConfigurations() ? configurations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AccessPreview)) {
            return false;
        }
        AccessPreview other = (AccessPreview) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(analyzerArn(), other.analyzerArn())
                && hasConfigurations() == other.hasConfigurations() && Objects.equals(configurations(), other.configurations())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusReason(), other.statusReason());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AccessPreview").add("Id", id()).add("AnalyzerArn", analyzerArn())
                .add("Configurations", hasConfigurations() ? configurations() : null).add("CreatedAt", createdAt())
                .add("Status", statusAsString()).add("StatusReason", statusReason()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "analyzerArn":
            return Optional.ofNullable(clazz.cast(analyzerArn()));
        case "configurations":
            return Optional.ofNullable(clazz.cast(configurations()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "statusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AccessPreview, T> g) {
        return obj -> g.apply((AccessPreview) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AccessPreview> {
        /**
         * <p>
         * The unique ID for the access preview.
         * </p>
         * 
         * @param id
         *        The unique ID for the access preview.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The ARN of the analyzer used to generate the access preview.
         * </p>
         * 
         * @param analyzerArn
         *        The ARN of the analyzer used to generate the access preview.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder analyzerArn(String analyzerArn);

        /**
         * <p>
         * A map of resource ARNs for the proposed resource configuration.
         * </p>
         * 
         * @param configurations
         *        A map of resource ARNs for the proposed resource configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurations(Map<String, Configuration> configurations);

        /**
         * <p>
         * The time at which the access preview was created.
         * </p>
         * 
         * @param createdAt
         *        The time at which the access preview was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The status of the access preview.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Creating</code> - The access preview creation is in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Completed</code> - The access preview is complete. You can preview findings for external access to the
         * resource.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Failed</code> - The access preview creation has failed.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the access preview.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Creating</code> - The access preview creation is in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Completed</code> - The access preview is complete. You can preview findings for external access
         *        to the resource.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Failed</code> - The access preview creation has failed.
         *        </p>
         *        </li>
         * @see AccessPreviewStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccessPreviewStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the access preview.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Creating</code> - The access preview creation is in progress.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Completed</code> - The access preview is complete. You can preview findings for external access to the
         * resource.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Failed</code> - The access preview creation has failed.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the access preview.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Creating</code> - The access preview creation is in progress.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Completed</code> - The access preview is complete. You can preview findings for external access
         *        to the resource.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Failed</code> - The access preview creation has failed.
         *        </p>
         *        </li>
         * @see AccessPreviewStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AccessPreviewStatus
         */
        Builder status(AccessPreviewStatus status);

        /**
         * <p>
         * Provides more details about the current status of the access preview.
         * </p>
         * <p>
         * For example, if the creation of the access preview fails, a <code>Failed</code> status is returned. This
         * failure can be due to an internal issue with the analysis or due to an invalid resource configuration.
         * </p>
         * 
         * @param statusReason
         *        Provides more details about the current status of the access preview.</p>
         *        <p>
         *        For example, if the creation of the access preview fails, a <code>Failed</code> status is returned.
         *        This failure can be due to an internal issue with the analysis or due to an invalid resource
         *        configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(AccessPreviewStatusReason statusReason);

        /**
         * <p>
         * Provides more details about the current status of the access preview.
         * </p>
         * <p>
         * For example, if the creation of the access preview fails, a <code>Failed</code> status is returned. This
         * failure can be due to an internal issue with the analysis or due to an invalid resource configuration.
         * </p>
         * This is a convenience method that creates an instance of the {@link AccessPreviewStatusReason.Builder}
         * avoiding the need to create one manually via {@link AccessPreviewStatusReason#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AccessPreviewStatusReason.Builder#build()} is called immediately
         * and its result is passed to {@link #statusReason(AccessPreviewStatusReason)}.
         * 
         * @param statusReason
         *        a consumer that will call methods on {@link AccessPreviewStatusReason.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statusReason(AccessPreviewStatusReason)
         */
        default Builder statusReason(Consumer<AccessPreviewStatusReason.Builder> statusReason) {
            return statusReason(AccessPreviewStatusReason.builder().applyMutation(statusReason).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String analyzerArn;

        private Map<String, Configuration> configurations = DefaultSdkAutoConstructMap.getInstance();

        private Instant createdAt;

        private String status;

        private AccessPreviewStatusReason statusReason;

        private BuilderImpl() {
        }

        private BuilderImpl(AccessPreview model) {
            id(model.id);
            analyzerArn(model.analyzerArn);
            configurations(model.configurations);
            createdAt(model.createdAt);
            status(model.status);
            statusReason(model.statusReason);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getAnalyzerArn() {
            return analyzerArn;
        }

        public final void setAnalyzerArn(String analyzerArn) {
            this.analyzerArn = analyzerArn;
        }

        @Override
        public final Builder analyzerArn(String analyzerArn) {
            this.analyzerArn = analyzerArn;
            return this;
        }

        public final Map<String, Configuration.Builder> getConfigurations() {
            Map<String, Configuration.Builder> result = ConfigurationsMapCopier.copyToBuilder(this.configurations);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setConfigurations(Map<String, Configuration.BuilderImpl> configurations) {
            this.configurations = ConfigurationsMapCopier.copyFromBuilder(configurations);
        }

        @Override
        public final Builder configurations(Map<String, Configuration> configurations) {
            this.configurations = ConfigurationsMapCopier.copy(configurations);
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AccessPreviewStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final AccessPreviewStatusReason.Builder getStatusReason() {
            return statusReason != null ? statusReason.toBuilder() : null;
        }

        public final void setStatusReason(AccessPreviewStatusReason.BuilderImpl statusReason) {
            this.statusReason = statusReason != null ? statusReason.build() : null;
        }

        @Override
        public final Builder statusReason(AccessPreviewStatusReason statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        @Override
        public AccessPreview build() {
            return new AccessPreview(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
