/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.security.oauth2.server.authorization.authentication;

import com.nimbusds.jose.jwk.JWK;
import com.nimbusds.jose.jwk.JWKMatcher;
import com.nimbusds.jose.jwk.JWKSet;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.PublicKey;
import java.security.cert.X509Certificate;
import java.text.ParseException;
import java.time.Clock;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Arrays;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import javax.security.auth.x500.X500Principal;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.RequestEntity;
import org.springframework.http.ResponseEntity;
import org.springframework.http.client.ClientHttpRequestFactory;
import org.springframework.http.client.SimpleClientHttpRequestFactory;
import org.springframework.security.oauth2.core.OAuth2AuthenticationException;
import org.springframework.security.oauth2.core.OAuth2Error;
import org.springframework.security.oauth2.server.authorization.authentication.OAuth2ClientAuthenticationContext;
import org.springframework.security.oauth2.server.authorization.authentication.OAuth2ClientAuthenticationToken;
import org.springframework.security.oauth2.server.authorization.client.RegisteredClient;
import org.springframework.util.MultiValueMap;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestOperations;
import org.springframework.web.client.RestTemplate;

final class X509SelfSignedCertificateVerifier
implements Consumer<OAuth2ClientAuthenticationContext> {
    private static final String ERROR_URI = "https://datatracker.ietf.org/doc/html/rfc6749#section-3.2.1";
    private static final JWKMatcher HAS_X509_CERT_CHAIN_MATCHER = new JWKMatcher.Builder().hasX509CertChain(true).build();
    private final Function<RegisteredClient, JWKSet> jwkSetSupplier = new JwkSetSupplier();

    X509SelfSignedCertificateVerifier() {
    }

    @Override
    public void accept(OAuth2ClientAuthenticationContext clientAuthenticationContext) {
        OAuth2ClientAuthenticationToken clientAuthentication = (OAuth2ClientAuthenticationToken)((Object)clientAuthenticationContext.getAuthentication());
        RegisteredClient registeredClient = clientAuthenticationContext.getRegisteredClient();
        X509Certificate[] clientCertificateChain = (X509Certificate[])clientAuthentication.getCredentials();
        X509Certificate clientCertificate = clientCertificateChain[0];
        X500Principal issuer = clientCertificate.getIssuerX500Principal();
        X500Principal subject = clientCertificate.getSubjectX500Principal();
        if (issuer == null || !issuer.equals(subject)) {
            X509SelfSignedCertificateVerifier.throwInvalidClient("x509_certificate_issuer");
        }
        JWKSet jwkSet = this.jwkSetSupplier.apply(registeredClient);
        boolean publicKeyMatches = false;
        for (JWK jwk : jwkSet.filter(HAS_X509_CERT_CHAIN_MATCHER).getKeys()) {
            X509Certificate x509Certificate = (X509Certificate)jwk.getParsedX509CertChain().get(0);
            PublicKey publicKey = x509Certificate.getPublicKey();
            if (!Arrays.equals(clientCertificate.getPublicKey().getEncoded(), publicKey.getEncoded())) continue;
            publicKeyMatches = true;
            break;
        }
        if (!publicKeyMatches) {
            X509SelfSignedCertificateVerifier.throwInvalidClient("x509_certificate");
        }
    }

    private static void throwInvalidClient(String parameterName) {
        X509SelfSignedCertificateVerifier.throwInvalidClient(parameterName, null);
    }

    private static void throwInvalidClient(String parameterName, Throwable cause) {
        OAuth2Error error = new OAuth2Error("invalid_client", "Client authentication failed: " + parameterName, ERROR_URI);
        throw new OAuth2AuthenticationException(error, error.toString(), cause);
    }

    private static final class JwkSetSupplier
    implements Function<RegisteredClient, JWKSet> {
        private static final MediaType APPLICATION_JWK_SET_JSON = new MediaType("application", "jwk-set+json");
        private final RestOperations restOperations;
        private final Map<String, Supplier<JWKSet>> jwkSets = new ConcurrentHashMap<String, Supplier<JWKSet>>();

        private JwkSetSupplier() {
            SimpleClientHttpRequestFactory requestFactory = new SimpleClientHttpRequestFactory();
            requestFactory.setConnectTimeout(15000);
            requestFactory.setReadTimeout(15000);
            this.restOperations = new RestTemplate((ClientHttpRequestFactory)requestFactory);
        }

        @Override
        public JWKSet apply(RegisteredClient registeredClient) {
            Supplier jwkSetSupplier = this.jwkSets.computeIfAbsent(registeredClient.getId(), key -> {
                if (!StringUtils.hasText((String)registeredClient.getClientSettings().getJwkSetUrl())) {
                    X509SelfSignedCertificateVerifier.throwInvalidClient("client_jwk_set_url");
                }
                return new JwkSetHolder(registeredClient.getClientSettings().getJwkSetUrl());
            });
            return (JWKSet)jwkSetSupplier.get();
        }

        private JWKSet retrieve(String jwkSetUrl) {
            URI jwkSetUri = null;
            try {
                jwkSetUri = new URI(jwkSetUrl);
            }
            catch (URISyntaxException ex) {
                X509SelfSignedCertificateVerifier.throwInvalidClient("jwk_set_uri", ex);
            }
            HttpHeaders headers = new HttpHeaders();
            headers.setAccept(Arrays.asList(MediaType.APPLICATION_JSON, APPLICATION_JWK_SET_JSON));
            RequestEntity request = new RequestEntity((MultiValueMap)headers, HttpMethod.GET, jwkSetUri);
            ResponseEntity response = null;
            try {
                response = this.restOperations.exchange(request, String.class);
            }
            catch (Exception ex) {
                X509SelfSignedCertificateVerifier.throwInvalidClient("jwk_set_response_error", ex);
            }
            if (response.getStatusCode().value() != 200) {
                X509SelfSignedCertificateVerifier.throwInvalidClient("jwk_set_response_status");
            }
            JWKSet jwkSet = null;
            try {
                jwkSet = JWKSet.parse((String)((String)response.getBody()));
            }
            catch (ParseException ex) {
                X509SelfSignedCertificateVerifier.throwInvalidClient("jwk_set_response_body", ex);
            }
            return jwkSet;
        }

        private final class JwkSetHolder
        implements Supplier<JWKSet> {
            private final ReentrantReadWriteLock rwLock = new ReentrantReadWriteLock();
            private final Clock clock = Clock.systemUTC();
            private final String jwkSetUrl;
            private JWKSet jwkSet;
            private Instant lastUpdatedAt;

            private JwkSetHolder(String jwkSetUrl) {
                this.jwkSetUrl = jwkSetUrl;
            }

            @Override
            public JWKSet get() {
                this.rwLock.readLock().lock();
                if (this.shouldRefresh()) {
                    this.rwLock.readLock().unlock();
                    this.rwLock.writeLock().lock();
                    try {
                        if (this.shouldRefresh()) {
                            this.jwkSet = JwkSetSupplier.this.retrieve(this.jwkSetUrl);
                            this.lastUpdatedAt = Instant.now();
                        }
                        this.rwLock.readLock().lock();
                    }
                    finally {
                        this.rwLock.writeLock().unlock();
                    }
                }
                try {
                    JWKSet jWKSet = this.jwkSet;
                    return jWKSet;
                }
                finally {
                    this.rwLock.readLock().unlock();
                }
            }

            private boolean shouldRefresh() {
                return this.jwkSet == null || this.clock.instant().isAfter(this.lastUpdatedAt.plus(5L, ChronoUnit.MINUTES));
            }
        }
    }
}

