/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.redis.cache;

import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import org.reactivestreams.Publisher;
import org.springframework.dao.PessimisticLockingFailureException;
import org.springframework.data.redis.cache.BatchStrategy;
import org.springframework.data.redis.cache.CacheStatistics;
import org.springframework.data.redis.cache.CacheStatisticsCollector;
import org.springframework.data.redis.cache.RedisCacheWriter;
import org.springframework.data.redis.connection.ReactiveRedisConnection;
import org.springframework.data.redis.connection.ReactiveRedisConnectionFactory;
import org.springframework.data.redis.connection.ReactiveStringCommands;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.connection.RedisStringCommands;
import org.springframework.data.redis.core.types.Expiration;
import org.springframework.data.redis.util.ByteUtils;
import org.springframework.lang.Nullable;
import org.springframework.util.Assert;
import org.springframework.util.ClassUtils;
import org.springframework.util.ObjectUtils;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

class DefaultRedisCacheWriter
implements RedisCacheWriter {
    private static final boolean REACTIVE_REDIS_CONNECTION_FACTORY_PRESENT = ClassUtils.isPresent((String)"org.springframework.data.redis.connection.ReactiveRedisConnectionFactory", null);
    private final BatchStrategy batchStrategy;
    private final CacheStatisticsCollector statistics;
    private final Duration sleepTime;
    private final RedisConnectionFactory connectionFactory;
    private final RedisCacheWriter.TtlFunction lockTtl;
    private final AsyncCacheWriter asyncCacheWriter;

    DefaultRedisCacheWriter(RedisConnectionFactory connectionFactory, BatchStrategy batchStrategy) {
        this(connectionFactory, Duration.ZERO, batchStrategy);
    }

    DefaultRedisCacheWriter(RedisConnectionFactory connectionFactory, Duration sleepTime, BatchStrategy batchStrategy) {
        this(connectionFactory, sleepTime, RedisCacheWriter.TtlFunction.persistent(), CacheStatisticsCollector.none(), batchStrategy);
    }

    DefaultRedisCacheWriter(RedisConnectionFactory connectionFactory, Duration sleepTime, RedisCacheWriter.TtlFunction lockTtl, CacheStatisticsCollector cacheStatisticsCollector, BatchStrategy batchStrategy) {
        Assert.notNull((Object)connectionFactory, (String)"ConnectionFactory must not be null");
        Assert.notNull((Object)sleepTime, (String)"SleepTime must not be null");
        Assert.notNull((Object)lockTtl, (String)"Lock TTL Function must not be null");
        Assert.notNull((Object)cacheStatisticsCollector, (String)"CacheStatisticsCollector must not be null");
        Assert.notNull((Object)batchStrategy, (String)"BatchStrategy must not be null");
        this.connectionFactory = connectionFactory;
        this.sleepTime = sleepTime;
        this.lockTtl = lockTtl;
        this.statistics = cacheStatisticsCollector;
        this.batchStrategy = batchStrategy;
        this.asyncCacheWriter = REACTIVE_REDIS_CONNECTION_FACTORY_PRESENT && this.connectionFactory instanceof ReactiveRedisConnectionFactory ? new AsynchronousCacheWriterDelegate() : UnsupportedAsyncCacheWriter.INSTANCE;
    }

    @Override
    public byte[] get(String name, byte[] key) {
        return this.get(name, key, null);
    }

    @Override
    public byte[] get(String name, byte[] key, @Nullable Duration ttl) {
        Assert.notNull((Object)name, (String)"Name must not be null");
        Assert.notNull((Object)key, (String)"Key must not be null");
        return this.execute(name, connection -> this.doGet((RedisConnection)connection, name, key, ttl));
    }

    @Nullable
    private byte[] doGet(RedisConnection connection, String name, byte[] key, @Nullable Duration ttl) {
        byte[] result = DefaultRedisCacheWriter.shouldExpireWithin(ttl) ? connection.stringCommands().getEx(key, Expiration.from(ttl)) : connection.stringCommands().get(key);
        this.statistics.incGets(name);
        if (result != null) {
            this.statistics.incHits(name);
        } else {
            this.statistics.incMisses(name);
        }
        return result;
    }

    @Override
    public byte[] get(String name, byte[] key, Supplier<byte[]> valueLoader, @Nullable Duration ttl, boolean timeToIdleEnabled) {
        byte[] bytes;
        Assert.notNull((Object)name, (String)"Name must not be null");
        Assert.notNull((Object)key, (String)"Key must not be null");
        boolean withTtl = DefaultRedisCacheWriter.shouldExpireWithin(ttl);
        if (this.isLockingCacheWriter() && (bytes = this.get(name, key, timeToIdleEnabled && withTtl ? ttl : null)) != null) {
            return bytes;
        }
        return this.execute(name, connection -> {
            if (this.isLockingCacheWriter()) {
                this.doLock(name, key, null, (RedisConnection)connection);
            }
            try {
                byte[] result = this.doGet((RedisConnection)connection, name, key, timeToIdleEnabled && withTtl ? ttl : null);
                if (result != null) {
                    byte[] byArray = result;
                    return byArray;
                }
                byte[] value = (byte[])valueLoader.get();
                this.doPut((RedisConnection)connection, name, key, value, ttl);
                byte[] byArray = value;
                return byArray;
            }
            finally {
                if (this.isLockingCacheWriter()) {
                    this.doUnlock(name, (RedisConnection)connection);
                }
            }
        });
    }

    @Override
    public boolean supportsAsyncRetrieve() {
        return this.asyncCacheWriter.isSupported();
    }

    @Override
    public CompletableFuture<byte[]> retrieve(String name, byte[] key, @Nullable Duration ttl) {
        Assert.notNull((Object)name, (String)"Name must not be null");
        Assert.notNull((Object)key, (String)"Key must not be null");
        return this.asyncCacheWriter.retrieve(name, key, ttl).thenApply(cachedValue -> {
            this.statistics.incGets(name);
            if (cachedValue != null) {
                this.statistics.incHits(name);
            } else {
                this.statistics.incMisses(name);
            }
            return cachedValue;
        });
    }

    @Override
    public void put(String name, byte[] key, byte[] value, @Nullable Duration ttl) {
        Assert.notNull((Object)name, (String)"Name must not be null");
        Assert.notNull((Object)key, (String)"Key must not be null");
        Assert.notNull((Object)value, (String)"Value must not be null");
        this.execute(name, connection -> {
            this.doPut((RedisConnection)connection, name, key, value, ttl);
            return "OK";
        });
    }

    private void doPut(RedisConnection connection, String name, byte[] key, byte[] value, @Nullable Duration ttl) {
        if (DefaultRedisCacheWriter.shouldExpireWithin(ttl)) {
            connection.stringCommands().set(key, value, Expiration.from(ttl.toMillis(), TimeUnit.MILLISECONDS), RedisStringCommands.SetOption.upsert());
        } else {
            connection.stringCommands().set(key, value);
        }
        this.statistics.incPuts(name);
    }

    @Override
    public CompletableFuture<Void> store(String name, byte[] key, byte[] value, @Nullable Duration ttl) {
        Assert.notNull((Object)name, (String)"Name must not be null");
        Assert.notNull((Object)key, (String)"Key must not be null");
        Assert.notNull((Object)value, (String)"Value must not be null");
        return this.asyncCacheWriter.store(name, key, value, ttl).thenRun(() -> this.statistics.incPuts(name));
    }

    @Override
    public byte[] putIfAbsent(String name, byte[] key, byte[] value, @Nullable Duration ttl) {
        Assert.notNull((Object)name, (String)"Name must not be null");
        Assert.notNull((Object)key, (String)"Key must not be null");
        Assert.notNull((Object)value, (String)"Value must not be null");
        return this.execute(name, connection -> {
            if (this.isLockingCacheWriter()) {
                this.doLock(name, key, value, (RedisConnection)connection);
            }
            try {
                boolean put = DefaultRedisCacheWriter.shouldExpireWithin(ttl) ? ObjectUtils.nullSafeEquals((Object)connection.stringCommands().set(key, value, Expiration.from(ttl), RedisStringCommands.SetOption.ifAbsent()), (Object)true) : ObjectUtils.nullSafeEquals((Object)connection.stringCommands().setNX(key, value), (Object)true);
                if (put) {
                    this.statistics.incPuts(name);
                    byte[] byArray = null;
                    return byArray;
                }
                byte[] byArray = connection.stringCommands().get(key);
                return byArray;
            }
            finally {
                if (this.isLockingCacheWriter()) {
                    this.doUnlock(name, (RedisConnection)connection);
                }
            }
        });
    }

    @Override
    public void remove(String name, byte[] key) {
        Assert.notNull((Object)name, (String)"Name must not be null");
        Assert.notNull((Object)key, (String)"Key must not be null");
        this.execute(name, connection -> connection.keyCommands().del(new byte[][]{key}));
        this.statistics.incDeletes(name);
    }

    @Override
    public void clean(String name, byte[] pattern) {
        Assert.notNull((Object)name, (String)"Name must not be null");
        Assert.notNull((Object)pattern, (String)"Pattern must not be null");
        this.execute(name, connection -> {
            try {
                long deleteCount;
                if (this.isLockingCacheWriter()) {
                    this.doLock(name, name, pattern, (RedisConnection)connection);
                }
                for (deleteCount = this.batchStrategy.cleanCache((RedisConnection)connection, name, pattern); deleteCount > Integer.MAX_VALUE; deleteCount -= Integer.MAX_VALUE) {
                    this.statistics.incDeletesBy(name, Integer.MAX_VALUE);
                }
                this.statistics.incDeletesBy(name, (int)deleteCount);
            }
            finally {
                if (this.isLockingCacheWriter()) {
                    this.doUnlock(name, (RedisConnection)connection);
                }
            }
            return "OK";
        });
    }

    @Override
    public CacheStatistics getCacheStatistics(String cacheName) {
        return this.statistics.getCacheStatistics(cacheName);
    }

    @Override
    public void clearStatistics(String name) {
        this.statistics.reset(name);
    }

    @Override
    public RedisCacheWriter withStatisticsCollector(CacheStatisticsCollector cacheStatisticsCollector) {
        return new DefaultRedisCacheWriter(this.connectionFactory, this.sleepTime, this.lockTtl, cacheStatisticsCollector, this.batchStrategy);
    }

    void lock(String name) {
        this.executeWithoutResult(name, connection -> this.doLock(name, name, null, (RedisConnection)connection));
    }

    void doLock(String name, Object contextualKey, @Nullable Object contextualValue, RedisConnection connection) {
        RedisStringCommands commands = connection.stringCommands();
        Expiration expiration = Expiration.from(this.lockTtl.getTimeToLive(contextualKey, contextualValue));
        byte[] cacheLockKey = this.createCacheLockKey(name);
        while (!ObjectUtils.nullSafeEquals((Object)commands.set(cacheLockKey, new byte[0], expiration, RedisStringCommands.SetOption.SET_IF_ABSENT), (Object)true)) {
            this.checkAndPotentiallyWaitUntilUnlocked(name, connection);
        }
    }

    void unlock(String name) {
        this.executeLockFree(connection -> this.doUnlock(name, (RedisConnection)connection));
    }

    @Nullable
    Long doUnlock(String name, RedisConnection connection) {
        return connection.keyCommands().del(new byte[][]{this.createCacheLockKey(name)});
    }

    private <T> T execute(String name, Function<RedisConnection, T> callback) {
        try (RedisConnection connection = this.connectionFactory.getConnection();){
            this.checkAndPotentiallyWaitUntilUnlocked(name, connection);
            T t = callback.apply(connection);
            return t;
        }
    }

    private void executeWithoutResult(String name, Consumer<RedisConnection> callback) {
        try (RedisConnection connection = this.connectionFactory.getConnection();){
            this.checkAndPotentiallyWaitUntilUnlocked(name, connection);
            callback.accept(connection);
        }
    }

    private <T> T executeLockFree(Function<RedisConnection, T> callback) {
        try (RedisConnection connection = this.connectionFactory.getConnection();){
            T t = callback.apply(connection);
            return t;
        }
    }

    private boolean isLockingCacheWriter() {
        return !this.sleepTime.isZero() && !this.sleepTime.isNegative();
    }

    private void checkAndPotentiallyWaitUntilUnlocked(String name, RedisConnection connection) {
        if (!this.isLockingCacheWriter()) {
            return;
        }
        long lockWaitTimeNs = System.nanoTime();
        try {
            while (this.doCheckLock(name, connection)) {
                Thread.sleep(this.sleepTime.toMillis());
            }
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new PessimisticLockingFailureException("Interrupted while waiting to unlock cache %s".formatted(name), (Throwable)ex);
        }
        finally {
            this.statistics.incLockTime(name, System.nanoTime() - lockWaitTimeNs);
        }
    }

    boolean doCheckLock(String name, RedisConnection connection) {
        return ObjectUtils.nullSafeEquals((Object)connection.keyCommands().exists(this.createCacheLockKey(name)), (Object)true);
    }

    byte[] createCacheLockKey(String name) {
        return (name + "~lock").getBytes(StandardCharsets.UTF_8);
    }

    private static boolean shouldExpireWithin(@Nullable Duration ttl) {
        return ttl != null && !ttl.isZero() && !ttl.isNegative();
    }

    class AsynchronousCacheWriterDelegate
    implements AsyncCacheWriter {
        AsynchronousCacheWriterDelegate() {
        }

        @Override
        public boolean isSupported() {
            return true;
        }

        @Override
        public CompletableFuture<byte[]> retrieve(String name, byte[] key, @Nullable Duration ttl) {
            return this.doWithConnection(connection -> {
                ByteBuffer wrappedKey = ByteBuffer.wrap(key);
                Mono<Void> cacheLockCheck = DefaultRedisCacheWriter.this.isLockingCacheWriter() ? this.waitForLock((ReactiveRedisConnection)connection, name) : Mono.empty();
                ReactiveStringCommands stringCommands = connection.stringCommands();
                Mono<ByteBuffer> get = DefaultRedisCacheWriter.shouldExpireWithin(ttl) ? stringCommands.getEx(wrappedKey, Expiration.from(ttl)) : stringCommands.get(wrappedKey);
                return cacheLockCheck.then(get).map(ByteUtils::getBytes).toFuture();
            });
        }

        @Override
        public CompletableFuture<Void> store(String name, byte[] key, byte[] value, @Nullable Duration ttl) {
            return this.doWithConnection(connection -> {
                Mono<Boolean> mono = DefaultRedisCacheWriter.this.isLockingCacheWriter() ? this.doStoreWithLocking(name, key, value, ttl, (ReactiveRedisConnection)connection) : this.doStore(key, value, ttl, (ReactiveRedisConnection)connection);
                return mono.then().toFuture();
            });
        }

        private Mono<Boolean> doStoreWithLocking(String name, byte[] key, byte[] value, @Nullable Duration ttl, ReactiveRedisConnection connection) {
            return Mono.usingWhen(this.doLock(name, key, value, connection), unused -> this.doStore(key, value, ttl, connection), unused -> this.doUnlock(name, connection));
        }

        private Mono<Boolean> doStore(byte[] cacheKey, byte[] value, @Nullable Duration ttl, ReactiveRedisConnection connection) {
            ByteBuffer wrappedKey = ByteBuffer.wrap(cacheKey);
            ByteBuffer wrappedValue = ByteBuffer.wrap(value);
            if (DefaultRedisCacheWriter.shouldExpireWithin(ttl)) {
                return connection.stringCommands().set(wrappedKey, wrappedValue, Expiration.from(ttl.toMillis(), TimeUnit.MILLISECONDS), RedisStringCommands.SetOption.upsert());
            }
            return connection.stringCommands().set(wrappedKey, wrappedValue);
        }

        private Mono<Object> doLock(String name, Object contextualKey, @Nullable Object contextualValue, ReactiveRedisConnection connection) {
            ByteBuffer key = ByteBuffer.wrap(DefaultRedisCacheWriter.this.createCacheLockKey(name));
            ByteBuffer value = ByteBuffer.wrap(new byte[0]);
            Expiration expiration = Expiration.from(DefaultRedisCacheWriter.this.lockTtl.getTimeToLive(contextualKey, contextualValue));
            return connection.stringCommands().set(key, value, expiration, RedisStringCommands.SetOption.SET_IF_ABSENT).thenReturn((Object)Boolean.TRUE);
        }

        private Mono<Void> doUnlock(String name, ReactiveRedisConnection connection) {
            return connection.keyCommands().del(ByteBuffer.wrap(DefaultRedisCacheWriter.this.createCacheLockKey(name))).then();
        }

        private Mono<Void> waitForLock(ReactiveRedisConnection connection, String cacheName) {
            AtomicLong lockWaitTimeNs = new AtomicLong();
            byte[] cacheLockKey = DefaultRedisCacheWriter.this.createCacheLockKey(cacheName);
            Flux wait = Flux.interval((Duration)Duration.ZERO, (Duration)DefaultRedisCacheWriter.this.sleepTime);
            Mono exists = connection.keyCommands().exists(ByteBuffer.wrap(cacheLockKey)).filter(it -> it == false);
            return wait.doOnSubscribe(subscription -> lockWaitTimeNs.set(System.nanoTime())).flatMap(it -> exists).doFinally(signalType -> DefaultRedisCacheWriter.this.statistics.incLockTime(cacheName, System.nanoTime() - lockWaitTimeNs.get())).next().then();
        }

        private <T> CompletableFuture<T> doWithConnection(Function<ReactiveRedisConnection, CompletableFuture<T>> callback) {
            ReactiveRedisConnectionFactory cf = (ReactiveRedisConnectionFactory)((Object)DefaultRedisCacheWriter.this.connectionFactory);
            return Mono.usingWhen((Publisher)Mono.fromSupplier(cf::getReactiveConnection), it -> Mono.fromCompletionStage((CompletionStage)((CompletionStage)callback.apply((ReactiveRedisConnection)it))), ReactiveRedisConnection::closeLater).toFuture();
        }
    }

    static interface AsyncCacheWriter {
        public boolean isSupported();

        public CompletableFuture<byte[]> retrieve(String var1, byte[] var2, @Nullable Duration var3);

        public CompletableFuture<Void> store(String var1, byte[] var2, byte[] var3, @Nullable Duration var4);
    }

    static enum UnsupportedAsyncCacheWriter implements AsyncCacheWriter
    {
        INSTANCE;


        @Override
        public boolean isSupported() {
            return false;
        }

        @Override
        public CompletableFuture<byte[]> retrieve(String name, byte[] key, @Nullable Duration ttl) {
            throw new UnsupportedOperationException("async retrieve not supported");
        }

        @Override
        public CompletableFuture<Void> store(String name, byte[] key, byte[] value, @Nullable Duration ttl) {
            throw new UnsupportedOperationException("async store not supported");
        }
    }
}

