package dotty.tools
package dotc
package cc

import core._
import Phases.*, DenotTransformers.*, SymDenotations.*
import Contexts.*, Names.*, Flags.*, Symbols.*, Decorators.*
import Types.*, StdNames.*
import config.Printers.capt
import ast.tpd
import transform.Recheck.*
import CaptureSet.IdentityCaptRefMap
import Synthetics.isExcluded
import util.Property

/** A tree traverser that prepares a compilation unit to be capture checked.
 *  It does the following:
 *    - For every inferred type, drop any retains annotations,
 *      add capture sets to all its parts, add refinements to class types and function types.
 *      (c.f. mapInferred)
 *    - For explicit capturing types, expand throws aliases to the underlying (pure) function,
 *      and add some implied capture sets to curried functions (c.f. expandThrowsAlias, expandAbbreviations).
 *    - Add capture sets to self types of classes and objects, unless the self type was written explicitly.
 *    - Box the types of mutable variables and type arguments to methods (type arguments of types
 *      are boxed on access).
 *    - Link the external types of val and def symbols with the inferred types based on their parameter symbols.
 */
class Setup(
  preRecheckPhase: DenotTransformer,
  thisPhase: DenotTransformer,
  recheckDef: (tpd.ValOrDefDef, Symbol) => Context ?=> Unit)
extends tpd.TreeTraverser:
  import tpd.*

  /** Create dependent function with underlying function class `tycon` and given
   *  arguments `argTypes` and result `resType`.
   */
  private def depFun(tycon: Type, argTypes: List[Type], resType: Type)(using Context): Type =
    MethodType.companion(
        isContextual = defn.isContextFunctionClass(tycon.classSymbol),
        isErased = defn.isErasedFunctionClass(tycon.classSymbol)
      )(argTypes, resType)
      .toFunctionType(isJava = false, alwaysDependent = true)

  /** If `tp` is an unboxed capturing type or a function returning an unboxed capturing type,
   *  convert it to be boxed.
   */
  private def box(tp: Type)(using Context): Type =
    def recur(tp: Type): Type = tp.dealias match
      case tp @ CapturingType(parent, refs) if !tp.isBoxed =>
        tp.boxed
      case tp1 @ AppliedType(tycon, args) if defn.isNonRefinedFunction(tp1) =>
        val res = args.last
        val boxedRes = recur(res)
        if boxedRes eq res then tp
        else tp1.derivedAppliedType(tycon, args.init :+ boxedRes)
      case tp1 @ RefinedType(_, _, rinfo) if defn.isFunctionType(tp1) =>
        val boxedRinfo = recur(rinfo)
        if boxedRinfo eq rinfo then tp
        else boxedRinfo.toFunctionType(isJava = false, alwaysDependent = true)
      case tp1: MethodOrPoly =>
        val res = tp1.resType
        val boxedRes = recur(res)
        if boxedRes eq res then tp
        else tp1.derivedLambdaType(resType = boxedRes)
      case _ => tp
    tp match
      case tp: MethodOrPoly => tp // don't box results of methods outside refinements
      case _ => recur(tp)

  /** Perform the following transformation steps everywhere in a type:
    *  1. Drop retains annotations
    *  2. Turn plain function types into dependent function types, so that
    *     we can refer to their parameters in capture sets. Currently this is
    *     only done at the toplevel, i.e. for function types that are not
    *     themselves argument types of other function types. Without this restriction
    *     pos.../lists.scala and pos/...curried-shorthands.scala fail.
    *     Need to figure out why.
    *  3. Refine other class types C by adding capture set variables to their parameter getters
    *     (see addCaptureRefinements)
    *  4. Add capture set variables to all types that can be tracked
    *
    *  Polytype bounds are only cleaned using step 1, but not otherwise transformed.
    */
  private def mapInferred(using Context) = new TypeMap:

    /** Drop @retains annotations everywhere */
    object cleanup extends TypeMap:
      def apply(t: Type) = t match
        case AnnotatedType(parent, annot) if annot.symbol == defn.RetainsAnnot =>
          apply(parent)
        case _ =>
          mapOver(t)

    /** Refine a possibly applied class type C where the class has tracked parameters
     *  x_1: T_1, ..., x_n: T_n to C { val x_1: CV_1 T_1, ..., val x_n: CV_n T_n }
     *  where CV_1, ..., CV_n are fresh capture sets.
     */
    def addCaptureRefinements(tp: Type): Type = tp match
      case _: TypeRef | _: AppliedType if tp.typeParams.isEmpty =>
        tp.typeSymbol match
          case cls: ClassSymbol
          if !defn.isFunctionClass(cls) && !cls.is(JavaDefined) =>
            // We assume that Java classes can refer to capturing Scala types only indirectly,
            // using type parameters. Hence, no need to refine them.
            cls.paramGetters.foldLeft(tp) { (core, getter) =>
              if getter.termRef.isTracked then
                val getterType = tp.memberInfo(getter).strippedDealias
                RefinedType(core, getter.name, CapturingType(getterType, CaptureSet.Var()))
                  .showing(i"add capture refinement $tp --> $result", capt)
              else
                core
            }
          case _ => tp
      case _ => tp

    private def superTypeIsImpure(tp: Type): Boolean = {
      tp.dealias match
        case CapturingType(_, refs) =>
          !refs.isAlwaysEmpty
        case tp: (TypeRef | AppliedType) =>
          val sym = tp.typeSymbol
          if sym.isClass then
            sym == defn.AnyClass
              // we assume Any is a shorthand of {*} Any, so if Any is an upper
              // bound, the type is taken to be impure.
          else superTypeIsImpure(tp.superType)
        case tp: (RefinedOrRecType | MatchType) =>
          superTypeIsImpure(tp.underlying)
        case tp: AndType =>
          superTypeIsImpure(tp.tp1) || needsVariable(tp.tp2)
        case tp: OrType =>
          superTypeIsImpure(tp.tp1) && superTypeIsImpure(tp.tp2)
        case _ =>
          false
    }.showing(i"super type is impure $tp = $result", capt)

    /** Should a capture set variable be added on type `tp`? */
    def needsVariable(tp: Type): Boolean = {
      tp.typeParams.isEmpty && tp.match
        case tp: (TypeRef | AppliedType) =>
          val tp1 = tp.dealias
          if tp1 ne tp then needsVariable(tp1)
          else
            val sym = tp1.typeSymbol
            if sym.isClass then
              !sym.isPureClass && sym != defn.AnyClass
            else superTypeIsImpure(tp1)
        case tp: (RefinedOrRecType | MatchType) =>
          needsVariable(tp.underlying)
        case tp: AndType =>
          needsVariable(tp.tp1) && needsVariable(tp.tp2)
        case tp: OrType =>
          needsVariable(tp.tp1) || needsVariable(tp.tp2)
        case CapturingType(parent, refs) =>
          needsVariable(parent)
          && refs.isConst      // if refs is a variable, no need to add another
          && !refs.isUniversal // if refs is {*}, an added variable would not change anything
        case _ =>
          false
    }.showing(i"can have inferred capture $tp = $result", capt)

    /** Add a capture set variable to `tp` if necessary, or maybe pull out
     *  an embedded capture set variable from a part of `tp`.
     */
    def addVar(tp: Type) = tp match
      case tp @ RefinedType(parent @ CapturingType(parent1, refs), rname, rinfo) =>
        CapturingType(tp.derivedRefinedType(parent1, rname, rinfo), refs, parent.isBoxed)
      case tp: RecType =>
        tp.parent match
          case parent @ CapturingType(parent1, refs) =>
            CapturingType(tp.derivedRecType(parent1), refs, parent.isBoxed)
          case _ =>
            tp // can return `tp` here since unlike RefinedTypes, RecTypes are never created
                // by `mapInferred`. Hence if the underlying type admits capture variables
                // a variable was already added, and the first case above would apply.
      case AndType(tp1 @ CapturingType(parent1, refs1), tp2 @ CapturingType(parent2, refs2)) =>
        assert(refs1.asVar.elems.isEmpty)
        assert(refs2.asVar.elems.isEmpty)
        assert(tp1.isBoxed == tp2.isBoxed)
        CapturingType(AndType(parent1, parent2), refs1 ** refs2, tp1.isBoxed)
      case tp @ OrType(tp1 @ CapturingType(parent1, refs1), tp2 @ CapturingType(parent2, refs2)) =>
        assert(refs1.asVar.elems.isEmpty)
        assert(refs2.asVar.elems.isEmpty)
        assert(tp1.isBoxed == tp2.isBoxed)
        CapturingType(OrType(parent1, parent2, tp.isSoft), refs1 ++ refs2, tp1.isBoxed)
      case tp @ OrType(tp1 @ CapturingType(parent1, refs1), tp2) =>
        CapturingType(OrType(parent1, tp2, tp.isSoft), refs1, tp1.isBoxed)
      case tp @ OrType(tp1, tp2 @ CapturingType(parent2, refs2)) =>
        CapturingType(OrType(tp1, parent2, tp.isSoft), refs2, tp2.isBoxed)
      case _ if needsVariable(tp) =>
        val cs = tp.dealias match
          case CapturingType(_, refs) => CaptureSet.Var(refs.elems)
          case _ => CaptureSet.Var()
        CapturingType(tp, cs)
      case _ =>
        tp

    private var isTopLevel = true

    private def mapNested(ts: List[Type]): List[Type] =
      val saved = isTopLevel
      isTopLevel = false
      try ts.mapConserve(this) finally isTopLevel = saved

    def apply(t: Type) =
      val tp = expandThrowsAlias(t)
      val tp1 = tp match
        case AnnotatedType(parent, annot) if annot.symbol == defn.RetainsAnnot =>
          // Drop explicit retains annotations
          apply(parent)
        case tp @ AppliedType(tycon, args) =>
          val tycon1 = this(tycon)
          if defn.isNonRefinedFunction(tp) then
            // Convert toplevel generic function types to dependent functions
            if !defn.isFunctionSymbol(tp.typeSymbol) && (tp.dealias ne tp) then
              // This type is a function after dealiasing, so we dealias and recurse.
              // See #15925.
              this(tp.dealias)
            else
              val args0 = args.init
              var res0 = args.last
              val args1 = mapNested(args0)
              val res1 = this(res0)
              if isTopLevel then
                depFun(tycon1, args1, res1)
                  .showing(i"add function refinement $tp ($tycon1, $args1, $res1) (${tp.dealias}) --> $result", capt)
              else if (tycon1 eq tycon) && (args1 eq args0) && (res1 eq res0) then
                tp
              else
                tp.derivedAppliedType(tycon1, args1 :+ res1)
          else
            tp.derivedAppliedType(tycon1, args.mapConserve(arg => this(arg)))
        case tp @ RefinedType(core, rname, rinfo) if defn.isFunctionType(tp) =>
          val rinfo1 = apply(rinfo)
          if rinfo1 ne rinfo then rinfo1.toFunctionType(isJava = false, alwaysDependent = true)
          else tp
        case tp: MethodType =>
          tp.derivedLambdaType(
            paramInfos = mapNested(tp.paramInfos),
            resType = this(tp.resType))
        case tp: TypeLambda =>
          // Don't recurse into parameter bounds, just cleanup any stray retains annotations
          tp.derivedLambdaType(
            paramInfos = tp.paramInfos.mapConserve(cleanup(_).bounds),
            resType = this(tp.resType))
        case _ =>
          mapOver(tp)
      addVar(addCaptureRefinements(tp1))
    end apply
  end mapInferred

  private def transformInferredType(tp: Type, boxed: Boolean)(using Context): Type =
    val tp1 = mapInferred(tp)
    if boxed then box(tp1) else tp1

  /** Expand some aliases of function types to the underlying functions.
   *  Right now, these are only $throws aliases, but this could be generalized.
   */
  private def expandThrowsAlias(tp: Type)(using Context) = tp match
    case AppliedType(tycon, res :: exc :: Nil) if tycon.typeSymbol == defn.throwsAlias =>
      // hard-coded expansion since $throws aliases in stdlib are defined with `?=>` rather than `?->`
      defn.FunctionOf(defn.CanThrowClass.typeRef.appliedTo(exc) :: Nil, res, isContextual = true, isErased = true)
    case _ => tp

  private def expandThrowsAliases(using Context) = new TypeMap:
    def apply(t: Type) = t match
      case _: AppliedType =>
        val t1 = expandThrowsAlias(t)
        if t1 ne t then apply(t1) else mapOver(t)
      case _: LazyRef =>
        t
      case t @ AnnotatedType(t1, ann) =>
        // Don't map capture sets, since that would implicitly normalize sets that
        // are not well-formed.
        t.derivedAnnotatedType(apply(t1), ann)
      case _ =>
        mapOver(t)

  /** Fill in capture sets of curried function types from left to right, using
   *  a combination of the following two rules:
   *
   *   1. Expand `{c} (x: A) -> (y: B) -> C`
   *          to `{c} (x: A) -> {c} (y: B) -> C`
   *   2. Expand `(x: A) -> (y: B) -> C` where `x` is tracked
   *          to `(x: A) -> {x} (y: B) -> C`
   *
   *  TODO: Should we also propagate capture sets to the left?
   */
  private def expandAbbreviations(using Context) = new TypeMap:

    /** Propagate `outerCs` as well as all tracked parameters as capture set to the result type
     *  of the dependent function type `tp`.
     */
    def propagateDepFunctionResult(tp: Type, outerCs: CaptureSet): Type = tp match
      case RefinedType(parent, nme.apply, rinfo: MethodType) =>
        val localCs = CaptureSet(rinfo.paramRefs.filter(_.isTracked)*)
        val rinfo1 = rinfo.derivedLambdaType(
          resType = propagateEnclosing(rinfo.resType, CaptureSet.empty, outerCs ++ localCs))
        if rinfo1 ne rinfo then rinfo1.toFunctionType(isJava = false, alwaysDependent = true)
        else tp

    /** If `tp` is a function type:
     *   - add `outerCs` as its capture set,
     *   - propagate `currentCs`, `outerCs`, and all tracked parameters of `tp` to the right.
     */
    def propagateEnclosing(tp: Type, currentCs: CaptureSet, outerCs: CaptureSet): Type = tp match
      case tp @ AppliedType(tycon, args) if defn.isFunctionClass(tycon.typeSymbol) =>
        val tycon1 = this(tycon)
        val args1 = args.init.mapConserve(this)
        val tp1 =
          if args1.exists(!_.captureSet.isAlwaysEmpty) then
            val propagated = propagateDepFunctionResult(
              depFun(tycon, args1, args.last), currentCs ++ outerCs)
            propagated match
              case RefinedType(_, _, mt: MethodType) =>
                if mt.isCaptureDependent then propagated
                else
                  // No need to introduce dependent type, switch back to generic function type
                  tp.derivedAppliedType(tycon1, args1 :+ mt.resType)
          else
            val resType1 = propagateEnclosing(
              args.last, CaptureSet.empty, currentCs ++ outerCs)
            tp.derivedAppliedType(tycon1, args1 :+ resType1)
        tp1.capturing(outerCs)
      case tp @ RefinedType(parent, nme.apply, rinfo: MethodType) if defn.isFunctionType(tp) =>
        propagateDepFunctionResult(mapOver(tp), currentCs ++ outerCs)
          .capturing(outerCs)
      case _ =>
        mapOver(tp)

    def apply(tp: Type): Type = tp match
      case CapturingType(parent, cs) =>
        tp.derivedCapturingType(propagateEnclosing(parent, cs, CaptureSet.empty), cs)
      case _ =>
        propagateEnclosing(tp, CaptureSet.empty, CaptureSet.empty)
  end expandAbbreviations

  private def transformExplicitType(tp: Type, boxed: Boolean)(using Context): Type =
    val tp1 = expandThrowsAliases(if boxed then box(tp) else tp)
    if tp1 ne tp then capt.println(i"expanded: $tp --> $tp1")
    if ctx.settings.YccNoAbbrev.value then tp1
    else expandAbbreviations(tp1)

  /** Transform type of type tree, and remember the transformed type as the type the tree */
  private def transformTT(tree: TypeTree, boxed: Boolean, exact: Boolean)(using Context): Unit =
    if !tree.hasRememberedType then
      tree.rememberType(
        if tree.isInstanceOf[InferredTypeTree] && !exact
        then transformInferredType(tree.tpe, boxed)
        else transformExplicitType(tree.tpe, boxed))

  /** Substitute parameter symbols in `from` to paramRefs in corresponding
   *  method or poly types `to`. We use a single BiTypeMap to do everything.
   *  @param from  a list of lists of type or term parameter symbols of a curried method
   *  @param to    a list of method or poly types corresponding one-to-one to the parameter lists
   */
  private class SubstParams(from: List[List[Symbol]], to: List[LambdaType])(using Context)
  extends DeepTypeMap, BiTypeMap:

    def apply(t: Type): Type = t match
      case t: NamedType =>
        val sym = t.symbol
        def outer(froms: List[List[Symbol]], tos: List[LambdaType]): Type =
          def inner(from: List[Symbol], to: List[ParamRef]): Type =
            if from.isEmpty then outer(froms.tail, tos.tail)
            else if sym eq from.head then to.head
            else inner(from.tail, to.tail)
          if tos.isEmpty then t
          else inner(froms.head, tos.head.paramRefs)
        outer(from, to)
      case _ =>
        mapOver(t)

    def inverse(t: Type): Type = t match
      case t: ParamRef =>
        def recur(from: List[LambdaType], to: List[List[Symbol]]): Type =
          if from.isEmpty then t
          else if t.binder eq from.head then to.head(t.paramNum).namedType
          else recur(from.tail, to.tail)
        recur(to, from)
      case _ =>
        mapOver(t)
  end SubstParams

  /** Update info of `sym` for CheckCaptures phase only */
  private def updateInfo(sym: Symbol, info: Type)(using Context) =
    sym.updateInfoBetween(preRecheckPhase, thisPhase, info)

  def traverse(tree: Tree)(using Context): Unit =
    tree match
      case tree: DefDef =>
        if isExcluded(tree.symbol) then
          return
        tree.tpt match
          case tpt: TypeTree if tree.symbol.allOverriddenSymbols.hasNext =>
            tree.paramss.foreach(traverse)
            transformTT(tpt, boxed = false, exact = true)
            traverse(tree.rhs)
            //println(i"TYPE of ${tree.symbol.showLocated} = ${tpt.knownType}")
          case _ =>
            traverseChildren(tree)
      case tree @ ValDef(_, tpt: TypeTree, _) =>
        transformTT(tpt,
          boxed = tree.symbol.is(Mutable),    // types of mutable variables are boxed
          exact = tree.symbol.allOverriddenSymbols.hasNext // types of symbols that override a parent don't get a capture set
        )
        traverse(tree.rhs)
      case tree @ TypeApply(fn, args) =>
        traverse(fn)
        for case arg: TypeTree <- args do
          transformTT(arg, boxed = true, exact = false) // type arguments in type applications are boxed
      case _ =>
        traverseChildren(tree)
    tree match
      case tree: TypeTree =>
        transformTT(tree, boxed = false, exact = false) // other types are not boxed
      case tree: ValOrDefDef =>
        val sym = tree.symbol

        // replace an existing symbol info with inferred types where capture sets of
        // TypeParamRefs and TermParamRefs put in correspondence by BiTypeMaps with the
        // capture sets of the types of the method's parameter symbols and result type.
        def integrateRT(
            info: Type,                     // symbol info to replace
            psymss: List[List[Symbol]],     // the local (type and term) parameter symbols corresponding to `info`
            prevPsymss: List[List[Symbol]], // the local parameter symbols seen previously in reverse order
            prevLambdas: List[LambdaType]   // the outer method and polytypes generated previously in reverse order
          ): Type =
          info match
            case mt: MethodOrPoly =>
              val psyms = psymss.head
              mt.companion(mt.paramNames)(
                mt1 =>
                  if !psyms.exists(_.isUpdatedAfter(preRecheckPhase)) && !mt.isParamDependent && prevLambdas.isEmpty then
                    mt.paramInfos
                  else
                    val subst = SubstParams(psyms :: prevPsymss, mt1 :: prevLambdas)
                    psyms.map(psym => subst(psym.info).asInstanceOf[mt.PInfo]),
                mt1 =>
                  integrateRT(mt.resType, psymss.tail, psyms :: prevPsymss, mt1 :: prevLambdas)
              )
            case info: ExprType =>
              info.derivedExprType(resType =
                integrateRT(info.resType, psymss, prevPsymss, prevLambdas))
            case _ =>
              val restp = tree.tpt.knownType
              if prevLambdas.isEmpty then restp
              else SubstParams(prevPsymss, prevLambdas)(restp)

        if tree.tpt.hasRememberedType && !sym.isConstructor then
          val newInfo = integrateRT(sym.info, sym.paramSymss, Nil, Nil)
            .showing(i"update info $sym: ${sym.info} --> $result", capt)
          if newInfo ne sym.info then
            val completer = new LazyType:
              def complete(denot: SymDenotation)(using Context) =
                denot.info = newInfo
                recheckDef(tree, sym)
            updateInfo(sym, completer)
      case tree: Bind =>
        val sym = tree.symbol
        updateInfo(sym, transformInferredType(sym.info, boxed = false))
      case tree: TypeDef =>
        tree.symbol match
          case cls: ClassSymbol =>
            val cinfo @ ClassInfo(prefix, _, ps, decls, selfInfo) = cls.classInfo
            if (selfInfo eq NoType) || cls.is(ModuleClass) && !cls.isStatic then
              // add capture set to self type of nested classes if no self type is given explicitly
              val localRefs = CaptureSet.Var()
              val newInfo = ClassInfo(prefix, cls, ps, decls,
                CapturingType(cinfo.selfType, localRefs)
                  .showing(i"inferred self type for $cls: $result", capt))
              updateInfo(cls, newInfo)
              cls.thisType.asInstanceOf[ThisType].invalidateCaches()
              if cls.is(ModuleClass) then
                // if it's a module, the capture set of the module reference is the capture set of the self type
                val modul = cls.sourceModule
                updateInfo(modul, CapturingType(modul.info, localRefs))
                modul.termRef.invalidateCaches()
          case _ =>
            val info = atPhase(preRecheckPhase)(tree.symbol.info)
            val newInfo = transformExplicitType(info, boxed = false)
            if newInfo ne info then
              updateInfo(tree.symbol, newInfo)
              capt.println(i"update info of ${tree.symbol} from $info to $newInfo")
      case _ =>
  end traverse

  def apply(tree: Tree)(using Context): Unit =
    traverse(tree)(using ctx.withProperty(Setup.IsDuringSetupKey, Some(())))
end Setup

object Setup:
  val IsDuringSetupKey = new Property.Key[Unit]

  def isDuringSetup(using Context): Boolean =
    ctx.property(IsDuringSetupKey).isDefined

