/*
 * (C) Copyright 2007 Nuxeo SAS (http://nuxeo.com/) and contributors.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * Contributors:
 *     Nuxeo - initial API and implementation
 *
 * $Id$
 */

package org.nuxeo.runtime.api;

import java.util.Map;
import java.util.Properties;

import org.nuxeo.common.xmap.annotation.XNode;
import org.nuxeo.common.xmap.annotation.XNodeList;
import org.nuxeo.common.xmap.annotation.XNodeMap;
import org.nuxeo.common.xmap.annotation.XObject;

/**
 * @author <a href="mailto:bs@nuxeo.com">Bogdan Stefanescu</a>
 *
 */
@XObject("server")
public class ServiceHost {

    public static final ServiceHost LOCAL_SERVER = new ServiceHost(RuntimeServiceLocator.class);
    
    @XNode("@class")
    private Class<? extends ServiceLocator> serviceLocatorClass;
    
    @XNode("@host")
    private String host;
    
    @XNode("@port")
    private int port;

    private Properties properties;


    private ServiceGroup[] groups;

    private ServiceLocator serviceLocator;

    
    public ServiceHost() {
    }

    public ServiceHost(Class<? extends ServiceLocator> serverClass) {
        serviceLocatorClass = serverClass;
    }

    public ServiceHost(Class<? extends ServiceLocator> serverClass, String[] groups) {
        this (serverClass);
        setGroups(groups);
    }

    protected ServiceLocator createServiceLocator() throws Exception {
        ServiceLocator serviceLocator = serviceLocatorClass.newInstance();
        serviceLocator.initialize(host, port, properties);
        return serviceLocator;
    }
    
    /**
     * @return the host.
     */
    public String getHost() {
        return host;
    }
    
    /**
     * @return the port.
     */
    public int getPort() {
        return port;
    }
    
    
    public void setAddress(String host, int port) {
        this.host = host;
        this.port = port;
    }

    /**
     * @param properties the properties to set.
     */
    @XNodeMap(value = "property", key = "@name", componentType = String.class, type = Properties.class, trim = true)
    public void setProperties(Properties properties) {
        this.properties = new Properties();
        // expand properties 
        for (Map.Entry<Object, Object> entry : properties.entrySet()) {
            Object value = entry.getValue();
            if (value.getClass() == String.class) {
                this.properties.put(entry.getKey().toString(), Framework.expandVars(value.toString().trim()));    
            } else {
                this.properties.put(entry.getKey(), value);
            }
        }
    }

    /**
     * @return the properties.
     */
    public Properties getProperties() {
        return properties;
    }

    public String getProperty(String key) {
        return properties.getProperty(key);
    }

    public String getProperty(String key, String defValue) {
        return properties.getProperty(key, defValue);
    }

    @XNodeList(value = "group",  componentType = String.class, type = String[].class, trim = true)
    public void setGroups(String[] groups) {
        this.groups = new ServiceGroup[groups.length];

        ServiceManager mgr = ServiceManager.getInstance();
        for (int i = 0; i < groups.length; i++) {
            this.groups[i] = mgr.getOrCreateGroup(groups[i]);
            this.groups[i].setServer(this);
        }
    }

    /**
     * @return the groups.
     */
    public ServiceGroup[] getGroups() {
        if (groups == null) {
            groups = new ServiceGroup[] {ServiceManager.getInstance().getRootGroup()};
        }
        return groups;
    }

    public ServiceLocator getServiceLocator() throws Exception {
        if (serviceLocator == null) {
            serviceLocator = createServiceLocator();
        }
        return serviceLocator;
    }

    public void dispose() {
        if (groups != null) {
            // remove cached information from groups
            for (ServiceGroup group : groups) {
                group.setServer(null);
            }
        }
        if (serviceLocator != null) {
            serviceLocator.dispose();
            serviceLocator = null;
        }
        serviceLocatorClass = null;
        properties = null;
        groups = null;
    }

    public Object lookup(ServiceDescriptor sd) throws Exception {
        Object service = getServiceLocator().lookup(sd);
        if (service == null) {
            return null;
        }
        ServiceAdapter adapter = sd.getAdapter();
        if (adapter != null) {
            return adapter.adapt(sd, service);
        }
        return service;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj == this) {
            return true;
        }
        if (!(obj instanceof ServiceHost)) {
            return false;
        }
        ServiceHost server = (ServiceHost) obj;
        return server.serviceLocatorClass == serviceLocatorClass;
    }

    @Override
    public int hashCode() {
        return serviceLocatorClass != null ? serviceLocatorClass.hashCode() : 0;
    }

}
