/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.env;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.lucene.store.Lock;
import org.apache.lucene.store.LockObtainFailedException;
import org.apache.lucene.store.NativeFSLockFactory;
import org.apache.lucene.util.IOUtils;
import org.apache.lucene.util.XIOUtils;
import org.elasticsearch.ElasticsearchIllegalArgumentException;
import org.elasticsearch.ElasticsearchIllegalStateException;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.collect.ImmutableSet;
import org.elasticsearch.common.collect.Sets;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.io.FileSystemUtils;
import org.elasticsearch.common.primitives.Ints;
import org.elasticsearch.common.settings.ImmutableSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.env.Environment;
import org.elasticsearch.env.ShardLock;
import org.elasticsearch.index.Index;
import org.elasticsearch.index.settings.IndexSettings;
import org.elasticsearch.index.shard.ShardId;

public class NodeEnvironment
extends AbstractComponent
implements Closeable {
    private final Path[] nodePaths;
    private final Path[] nodeIndicesPaths;
    private final Lock[] locks;
    private final boolean addNodeId;
    private final int localNodeId;
    private final AtomicBoolean closed = new AtomicBoolean(false);
    private final Map<ShardId, InternalShardLock> shardLocks = new HashMap<ShardId, InternalShardLock>();
    private final boolean customPathsEnabled;
    public static final String ADD_NODE_ID_TO_CUSTOM_PATH = "node.add_id_to_custom_path";
    public static final String SETTING_CUSTOM_DATA_PATH_ENABLED = "node.enable_custom_paths";
    public static final String NODES_FOLDER = "nodes";
    public static final String INDICES_FOLDER = "indices";
    public static final String NODE_LOCK_FILENAME = "node.lock";

    @Inject
    public NodeEnvironment(Settings settings, Environment environment) throws IOException {
        super(settings);
        this.addNodeId = settings.getAsBoolean(ADD_NODE_ID_TO_CUSTOM_PATH, (Boolean)true);
        this.customPathsEnabled = settings.getAsBoolean(SETTING_CUSTOM_DATA_PATH_ENABLED, (Boolean)false);
        if (!DiscoveryNode.nodeRequiresLocalStorage(settings)) {
            this.nodePaths = null;
            this.nodeIndicesPaths = null;
            this.locks = null;
            this.localNodeId = -1;
            return;
        }
        Path[] nodePaths = new Path[environment.dataWithClusterFiles().length];
        Lock[] locks = new Lock[environment.dataWithClusterFiles().length];
        int localNodeId = -1;
        IOException lastException = null;
        int maxLocalStorageNodes = settings.getAsInt("node.max_local_storage_nodes", (Integer)50);
        for (int possibleLockId = 0; possibleLockId < maxLocalStorageNodes; ++possibleLockId) {
            for (int dirIndex = 0; dirIndex < environment.dataWithClusterFiles().length; ++dirIndex) {
                Path dir = environment.dataWithClusterFiles()[dirIndex].toPath().resolve(Paths.get(NODES_FOLDER, Integer.toString(possibleLockId)));
                if (!Files.exists(dir, new LinkOption[0])) {
                    Files.createDirectories(dir, new FileAttribute[0]);
                }
                this.logger.trace("obtaining node lock on {} ...", dir.toAbsolutePath());
                try {
                    NativeFSLockFactory lockFactory = new NativeFSLockFactory(dir.toFile());
                    Lock tmpLock = lockFactory.makeLock(NODE_LOCK_FILENAME);
                    boolean obtained = tmpLock.obtain();
                    if (obtained) {
                        locks[dirIndex] = tmpLock;
                        nodePaths[dirIndex] = dir;
                        localNodeId = possibleLockId;
                        continue;
                    }
                    this.logger.trace("failed to obtain node lock on {}", dir.toAbsolutePath());
                    for (int i = 0; i < locks.length; ++i) {
                        if (locks[i] != null) {
                            IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{locks[i]});
                        }
                        locks[i] = null;
                    }
                    break;
                }
                catch (IOException e) {
                    this.logger.trace("failed to obtain node lock on {}", e, dir.toAbsolutePath());
                    lastException = new IOException("failed to obtain lock on " + dir.toAbsolutePath(), e);
                    for (int i = 0; i < locks.length; ++i) {
                        IOUtils.closeWhileHandlingException((Closeable[])new Closeable[]{locks[i]});
                        locks[i] = null;
                    }
                    break;
                }
            }
            if (locks[0] != null) break;
        }
        if (locks[0] == null) {
            throw new ElasticsearchIllegalStateException("Failed to obtain node lock, is the following location writable?: " + Arrays.toString(environment.dataWithClusterFiles()), lastException);
        }
        this.localNodeId = localNodeId;
        this.locks = locks;
        this.nodePaths = nodePaths;
        if (this.logger.isDebugEnabled()) {
            this.logger.debug("using node location [{}], local_node_id [{}]", nodePaths, localNodeId);
        }
        if (this.logger.isTraceEnabled()) {
            StringBuilder sb = new StringBuilder("node data locations details:\n");
            for (Path file : nodePaths) {
                sb.append(" -> ").append(file.toAbsolutePath()).append(", free_space [").append(new ByteSizeValue(Files.getFileStore(file).getUnallocatedSpace())).append("], usable_space [").append(new ByteSizeValue(Files.getFileStore(file).getUsableSpace())).append("]\n");
            }
            this.logger.trace(sb.toString(), new Object[0]);
        }
        this.nodeIndicesPaths = new Path[nodePaths.length];
        for (int i = 0; i < nodePaths.length; ++i) {
            this.nodeIndicesPaths[i] = nodePaths[i].resolve(INDICES_FOLDER);
        }
    }

    public void deleteShardDirectorySafe(ShardId shardId, @IndexSettings Settings indexSettings) throws IOException {
        assert (indexSettings != ImmutableSettings.EMPTY);
        Path[] paths = this.shardPaths(shardId);
        this.logger.trace("deleting shard {} directory, paths: [{}]", shardId, paths);
        try (ShardLock lock = this.shardLock(shardId);){
            this.deleteShardDirectoryUnderLock(lock, indexSettings);
        }
    }

    public void deleteShardDirectoryUnderLock(ShardLock lock, @IndexSettings Settings indexSettings) throws IOException {
        assert (indexSettings != ImmutableSettings.EMPTY);
        ShardId shardId = lock.getShardId();
        assert (this.isShardLocked(shardId)) : "shard " + shardId + " is not locked";
        Path[] paths = this.shardPaths(shardId);
        XIOUtils.rm(paths);
        if (NodeEnvironment.hasCustomDataPath(indexSettings)) {
            Path customLocation = this.resolveCustomLocation(indexSettings, shardId);
            this.logger.trace("deleting custom shard {} directory [{}]", shardId, customLocation);
            XIOUtils.rm(customLocation);
        }
        this.logger.trace("deleted shard {} directory, paths: [{}]", shardId, paths);
        assert (!FileSystemUtils.exists(paths));
    }

    private boolean isShardLocked(ShardId id) {
        try {
            this.shardLock(id, 0L).close();
            return false;
        }
        catch (IOException ex) {
            return true;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteIndexDirectorySafe(Index index, long lockTimeoutMS, @IndexSettings Settings indexSettings) throws IOException {
        assert (indexSettings != ImmutableSettings.EMPTY);
        List<ShardLock> locks = this.lockAllForIndex(index, indexSettings, lockTimeoutMS);
        try {
            this.deleteIndexDirectoryUnderLock(index, indexSettings);
        }
        finally {
            IOUtils.closeWhileHandlingException(locks);
        }
    }

    public void deleteIndexDirectoryUnderLock(Index index, @IndexSettings Settings indexSettings) throws IOException {
        assert (indexSettings != ImmutableSettings.EMPTY);
        Path[] indexPaths = this.indexPaths(index);
        this.logger.trace("deleting index {} directory, paths({}): [{}]", index, indexPaths.length, indexPaths);
        XIOUtils.rm(indexPaths);
        if (NodeEnvironment.hasCustomDataPath(indexSettings)) {
            Path customLocation = this.resolveCustomLocation(indexSettings, index.name());
            this.logger.trace("deleting custom index {} directory [{}]", index, customLocation);
            XIOUtils.rm(customLocation);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public List<ShardLock> lockAllForIndex(Index index, @IndexSettings Settings settings, long lockTimeoutMS) throws IOException {
        Integer numShards = settings.getAsInt("index.number_of_shards", null);
        if (numShards == null || numShards <= 0) {
            throw new IllegalArgumentException("settings must contain a non-null > 0 number of shards");
        }
        this.logger.trace("locking all shards for index {} - [{}]", index, numShards);
        ArrayList<ShardLock> allLocks = new ArrayList<ShardLock>(numShards);
        boolean success = false;
        long startTime = System.currentTimeMillis();
        try {
            for (int i = 0; i < numShards; ++i) {
                long timeoutLeft = Math.max(0L, lockTimeoutMS - (System.currentTimeMillis() - startTime));
                allLocks.add(this.shardLock(new ShardId(index, i), timeoutLeft));
            }
            success = true;
            if (success) return allLocks;
        }
        catch (Throwable throwable) {
            if (success) throw throwable;
            this.logger.trace("unable to lock all shards for index {}", index);
            IOUtils.closeWhileHandlingException(allLocks);
            throw throwable;
        }
        this.logger.trace("unable to lock all shards for index {}", index);
        IOUtils.closeWhileHandlingException(allLocks);
        return allLocks;
    }

    public ShardLock shardLock(ShardId id) throws IOException {
        return this.shardLock(id, 0L);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ShardLock shardLock(final ShardId id, long lockTimeoutMS) throws IOException {
        boolean acquired;
        InternalShardLock shardLock;
        this.logger.trace("acquiring node shardlock on [{}], timeout [{}]", id, lockTimeoutMS);
        Map<ShardId, InternalShardLock> map = this.shardLocks;
        synchronized (map) {
            if (this.shardLocks.containsKey(id)) {
                shardLock = this.shardLocks.get(id);
                shardLock.incWaitCount();
                acquired = false;
            } else {
                shardLock = new InternalShardLock(id);
                this.shardLocks.put(id, shardLock);
                acquired = true;
            }
        }
        if (!acquired) {
            boolean success = false;
            try {
                shardLock.acquire(lockTimeoutMS);
                success = true;
            }
            finally {
                if (!success) {
                    shardLock.decWaitCount();
                }
            }
        }
        this.logger.trace("successfully acquired shardlock for [{}]", id);
        return new ShardLock(id){

            @Override
            protected void closeInternal() {
                shardLock.release();
                NodeEnvironment.this.logger.trace("released shard lock for [{}]", id);
            }
        };
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Set<ShardId> lockedShards() {
        Map<ShardId, InternalShardLock> map = this.shardLocks;
        synchronized (map) {
            ImmutableSet.Builder builder = ImmutableSet.builder();
            return ((ImmutableSet.Builder)builder.addAll(this.shardLocks.keySet())).build();
        }
    }

    public int localNodeId() {
        return this.localNodeId;
    }

    public boolean hasNodeFile() {
        return this.nodePaths != null && this.locks != null;
    }

    public Path[] nodeDataPaths() {
        assert (this.assertEnvIsLocked());
        if (this.nodePaths == null || this.locks == null) {
            throw new ElasticsearchIllegalStateException("node is not configured to store local location");
        }
        return this.nodePaths;
    }

    @Deprecated
    public File[] nodeDataLocations() {
        return NodeEnvironment.toFiles(this.nodeDataPaths());
    }

    @Deprecated
    public File[] indexLocations(Index index) {
        return NodeEnvironment.toFiles(this.indexPaths(index));
    }

    @Deprecated
    public File[] shardLocations(ShardId shardId) {
        return NodeEnvironment.toFiles(this.shardPaths(shardId));
    }

    @Deprecated
    public File[] shardDataLocations(ShardId shardId, @IndexSettings Settings settings) {
        return NodeEnvironment.toFiles(this.shardDataPaths(shardId, settings));
    }

    public Path[] indexPaths(Index index) {
        assert (this.assertEnvIsLocked());
        Path[] indexPaths = new Path[this.nodeIndicesPaths.length];
        for (int i = 0; i < this.nodeIndicesPaths.length; ++i) {
            indexPaths[i] = this.nodeIndicesPaths[i].resolve(index.name());
        }
        return indexPaths;
    }

    public Path[] shardDataPaths(ShardId shardId, @IndexSettings Settings indexSettings) {
        assert (indexSettings != ImmutableSettings.EMPTY);
        assert (this.assertEnvIsLocked());
        if (NodeEnvironment.hasCustomDataPath(indexSettings)) {
            return new Path[]{this.resolveCustomLocation(indexSettings, shardId)};
        }
        return this.shardPaths(shardId);
    }

    public Path[] shardPaths(ShardId shardId) {
        assert (this.assertEnvIsLocked());
        Path[] nodePaths = this.nodeDataPaths();
        Path[] shardLocations = new Path[nodePaths.length];
        for (int i = 0; i < nodePaths.length; ++i) {
            shardLocations[i] = nodePaths[i].resolve(Paths.get(INDICES_FOLDER, shardId.index().name(), Integer.toString(shardId.id())));
        }
        return shardLocations;
    }

    public Set<String> findAllIndices() throws Exception {
        if (this.nodePaths == null || this.locks == null) {
            throw new ElasticsearchIllegalStateException("node is not configured to store local location");
        }
        assert (this.assertEnvIsLocked());
        HashSet<String> indices = Sets.newHashSet();
        for (Path indicesLocation : this.nodeIndicesPaths) {
            if (!Files.exists(indicesLocation, new LinkOption[0]) || !Files.isDirectory(indicesLocation, new LinkOption[0])) continue;
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(indicesLocation);){
                for (Path index : stream) {
                    if (!Files.isDirectory(index, new LinkOption[0])) continue;
                    indices.add(index.getFileName().toString());
                }
            }
        }
        return indices;
    }

    public Set<ShardId> findAllShardIds(Index index) throws IOException {
        if (this.nodePaths == null || this.locks == null) {
            throw new ElasticsearchIllegalStateException("node is not configured to store local location");
        }
        assert (this.assertEnvIsLocked());
        return NodeEnvironment.findAllShardIds(index == null ? null : index.getName(), this.nodeIndicesPaths);
    }

    private static Set<ShardId> findAllShardIds(@Nullable String index, Path ... locations) throws IOException {
        HashSet<ShardId> shardIds = Sets.newHashSet();
        for (Path location : locations) {
            if (!Files.isDirectory(location, new LinkOption[0])) continue;
            try (DirectoryStream<Path> indexStream = Files.newDirectoryStream(location);){
                for (Path indexPath : indexStream) {
                    if (index != null && !index.equals(indexPath.getFileName().toString())) continue;
                    shardIds.addAll(NodeEnvironment.findAllShardsForIndex(indexPath));
                }
            }
        }
        return shardIds;
    }

    private static Set<ShardId> findAllShardsForIndex(Path indexPath) throws IOException {
        HashSet<ShardId> shardIds = new HashSet<ShardId>();
        if (Files.isDirectory(indexPath, new LinkOption[0])) {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(indexPath);){
                String currentIndex = indexPath.getFileName().toString();
                for (Path shardPath : stream) {
                    Integer shardId;
                    if (!Files.isDirectory(shardPath, new LinkOption[0]) || (shardId = Ints.tryParse(shardPath.getFileName().toString())) == null) continue;
                    ShardId id = new ShardId(currentIndex, (int)shardId);
                    shardIds.add(id);
                }
            }
        }
        return shardIds;
    }

    public Set<ShardId> findAllShardIds() throws IOException {
        return this.findAllShardIds(null);
    }

    @Override
    public void close() {
        if (this.closed.compareAndSet(false, true) && this.locks != null) {
            for (Lock lock : this.locks) {
                try {
                    this.logger.trace("releasing lock [{}]", lock);
                    lock.close();
                }
                catch (IOException e) {
                    this.logger.trace("failed to release lock [{}]", e, lock);
                }
            }
        }
    }

    private boolean assertEnvIsLocked() {
        if (!this.closed.get() && this.locks != null) {
            for (Lock lock : this.locks) {
                try {
                    if ($assertionsDisabled || lock.isLocked()) continue;
                    throw new AssertionError((Object)("Lock: " + lock + "is not locked"));
                }
                catch (IOException e) {
                    this.logger.warn("lock assertion failed", e, new Object[0]);
                    return false;
                }
            }
        }
        return true;
    }

    public void ensureAtomicMoveSupported() throws IOException {
        Path[] nodePaths;
        for (Path directory : nodePaths = this.nodeDataPaths()) {
            assert (Files.isDirectory(directory, new LinkOption[0])) : directory + " is not a directory";
            Path src = directory.resolve("__es__.tmp");
            Files.createFile(src, new FileAttribute[0]);
            Path target = directory.resolve("__es__.final");
            try {
                Files.move(src, target, StandardCopyOption.ATOMIC_MOVE);
            }
            catch (AtomicMoveNotSupportedException ex) {
                throw new ElasticsearchIllegalStateException("atomic_move is not supported by the filesystem on path [" + directory + "] atomic_move is required for elasticsearch to work correctly.", ex);
            }
            finally {
                Files.deleteIfExists(src);
                Files.deleteIfExists(target);
            }
        }
    }

    @Deprecated
    private static File[] toFiles(Path ... files) {
        File[] paths = new File[files.length];
        for (int i = 0; i < files.length; ++i) {
            paths[i] = files[i].toFile();
        }
        return paths;
    }

    public boolean isCustomPathsEnabled() {
        return this.customPathsEnabled;
    }

    public static boolean hasCustomDataPath(@IndexSettings Settings indexSettings) {
        return indexSettings.get("index.data_path") != null;
    }

    private Path resolveCustomLocation(@IndexSettings Settings indexSettings) {
        assert (indexSettings != ImmutableSettings.EMPTY);
        String customDataDir = indexSettings.get("index.data_path");
        if (customDataDir != null) {
            assert (this.customPathsEnabled);
            if (this.addNodeId) {
                return Paths.get(customDataDir, Integer.toString(this.localNodeId));
            }
            return Paths.get(customDataDir, new String[0]);
        }
        throw new ElasticsearchIllegalArgumentException("no custom index.data_path setting available");
    }

    private Path resolveCustomLocation(@IndexSettings Settings indexSettings, String indexName) {
        return this.resolveCustomLocation(indexSettings).resolve(indexName);
    }

    public Path resolveCustomLocation(@IndexSettings Settings indexSettings, ShardId shardId) {
        return this.resolveCustomLocation(indexSettings, shardId.index().name()).resolve(Integer.toString(shardId.id()));
    }

    private final class InternalShardLock {
        private final Semaphore mutex = new Semaphore(1);
        private int waitCount = 1;
        private ShardId shardId;

        InternalShardLock(ShardId id) {
            this.shardId = id;
            this.mutex.acquireUninterruptibly();
        }

        protected void release() {
            this.mutex.release();
            this.decWaitCount();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void incWaitCount() {
            Map map = NodeEnvironment.this.shardLocks;
            synchronized (map) {
                assert (this.waitCount > 0) : "waitCount is " + this.waitCount + " but should be > 0";
                ++this.waitCount;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void decWaitCount() {
            Map map = NodeEnvironment.this.shardLocks;
            synchronized (map) {
                assert (this.waitCount > 0) : "waitCount is " + this.waitCount + " but should be > 0";
                --this.waitCount;
                NodeEnvironment.this.logger.trace("shard lock wait count for [{}] is now [{}]", this.shardId, this.waitCount);
                if (this.waitCount == 0) {
                    NodeEnvironment.this.logger.trace("last shard lock wait decremented, removing lock for [{}]", this.shardId);
                    InternalShardLock remove = (InternalShardLock)NodeEnvironment.this.shardLocks.remove(this.shardId);
                    assert (remove != null) : "Removed lock was null";
                }
            }
        }

        void acquire(long timeoutInMillis) throws LockObtainFailedException {
            try {
                if (!this.mutex.tryAcquire(timeoutInMillis, TimeUnit.MILLISECONDS)) {
                    throw new LockObtainFailedException("Can't lock shard " + this.shardId + ", timed out after " + timeoutInMillis + "ms");
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new LockObtainFailedException("Can't lock shard " + this.shardId + ", interrupted", (Throwable)e);
            }
        }
    }
}

