/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.system.metadata;

import java.beans.PropertyEditor;
import java.beans.PropertyEditorManager;
import java.io.Serializable;

import org.jboss.logging.Logger;
import org.jboss.util.StringPropertyReplacer;
import org.w3c.dom.Element;

/**
 * ServiceElementValueMetaData.
 * 
 * This class is based on the old ServiceConfigurator.
 * If an MBean's <code>attribute</code> element contains an element which does not match any of the other
 * ServiceValueMetaData implementations, that element is passed in to the MBean attribute. 
 * 
 * An example configuration for an MBean with an Element property:
 * <pre>
 *   &lt;mbean name="jboss.test:type=test" code="org.jboss.test.system.controller.support.Simple"&gt;
 *      &lt;attribute name="Element"&gt;&lt;test/&gt;&lt;/property&gt;
 *     &lt;/attribute&gt;
 *   &lt;/mbean&gt;
 * </pre>
 * Will get hold of the Element for <code>dummy</code> and pass that in to the MBean's <code>Element</code>
 * property. 
 *
 * @author <a href="mailto:marc@jboss.org">Marc Fleury</a>
 * @author <a href="mailto:hiram@jboss.org">Hiram Chirino</a>
 * @author <a href="mailto:d_jencks@users.sourceforge.net">David Jencks</a>
 * @author <a href="mailto:jason@planet57.com">Jason Dillon</a>
 * @author <a href="mailto:dimitris@jboss.org">Dimitris Andreadis</a>
 * @author <a href="adrian@jboss.com">Adrian Brock</a>
 * @version $Revision: 89990 $
 */
public class ServiceElementValueMetaData extends AbstractMetaDataVisitorNode
   implements ServiceValueMetaData, Serializable
{
   private static final long serialVersionUID = 1;

   /** The log */
   private static final Logger log = Logger.getLogger(ServiceElementValueMetaData.class); 
   
   /** The element */
   private Element element;

   /**
    * Create a new ServiceElementValueMetaData.
    */
   public ServiceElementValueMetaData()
   {
   }
   
   /**
    * Create a new ServiceElementValueMetaData.
    * 
    * @param element the element
    */
   public ServiceElementValueMetaData(Element element)
   {
      setElement(element);
   }

   /**
    * Get the element.
    * 
    * @return the element.
    */
   public Element getElement()
   {
      return element;
   }

   /**
    * Set the element.
    * 
    * @param element the element.
    */
   public void setElement(Element element)
   {
      if (element == null)
         throw new IllegalArgumentException("Null element");
      this.element = element;
   }

   public Object getValue(ServiceValueContext valueContext) throws Exception
   {         
      // Replace any ${x} references in the element text
      if (valueContext.isReplace())
      {
         PropertyEditor editor = PropertyEditorManager.findEditor(Element.class);
         if (editor == null)
            log.warn("Cannot perform property replace on Element");
         else
         {
            editor.setValue(element);
            String text = editor.getAsText();
            text = StringPropertyReplacer.replaceProperties(text);
            editor.setAsText(text);
            return editor.getValue();
         }
      }

      return element;
   }
}
