package org.jboss.jsr299.tck.tests.implementation.enterprise.lifecycle;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import javax.context.Context;
import javax.context.CreationalContext;
import javax.context.RequestScoped;
import javax.inject.manager.Bean;

import org.hibernate.tck.annotations.SpecAssertion;
import org.hibernate.tck.annotations.SpecAssertions;
import org.jboss.jsr299.tck.AbstractJSR299Test;
import org.jboss.testharness.impl.packaging.Artifact;
import org.jboss.testharness.impl.packaging.IntegrationTest;
import org.jboss.testharness.impl.packaging.Packaging;
import org.jboss.testharness.impl.packaging.PackagingType;
import org.testng.annotations.Test;

/**
 * Sections
 * 
 * 6.5. Lifecycle of stateful session beans 
 * 6.6. Lifecycle of stateless session and singleton beans 
 * 6.11. Lifecycle of EJBs
 * 
 * Mostly overlapping with other tests...
 * 
 * @author Nicklas Karlsson
 * @author David Allen
 * 
 * Spec version: Public Release Draft 2
 * 
 */
@Artifact
@Packaging(PackagingType.EAR)
@IntegrationTest
public class EnterpriseBeanLifecycleTest extends AbstractJSR299Test
{

   /**
    * When the create() method of a Bean object that represents a stateful
    * session bean that is called, the container creates and returns a session
    * bean proxy, as defined in Section 3.3.9, "Session bean proxies".
    */
   @Test(groups = { "enterpriseBeans", "clientProxy", "lifecycle", "integration" })
   @SpecAssertions( { 
      @SpecAssertion(section = "3.3.8", id = "a"), 
      @SpecAssertion(section = "6", id = "a"),
      @SpecAssertion(section = "6.5", id = "a") })
   public void testCreateSFSB()
   {
      GrossStadt frankfurt = getCurrentManager().getInstanceByType(GrossStadt.class);
      Bean<KleinStadt> stadtBean = getCurrentManager().resolveByType(KleinStadt.class).iterator().next();
      assert stadtBean != null : "Expected a bean for stateful session bean Kassel";
      CreationalContext<KleinStadt> creationalContext = new MockCreationalContext<KleinStadt>();
      KleinStadt stadtInstance = stadtBean.create(creationalContext);
      assert stadtInstance != null : "Expected instance to be created by container";
      //assert frankfurt.isKleinStadtCreated() : "PostConstruct should be invoked when bean instance is created";
      frankfurt.resetCreatedFlags();
      
      // Create a second one to make sure create always does create a new session bean
      KleinStadt anotherStadtInstance = stadtBean.create(creationalContext);
      assert anotherStadtInstance != null : "Expected second instance of session bean";
      //assert frankfurt.isKleinStadtCreated();
      assert anotherStadtInstance != stadtInstance : "create() should not return same bean as before";
      
      // Verify that the instance returned is a proxy by checking for all local interfaces
      Set<Class<?>> interfaces = new HashSet<Class<?>>(Arrays.asList(stadtInstance.getClass().getInterfaces()));
      assert interfaces.contains(KleinStadt.class);
      assert interfaces.contains(SchoeneStadt.class);
      //frankfurt.dispose();
   }

   @Test(groups = { "enterpriseBeans", "clientProxy", "lifecycle", "integration", "ri-broken" })
   @SpecAssertions({
      @SpecAssertion(section = "6.5", id = "b"),
      @SpecAssertion(section = "6", id = "e")
   })
   public void testDestroyRemovesSFSB() throws Exception
   {
      GrossStadt frankfurt = getCurrentManager().getInstanceByType(GrossStadt.class);
      Bean<KleinStadt> stadtBean = getCurrentManager().resolveByType(KleinStadt.class).iterator().next();
      assert stadtBean != null : "Expected a bean for stateful session bean Kassel";
      Context requestContext = getCurrentManager().getContext(RequestScoped.class);
      CreationalContext<KleinStadt> creationalContext = new MockCreationalContext<KleinStadt>();
      KleinStadt kassel = requestContext.get(stadtBean, creationalContext);
      stadtBean.destroy(kassel);
      
      assert frankfurt.isKleinStadtDestroyed() : "Expected SFSB bean to be destroyed";
      kassel = requestContext.get(stadtBean);
      assert kassel == null : "SFSB bean should not exist after being destroyed";
      //frankfurt.dispose();
   }

   @Test(groups = { "enterpriseBeans", "lifecycle", "integration" })
   @SpecAssertion(section = "6.5", id = "c")
   public void testRemovedEjbIgnored()
   {
      KleinStadt stadtInstance = getCurrentManager().getInstanceByType(KleinStadt.class);
      assert stadtInstance != null : "Expected instance to be created by container";
      stadtInstance.zustandVergessen();
      
      // Now make sure that the container does not return this instance again
      KleinStadt newStadtInstance = getCurrentManager().getInstanceByType(KleinStadt.class);
      assert newStadtInstance != null : "Failed to get SFSB instance the second time";
      assert !newStadtInstance.equals(stadtInstance) : "The destroyed SFSB was not ignored";
   }

   @Test(groups = { "enterpriseBeans", "lifecycle", "integration" })
   @SpecAssertion(section = "6.6", id = "a")
   public void testCreateSLSB()
   {
      Bean<NeueStadt> stadtBean = getCurrentManager().resolveByType(NeueStadt.class).iterator().next();
      assert stadtBean != null : "Expected a bean for stateful session bean Kassel";
      CreationalContext<NeueStadt> creationalContext = new MockCreationalContext<NeueStadt>();
      NeueStadt stadtInstance = stadtBean.create(creationalContext);
      assert stadtInstance != null : "Expected instance to be created by container";
      
      // Verify that the instance returned is a proxy by checking for all local interfaces
      Set<Class<?>> interfaces = new HashSet<Class<?>>(Arrays.asList(stadtInstance.getClass().getInterfaces()));
      assert interfaces.contains(NeueStadt.class);
      assert interfaces.contains(GeschichtslosStadt.class);
   }

   @Test(groups = { "enterpriseBeans", "lifecycle", "integration" })
   @SpecAssertion(section = "6.11", id = "a")
   public void testFieldInjectionsOnNonContextualEjbs()
   {
      AlteStadt alteStadt = this.getCurrentConfiguration().getBeans().getEnterpriseBean(Mainz.class, AlteStadt.class);
      assert alteStadt != null : "Could not find the AlteStadt EJB";
      assert alteStadt.getPlaceOfInterest().equals(RoemerPassage.name);
   }

   @Test(groups = { "enterpriseBeans", "lifecycle", "integration" })
   @SpecAssertion(section = "6.11", id = "c")
   public void testInitializerMethodsCalledWithCurrentParameterValues()
   {
      AlteStadt alteStadt = this.getCurrentConfiguration().getBeans().getEnterpriseBean(Mainz.class, AlteStadt.class);
      assert alteStadt != null : "Could not find the AlteStadt EJB";
      assert alteStadt.getAnotherPlaceOfInterest() != null;
   }

   @Test(groups = { "enterpriseBeans", "lifecycle", "ri-broken" })
   @SpecAssertion(section = "6.11", id = "f")
   public void testDependentObjectsDestroyed()
   {
      UniStadt marburg = getCurrentManager().getInstanceByType(UniStadt.class);
      assert marburg != null : "Couldn't find the main SFSB";
      Bean<UniStadt> uniStadtBean = getCurrentManager().resolveByType(UniStadt.class).iterator().next();
      uniStadtBean.destroy(marburg);
      GrossStadt frankfurt = getCurrentManager().getInstanceByType(GrossStadt.class);
      assert frankfurt.isSchlossDestroyed();
   }

   @Test
   @SpecAssertion(section = "4.2", id = "bab")
   public void testDirectSubClassInheritsPostConstructOnSuperclass() throws Exception
   {
      OrderProcessor.postConstructCalled = false;
      assert getCurrentManager().resolveByType(DirectOrderProcessorLocal.class).size() == 1;
      new RunInDependentContext()
      {
         @Override
         protected void execute() throws Exception
         {
            getCurrentManager().getInstanceByType(DirectOrderProcessorLocal.class).order();
         }
      }.run();
      assert OrderProcessor.postConstructCalled;
   } 
   
   @Test
   @SpecAssertion(section = "4.2", id = "bad")
   public void testIndirectSubClassInheritsPostConstructOnSuperclass() throws Exception
   {
      OrderProcessor.postConstructCalled = false;
      assert getCurrentManager().resolveByType(OrderProcessorLocal.class).size() == 1;
      new RunInDependentContext()
      {
         @Override
         protected void execute() throws Exception
         {
            getCurrentManager().getInstanceByType(OrderProcessorLocal.class).order();
         }
      }.run();
      assert OrderProcessor.postConstructCalled;
   }   
   
   @Test(groups = "ri-broken")
   @SpecAssertion(section = "4.2", id = "bbb")
   public void testSubClassInheritsPreDestroyOnSuperclass() throws Exception
   {
      OrderProcessor.preDestroyCalled = false;
      assert getCurrentManager().resolveByType(DirectOrderProcessorLocal.class).size() == 1;
      new RunInDependentContext()
      {
         @Override
         protected void execute() throws Exception
         {  
            Bean<DirectOrderProcessorLocal> bean = getCurrentManager().resolveByType(DirectOrderProcessorLocal.class).iterator().next();
            DirectOrderProcessorLocal instance = getCurrentManager().getInstanceByType(DirectOrderProcessorLocal.class);
            bean.destroy(instance);
         }
      }.run();
      assert OrderProcessor.preDestroyCalled;
   }
   
   @Test(groups = "ri-broken")
   @SpecAssertion(section = "4.2", id = "bbd")
   public void testIndirectSubClassInheritsPreDestroyOnSuperclass() throws Exception
   {
      OrderProcessor.preDestroyCalled = false;
      assert getCurrentManager().resolveByType(IndirectOrderProcessor.class).size() == 1;
      new RunInDependentContext()
      {
         @Override
         protected void execute() throws Exception
         {  
            Bean<IndirectOrderProcessor> bean = getCurrentManager().resolveByType(IndirectOrderProcessor.class).iterator().next();
            IndirectOrderProcessor instance = getCurrentManager().getInstanceByType(IndirectOrderProcessor.class);
            bean.destroy(instance);
         }
      }.run();
      assert OrderProcessor.preDestroyCalled;
   }     
}
