/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */ 
package org.jboss.aop.classpool;

import java.lang.ref.WeakReference;
import java.lang.reflect.Field;
import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.security.PrivilegedExceptionAction;
import java.util.HashMap;
import java.util.HashSet;

import org.jboss.aop.Advisor;
import org.jboss.aop.AspectManager;
import org.jboss.aop.advice.SecurityActions;
import org.jboss.aop.instrument.Instrumentor;
import org.jboss.classpool.spi.ClassPoolRepositoryCallback;
import org.jboss.logging.Logger;

/**
 * Singleton classpool repository used by aop
 * 
 * @author <a href="kabir.khan@jboss.com">Kabir Khan</a>
 * @author <a href="flavia.rainone@jboss.com">Flavia Rainone</a>
 * @version $Revision: 102779 $
 */
public class ClassLoaderRepository implements ClassPoolRepositoryCallback
{
   private static final Logger logger = Logger.getLogger(ClassLoaderRepository.class);
   
   private final static ClassLoaderRepository instance = new ClassLoaderRepository();
   
   /** The classes per classppol */
   protected final HashMap<ClassLoader, HashSet<Class<?>>> ucl2classes = new HashMap<ClassLoader, HashSet<Class<?>>>();

   /** The top-level AspectManager this pool belongs to */
   AspectManager manager;
   
   public static ClassLoaderRepository getInstance()
   {
      return instance;
   }
   
   private ClassLoaderRepository() {}

   public void setAspectManager(AspectManager manager)
   {
      this.manager = manager;
   }

   
   public void registerClass(Class<?> clazz)
   {
      ClassLoader classLoader = SecurityActions.getClassLoader(clazz);
      HashSet<Class<?>> classes = ucl2classes.get(classLoader);
      if (classes == null)
      {
         classes = new HashSet<Class<?>>();
         ucl2classes.put(classLoader, classes);
      }
      classes.add(clazz);
   }

   public void classLoaderRegistered(ClassLoader classLoader)
   {
      // do nothing
   }

   public void classLoaderUnregistered(ClassLoader classLoader)
   {
      if (System.getSecurityManager() == null)
      {
         UnregisterClassLoaderAction.NON_PRIVILEGED.unregister(this, classLoader);
      }
      else
      {
         UnregisterClassLoaderAction.PRIVILEGED.unregister(this, classLoader);
      }
   }
   
   private void doUnregisterClassLoader(ClassLoader cl)
   {
      synchronized (this)
      {
         HashSet<Class<?>> classes = ucl2classes.remove(cl);
         if (classes != null)
         {
            for (Class<?> clazz : classes)
            {
               synchronized (manager.getAdvisors())
               {
                  WeakReference<Advisor> ref = manager.getAdvisors().get(clazz);
                  if (ref != null)
                  {
                     Advisor advisor = ref.get();
                     manager.getAdvisors().remove(clazz);
                     if (advisor != null)
                     {
                        advisor.cleanup();
                     }
                  }
                  Class<?> advisedClass = clazz;
                  try
                  {
                     //The static advisor field should be the only remaining hard reference to the advisor
                     Field f = advisedClass.getDeclaredField(Instrumentor.HELPER_FIELD_NAME);
                     f.setAccessible(true);
                     f.set(null, null);
                  }
                  catch(NoSuchFieldException e)
                  {
                     logger.warn("Error unsetting advisor for " + advisedClass.getName() + " " + e);
                  }
                  catch(IllegalAccessException e)
                  {
                     logger.warn("Error unsetting advisor for " + advisedClass.getName() + " " + e);
                  }
               }
            }
         }
      }
   }

   
   interface UnregisterClassLoaderAction
   {
      void unregister(ClassLoaderRepository repository, ClassLoader loader);
      
      UnregisterClassLoaderAction PRIVILEGED = new UnregisterClassLoaderAction()
      {
         public void unregister(final ClassLoaderRepository repository, final ClassLoader loader)
         {
            try
            {
               AccessController.doPrivileged(new PrivilegedExceptionAction<Object>()
               {
                  public Object run()
                  {
                     repository.doUnregisterClassLoader(loader);
                     return null;
                  }
               });
            }
            catch (PrivilegedActionException e)
            {
               Exception ex = e.getException();
               if (ex instanceof RuntimeException) 
               { 
                  throw (RuntimeException)ex;
               }
               throw new RuntimeException(ex);
            }
         }
      };

      UnregisterClassLoaderAction NON_PRIVILEGED = new UnregisterClassLoaderAction()
      {
         public void unregister(ClassLoaderRepository repository, ClassLoader loader)
         {
            repository.doUnregisterClassLoader(loader);
         }
      };
   }
}
