/*
 * Copyright 2002-2009 Andy Clark, Marc Guillemot
 * Copyright 2017-2023 Ronald Brill
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.htmlunit.cyberneko;

/**
 * Parser for the Pre-defined named HTML entities.
 * <a href="https://html.spec.whatwg.org/multipage/parsing.html#character-reference-state">12.2.5.72 Character reference state</a>
 * <p>
 * From the spec:<br>
 * Consume the maximum number of characters possible, with the consumed characters matching
 * one of the identifiers in the first column of the named character references table
 * (in a case-sensitive manner).
 * Append each character to the temporary buffer when it's consumed.
 *
 * @author Ronald Brill
 */
public class HTMLEntitiesParser {
    public static final int STATE_START = 0;
    private static final int STATE_ENDS_WITH_SEMICOLON = -2;

    private static final int STATE_HEXADECIMAL_CHAR = -102;
    private static final int STATE_DECIMAL_CHAR = -104;

    private static final int STATE_NUMERIC_CHAR_END_SEMICOLON_MISSING = -105;

    private int state_;
    private int consumedCount_;
    private String match_;
    private int code_;
    private int matchLength_;

    public String getMatch() {
        return match_;
    }

    public int getMatchLength() {
        return matchLength_;
    }

    public int getRewindCount() {
        return consumedCount_ - matchLength_;
    }

    public boolean endsWithSemicolon() {
        return STATE_ENDS_WITH_SEMICOLON == state_;
    }

    public HTMLEntitiesParser() {
        state_ = STATE_START;
    }

    public void setMatchFromCode() {
        // If the number is 0x00, then this is a null-character-reference parse error. Set the character reference code to 0xFFFD.
        // If the number is greater than 0x10FFFF, then this is a character-reference-outside-unicode-range parse error. Set the character reference code to 0xFFFD.
        if ((0x00 == code_) || (code_ > 0x10FFFF)) {
            match_ = "\uFFFD";
            matchLength_ = consumedCount_;
            return;
        }

        // If the number is a surrogate, then this is a surrogate-character-reference parse error. Set the character reference code to 0xFFFD
        if (Character.isSurrogate((char) code_)) {
            match_ = "\uFFFD";
            return;
        }

        // If the number is a noncharacter, then this is a noncharacter-character-reference parse error.

        // If the number is 0x0D, or a control that's not ASCII whitespace, then this is a control-character-reference parse error.

        // If the number is one of the numbers in the first column of the following table, then find the row with that number in the first column,
        // and set the character reference code to the number in the second column of that row.
        switch (code_) {
            case 0x80:
                match_ = "\u20AC";
                matchLength_ = consumedCount_;
                return;

            case 0x82:
                match_ = "\u201A";
                matchLength_ = consumedCount_;
                return;

            case 0x83:
                match_ = "\u0192";
                matchLength_ = consumedCount_;
                return;

            case 0x84:
                match_ = "\u201E";
                matchLength_ = consumedCount_;
                return;

            case 0x85:
                match_ = "\u2026";
                matchLength_ = consumedCount_;
                return;

            case 0x86:
                match_ = "\u2020";
                matchLength_ = consumedCount_;
                return;

            case 0x87:
                match_ = "\u2021";
                matchLength_ = consumedCount_;
                return;

            case 0x88:
                match_ = "\u02C6";
                matchLength_ = consumedCount_;
                return;

            case 0x89:
                match_ = "\u2030";
                matchLength_ = consumedCount_;
                return;

            case 0x8A:
                match_ = "\u0160";
                matchLength_ = consumedCount_;
                return;

            case 0x8B:
                match_ = "\u2039";
                matchLength_ = consumedCount_;
                return;

            case 0x8C:
                match_ = "\u0152";
                matchLength_ = consumedCount_;
                return;

            case 0x8E:
                match_ = "\u017D";
                matchLength_ = consumedCount_;
                return;

            case 0x91:
                match_ = "\u2018";
                matchLength_ = consumedCount_;
                return;

            case 0x92:
                match_ = "\u2019";
                matchLength_ = consumedCount_;
                return;

            case 0x93:
                match_ = "\u201C";
                matchLength_ = consumedCount_;
                return;

            case 0x94:
                match_ = "\u201D";
                matchLength_ = consumedCount_;
                return;

            case 0x95:
                match_ = "\u2022";
                matchLength_ = consumedCount_;
                return;

            case 0x96:
                match_ = "\u2013";
                matchLength_ = consumedCount_;
                return;

            case 0x97:
                match_ = "\u2014";
                matchLength_ = consumedCount_;
                return;

            case 0x98:
                match_ = "\u20DC";
                matchLength_ = consumedCount_;
                return;

            case 0x99:
                match_ = "\u2122";
                matchLength_ = consumedCount_;
                return;

            case 0x9A:
                match_ = "\u0161";
                matchLength_ = consumedCount_;
                return;

            case 0x9B:
                match_ = "\u203A";
                matchLength_ = consumedCount_;
                return;

            case 0x9C:
                match_ = "\u0153";
                matchLength_ = consumedCount_;
                return;

            case 0x9E:
                match_ = "\u017E";
                matchLength_ = consumedCount_;
                return;

            case 0x9F:
                match_ = "\u0178";
                matchLength_ = consumedCount_;
                return;

            default:
                break;
        }
        match_ = new String(Character.toChars(code_));
        matchLength_ = consumedCount_;
    }

    public boolean parseNumeric(final int current) {
        consumedCount_++;
        switch (state_) {
            case STATE_START:
                if ('X' == current || 'x' == current) {
                    state_ = STATE_HEXADECIMAL_CHAR;
                    return true;
                }
                if ('0' <= current && current <= '9') {
                    state_ = STATE_DECIMAL_CHAR;
                    code_ = (code_ * 10) + current - 0x30;
                    return true;
                }
                break;
            case STATE_HEXADECIMAL_CHAR:
                if ('0' <= current && current <= '9') {
                    code_ = (code_ * 16) + current - 0x30;
                    return true;
                }
                if ('A' <= current && current <= 'F') {
                    code_ = (code_ * 16) + current - 0x37;
                    return true;
                }
                if ('a' <= current && current <= 'f') {
                    code_ = (code_ * 16) + current - 0x57;
                    return true;
                }
                if (';' == current) {
                    setMatchFromCode();
                    return false;
                }

                state_ = STATE_NUMERIC_CHAR_END_SEMICOLON_MISSING;
                setMatchFromCode();
                matchLength_ = consumedCount_ - 1;
                break;
            case STATE_DECIMAL_CHAR:
                if ('0' <= current && current <= '9') {
                    code_ = (code_ * 10) + current - 0x30;
                    return true;
                }
                if (';' == current) {
                    setMatchFromCode();
                    return false;
                }

                state_ = STATE_NUMERIC_CHAR_END_SEMICOLON_MISSING;
                setMatchFromCode();
                matchLength_ = consumedCount_ - 1;
                break;
        }
        return false;
    }

    private boolean parse1(final int current) {
        consumedCount_++;
        switch (state_) {
            case 0:
                switch (current) {
                    case 'A':
                        state_ = 1;
                        return true;
                    case 'B':
                        state_ = 77;
                        return true;
                    case 'C':
                        state_ = 126;
                        return true;
                    case 'D':
                        state_ = 342;
                        return true;
                    case 'E':
                        state_ = 666;
                        return true;
                    case 'F':
                        state_ = 788;
                        return true;
                    case 'G':
                        state_ = 842;
                        return true;
                    case 'H':
                        state_ = 929;
                        return true;
                    case 'I':
                        state_ = 995;
                        return true;
                    case 'J':
                        state_ = 1096;
                        return true;
                    case 'K':
                        state_ = 1118;
                        return true;
                    case 'L':
                        state_ = 1143;
                        return true;
                    case 'M':
                        state_ = 1514;
                        return true;
                    case 'N':
                        state_ = 1553;
                        return true;
                    case 'O':
                        state_ = 2064;
                        return true;
                    case 'P':
                        state_ = 2173;
                        return true;
                    case 'Q':
                        state_ = 2258;
                        return true;
                    case 'R':
                        state_ = 2270;
                        return true;
                    case 'S':
                        state_ = 2525;
                        return true;
                    case 'T':
                        state_ = 2722;
                        return true;
                    case 'U':
                        state_ = 2817;
                        return true;
                    case 'V':
                        state_ = 2997;
                        return true;
                    case 'W':
                        state_ = 3067;
                        return true;
                    case 'X':
                        state_ = 3084;
                        return true;
                    case 'Y':
                        state_ = 3094;
                        return true;
                    case 'Z':
                        state_ = 3125;
                        return true;
                    case 'a':
                        state_ = 3166;
                        return true;
                    case 'b':
                        state_ = 3311;
                        return true;
                    case 'c':
                        state_ = 3623;
                        return true;
                    case 'd':
                        state_ = 3915;
                        return true;
                    case 'e':
                        state_ = 4156;
                        return true;
                    case 'f':
                        state_ = 4342;
                        return true;
                    case 'g':
                        state_ = 4442;
                        return true;
                    case 'h':
                        state_ = 4588;
                        return true;
                    case 'i':
                        state_ = 4704;
                        return true;
                    case 'j':
                        state_ = 4847;
                        return true;
                    case 'k':
                        state_ = 4873;
                        return true;
                    case 'l':
                        state_ = 4904;
                        return true;
                    case 'm':
                        state_ = 5397;
                        return true;
                    case 'n':
                        state_ = 5510;
                        return true;
                    case 'o':
                        state_ = 5945;
                        return true;
                    case 'p':
                        state_ = 6092;
                        return true;
                    case 'q':
                        state_ = 6272;
                        return true;
                    case 'r':
                        state_ = 6309;
                        return true;
                    case 's':
                        state_ = 6642;
                        return true;
                    case 't':
                        state_ = 7043;
                        return true;
                    case 'u':
                        state_ = 7238;
                        return true;
                    case 'v':
                        state_ = 7408;
                        return true;
                    case 'w':
                        state_ = 7562;
                        return true;
                    case 'x':
                        state_ = 7593;
                        return true;
                    case 'y':
                        state_ = 7672;
                        return true;
                    case 'z':
                        state_ = 7702;
                        return true;
                }
                break;
            case 1:
                switch (current) {
                    case 'E':
                        state_ = 2;
                        return true;
                    case 'M':
                        state_ = 6;
                        return true;
                    case 'a':
                        state_ = 8;
                        return true;
                    case 'b':
                        state_ = 13;
                        return true;
                    case 'c':
                        state_ = 18;
                        return true;
                    case 'f':
                        state_ = 23;
                        return true;
                    case 'g':
                        state_ = 25;
                        return true;
                    case 'l':
                        state_ = 30;
                        return true;
                    case 'm':
                        state_ = 34;
                        return true;
                    case 'n':
                        state_ = 38;
                        return true;
                    case 'o':
                        state_ = 40;
                        return true;
                    case 'p':
                        state_ = 46;
                        return true;
                    case 'r':
                        state_ = 58;
                        return true;
                    case 's':
                        state_ = 62;
                        return true;
                    case 't':
                        state_ = 69;
                        return true;
                    case 'u':
                        state_ = 74;
                        return true;
                }
                break;
            case 2:
                if ('l' == current) {
                    state_ = 3;
                    return true;
                }
                break;
            case 3:
                if ('i' == current) {
                    state_ = 4;
                    return true;
                }
                break;
            case 4:
                // AElig
                if ('g' == current) {
                    match_ = "\u00C6";
                    matchLength_ = consumedCount_;
                    state_ = 5;
                    return true;
                }
                break;
            case 5:
                // AElig;
                if (';' == current) {
                    match_ = "\u00C6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 6:
                // AMP
                if ('P' == current) {
                    match_ = "&";
                    matchLength_ = consumedCount_;
                    state_ = 7;
                    return true;
                }
                break;
            case 7:
                // AMP;
                if (';' == current) {
                    match_ = "&";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 8:
                if ('c' == current) {
                    state_ = 9;
                    return true;
                }
                break;
            case 9:
                if ('u' == current) {
                    state_ = 10;
                    return true;
                }
                break;
            case 10:
                if ('t' == current) {
                    state_ = 11;
                    return true;
                }
                break;
            case 11:
                // Aacute
                if ('e' == current) {
                    match_ = "\u00C1";
                    matchLength_ = consumedCount_;
                    state_ = 12;
                    return true;
                }
                break;
            case 12:
                // Aacute;
                if (';' == current) {
                    match_ = "\u00C1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 13:
                if ('r' == current) {
                    state_ = 14;
                    return true;
                }
                break;
            case 14:
                if ('e' == current) {
                    state_ = 15;
                    return true;
                }
                break;
            case 15:
                if ('v' == current) {
                    state_ = 16;
                    return true;
                }
                break;
            case 16:
                if ('e' == current) {
                    state_ = 17;
                    return true;
                }
                break;
            case 17:
                // Abreve;
                if (';' == current) {
                    match_ = "\u0102";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 18:
                if ('i' == current) {
                    state_ = 19;
                    return true;
                }
                else if ('y' == current) {
                    state_ = 22;
                    return true;
                }
                break;
            case 19:
                if ('r' == current) {
                    state_ = 20;
                    return true;
                }
                break;
            case 20:
                // Acirc
                if ('c' == current) {
                    match_ = "\u00C2";
                    matchLength_ = consumedCount_;
                    state_ = 21;
                    return true;
                }
                break;
            case 21:
                // Acirc;
                if (';' == current) {
                    match_ = "\u00C2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 22:
                // Acy;
                if (';' == current) {
                    match_ = "\u0410";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 23:
                if ('r' == current) {
                    state_ = 24;
                    return true;
                }
                break;
            case 24:
                // Afr;
                if (';' == current) {
                    match_ = "\uD835\uDD04";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 25:
                if ('r' == current) {
                    state_ = 26;
                    return true;
                }
                break;
            case 26:
                if ('a' == current) {
                    state_ = 27;
                    return true;
                }
                break;
            case 27:
                if ('v' == current) {
                    state_ = 28;
                    return true;
                }
                break;
            case 28:
                // Agrave
                if ('e' == current) {
                    match_ = "\u00C0";
                    matchLength_ = consumedCount_;
                    state_ = 29;
                    return true;
                }
                break;
            case 29:
                // Agrave;
                if (';' == current) {
                    match_ = "\u00C0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 30:
                if ('p' == current) {
                    state_ = 31;
                    return true;
                }
                break;
            case 31:
                if ('h' == current) {
                    state_ = 32;
                    return true;
                }
                break;
            case 32:
                if ('a' == current) {
                    state_ = 33;
                    return true;
                }
                break;
            case 33:
                // Alpha;
                if (';' == current) {
                    match_ = "\u0391";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 34:
                if ('a' == current) {
                    state_ = 35;
                    return true;
                }
                break;
            case 35:
                if ('c' == current) {
                    state_ = 36;
                    return true;
                }
                break;
            case 36:
                if ('r' == current) {
                    state_ = 37;
                    return true;
                }
                break;
            case 37:
                // Amacr;
                if (';' == current) {
                    match_ = "\u0100";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 38:
                if ('d' == current) {
                    state_ = 39;
                    return true;
                }
                break;
            case 39:
                // And;
                if (';' == current) {
                    match_ = "\u2A53";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 40:
                if ('g' == current) {
                    state_ = 41;
                    return true;
                }
                else if ('p' == current) {
                    state_ = 44;
                    return true;
                }
                break;
            case 41:
                if ('o' == current) {
                    state_ = 42;
                    return true;
                }
                break;
            case 42:
                if ('n' == current) {
                    state_ = 43;
                    return true;
                }
                break;
            case 43:
                // Aogon;
                if (';' == current) {
                    match_ = "\u0104";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 44:
                if ('f' == current) {
                    state_ = 45;
                    return true;
                }
                break;
            case 45:
                // Aopf;
                if (';' == current) {
                    match_ = "\uD835\uDD38";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 46:
                if ('p' == current) {
                    state_ = 47;
                    return true;
                }
                break;
            case 47:
                if ('l' == current) {
                    state_ = 48;
                    return true;
                }
                break;
            case 48:
                if ('y' == current) {
                    state_ = 49;
                    return true;
                }
                break;
            case 49:
                if ('F' == current) {
                    state_ = 50;
                    return true;
                }
                break;
            case 50:
                if ('u' == current) {
                    state_ = 51;
                    return true;
                }
                break;
            case 51:
                if ('n' == current) {
                    state_ = 52;
                    return true;
                }
                break;
            case 52:
                if ('c' == current) {
                    state_ = 53;
                    return true;
                }
                break;
            case 53:
                if ('t' == current) {
                    state_ = 54;
                    return true;
                }
                break;
            case 54:
                if ('i' == current) {
                    state_ = 55;
                    return true;
                }
                break;
            case 55:
                if ('o' == current) {
                    state_ = 56;
                    return true;
                }
                break;
            case 56:
                if ('n' == current) {
                    state_ = 57;
                    return true;
                }
                break;
            case 57:
                // ApplyFunction;
                if (';' == current) {
                    match_ = "\u2061";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 58:
                if ('i' == current) {
                    state_ = 59;
                    return true;
                }
                break;
            case 59:
                if ('n' == current) {
                    state_ = 60;
                    return true;
                }
                break;
            case 60:
                // Aring
                if ('g' == current) {
                    match_ = "\u00C5";
                    matchLength_ = consumedCount_;
                    state_ = 61;
                    return true;
                }
                break;
            case 61:
                // Aring;
                if (';' == current) {
                    match_ = "\u00C5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 62:
                if ('c' == current) {
                    state_ = 63;
                    return true;
                }
                else if ('s' == current) {
                    state_ = 65;
                    return true;
                }
                break;
            case 63:
                if ('r' == current) {
                    state_ = 64;
                    return true;
                }
                break;
            case 64:
                // Ascr;
                if (';' == current) {
                    match_ = "\uD835\uDC9C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 65:
                if ('i' == current) {
                    state_ = 66;
                    return true;
                }
                break;
            case 66:
                if ('g' == current) {
                    state_ = 67;
                    return true;
                }
                break;
            case 67:
                if ('n' == current) {
                    state_ = 68;
                    return true;
                }
                break;
            case 68:
                // Assign;
                if (';' == current) {
                    match_ = "\u2254";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 69:
                if ('i' == current) {
                    state_ = 70;
                    return true;
                }
                break;
            case 70:
                if ('l' == current) {
                    state_ = 71;
                    return true;
                }
                break;
            case 71:
                if ('d' == current) {
                    state_ = 72;
                    return true;
                }
                break;
            case 72:
                // Atilde
                if ('e' == current) {
                    match_ = "\u00C3";
                    matchLength_ = consumedCount_;
                    state_ = 73;
                    return true;
                }
                break;
            case 73:
                // Atilde;
                if (';' == current) {
                    match_ = "\u00C3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 74:
                if ('m' == current) {
                    state_ = 75;
                    return true;
                }
                break;
            case 75:
                // Auml
                if ('l' == current) {
                    match_ = "\u00C4";
                    matchLength_ = consumedCount_;
                    state_ = 76;
                    return true;
                }
                break;
            case 76:
                // Auml;
                if (';' == current) {
                    match_ = "\u00C4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 77:
                switch (current) {
                    case 'a':
                        state_ = 78;
                        return true;
                    case 'c':
                        state_ = 91;
                        return true;
                    case 'e':
                        state_ = 93;
                        return true;
                    case 'f':
                        state_ = 109;
                        return true;
                    case 'o':
                        state_ = 111;
                        return true;
                    case 'r':
                        state_ = 114;
                        return true;
                    case 's':
                        state_ = 118;
                        return true;
                    case 'u':
                        state_ = 121;
                        return true;
                }
                break;
            case 78:
                if ('c' == current) {
                    state_ = 79;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 86;
                    return true;
                }
                break;
            case 79:
                if ('k' == current) {
                    state_ = 80;
                    return true;
                }
                break;
            case 80:
                if ('s' == current) {
                    state_ = 81;
                    return true;
                }
                break;
            case 81:
                if ('l' == current) {
                    state_ = 82;
                    return true;
                }
                break;
            case 82:
                if ('a' == current) {
                    state_ = 83;
                    return true;
                }
                break;
            case 83:
                if ('s' == current) {
                    state_ = 84;
                    return true;
                }
                break;
            case 84:
                if ('h' == current) {
                    state_ = 85;
                    return true;
                }
                break;
            case 85:
                // Backslash;
                if (';' == current) {
                    match_ = "\u2216";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 86:
                if ('v' == current) {
                    state_ = 87;
                    return true;
                }
                else if ('w' == current) {
                    state_ = 88;
                    return true;
                }
                break;
            case 87:
                // Barv;
                if (';' == current) {
                    match_ = "\u2AE7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 88:
                if ('e' == current) {
                    state_ = 89;
                    return true;
                }
                break;
            case 89:
                if ('d' == current) {
                    state_ = 90;
                    return true;
                }
                break;
            case 90:
                // Barwed;
                if (';' == current) {
                    match_ = "\u2306";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 91:
                if ('y' == current) {
                    state_ = 92;
                    return true;
                }
                break;
            case 92:
                // Bcy;
                if (';' == current) {
                    match_ = "\u0411";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 93:
                switch (current) {
                    case 'c':
                        state_ = 94;
                        return true;
                    case 'r':
                        state_ = 99;
                        return true;
                    case 't':
                        state_ = 107;
                        return true;
                }
                break;
            case 94:
                if ('a' == current) {
                    state_ = 95;
                    return true;
                }
                break;
            case 95:
                if ('u' == current) {
                    state_ = 96;
                    return true;
                }
                break;
            case 96:
                if ('s' == current) {
                    state_ = 97;
                    return true;
                }
                break;
            case 97:
                if ('e' == current) {
                    state_ = 98;
                    return true;
                }
                break;
            case 98:
                // Because;
                if (';' == current) {
                    match_ = "\u2235";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 99:
                if ('n' == current) {
                    state_ = 100;
                    return true;
                }
                break;
            case 100:
                if ('o' == current) {
                    state_ = 101;
                    return true;
                }
                break;
            case 101:
                if ('u' == current) {
                    state_ = 102;
                    return true;
                }
                break;
            case 102:
                if ('l' == current) {
                    state_ = 103;
                    return true;
                }
                break;
            case 103:
                if ('l' == current) {
                    state_ = 104;
                    return true;
                }
                break;
            case 104:
                if ('i' == current) {
                    state_ = 105;
                    return true;
                }
                break;
            case 105:
                if ('s' == current) {
                    state_ = 106;
                    return true;
                }
                break;
            case 106:
                // Bernoullis;
                if (';' == current) {
                    match_ = "\u212C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 107:
                if ('a' == current) {
                    state_ = 108;
                    return true;
                }
                break;
            case 108:
                // Beta;
                if (';' == current) {
                    match_ = "\u0392";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 109:
                if ('r' == current) {
                    state_ = 110;
                    return true;
                }
                break;
            case 110:
                // Bfr;
                if (';' == current) {
                    match_ = "\uD835\uDD05";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 111:
                if ('p' == current) {
                    state_ = 112;
                    return true;
                }
                break;
            case 112:
                if ('f' == current) {
                    state_ = 113;
                    return true;
                }
                break;
            case 113:
                // Bopf;
                if (';' == current) {
                    match_ = "\uD835\uDD39";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 114:
                if ('e' == current) {
                    state_ = 115;
                    return true;
                }
                break;
            case 115:
                if ('v' == current) {
                    state_ = 116;
                    return true;
                }
                break;
            case 116:
                if ('e' == current) {
                    state_ = 117;
                    return true;
                }
                break;
            case 117:
                // Breve;
                if (';' == current) {
                    match_ = "\u02D8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 118:
                if ('c' == current) {
                    state_ = 119;
                    return true;
                }
                break;
            case 119:
                if ('r' == current) {
                    state_ = 120;
                    return true;
                }
                break;
            case 120:
                // Bscr;
                if (';' == current) {
                    match_ = "\u212C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 121:
                if ('m' == current) {
                    state_ = 122;
                    return true;
                }
                break;
            case 122:
                if ('p' == current) {
                    state_ = 123;
                    return true;
                }
                break;
            case 123:
                if ('e' == current) {
                    state_ = 124;
                    return true;
                }
                break;
            case 124:
                if ('q' == current) {
                    state_ = 125;
                    return true;
                }
                break;
            case 125:
                // Bumpeq;
                if (';' == current) {
                    match_ = "\u224E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 126:
                switch (current) {
                    case 'H':
                        state_ = 127;
                        return true;
                    case 'O':
                        state_ = 130;
                        return true;
                    case 'a':
                        state_ = 133;
                        return true;
                    case 'c':
                        state_ = 161;
                        return true;
                    case 'd':
                        state_ = 178;
                        return true;
                    case 'e':
                        state_ = 181;
                        return true;
                    case 'f':
                        state_ = 194;
                        return true;
                    case 'h':
                        state_ = 196;
                        return true;
                    case 'i':
                        state_ = 198;
                        return true;
                    case 'l':
                        state_ = 220;
                        return true;
                    case 'o':
                        state_ = 266;
                        return true;
                    case 'r':
                        state_ = 330;
                        return true;
                    case 's':
                        state_ = 334;
                        return true;
                    case 'u':
                        state_ = 337;
                        return true;
                }
                break;
            case 127:
                if ('c' == current) {
                    state_ = 128;
                    return true;
                }
                break;
            case 128:
                if ('y' == current) {
                    state_ = 129;
                    return true;
                }
                break;
            case 129:
                // CHcy;
                if (';' == current) {
                    match_ = "\u0427";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 130:
                if ('P' == current) {
                    state_ = 131;
                    return true;
                }
                break;
            case 131:
                // COPY
                if ('Y' == current) {
                    match_ = "\u00A9";
                    matchLength_ = consumedCount_;
                    state_ = 132;
                    return true;
                }
                break;
            case 132:
                // COPY;
                if (';' == current) {
                    match_ = "\u00A9";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 133:
                switch (current) {
                    case 'c':
                        state_ = 134;
                        return true;
                    case 'p':
                        state_ = 138;
                        return true;
                    case 'y':
                        state_ = 156;
                        return true;
                }
                break;
            case 134:
                if ('u' == current) {
                    state_ = 135;
                    return true;
                }
                break;
            case 135:
                if ('t' == current) {
                    state_ = 136;
                    return true;
                }
                break;
            case 136:
                if ('e' == current) {
                    state_ = 137;
                    return true;
                }
                break;
            case 137:
                // Cacute;
                if (';' == current) {
                    match_ = "\u0106";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 138:
                // Cap;
                if (';' == current) {
                    match_ = "\u22D2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('i' == current) {
                    state_ = 139;
                    return true;
                }
                break;
            case 139:
                if ('t' == current) {
                    state_ = 140;
                    return true;
                }
                break;
            case 140:
                if ('a' == current) {
                    state_ = 141;
                    return true;
                }
                break;
            case 141:
                if ('l' == current) {
                    state_ = 142;
                    return true;
                }
                break;
            case 142:
                if ('D' == current) {
                    state_ = 143;
                    return true;
                }
                break;
            case 143:
                if ('i' == current) {
                    state_ = 144;
                    return true;
                }
                break;
            case 144:
                if ('f' == current) {
                    state_ = 145;
                    return true;
                }
                break;
            case 145:
                if ('f' == current) {
                    state_ = 146;
                    return true;
                }
                break;
            case 146:
                if ('e' == current) {
                    state_ = 147;
                    return true;
                }
                break;
            case 147:
                if ('r' == current) {
                    state_ = 148;
                    return true;
                }
                break;
            case 148:
                if ('e' == current) {
                    state_ = 149;
                    return true;
                }
                break;
            case 149:
                if ('n' == current) {
                    state_ = 150;
                    return true;
                }
                break;
            case 150:
                if ('t' == current) {
                    state_ = 151;
                    return true;
                }
                break;
            case 151:
                if ('i' == current) {
                    state_ = 152;
                    return true;
                }
                break;
            case 152:
                if ('a' == current) {
                    state_ = 153;
                    return true;
                }
                break;
            case 153:
                if ('l' == current) {
                    state_ = 154;
                    return true;
                }
                break;
            case 154:
                if ('D' == current) {
                    state_ = 155;
                    return true;
                }
                break;
            case 155:
                // CapitalDifferentialD;
                if (';' == current) {
                    match_ = "\u2145";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 156:
                if ('l' == current) {
                    state_ = 157;
                    return true;
                }
                break;
            case 157:
                if ('e' == current) {
                    state_ = 158;
                    return true;
                }
                break;
            case 158:
                if ('y' == current) {
                    state_ = 159;
                    return true;
                }
                break;
            case 159:
                if ('s' == current) {
                    state_ = 160;
                    return true;
                }
                break;
            case 160:
                // Cayleys;
                if (';' == current) {
                    match_ = "\u212D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 161:
                switch (current) {
                    case 'a':
                        state_ = 162;
                        return true;
                    case 'e':
                        state_ = 166;
                        return true;
                    case 'i':
                        state_ = 170;
                        return true;
                    case 'o':
                        state_ = 173;
                        return true;
                }
                break;
            case 162:
                if ('r' == current) {
                    state_ = 163;
                    return true;
                }
                break;
            case 163:
                if ('o' == current) {
                    state_ = 164;
                    return true;
                }
                break;
            case 164:
                if ('n' == current) {
                    state_ = 165;
                    return true;
                }
                break;
            case 165:
                // Ccaron;
                if (';' == current) {
                    match_ = "\u010C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 166:
                if ('d' == current) {
                    state_ = 167;
                    return true;
                }
                break;
            case 167:
                if ('i' == current) {
                    state_ = 168;
                    return true;
                }
                break;
            case 168:
                // Ccedil
                if ('l' == current) {
                    match_ = "\u00C7";
                    matchLength_ = consumedCount_;
                    state_ = 169;
                    return true;
                }
                break;
            case 169:
                // Ccedil;
                if (';' == current) {
                    match_ = "\u00C7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 170:
                if ('r' == current) {
                    state_ = 171;
                    return true;
                }
                break;
            case 171:
                if ('c' == current) {
                    state_ = 172;
                    return true;
                }
                break;
            case 172:
                // Ccirc;
                if (';' == current) {
                    match_ = "\u0108";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 173:
                if ('n' == current) {
                    state_ = 174;
                    return true;
                }
                break;
            case 174:
                if ('i' == current) {
                    state_ = 175;
                    return true;
                }
                break;
            case 175:
                if ('n' == current) {
                    state_ = 176;
                    return true;
                }
                break;
            case 176:
                if ('t' == current) {
                    state_ = 177;
                    return true;
                }
                break;
            case 177:
                // Cconint;
                if (';' == current) {
                    match_ = "\u2230";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 178:
                if ('o' == current) {
                    state_ = 179;
                    return true;
                }
                break;
            case 179:
                if ('t' == current) {
                    state_ = 180;
                    return true;
                }
                break;
            case 180:
                // Cdot;
                if (';' == current) {
                    match_ = "\u010A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 181:
                if ('d' == current) {
                    state_ = 182;
                    return true;
                }
                else if ('n' == current) {
                    state_ = 187;
                    return true;
                }
                break;
            case 182:
                if ('i' == current) {
                    state_ = 183;
                    return true;
                }
                break;
            case 183:
                if ('l' == current) {
                    state_ = 184;
                    return true;
                }
                break;
            case 184:
                if ('l' == current) {
                    state_ = 185;
                    return true;
                }
                break;
            case 185:
                if ('a' == current) {
                    state_ = 186;
                    return true;
                }
                break;
            case 186:
                // Cedilla;
                if (';' == current) {
                    match_ = "\u00B8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 187:
                if ('t' == current) {
                    state_ = 188;
                    return true;
                }
                break;
            case 188:
                if ('e' == current) {
                    state_ = 189;
                    return true;
                }
                break;
            case 189:
                if ('r' == current) {
                    state_ = 190;
                    return true;
                }
                break;
            case 190:
                if ('D' == current) {
                    state_ = 191;
                    return true;
                }
                break;
            case 191:
                if ('o' == current) {
                    state_ = 192;
                    return true;
                }
                break;
            case 192:
                if ('t' == current) {
                    state_ = 193;
                    return true;
                }
                break;
            case 193:
                // CenterDot;
                if (';' == current) {
                    match_ = "\u00B7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 194:
                if ('r' == current) {
                    state_ = 195;
                    return true;
                }
                break;
            case 195:
                // Cfr;
                if (';' == current) {
                    match_ = "\u212D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 196:
                if ('i' == current) {
                    state_ = 197;
                    return true;
                }
                break;
            case 197:
                // Chi;
                if (';' == current) {
                    match_ = "\u03A7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 198:
                if ('r' == current) {
                    state_ = 199;
                    return true;
                }
                break;
            case 199:
                if ('c' == current) {
                    state_ = 200;
                    return true;
                }
                break;
            case 200:
                if ('l' == current) {
                    state_ = 201;
                    return true;
                }
                break;
            case 201:
                if ('e' == current) {
                    state_ = 202;
                    return true;
                }
                break;
            case 202:
                switch (current) {
                    case 'D':
                        state_ = 203;
                        return true;
                    case 'M':
                        state_ = 206;
                        return true;
                    case 'P':
                        state_ = 211;
                        return true;
                    case 'T':
                        state_ = 215;
                        return true;
                }
                break;
            case 203:
                if ('o' == current) {
                    state_ = 204;
                    return true;
                }
                break;
            case 204:
                if ('t' == current) {
                    state_ = 205;
                    return true;
                }
                break;
            case 205:
                // CircleDot;
                if (';' == current) {
                    match_ = "\u2299";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 206:
                if ('i' == current) {
                    state_ = 207;
                    return true;
                }
                break;
            case 207:
                if ('n' == current) {
                    state_ = 208;
                    return true;
                }
                break;
            case 208:
                if ('u' == current) {
                    state_ = 209;
                    return true;
                }
                break;
            case 209:
                if ('s' == current) {
                    state_ = 210;
                    return true;
                }
                break;
            case 210:
                // CircleMinus;
                if (';' == current) {
                    match_ = "\u2296";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 211:
                if ('l' == current) {
                    state_ = 212;
                    return true;
                }
                break;
            case 212:
                if ('u' == current) {
                    state_ = 213;
                    return true;
                }
                break;
            case 213:
                if ('s' == current) {
                    state_ = 214;
                    return true;
                }
                break;
            case 214:
                // CirclePlus;
                if (';' == current) {
                    match_ = "\u2295";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 215:
                if ('i' == current) {
                    state_ = 216;
                    return true;
                }
                break;
            case 216:
                if ('m' == current) {
                    state_ = 217;
                    return true;
                }
                break;
            case 217:
                if ('e' == current) {
                    state_ = 218;
                    return true;
                }
                break;
            case 218:
                if ('s' == current) {
                    state_ = 219;
                    return true;
                }
                break;
            case 219:
                // CircleTimes;
                if (';' == current) {
                    match_ = "\u2297";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 220:
                if ('o' == current) {
                    state_ = 221;
                    return true;
                }
                break;
            case 221:
                if ('c' == current) {
                    state_ = 222;
                    return true;
                }
                else if ('s' == current) {
                    state_ = 243;
                    return true;
                }
                break;
            case 222:
                if ('k' == current) {
                    state_ = 223;
                    return true;
                }
                break;
            case 223:
                if ('w' == current) {
                    state_ = 224;
                    return true;
                }
                break;
            case 224:
                if ('i' == current) {
                    state_ = 225;
                    return true;
                }
                break;
            case 225:
                if ('s' == current) {
                    state_ = 226;
                    return true;
                }
                break;
            case 226:
                if ('e' == current) {
                    state_ = 227;
                    return true;
                }
                break;
            case 227:
                if ('C' == current) {
                    state_ = 228;
                    return true;
                }
                break;
            case 228:
                if ('o' == current) {
                    state_ = 229;
                    return true;
                }
                break;
            case 229:
                if ('n' == current) {
                    state_ = 230;
                    return true;
                }
                break;
            case 230:
                if ('t' == current) {
                    state_ = 231;
                    return true;
                }
                break;
            case 231:
                if ('o' == current) {
                    state_ = 232;
                    return true;
                }
                break;
            case 232:
                if ('u' == current) {
                    state_ = 233;
                    return true;
                }
                break;
            case 233:
                if ('r' == current) {
                    state_ = 234;
                    return true;
                }
                break;
            case 234:
                if ('I' == current) {
                    state_ = 235;
                    return true;
                }
                break;
            case 235:
                if ('n' == current) {
                    state_ = 236;
                    return true;
                }
                break;
            case 236:
                if ('t' == current) {
                    state_ = 237;
                    return true;
                }
                break;
            case 237:
                if ('e' == current) {
                    state_ = 238;
                    return true;
                }
                break;
            case 238:
                if ('g' == current) {
                    state_ = 239;
                    return true;
                }
                break;
            case 239:
                if ('r' == current) {
                    state_ = 240;
                    return true;
                }
                break;
            case 240:
                if ('a' == current) {
                    state_ = 241;
                    return true;
                }
                break;
            case 241:
                if ('l' == current) {
                    state_ = 242;
                    return true;
                }
                break;
            case 242:
                // ClockwiseContourIntegral;
                if (';' == current) {
                    match_ = "\u2232";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 243:
                if ('e' == current) {
                    state_ = 244;
                    return true;
                }
                break;
            case 244:
                if ('C' == current) {
                    state_ = 245;
                    return true;
                }
                break;
            case 245:
                if ('u' == current) {
                    state_ = 246;
                    return true;
                }
                break;
            case 246:
                if ('r' == current) {
                    state_ = 247;
                    return true;
                }
                break;
            case 247:
                if ('l' == current) {
                    state_ = 248;
                    return true;
                }
                break;
            case 248:
                if ('y' == current) {
                    state_ = 249;
                    return true;
                }
                break;
            case 249:
                if ('D' == current) {
                    state_ = 250;
                    return true;
                }
                else if ('Q' == current) {
                    state_ = 261;
                    return true;
                }
                break;
            case 250:
                if ('o' == current) {
                    state_ = 251;
                    return true;
                }
                break;
            case 251:
                if ('u' == current) {
                    state_ = 252;
                    return true;
                }
                break;
            case 252:
                if ('b' == current) {
                    state_ = 253;
                    return true;
                }
                break;
            case 253:
                if ('l' == current) {
                    state_ = 254;
                    return true;
                }
                break;
            case 254:
                if ('e' == current) {
                    state_ = 255;
                    return true;
                }
                break;
            case 255:
                if ('Q' == current) {
                    state_ = 256;
                    return true;
                }
                break;
            case 256:
                if ('u' == current) {
                    state_ = 257;
                    return true;
                }
                break;
            case 257:
                if ('o' == current) {
                    state_ = 258;
                    return true;
                }
                break;
            case 258:
                if ('t' == current) {
                    state_ = 259;
                    return true;
                }
                break;
            case 259:
                if ('e' == current) {
                    state_ = 260;
                    return true;
                }
                break;
            case 260:
                // CloseCurlyDoubleQuote;
                if (';' == current) {
                    match_ = "\u201D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 261:
                if ('u' == current) {
                    state_ = 262;
                    return true;
                }
                break;
            case 262:
                if ('o' == current) {
                    state_ = 263;
                    return true;
                }
                break;
            case 263:
                if ('t' == current) {
                    state_ = 264;
                    return true;
                }
                break;
            case 264:
                if ('e' == current) {
                    state_ = 265;
                    return true;
                }
                break;
            case 265:
                // CloseCurlyQuote;
                if (';' == current) {
                    match_ = "\u2019";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 266:
                switch (current) {
                    case 'l':
                        state_ = 267;
                        return true;
                    case 'n':
                        state_ = 271;
                        return true;
                    case 'p':
                        state_ = 293;
                        return true;
                    case 'u':
                        state_ = 301;
                        return true;
                }
                break;
            case 267:
                if ('o' == current) {
                    state_ = 268;
                    return true;
                }
                break;
            case 268:
                if ('n' == current) {
                    state_ = 269;
                    return true;
                }
                break;
            case 269:
                // Colon;
                if (';' == current) {
                    match_ = "\u2237";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state_ = 270;
                    return true;
                }
                break;
            case 270:
                // Colone;
                if (';' == current) {
                    match_ = "\u2A74";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 271:
                switch (current) {
                    case 'g':
                        state_ = 272;
                        return true;
                    case 'i':
                        state_ = 278;
                        return true;
                    case 't':
                        state_ = 281;
                        return true;
                }
                break;
            case 272:
                if ('r' == current) {
                    state_ = 273;
                    return true;
                }
                break;
            case 273:
                if ('u' == current) {
                    state_ = 274;
                    return true;
                }
                break;
            case 274:
                if ('e' == current) {
                    state_ = 275;
                    return true;
                }
                break;
            case 275:
                if ('n' == current) {
                    state_ = 276;
                    return true;
                }
                break;
            case 276:
                if ('t' == current) {
                    state_ = 277;
                    return true;
                }
                break;
            case 277:
                // Congruent;
                if (';' == current) {
                    match_ = "\u2261";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 278:
                if ('n' == current) {
                    state_ = 279;
                    return true;
                }
                break;
            case 279:
                if ('t' == current) {
                    state_ = 280;
                    return true;
                }
                break;
            case 280:
                // Conint;
                if (';' == current) {
                    match_ = "\u222F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 281:
                if ('o' == current) {
                    state_ = 282;
                    return true;
                }
                break;
            case 282:
                if ('u' == current) {
                    state_ = 283;
                    return true;
                }
                break;
            case 283:
                if ('r' == current) {
                    state_ = 284;
                    return true;
                }
                break;
            case 284:
                if ('I' == current) {
                    state_ = 285;
                    return true;
                }
                break;
            case 285:
                if ('n' == current) {
                    state_ = 286;
                    return true;
                }
                break;
            case 286:
                if ('t' == current) {
                    state_ = 287;
                    return true;
                }
                break;
            case 287:
                if ('e' == current) {
                    state_ = 288;
                    return true;
                }
                break;
            case 288:
                if ('g' == current) {
                    state_ = 289;
                    return true;
                }
                break;
            case 289:
                if ('r' == current) {
                    state_ = 290;
                    return true;
                }
                break;
            case 290:
                if ('a' == current) {
                    state_ = 291;
                    return true;
                }
                break;
            case 291:
                if ('l' == current) {
                    state_ = 292;
                    return true;
                }
                break;
            case 292:
                // ContourIntegral;
                if (';' == current) {
                    match_ = "\u222E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 293:
                if ('f' == current) {
                    state_ = 294;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 295;
                    return true;
                }
                break;
            case 294:
                // Copf;
                if (';' == current) {
                    match_ = "\u2102";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 295:
                if ('o' == current) {
                    state_ = 296;
                    return true;
                }
                break;
            case 296:
                if ('d' == current) {
                    state_ = 297;
                    return true;
                }
                break;
            case 297:
                if ('u' == current) {
                    state_ = 298;
                    return true;
                }
                break;
            case 298:
                if ('c' == current) {
                    state_ = 299;
                    return true;
                }
                break;
            case 299:
                if ('t' == current) {
                    state_ = 300;
                    return true;
                }
                break;
            case 300:
                // Coproduct;
                if (';' == current) {
                    match_ = "\u2210";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 301:
                if ('n' == current) {
                    state_ = 302;
                    return true;
                }
                break;
            case 302:
                if ('t' == current) {
                    state_ = 303;
                    return true;
                }
                break;
            case 303:
                if ('e' == current) {
                    state_ = 304;
                    return true;
                }
                break;
            case 304:
                if ('r' == current) {
                    state_ = 305;
                    return true;
                }
                break;
            case 305:
                if ('C' == current) {
                    state_ = 306;
                    return true;
                }
                break;
            case 306:
                if ('l' == current) {
                    state_ = 307;
                    return true;
                }
                break;
            case 307:
                if ('o' == current) {
                    state_ = 308;
                    return true;
                }
                break;
            case 308:
                if ('c' == current) {
                    state_ = 309;
                    return true;
                }
                break;
            case 309:
                if ('k' == current) {
                    state_ = 310;
                    return true;
                }
                break;
            case 310:
                if ('w' == current) {
                    state_ = 311;
                    return true;
                }
                break;
            case 311:
                if ('i' == current) {
                    state_ = 312;
                    return true;
                }
                break;
            case 312:
                if ('s' == current) {
                    state_ = 313;
                    return true;
                }
                break;
            case 313:
                if ('e' == current) {
                    state_ = 314;
                    return true;
                }
                break;
            case 314:
                if ('C' == current) {
                    state_ = 315;
                    return true;
                }
                break;
            case 315:
                if ('o' == current) {
                    state_ = 316;
                    return true;
                }
                break;
            case 316:
                if ('n' == current) {
                    state_ = 317;
                    return true;
                }
                break;
            case 317:
                if ('t' == current) {
                    state_ = 318;
                    return true;
                }
                break;
            case 318:
                if ('o' == current) {
                    state_ = 319;
                    return true;
                }
                break;
            case 319:
                if ('u' == current) {
                    state_ = 320;
                    return true;
                }
                break;
            case 320:
                if ('r' == current) {
                    state_ = 321;
                    return true;
                }
                break;
            case 321:
                if ('I' == current) {
                    state_ = 322;
                    return true;
                }
                break;
            case 322:
                if ('n' == current) {
                    state_ = 323;
                    return true;
                }
                break;
            case 323:
                if ('t' == current) {
                    state_ = 324;
                    return true;
                }
                break;
            case 324:
                if ('e' == current) {
                    state_ = 325;
                    return true;
                }
                break;
            case 325:
                if ('g' == current) {
                    state_ = 326;
                    return true;
                }
                break;
            case 326:
                if ('r' == current) {
                    state_ = 327;
                    return true;
                }
                break;
            case 327:
                if ('a' == current) {
                    state_ = 328;
                    return true;
                }
                break;
            case 328:
                if ('l' == current) {
                    state_ = 329;
                    return true;
                }
                break;
            case 329:
                // CounterClockwiseContourIntegral;
                if (';' == current) {
                    match_ = "\u2233";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 330:
                if ('o' == current) {
                    state_ = 331;
                    return true;
                }
                break;
            case 331:
                if ('s' == current) {
                    state_ = 332;
                    return true;
                }
                break;
            case 332:
                if ('s' == current) {
                    state_ = 333;
                    return true;
                }
                break;
            case 333:
                // Cross;
                if (';' == current) {
                    match_ = "\u2A2F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 334:
                if ('c' == current) {
                    state_ = 335;
                    return true;
                }
                break;
            case 335:
                if ('r' == current) {
                    state_ = 336;
                    return true;
                }
                break;
            case 336:
                // Cscr;
                if (';' == current) {
                    match_ = "\uD835\uDC9E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 337:
                if ('p' == current) {
                    state_ = 338;
                    return true;
                }
                break;
            case 338:
                // Cup;
                if (';' == current) {
                    match_ = "\u22D3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('C' == current) {
                    state_ = 339;
                    return true;
                }
                break;
            case 339:
                if ('a' == current) {
                    state_ = 340;
                    return true;
                }
                break;
            case 340:
                if ('p' == current) {
                    state_ = 341;
                    return true;
                }
                break;
            case 341:
                // CupCap;
                if (';' == current) {
                    match_ = "\u224D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 342:
                switch (current) {
                    case 'D':
                        state_ = 343;
                        return true;
                    case 'J':
                        state_ = 350;
                        return true;
                    case 'S':
                        state_ = 353;
                        return true;
                    case 'Z':
                        state_ = 356;
                        return true;
                    case 'a':
                        state_ = 359;
                        return true;
                    case 'c':
                        state_ = 369;
                        return true;
                    case 'e':
                        state_ = 375;
                        return true;
                    case 'f':
                        state_ = 379;
                        return true;
                    case 'i':
                        state_ = 381;
                        return true;
                    case 'o':
                        state_ = 433;
                        return true;
                    case 's':
                        state_ = 659;
                        return true;
                }
                break;
            case 343:
                // DD;
                if (';' == current) {
                    match_ = "\u2145";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('o' == current) {
                    state_ = 344;
                    return true;
                }
                break;
            case 344:
                if ('t' == current) {
                    state_ = 345;
                    return true;
                }
                break;
            case 345:
                if ('r' == current) {
                    state_ = 346;
                    return true;
                }
                break;
            case 346:
                if ('a' == current) {
                    state_ = 347;
                    return true;
                }
                break;
            case 347:
                if ('h' == current) {
                    state_ = 348;
                    return true;
                }
                break;
            case 348:
                if ('d' == current) {
                    state_ = 349;
                    return true;
                }
                break;
            case 349:
                // DDotrahd;
                if (';' == current) {
                    match_ = "\u2911";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 350:
                if ('c' == current) {
                    state_ = 351;
                    return true;
                }
                break;
            case 351:
                if ('y' == current) {
                    state_ = 352;
                    return true;
                }
                break;
            case 352:
                // DJcy;
                if (';' == current) {
                    match_ = "\u0402";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 353:
                if ('c' == current) {
                    state_ = 354;
                    return true;
                }
                break;
            case 354:
                if ('y' == current) {
                    state_ = 355;
                    return true;
                }
                break;
            case 355:
                // DScy;
                if (';' == current) {
                    match_ = "\u0405";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 356:
                if ('c' == current) {
                    state_ = 357;
                    return true;
                }
                break;
            case 357:
                if ('y' == current) {
                    state_ = 358;
                    return true;
                }
                break;
            case 358:
                // DZcy;
                if (';' == current) {
                    match_ = "\u040F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 359:
                switch (current) {
                    case 'g':
                        state_ = 360;
                        return true;
                    case 'r':
                        state_ = 364;
                        return true;
                    case 's':
                        state_ = 366;
                        return true;
                }
                break;
            case 360:
                if ('g' == current) {
                    state_ = 361;
                    return true;
                }
                break;
            case 361:
                if ('e' == current) {
                    state_ = 362;
                    return true;
                }
                break;
            case 362:
                if ('r' == current) {
                    state_ = 363;
                    return true;
                }
                break;
            case 363:
                // Dagger;
                if (';' == current) {
                    match_ = "\u2021";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 364:
                if ('r' == current) {
                    state_ = 365;
                    return true;
                }
                break;
            case 365:
                // Darr;
                if (';' == current) {
                    match_ = "\u21A1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 366:
                if ('h' == current) {
                    state_ = 367;
                    return true;
                }
                break;
            case 367:
                if ('v' == current) {
                    state_ = 368;
                    return true;
                }
                break;
            case 368:
                // Dashv;
                if (';' == current) {
                    match_ = "\u2AE4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 369:
                if ('a' == current) {
                    state_ = 370;
                    return true;
                }
                else if ('y' == current) {
                    state_ = 374;
                    return true;
                }
                break;
            case 370:
                if ('r' == current) {
                    state_ = 371;
                    return true;
                }
                break;
            case 371:
                if ('o' == current) {
                    state_ = 372;
                    return true;
                }
                break;
            case 372:
                if ('n' == current) {
                    state_ = 373;
                    return true;
                }
                break;
            case 373:
                // Dcaron;
                if (';' == current) {
                    match_ = "\u010E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 374:
                // Dcy;
                if (';' == current) {
                    match_ = "\u0414";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 375:
                if ('l' == current) {
                    state_ = 376;
                    return true;
                }
                break;
            case 376:
                // Del;
                if (';' == current) {
                    match_ = "\u2207";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('t' == current) {
                    state_ = 377;
                    return true;
                }
                break;
            case 377:
                if ('a' == current) {
                    state_ = 378;
                    return true;
                }
                break;
            case 378:
                // Delta;
                if (';' == current) {
                    match_ = "\u0394";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 379:
                if ('r' == current) {
                    state_ = 380;
                    return true;
                }
                break;
            case 380:
                // Dfr;
                if (';' == current) {
                    match_ = "\uD835\uDD07";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 381:
                if ('a' == current) {
                    state_ = 382;
                    return true;
                }
                else if ('f' == current) {
                    state_ = 422;
                    return true;
                }
                break;
            case 382:
                if ('c' == current) {
                    state_ = 383;
                    return true;
                }
                else if ('m' == current) {
                    state_ = 418;
                    return true;
                }
                break;
            case 383:
                if ('r' == current) {
                    state_ = 384;
                    return true;
                }
                break;
            case 384:
                if ('i' == current) {
                    state_ = 385;
                    return true;
                }
                break;
            case 385:
                if ('t' == current) {
                    state_ = 386;
                    return true;
                }
                break;
            case 386:
                if ('i' == current) {
                    state_ = 387;
                    return true;
                }
                break;
            case 387:
                if ('c' == current) {
                    state_ = 388;
                    return true;
                }
                break;
            case 388:
                if ('a' == current) {
                    state_ = 389;
                    return true;
                }
                break;
            case 389:
                if ('l' == current) {
                    state_ = 390;
                    return true;
                }
                break;
            case 390:
                switch (current) {
                    case 'A':
                        state_ = 391;
                        return true;
                    case 'D':
                        state_ = 396;
                        return true;
                    case 'G':
                        state_ = 408;
                        return true;
                    case 'T':
                        state_ = 413;
                        return true;
                }
                break;
            case 391:
                if ('c' == current) {
                    state_ = 392;
                    return true;
                }
                break;
            case 392:
                if ('u' == current) {
                    state_ = 393;
                    return true;
                }
                break;
            case 393:
                if ('t' == current) {
                    state_ = 394;
                    return true;
                }
                break;
            case 394:
                if ('e' == current) {
                    state_ = 395;
                    return true;
                }
                break;
            case 395:
                // DiacriticalAcute;
                if (';' == current) {
                    match_ = "\u00B4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 396:
                if ('o' == current) {
                    state_ = 397;
                    return true;
                }
                break;
            case 397:
                if ('t' == current) {
                    state_ = 398;
                    return true;
                }
                else if ('u' == current) {
                    state_ = 399;
                    return true;
                }
                break;
            case 398:
                // DiacriticalDot;
                if (';' == current) {
                    match_ = "\u02D9";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 399:
                if ('b' == current) {
                    state_ = 400;
                    return true;
                }
                break;
            case 400:
                if ('l' == current) {
                    state_ = 401;
                    return true;
                }
                break;
            case 401:
                if ('e' == current) {
                    state_ = 402;
                    return true;
                }
                break;
            case 402:
                if ('A' == current) {
                    state_ = 403;
                    return true;
                }
                break;
            case 403:
                if ('c' == current) {
                    state_ = 404;
                    return true;
                }
                break;
            case 404:
                if ('u' == current) {
                    state_ = 405;
                    return true;
                }
                break;
            case 405:
                if ('t' == current) {
                    state_ = 406;
                    return true;
                }
                break;
            case 406:
                if ('e' == current) {
                    state_ = 407;
                    return true;
                }
                break;
            case 407:
                // DiacriticalDoubleAcute;
                if (';' == current) {
                    match_ = "\u02DD";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 408:
                if ('r' == current) {
                    state_ = 409;
                    return true;
                }
                break;
            case 409:
                if ('a' == current) {
                    state_ = 410;
                    return true;
                }
                break;
            case 410:
                if ('v' == current) {
                    state_ = 411;
                    return true;
                }
                break;
            case 411:
                if ('e' == current) {
                    state_ = 412;
                    return true;
                }
                break;
            case 412:
                // DiacriticalGrave;
                if (';' == current) {
                    match_ = "`";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 413:
                if ('i' == current) {
                    state_ = 414;
                    return true;
                }
                break;
            case 414:
                if ('l' == current) {
                    state_ = 415;
                    return true;
                }
                break;
            case 415:
                if ('d' == current) {
                    state_ = 416;
                    return true;
                }
                break;
            case 416:
                if ('e' == current) {
                    state_ = 417;
                    return true;
                }
                break;
            case 417:
                // DiacriticalTilde;
                if (';' == current) {
                    match_ = "\u02DC";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 418:
                if ('o' == current) {
                    state_ = 419;
                    return true;
                }
                break;
            case 419:
                if ('n' == current) {
                    state_ = 420;
                    return true;
                }
                break;
            case 420:
                if ('d' == current) {
                    state_ = 421;
                    return true;
                }
                break;
            case 421:
                // Diamond;
                if (';' == current) {
                    match_ = "\u22C4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 422:
                if ('f' == current) {
                    state_ = 423;
                    return true;
                }
                break;
            case 423:
                if ('e' == current) {
                    state_ = 424;
                    return true;
                }
                break;
            case 424:
                if ('r' == current) {
                    state_ = 425;
                    return true;
                }
                break;
            case 425:
                if ('e' == current) {
                    state_ = 426;
                    return true;
                }
                break;
            case 426:
                if ('n' == current) {
                    state_ = 427;
                    return true;
                }
                break;
            case 427:
                if ('t' == current) {
                    state_ = 428;
                    return true;
                }
                break;
            case 428:
                if ('i' == current) {
                    state_ = 429;
                    return true;
                }
                break;
            case 429:
                if ('a' == current) {
                    state_ = 430;
                    return true;
                }
                break;
            case 430:
                if ('l' == current) {
                    state_ = 431;
                    return true;
                }
                break;
            case 431:
                if ('D' == current) {
                    state_ = 432;
                    return true;
                }
                break;
            case 432:
                // DifferentialD;
                if (';' == current) {
                    match_ = "\u2146";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 433:
                switch (current) {
                    case 'p':
                        state_ = 434;
                        return true;
                    case 't':
                        state_ = 436;
                        return true;
                    case 'u':
                        state_ = 445;
                        return true;
                    case 'w':
                        state_ = 568;
                        return true;
                }
                break;
            case 434:
                if ('f' == current) {
                    state_ = 435;
                    return true;
                }
                break;
            case 435:
                // Dopf;
                if (';' == current) {
                    match_ = "\uD835\uDD3B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 436:
                switch (current) {
                    case ';': // Dot;
                        match_ = "\u00A8";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'D':
                        state_ = 437;
                        return true;
                    case 'E':
                        state_ = 440;
                        return true;
                }
                break;
            case 437:
                if ('o' == current) {
                    state_ = 438;
                    return true;
                }
                break;
            case 438:
                if ('t' == current) {
                    state_ = 439;
                    return true;
                }
                break;
            case 439:
                // DotDot;
                if (';' == current) {
                    match_ = "\u20DC";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 440:
                if ('q' == current) {
                    state_ = 441;
                    return true;
                }
                break;
            case 441:
                if ('u' == current) {
                    state_ = 442;
                    return true;
                }
                break;
            case 442:
                if ('a' == current) {
                    state_ = 443;
                    return true;
                }
                break;
            case 443:
                if ('l' == current) {
                    state_ = 444;
                    return true;
                }
                break;
            case 444:
                // DotEqual;
                if (';' == current) {
                    match_ = "\u2250";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 445:
                if ('b' == current) {
                    state_ = 446;
                    return true;
                }
                break;
            case 446:
                if ('l' == current) {
                    state_ = 447;
                    return true;
                }
                break;
            case 447:
                if ('e' == current) {
                    state_ = 448;
                    return true;
                }
                break;
            case 448:
                switch (current) {
                    case 'C':
                        state_ = 449;
                        return true;
                    case 'D':
                        state_ = 464;
                        return true;
                    case 'L':
                        state_ = 474;
                        return true;
                    case 'R':
                        state_ = 528;
                        return true;
                    case 'U':
                        state_ = 541;
                        return true;
                    case 'V':
                        state_ = 557;
                        return true;
                }
                break;
            case 449:
                if ('o' == current) {
                    state_ = 450;
                    return true;
                }
                break;
            case 450:
                if ('n' == current) {
                    state_ = 451;
                    return true;
                }
                break;
            case 451:
                if ('t' == current) {
                    state_ = 452;
                    return true;
                }
                break;
            case 452:
                if ('o' == current) {
                    state_ = 453;
                    return true;
                }
                break;
            case 453:
                if ('u' == current) {
                    state_ = 454;
                    return true;
                }
                break;
            case 454:
                if ('r' == current) {
                    state_ = 455;
                    return true;
                }
                break;
            case 455:
                if ('I' == current) {
                    state_ = 456;
                    return true;
                }
                break;
            case 456:
                if ('n' == current) {
                    state_ = 457;
                    return true;
                }
                break;
            case 457:
                if ('t' == current) {
                    state_ = 458;
                    return true;
                }
                break;
            case 458:
                if ('e' == current) {
                    state_ = 459;
                    return true;
                }
                break;
            case 459:
                if ('g' == current) {
                    state_ = 460;
                    return true;
                }
                break;
            case 460:
                if ('r' == current) {
                    state_ = 461;
                    return true;
                }
                break;
            case 461:
                if ('a' == current) {
                    state_ = 462;
                    return true;
                }
                break;
            case 462:
                if ('l' == current) {
                    state_ = 463;
                    return true;
                }
                break;
            case 463:
                // DoubleContourIntegral;
                if (';' == current) {
                    match_ = "\u222F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 464:
                if ('o' == current) {
                    state_ = 465;
                    return true;
                }
                break;
            case 465:
                if ('t' == current) {
                    state_ = 466;
                    return true;
                }
                else if ('w' == current) {
                    state_ = 467;
                    return true;
                }
                break;
            case 466:
                // DoubleDot;
                if (';' == current) {
                    match_ = "\u00A8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 467:
                if ('n' == current) {
                    state_ = 468;
                    return true;
                }
                break;
            case 468:
                if ('A' == current) {
                    state_ = 469;
                    return true;
                }
                break;
            case 469:
                if ('r' == current) {
                    state_ = 470;
                    return true;
                }
                break;
            case 470:
                if ('r' == current) {
                    state_ = 471;
                    return true;
                }
                break;
            case 471:
                if ('o' == current) {
                    state_ = 472;
                    return true;
                }
                break;
            case 472:
                if ('w' == current) {
                    state_ = 473;
                    return true;
                }
                break;
            case 473:
                // DoubleDownArrow;
                if (';' == current) {
                    match_ = "\u21D3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 474:
                if ('e' == current) {
                    state_ = 475;
                    return true;
                }
                else if ('o' == current) {
                    state_ = 496;
                    return true;
                }
                break;
            case 475:
                if ('f' == current) {
                    state_ = 476;
                    return true;
                }
                break;
            case 476:
                if ('t' == current) {
                    state_ = 477;
                    return true;
                }
                break;
            case 477:
                switch (current) {
                    case 'A':
                        state_ = 478;
                        return true;
                    case 'R':
                        state_ = 483;
                        return true;
                    case 'T':
                        state_ = 493;
                        return true;
                }
                break;
            case 478:
                if ('r' == current) {
                    state_ = 479;
                    return true;
                }
                break;
            case 479:
                if ('r' == current) {
                    state_ = 480;
                    return true;
                }
                break;
            case 480:
                if ('o' == current) {
                    state_ = 481;
                    return true;
                }
                break;
            case 481:
                if ('w' == current) {
                    state_ = 482;
                    return true;
                }
                break;
            case 482:
                // DoubleLeftArrow;
                if (';' == current) {
                    match_ = "\u21D0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 483:
                if ('i' == current) {
                    state_ = 484;
                    return true;
                }
                break;
            case 484:
                if ('g' == current) {
                    state_ = 485;
                    return true;
                }
                break;
            case 485:
                if ('h' == current) {
                    state_ = 486;
                    return true;
                }
                break;
            case 486:
                if ('t' == current) {
                    state_ = 487;
                    return true;
                }
                break;
            case 487:
                if ('A' == current) {
                    state_ = 488;
                    return true;
                }
                break;
            case 488:
                if ('r' == current) {
                    state_ = 489;
                    return true;
                }
                break;
            case 489:
                if ('r' == current) {
                    state_ = 490;
                    return true;
                }
                break;
            case 490:
                if ('o' == current) {
                    state_ = 491;
                    return true;
                }
                break;
            case 491:
                if ('w' == current) {
                    state_ = 492;
                    return true;
                }
                break;
            case 492:
                // DoubleLeftRightArrow;
                if (';' == current) {
                    match_ = "\u21D4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 493:
                if ('e' == current) {
                    state_ = 494;
                    return true;
                }
                break;
            case 494:
                if ('e' == current) {
                    state_ = 495;
                    return true;
                }
                break;
            case 495:
                // DoubleLeftTee;
                if (';' == current) {
                    match_ = "\u2AE4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 496:
                if ('n' == current) {
                    state_ = 497;
                    return true;
                }
                break;
            case 497:
                if ('g' == current) {
                    state_ = 498;
                    return true;
                }
                break;
            case 498:
                if ('L' == current) {
                    state_ = 499;
                    return true;
                }
                else if ('R' == current) {
                    state_ = 518;
                    return true;
                }
                break;
            case 499:
                if ('e' == current) {
                    state_ = 500;
                    return true;
                }
                break;
            case 500:
                if ('f' == current) {
                    state_ = 501;
                    return true;
                }
                break;
            case 501:
                if ('t' == current) {
                    state_ = 502;
                    return true;
                }
                break;
            case 502:
                if ('A' == current) {
                    state_ = 503;
                    return true;
                }
                else if ('R' == current) {
                    state_ = 508;
                    return true;
                }
                break;
            case 503:
                if ('r' == current) {
                    state_ = 504;
                    return true;
                }
                break;
            case 504:
                if ('r' == current) {
                    state_ = 505;
                    return true;
                }
                break;
            case 505:
                if ('o' == current) {
                    state_ = 506;
                    return true;
                }
                break;
            case 506:
                if ('w' == current) {
                    state_ = 507;
                    return true;
                }
                break;
            case 507:
                // DoubleLongLeftArrow;
                if (';' == current) {
                    match_ = "\u27F8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 508:
                if ('i' == current) {
                    state_ = 509;
                    return true;
                }
                break;
            case 509:
                if ('g' == current) {
                    state_ = 510;
                    return true;
                }
                break;
            case 510:
                if ('h' == current) {
                    state_ = 511;
                    return true;
                }
                break;
            case 511:
                if ('t' == current) {
                    state_ = 512;
                    return true;
                }
                break;
            case 512:
                if ('A' == current) {
                    state_ = 513;
                    return true;
                }
                break;
            case 513:
                if ('r' == current) {
                    state_ = 514;
                    return true;
                }
                break;
            case 514:
                if ('r' == current) {
                    state_ = 515;
                    return true;
                }
                break;
            case 515:
                if ('o' == current) {
                    state_ = 516;
                    return true;
                }
                break;
            case 516:
                if ('w' == current) {
                    state_ = 517;
                    return true;
                }
                break;
            case 517:
                // DoubleLongLeftRightArrow;
                if (';' == current) {
                    match_ = "\u27FA";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 518:
                if ('i' == current) {
                    state_ = 519;
                    return true;
                }
                break;
            case 519:
                if ('g' == current) {
                    state_ = 520;
                    return true;
                }
                break;
            case 520:
                if ('h' == current) {
                    state_ = 521;
                    return true;
                }
                break;
            case 521:
                if ('t' == current) {
                    state_ = 522;
                    return true;
                }
                break;
            case 522:
                if ('A' == current) {
                    state_ = 523;
                    return true;
                }
                break;
            case 523:
                if ('r' == current) {
                    state_ = 524;
                    return true;
                }
                break;
            case 524:
                if ('r' == current) {
                    state_ = 525;
                    return true;
                }
                break;
            case 525:
                if ('o' == current) {
                    state_ = 526;
                    return true;
                }
                break;
            case 526:
                if ('w' == current) {
                    state_ = 527;
                    return true;
                }
                break;
            case 527:
                // DoubleLongRightArrow;
                if (';' == current) {
                    match_ = "\u27F9";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 528:
                if ('i' == current) {
                    state_ = 529;
                    return true;
                }
                break;
            case 529:
                if ('g' == current) {
                    state_ = 530;
                    return true;
                }
                break;
            case 530:
                if ('h' == current) {
                    state_ = 531;
                    return true;
                }
                break;
            case 531:
                if ('t' == current) {
                    state_ = 532;
                    return true;
                }
                break;
            case 532:
                if ('A' == current) {
                    state_ = 533;
                    return true;
                }
                else if ('T' == current) {
                    state_ = 538;
                    return true;
                }
                break;
            case 533:
                if ('r' == current) {
                    state_ = 534;
                    return true;
                }
                break;
            case 534:
                if ('r' == current) {
                    state_ = 535;
                    return true;
                }
                break;
            case 535:
                if ('o' == current) {
                    state_ = 536;
                    return true;
                }
                break;
            case 536:
                if ('w' == current) {
                    state_ = 537;
                    return true;
                }
                break;
            case 537:
                // DoubleRightArrow;
                if (';' == current) {
                    match_ = "\u21D2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 538:
                if ('e' == current) {
                    state_ = 539;
                    return true;
                }
                break;
            case 539:
                if ('e' == current) {
                    state_ = 540;
                    return true;
                }
                break;
            case 540:
                // DoubleRightTee;
                if (';' == current) {
                    match_ = "\u22A8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 541:
                if ('p' == current) {
                    state_ = 542;
                    return true;
                }
                break;
            case 542:
                if ('A' == current) {
                    state_ = 543;
                    return true;
                }
                else if ('D' == current) {
                    state_ = 548;
                    return true;
                }
                break;
            case 543:
                if ('r' == current) {
                    state_ = 544;
                    return true;
                }
                break;
            case 544:
                if ('r' == current) {
                    state_ = 545;
                    return true;
                }
                break;
            case 545:
                if ('o' == current) {
                    state_ = 546;
                    return true;
                }
                break;
            case 546:
                if ('w' == current) {
                    state_ = 547;
                    return true;
                }
                break;
            case 547:
                // DoubleUpArrow;
                if (';' == current) {
                    match_ = "\u21D1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 548:
                if ('o' == current) {
                    state_ = 549;
                    return true;
                }
                break;
            case 549:
                if ('w' == current) {
                    state_ = 550;
                    return true;
                }
                break;
            case 550:
                if ('n' == current) {
                    state_ = 551;
                    return true;
                }
                break;
            case 551:
                if ('A' == current) {
                    state_ = 552;
                    return true;
                }
                break;
            case 552:
                if ('r' == current) {
                    state_ = 553;
                    return true;
                }
                break;
            case 553:
                if ('r' == current) {
                    state_ = 554;
                    return true;
                }
                break;
            case 554:
                if ('o' == current) {
                    state_ = 555;
                    return true;
                }
                break;
            case 555:
                if ('w' == current) {
                    state_ = 556;
                    return true;
                }
                break;
            case 556:
                // DoubleUpDownArrow;
                if (';' == current) {
                    match_ = "\u21D5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 557:
                if ('e' == current) {
                    state_ = 558;
                    return true;
                }
                break;
            case 558:
                if ('r' == current) {
                    state_ = 559;
                    return true;
                }
                break;
            case 559:
                if ('t' == current) {
                    state_ = 560;
                    return true;
                }
                break;
            case 560:
                if ('i' == current) {
                    state_ = 561;
                    return true;
                }
                break;
            case 561:
                if ('c' == current) {
                    state_ = 562;
                    return true;
                }
                break;
            case 562:
                if ('a' == current) {
                    state_ = 563;
                    return true;
                }
                break;
            case 563:
                if ('l' == current) {
                    state_ = 564;
                    return true;
                }
                break;
            case 564:
                if ('B' == current) {
                    state_ = 565;
                    return true;
                }
                break;
            case 565:
                if ('a' == current) {
                    state_ = 566;
                    return true;
                }
                break;
            case 566:
                if ('r' == current) {
                    state_ = 567;
                    return true;
                }
                break;
            case 567:
                // DoubleVerticalBar;
                if (';' == current) {
                    match_ = "\u2225";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 568:
                if ('n' == current) {
                    state_ = 569;
                    return true;
                }
                break;
            case 569:
                switch (current) {
                    case 'A':
                        state_ = 570;
                        return true;
                    case 'B':
                        state_ = 585;
                        return true;
                    case 'L':
                        state_ = 590;
                        return true;
                    case 'R':
                        state_ = 623;
                        return true;
                    case 'T':
                        state_ = 646;
                        return true;
                    case 'a':
                        state_ = 654;
                        return true;
                }
                break;
            case 570:
                if ('r' == current) {
                    state_ = 571;
                    return true;
                }
                break;
            case 571:
                if ('r' == current) {
                    state_ = 572;
                    return true;
                }
                break;
            case 572:
                if ('o' == current) {
                    state_ = 573;
                    return true;
                }
                break;
            case 573:
                if ('w' == current) {
                    state_ = 574;
                    return true;
                }
                break;
            case 574:
                switch (current) {
                    case ';': // DownArrow;
                        match_ = "\u2193";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B':
                        state_ = 575;
                        return true;
                    case 'U':
                        state_ = 578;
                        return true;
                }
                break;
            case 575:
                if ('a' == current) {
                    state_ = 576;
                    return true;
                }
                break;
            case 576:
                if ('r' == current) {
                    state_ = 577;
                    return true;
                }
                break;
            case 577:
                // DownArrowBar;
                if (';' == current) {
                    match_ = "\u2913";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 578:
                if ('p' == current) {
                    state_ = 579;
                    return true;
                }
                break;
            case 579:
                if ('A' == current) {
                    state_ = 580;
                    return true;
                }
                break;
            case 580:
                if ('r' == current) {
                    state_ = 581;
                    return true;
                }
                break;
            case 581:
                if ('r' == current) {
                    state_ = 582;
                    return true;
                }
                break;
            case 582:
                if ('o' == current) {
                    state_ = 583;
                    return true;
                }
                break;
            case 583:
                if ('w' == current) {
                    state_ = 584;
                    return true;
                }
                break;
            case 584:
                // DownArrowUpArrow;
                if (';' == current) {
                    match_ = "\u21F5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 585:
                if ('r' == current) {
                    state_ = 586;
                    return true;
                }
                break;
            case 586:
                if ('e' == current) {
                    state_ = 587;
                    return true;
                }
                break;
            case 587:
                if ('v' == current) {
                    state_ = 588;
                    return true;
                }
                break;
            case 588:
                if ('e' == current) {
                    state_ = 589;
                    return true;
                }
                break;
            case 589:
                // DownBreve;
                if (';' == current) {
                    match_ = "\u0311";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 590:
                if ('e' == current) {
                    state_ = 591;
                    return true;
                }
                break;
            case 591:
                if ('f' == current) {
                    state_ = 592;
                    return true;
                }
                break;
            case 592:
                if ('t' == current) {
                    state_ = 593;
                    return true;
                }
                break;
            case 593:
                switch (current) {
                    case 'R':
                        state_ = 594;
                        return true;
                    case 'T':
                        state_ = 605;
                        return true;
                    case 'V':
                        state_ = 614;
                        return true;
                }
                break;
            case 594:
                if ('i' == current) {
                    state_ = 595;
                    return true;
                }
                break;
            case 595:
                if ('g' == current) {
                    state_ = 596;
                    return true;
                }
                break;
            case 596:
                if ('h' == current) {
                    state_ = 597;
                    return true;
                }
                break;
            case 597:
                if ('t' == current) {
                    state_ = 598;
                    return true;
                }
                break;
            case 598:
                if ('V' == current) {
                    state_ = 599;
                    return true;
                }
                break;
            case 599:
                if ('e' == current) {
                    state_ = 600;
                    return true;
                }
                break;
            case 600:
                if ('c' == current) {
                    state_ = 601;
                    return true;
                }
                break;
            case 601:
                if ('t' == current) {
                    state_ = 602;
                    return true;
                }
                break;
            case 602:
                if ('o' == current) {
                    state_ = 603;
                    return true;
                }
                break;
            case 603:
                if ('r' == current) {
                    state_ = 604;
                    return true;
                }
                break;
            case 604:
                // DownLeftRightVector;
                if (';' == current) {
                    match_ = "\u2950";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 605:
                if ('e' == current) {
                    state_ = 606;
                    return true;
                }
                break;
            case 606:
                if ('e' == current) {
                    state_ = 607;
                    return true;
                }
                break;
            case 607:
                if ('V' == current) {
                    state_ = 608;
                    return true;
                }
                break;
            case 608:
                if ('e' == current) {
                    state_ = 609;
                    return true;
                }
                break;
            case 609:
                if ('c' == current) {
                    state_ = 610;
                    return true;
                }
                break;
            case 610:
                if ('t' == current) {
                    state_ = 611;
                    return true;
                }
                break;
            case 611:
                if ('o' == current) {
                    state_ = 612;
                    return true;
                }
                break;
            case 612:
                if ('r' == current) {
                    state_ = 613;
                    return true;
                }
                break;
            case 613:
                // DownLeftTeeVector;
                if (';' == current) {
                    match_ = "\u295E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 614:
                if ('e' == current) {
                    state_ = 615;
                    return true;
                }
                break;
            case 615:
                if ('c' == current) {
                    state_ = 616;
                    return true;
                }
                break;
            case 616:
                if ('t' == current) {
                    state_ = 617;
                    return true;
                }
                break;
            case 617:
                if ('o' == current) {
                    state_ = 618;
                    return true;
                }
                break;
            case 618:
                if ('r' == current) {
                    state_ = 619;
                    return true;
                }
                break;
            case 619:
                // DownLeftVector;
                if (';' == current) {
                    match_ = "\u21BD";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('B' == current) {
                    state_ = 620;
                    return true;
                }
                break;
            case 620:
                if ('a' == current) {
                    state_ = 621;
                    return true;
                }
                break;
            case 621:
                if ('r' == current) {
                    state_ = 622;
                    return true;
                }
                break;
            case 622:
                // DownLeftVectorBar;
                if (';' == current) {
                    match_ = "\u2956";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 623:
                if ('i' == current) {
                    state_ = 624;
                    return true;
                }
                break;
            case 624:
                if ('g' == current) {
                    state_ = 625;
                    return true;
                }
                break;
            case 625:
                if ('h' == current) {
                    state_ = 626;
                    return true;
                }
                break;
            case 626:
                if ('t' == current) {
                    state_ = 627;
                    return true;
                }
                break;
            case 627:
                if ('T' == current) {
                    state_ = 628;
                    return true;
                }
                else if ('V' == current) {
                    state_ = 637;
                    return true;
                }
                break;
            case 628:
                if ('e' == current) {
                    state_ = 629;
                    return true;
                }
                break;
            case 629:
                if ('e' == current) {
                    state_ = 630;
                    return true;
                }
                break;
            case 630:
                if ('V' == current) {
                    state_ = 631;
                    return true;
                }
                break;
            case 631:
                if ('e' == current) {
                    state_ = 632;
                    return true;
                }
                break;
            case 632:
                if ('c' == current) {
                    state_ = 633;
                    return true;
                }
                break;
            case 633:
                if ('t' == current) {
                    state_ = 634;
                    return true;
                }
                break;
            case 634:
                if ('o' == current) {
                    state_ = 635;
                    return true;
                }
                break;
            case 635:
                if ('r' == current) {
                    state_ = 636;
                    return true;
                }
                break;
            case 636:
                // DownRightTeeVector;
                if (';' == current) {
                    match_ = "\u295F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 637:
                if ('e' == current) {
                    state_ = 638;
                    return true;
                }
                break;
            case 638:
                if ('c' == current) {
                    state_ = 639;
                    return true;
                }
                break;
            case 639:
                if ('t' == current) {
                    state_ = 640;
                    return true;
                }
                break;
            case 640:
                if ('o' == current) {
                    state_ = 641;
                    return true;
                }
                break;
            case 641:
                if ('r' == current) {
                    state_ = 642;
                    return true;
                }
                break;
            case 642:
                // DownRightVector;
                if (';' == current) {
                    match_ = "\u21C1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('B' == current) {
                    state_ = 643;
                    return true;
                }
                break;
            case 643:
                if ('a' == current) {
                    state_ = 644;
                    return true;
                }
                break;
            case 644:
                if ('r' == current) {
                    state_ = 645;
                    return true;
                }
                break;
            case 645:
                // DownRightVectorBar;
                if (';' == current) {
                    match_ = "\u2957";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 646:
                if ('e' == current) {
                    state_ = 647;
                    return true;
                }
                break;
            case 647:
                if ('e' == current) {
                    state_ = 648;
                    return true;
                }
                break;
            case 648:
                // DownTee;
                if (';' == current) {
                    match_ = "\u22A4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('A' == current) {
                    state_ = 649;
                    return true;
                }
                break;
            case 649:
                if ('r' == current) {
                    state_ = 650;
                    return true;
                }
                break;
            case 650:
                if ('r' == current) {
                    state_ = 651;
                    return true;
                }
                break;
            case 651:
                if ('o' == current) {
                    state_ = 652;
                    return true;
                }
                break;
            case 652:
                if ('w' == current) {
                    state_ = 653;
                    return true;
                }
                break;
            case 653:
                // DownTeeArrow;
                if (';' == current) {
                    match_ = "\u21A7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 654:
                if ('r' == current) {
                    state_ = 655;
                    return true;
                }
                break;
            case 655:
                if ('r' == current) {
                    state_ = 656;
                    return true;
                }
                break;
            case 656:
                if ('o' == current) {
                    state_ = 657;
                    return true;
                }
                break;
            case 657:
                if ('w' == current) {
                    state_ = 658;
                    return true;
                }
                break;
            case 658:
                // Downarrow;
                if (';' == current) {
                    match_ = "\u21D3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 659:
                if ('c' == current) {
                    state_ = 660;
                    return true;
                }
                else if ('t' == current) {
                    state_ = 662;
                    return true;
                }
                break;
            case 660:
                if ('r' == current) {
                    state_ = 661;
                    return true;
                }
                break;
            case 661:
                // Dscr;
                if (';' == current) {
                    match_ = "\uD835\uDC9F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 662:
                if ('r' == current) {
                    state_ = 663;
                    return true;
                }
                break;
            case 663:
                if ('o' == current) {
                    state_ = 664;
                    return true;
                }
                break;
            case 664:
                if ('k' == current) {
                    state_ = 665;
                    return true;
                }
                break;
            case 665:
                // Dstrok;
                if (';' == current) {
                    match_ = "\u0110";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 666:
                switch (current) {
                    case 'N':
                        state_ = 667;
                        return true;
                    case 'T':
                        state_ = 669;
                        return true;
                    case 'a':
                        state_ = 671;
                        return true;
                    case 'c':
                        state_ = 676;
                        return true;
                    case 'd':
                        state_ = 685;
                        return true;
                    case 'f':
                        state_ = 688;
                        return true;
                    case 'g':
                        state_ = 690;
                        return true;
                    case 'l':
                        state_ = 695;
                        return true;
                    case 'm':
                        state_ = 701;
                        return true;
                    case 'o':
                        state_ = 734;
                        return true;
                    case 'p':
                        state_ = 740;
                        return true;
                    case 'q':
                        state_ = 746;
                        return true;
                    case 's':
                        state_ = 763;
                        return true;
                    case 't':
                        state_ = 768;
                        return true;
                    case 'u':
                        state_ = 770;
                        return true;
                    case 'x':
                        state_ = 773;
                        return true;
                }
                break;
            case 667:
                if ('G' == current) {
                    state_ = 668;
                    return true;
                }
                break;
            case 668:
                // ENG;
                if (';' == current) {
                    match_ = "\u014A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 669:
                // ETH
                if ('H' == current) {
                    match_ = "\u00D0";
                    matchLength_ = consumedCount_;
                    state_ = 670;
                    return true;
                }
                break;
            case 670:
                // ETH;
                if (';' == current) {
                    match_ = "\u00D0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 671:
                if ('c' == current) {
                    state_ = 672;
                    return true;
                }
                break;
            case 672:
                if ('u' == current) {
                    state_ = 673;
                    return true;
                }
                break;
            case 673:
                if ('t' == current) {
                    state_ = 674;
                    return true;
                }
                break;
            case 674:
                // Eacute
                if ('e' == current) {
                    match_ = "\u00C9";
                    matchLength_ = consumedCount_;
                    state_ = 675;
                    return true;
                }
                break;
            case 675:
                // Eacute;
                if (';' == current) {
                    match_ = "\u00C9";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 676:
                switch (current) {
                    case 'a':
                        state_ = 677;
                        return true;
                    case 'i':
                        state_ = 681;
                        return true;
                    case 'y':
                        state_ = 684;
                        return true;
                }
                break;
            case 677:
                if ('r' == current) {
                    state_ = 678;
                    return true;
                }
                break;
            case 678:
                if ('o' == current) {
                    state_ = 679;
                    return true;
                }
                break;
            case 679:
                if ('n' == current) {
                    state_ = 680;
                    return true;
                }
                break;
            case 680:
                // Ecaron;
                if (';' == current) {
                    match_ = "\u011A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 681:
                if ('r' == current) {
                    state_ = 682;
                    return true;
                }
                break;
            case 682:
                // Ecirc
                if ('c' == current) {
                    match_ = "\u00CA";
                    matchLength_ = consumedCount_;
                    state_ = 683;
                    return true;
                }
                break;
            case 683:
                // Ecirc;
                if (';' == current) {
                    match_ = "\u00CA";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 684:
                // Ecy;
                if (';' == current) {
                    match_ = "\u042D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 685:
                if ('o' == current) {
                    state_ = 686;
                    return true;
                }
                break;
            case 686:
                if ('t' == current) {
                    state_ = 687;
                    return true;
                }
                break;
            case 687:
                // Edot;
                if (';' == current) {
                    match_ = "\u0116";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 688:
                if ('r' == current) {
                    state_ = 689;
                    return true;
                }
                break;
            case 689:
                // Efr;
                if (';' == current) {
                    match_ = "\uD835\uDD08";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 690:
                if ('r' == current) {
                    state_ = 691;
                    return true;
                }
                break;
            case 691:
                if ('a' == current) {
                    state_ = 692;
                    return true;
                }
                break;
            case 692:
                if ('v' == current) {
                    state_ = 693;
                    return true;
                }
                break;
            case 693:
                // Egrave
                if ('e' == current) {
                    match_ = "\u00C8";
                    matchLength_ = consumedCount_;
                    state_ = 694;
                    return true;
                }
                break;
            case 694:
                // Egrave;
                if (';' == current) {
                    match_ = "\u00C8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 695:
                if ('e' == current) {
                    state_ = 696;
                    return true;
                }
                break;
            case 696:
                if ('m' == current) {
                    state_ = 697;
                    return true;
                }
                break;
            case 697:
                if ('e' == current) {
                    state_ = 698;
                    return true;
                }
                break;
            case 698:
                if ('n' == current) {
                    state_ = 699;
                    return true;
                }
                break;
            case 699:
                if ('t' == current) {
                    state_ = 700;
                    return true;
                }
                break;
            case 700:
                // Element;
                if (';' == current) {
                    match_ = "\u2208";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 701:
                if ('a' == current) {
                    state_ = 702;
                    return true;
                }
                else if ('p' == current) {
                    state_ = 705;
                    return true;
                }
                break;
            case 702:
                if ('c' == current) {
                    state_ = 703;
                    return true;
                }
                break;
            case 703:
                if ('r' == current) {
                    state_ = 704;
                    return true;
                }
                break;
            case 704:
                // Emacr;
                if (';' == current) {
                    match_ = "\u0112";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 705:
                if ('t' == current) {
                    state_ = 706;
                    return true;
                }
                break;
            case 706:
                if ('y' == current) {
                    state_ = 707;
                    return true;
                }
                break;
            case 707:
                if ('S' == current) {
                    state_ = 708;
                    return true;
                }
                else if ('V' == current) {
                    state_ = 719;
                    return true;
                }
                break;
            case 708:
                if ('m' == current) {
                    state_ = 709;
                    return true;
                }
                break;
            case 709:
                if ('a' == current) {
                    state_ = 710;
                    return true;
                }
                break;
            case 710:
                if ('l' == current) {
                    state_ = 711;
                    return true;
                }
                break;
            case 711:
                if ('l' == current) {
                    state_ = 712;
                    return true;
                }
                break;
            case 712:
                if ('S' == current) {
                    state_ = 713;
                    return true;
                }
                break;
            case 713:
                if ('q' == current) {
                    state_ = 714;
                    return true;
                }
                break;
            case 714:
                if ('u' == current) {
                    state_ = 715;
                    return true;
                }
                break;
            case 715:
                if ('a' == current) {
                    state_ = 716;
                    return true;
                }
                break;
            case 716:
                if ('r' == current) {
                    state_ = 717;
                    return true;
                }
                break;
            case 717:
                if ('e' == current) {
                    state_ = 718;
                    return true;
                }
                break;
            case 718:
                // EmptySmallSquare;
                if (';' == current) {
                    match_ = "\u25FB";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 719:
                if ('e' == current) {
                    state_ = 720;
                    return true;
                }
                break;
            case 720:
                if ('r' == current) {
                    state_ = 721;
                    return true;
                }
                break;
            case 721:
                if ('y' == current) {
                    state_ = 722;
                    return true;
                }
                break;
            case 722:
                if ('S' == current) {
                    state_ = 723;
                    return true;
                }
                break;
            case 723:
                if ('m' == current) {
                    state_ = 724;
                    return true;
                }
                break;
            case 724:
                if ('a' == current) {
                    state_ = 725;
                    return true;
                }
                break;
            case 725:
                if ('l' == current) {
                    state_ = 726;
                    return true;
                }
                break;
            case 726:
                if ('l' == current) {
                    state_ = 727;
                    return true;
                }
                break;
            case 727:
                if ('S' == current) {
                    state_ = 728;
                    return true;
                }
                break;
            case 728:
                if ('q' == current) {
                    state_ = 729;
                    return true;
                }
                break;
            case 729:
                if ('u' == current) {
                    state_ = 730;
                    return true;
                }
                break;
            case 730:
                if ('a' == current) {
                    state_ = 731;
                    return true;
                }
                break;
            case 731:
                if ('r' == current) {
                    state_ = 732;
                    return true;
                }
                break;
            case 732:
                if ('e' == current) {
                    state_ = 733;
                    return true;
                }
                break;
            case 733:
                // EmptyVerySmallSquare;
                if (';' == current) {
                    match_ = "\u25AB";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 734:
                if ('g' == current) {
                    state_ = 735;
                    return true;
                }
                else if ('p' == current) {
                    state_ = 738;
                    return true;
                }
                break;
            case 735:
                if ('o' == current) {
                    state_ = 736;
                    return true;
                }
                break;
            case 736:
                if ('n' == current) {
                    state_ = 737;
                    return true;
                }
                break;
            case 737:
                // Eogon;
                if (';' == current) {
                    match_ = "\u0118";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 738:
                if ('f' == current) {
                    state_ = 739;
                    return true;
                }
                break;
            case 739:
                // Eopf;
                if (';' == current) {
                    match_ = "\uD835\uDD3C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 740:
                if ('s' == current) {
                    state_ = 741;
                    return true;
                }
                break;
            case 741:
                if ('i' == current) {
                    state_ = 742;
                    return true;
                }
                break;
            case 742:
                if ('l' == current) {
                    state_ = 743;
                    return true;
                }
                break;
            case 743:
                if ('o' == current) {
                    state_ = 744;
                    return true;
                }
                break;
            case 744:
                if ('n' == current) {
                    state_ = 745;
                    return true;
                }
                break;
            case 745:
                // Epsilon;
                if (';' == current) {
                    match_ = "\u0395";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 746:
                if ('u' == current) {
                    state_ = 747;
                    return true;
                }
                break;
            case 747:
                if ('a' == current) {
                    state_ = 748;
                    return true;
                }
                else if ('i' == current) {
                    state_ = 755;
                    return true;
                }
                break;
            case 748:
                if ('l' == current) {
                    state_ = 749;
                    return true;
                }
                break;
            case 749:
                // Equal;
                if (';' == current) {
                    match_ = "\u2A75";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('T' == current) {
                    state_ = 750;
                    return true;
                }
                break;
            case 750:
                if ('i' == current) {
                    state_ = 751;
                    return true;
                }
                break;
            case 751:
                if ('l' == current) {
                    state_ = 752;
                    return true;
                }
                break;
            case 752:
                if ('d' == current) {
                    state_ = 753;
                    return true;
                }
                break;
            case 753:
                if ('e' == current) {
                    state_ = 754;
                    return true;
                }
                break;
            case 754:
                // EqualTilde;
                if (';' == current) {
                    match_ = "\u2242";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 755:
                if ('l' == current) {
                    state_ = 756;
                    return true;
                }
                break;
            case 756:
                if ('i' == current) {
                    state_ = 757;
                    return true;
                }
                break;
            case 757:
                if ('b' == current) {
                    state_ = 758;
                    return true;
                }
                break;
            case 758:
                if ('r' == current) {
                    state_ = 759;
                    return true;
                }
                break;
            case 759:
                if ('i' == current) {
                    state_ = 760;
                    return true;
                }
                break;
            case 760:
                if ('u' == current) {
                    state_ = 761;
                    return true;
                }
                break;
            case 761:
                if ('m' == current) {
                    state_ = 762;
                    return true;
                }
                break;
            case 762:
                // Equilibrium;
                if (';' == current) {
                    match_ = "\u21CC";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 763:
                if ('c' == current) {
                    state_ = 764;
                    return true;
                }
                else if ('i' == current) {
                    state_ = 766;
                    return true;
                }
                break;
            case 764:
                if ('r' == current) {
                    state_ = 765;
                    return true;
                }
                break;
            case 765:
                // Escr;
                if (';' == current) {
                    match_ = "\u2130";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 766:
                if ('m' == current) {
                    state_ = 767;
                    return true;
                }
                break;
            case 767:
                // Esim;
                if (';' == current) {
                    match_ = "\u2A73";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 768:
                if ('a' == current) {
                    state_ = 769;
                    return true;
                }
                break;
            case 769:
                // Eta;
                if (';' == current) {
                    match_ = "\u0397";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 770:
                if ('m' == current) {
                    state_ = 771;
                    return true;
                }
                break;
            case 771:
                // Euml
                if ('l' == current) {
                    match_ = "\u00CB";
                    matchLength_ = consumedCount_;
                    state_ = 772;
                    return true;
                }
                break;
            case 772:
                // Euml;
                if (';' == current) {
                    match_ = "\u00CB";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 773:
                if ('i' == current) {
                    state_ = 774;
                    return true;
                }
                else if ('p' == current) {
                    state_ = 778;
                    return true;
                }
                break;
            case 774:
                if ('s' == current) {
                    state_ = 775;
                    return true;
                }
                break;
            case 775:
                if ('t' == current) {
                    state_ = 776;
                    return true;
                }
                break;
            case 776:
                if ('s' == current) {
                    state_ = 777;
                    return true;
                }
                break;
            case 777:
                // Exists;
                if (';' == current) {
                    match_ = "\u2203";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 778:
                if ('o' == current) {
                    state_ = 779;
                    return true;
                }
                break;
            case 779:
                if ('n' == current) {
                    state_ = 780;
                    return true;
                }
                break;
            case 780:
                if ('e' == current) {
                    state_ = 781;
                    return true;
                }
                break;
            case 781:
                if ('n' == current) {
                    state_ = 782;
                    return true;
                }
                break;
            case 782:
                if ('t' == current) {
                    state_ = 783;
                    return true;
                }
                break;
            case 783:
                if ('i' == current) {
                    state_ = 784;
                    return true;
                }
                break;
            case 784:
                if ('a' == current) {
                    state_ = 785;
                    return true;
                }
                break;
            case 785:
                if ('l' == current) {
                    state_ = 786;
                    return true;
                }
                break;
            case 786:
                if ('E' == current) {
                    state_ = 787;
                    return true;
                }
                break;
            case 787:
                // ExponentialE;
                if (';' == current) {
                    match_ = "\u2147";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 788:
                switch (current) {
                    case 'c':
                        state_ = 789;
                        return true;
                    case 'f':
                        state_ = 791;
                        return true;
                    case 'i':
                        state_ = 793;
                        return true;
                    case 'o':
                        state_ = 824;
                        return true;
                    case 's':
                        state_ = 839;
                        return true;
                }
                break;
            case 789:
                if ('y' == current) {
                    state_ = 790;
                    return true;
                }
                break;
            case 790:
                // Fcy;
                if (';' == current) {
                    match_ = "\u0424";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 791:
                if ('r' == current) {
                    state_ = 792;
                    return true;
                }
                break;
            case 792:
                // Ffr;
                if (';' == current) {
                    match_ = "\uD835\uDD09";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 793:
                if ('l' == current) {
                    state_ = 794;
                    return true;
                }
                break;
            case 794:
                if ('l' == current) {
                    state_ = 795;
                    return true;
                }
                break;
            case 795:
                if ('e' == current) {
                    state_ = 796;
                    return true;
                }
                break;
            case 796:
                if ('d' == current) {
                    state_ = 797;
                    return true;
                }
                break;
            case 797:
                if ('S' == current) {
                    state_ = 798;
                    return true;
                }
                else if ('V' == current) {
                    state_ = 809;
                    return true;
                }
                break;
            case 798:
                if ('m' == current) {
                    state_ = 799;
                    return true;
                }
                break;
            case 799:
                if ('a' == current) {
                    state_ = 800;
                    return true;
                }
                break;
            case 800:
                if ('l' == current) {
                    state_ = 801;
                    return true;
                }
                break;
            case 801:
                if ('l' == current) {
                    state_ = 802;
                    return true;
                }
                break;
            case 802:
                if ('S' == current) {
                    state_ = 803;
                    return true;
                }
                break;
            case 803:
                if ('q' == current) {
                    state_ = 804;
                    return true;
                }
                break;
            case 804:
                if ('u' == current) {
                    state_ = 805;
                    return true;
                }
                break;
            case 805:
                if ('a' == current) {
                    state_ = 806;
                    return true;
                }
                break;
            case 806:
                if ('r' == current) {
                    state_ = 807;
                    return true;
                }
                break;
            case 807:
                if ('e' == current) {
                    state_ = 808;
                    return true;
                }
                break;
            case 808:
                // FilledSmallSquare;
                if (';' == current) {
                    match_ = "\u25FC";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 809:
                if ('e' == current) {
                    state_ = 810;
                    return true;
                }
                break;
            case 810:
                if ('r' == current) {
                    state_ = 811;
                    return true;
                }
                break;
            case 811:
                if ('y' == current) {
                    state_ = 812;
                    return true;
                }
                break;
            case 812:
                if ('S' == current) {
                    state_ = 813;
                    return true;
                }
                break;
            case 813:
                if ('m' == current) {
                    state_ = 814;
                    return true;
                }
                break;
            case 814:
                if ('a' == current) {
                    state_ = 815;
                    return true;
                }
                break;
            case 815:
                if ('l' == current) {
                    state_ = 816;
                    return true;
                }
                break;
            case 816:
                if ('l' == current) {
                    state_ = 817;
                    return true;
                }
                break;
            case 817:
                if ('S' == current) {
                    state_ = 818;
                    return true;
                }
                break;
            case 818:
                if ('q' == current) {
                    state_ = 819;
                    return true;
                }
                break;
            case 819:
                if ('u' == current) {
                    state_ = 820;
                    return true;
                }
                break;
            case 820:
                if ('a' == current) {
                    state_ = 821;
                    return true;
                }
                break;
            case 821:
                if ('r' == current) {
                    state_ = 822;
                    return true;
                }
                break;
            case 822:
                if ('e' == current) {
                    state_ = 823;
                    return true;
                }
                break;
            case 823:
                // FilledVerySmallSquare;
                if (';' == current) {
                    match_ = "\u25AA";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 824:
                switch (current) {
                    case 'p':
                        state_ = 825;
                        return true;
                    case 'r':
                        state_ = 827;
                        return true;
                    case 'u':
                        state_ = 831;
                        return true;
                }
                break;
            case 825:
                if ('f' == current) {
                    state_ = 826;
                    return true;
                }
                break;
            case 826:
                // Fopf;
                if (';' == current) {
                    match_ = "\uD835\uDD3D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 827:
                if ('A' == current) {
                    state_ = 828;
                    return true;
                }
                break;
            case 828:
                if ('l' == current) {
                    state_ = 829;
                    return true;
                }
                break;
            case 829:
                if ('l' == current) {
                    state_ = 830;
                    return true;
                }
                break;
            case 830:
                // ForAll;
                if (';' == current) {
                    match_ = "\u2200";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 831:
                if ('r' == current) {
                    state_ = 832;
                    return true;
                }
                break;
            case 832:
                if ('i' == current) {
                    state_ = 833;
                    return true;
                }
                break;
            case 833:
                if ('e' == current) {
                    state_ = 834;
                    return true;
                }
                break;
            case 834:
                if ('r' == current) {
                    state_ = 835;
                    return true;
                }
                break;
            case 835:
                if ('t' == current) {
                    state_ = 836;
                    return true;
                }
                break;
            case 836:
                if ('r' == current) {
                    state_ = 837;
                    return true;
                }
                break;
            case 837:
                if ('f' == current) {
                    state_ = 838;
                    return true;
                }
                break;
            case 838:
                // Fouriertrf;
                if (';' == current) {
                    match_ = "\u2131";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 839:
                if ('c' == current) {
                    state_ = 840;
                    return true;
                }
                break;
            case 840:
                if ('r' == current) {
                    state_ = 841;
                    return true;
                }
                break;
            case 841:
                // Fscr;
                if (';' == current) {
                    match_ = "\u2131";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 842:
                switch (current) {
                    case 'J':
                        state_ = 843;
                        return true;
                    case 'T': // GT
                        match_ = ">";
                        matchLength_ = consumedCount_;
                        state_ = 846;
                        return true;
                    case 'a':
                        state_ = 847;
                        return true;
                    case 'b':
                        state_ = 852;
                        return true;
                    case 'c':
                        state_ = 857;
                        return true;
                    case 'd':
                        state_ = 866;
                        return true;
                    case 'f':
                        state_ = 869;
                        return true;
                    case 'g':
                        state_ = 871;
                        return true;
                    case 'o':
                        state_ = 872;
                        return true;
                    case 'r':
                        state_ = 875;
                        return true;
                    case 's':
                        state_ = 925;
                        return true;
                    case 't':
                        state_ = 928;
                        return true;
                }
                break;
            case 843:
                if ('c' == current) {
                    state_ = 844;
                    return true;
                }
                break;
            case 844:
                if ('y' == current) {
                    state_ = 845;
                    return true;
                }
                break;
            case 845:
                // GJcy;
                if (';' == current) {
                    match_ = "\u0403";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 846:
                // GT;
                if (';' == current) {
                    match_ = ">";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 847:
                if ('m' == current) {
                    state_ = 848;
                    return true;
                }
                break;
            case 848:
                if ('m' == current) {
                    state_ = 849;
                    return true;
                }
                break;
            case 849:
                if ('a' == current) {
                    state_ = 850;
                    return true;
                }
                break;
            case 850:
                // Gamma;
                if (';' == current) {
                    match_ = "\u0393";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('d' == current) {
                    state_ = 851;
                    return true;
                }
                break;
            case 851:
                // Gammad;
                if (';' == current) {
                    match_ = "\u03DC";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 852:
                if ('r' == current) {
                    state_ = 853;
                    return true;
                }
                break;
            case 853:
                if ('e' == current) {
                    state_ = 854;
                    return true;
                }
                break;
            case 854:
                if ('v' == current) {
                    state_ = 855;
                    return true;
                }
                break;
            case 855:
                if ('e' == current) {
                    state_ = 856;
                    return true;
                }
                break;
            case 856:
                // Gbreve;
                if (';' == current) {
                    match_ = "\u011E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 857:
                switch (current) {
                    case 'e':
                        state_ = 858;
                        return true;
                    case 'i':
                        state_ = 862;
                        return true;
                    case 'y':
                        state_ = 865;
                        return true;
                }
                break;
            case 858:
                if ('d' == current) {
                    state_ = 859;
                    return true;
                }
                break;
            case 859:
                if ('i' == current) {
                    state_ = 860;
                    return true;
                }
                break;
            case 860:
                if ('l' == current) {
                    state_ = 861;
                    return true;
                }
                break;
            case 861:
                // Gcedil;
                if (';' == current) {
                    match_ = "\u0122";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 862:
                if ('r' == current) {
                    state_ = 863;
                    return true;
                }
                break;
            case 863:
                if ('c' == current) {
                    state_ = 864;
                    return true;
                }
                break;
            case 864:
                // Gcirc;
                if (';' == current) {
                    match_ = "\u011C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 865:
                // Gcy;
                if (';' == current) {
                    match_ = "\u0413";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 866:
                if ('o' == current) {
                    state_ = 867;
                    return true;
                }
                break;
            case 867:
                if ('t' == current) {
                    state_ = 868;
                    return true;
                }
                break;
            case 868:
                // Gdot;
                if (';' == current) {
                    match_ = "\u0120";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 869:
                if ('r' == current) {
                    state_ = 870;
                    return true;
                }
                break;
            case 870:
                // Gfr;
                if (';' == current) {
                    match_ = "\uD835\uDD0A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 871:
                // Gg;
                if (';' == current) {
                    match_ = "\u22D9";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 872:
                if ('p' == current) {
                    state_ = 873;
                    return true;
                }
                break;
            case 873:
                if ('f' == current) {
                    state_ = 874;
                    return true;
                }
                break;
            case 874:
                // Gopf;
                if (';' == current) {
                    match_ = "\uD835\uDD3E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 875:
                if ('e' == current) {
                    state_ = 876;
                    return true;
                }
                break;
            case 876:
                if ('a' == current) {
                    state_ = 877;
                    return true;
                }
                break;
            case 877:
                if ('t' == current) {
                    state_ = 878;
                    return true;
                }
                break;
            case 878:
                if ('e' == current) {
                    state_ = 879;
                    return true;
                }
                break;
            case 879:
                if ('r' == current) {
                    state_ = 880;
                    return true;
                }
                break;
            case 880:
                switch (current) {
                    case 'E':
                        state_ = 881;
                        return true;
                    case 'F':
                        state_ = 890;
                        return true;
                    case 'G':
                        state_ = 899;
                        return true;
                    case 'L':
                        state_ = 906;
                        return true;
                    case 'S':
                        state_ = 910;
                        return true;
                    case 'T':
                        state_ = 920;
                        return true;
                }
                break;
            case 881:
                if ('q' == current) {
                    state_ = 882;
                    return true;
                }
                break;
            case 882:
                if ('u' == current) {
                    state_ = 883;
                    return true;
                }
                break;
            case 883:
                if ('a' == current) {
                    state_ = 884;
                    return true;
                }
                break;
            case 884:
                if ('l' == current) {
                    state_ = 885;
                    return true;
                }
                break;
            case 885:
                // GreaterEqual;
                if (';' == current) {
                    match_ = "\u2265";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('L' == current) {
                    state_ = 886;
                    return true;
                }
                break;
            case 886:
                if ('e' == current) {
                    state_ = 887;
                    return true;
                }
                break;
            case 887:
                if ('s' == current) {
                    state_ = 888;
                    return true;
                }
                break;
            case 888:
                if ('s' == current) {
                    state_ = 889;
                    return true;
                }
                break;
            case 889:
                // GreaterEqualLess;
                if (';' == current) {
                    match_ = "\u22DB";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 890:
                if ('u' == current) {
                    state_ = 891;
                    return true;
                }
                break;
            case 891:
                if ('l' == current) {
                    state_ = 892;
                    return true;
                }
                break;
            case 892:
                if ('l' == current) {
                    state_ = 893;
                    return true;
                }
                break;
            case 893:
                if ('E' == current) {
                    state_ = 894;
                    return true;
                }
                break;
            case 894:
                if ('q' == current) {
                    state_ = 895;
                    return true;
                }
                break;
            case 895:
                if ('u' == current) {
                    state_ = 896;
                    return true;
                }
                break;
            case 896:
                if ('a' == current) {
                    state_ = 897;
                    return true;
                }
                break;
            case 897:
                if ('l' == current) {
                    state_ = 898;
                    return true;
                }
                break;
            case 898:
                // GreaterFullEqual;
                if (';' == current) {
                    match_ = "\u2267";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 899:
                if ('r' == current) {
                    state_ = 900;
                    return true;
                }
                break;
            case 900:
                if ('e' == current) {
                    state_ = 901;
                    return true;
                }
                break;
            case 901:
                if ('a' == current) {
                    state_ = 902;
                    return true;
                }
                break;
            case 902:
                if ('t' == current) {
                    state_ = 903;
                    return true;
                }
                break;
            case 903:
                if ('e' == current) {
                    state_ = 904;
                    return true;
                }
                break;
            case 904:
                if ('r' == current) {
                    state_ = 905;
                    return true;
                }
                break;
            case 905:
                // GreaterGreater;
                if (';' == current) {
                    match_ = "\u2AA2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 906:
                if ('e' == current) {
                    state_ = 907;
                    return true;
                }
                break;
            case 907:
                if ('s' == current) {
                    state_ = 908;
                    return true;
                }
                break;
            case 908:
                if ('s' == current) {
                    state_ = 909;
                    return true;
                }
                break;
            case 909:
                // GreaterLess;
                if (';' == current) {
                    match_ = "\u2277";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 910:
                if ('l' == current) {
                    state_ = 911;
                    return true;
                }
                break;
            case 911:
                if ('a' == current) {
                    state_ = 912;
                    return true;
                }
                break;
            case 912:
                if ('n' == current) {
                    state_ = 913;
                    return true;
                }
                break;
            case 913:
                if ('t' == current) {
                    state_ = 914;
                    return true;
                }
                break;
            case 914:
                if ('E' == current) {
                    state_ = 915;
                    return true;
                }
                break;
            case 915:
                if ('q' == current) {
                    state_ = 916;
                    return true;
                }
                break;
            case 916:
                if ('u' == current) {
                    state_ = 917;
                    return true;
                }
                break;
            case 917:
                if ('a' == current) {
                    state_ = 918;
                    return true;
                }
                break;
            case 918:
                if ('l' == current) {
                    state_ = 919;
                    return true;
                }
                break;
            case 919:
                // GreaterSlantEqual;
                if (';' == current) {
                    match_ = "\u2A7E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 920:
                if ('i' == current) {
                    state_ = 921;
                    return true;
                }
                break;
            case 921:
                if ('l' == current) {
                    state_ = 922;
                    return true;
                }
                break;
            case 922:
                if ('d' == current) {
                    state_ = 923;
                    return true;
                }
                break;
            case 923:
                if ('e' == current) {
                    state_ = 924;
                    return true;
                }
                break;
            case 924:
                // GreaterTilde;
                if (';' == current) {
                    match_ = "\u2273";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 925:
                if ('c' == current) {
                    state_ = 926;
                    return true;
                }
                break;
            case 926:
                if ('r' == current) {
                    state_ = 927;
                    return true;
                }
                break;
            case 927:
                // Gscr;
                if (';' == current) {
                    match_ = "\uD835\uDCA2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 928:
                // Gt;
                if (';' == current) {
                    match_ = "\u226B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 929:
                switch (current) {
                    case 'A':
                        state_ = 930;
                        return true;
                    case 'a':
                        state_ = 935;
                        return true;
                    case 'c':
                        state_ = 940;
                        return true;
                    case 'f':
                        state_ = 944;
                        return true;
                    case 'i':
                        state_ = 946;
                        return true;
                    case 'o':
                        state_ = 957;
                        return true;
                    case 's':
                        state_ = 972;
                        return true;
                    case 'u':
                        state_ = 979;
                        return true;
                }
                break;
            case 930:
                if ('R' == current) {
                    state_ = 931;
                    return true;
                }
                break;
            case 931:
                if ('D' == current) {
                    state_ = 932;
                    return true;
                }
                break;
            case 932:
                if ('c' == current) {
                    state_ = 933;
                    return true;
                }
                break;
            case 933:
                if ('y' == current) {
                    state_ = 934;
                    return true;
                }
                break;
            case 934:
                // HARDcy;
                if (';' == current) {
                    match_ = "\u042A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 935:
                if ('c' == current) {
                    state_ = 936;
                    return true;
                }
                else if ('t' == current) {
                    state_ = 939;
                    return true;
                }
                break;
            case 936:
                if ('e' == current) {
                    state_ = 937;
                    return true;
                }
                break;
            case 937:
                if ('k' == current) {
                    state_ = 938;
                    return true;
                }
                break;
            case 938:
                // Hacek;
                if (';' == current) {
                    match_ = "\u02C7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 939:
                // Hat;
                if (';' == current) {
                    match_ = "^";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 940:
                if ('i' == current) {
                    state_ = 941;
                    return true;
                }
                break;
            case 941:
                if ('r' == current) {
                    state_ = 942;
                    return true;
                }
                break;
            case 942:
                if ('c' == current) {
                    state_ = 943;
                    return true;
                }
                break;
            case 943:
                // Hcirc;
                if (';' == current) {
                    match_ = "\u0124";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 944:
                if ('r' == current) {
                    state_ = 945;
                    return true;
                }
                break;
            case 945:
                // Hfr;
                if (';' == current) {
                    match_ = "\u210C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 946:
                if ('l' == current) {
                    state_ = 947;
                    return true;
                }
                break;
            case 947:
                if ('b' == current) {
                    state_ = 948;
                    return true;
                }
                break;
            case 948:
                if ('e' == current) {
                    state_ = 949;
                    return true;
                }
                break;
            case 949:
                if ('r' == current) {
                    state_ = 950;
                    return true;
                }
                break;
            case 950:
                if ('t' == current) {
                    state_ = 951;
                    return true;
                }
                break;
            case 951:
                if ('S' == current) {
                    state_ = 952;
                    return true;
                }
                break;
            case 952:
                if ('p' == current) {
                    state_ = 953;
                    return true;
                }
                break;
            case 953:
                if ('a' == current) {
                    state_ = 954;
                    return true;
                }
                break;
            case 954:
                if ('c' == current) {
                    state_ = 955;
                    return true;
                }
                break;
            case 955:
                if ('e' == current) {
                    state_ = 956;
                    return true;
                }
                break;
            case 956:
                // HilbertSpace;
                if (';' == current) {
                    match_ = "\u210B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 957:
                if ('p' == current) {
                    state_ = 958;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 960;
                    return true;
                }
                break;
            case 958:
                if ('f' == current) {
                    state_ = 959;
                    return true;
                }
                break;
            case 959:
                // Hopf;
                if (';' == current) {
                    match_ = "\u210D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 960:
                if ('i' == current) {
                    state_ = 961;
                    return true;
                }
                break;
            case 961:
                if ('z' == current) {
                    state_ = 962;
                    return true;
                }
                break;
            case 962:
                if ('o' == current) {
                    state_ = 963;
                    return true;
                }
                break;
            case 963:
                if ('n' == current) {
                    state_ = 964;
                    return true;
                }
                break;
            case 964:
                if ('t' == current) {
                    state_ = 965;
                    return true;
                }
                break;
            case 965:
                if ('a' == current) {
                    state_ = 966;
                    return true;
                }
                break;
            case 966:
                if ('l' == current) {
                    state_ = 967;
                    return true;
                }
                break;
            case 967:
                if ('L' == current) {
                    state_ = 968;
                    return true;
                }
                break;
            case 968:
                if ('i' == current) {
                    state_ = 969;
                    return true;
                }
                break;
            case 969:
                if ('n' == current) {
                    state_ = 970;
                    return true;
                }
                break;
            case 970:
                if ('e' == current) {
                    state_ = 971;
                    return true;
                }
                break;
            case 971:
                // HorizontalLine;
                if (';' == current) {
                    match_ = "\u2500";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 972:
                if ('c' == current) {
                    state_ = 973;
                    return true;
                }
                else if ('t' == current) {
                    state_ = 975;
                    return true;
                }
                break;
            case 973:
                if ('r' == current) {
                    state_ = 974;
                    return true;
                }
                break;
            case 974:
                // Hscr;
                if (';' == current) {
                    match_ = "\u210B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 975:
                if ('r' == current) {
                    state_ = 976;
                    return true;
                }
                break;
            case 976:
                if ('o' == current) {
                    state_ = 977;
                    return true;
                }
                break;
            case 977:
                if ('k' == current) {
                    state_ = 978;
                    return true;
                }
                break;
            case 978:
                // Hstrok;
                if (';' == current) {
                    match_ = "\u0126";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 979:
                if ('m' == current) {
                    state_ = 980;
                    return true;
                }
                break;
            case 980:
                if ('p' == current) {
                    state_ = 981;
                    return true;
                }
                break;
            case 981:
                if ('D' == current) {
                    state_ = 982;
                    return true;
                }
                else if ('E' == current) {
                    state_ = 990;
                    return true;
                }
                break;
            case 982:
                if ('o' == current) {
                    state_ = 983;
                    return true;
                }
                break;
            case 983:
                if ('w' == current) {
                    state_ = 984;
                    return true;
                }
                break;
            case 984:
                if ('n' == current) {
                    state_ = 985;
                    return true;
                }
                break;
            case 985:
                if ('H' == current) {
                    state_ = 986;
                    return true;
                }
                break;
            case 986:
                if ('u' == current) {
                    state_ = 987;
                    return true;
                }
                break;
            case 987:
                if ('m' == current) {
                    state_ = 988;
                    return true;
                }
                break;
            case 988:
                if ('p' == current) {
                    state_ = 989;
                    return true;
                }
                break;
            case 989:
                // HumpDownHump;
                if (';' == current) {
                    match_ = "\u224E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 990:
                if ('q' == current) {
                    state_ = 991;
                    return true;
                }
                break;
            case 991:
                if ('u' == current) {
                    state_ = 992;
                    return true;
                }
                break;
            case 992:
                if ('a' == current) {
                    state_ = 993;
                    return true;
                }
                break;
            case 993:
                if ('l' == current) {
                    state_ = 994;
                    return true;
                }
                break;
            case 994:
                // HumpEqual;
                if (';' == current) {
                    match_ = "\u224F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 995:
                switch (current) {
                    case 'E':
                        state_ = 996;
                        return true;
                    case 'J':
                        state_ = 999;
                        return true;
                    case 'O':
                        state_ = 1003;
                        return true;
                    case 'a':
                        state_ = 1006;
                        return true;
                    case 'c':
                        state_ = 1011;
                        return true;
                    case 'd':
                        state_ = 1016;
                        return true;
                    case 'f':
                        state_ = 1019;
                        return true;
                    case 'g':
                        state_ = 1021;
                        return true;
                    case 'm':
                        state_ = 1026;
                        return true;
                    case 'n':
                        state_ = 1042;
                        return true;
                    case 'o':
                        state_ = 1074;
                        return true;
                    case 's':
                        state_ = 1082;
                        return true;
                    case 't':
                        state_ = 1085;
                        return true;
                    case 'u':
                        state_ = 1090;
                        return true;
                }
                break;
            case 996:
                if ('c' == current) {
                    state_ = 997;
                    return true;
                }
                break;
            case 997:
                if ('y' == current) {
                    state_ = 998;
                    return true;
                }
                break;
            case 998:
                // IEcy;
                if (';' == current) {
                    match_ = "\u0415";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 999:
                if ('l' == current) {
                    state_ = 1000;
                    return true;
                }
                break;
        }
        return false;
    }

    private boolean parse2(final int current) {
        consumedCount_++;
        switch (state_) {
            case 1000:
                if ('i' == current) {
                    state_ = 1001;
                    return true;
                }
                break;
            case 1001:
                if ('g' == current) {
                    state_ = 1002;
                    return true;
                }
                break;
            case 1002:
                // IJlig;
                if (';' == current) {
                    match_ = "\u0132";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1003:
                if ('c' == current) {
                    state_ = 1004;
                    return true;
                }
                break;
            case 1004:
                if ('y' == current) {
                    state_ = 1005;
                    return true;
                }
                break;
            case 1005:
                // IOcy;
                if (';' == current) {
                    match_ = "\u0401";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1006:
                if ('c' == current) {
                    state_ = 1007;
                    return true;
                }
                break;
            case 1007:
                if ('u' == current) {
                    state_ = 1008;
                    return true;
                }
                break;
            case 1008:
                if ('t' == current) {
                    state_ = 1009;
                    return true;
                }
                break;
            case 1009:
                // Iacute
                if ('e' == current) {
                    match_ = "\u00CD";
                    matchLength_ = consumedCount_;
                    state_ = 1010;
                    return true;
                }
                break;
            case 1010:
                // Iacute;
                if (';' == current) {
                    match_ = "\u00CD";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1011:
                if ('i' == current) {
                    state_ = 1012;
                    return true;
                }
                else if ('y' == current) {
                    state_ = 1015;
                    return true;
                }
                break;
            case 1012:
                if ('r' == current) {
                    state_ = 1013;
                    return true;
                }
                break;
            case 1013:
                // Icirc
                if ('c' == current) {
                    match_ = "\u00CE";
                    matchLength_ = consumedCount_;
                    state_ = 1014;
                    return true;
                }
                break;
            case 1014:
                // Icirc;
                if (';' == current) {
                    match_ = "\u00CE";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1015:
                // Icy;
                if (';' == current) {
                    match_ = "\u0418";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1016:
                if ('o' == current) {
                    state_ = 1017;
                    return true;
                }
                break;
            case 1017:
                if ('t' == current) {
                    state_ = 1018;
                    return true;
                }
                break;
            case 1018:
                // Idot;
                if (';' == current) {
                    match_ = "\u0130";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1019:
                if ('r' == current) {
                    state_ = 1020;
                    return true;
                }
                break;
            case 1020:
                // Ifr;
                if (';' == current) {
                    match_ = "\u2111";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1021:
                if ('r' == current) {
                    state_ = 1022;
                    return true;
                }
                break;
            case 1022:
                if ('a' == current) {
                    state_ = 1023;
                    return true;
                }
                break;
            case 1023:
                if ('v' == current) {
                    state_ = 1024;
                    return true;
                }
                break;
            case 1024:
                // Igrave
                if ('e' == current) {
                    match_ = "\u00CC";
                    matchLength_ = consumedCount_;
                    state_ = 1025;
                    return true;
                }
                break;
            case 1025:
                // Igrave;
                if (';' == current) {
                    match_ = "\u00CC";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1026:
                switch (current) {
                    case ';': // Im;
                        match_ = "\u2111";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'a':
                        state_ = 1027;
                        return true;
                    case 'p':
                        state_ = 1037;
                        return true;
                }
                break;
            case 1027:
                if ('c' == current) {
                    state_ = 1028;
                    return true;
                }
                else if ('g' == current) {
                    state_ = 1030;
                    return true;
                }
                break;
            case 1028:
                if ('r' == current) {
                    state_ = 1029;
                    return true;
                }
                break;
            case 1029:
                // Imacr;
                if (';' == current) {
                    match_ = "\u012A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1030:
                if ('i' == current) {
                    state_ = 1031;
                    return true;
                }
                break;
            case 1031:
                if ('n' == current) {
                    state_ = 1032;
                    return true;
                }
                break;
            case 1032:
                if ('a' == current) {
                    state_ = 1033;
                    return true;
                }
                break;
            case 1033:
                if ('r' == current) {
                    state_ = 1034;
                    return true;
                }
                break;
            case 1034:
                if ('y' == current) {
                    state_ = 1035;
                    return true;
                }
                break;
            case 1035:
                if ('I' == current) {
                    state_ = 1036;
                    return true;
                }
                break;
            case 1036:
                // ImaginaryI;
                if (';' == current) {
                    match_ = "\u2148";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1037:
                if ('l' == current) {
                    state_ = 1038;
                    return true;
                }
                break;
            case 1038:
                if ('i' == current) {
                    state_ = 1039;
                    return true;
                }
                break;
            case 1039:
                if ('e' == current) {
                    state_ = 1040;
                    return true;
                }
                break;
            case 1040:
                if ('s' == current) {
                    state_ = 1041;
                    return true;
                }
                break;
            case 1041:
                // Implies;
                if (';' == current) {
                    match_ = "\u21D2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1042:
                if ('t' == current) {
                    state_ = 1043;
                    return true;
                }
                else if ('v' == current) {
                    state_ = 1057;
                    return true;
                }
                break;
            case 1043:
                // Int;
                if (';' == current) {
                    match_ = "\u222C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state_ = 1044;
                    return true;
                }
                break;
            case 1044:
                if ('g' == current) {
                    state_ = 1045;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 1049;
                    return true;
                }
                break;
            case 1045:
                if ('r' == current) {
                    state_ = 1046;
                    return true;
                }
                break;
            case 1046:
                if ('a' == current) {
                    state_ = 1047;
                    return true;
                }
                break;
            case 1047:
                if ('l' == current) {
                    state_ = 1048;
                    return true;
                }
                break;
            case 1048:
                // Integral;
                if (';' == current) {
                    match_ = "\u222B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1049:
                if ('s' == current) {
                    state_ = 1050;
                    return true;
                }
                break;
            case 1050:
                if ('e' == current) {
                    state_ = 1051;
                    return true;
                }
                break;
            case 1051:
                if ('c' == current) {
                    state_ = 1052;
                    return true;
                }
                break;
            case 1052:
                if ('t' == current) {
                    state_ = 1053;
                    return true;
                }
                break;
            case 1053:
                if ('i' == current) {
                    state_ = 1054;
                    return true;
                }
                break;
            case 1054:
                if ('o' == current) {
                    state_ = 1055;
                    return true;
                }
                break;
            case 1055:
                if ('n' == current) {
                    state_ = 1056;
                    return true;
                }
                break;
            case 1056:
                // Intersection;
                if (';' == current) {
                    match_ = "\u22C2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1057:
                if ('i' == current) {
                    state_ = 1058;
                    return true;
                }
                break;
            case 1058:
                if ('s' == current) {
                    state_ = 1059;
                    return true;
                }
                break;
            case 1059:
                if ('i' == current) {
                    state_ = 1060;
                    return true;
                }
                break;
            case 1060:
                if ('b' == current) {
                    state_ = 1061;
                    return true;
                }
                break;
            case 1061:
                if ('l' == current) {
                    state_ = 1062;
                    return true;
                }
                break;
            case 1062:
                if ('e' == current) {
                    state_ = 1063;
                    return true;
                }
                break;
            case 1063:
                if ('C' == current) {
                    state_ = 1064;
                    return true;
                }
                else if ('T' == current) {
                    state_ = 1069;
                    return true;
                }
                break;
            case 1064:
                if ('o' == current) {
                    state_ = 1065;
                    return true;
                }
                break;
            case 1065:
                if ('m' == current) {
                    state_ = 1066;
                    return true;
                }
                break;
            case 1066:
                if ('m' == current) {
                    state_ = 1067;
                    return true;
                }
                break;
            case 1067:
                if ('a' == current) {
                    state_ = 1068;
                    return true;
                }
                break;
            case 1068:
                // InvisibleComma;
                if (';' == current) {
                    match_ = "\u2063";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1069:
                if ('i' == current) {
                    state_ = 1070;
                    return true;
                }
                break;
            case 1070:
                if ('m' == current) {
                    state_ = 1071;
                    return true;
                }
                break;
            case 1071:
                if ('e' == current) {
                    state_ = 1072;
                    return true;
                }
                break;
            case 1072:
                if ('s' == current) {
                    state_ = 1073;
                    return true;
                }
                break;
            case 1073:
                // InvisibleTimes;
                if (';' == current) {
                    match_ = "\u2062";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1074:
                switch (current) {
                    case 'g':
                        state_ = 1075;
                        return true;
                    case 'p':
                        state_ = 1078;
                        return true;
                    case 't':
                        state_ = 1080;
                        return true;
                }
                break;
            case 1075:
                if ('o' == current) {
                    state_ = 1076;
                    return true;
                }
                break;
            case 1076:
                if ('n' == current) {
                    state_ = 1077;
                    return true;
                }
                break;
            case 1077:
                // Iogon;
                if (';' == current) {
                    match_ = "\u012E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1078:
                if ('f' == current) {
                    state_ = 1079;
                    return true;
                }
                break;
            case 1079:
                // Iopf;
                if (';' == current) {
                    match_ = "\uD835\uDD40";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1080:
                if ('a' == current) {
                    state_ = 1081;
                    return true;
                }
                break;
            case 1081:
                // Iota;
                if (';' == current) {
                    match_ = "\u0399";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1082:
                if ('c' == current) {
                    state_ = 1083;
                    return true;
                }
                break;
            case 1083:
                if ('r' == current) {
                    state_ = 1084;
                    return true;
                }
                break;
            case 1084:
                // Iscr;
                if (';' == current) {
                    match_ = "\u2110";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1085:
                if ('i' == current) {
                    state_ = 1086;
                    return true;
                }
                break;
            case 1086:
                if ('l' == current) {
                    state_ = 1087;
                    return true;
                }
                break;
            case 1087:
                if ('d' == current) {
                    state_ = 1088;
                    return true;
                }
                break;
            case 1088:
                if ('e' == current) {
                    state_ = 1089;
                    return true;
                }
                break;
            case 1089:
                // Itilde;
                if (';' == current) {
                    match_ = "\u0128";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1090:
                if ('k' == current) {
                    state_ = 1091;
                    return true;
                }
                else if ('m' == current) {
                    state_ = 1094;
                    return true;
                }
                break;
            case 1091:
                if ('c' == current) {
                    state_ = 1092;
                    return true;
                }
                break;
            case 1092:
                if ('y' == current) {
                    state_ = 1093;
                    return true;
                }
                break;
            case 1093:
                // Iukcy;
                if (';' == current) {
                    match_ = "\u0406";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1094:
                // Iuml
                if ('l' == current) {
                    match_ = "\u00CF";
                    matchLength_ = consumedCount_;
                    state_ = 1095;
                    return true;
                }
                break;
            case 1095:
                // Iuml;
                if (';' == current) {
                    match_ = "\u00CF";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1096:
                switch (current) {
                    case 'c':
                        state_ = 1097;
                        return true;
                    case 'f':
                        state_ = 1102;
                        return true;
                    case 'o':
                        state_ = 1104;
                        return true;
                    case 's':
                        state_ = 1107;
                        return true;
                    case 'u':
                        state_ = 1114;
                        return true;
                }
                break;
            case 1097:
                if ('i' == current) {
                    state_ = 1098;
                    return true;
                }
                else if ('y' == current) {
                    state_ = 1101;
                    return true;
                }
                break;
            case 1098:
                if ('r' == current) {
                    state_ = 1099;
                    return true;
                }
                break;
            case 1099:
                if ('c' == current) {
                    state_ = 1100;
                    return true;
                }
                break;
            case 1100:
                // Jcirc;
                if (';' == current) {
                    match_ = "\u0134";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1101:
                // Jcy;
                if (';' == current) {
                    match_ = "\u0419";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1102:
                if ('r' == current) {
                    state_ = 1103;
                    return true;
                }
                break;
            case 1103:
                // Jfr;
                if (';' == current) {
                    match_ = "\uD835\uDD0D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1104:
                if ('p' == current) {
                    state_ = 1105;
                    return true;
                }
                break;
            case 1105:
                if ('f' == current) {
                    state_ = 1106;
                    return true;
                }
                break;
            case 1106:
                // Jopf;
                if (';' == current) {
                    match_ = "\uD835\uDD41";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1107:
                if ('c' == current) {
                    state_ = 1108;
                    return true;
                }
                else if ('e' == current) {
                    state_ = 1110;
                    return true;
                }
                break;
            case 1108:
                if ('r' == current) {
                    state_ = 1109;
                    return true;
                }
                break;
            case 1109:
                // Jscr;
                if (';' == current) {
                    match_ = "\uD835\uDCA5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1110:
                if ('r' == current) {
                    state_ = 1111;
                    return true;
                }
                break;
            case 1111:
                if ('c' == current) {
                    state_ = 1112;
                    return true;
                }
                break;
            case 1112:
                if ('y' == current) {
                    state_ = 1113;
                    return true;
                }
                break;
            case 1113:
                // Jsercy;
                if (';' == current) {
                    match_ = "\u0408";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1114:
                if ('k' == current) {
                    state_ = 1115;
                    return true;
                }
                break;
            case 1115:
                if ('c' == current) {
                    state_ = 1116;
                    return true;
                }
                break;
            case 1116:
                if ('y' == current) {
                    state_ = 1117;
                    return true;
                }
                break;
            case 1117:
                // Jukcy;
                if (';' == current) {
                    match_ = "\u0404";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1118:
                switch (current) {
                    case 'H':
                        state_ = 1119;
                        return true;
                    case 'J':
                        state_ = 1122;
                        return true;
                    case 'a':
                        state_ = 1125;
                        return true;
                    case 'c':
                        state_ = 1129;
                        return true;
                    case 'f':
                        state_ = 1135;
                        return true;
                    case 'o':
                        state_ = 1137;
                        return true;
                    case 's':
                        state_ = 1140;
                        return true;
                }
                break;
            case 1119:
                if ('c' == current) {
                    state_ = 1120;
                    return true;
                }
                break;
            case 1120:
                if ('y' == current) {
                    state_ = 1121;
                    return true;
                }
                break;
            case 1121:
                // KHcy;
                if (';' == current) {
                    match_ = "\u0425";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1122:
                if ('c' == current) {
                    state_ = 1123;
                    return true;
                }
                break;
            case 1123:
                if ('y' == current) {
                    state_ = 1124;
                    return true;
                }
                break;
            case 1124:
                // KJcy;
                if (';' == current) {
                    match_ = "\u040C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1125:
                if ('p' == current) {
                    state_ = 1126;
                    return true;
                }
                break;
            case 1126:
                if ('p' == current) {
                    state_ = 1127;
                    return true;
                }
                break;
            case 1127:
                if ('a' == current) {
                    state_ = 1128;
                    return true;
                }
                break;
            case 1128:
                // Kappa;
                if (';' == current) {
                    match_ = "\u039A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1129:
                if ('e' == current) {
                    state_ = 1130;
                    return true;
                }
                else if ('y' == current) {
                    state_ = 1134;
                    return true;
                }
                break;
            case 1130:
                if ('d' == current) {
                    state_ = 1131;
                    return true;
                }
                break;
            case 1131:
                if ('i' == current) {
                    state_ = 1132;
                    return true;
                }
                break;
            case 1132:
                if ('l' == current) {
                    state_ = 1133;
                    return true;
                }
                break;
            case 1133:
                // Kcedil;
                if (';' == current) {
                    match_ = "\u0136";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1134:
                // Kcy;
                if (';' == current) {
                    match_ = "\u041A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1135:
                if ('r' == current) {
                    state_ = 1136;
                    return true;
                }
                break;
            case 1136:
                // Kfr;
                if (';' == current) {
                    match_ = "\uD835\uDD0E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1137:
                if ('p' == current) {
                    state_ = 1138;
                    return true;
                }
                break;
            case 1138:
                if ('f' == current) {
                    state_ = 1139;
                    return true;
                }
                break;
            case 1139:
                // Kopf;
                if (';' == current) {
                    match_ = "\uD835\uDD42";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1140:
                if ('c' == current) {
                    state_ = 1141;
                    return true;
                }
                break;
            case 1141:
                if ('r' == current) {
                    state_ = 1142;
                    return true;
                }
                break;
            case 1142:
                // Kscr;
                if (';' == current) {
                    match_ = "\uD835\uDCA6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1143:
                switch (current) {
                    case 'J':
                        state_ = 1144;
                        return true;
                    case 'T': // LT
                        match_ = "<";
                        matchLength_ = consumedCount_;
                        state_ = 1147;
                        return true;
                    case 'a':
                        state_ = 1148;
                        return true;
                    case 'c':
                        state_ = 1169;
                        return true;
                    case 'e':
                        state_ = 1179;
                        return true;
                    case 'f':
                        state_ = 1404;
                        return true;
                    case 'l':
                        state_ = 1406;
                        return true;
                    case 'm':
                        state_ = 1415;
                        return true;
                    case 'o':
                        state_ = 1420;
                        return true;
                    case 's':
                        state_ = 1505;
                        return true;
                    case 't':
                        state_ = 1513;
                        return true;
                }
                break;
            case 1144:
                if ('c' == current) {
                    state_ = 1145;
                    return true;
                }
                break;
            case 1145:
                if ('y' == current) {
                    state_ = 1146;
                    return true;
                }
                break;
            case 1146:
                // LJcy;
                if (';' == current) {
                    match_ = "\u0409";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1147:
                // LT;
                if (';' == current) {
                    match_ = "<";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1148:
                switch (current) {
                    case 'c':
                        state_ = 1149;
                        return true;
                    case 'm':
                        state_ = 1153;
                        return true;
                    case 'n':
                        state_ = 1157;
                        return true;
                    case 'p':
                        state_ = 1159;
                        return true;
                    case 'r':
                        state_ = 1167;
                        return true;
                }
                break;
            case 1149:
                if ('u' == current) {
                    state_ = 1150;
                    return true;
                }
                break;
            case 1150:
                if ('t' == current) {
                    state_ = 1151;
                    return true;
                }
                break;
            case 1151:
                if ('e' == current) {
                    state_ = 1152;
                    return true;
                }
                break;
            case 1152:
                // Lacute;
                if (';' == current) {
                    match_ = "\u0139";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1153:
                if ('b' == current) {
                    state_ = 1154;
                    return true;
                }
                break;
            case 1154:
                if ('d' == current) {
                    state_ = 1155;
                    return true;
                }
                break;
            case 1155:
                if ('a' == current) {
                    state_ = 1156;
                    return true;
                }
                break;
            case 1156:
                // Lambda;
                if (';' == current) {
                    match_ = "\u039B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1157:
                if ('g' == current) {
                    state_ = 1158;
                    return true;
                }
                break;
            case 1158:
                // Lang;
                if (';' == current) {
                    match_ = "\u27EA";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1159:
                if ('l' == current) {
                    state_ = 1160;
                    return true;
                }
                break;
            case 1160:
                if ('a' == current) {
                    state_ = 1161;
                    return true;
                }
                break;
            case 1161:
                if ('c' == current) {
                    state_ = 1162;
                    return true;
                }
                break;
            case 1162:
                if ('e' == current) {
                    state_ = 1163;
                    return true;
                }
                break;
            case 1163:
                if ('t' == current) {
                    state_ = 1164;
                    return true;
                }
                break;
            case 1164:
                if ('r' == current) {
                    state_ = 1165;
                    return true;
                }
                break;
            case 1165:
                if ('f' == current) {
                    state_ = 1166;
                    return true;
                }
                break;
            case 1166:
                // Laplacetrf;
                if (';' == current) {
                    match_ = "\u2112";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1167:
                if ('r' == current) {
                    state_ = 1168;
                    return true;
                }
                break;
            case 1168:
                // Larr;
                if (';' == current) {
                    match_ = "\u219E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1169:
                switch (current) {
                    case 'a':
                        state_ = 1170;
                        return true;
                    case 'e':
                        state_ = 1174;
                        return true;
                    case 'y':
                        state_ = 1178;
                        return true;
                }
                break;
            case 1170:
                if ('r' == current) {
                    state_ = 1171;
                    return true;
                }
                break;
            case 1171:
                if ('o' == current) {
                    state_ = 1172;
                    return true;
                }
                break;
            case 1172:
                if ('n' == current) {
                    state_ = 1173;
                    return true;
                }
                break;
            case 1173:
                // Lcaron;
                if (';' == current) {
                    match_ = "\u013D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1174:
                if ('d' == current) {
                    state_ = 1175;
                    return true;
                }
                break;
            case 1175:
                if ('i' == current) {
                    state_ = 1176;
                    return true;
                }
                break;
            case 1176:
                if ('l' == current) {
                    state_ = 1177;
                    return true;
                }
                break;
            case 1177:
                // Lcedil;
                if (';' == current) {
                    match_ = "\u013B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1178:
                // Lcy;
                if (';' == current) {
                    match_ = "\u041B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1179:
                if ('f' == current) {
                    state_ = 1180;
                    return true;
                }
                else if ('s' == current) {
                    state_ = 1355;
                    return true;
                }
                break;
            case 1180:
                if ('t' == current) {
                    state_ = 1181;
                    return true;
                }
                break;
            case 1181:
                switch (current) {
                    case 'A':
                        state_ = 1182;
                        return true;
                    case 'C':
                        state_ = 1211;
                        return true;
                    case 'D':
                        state_ = 1218;
                        return true;
                    case 'F':
                        state_ = 1251;
                        return true;
                    case 'R':
                        state_ = 1256;
                        return true;
                    case 'T':
                        state_ = 1272;
                        return true;
                    case 'U':
                        state_ = 1301;
                        return true;
                    case 'V':
                        state_ = 1331;
                        return true;
                    case 'a':
                        state_ = 1340;
                        return true;
                    case 'r':
                        state_ = 1345;
                        return true;
                }
                break;
            case 1182:
                if ('n' == current) {
                    state_ = 1183;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 1194;
                    return true;
                }
                break;
            case 1183:
                if ('g' == current) {
                    state_ = 1184;
                    return true;
                }
                break;
            case 1184:
                if ('l' == current) {
                    state_ = 1185;
                    return true;
                }
                break;
            case 1185:
                if ('e' == current) {
                    state_ = 1186;
                    return true;
                }
                break;
            case 1186:
                if ('B' == current) {
                    state_ = 1187;
                    return true;
                }
                break;
            case 1187:
                if ('r' == current) {
                    state_ = 1188;
                    return true;
                }
                break;
            case 1188:
                if ('a' == current) {
                    state_ = 1189;
                    return true;
                }
                break;
            case 1189:
                if ('c' == current) {
                    state_ = 1190;
                    return true;
                }
                break;
            case 1190:
                if ('k' == current) {
                    state_ = 1191;
                    return true;
                }
                break;
            case 1191:
                if ('e' == current) {
                    state_ = 1192;
                    return true;
                }
                break;
            case 1192:
                if ('t' == current) {
                    state_ = 1193;
                    return true;
                }
                break;
            case 1193:
                // LeftAngleBracket;
                if (';' == current) {
                    match_ = "\u27E8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1194:
                if ('r' == current) {
                    state_ = 1195;
                    return true;
                }
                break;
            case 1195:
                if ('o' == current) {
                    state_ = 1196;
                    return true;
                }
                break;
            case 1196:
                if ('w' == current) {
                    state_ = 1197;
                    return true;
                }
                break;
            case 1197:
                switch (current) {
                    case ';': // LeftArrow;
                        match_ = "\u2190";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B':
                        state_ = 1198;
                        return true;
                    case 'R':
                        state_ = 1201;
                        return true;
                }
                break;
            case 1198:
                if ('a' == current) {
                    state_ = 1199;
                    return true;
                }
                break;
            case 1199:
                if ('r' == current) {
                    state_ = 1200;
                    return true;
                }
                break;
            case 1200:
                // LeftArrowBar;
                if (';' == current) {
                    match_ = "\u21E4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1201:
                if ('i' == current) {
                    state_ = 1202;
                    return true;
                }
                break;
            case 1202:
                if ('g' == current) {
                    state_ = 1203;
                    return true;
                }
                break;
            case 1203:
                if ('h' == current) {
                    state_ = 1204;
                    return true;
                }
                break;
            case 1204:
                if ('t' == current) {
                    state_ = 1205;
                    return true;
                }
                break;
            case 1205:
                if ('A' == current) {
                    state_ = 1206;
                    return true;
                }
                break;
            case 1206:
                if ('r' == current) {
                    state_ = 1207;
                    return true;
                }
                break;
            case 1207:
                if ('r' == current) {
                    state_ = 1208;
                    return true;
                }
                break;
            case 1208:
                if ('o' == current) {
                    state_ = 1209;
                    return true;
                }
                break;
            case 1209:
                if ('w' == current) {
                    state_ = 1210;
                    return true;
                }
                break;
            case 1210:
                // LeftArrowRightArrow;
                if (';' == current) {
                    match_ = "\u21C6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1211:
                if ('e' == current) {
                    state_ = 1212;
                    return true;
                }
                break;
            case 1212:
                if ('i' == current) {
                    state_ = 1213;
                    return true;
                }
                break;
            case 1213:
                if ('l' == current) {
                    state_ = 1214;
                    return true;
                }
                break;
            case 1214:
                if ('i' == current) {
                    state_ = 1215;
                    return true;
                }
                break;
            case 1215:
                if ('n' == current) {
                    state_ = 1216;
                    return true;
                }
                break;
            case 1216:
                if ('g' == current) {
                    state_ = 1217;
                    return true;
                }
                break;
            case 1217:
                // LeftCeiling;
                if (';' == current) {
                    match_ = "\u2308";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1218:
                if ('o' == current) {
                    state_ = 1219;
                    return true;
                }
                break;
            case 1219:
                if ('u' == current) {
                    state_ = 1220;
                    return true;
                }
                else if ('w' == current) {
                    state_ = 1231;
                    return true;
                }
                break;
            case 1220:
                if ('b' == current) {
                    state_ = 1221;
                    return true;
                }
                break;
            case 1221:
                if ('l' == current) {
                    state_ = 1222;
                    return true;
                }
                break;
            case 1222:
                if ('e' == current) {
                    state_ = 1223;
                    return true;
                }
                break;
            case 1223:
                if ('B' == current) {
                    state_ = 1224;
                    return true;
                }
                break;
            case 1224:
                if ('r' == current) {
                    state_ = 1225;
                    return true;
                }
                break;
            case 1225:
                if ('a' == current) {
                    state_ = 1226;
                    return true;
                }
                break;
            case 1226:
                if ('c' == current) {
                    state_ = 1227;
                    return true;
                }
                break;
            case 1227:
                if ('k' == current) {
                    state_ = 1228;
                    return true;
                }
                break;
            case 1228:
                if ('e' == current) {
                    state_ = 1229;
                    return true;
                }
                break;
            case 1229:
                if ('t' == current) {
                    state_ = 1230;
                    return true;
                }
                break;
            case 1230:
                // LeftDoubleBracket;
                if (';' == current) {
                    match_ = "\u27E6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1231:
                if ('n' == current) {
                    state_ = 1232;
                    return true;
                }
                break;
            case 1232:
                if ('T' == current) {
                    state_ = 1233;
                    return true;
                }
                else if ('V' == current) {
                    state_ = 1242;
                    return true;
                }
                break;
            case 1233:
                if ('e' == current) {
                    state_ = 1234;
                    return true;
                }
                break;
            case 1234:
                if ('e' == current) {
                    state_ = 1235;
                    return true;
                }
                break;
            case 1235:
                if ('V' == current) {
                    state_ = 1236;
                    return true;
                }
                break;
            case 1236:
                if ('e' == current) {
                    state_ = 1237;
                    return true;
                }
                break;
            case 1237:
                if ('c' == current) {
                    state_ = 1238;
                    return true;
                }
                break;
            case 1238:
                if ('t' == current) {
                    state_ = 1239;
                    return true;
                }
                break;
            case 1239:
                if ('o' == current) {
                    state_ = 1240;
                    return true;
                }
                break;
            case 1240:
                if ('r' == current) {
                    state_ = 1241;
                    return true;
                }
                break;
            case 1241:
                // LeftDownTeeVector;
                if (';' == current) {
                    match_ = "\u2961";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1242:
                if ('e' == current) {
                    state_ = 1243;
                    return true;
                }
                break;
            case 1243:
                if ('c' == current) {
                    state_ = 1244;
                    return true;
                }
                break;
            case 1244:
                if ('t' == current) {
                    state_ = 1245;
                    return true;
                }
                break;
            case 1245:
                if ('o' == current) {
                    state_ = 1246;
                    return true;
                }
                break;
            case 1246:
                if ('r' == current) {
                    state_ = 1247;
                    return true;
                }
                break;
            case 1247:
                // LeftDownVector;
                if (';' == current) {
                    match_ = "\u21C3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('B' == current) {
                    state_ = 1248;
                    return true;
                }
                break;
            case 1248:
                if ('a' == current) {
                    state_ = 1249;
                    return true;
                }
                break;
            case 1249:
                if ('r' == current) {
                    state_ = 1250;
                    return true;
                }
                break;
            case 1250:
                // LeftDownVectorBar;
                if (';' == current) {
                    match_ = "\u2959";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1251:
                if ('l' == current) {
                    state_ = 1252;
                    return true;
                }
                break;
            case 1252:
                if ('o' == current) {
                    state_ = 1253;
                    return true;
                }
                break;
            case 1253:
                if ('o' == current) {
                    state_ = 1254;
                    return true;
                }
                break;
            case 1254:
                if ('r' == current) {
                    state_ = 1255;
                    return true;
                }
                break;
            case 1255:
                // LeftFloor;
                if (';' == current) {
                    match_ = "\u230A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1256:
                if ('i' == current) {
                    state_ = 1257;
                    return true;
                }
                break;
            case 1257:
                if ('g' == current) {
                    state_ = 1258;
                    return true;
                }
                break;
            case 1258:
                if ('h' == current) {
                    state_ = 1259;
                    return true;
                }
                break;
            case 1259:
                if ('t' == current) {
                    state_ = 1260;
                    return true;
                }
                break;
            case 1260:
                if ('A' == current) {
                    state_ = 1261;
                    return true;
                }
                else if ('V' == current) {
                    state_ = 1266;
                    return true;
                }
                break;
            case 1261:
                if ('r' == current) {
                    state_ = 1262;
                    return true;
                }
                break;
            case 1262:
                if ('r' == current) {
                    state_ = 1263;
                    return true;
                }
                break;
            case 1263:
                if ('o' == current) {
                    state_ = 1264;
                    return true;
                }
                break;
            case 1264:
                if ('w' == current) {
                    state_ = 1265;
                    return true;
                }
                break;
            case 1265:
                // LeftRightArrow;
                if (';' == current) {
                    match_ = "\u2194";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1266:
                if ('e' == current) {
                    state_ = 1267;
                    return true;
                }
                break;
            case 1267:
                if ('c' == current) {
                    state_ = 1268;
                    return true;
                }
                break;
            case 1268:
                if ('t' == current) {
                    state_ = 1269;
                    return true;
                }
                break;
            case 1269:
                if ('o' == current) {
                    state_ = 1270;
                    return true;
                }
                break;
            case 1270:
                if ('r' == current) {
                    state_ = 1271;
                    return true;
                }
                break;
            case 1271:
                // LeftRightVector;
                if (';' == current) {
                    match_ = "\u294E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1272:
                if ('e' == current) {
                    state_ = 1273;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 1286;
                    return true;
                }
                break;
            case 1273:
                if ('e' == current) {
                    state_ = 1274;
                    return true;
                }
                break;
            case 1274:
                switch (current) {
                    case ';': // LeftTee;
                        match_ = "\u22A3";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'A':
                        state_ = 1275;
                        return true;
                    case 'V':
                        state_ = 1280;
                        return true;
                }
                break;
            case 1275:
                if ('r' == current) {
                    state_ = 1276;
                    return true;
                }
                break;
            case 1276:
                if ('r' == current) {
                    state_ = 1277;
                    return true;
                }
                break;
            case 1277:
                if ('o' == current) {
                    state_ = 1278;
                    return true;
                }
                break;
            case 1278:
                if ('w' == current) {
                    state_ = 1279;
                    return true;
                }
                break;
            case 1279:
                // LeftTeeArrow;
                if (';' == current) {
                    match_ = "\u21A4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1280:
                if ('e' == current) {
                    state_ = 1281;
                    return true;
                }
                break;
            case 1281:
                if ('c' == current) {
                    state_ = 1282;
                    return true;
                }
                break;
            case 1282:
                if ('t' == current) {
                    state_ = 1283;
                    return true;
                }
                break;
            case 1283:
                if ('o' == current) {
                    state_ = 1284;
                    return true;
                }
                break;
            case 1284:
                if ('r' == current) {
                    state_ = 1285;
                    return true;
                }
                break;
            case 1285:
                // LeftTeeVector;
                if (';' == current) {
                    match_ = "\u295A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1286:
                if ('i' == current) {
                    state_ = 1287;
                    return true;
                }
                break;
            case 1287:
                if ('a' == current) {
                    state_ = 1288;
                    return true;
                }
                break;
            case 1288:
                if ('n' == current) {
                    state_ = 1289;
                    return true;
                }
                break;
            case 1289:
                if ('g' == current) {
                    state_ = 1290;
                    return true;
                }
                break;
            case 1290:
                if ('l' == current) {
                    state_ = 1291;
                    return true;
                }
                break;
            case 1291:
                if ('e' == current) {
                    state_ = 1292;
                    return true;
                }
                break;
            case 1292:
                switch (current) {
                    case ';': // LeftTriangle;
                        match_ = "\u22B2";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B':
                        state_ = 1293;
                        return true;
                    case 'E':
                        state_ = 1296;
                        return true;
                }
                break;
            case 1293:
                if ('a' == current) {
                    state_ = 1294;
                    return true;
                }
                break;
            case 1294:
                if ('r' == current) {
                    state_ = 1295;
                    return true;
                }
                break;
            case 1295:
                // LeftTriangleBar;
                if (';' == current) {
                    match_ = "\u29CF";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1296:
                if ('q' == current) {
                    state_ = 1297;
                    return true;
                }
                break;
            case 1297:
                if ('u' == current) {
                    state_ = 1298;
                    return true;
                }
                break;
            case 1298:
                if ('a' == current) {
                    state_ = 1299;
                    return true;
                }
                break;
            case 1299:
                if ('l' == current) {
                    state_ = 1300;
                    return true;
                }
                break;
            case 1300:
                // LeftTriangleEqual;
                if (';' == current) {
                    match_ = "\u22B4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1301:
                if ('p' == current) {
                    state_ = 1302;
                    return true;
                }
                break;
            case 1302:
                switch (current) {
                    case 'D':
                        state_ = 1303;
                        return true;
                    case 'T':
                        state_ = 1313;
                        return true;
                    case 'V':
                        state_ = 1322;
                        return true;
                }
                break;
            case 1303:
                if ('o' == current) {
                    state_ = 1304;
                    return true;
                }
                break;
            case 1304:
                if ('w' == current) {
                    state_ = 1305;
                    return true;
                }
                break;
            case 1305:
                if ('n' == current) {
                    state_ = 1306;
                    return true;
                }
                break;
            case 1306:
                if ('V' == current) {
                    state_ = 1307;
                    return true;
                }
                break;
            case 1307:
                if ('e' == current) {
                    state_ = 1308;
                    return true;
                }
                break;
            case 1308:
                if ('c' == current) {
                    state_ = 1309;
                    return true;
                }
                break;
            case 1309:
                if ('t' == current) {
                    state_ = 1310;
                    return true;
                }
                break;
            case 1310:
                if ('o' == current) {
                    state_ = 1311;
                    return true;
                }
                break;
            case 1311:
                if ('r' == current) {
                    state_ = 1312;
                    return true;
                }
                break;
            case 1312:
                // LeftUpDownVector;
                if (';' == current) {
                    match_ = "\u2951";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1313:
                if ('e' == current) {
                    state_ = 1314;
                    return true;
                }
                break;
            case 1314:
                if ('e' == current) {
                    state_ = 1315;
                    return true;
                }
                break;
            case 1315:
                if ('V' == current) {
                    state_ = 1316;
                    return true;
                }
                break;
            case 1316:
                if ('e' == current) {
                    state_ = 1317;
                    return true;
                }
                break;
            case 1317:
                if ('c' == current) {
                    state_ = 1318;
                    return true;
                }
                break;
            case 1318:
                if ('t' == current) {
                    state_ = 1319;
                    return true;
                }
                break;
            case 1319:
                if ('o' == current) {
                    state_ = 1320;
                    return true;
                }
                break;
            case 1320:
                if ('r' == current) {
                    state_ = 1321;
                    return true;
                }
                break;
            case 1321:
                // LeftUpTeeVector;
                if (';' == current) {
                    match_ = "\u2960";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1322:
                if ('e' == current) {
                    state_ = 1323;
                    return true;
                }
                break;
            case 1323:
                if ('c' == current) {
                    state_ = 1324;
                    return true;
                }
                break;
            case 1324:
                if ('t' == current) {
                    state_ = 1325;
                    return true;
                }
                break;
            case 1325:
                if ('o' == current) {
                    state_ = 1326;
                    return true;
                }
                break;
            case 1326:
                if ('r' == current) {
                    state_ = 1327;
                    return true;
                }
                break;
            case 1327:
                // LeftUpVector;
                if (';' == current) {
                    match_ = "\u21BF";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('B' == current) {
                    state_ = 1328;
                    return true;
                }
                break;
            case 1328:
                if ('a' == current) {
                    state_ = 1329;
                    return true;
                }
                break;
            case 1329:
                if ('r' == current) {
                    state_ = 1330;
                    return true;
                }
                break;
            case 1330:
                // LeftUpVectorBar;
                if (';' == current) {
                    match_ = "\u2958";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1331:
                if ('e' == current) {
                    state_ = 1332;
                    return true;
                }
                break;
            case 1332:
                if ('c' == current) {
                    state_ = 1333;
                    return true;
                }
                break;
            case 1333:
                if ('t' == current) {
                    state_ = 1334;
                    return true;
                }
                break;
            case 1334:
                if ('o' == current) {
                    state_ = 1335;
                    return true;
                }
                break;
            case 1335:
                if ('r' == current) {
                    state_ = 1336;
                    return true;
                }
                break;
            case 1336:
                // LeftVector;
                if (';' == current) {
                    match_ = "\u21BC";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('B' == current) {
                    state_ = 1337;
                    return true;
                }
                break;
            case 1337:
                if ('a' == current) {
                    state_ = 1338;
                    return true;
                }
                break;
            case 1338:
                if ('r' == current) {
                    state_ = 1339;
                    return true;
                }
                break;
            case 1339:
                // LeftVectorBar;
                if (';' == current) {
                    match_ = "\u2952";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1340:
                if ('r' == current) {
                    state_ = 1341;
                    return true;
                }
                break;
            case 1341:
                if ('r' == current) {
                    state_ = 1342;
                    return true;
                }
                break;
            case 1342:
                if ('o' == current) {
                    state_ = 1343;
                    return true;
                }
                break;
            case 1343:
                if ('w' == current) {
                    state_ = 1344;
                    return true;
                }
                break;
            case 1344:
                // Leftarrow;
                if (';' == current) {
                    match_ = "\u21D0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1345:
                if ('i' == current) {
                    state_ = 1346;
                    return true;
                }
                break;
            case 1346:
                if ('g' == current) {
                    state_ = 1347;
                    return true;
                }
                break;
            case 1347:
                if ('h' == current) {
                    state_ = 1348;
                    return true;
                }
                break;
            case 1348:
                if ('t' == current) {
                    state_ = 1349;
                    return true;
                }
                break;
            case 1349:
                if ('a' == current) {
                    state_ = 1350;
                    return true;
                }
                break;
            case 1350:
                if ('r' == current) {
                    state_ = 1351;
                    return true;
                }
                break;
            case 1351:
                if ('r' == current) {
                    state_ = 1352;
                    return true;
                }
                break;
            case 1352:
                if ('o' == current) {
                    state_ = 1353;
                    return true;
                }
                break;
            case 1353:
                if ('w' == current) {
                    state_ = 1354;
                    return true;
                }
                break;
            case 1354:
                // Leftrightarrow;
                if (';' == current) {
                    match_ = "\u21D4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1355:
                if ('s' == current) {
                    state_ = 1356;
                    return true;
                }
                break;
            case 1356:
                switch (current) {
                    case 'E':
                        state_ = 1357;
                        return true;
                    case 'F':
                        state_ = 1369;
                        return true;
                    case 'G':
                        state_ = 1378;
                        return true;
                    case 'L':
                        state_ = 1385;
                        return true;
                    case 'S':
                        state_ = 1389;
                        return true;
                    case 'T':
                        state_ = 1399;
                        return true;
                }
                break;
            case 1357:
                if ('q' == current) {
                    state_ = 1358;
                    return true;
                }
                break;
            case 1358:
                if ('u' == current) {
                    state_ = 1359;
                    return true;
                }
                break;
            case 1359:
                if ('a' == current) {
                    state_ = 1360;
                    return true;
                }
                break;
            case 1360:
                if ('l' == current) {
                    state_ = 1361;
                    return true;
                }
                break;
            case 1361:
                if ('G' == current) {
                    state_ = 1362;
                    return true;
                }
                break;
            case 1362:
                if ('r' == current) {
                    state_ = 1363;
                    return true;
                }
                break;
            case 1363:
                if ('e' == current) {
                    state_ = 1364;
                    return true;
                }
                break;
            case 1364:
                if ('a' == current) {
                    state_ = 1365;
                    return true;
                }
                break;
            case 1365:
                if ('t' == current) {
                    state_ = 1366;
                    return true;
                }
                break;
            case 1366:
                if ('e' == current) {
                    state_ = 1367;
                    return true;
                }
                break;
            case 1367:
                if ('r' == current) {
                    state_ = 1368;
                    return true;
                }
                break;
            case 1368:
                // LessEqualGreater;
                if (';' == current) {
                    match_ = "\u22DA";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1369:
                if ('u' == current) {
                    state_ = 1370;
                    return true;
                }
                break;
            case 1370:
                if ('l' == current) {
                    state_ = 1371;
                    return true;
                }
                break;
            case 1371:
                if ('l' == current) {
                    state_ = 1372;
                    return true;
                }
                break;
            case 1372:
                if ('E' == current) {
                    state_ = 1373;
                    return true;
                }
                break;
            case 1373:
                if ('q' == current) {
                    state_ = 1374;
                    return true;
                }
                break;
            case 1374:
                if ('u' == current) {
                    state_ = 1375;
                    return true;
                }
                break;
            case 1375:
                if ('a' == current) {
                    state_ = 1376;
                    return true;
                }
                break;
            case 1376:
                if ('l' == current) {
                    state_ = 1377;
                    return true;
                }
                break;
            case 1377:
                // LessFullEqual;
                if (';' == current) {
                    match_ = "\u2266";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1378:
                if ('r' == current) {
                    state_ = 1379;
                    return true;
                }
                break;
            case 1379:
                if ('e' == current) {
                    state_ = 1380;
                    return true;
                }
                break;
            case 1380:
                if ('a' == current) {
                    state_ = 1381;
                    return true;
                }
                break;
            case 1381:
                if ('t' == current) {
                    state_ = 1382;
                    return true;
                }
                break;
            case 1382:
                if ('e' == current) {
                    state_ = 1383;
                    return true;
                }
                break;
            case 1383:
                if ('r' == current) {
                    state_ = 1384;
                    return true;
                }
                break;
            case 1384:
                // LessGreater;
                if (';' == current) {
                    match_ = "\u2276";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1385:
                if ('e' == current) {
                    state_ = 1386;
                    return true;
                }
                break;
            case 1386:
                if ('s' == current) {
                    state_ = 1387;
                    return true;
                }
                break;
            case 1387:
                if ('s' == current) {
                    state_ = 1388;
                    return true;
                }
                break;
            case 1388:
                // LessLess;
                if (';' == current) {
                    match_ = "\u2AA1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1389:
                if ('l' == current) {
                    state_ = 1390;
                    return true;
                }
                break;
            case 1390:
                if ('a' == current) {
                    state_ = 1391;
                    return true;
                }
                break;
            case 1391:
                if ('n' == current) {
                    state_ = 1392;
                    return true;
                }
                break;
            case 1392:
                if ('t' == current) {
                    state_ = 1393;
                    return true;
                }
                break;
            case 1393:
                if ('E' == current) {
                    state_ = 1394;
                    return true;
                }
                break;
            case 1394:
                if ('q' == current) {
                    state_ = 1395;
                    return true;
                }
                break;
            case 1395:
                if ('u' == current) {
                    state_ = 1396;
                    return true;
                }
                break;
            case 1396:
                if ('a' == current) {
                    state_ = 1397;
                    return true;
                }
                break;
            case 1397:
                if ('l' == current) {
                    state_ = 1398;
                    return true;
                }
                break;
            case 1398:
                // LessSlantEqual;
                if (';' == current) {
                    match_ = "\u2A7D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1399:
                if ('i' == current) {
                    state_ = 1400;
                    return true;
                }
                break;
            case 1400:
                if ('l' == current) {
                    state_ = 1401;
                    return true;
                }
                break;
            case 1401:
                if ('d' == current) {
                    state_ = 1402;
                    return true;
                }
                break;
            case 1402:
                if ('e' == current) {
                    state_ = 1403;
                    return true;
                }
                break;
            case 1403:
                // LessTilde;
                if (';' == current) {
                    match_ = "\u2272";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1404:
                if ('r' == current) {
                    state_ = 1405;
                    return true;
                }
                break;
            case 1405:
                // Lfr;
                if (';' == current) {
                    match_ = "\uD835\uDD0F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1406:
                // Ll;
                if (';' == current) {
                    match_ = "\u22D8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state_ = 1407;
                    return true;
                }
                break;
            case 1407:
                if ('f' == current) {
                    state_ = 1408;
                    return true;
                }
                break;
            case 1408:
                if ('t' == current) {
                    state_ = 1409;
                    return true;
                }
                break;
            case 1409:
                if ('a' == current) {
                    state_ = 1410;
                    return true;
                }
                break;
            case 1410:
                if ('r' == current) {
                    state_ = 1411;
                    return true;
                }
                break;
            case 1411:
                if ('r' == current) {
                    state_ = 1412;
                    return true;
                }
                break;
            case 1412:
                if ('o' == current) {
                    state_ = 1413;
                    return true;
                }
                break;
            case 1413:
                if ('w' == current) {
                    state_ = 1414;
                    return true;
                }
                break;
            case 1414:
                // Lleftarrow;
                if (';' == current) {
                    match_ = "\u21DA";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1415:
                if ('i' == current) {
                    state_ = 1416;
                    return true;
                }
                break;
            case 1416:
                if ('d' == current) {
                    state_ = 1417;
                    return true;
                }
                break;
            case 1417:
                if ('o' == current) {
                    state_ = 1418;
                    return true;
                }
                break;
            case 1418:
                if ('t' == current) {
                    state_ = 1419;
                    return true;
                }
                break;
            case 1419:
                // Lmidot;
                if (';' == current) {
                    match_ = "\u013F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1420:
                switch (current) {
                    case 'n':
                        state_ = 1421;
                        return true;
                    case 'p':
                        state_ = 1481;
                        return true;
                    case 'w':
                        state_ = 1483;
                        return true;
                }
                break;
            case 1421:
                if ('g' == current) {
                    state_ = 1422;
                    return true;
                }
                break;
            case 1422:
                switch (current) {
                    case 'L':
                        state_ = 1423;
                        return true;
                    case 'R':
                        state_ = 1442;
                        return true;
                    case 'l':
                        state_ = 1452;
                        return true;
                    case 'r':
                        state_ = 1471;
                        return true;
                }
                break;
            case 1423:
                if ('e' == current) {
                    state_ = 1424;
                    return true;
                }
                break;
            case 1424:
                if ('f' == current) {
                    state_ = 1425;
                    return true;
                }
                break;
            case 1425:
                if ('t' == current) {
                    state_ = 1426;
                    return true;
                }
                break;
            case 1426:
                if ('A' == current) {
                    state_ = 1427;
                    return true;
                }
                else if ('R' == current) {
                    state_ = 1432;
                    return true;
                }
                break;
            case 1427:
                if ('r' == current) {
                    state_ = 1428;
                    return true;
                }
                break;
            case 1428:
                if ('r' == current) {
                    state_ = 1429;
                    return true;
                }
                break;
            case 1429:
                if ('o' == current) {
                    state_ = 1430;
                    return true;
                }
                break;
            case 1430:
                if ('w' == current) {
                    state_ = 1431;
                    return true;
                }
                break;
            case 1431:
                // LongLeftArrow;
                if (';' == current) {
                    match_ = "\u27F5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1432:
                if ('i' == current) {
                    state_ = 1433;
                    return true;
                }
                break;
            case 1433:
                if ('g' == current) {
                    state_ = 1434;
                    return true;
                }
                break;
            case 1434:
                if ('h' == current) {
                    state_ = 1435;
                    return true;
                }
                break;
            case 1435:
                if ('t' == current) {
                    state_ = 1436;
                    return true;
                }
                break;
            case 1436:
                if ('A' == current) {
                    state_ = 1437;
                    return true;
                }
                break;
            case 1437:
                if ('r' == current) {
                    state_ = 1438;
                    return true;
                }
                break;
            case 1438:
                if ('r' == current) {
                    state_ = 1439;
                    return true;
                }
                break;
            case 1439:
                if ('o' == current) {
                    state_ = 1440;
                    return true;
                }
                break;
            case 1440:
                if ('w' == current) {
                    state_ = 1441;
                    return true;
                }
                break;
            case 1441:
                // LongLeftRightArrow;
                if (';' == current) {
                    match_ = "\u27F7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1442:
                if ('i' == current) {
                    state_ = 1443;
                    return true;
                }
                break;
            case 1443:
                if ('g' == current) {
                    state_ = 1444;
                    return true;
                }
                break;
            case 1444:
                if ('h' == current) {
                    state_ = 1445;
                    return true;
                }
                break;
            case 1445:
                if ('t' == current) {
                    state_ = 1446;
                    return true;
                }
                break;
            case 1446:
                if ('A' == current) {
                    state_ = 1447;
                    return true;
                }
                break;
            case 1447:
                if ('r' == current) {
                    state_ = 1448;
                    return true;
                }
                break;
            case 1448:
                if ('r' == current) {
                    state_ = 1449;
                    return true;
                }
                break;
            case 1449:
                if ('o' == current) {
                    state_ = 1450;
                    return true;
                }
                break;
            case 1450:
                if ('w' == current) {
                    state_ = 1451;
                    return true;
                }
                break;
            case 1451:
                // LongRightArrow;
                if (';' == current) {
                    match_ = "\u27F6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1452:
                if ('e' == current) {
                    state_ = 1453;
                    return true;
                }
                break;
            case 1453:
                if ('f' == current) {
                    state_ = 1454;
                    return true;
                }
                break;
            case 1454:
                if ('t' == current) {
                    state_ = 1455;
                    return true;
                }
                break;
            case 1455:
                if ('a' == current) {
                    state_ = 1456;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 1461;
                    return true;
                }
                break;
            case 1456:
                if ('r' == current) {
                    state_ = 1457;
                    return true;
                }
                break;
            case 1457:
                if ('r' == current) {
                    state_ = 1458;
                    return true;
                }
                break;
            case 1458:
                if ('o' == current) {
                    state_ = 1459;
                    return true;
                }
                break;
            case 1459:
                if ('w' == current) {
                    state_ = 1460;
                    return true;
                }
                break;
            case 1460:
                // Longleftarrow;
                if (';' == current) {
                    match_ = "\u27F8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1461:
                if ('i' == current) {
                    state_ = 1462;
                    return true;
                }
                break;
            case 1462:
                if ('g' == current) {
                    state_ = 1463;
                    return true;
                }
                break;
            case 1463:
                if ('h' == current) {
                    state_ = 1464;
                    return true;
                }
                break;
            case 1464:
                if ('t' == current) {
                    state_ = 1465;
                    return true;
                }
                break;
            case 1465:
                if ('a' == current) {
                    state_ = 1466;
                    return true;
                }
                break;
            case 1466:
                if ('r' == current) {
                    state_ = 1467;
                    return true;
                }
                break;
            case 1467:
                if ('r' == current) {
                    state_ = 1468;
                    return true;
                }
                break;
            case 1468:
                if ('o' == current) {
                    state_ = 1469;
                    return true;
                }
                break;
            case 1469:
                if ('w' == current) {
                    state_ = 1470;
                    return true;
                }
                break;
            case 1470:
                // Longleftrightarrow;
                if (';' == current) {
                    match_ = "\u27FA";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1471:
                if ('i' == current) {
                    state_ = 1472;
                    return true;
                }
                break;
            case 1472:
                if ('g' == current) {
                    state_ = 1473;
                    return true;
                }
                break;
            case 1473:
                if ('h' == current) {
                    state_ = 1474;
                    return true;
                }
                break;
            case 1474:
                if ('t' == current) {
                    state_ = 1475;
                    return true;
                }
                break;
            case 1475:
                if ('a' == current) {
                    state_ = 1476;
                    return true;
                }
                break;
            case 1476:
                if ('r' == current) {
                    state_ = 1477;
                    return true;
                }
                break;
            case 1477:
                if ('r' == current) {
                    state_ = 1478;
                    return true;
                }
                break;
            case 1478:
                if ('o' == current) {
                    state_ = 1479;
                    return true;
                }
                break;
            case 1479:
                if ('w' == current) {
                    state_ = 1480;
                    return true;
                }
                break;
            case 1480:
                // Longrightarrow;
                if (';' == current) {
                    match_ = "\u27F9";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1481:
                if ('f' == current) {
                    state_ = 1482;
                    return true;
                }
                break;
            case 1482:
                // Lopf;
                if (';' == current) {
                    match_ = "\uD835\uDD43";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1483:
                if ('e' == current) {
                    state_ = 1484;
                    return true;
                }
                break;
            case 1484:
                if ('r' == current) {
                    state_ = 1485;
                    return true;
                }
                break;
            case 1485:
                if ('L' == current) {
                    state_ = 1486;
                    return true;
                }
                else if ('R' == current) {
                    state_ = 1495;
                    return true;
                }
                break;
            case 1486:
                if ('e' == current) {
                    state_ = 1487;
                    return true;
                }
                break;
            case 1487:
                if ('f' == current) {
                    state_ = 1488;
                    return true;
                }
                break;
            case 1488:
                if ('t' == current) {
                    state_ = 1489;
                    return true;
                }
                break;
            case 1489:
                if ('A' == current) {
                    state_ = 1490;
                    return true;
                }
                break;
            case 1490:
                if ('r' == current) {
                    state_ = 1491;
                    return true;
                }
                break;
            case 1491:
                if ('r' == current) {
                    state_ = 1492;
                    return true;
                }
                break;
            case 1492:
                if ('o' == current) {
                    state_ = 1493;
                    return true;
                }
                break;
            case 1493:
                if ('w' == current) {
                    state_ = 1494;
                    return true;
                }
                break;
            case 1494:
                // LowerLeftArrow;
                if (';' == current) {
                    match_ = "\u2199";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1495:
                if ('i' == current) {
                    state_ = 1496;
                    return true;
                }
                break;
            case 1496:
                if ('g' == current) {
                    state_ = 1497;
                    return true;
                }
                break;
            case 1497:
                if ('h' == current) {
                    state_ = 1498;
                    return true;
                }
                break;
            case 1498:
                if ('t' == current) {
                    state_ = 1499;
                    return true;
                }
                break;
            case 1499:
                if ('A' == current) {
                    state_ = 1500;
                    return true;
                }
                break;
            case 1500:
                if ('r' == current) {
                    state_ = 1501;
                    return true;
                }
                break;
            case 1501:
                if ('r' == current) {
                    state_ = 1502;
                    return true;
                }
                break;
            case 1502:
                if ('o' == current) {
                    state_ = 1503;
                    return true;
                }
                break;
            case 1503:
                if ('w' == current) {
                    state_ = 1504;
                    return true;
                }
                break;
            case 1504:
                // LowerRightArrow;
                if (';' == current) {
                    match_ = "\u2198";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1505:
                switch (current) {
                    case 'c':
                        state_ = 1506;
                        return true;
                    case 'h':
                        state_ = 1508;
                        return true;
                    case 't':
                        state_ = 1509;
                        return true;
                }
                break;
            case 1506:
                if ('r' == current) {
                    state_ = 1507;
                    return true;
                }
                break;
            case 1507:
                // Lscr;
                if (';' == current) {
                    match_ = "\u2112";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1508:
                // Lsh;
                if (';' == current) {
                    match_ = "\u21B0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1509:
                if ('r' == current) {
                    state_ = 1510;
                    return true;
                }
                break;
            case 1510:
                if ('o' == current) {
                    state_ = 1511;
                    return true;
                }
                break;
            case 1511:
                if ('k' == current) {
                    state_ = 1512;
                    return true;
                }
                break;
            case 1512:
                // Lstrok;
                if (';' == current) {
                    match_ = "\u0141";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1513:
                // Lt;
                if (';' == current) {
                    match_ = "\u226A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1514:
                switch (current) {
                    case 'a':
                        state_ = 1515;
                        return true;
                    case 'c':
                        state_ = 1517;
                        return true;
                    case 'e':
                        state_ = 1519;
                        return true;
                    case 'f':
                        state_ = 1536;
                        return true;
                    case 'i':
                        state_ = 1538;
                        return true;
                    case 'o':
                        state_ = 1546;
                        return true;
                    case 's':
                        state_ = 1549;
                        return true;
                    case 'u':
                        state_ = 1552;
                        return true;
                }
                break;
            case 1515:
                if ('p' == current) {
                    state_ = 1516;
                    return true;
                }
                break;
            case 1516:
                // Map;
                if (';' == current) {
                    match_ = "\u2905";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1517:
                if ('y' == current) {
                    state_ = 1518;
                    return true;
                }
                break;
            case 1518:
                // Mcy;
                if (';' == current) {
                    match_ = "\u041C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1519:
                if ('d' == current) {
                    state_ = 1520;
                    return true;
                }
                else if ('l' == current) {
                    state_ = 1529;
                    return true;
                }
                break;
            case 1520:
                if ('i' == current) {
                    state_ = 1521;
                    return true;
                }
                break;
            case 1521:
                if ('u' == current) {
                    state_ = 1522;
                    return true;
                }
                break;
            case 1522:
                if ('m' == current) {
                    state_ = 1523;
                    return true;
                }
                break;
            case 1523:
                if ('S' == current) {
                    state_ = 1524;
                    return true;
                }
                break;
            case 1524:
                if ('p' == current) {
                    state_ = 1525;
                    return true;
                }
                break;
            case 1525:
                if ('a' == current) {
                    state_ = 1526;
                    return true;
                }
                break;
            case 1526:
                if ('c' == current) {
                    state_ = 1527;
                    return true;
                }
                break;
            case 1527:
                if ('e' == current) {
                    state_ = 1528;
                    return true;
                }
                break;
            case 1528:
                // MediumSpace;
                if (';' == current) {
                    match_ = "\u205F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1529:
                if ('l' == current) {
                    state_ = 1530;
                    return true;
                }
                break;
            case 1530:
                if ('i' == current) {
                    state_ = 1531;
                    return true;
                }
                break;
            case 1531:
                if ('n' == current) {
                    state_ = 1532;
                    return true;
                }
                break;
            case 1532:
                if ('t' == current) {
                    state_ = 1533;
                    return true;
                }
                break;
            case 1533:
                if ('r' == current) {
                    state_ = 1534;
                    return true;
                }
                break;
            case 1534:
                if ('f' == current) {
                    state_ = 1535;
                    return true;
                }
                break;
            case 1535:
                // Mellintrf;
                if (';' == current) {
                    match_ = "\u2133";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1536:
                if ('r' == current) {
                    state_ = 1537;
                    return true;
                }
                break;
            case 1537:
                // Mfr;
                if (';' == current) {
                    match_ = "\uD835\uDD10";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1538:
                if ('n' == current) {
                    state_ = 1539;
                    return true;
                }
                break;
            case 1539:
                if ('u' == current) {
                    state_ = 1540;
                    return true;
                }
                break;
            case 1540:
                if ('s' == current) {
                    state_ = 1541;
                    return true;
                }
                break;
            case 1541:
                if ('P' == current) {
                    state_ = 1542;
                    return true;
                }
                break;
            case 1542:
                if ('l' == current) {
                    state_ = 1543;
                    return true;
                }
                break;
            case 1543:
                if ('u' == current) {
                    state_ = 1544;
                    return true;
                }
                break;
            case 1544:
                if ('s' == current) {
                    state_ = 1545;
                    return true;
                }
                break;
            case 1545:
                // MinusPlus;
                if (';' == current) {
                    match_ = "\u2213";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1546:
                if ('p' == current) {
                    state_ = 1547;
                    return true;
                }
                break;
            case 1547:
                if ('f' == current) {
                    state_ = 1548;
                    return true;
                }
                break;
            case 1548:
                // Mopf;
                if (';' == current) {
                    match_ = "\uD835\uDD44";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1549:
                if ('c' == current) {
                    state_ = 1550;
                    return true;
                }
                break;
            case 1550:
                if ('r' == current) {
                    state_ = 1551;
                    return true;
                }
                break;
            case 1551:
                // Mscr;
                if (';' == current) {
                    match_ = "\u2133";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1552:
                // Mu;
                if (';' == current) {
                    match_ = "\u039C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1553:
                switch (current) {
                    case 'J':
                        state_ = 1554;
                        return true;
                    case 'a':
                        state_ = 1557;
                        return true;
                    case 'c':
                        state_ = 1562;
                        return true;
                    case 'e':
                        state_ = 1572;
                        return true;
                    case 'f':
                        state_ = 1650;
                        return true;
                    case 'o':
                        state_ = 1652;
                        return true;
                    case 's':
                        state_ = 2055;
                        return true;
                    case 't':
                        state_ = 2058;
                        return true;
                    case 'u':
                        state_ = 2063;
                        return true;
                }
                break;
            case 1554:
                if ('c' == current) {
                    state_ = 1555;
                    return true;
                }
                break;
            case 1555:
                if ('y' == current) {
                    state_ = 1556;
                    return true;
                }
                break;
            case 1556:
                // NJcy;
                if (';' == current) {
                    match_ = "\u040A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1557:
                if ('c' == current) {
                    state_ = 1558;
                    return true;
                }
                break;
            case 1558:
                if ('u' == current) {
                    state_ = 1559;
                    return true;
                }
                break;
            case 1559:
                if ('t' == current) {
                    state_ = 1560;
                    return true;
                }
                break;
            case 1560:
                if ('e' == current) {
                    state_ = 1561;
                    return true;
                }
                break;
            case 1561:
                // Nacute;
                if (';' == current) {
                    match_ = "\u0143";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1562:
                switch (current) {
                    case 'a':
                        state_ = 1563;
                        return true;
                    case 'e':
                        state_ = 1567;
                        return true;
                    case 'y':
                        state_ = 1571;
                        return true;
                }
                break;
            case 1563:
                if ('r' == current) {
                    state_ = 1564;
                    return true;
                }
                break;
            case 1564:
                if ('o' == current) {
                    state_ = 1565;
                    return true;
                }
                break;
            case 1565:
                if ('n' == current) {
                    state_ = 1566;
                    return true;
                }
                break;
            case 1566:
                // Ncaron;
                if (';' == current) {
                    match_ = "\u0147";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1567:
                if ('d' == current) {
                    state_ = 1568;
                    return true;
                }
                break;
            case 1568:
                if ('i' == current) {
                    state_ = 1569;
                    return true;
                }
                break;
            case 1569:
                if ('l' == current) {
                    state_ = 1570;
                    return true;
                }
                break;
            case 1570:
                // Ncedil;
                if (';' == current) {
                    match_ = "\u0145";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1571:
                // Ncy;
                if (';' == current) {
                    match_ = "\u041D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1572:
                switch (current) {
                    case 'g':
                        state_ = 1573;
                        return true;
                    case 's':
                        state_ = 1619;
                        return true;
                    case 'w':
                        state_ = 1645;
                        return true;
                }
                break;
            case 1573:
                if ('a' == current) {
                    state_ = 1574;
                    return true;
                }
                break;
            case 1574:
                if ('t' == current) {
                    state_ = 1575;
                    return true;
                }
                break;
            case 1575:
                if ('i' == current) {
                    state_ = 1576;
                    return true;
                }
                break;
            case 1576:
                if ('v' == current) {
                    state_ = 1577;
                    return true;
                }
                break;
            case 1577:
                if ('e' == current) {
                    state_ = 1578;
                    return true;
                }
                break;
            case 1578:
                switch (current) {
                    case 'M':
                        state_ = 1579;
                        return true;
                    case 'T':
                        state_ = 1590;
                        return true;
                    case 'V':
                        state_ = 1606;
                        return true;
                }
                break;
            case 1579:
                if ('e' == current) {
                    state_ = 1580;
                    return true;
                }
                break;
            case 1580:
                if ('d' == current) {
                    state_ = 1581;
                    return true;
                }
                break;
            case 1581:
                if ('i' == current) {
                    state_ = 1582;
                    return true;
                }
                break;
            case 1582:
                if ('u' == current) {
                    state_ = 1583;
                    return true;
                }
                break;
            case 1583:
                if ('m' == current) {
                    state_ = 1584;
                    return true;
                }
                break;
            case 1584:
                if ('S' == current) {
                    state_ = 1585;
                    return true;
                }
                break;
            case 1585:
                if ('p' == current) {
                    state_ = 1586;
                    return true;
                }
                break;
            case 1586:
                if ('a' == current) {
                    state_ = 1587;
                    return true;
                }
                break;
            case 1587:
                if ('c' == current) {
                    state_ = 1588;
                    return true;
                }
                break;
            case 1588:
                if ('e' == current) {
                    state_ = 1589;
                    return true;
                }
                break;
            case 1589:
                // NegativeMediumSpace;
                if (';' == current) {
                    match_ = "\u200B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1590:
                if ('h' == current) {
                    state_ = 1591;
                    return true;
                }
                break;
            case 1591:
                if ('i' == current) {
                    state_ = 1592;
                    return true;
                }
                break;
            case 1592:
                if ('c' == current) {
                    state_ = 1593;
                    return true;
                }
                else if ('n' == current) {
                    state_ = 1600;
                    return true;
                }
                break;
            case 1593:
                if ('k' == current) {
                    state_ = 1594;
                    return true;
                }
                break;
            case 1594:
                if ('S' == current) {
                    state_ = 1595;
                    return true;
                }
                break;
            case 1595:
                if ('p' == current) {
                    state_ = 1596;
                    return true;
                }
                break;
            case 1596:
                if ('a' == current) {
                    state_ = 1597;
                    return true;
                }
                break;
            case 1597:
                if ('c' == current) {
                    state_ = 1598;
                    return true;
                }
                break;
            case 1598:
                if ('e' == current) {
                    state_ = 1599;
                    return true;
                }
                break;
            case 1599:
                // NegativeThickSpace;
                if (';' == current) {
                    match_ = "\u200B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1600:
                if ('S' == current) {
                    state_ = 1601;
                    return true;
                }
                break;
            case 1601:
                if ('p' == current) {
                    state_ = 1602;
                    return true;
                }
                break;
            case 1602:
                if ('a' == current) {
                    state_ = 1603;
                    return true;
                }
                break;
            case 1603:
                if ('c' == current) {
                    state_ = 1604;
                    return true;
                }
                break;
            case 1604:
                if ('e' == current) {
                    state_ = 1605;
                    return true;
                }
                break;
            case 1605:
                // NegativeThinSpace;
                if (';' == current) {
                    match_ = "\u200B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1606:
                if ('e' == current) {
                    state_ = 1607;
                    return true;
                }
                break;
            case 1607:
                if ('r' == current) {
                    state_ = 1608;
                    return true;
                }
                break;
            case 1608:
                if ('y' == current) {
                    state_ = 1609;
                    return true;
                }
                break;
            case 1609:
                if ('T' == current) {
                    state_ = 1610;
                    return true;
                }
                break;
            case 1610:
                if ('h' == current) {
                    state_ = 1611;
                    return true;
                }
                break;
            case 1611:
                if ('i' == current) {
                    state_ = 1612;
                    return true;
                }
                break;
            case 1612:
                if ('n' == current) {
                    state_ = 1613;
                    return true;
                }
                break;
            case 1613:
                if ('S' == current) {
                    state_ = 1614;
                    return true;
                }
                break;
            case 1614:
                if ('p' == current) {
                    state_ = 1615;
                    return true;
                }
                break;
            case 1615:
                if ('a' == current) {
                    state_ = 1616;
                    return true;
                }
                break;
            case 1616:
                if ('c' == current) {
                    state_ = 1617;
                    return true;
                }
                break;
            case 1617:
                if ('e' == current) {
                    state_ = 1618;
                    return true;
                }
                break;
            case 1618:
                // NegativeVeryThinSpace;
                if (';' == current) {
                    match_ = "\u200B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1619:
                if ('t' == current) {
                    state_ = 1620;
                    return true;
                }
                break;
            case 1620:
                if ('e' == current) {
                    state_ = 1621;
                    return true;
                }
                break;
            case 1621:
                if ('d' == current) {
                    state_ = 1622;
                    return true;
                }
                break;
            case 1622:
                if ('G' == current) {
                    state_ = 1623;
                    return true;
                }
                else if ('L' == current) {
                    state_ = 1637;
                    return true;
                }
                break;
            case 1623:
                if ('r' == current) {
                    state_ = 1624;
                    return true;
                }
                break;
            case 1624:
                if ('e' == current) {
                    state_ = 1625;
                    return true;
                }
                break;
            case 1625:
                if ('a' == current) {
                    state_ = 1626;
                    return true;
                }
                break;
            case 1626:
                if ('t' == current) {
                    state_ = 1627;
                    return true;
                }
                break;
            case 1627:
                if ('e' == current) {
                    state_ = 1628;
                    return true;
                }
                break;
            case 1628:
                if ('r' == current) {
                    state_ = 1629;
                    return true;
                }
                break;
            case 1629:
                if ('G' == current) {
                    state_ = 1630;
                    return true;
                }
                break;
            case 1630:
                if ('r' == current) {
                    state_ = 1631;
                    return true;
                }
                break;
            case 1631:
                if ('e' == current) {
                    state_ = 1632;
                    return true;
                }
                break;
            case 1632:
                if ('a' == current) {
                    state_ = 1633;
                    return true;
                }
                break;
            case 1633:
                if ('t' == current) {
                    state_ = 1634;
                    return true;
                }
                break;
            case 1634:
                if ('e' == current) {
                    state_ = 1635;
                    return true;
                }
                break;
            case 1635:
                if ('r' == current) {
                    state_ = 1636;
                    return true;
                }
                break;
            case 1636:
                // NestedGreaterGreater;
                if (';' == current) {
                    match_ = "\u226B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1637:
                if ('e' == current) {
                    state_ = 1638;
                    return true;
                }
                break;
            case 1638:
                if ('s' == current) {
                    state_ = 1639;
                    return true;
                }
                break;
            case 1639:
                if ('s' == current) {
                    state_ = 1640;
                    return true;
                }
                break;
            case 1640:
                if ('L' == current) {
                    state_ = 1641;
                    return true;
                }
                break;
            case 1641:
                if ('e' == current) {
                    state_ = 1642;
                    return true;
                }
                break;
            case 1642:
                if ('s' == current) {
                    state_ = 1643;
                    return true;
                }
                break;
            case 1643:
                if ('s' == current) {
                    state_ = 1644;
                    return true;
                }
                break;
            case 1644:
                // NestedLessLess;
                if (';' == current) {
                    match_ = "\u226A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1645:
                if ('L' == current) {
                    state_ = 1646;
                    return true;
                }
                break;
            case 1646:
                if ('i' == current) {
                    state_ = 1647;
                    return true;
                }
                break;
            case 1647:
                if ('n' == current) {
                    state_ = 1648;
                    return true;
                }
                break;
            case 1648:
                if ('e' == current) {
                    state_ = 1649;
                    return true;
                }
                break;
            case 1649:
                // NewLine;
                if (';' == current) {
                    match_ = "\n";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1650:
                if ('r' == current) {
                    state_ = 1651;
                    return true;
                }
                break;
            case 1651:
                // Nfr;
                if (';' == current) {
                    match_ = "\uD835\uDD11";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1652:
                switch (current) {
                    case 'B':
                        state_ = 1653;
                        return true;
                    case 'n':
                        state_ = 1658;
                        return true;
                    case 'p':
                        state_ = 1672;
                        return true;
                    case 't':
                        state_ = 1674;
                        return true;
                }
                break;
            case 1653:
                if ('r' == current) {
                    state_ = 1654;
                    return true;
                }
                break;
            case 1654:
                if ('e' == current) {
                    state_ = 1655;
                    return true;
                }
                break;
            case 1655:
                if ('a' == current) {
                    state_ = 1656;
                    return true;
                }
                break;
            case 1656:
                if ('k' == current) {
                    state_ = 1657;
                    return true;
                }
                break;
            case 1657:
                // NoBreak;
                if (';' == current) {
                    match_ = "\u2060";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1658:
                if ('B' == current) {
                    state_ = 1659;
                    return true;
                }
                break;
            case 1659:
                if ('r' == current) {
                    state_ = 1660;
                    return true;
                }
                break;
            case 1660:
                if ('e' == current) {
                    state_ = 1661;
                    return true;
                }
                break;
            case 1661:
                if ('a' == current) {
                    state_ = 1662;
                    return true;
                }
                break;
            case 1662:
                if ('k' == current) {
                    state_ = 1663;
                    return true;
                }
                break;
            case 1663:
                if ('i' == current) {
                    state_ = 1664;
                    return true;
                }
                break;
            case 1664:
                if ('n' == current) {
                    state_ = 1665;
                    return true;
                }
                break;
            case 1665:
                if ('g' == current) {
                    state_ = 1666;
                    return true;
                }
                break;
            case 1666:
                if ('S' == current) {
                    state_ = 1667;
                    return true;
                }
                break;
            case 1667:
                if ('p' == current) {
                    state_ = 1668;
                    return true;
                }
                break;
            case 1668:
                if ('a' == current) {
                    state_ = 1669;
                    return true;
                }
                break;
            case 1669:
                if ('c' == current) {
                    state_ = 1670;
                    return true;
                }
                break;
            case 1670:
                if ('e' == current) {
                    state_ = 1671;
                    return true;
                }
                break;
            case 1671:
                // NonBreakingSpace;
                if (';' == current) {
                    match_ = "\u00A0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1672:
                if ('f' == current) {
                    state_ = 1673;
                    return true;
                }
                break;
            case 1673:
                // Nopf;
                if (';' == current) {
                    match_ = "\u2115";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1674:
                switch (current) {
                    case ';': // Not;
                        match_ = "\u2AEC";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'C':
                        state_ = 1675;
                        return true;
                    case 'D':
                        state_ = 1689;
                        return true;
                    case 'E':
                        state_ = 1706;
                        return true;
                    case 'G':
                        state_ = 1727;
                        return true;
                    case 'H':
                        state_ = 1780;
                        return true;
                    case 'L':
                        state_ = 1801;
                        return true;
                    case 'N':
                        state_ = 1860;
                        return true;
                    case 'P':
                        state_ = 1888;
                        return true;
                    case 'R':
                        state_ = 1911;
                        return true;
                    case 'S':
                        state_ = 1945;
                        return true;
                    case 'T':
                        state_ = 2020;
                        return true;
                    case 'V':
                        state_ = 2044;
                        return true;
                }
                break;
            case 1675:
                if ('o' == current) {
                    state_ = 1676;
                    return true;
                }
                else if ('u' == current) {
                    state_ = 1684;
                    return true;
                }
                break;
            case 1676:
                if ('n' == current) {
                    state_ = 1677;
                    return true;
                }
                break;
            case 1677:
                if ('g' == current) {
                    state_ = 1678;
                    return true;
                }
                break;
            case 1678:
                if ('r' == current) {
                    state_ = 1679;
                    return true;
                }
                break;
            case 1679:
                if ('u' == current) {
                    state_ = 1680;
                    return true;
                }
                break;
            case 1680:
                if ('e' == current) {
                    state_ = 1681;
                    return true;
                }
                break;
            case 1681:
                if ('n' == current) {
                    state_ = 1682;
                    return true;
                }
                break;
            case 1682:
                if ('t' == current) {
                    state_ = 1683;
                    return true;
                }
                break;
            case 1683:
                // NotCongruent;
                if (';' == current) {
                    match_ = "\u2262";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1684:
                if ('p' == current) {
                    state_ = 1685;
                    return true;
                }
                break;
            case 1685:
                if ('C' == current) {
                    state_ = 1686;
                    return true;
                }
                break;
            case 1686:
                if ('a' == current) {
                    state_ = 1687;
                    return true;
                }
                break;
            case 1687:
                if ('p' == current) {
                    state_ = 1688;
                    return true;
                }
                break;
            case 1688:
                // NotCupCap;
                if (';' == current) {
                    match_ = "\u226D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1689:
                if ('o' == current) {
                    state_ = 1690;
                    return true;
                }
                break;
            case 1690:
                if ('u' == current) {
                    state_ = 1691;
                    return true;
                }
                break;
            case 1691:
                if ('b' == current) {
                    state_ = 1692;
                    return true;
                }
                break;
            case 1692:
                if ('l' == current) {
                    state_ = 1693;
                    return true;
                }
                break;
            case 1693:
                if ('e' == current) {
                    state_ = 1694;
                    return true;
                }
                break;
            case 1694:
                if ('V' == current) {
                    state_ = 1695;
                    return true;
                }
                break;
            case 1695:
                if ('e' == current) {
                    state_ = 1696;
                    return true;
                }
                break;
            case 1696:
                if ('r' == current) {
                    state_ = 1697;
                    return true;
                }
                break;
            case 1697:
                if ('t' == current) {
                    state_ = 1698;
                    return true;
                }
                break;
            case 1698:
                if ('i' == current) {
                    state_ = 1699;
                    return true;
                }
                break;
            case 1699:
                if ('c' == current) {
                    state_ = 1700;
                    return true;
                }
                break;
            case 1700:
                if ('a' == current) {
                    state_ = 1701;
                    return true;
                }
                break;
            case 1701:
                if ('l' == current) {
                    state_ = 1702;
                    return true;
                }
                break;
            case 1702:
                if ('B' == current) {
                    state_ = 1703;
                    return true;
                }
                break;
            case 1703:
                if ('a' == current) {
                    state_ = 1704;
                    return true;
                }
                break;
            case 1704:
                if ('r' == current) {
                    state_ = 1705;
                    return true;
                }
                break;
            case 1705:
                // NotDoubleVerticalBar;
                if (';' == current) {
                    match_ = "\u2226";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1706:
                switch (current) {
                    case 'l':
                        state_ = 1707;
                        return true;
                    case 'q':
                        state_ = 1713;
                        return true;
                    case 'x':
                        state_ = 1722;
                        return true;
                }
                break;
            case 1707:
                if ('e' == current) {
                    state_ = 1708;
                    return true;
                }
                break;
            case 1708:
                if ('m' == current) {
                    state_ = 1709;
                    return true;
                }
                break;
            case 1709:
                if ('e' == current) {
                    state_ = 1710;
                    return true;
                }
                break;
            case 1710:
                if ('n' == current) {
                    state_ = 1711;
                    return true;
                }
                break;
            case 1711:
                if ('t' == current) {
                    state_ = 1712;
                    return true;
                }
                break;
            case 1712:
                // NotElement;
                if (';' == current) {
                    match_ = "\u2209";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1713:
                if ('u' == current) {
                    state_ = 1714;
                    return true;
                }
                break;
            case 1714:
                if ('a' == current) {
                    state_ = 1715;
                    return true;
                }
                break;
            case 1715:
                if ('l' == current) {
                    state_ = 1716;
                    return true;
                }
                break;
            case 1716:
                // NotEqual;
                if (';' == current) {
                    match_ = "\u2260";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('T' == current) {
                    state_ = 1717;
                    return true;
                }
                break;
            case 1717:
                if ('i' == current) {
                    state_ = 1718;
                    return true;
                }
                break;
            case 1718:
                if ('l' == current) {
                    state_ = 1719;
                    return true;
                }
                break;
            case 1719:
                if ('d' == current) {
                    state_ = 1720;
                    return true;
                }
                break;
            case 1720:
                if ('e' == current) {
                    state_ = 1721;
                    return true;
                }
                break;
            case 1721:
                // NotEqualTilde;
                if (';' == current) {
                    match_ = "\u2242\u0338";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1722:
                if ('i' == current) {
                    state_ = 1723;
                    return true;
                }
                break;
            case 1723:
                if ('s' == current) {
                    state_ = 1724;
                    return true;
                }
                break;
            case 1724:
                if ('t' == current) {
                    state_ = 1725;
                    return true;
                }
                break;
            case 1725:
                if ('s' == current) {
                    state_ = 1726;
                    return true;
                }
                break;
            case 1726:
                // NotExists;
                if (';' == current) {
                    match_ = "\u2204";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1727:
                if ('r' == current) {
                    state_ = 1728;
                    return true;
                }
                break;
            case 1728:
                if ('e' == current) {
                    state_ = 1729;
                    return true;
                }
                break;
            case 1729:
                if ('a' == current) {
                    state_ = 1730;
                    return true;
                }
                break;
            case 1730:
                if ('t' == current) {
                    state_ = 1731;
                    return true;
                }
                break;
            case 1731:
                if ('e' == current) {
                    state_ = 1732;
                    return true;
                }
                break;
            case 1732:
                if ('r' == current) {
                    state_ = 1733;
                    return true;
                }
                break;
            case 1733:
                switch (current) {
                    case ';': // NotGreater;
                        match_ = "\u226F";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E':
                        state_ = 1734;
                        return true;
                    case 'F':
                        state_ = 1739;
                        return true;
                    case 'G':
                        state_ = 1750;
                        return true;
                    case 'L':
                        state_ = 1759;
                        return true;
                    case 'S':
                        state_ = 1763;
                        return true;
                    case 'T':
                        state_ = 1775;
                        return true;
                }
                break;
            case 1734:
                if ('q' == current) {
                    state_ = 1735;
                    return true;
                }
                break;
            case 1735:
                if ('u' == current) {
                    state_ = 1736;
                    return true;
                }
                break;
            case 1736:
                if ('a' == current) {
                    state_ = 1737;
                    return true;
                }
                break;
            case 1737:
                if ('l' == current) {
                    state_ = 1738;
                    return true;
                }
                break;
            case 1738:
                // NotGreaterEqual;
                if (';' == current) {
                    match_ = "\u2271";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1739:
                if ('u' == current) {
                    state_ = 1740;
                    return true;
                }
                break;
            case 1740:
                if ('l' == current) {
                    state_ = 1741;
                    return true;
                }
                break;
            case 1741:
                if ('l' == current) {
                    state_ = 1742;
                    return true;
                }
                break;
            case 1742:
                if ('E' == current) {
                    state_ = 1743;
                    return true;
                }
                break;
            case 1743:
                if ('q' == current) {
                    state_ = 1744;
                    return true;
                }
                break;
            case 1744:
                if ('u' == current) {
                    state_ = 1745;
                    return true;
                }
                break;
            case 1745:
                if ('a' == current) {
                    state_ = 1746;
                    return true;
                }
                break;
            case 1746:
                if ('l' == current) {
                    state_ = 1747;
                    return true;
                }
                break;
            case 1747:
                if (';' == current) {
                    state_ = 1748;
                    return true;
                }
                break;
            case 1748:
                if ('≧' == current) {
                    state_ = 1749;
                    return true;
                }
                break;
            case 1749:
                // NotGreaterFullEqual;≧̸
                if ('̸' == current) {
                    match_ = "";
                    matchLength_ = consumedCount_;
                    return false;
                }
                break;
            case 1750:
                if ('r' == current) {
                    state_ = 1751;
                    return true;
                }
                break;
            case 1751:
                if ('e' == current) {
                    state_ = 1752;
                    return true;
                }
                break;
            case 1752:
                if ('a' == current) {
                    state_ = 1753;
                    return true;
                }
                break;
            case 1753:
                if ('t' == current) {
                    state_ = 1754;
                    return true;
                }
                break;
            case 1754:
                if ('e' == current) {
                    state_ = 1755;
                    return true;
                }
                break;
            case 1755:
                if ('r' == current) {
                    state_ = 1756;
                    return true;
                }
                break;
            case 1756:
                if (';' == current) {
                    state_ = 1757;
                    return true;
                }
                break;
            case 1757:
                if ('≫' == current) {
                    state_ = 1758;
                    return true;
                }
                break;
            case 1758:
                // NotGreaterGreater;≫̸
                if ('̸' == current) {
                    match_ = "";
                    matchLength_ = consumedCount_;
                    return false;
                }
                break;
            case 1759:
                if ('e' == current) {
                    state_ = 1760;
                    return true;
                }
                break;
            case 1760:
                if ('s' == current) {
                    state_ = 1761;
                    return true;
                }
                break;
            case 1761:
                if ('s' == current) {
                    state_ = 1762;
                    return true;
                }
                break;
            case 1762:
                // NotGreaterLess;
                if (';' == current) {
                    match_ = "\u2279";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1763:
                if ('l' == current) {
                    state_ = 1764;
                    return true;
                }
                break;
            case 1764:
                if ('a' == current) {
                    state_ = 1765;
                    return true;
                }
                break;
            case 1765:
                if ('n' == current) {
                    state_ = 1766;
                    return true;
                }
                break;
            case 1766:
                if ('t' == current) {
                    state_ = 1767;
                    return true;
                }
                break;
            case 1767:
                if ('E' == current) {
                    state_ = 1768;
                    return true;
                }
                break;
            case 1768:
                if ('q' == current) {
                    state_ = 1769;
                    return true;
                }
                break;
            case 1769:
                if ('u' == current) {
                    state_ = 1770;
                    return true;
                }
                break;
            case 1770:
                if ('a' == current) {
                    state_ = 1771;
                    return true;
                }
                break;
            case 1771:
                if ('l' == current) {
                    state_ = 1772;
                    return true;
                }
                break;
            case 1772:
                if (';' == current) {
                    state_ = 1773;
                    return true;
                }
                break;
            case 1773:
                if ('⩾' == current) {
                    state_ = 1774;
                    return true;
                }
                break;
            case 1774:
                // NotGreaterSlantEqual;⩾̸
                if ('̸' == current) {
                    match_ = "";
                    matchLength_ = consumedCount_;
                    return false;
                }
                break;
            case 1775:
                if ('i' == current) {
                    state_ = 1776;
                    return true;
                }
                break;
            case 1776:
                if ('l' == current) {
                    state_ = 1777;
                    return true;
                }
                break;
            case 1777:
                if ('d' == current) {
                    state_ = 1778;
                    return true;
                }
                break;
            case 1778:
                if ('e' == current) {
                    state_ = 1779;
                    return true;
                }
                break;
            case 1779:
                // NotGreaterTilde;
                if (';' == current) {
                    match_ = "\u2275";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1780:
                if ('u' == current) {
                    state_ = 1781;
                    return true;
                }
                break;
            case 1781:
                if ('m' == current) {
                    state_ = 1782;
                    return true;
                }
                break;
            case 1782:
                if ('p' == current) {
                    state_ = 1783;
                    return true;
                }
                break;
            case 1783:
                if ('D' == current) {
                    state_ = 1784;
                    return true;
                }
                else if ('E' == current) {
                    state_ = 1794;
                    return true;
                }
                break;
            case 1784:
                if ('o' == current) {
                    state_ = 1785;
                    return true;
                }
                break;
            case 1785:
                if ('w' == current) {
                    state_ = 1786;
                    return true;
                }
                break;
            case 1786:
                if ('n' == current) {
                    state_ = 1787;
                    return true;
                }
                break;
            case 1787:
                if ('H' == current) {
                    state_ = 1788;
                    return true;
                }
                break;
            case 1788:
                if ('u' == current) {
                    state_ = 1789;
                    return true;
                }
                break;
            case 1789:
                if ('m' == current) {
                    state_ = 1790;
                    return true;
                }
                break;
            case 1790:
                if ('p' == current) {
                    state_ = 1791;
                    return true;
                }
                break;
            case 1791:
                if (';' == current) {
                    state_ = 1792;
                    return true;
                }
                break;
            case 1792:
                if ('≎' == current) {
                    state_ = 1793;
                    return true;
                }
                break;
            case 1793:
                // NotHumpDownHump;≎̸
                if ('̸' == current) {
                    match_ = "";
                    matchLength_ = consumedCount_;
                    return false;
                }
                break;
            case 1794:
                if ('q' == current) {
                    state_ = 1795;
                    return true;
                }
                break;
            case 1795:
                if ('u' == current) {
                    state_ = 1796;
                    return true;
                }
                break;
            case 1796:
                if ('a' == current) {
                    state_ = 1797;
                    return true;
                }
                break;
            case 1797:
                if ('l' == current) {
                    state_ = 1798;
                    return true;
                }
                break;
            case 1798:
                if (';' == current) {
                    state_ = 1799;
                    return true;
                }
                break;
            case 1799:
                if ('≏' == current) {
                    state_ = 1800;
                    return true;
                }
                break;
            case 1800:
                // NotHumpEqual;≏̸
                if ('̸' == current) {
                    match_ = "";
                    matchLength_ = consumedCount_;
                    return false;
                }
                break;
            case 1801:
                if ('e' == current) {
                    state_ = 1802;
                    return true;
                }
                break;
            case 1802:
                if ('f' == current) {
                    state_ = 1803;
                    return true;
                }
                else if ('s' == current) {
                    state_ = 1823;
                    return true;
                }
                break;
            case 1803:
                if ('t' == current) {
                    state_ = 1804;
                    return true;
                }
                break;
            case 1804:
                if ('T' == current) {
                    state_ = 1805;
                    return true;
                }
                break;
            case 1805:
                if ('r' == current) {
                    state_ = 1806;
                    return true;
                }
                break;
            case 1806:
                if ('i' == current) {
                    state_ = 1807;
                    return true;
                }
                break;
            case 1807:
                if ('a' == current) {
                    state_ = 1808;
                    return true;
                }
                break;
            case 1808:
                if ('n' == current) {
                    state_ = 1809;
                    return true;
                }
                break;
            case 1809:
                if ('g' == current) {
                    state_ = 1810;
                    return true;
                }
                break;
            case 1810:
                if ('l' == current) {
                    state_ = 1811;
                    return true;
                }
                break;
            case 1811:
                if ('e' == current) {
                    state_ = 1812;
                    return true;
                }
                break;
            case 1812:
                switch (current) {
                    case ';': // NotLeftTriangle;
                        match_ = "\u22EA";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B':
                        state_ = 1813;
                        return true;
                    case 'E':
                        state_ = 1818;
                        return true;
                }
                break;
            case 1813:
                if ('a' == current) {
                    state_ = 1814;
                    return true;
                }
                break;
            case 1814:
                if ('r' == current) {
                    state_ = 1815;
                    return true;
                }
                break;
            case 1815:
                if (';' == current) {
                    state_ = 1816;
                    return true;
                }
                break;
            case 1816:
                if ('⧏' == current) {
                    state_ = 1817;
                    return true;
                }
                break;
            case 1817:
                // NotLeftTriangleBar;⧏̸
                if ('̸' == current) {
                    match_ = "";
                    matchLength_ = consumedCount_;
                    return false;
                }
                break;
            case 1818:
                if ('q' == current) {
                    state_ = 1819;
                    return true;
                }
                break;
            case 1819:
                if ('u' == current) {
                    state_ = 1820;
                    return true;
                }
                break;
            case 1820:
                if ('a' == current) {
                    state_ = 1821;
                    return true;
                }
                break;
            case 1821:
                if ('l' == current) {
                    state_ = 1822;
                    return true;
                }
                break;
            case 1822:
                // NotLeftTriangleEqual;
                if (';' == current) {
                    match_ = "\u22EC";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1823:
                if ('s' == current) {
                    state_ = 1824;
                    return true;
                }
                break;
            case 1824:
                switch (current) {
                    case ';': // NotLess;
                        match_ = "\u226E";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E':
                        state_ = 1825;
                        return true;
                    case 'G':
                        state_ = 1830;
                        return true;
                    case 'L':
                        state_ = 1837;
                        return true;
                    case 'S':
                        state_ = 1843;
                        return true;
                    case 'T':
                        state_ = 1855;
                        return true;
                }
                break;
            case 1825:
                if ('q' == current) {
                    state_ = 1826;
                    return true;
                }
                break;
            case 1826:
                if ('u' == current) {
                    state_ = 1827;
                    return true;
                }
                break;
            case 1827:
                if ('a' == current) {
                    state_ = 1828;
                    return true;
                }
                break;
            case 1828:
                if ('l' == current) {
                    state_ = 1829;
                    return true;
                }
                break;
            case 1829:
                // NotLessEqual;
                if (';' == current) {
                    match_ = "\u2270";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1830:
                if ('r' == current) {
                    state_ = 1831;
                    return true;
                }
                break;
            case 1831:
                if ('e' == current) {
                    state_ = 1832;
                    return true;
                }
                break;
            case 1832:
                if ('a' == current) {
                    state_ = 1833;
                    return true;
                }
                break;
            case 1833:
                if ('t' == current) {
                    state_ = 1834;
                    return true;
                }
                break;
            case 1834:
                if ('e' == current) {
                    state_ = 1835;
                    return true;
                }
                break;
            case 1835:
                if ('r' == current) {
                    state_ = 1836;
                    return true;
                }
                break;
            case 1836:
                // NotLessGreater;
                if (';' == current) {
                    match_ = "\u2278";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1837:
                if ('e' == current) {
                    state_ = 1838;
                    return true;
                }
                break;
            case 1838:
                if ('s' == current) {
                    state_ = 1839;
                    return true;
                }
                break;
            case 1839:
                if ('s' == current) {
                    state_ = 1840;
                    return true;
                }
                break;
            case 1840:
                if (';' == current) {
                    state_ = 1841;
                    return true;
                }
                break;
            case 1841:
                if ('≪' == current) {
                    state_ = 1842;
                    return true;
                }
                break;
            case 1842:
                // NotLessLess;≪̸
                if ('̸' == current) {
                    match_ = "";
                    matchLength_ = consumedCount_;
                    return false;
                }
                break;
            case 1843:
                if ('l' == current) {
                    state_ = 1844;
                    return true;
                }
                break;
            case 1844:
                if ('a' == current) {
                    state_ = 1845;
                    return true;
                }
                break;
            case 1845:
                if ('n' == current) {
                    state_ = 1846;
                    return true;
                }
                break;
            case 1846:
                if ('t' == current) {
                    state_ = 1847;
                    return true;
                }
                break;
            case 1847:
                if ('E' == current) {
                    state_ = 1848;
                    return true;
                }
                break;
            case 1848:
                if ('q' == current) {
                    state_ = 1849;
                    return true;
                }
                break;
            case 1849:
                if ('u' == current) {
                    state_ = 1850;
                    return true;
                }
                break;
            case 1850:
                if ('a' == current) {
                    state_ = 1851;
                    return true;
                }
                break;
            case 1851:
                if ('l' == current) {
                    state_ = 1852;
                    return true;
                }
                break;
            case 1852:
                if (';' == current) {
                    state_ = 1853;
                    return true;
                }
                break;
            case 1853:
                if ('⩽' == current) {
                    state_ = 1854;
                    return true;
                }
                break;
            case 1854:
                // NotLessSlantEqual;⩽̸
                if ('̸' == current) {
                    match_ = "";
                    matchLength_ = consumedCount_;
                    return false;
                }
                break;
            case 1855:
                if ('i' == current) {
                    state_ = 1856;
                    return true;
                }
                break;
            case 1856:
                if ('l' == current) {
                    state_ = 1857;
                    return true;
                }
                break;
            case 1857:
                if ('d' == current) {
                    state_ = 1858;
                    return true;
                }
                break;
            case 1858:
                if ('e' == current) {
                    state_ = 1859;
                    return true;
                }
                break;
            case 1859:
                // NotLessTilde;
                if (';' == current) {
                    match_ = "\u2274";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1860:
                if ('e' == current) {
                    state_ = 1861;
                    return true;
                }
                break;
            case 1861:
                if ('s' == current) {
                    state_ = 1862;
                    return true;
                }
                break;
            case 1862:
                if ('t' == current) {
                    state_ = 1863;
                    return true;
                }
                break;
            case 1863:
                if ('e' == current) {
                    state_ = 1864;
                    return true;
                }
                break;
            case 1864:
                if ('d' == current) {
                    state_ = 1865;
                    return true;
                }
                break;
            case 1865:
                if ('G' == current) {
                    state_ = 1866;
                    return true;
                }
                else if ('L' == current) {
                    state_ = 1880;
                    return true;
                }
                break;
            case 1866:
                if ('r' == current) {
                    state_ = 1867;
                    return true;
                }
                break;
            case 1867:
                if ('e' == current) {
                    state_ = 1868;
                    return true;
                }
                break;
            case 1868:
                if ('a' == current) {
                    state_ = 1869;
                    return true;
                }
                break;
            case 1869:
                if ('t' == current) {
                    state_ = 1870;
                    return true;
                }
                break;
            case 1870:
                if ('e' == current) {
                    state_ = 1871;
                    return true;
                }
                break;
            case 1871:
                if ('r' == current) {
                    state_ = 1872;
                    return true;
                }
                break;
            case 1872:
                if ('G' == current) {
                    state_ = 1873;
                    return true;
                }
                break;
            case 1873:
                if ('r' == current) {
                    state_ = 1874;
                    return true;
                }
                break;
            case 1874:
                if ('e' == current) {
                    state_ = 1875;
                    return true;
                }
                break;
            case 1875:
                if ('a' == current) {
                    state_ = 1876;
                    return true;
                }
                break;
            case 1876:
                if ('t' == current) {
                    state_ = 1877;
                    return true;
                }
                break;
            case 1877:
                if ('e' == current) {
                    state_ = 1878;
                    return true;
                }
                break;
            case 1878:
                if ('r' == current) {
                    state_ = 1879;
                    return true;
                }
                break;
            case 1879:
                // NotNestedGreaterGreater;
                if (';' == current) {
                    match_ = "\u2AA2\u0338";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1880:
                if ('e' == current) {
                    state_ = 1881;
                    return true;
                }
                break;
            case 1881:
                if ('s' == current) {
                    state_ = 1882;
                    return true;
                }
                break;
            case 1882:
                if ('s' == current) {
                    state_ = 1883;
                    return true;
                }
                break;
            case 1883:
                if ('L' == current) {
                    state_ = 1884;
                    return true;
                }
                break;
            case 1884:
                if ('e' == current) {
                    state_ = 1885;
                    return true;
                }
                break;
            case 1885:
                if ('s' == current) {
                    state_ = 1886;
                    return true;
                }
                break;
            case 1886:
                if ('s' == current) {
                    state_ = 1887;
                    return true;
                }
                break;
            case 1887:
                // NotNestedLessLess;
                if (';' == current) {
                    match_ = "\u2AA1\u0338";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1888:
                if ('r' == current) {
                    state_ = 1889;
                    return true;
                }
                break;
            case 1889:
                if ('e' == current) {
                    state_ = 1890;
                    return true;
                }
                break;
            case 1890:
                if ('c' == current) {
                    state_ = 1891;
                    return true;
                }
                break;
            case 1891:
                if ('e' == current) {
                    state_ = 1892;
                    return true;
                }
                break;
            case 1892:
                if ('d' == current) {
                    state_ = 1893;
                    return true;
                }
                break;
            case 1893:
                if ('e' == current) {
                    state_ = 1894;
                    return true;
                }
                break;
            case 1894:
                if ('s' == current) {
                    state_ = 1895;
                    return true;
                }
                break;
            case 1895:
                switch (current) {
                    case ';': // NotPrecedes;
                        match_ = "\u2280";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E':
                        state_ = 1896;
                        return true;
                    case 'S':
                        state_ = 1901;
                        return true;
                }
                break;
            case 1896:
                if ('q' == current) {
                    state_ = 1897;
                    return true;
                }
                break;
            case 1897:
                if ('u' == current) {
                    state_ = 1898;
                    return true;
                }
                break;
            case 1898:
                if ('a' == current) {
                    state_ = 1899;
                    return true;
                }
                break;
            case 1899:
                if ('l' == current) {
                    state_ = 1900;
                    return true;
                }
                break;
            case 1900:
                // NotPrecedesEqual;
                if (';' == current) {
                    match_ = "\u2AAF\u0338";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1901:
                if ('l' == current) {
                    state_ = 1902;
                    return true;
                }
                break;
            case 1902:
                if ('a' == current) {
                    state_ = 1903;
                    return true;
                }
                break;
            case 1903:
                if ('n' == current) {
                    state_ = 1904;
                    return true;
                }
                break;
            case 1904:
                if ('t' == current) {
                    state_ = 1905;
                    return true;
                }
                break;
            case 1905:
                if ('E' == current) {
                    state_ = 1906;
                    return true;
                }
                break;
            case 1906:
                if ('q' == current) {
                    state_ = 1907;
                    return true;
                }
                break;
            case 1907:
                if ('u' == current) {
                    state_ = 1908;
                    return true;
                }
                break;
            case 1908:
                if ('a' == current) {
                    state_ = 1909;
                    return true;
                }
                break;
            case 1909:
                if ('l' == current) {
                    state_ = 1910;
                    return true;
                }
                break;
            case 1910:
                // NotPrecedesSlantEqual;
                if (';' == current) {
                    match_ = "\u22E0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1911:
                if ('e' == current) {
                    state_ = 1912;
                    return true;
                }
                else if ('i' == current) {
                    state_ = 1925;
                    return true;
                }
                break;
            case 1912:
                if ('v' == current) {
                    state_ = 1913;
                    return true;
                }
                break;
            case 1913:
                if ('e' == current) {
                    state_ = 1914;
                    return true;
                }
                break;
            case 1914:
                if ('r' == current) {
                    state_ = 1915;
                    return true;
                }
                break;
            case 1915:
                if ('s' == current) {
                    state_ = 1916;
                    return true;
                }
                break;
            case 1916:
                if ('e' == current) {
                    state_ = 1917;
                    return true;
                }
                break;
            case 1917:
                if ('E' == current) {
                    state_ = 1918;
                    return true;
                }
                break;
            case 1918:
                if ('l' == current) {
                    state_ = 1919;
                    return true;
                }
                break;
            case 1919:
                if ('e' == current) {
                    state_ = 1920;
                    return true;
                }
                break;
            case 1920:
                if ('m' == current) {
                    state_ = 1921;
                    return true;
                }
                break;
            case 1921:
                if ('e' == current) {
                    state_ = 1922;
                    return true;
                }
                break;
            case 1922:
                if ('n' == current) {
                    state_ = 1923;
                    return true;
                }
                break;
            case 1923:
                if ('t' == current) {
                    state_ = 1924;
                    return true;
                }
                break;
            case 1924:
                // NotReverseElement;
                if (';' == current) {
                    match_ = "\u220C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1925:
                if ('g' == current) {
                    state_ = 1926;
                    return true;
                }
                break;
            case 1926:
                if ('h' == current) {
                    state_ = 1927;
                    return true;
                }
                break;
            case 1927:
                if ('t' == current) {
                    state_ = 1928;
                    return true;
                }
                break;
            case 1928:
                if ('T' == current) {
                    state_ = 1929;
                    return true;
                }
                break;
            case 1929:
                if ('r' == current) {
                    state_ = 1930;
                    return true;
                }
                break;
            case 1930:
                if ('i' == current) {
                    state_ = 1931;
                    return true;
                }
                break;
            case 1931:
                if ('a' == current) {
                    state_ = 1932;
                    return true;
                }
                break;
            case 1932:
                if ('n' == current) {
                    state_ = 1933;
                    return true;
                }
                break;
            case 1933:
                if ('g' == current) {
                    state_ = 1934;
                    return true;
                }
                break;
            case 1934:
                if ('l' == current) {
                    state_ = 1935;
                    return true;
                }
                break;
            case 1935:
                if ('e' == current) {
                    state_ = 1936;
                    return true;
                }
                break;
            case 1936:
                switch (current) {
                    case ';': // NotRightTriangle;
                        match_ = "\u22EB";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B':
                        state_ = 1937;
                        return true;
                    case 'E':
                        state_ = 1940;
                        return true;
                }
                break;
            case 1937:
                if ('a' == current) {
                    state_ = 1938;
                    return true;
                }
                break;
            case 1938:
                if ('r' == current) {
                    state_ = 1939;
                    return true;
                }
                break;
            case 1939:
                // NotRightTriangleBar;
                if (';' == current) {
                    match_ = "\u29D0\u0338";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1940:
                if ('q' == current) {
                    state_ = 1941;
                    return true;
                }
                break;
            case 1941:
                if ('u' == current) {
                    state_ = 1942;
                    return true;
                }
                break;
            case 1942:
                if ('a' == current) {
                    state_ = 1943;
                    return true;
                }
                break;
            case 1943:
                if ('l' == current) {
                    state_ = 1944;
                    return true;
                }
                break;
            case 1944:
                // NotRightTriangleEqual;
                if (';' == current) {
                    match_ = "\u22ED";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1945:
                if ('q' == current) {
                    state_ = 1946;
                    return true;
                }
                else if ('u' == current) {
                    state_ = 1973;
                    return true;
                }
                break;
            case 1946:
                if ('u' == current) {
                    state_ = 1947;
                    return true;
                }
                break;
            case 1947:
                if ('a' == current) {
                    state_ = 1948;
                    return true;
                }
                break;
            case 1948:
                if ('r' == current) {
                    state_ = 1949;
                    return true;
                }
                break;
            case 1949:
                if ('e' == current) {
                    state_ = 1950;
                    return true;
                }
                break;
            case 1950:
                if ('S' == current) {
                    state_ = 1951;
                    return true;
                }
                break;
            case 1951:
                if ('u' == current) {
                    state_ = 1952;
                    return true;
                }
                break;
            case 1952:
                if ('b' == current) {
                    state_ = 1953;
                    return true;
                }
                else if ('p' == current) {
                    state_ = 1962;
                    return true;
                }
                break;
            case 1953:
                if ('s' == current) {
                    state_ = 1954;
                    return true;
                }
                break;
            case 1954:
                if ('e' == current) {
                    state_ = 1955;
                    return true;
                }
                break;
            case 1955:
                if ('t' == current) {
                    state_ = 1956;
                    return true;
                }
                break;
            case 1956:
                // NotSquareSubset;
                if (';' == current) {
                    match_ = "\u228F\u0338";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('E' == current) {
                    state_ = 1957;
                    return true;
                }
                break;
            case 1957:
                if ('q' == current) {
                    state_ = 1958;
                    return true;
                }
                break;
            case 1958:
                if ('u' == current) {
                    state_ = 1959;
                    return true;
                }
                break;
            case 1959:
                if ('a' == current) {
                    state_ = 1960;
                    return true;
                }
                break;
            case 1960:
                if ('l' == current) {
                    state_ = 1961;
                    return true;
                }
                break;
            case 1961:
                // NotSquareSubsetEqual;
                if (';' == current) {
                    match_ = "\u22E2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1962:
                if ('e' == current) {
                    state_ = 1963;
                    return true;
                }
                break;
            case 1963:
                if ('r' == current) {
                    state_ = 1964;
                    return true;
                }
                break;
            case 1964:
                if ('s' == current) {
                    state_ = 1965;
                    return true;
                }
                break;
            case 1965:
                if ('e' == current) {
                    state_ = 1966;
                    return true;
                }
                break;
            case 1966:
                if ('t' == current) {
                    state_ = 1967;
                    return true;
                }
                break;
            case 1967:
                // NotSquareSuperset;
                if (';' == current) {
                    match_ = "\u2290\u0338";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('E' == current) {
                    state_ = 1968;
                    return true;
                }
                break;
            case 1968:
                if ('q' == current) {
                    state_ = 1969;
                    return true;
                }
                break;
            case 1969:
                if ('u' == current) {
                    state_ = 1970;
                    return true;
                }
                break;
            case 1970:
                if ('a' == current) {
                    state_ = 1971;
                    return true;
                }
                break;
            case 1971:
                if ('l' == current) {
                    state_ = 1972;
                    return true;
                }
                break;
            case 1972:
                // NotSquareSupersetEqual;
                if (';' == current) {
                    match_ = "\u22E3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1973:
                switch (current) {
                    case 'b':
                        state_ = 1974;
                        return true;
                    case 'c':
                        state_ = 1983;
                        return true;
                    case 'p':
                        state_ = 2009;
                        return true;
                }
                break;
            case 1974:
                if ('s' == current) {
                    state_ = 1975;
                    return true;
                }
                break;
            case 1975:
                if ('e' == current) {
                    state_ = 1976;
                    return true;
                }
                break;
            case 1976:
                if ('t' == current) {
                    state_ = 1977;
                    return true;
                }
                break;
            case 1977:
                // NotSubset;
                if (';' == current) {
                    match_ = "\u2282\u20D2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('E' == current) {
                    state_ = 1978;
                    return true;
                }
                break;
            case 1978:
                if ('q' == current) {
                    state_ = 1979;
                    return true;
                }
                break;
            case 1979:
                if ('u' == current) {
                    state_ = 1980;
                    return true;
                }
                break;
            case 1980:
                if ('a' == current) {
                    state_ = 1981;
                    return true;
                }
                break;
            case 1981:
                if ('l' == current) {
                    state_ = 1982;
                    return true;
                }
                break;
            case 1982:
                // NotSubsetEqual;
                if (';' == current) {
                    match_ = "\u2288";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1983:
                if ('c' == current) {
                    state_ = 1984;
                    return true;
                }
                break;
            case 1984:
                if ('e' == current) {
                    state_ = 1985;
                    return true;
                }
                break;
            case 1985:
                if ('e' == current) {
                    state_ = 1986;
                    return true;
                }
                break;
            case 1986:
                if ('d' == current) {
                    state_ = 1987;
                    return true;
                }
                break;
            case 1987:
                if ('s' == current) {
                    state_ = 1988;
                    return true;
                }
                break;
            case 1988:
                switch (current) {
                    case ';': // NotSucceeds;
                        match_ = "\u2281";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E':
                        state_ = 1989;
                        return true;
                    case 'S':
                        state_ = 1994;
                        return true;
                    case 'T':
                        state_ = 2004;
                        return true;
                }
                break;
            case 1989:
                if ('q' == current) {
                    state_ = 1990;
                    return true;
                }
                break;
            case 1990:
                if ('u' == current) {
                    state_ = 1991;
                    return true;
                }
                break;
            case 1991:
                if ('a' == current) {
                    state_ = 1992;
                    return true;
                }
                break;
            case 1992:
                if ('l' == current) {
                    state_ = 1993;
                    return true;
                }
                break;
            case 1993:
                // NotSucceedsEqual;
                if (';' == current) {
                    match_ = "\u2AB0\u0338";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 1994:
                if ('l' == current) {
                    state_ = 1995;
                    return true;
                }
                break;
            case 1995:
                if ('a' == current) {
                    state_ = 1996;
                    return true;
                }
                break;
            case 1996:
                if ('n' == current) {
                    state_ = 1997;
                    return true;
                }
                break;
            case 1997:
                if ('t' == current) {
                    state_ = 1998;
                    return true;
                }
                break;
            case 1998:
                if ('E' == current) {
                    state_ = 1999;
                    return true;
                }
                break;
            case 1999:
                if ('q' == current) {
                    state_ = 2000;
                    return true;
                }
                break;
        }
        return false;
    }

    private boolean parse3(final int current) {
        consumedCount_++;
        switch (state_) {
            case 2000:
                if ('u' == current) {
                    state_ = 2001;
                    return true;
                }
                break;
            case 2001:
                if ('a' == current) {
                    state_ = 2002;
                    return true;
                }
                break;
            case 2002:
                if ('l' == current) {
                    state_ = 2003;
                    return true;
                }
                break;
            case 2003:
                // NotSucceedsSlantEqual;
                if (';' == current) {
                    match_ = "\u22E1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2004:
                if ('i' == current) {
                    state_ = 2005;
                    return true;
                }
                break;
            case 2005:
                if ('l' == current) {
                    state_ = 2006;
                    return true;
                }
                break;
            case 2006:
                if ('d' == current) {
                    state_ = 2007;
                    return true;
                }
                break;
            case 2007:
                if ('e' == current) {
                    state_ = 2008;
                    return true;
                }
                break;
            case 2008:
                // NotSucceedsTilde;
                if (';' == current) {
                    match_ = "\u227F\u0338";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2009:
                if ('e' == current) {
                    state_ = 2010;
                    return true;
                }
                break;
            case 2010:
                if ('r' == current) {
                    state_ = 2011;
                    return true;
                }
                break;
            case 2011:
                if ('s' == current) {
                    state_ = 2012;
                    return true;
                }
                break;
            case 2012:
                if ('e' == current) {
                    state_ = 2013;
                    return true;
                }
                break;
            case 2013:
                if ('t' == current) {
                    state_ = 2014;
                    return true;
                }
                break;
            case 2014:
                // NotSuperset;
                if (';' == current) {
                    match_ = "\u2283\u20D2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('E' == current) {
                    state_ = 2015;
                    return true;
                }
                break;
            case 2015:
                if ('q' == current) {
                    state_ = 2016;
                    return true;
                }
                break;
            case 2016:
                if ('u' == current) {
                    state_ = 2017;
                    return true;
                }
                break;
            case 2017:
                if ('a' == current) {
                    state_ = 2018;
                    return true;
                }
                break;
            case 2018:
                if ('l' == current) {
                    state_ = 2019;
                    return true;
                }
                break;
            case 2019:
                // NotSupersetEqual;
                if (';' == current) {
                    match_ = "\u2289";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2020:
                if ('i' == current) {
                    state_ = 2021;
                    return true;
                }
                break;
            case 2021:
                if ('l' == current) {
                    state_ = 2022;
                    return true;
                }
                break;
            case 2022:
                if ('d' == current) {
                    state_ = 2023;
                    return true;
                }
                break;
            case 2023:
                if ('e' == current) {
                    state_ = 2024;
                    return true;
                }
                break;
            case 2024:
                switch (current) {
                    case ';': // NotTilde;
                        match_ = "\u2241";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E':
                        state_ = 2025;
                        return true;
                    case 'F':
                        state_ = 2030;
                        return true;
                    case 'T':
                        state_ = 2039;
                        return true;
                }
                break;
            case 2025:
                if ('q' == current) {
                    state_ = 2026;
                    return true;
                }
                break;
            case 2026:
                if ('u' == current) {
                    state_ = 2027;
                    return true;
                }
                break;
            case 2027:
                if ('a' == current) {
                    state_ = 2028;
                    return true;
                }
                break;
            case 2028:
                if ('l' == current) {
                    state_ = 2029;
                    return true;
                }
                break;
            case 2029:
                // NotTildeEqual;
                if (';' == current) {
                    match_ = "\u2244";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2030:
                if ('u' == current) {
                    state_ = 2031;
                    return true;
                }
                break;
            case 2031:
                if ('l' == current) {
                    state_ = 2032;
                    return true;
                }
                break;
            case 2032:
                if ('l' == current) {
                    state_ = 2033;
                    return true;
                }
                break;
            case 2033:
                if ('E' == current) {
                    state_ = 2034;
                    return true;
                }
                break;
            case 2034:
                if ('q' == current) {
                    state_ = 2035;
                    return true;
                }
                break;
            case 2035:
                if ('u' == current) {
                    state_ = 2036;
                    return true;
                }
                break;
            case 2036:
                if ('a' == current) {
                    state_ = 2037;
                    return true;
                }
                break;
            case 2037:
                if ('l' == current) {
                    state_ = 2038;
                    return true;
                }
                break;
            case 2038:
                // NotTildeFullEqual;
                if (';' == current) {
                    match_ = "\u2247";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2039:
                if ('i' == current) {
                    state_ = 2040;
                    return true;
                }
                break;
            case 2040:
                if ('l' == current) {
                    state_ = 2041;
                    return true;
                }
                break;
            case 2041:
                if ('d' == current) {
                    state_ = 2042;
                    return true;
                }
                break;
            case 2042:
                if ('e' == current) {
                    state_ = 2043;
                    return true;
                }
                break;
            case 2043:
                // NotTildeTilde;
                if (';' == current) {
                    match_ = "\u2249";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2044:
                if ('e' == current) {
                    state_ = 2045;
                    return true;
                }
                break;
            case 2045:
                if ('r' == current) {
                    state_ = 2046;
                    return true;
                }
                break;
            case 2046:
                if ('t' == current) {
                    state_ = 2047;
                    return true;
                }
                break;
            case 2047:
                if ('i' == current) {
                    state_ = 2048;
                    return true;
                }
                break;
            case 2048:
                if ('c' == current) {
                    state_ = 2049;
                    return true;
                }
                break;
            case 2049:
                if ('a' == current) {
                    state_ = 2050;
                    return true;
                }
                break;
            case 2050:
                if ('l' == current) {
                    state_ = 2051;
                    return true;
                }
                break;
            case 2051:
                if ('B' == current) {
                    state_ = 2052;
                    return true;
                }
                break;
            case 2052:
                if ('a' == current) {
                    state_ = 2053;
                    return true;
                }
                break;
            case 2053:
                if ('r' == current) {
                    state_ = 2054;
                    return true;
                }
                break;
            case 2054:
                // NotVerticalBar;
                if (';' == current) {
                    match_ = "\u2224";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2055:
                if ('c' == current) {
                    state_ = 2056;
                    return true;
                }
                break;
            case 2056:
                if ('r' == current) {
                    state_ = 2057;
                    return true;
                }
                break;
            case 2057:
                // Nscr;
                if (';' == current) {
                    match_ = "\uD835\uDCA9";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2058:
                if ('i' == current) {
                    state_ = 2059;
                    return true;
                }
                break;
            case 2059:
                if ('l' == current) {
                    state_ = 2060;
                    return true;
                }
                break;
            case 2060:
                if ('d' == current) {
                    state_ = 2061;
                    return true;
                }
                break;
            case 2061:
                // Ntilde
                if ('e' == current) {
                    match_ = "\u00D1";
                    matchLength_ = consumedCount_;
                    state_ = 2062;
                    return true;
                }
                break;
            case 2062:
                // Ntilde;
                if (';' == current) {
                    match_ = "\u00D1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2063:
                // Nu;
                if (';' == current) {
                    match_ = "\u039D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2064:
                switch (current) {
                    case 'E':
                        state_ = 2065;
                        return true;
                    case 'a':
                        state_ = 2069;
                        return true;
                    case 'c':
                        state_ = 2074;
                        return true;
                    case 'd':
                        state_ = 2079;
                        return true;
                    case 'f':
                        state_ = 2084;
                        return true;
                    case 'g':
                        state_ = 2086;
                        return true;
                    case 'm':
                        state_ = 2091;
                        return true;
                    case 'o':
                        state_ = 2103;
                        return true;
                    case 'p':
                        state_ = 2106;
                        return true;
                    case 'r':
                        state_ = 2130;
                        return true;
                    case 's':
                        state_ = 2131;
                        return true;
                    case 't':
                        state_ = 2138;
                        return true;
                    case 'u':
                        state_ = 2146;
                        return true;
                    case 'v':
                        state_ = 2149;
                        return true;
                }
                break;
            case 2065:
                if ('l' == current) {
                    state_ = 2066;
                    return true;
                }
                break;
            case 2066:
                if ('i' == current) {
                    state_ = 2067;
                    return true;
                }
                break;
            case 2067:
                if ('g' == current) {
                    state_ = 2068;
                    return true;
                }
                break;
            case 2068:
                // OElig;
                if (';' == current) {
                    match_ = "\u0152";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2069:
                if ('c' == current) {
                    state_ = 2070;
                    return true;
                }
                break;
            case 2070:
                if ('u' == current) {
                    state_ = 2071;
                    return true;
                }
                break;
            case 2071:
                if ('t' == current) {
                    state_ = 2072;
                    return true;
                }
                break;
            case 2072:
                // Oacute
                if ('e' == current) {
                    match_ = "\u00D3";
                    matchLength_ = consumedCount_;
                    state_ = 2073;
                    return true;
                }
                break;
            case 2073:
                // Oacute;
                if (';' == current) {
                    match_ = "\u00D3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2074:
                if ('i' == current) {
                    state_ = 2075;
                    return true;
                }
                else if ('y' == current) {
                    state_ = 2078;
                    return true;
                }
                break;
            case 2075:
                if ('r' == current) {
                    state_ = 2076;
                    return true;
                }
                break;
            case 2076:
                // Ocirc
                if ('c' == current) {
                    match_ = "\u00D4";
                    matchLength_ = consumedCount_;
                    state_ = 2077;
                    return true;
                }
                break;
            case 2077:
                // Ocirc;
                if (';' == current) {
                    match_ = "\u00D4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2078:
                // Ocy;
                if (';' == current) {
                    match_ = "\u041E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2079:
                if ('b' == current) {
                    state_ = 2080;
                    return true;
                }
                break;
            case 2080:
                if ('l' == current) {
                    state_ = 2081;
                    return true;
                }
                break;
            case 2081:
                if ('a' == current) {
                    state_ = 2082;
                    return true;
                }
                break;
            case 2082:
                if ('c' == current) {
                    state_ = 2083;
                    return true;
                }
                break;
            case 2083:
                // Odblac;
                if (';' == current) {
                    match_ = "\u0150";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2084:
                if ('r' == current) {
                    state_ = 2085;
                    return true;
                }
                break;
            case 2085:
                // Ofr;
                if (';' == current) {
                    match_ = "\uD835\uDD12";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2086:
                if ('r' == current) {
                    state_ = 2087;
                    return true;
                }
                break;
            case 2087:
                if ('a' == current) {
                    state_ = 2088;
                    return true;
                }
                break;
            case 2088:
                if ('v' == current) {
                    state_ = 2089;
                    return true;
                }
                break;
            case 2089:
                // Ograve
                if ('e' == current) {
                    match_ = "\u00D2";
                    matchLength_ = consumedCount_;
                    state_ = 2090;
                    return true;
                }
                break;
            case 2090:
                // Ograve;
                if (';' == current) {
                    match_ = "\u00D2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2091:
                switch (current) {
                    case 'a':
                        state_ = 2092;
                        return true;
                    case 'e':
                        state_ = 2095;
                        return true;
                    case 'i':
                        state_ = 2098;
                        return true;
                }
                break;
            case 2092:
                if ('c' == current) {
                    state_ = 2093;
                    return true;
                }
                break;
            case 2093:
                if ('r' == current) {
                    state_ = 2094;
                    return true;
                }
                break;
            case 2094:
                // Omacr;
                if (';' == current) {
                    match_ = "\u014C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2095:
                if ('g' == current) {
                    state_ = 2096;
                    return true;
                }
                break;
            case 2096:
                if ('a' == current) {
                    state_ = 2097;
                    return true;
                }
                break;
            case 2097:
                // Omega;
                if (';' == current) {
                    match_ = "\u03A9";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2098:
                if ('c' == current) {
                    state_ = 2099;
                    return true;
                }
                break;
            case 2099:
                if ('r' == current) {
                    state_ = 2100;
                    return true;
                }
                break;
            case 2100:
                if ('o' == current) {
                    state_ = 2101;
                    return true;
                }
                break;
            case 2101:
                if ('n' == current) {
                    state_ = 2102;
                    return true;
                }
                break;
            case 2102:
                // Omicron;
                if (';' == current) {
                    match_ = "\u039F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2103:
                if ('p' == current) {
                    state_ = 2104;
                    return true;
                }
                break;
            case 2104:
                if ('f' == current) {
                    state_ = 2105;
                    return true;
                }
                break;
            case 2105:
                // Oopf;
                if (';' == current) {
                    match_ = "\uD835\uDD46";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2106:
                if ('e' == current) {
                    state_ = 2107;
                    return true;
                }
                break;
            case 2107:
                if ('n' == current) {
                    state_ = 2108;
                    return true;
                }
                break;
            case 2108:
                if ('C' == current) {
                    state_ = 2109;
                    return true;
                }
                break;
            case 2109:
                if ('u' == current) {
                    state_ = 2110;
                    return true;
                }
                break;
            case 2110:
                if ('r' == current) {
                    state_ = 2111;
                    return true;
                }
                break;
            case 2111:
                if ('l' == current) {
                    state_ = 2112;
                    return true;
                }
                break;
            case 2112:
                if ('y' == current) {
                    state_ = 2113;
                    return true;
                }
                break;
            case 2113:
                if ('D' == current) {
                    state_ = 2114;
                    return true;
                }
                else if ('Q' == current) {
                    state_ = 2125;
                    return true;
                }
                break;
            case 2114:
                if ('o' == current) {
                    state_ = 2115;
                    return true;
                }
                break;
            case 2115:
                if ('u' == current) {
                    state_ = 2116;
                    return true;
                }
                break;
            case 2116:
                if ('b' == current) {
                    state_ = 2117;
                    return true;
                }
                break;
            case 2117:
                if ('l' == current) {
                    state_ = 2118;
                    return true;
                }
                break;
            case 2118:
                if ('e' == current) {
                    state_ = 2119;
                    return true;
                }
                break;
            case 2119:
                if ('Q' == current) {
                    state_ = 2120;
                    return true;
                }
                break;
            case 2120:
                if ('u' == current) {
                    state_ = 2121;
                    return true;
                }
                break;
            case 2121:
                if ('o' == current) {
                    state_ = 2122;
                    return true;
                }
                break;
            case 2122:
                if ('t' == current) {
                    state_ = 2123;
                    return true;
                }
                break;
            case 2123:
                if ('e' == current) {
                    state_ = 2124;
                    return true;
                }
                break;
            case 2124:
                // OpenCurlyDoubleQuote;
                if (';' == current) {
                    match_ = "\u201C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2125:
                if ('u' == current) {
                    state_ = 2126;
                    return true;
                }
                break;
            case 2126:
                if ('o' == current) {
                    state_ = 2127;
                    return true;
                }
                break;
            case 2127:
                if ('t' == current) {
                    state_ = 2128;
                    return true;
                }
                break;
            case 2128:
                if ('e' == current) {
                    state_ = 2129;
                    return true;
                }
                break;
            case 2129:
                // OpenCurlyQuote;
                if (';' == current) {
                    match_ = "\u2018";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2130:
                // Or;
                if (';' == current) {
                    match_ = "\u2A54";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2131:
                if ('c' == current) {
                    state_ = 2132;
                    return true;
                }
                else if ('l' == current) {
                    state_ = 2134;
                    return true;
                }
                break;
            case 2132:
                if ('r' == current) {
                    state_ = 2133;
                    return true;
                }
                break;
            case 2133:
                // Oscr;
                if (';' == current) {
                    match_ = "\uD835\uDCAA";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2134:
                if ('a' == current) {
                    state_ = 2135;
                    return true;
                }
                break;
            case 2135:
                if ('s' == current) {
                    state_ = 2136;
                    return true;
                }
                break;
            case 2136:
                // Oslash
                if ('h' == current) {
                    match_ = "\u00D8";
                    matchLength_ = consumedCount_;
                    state_ = 2137;
                    return true;
                }
                break;
            case 2137:
                // Oslash;
                if (';' == current) {
                    match_ = "\u00D8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2138:
                if ('i' == current) {
                    state_ = 2139;
                    return true;
                }
                break;
            case 2139:
                if ('l' == current) {
                    state_ = 2140;
                    return true;
                }
                else if ('m' == current) {
                    state_ = 2143;
                    return true;
                }
                break;
            case 2140:
                if ('d' == current) {
                    state_ = 2141;
                    return true;
                }
                break;
            case 2141:
                // Otilde
                if ('e' == current) {
                    match_ = "\u00D5";
                    matchLength_ = consumedCount_;
                    state_ = 2142;
                    return true;
                }
                break;
            case 2142:
                // Otilde;
                if (';' == current) {
                    match_ = "\u00D5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2143:
                if ('e' == current) {
                    state_ = 2144;
                    return true;
                }
                break;
            case 2144:
                if ('s' == current) {
                    state_ = 2145;
                    return true;
                }
                break;
            case 2145:
                // Otimes;
                if (';' == current) {
                    match_ = "\u2A37";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2146:
                if ('m' == current) {
                    state_ = 2147;
                    return true;
                }
                break;
            case 2147:
                // Ouml
                if ('l' == current) {
                    match_ = "\u00D6";
                    matchLength_ = consumedCount_;
                    state_ = 2148;
                    return true;
                }
                break;
            case 2148:
                // Ouml;
                if (';' == current) {
                    match_ = "\u00D6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2149:
                if ('e' == current) {
                    state_ = 2150;
                    return true;
                }
                break;
            case 2150:
                if ('r' == current) {
                    state_ = 2151;
                    return true;
                }
                break;
            case 2151:
                if ('B' == current) {
                    state_ = 2152;
                    return true;
                }
                else if ('P' == current) {
                    state_ = 2162;
                    return true;
                }
                break;
            case 2152:
                if ('a' == current) {
                    state_ = 2153;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 2155;
                    return true;
                }
                break;
            case 2153:
                if ('r' == current) {
                    state_ = 2154;
                    return true;
                }
                break;
            case 2154:
                // OverBar;
                if (';' == current) {
                    match_ = "\u203E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2155:
                if ('a' == current) {
                    state_ = 2156;
                    return true;
                }
                break;
            case 2156:
                if ('c' == current) {
                    state_ = 2157;
                    return true;
                }
                break;
            case 2157:
                if ('e' == current) {
                    state_ = 2158;
                    return true;
                }
                else if ('k' == current) {
                    state_ = 2159;
                    return true;
                }
                break;
            case 2158:
                // OverBrace;
                if (';' == current) {
                    match_ = "\u23DE";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2159:
                if ('e' == current) {
                    state_ = 2160;
                    return true;
                }
                break;
            case 2160:
                if ('t' == current) {
                    state_ = 2161;
                    return true;
                }
                break;
            case 2161:
                // OverBracket;
                if (';' == current) {
                    match_ = "\u23B4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2162:
                if ('a' == current) {
                    state_ = 2163;
                    return true;
                }
                break;
            case 2163:
                if ('r' == current) {
                    state_ = 2164;
                    return true;
                }
                break;
            case 2164:
                if ('e' == current) {
                    state_ = 2165;
                    return true;
                }
                break;
            case 2165:
                if ('n' == current) {
                    state_ = 2166;
                    return true;
                }
                break;
            case 2166:
                if ('t' == current) {
                    state_ = 2167;
                    return true;
                }
                break;
            case 2167:
                if ('h' == current) {
                    state_ = 2168;
                    return true;
                }
                break;
            case 2168:
                if ('e' == current) {
                    state_ = 2169;
                    return true;
                }
                break;
            case 2169:
                if ('s' == current) {
                    state_ = 2170;
                    return true;
                }
                break;
            case 2170:
                if ('i' == current) {
                    state_ = 2171;
                    return true;
                }
                break;
            case 2171:
                if ('s' == current) {
                    state_ = 2172;
                    return true;
                }
                break;
            case 2172:
                // OverParenthesis;
                if (';' == current) {
                    match_ = "\u23DC";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2173:
                switch (current) {
                    case 'a':
                        state_ = 2174;
                        return true;
                    case 'c':
                        state_ = 2181;
                        return true;
                    case 'f':
                        state_ = 2183;
                        return true;
                    case 'h':
                        state_ = 2185;
                        return true;
                    case 'i':
                        state_ = 2187;
                        return true;
                    case 'l':
                        state_ = 2188;
                        return true;
                    case 'o':
                        state_ = 2196;
                        return true;
                    case 'r':
                        state_ = 2210;
                        return true;
                    case 's':
                        state_ = 2254;
                        return true;
                }
                break;
            case 2174:
                if ('r' == current) {
                    state_ = 2175;
                    return true;
                }
                break;
            case 2175:
                if ('t' == current) {
                    state_ = 2176;
                    return true;
                }
                break;
            case 2176:
                if ('i' == current) {
                    state_ = 2177;
                    return true;
                }
                break;
            case 2177:
                if ('a' == current) {
                    state_ = 2178;
                    return true;
                }
                break;
            case 2178:
                if ('l' == current) {
                    state_ = 2179;
                    return true;
                }
                break;
            case 2179:
                if ('D' == current) {
                    state_ = 2180;
                    return true;
                }
                break;
            case 2180:
                // PartialD;
                if (';' == current) {
                    match_ = "\u2202";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2181:
                if ('y' == current) {
                    state_ = 2182;
                    return true;
                }
                break;
            case 2182:
                // Pcy;
                if (';' == current) {
                    match_ = "\u041F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2183:
                if ('r' == current) {
                    state_ = 2184;
                    return true;
                }
                break;
            case 2184:
                // Pfr;
                if (';' == current) {
                    match_ = "\uD835\uDD13";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2185:
                if ('i' == current) {
                    state_ = 2186;
                    return true;
                }
                break;
            case 2186:
                // Phi;
                if (';' == current) {
                    match_ = "\u03A6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2187:
                // Pi;
                if (';' == current) {
                    match_ = "\u03A0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2188:
                if ('u' == current) {
                    state_ = 2189;
                    return true;
                }
                break;
            case 2189:
                if ('s' == current) {
                    state_ = 2190;
                    return true;
                }
                break;
            case 2190:
                if ('M' == current) {
                    state_ = 2191;
                    return true;
                }
                break;
            case 2191:
                if ('i' == current) {
                    state_ = 2192;
                    return true;
                }
                break;
            case 2192:
                if ('n' == current) {
                    state_ = 2193;
                    return true;
                }
                break;
            case 2193:
                if ('u' == current) {
                    state_ = 2194;
                    return true;
                }
                break;
            case 2194:
                if ('s' == current) {
                    state_ = 2195;
                    return true;
                }
                break;
            case 2195:
                // PlusMinus;
                if (';' == current) {
                    match_ = "\u00B1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2196:
                if ('i' == current) {
                    state_ = 2197;
                    return true;
                }
                else if ('p' == current) {
                    state_ = 2208;
                    return true;
                }
                break;
            case 2197:
                if ('n' == current) {
                    state_ = 2198;
                    return true;
                }
                break;
            case 2198:
                if ('c' == current) {
                    state_ = 2199;
                    return true;
                }
                break;
            case 2199:
                if ('a' == current) {
                    state_ = 2200;
                    return true;
                }
                break;
            case 2200:
                if ('r' == current) {
                    state_ = 2201;
                    return true;
                }
                break;
            case 2201:
                if ('e' == current) {
                    state_ = 2202;
                    return true;
                }
                break;
            case 2202:
                if ('p' == current) {
                    state_ = 2203;
                    return true;
                }
                break;
            case 2203:
                if ('l' == current) {
                    state_ = 2204;
                    return true;
                }
                break;
            case 2204:
                if ('a' == current) {
                    state_ = 2205;
                    return true;
                }
                break;
            case 2205:
                if ('n' == current) {
                    state_ = 2206;
                    return true;
                }
                break;
            case 2206:
                if ('e' == current) {
                    state_ = 2207;
                    return true;
                }
                break;
            case 2207:
                // Poincareplane;
                if (';' == current) {
                    match_ = "\u210C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2208:
                if ('f' == current) {
                    state_ = 2209;
                    return true;
                }
                break;
            case 2209:
                // Popf;
                if (';' == current) {
                    match_ = "\u2119";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2210:
                switch (current) {
                    case ';': // Pr;
                        match_ = "\u2ABB";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e':
                        state_ = 2211;
                        return true;
                    case 'i':
                        state_ = 2237;
                        return true;
                    case 'o':
                        state_ = 2240;
                        return true;
                }
                break;
            case 2211:
                if ('c' == current) {
                    state_ = 2212;
                    return true;
                }
                break;
            case 2212:
                if ('e' == current) {
                    state_ = 2213;
                    return true;
                }
                break;
            case 2213:
                if ('d' == current) {
                    state_ = 2214;
                    return true;
                }
                break;
            case 2214:
                if ('e' == current) {
                    state_ = 2215;
                    return true;
                }
                break;
            case 2215:
                if ('s' == current) {
                    state_ = 2216;
                    return true;
                }
                break;
            case 2216:
                switch (current) {
                    case ';': // Precedes;
                        match_ = "\u227A";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E':
                        state_ = 2217;
                        return true;
                    case 'S':
                        state_ = 2222;
                        return true;
                    case 'T':
                        state_ = 2232;
                        return true;
                }
                break;
            case 2217:
                if ('q' == current) {
                    state_ = 2218;
                    return true;
                }
                break;
            case 2218:
                if ('u' == current) {
                    state_ = 2219;
                    return true;
                }
                break;
            case 2219:
                if ('a' == current) {
                    state_ = 2220;
                    return true;
                }
                break;
            case 2220:
                if ('l' == current) {
                    state_ = 2221;
                    return true;
                }
                break;
            case 2221:
                // PrecedesEqual;
                if (';' == current) {
                    match_ = "\u2AAF";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2222:
                if ('l' == current) {
                    state_ = 2223;
                    return true;
                }
                break;
            case 2223:
                if ('a' == current) {
                    state_ = 2224;
                    return true;
                }
                break;
            case 2224:
                if ('n' == current) {
                    state_ = 2225;
                    return true;
                }
                break;
            case 2225:
                if ('t' == current) {
                    state_ = 2226;
                    return true;
                }
                break;
            case 2226:
                if ('E' == current) {
                    state_ = 2227;
                    return true;
                }
                break;
            case 2227:
                if ('q' == current) {
                    state_ = 2228;
                    return true;
                }
                break;
            case 2228:
                if ('u' == current) {
                    state_ = 2229;
                    return true;
                }
                break;
            case 2229:
                if ('a' == current) {
                    state_ = 2230;
                    return true;
                }
                break;
            case 2230:
                if ('l' == current) {
                    state_ = 2231;
                    return true;
                }
                break;
            case 2231:
                // PrecedesSlantEqual;
                if (';' == current) {
                    match_ = "\u227C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2232:
                if ('i' == current) {
                    state_ = 2233;
                    return true;
                }
                break;
            case 2233:
                if ('l' == current) {
                    state_ = 2234;
                    return true;
                }
                break;
            case 2234:
                if ('d' == current) {
                    state_ = 2235;
                    return true;
                }
                break;
            case 2235:
                if ('e' == current) {
                    state_ = 2236;
                    return true;
                }
                break;
            case 2236:
                // PrecedesTilde;
                if (';' == current) {
                    match_ = "\u227E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2237:
                if ('m' == current) {
                    state_ = 2238;
                    return true;
                }
                break;
            case 2238:
                if ('e' == current) {
                    state_ = 2239;
                    return true;
                }
                break;
            case 2239:
                // Prime;
                if (';' == current) {
                    match_ = "\u2033";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2240:
                if ('d' == current) {
                    state_ = 2241;
                    return true;
                }
                else if ('p' == current) {
                    state_ = 2245;
                    return true;
                }
                break;
            case 2241:
                if ('u' == current) {
                    state_ = 2242;
                    return true;
                }
                break;
            case 2242:
                if ('c' == current) {
                    state_ = 2243;
                    return true;
                }
                break;
            case 2243:
                if ('t' == current) {
                    state_ = 2244;
                    return true;
                }
                break;
            case 2244:
                // Product;
                if (';' == current) {
                    match_ = "\u220F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2245:
                if ('o' == current) {
                    state_ = 2246;
                    return true;
                }
                break;
            case 2246:
                if ('r' == current) {
                    state_ = 2247;
                    return true;
                }
                break;
            case 2247:
                if ('t' == current) {
                    state_ = 2248;
                    return true;
                }
                break;
            case 2248:
                if ('i' == current) {
                    state_ = 2249;
                    return true;
                }
                break;
            case 2249:
                if ('o' == current) {
                    state_ = 2250;
                    return true;
                }
                break;
            case 2250:
                if ('n' == current) {
                    state_ = 2251;
                    return true;
                }
                break;
            case 2251:
                // Proportion;
                if (';' == current) {
                    match_ = "\u2237";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('a' == current) {
                    state_ = 2252;
                    return true;
                }
                break;
            case 2252:
                if ('l' == current) {
                    state_ = 2253;
                    return true;
                }
                break;
            case 2253:
                // Proportional;
                if (';' == current) {
                    match_ = "\u221D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2254:
                if ('c' == current) {
                    state_ = 2255;
                    return true;
                }
                else if ('i' == current) {
                    state_ = 2257;
                    return true;
                }
                break;
            case 2255:
                if ('r' == current) {
                    state_ = 2256;
                    return true;
                }
                break;
            case 2256:
                // Pscr;
                if (';' == current) {
                    match_ = "\uD835\uDCAB";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2257:
                // Psi;
                if (';' == current) {
                    match_ = "\u03A8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2258:
                switch (current) {
                    case 'U':
                        state_ = 2259;
                        return true;
                    case 'f':
                        state_ = 2262;
                        return true;
                    case 'o':
                        state_ = 2264;
                        return true;
                    case 's':
                        state_ = 2267;
                        return true;
                }
                break;
            case 2259:
                if ('O' == current) {
                    state_ = 2260;
                    return true;
                }
                break;
            case 2260:
                // QUOT
                if ('T' == current) {
                    match_ = "\"";
                    matchLength_ = consumedCount_;
                    state_ = 2261;
                    return true;
                }
                break;
            case 2261:
                // QUOT;
                if (';' == current) {
                    match_ = "\"";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2262:
                if ('r' == current) {
                    state_ = 2263;
                    return true;
                }
                break;
            case 2263:
                // Qfr;
                if (';' == current) {
                    match_ = "\uD835\uDD14";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2264:
                if ('p' == current) {
                    state_ = 2265;
                    return true;
                }
                break;
            case 2265:
                if ('f' == current) {
                    state_ = 2266;
                    return true;
                }
                break;
            case 2266:
                // Qopf;
                if (';' == current) {
                    match_ = "\u211A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2267:
                if ('c' == current) {
                    state_ = 2268;
                    return true;
                }
                break;
            case 2268:
                if ('r' == current) {
                    state_ = 2269;
                    return true;
                }
                break;
            case 2269:
                // Qscr;
                if (';' == current) {
                    match_ = "\uD835\uDCAC";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2270:
                switch (current) {
                    case 'B':
                        state_ = 2271;
                        return true;
                    case 'E':
                        state_ = 2275;
                        return true;
                    case 'a':
                        state_ = 2277;
                        return true;
                    case 'c':
                        state_ = 2288;
                        return true;
                    case 'e':
                        state_ = 2298;
                        return true;
                    case 'f':
                        state_ = 2334;
                        return true;
                    case 'h':
                        state_ = 2336;
                        return true;
                    case 'i':
                        state_ = 2338;
                        return true;
                    case 'o':
                        state_ = 2488;
                        return true;
                    case 'r':
                        state_ = 2501;
                        return true;
                    case 's':
                        state_ = 2511;
                        return true;
                    case 'u':
                        state_ = 2515;
                        return true;
                }
                break;
            case 2271:
                if ('a' == current) {
                    state_ = 2272;
                    return true;
                }
                break;
            case 2272:
                if ('r' == current) {
                    state_ = 2273;
                    return true;
                }
                break;
            case 2273:
                if ('r' == current) {
                    state_ = 2274;
                    return true;
                }
                break;
            case 2274:
                // RBarr;
                if (';' == current) {
                    match_ = "\u2910";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2275:
                // REG
                if ('G' == current) {
                    match_ = "\u00AE";
                    matchLength_ = consumedCount_;
                    state_ = 2276;
                    return true;
                }
                break;
            case 2276:
                // REG;
                if (';' == current) {
                    match_ = "\u00AE";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2277:
                switch (current) {
                    case 'c':
                        state_ = 2278;
                        return true;
                    case 'n':
                        state_ = 2282;
                        return true;
                    case 'r':
                        state_ = 2284;
                        return true;
                }
                break;
            case 2278:
                if ('u' == current) {
                    state_ = 2279;
                    return true;
                }
                break;
            case 2279:
                if ('t' == current) {
                    state_ = 2280;
                    return true;
                }
                break;
            case 2280:
                if ('e' == current) {
                    state_ = 2281;
                    return true;
                }
                break;
            case 2281:
                // Racute;
                if (';' == current) {
                    match_ = "\u0154";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2282:
                if ('g' == current) {
                    state_ = 2283;
                    return true;
                }
                break;
            case 2283:
                // Rang;
                if (';' == current) {
                    match_ = "\u27EB";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2284:
                if ('r' == current) {
                    state_ = 2285;
                    return true;
                }
                break;
            case 2285:
                // Rarr;
                if (';' == current) {
                    match_ = "\u21A0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('t' == current) {
                    state_ = 2286;
                    return true;
                }
                break;
            case 2286:
                if ('l' == current) {
                    state_ = 2287;
                    return true;
                }
                break;
            case 2287:
                // Rarrtl;
                if (';' == current) {
                    match_ = "\u2916";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2288:
                switch (current) {
                    case 'a':
                        state_ = 2289;
                        return true;
                    case 'e':
                        state_ = 2293;
                        return true;
                    case 'y':
                        state_ = 2297;
                        return true;
                }
                break;
            case 2289:
                if ('r' == current) {
                    state_ = 2290;
                    return true;
                }
                break;
            case 2290:
                if ('o' == current) {
                    state_ = 2291;
                    return true;
                }
                break;
            case 2291:
                if ('n' == current) {
                    state_ = 2292;
                    return true;
                }
                break;
            case 2292:
                // Rcaron;
                if (';' == current) {
                    match_ = "\u0158";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2293:
                if ('d' == current) {
                    state_ = 2294;
                    return true;
                }
                break;
            case 2294:
                if ('i' == current) {
                    state_ = 2295;
                    return true;
                }
                break;
            case 2295:
                if ('l' == current) {
                    state_ = 2296;
                    return true;
                }
                break;
            case 2296:
                // Rcedil;
                if (';' == current) {
                    match_ = "\u0156";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2297:
                // Rcy;
                if (';' == current) {
                    match_ = "\u0420";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2298:
                // Re;
                if (';' == current) {
                    match_ = "\u211C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('v' == current) {
                    state_ = 2299;
                    return true;
                }
                break;
            case 2299:
                if ('e' == current) {
                    state_ = 2300;
                    return true;
                }
                break;
            case 2300:
                if ('r' == current) {
                    state_ = 2301;
                    return true;
                }
                break;
            case 2301:
                if ('s' == current) {
                    state_ = 2302;
                    return true;
                }
                break;
            case 2302:
                if ('e' == current) {
                    state_ = 2303;
                    return true;
                }
                break;
            case 2303:
                if ('E' == current) {
                    state_ = 2304;
                    return true;
                }
                else if ('U' == current) {
                    state_ = 2321;
                    return true;
                }
                break;
            case 2304:
                if ('l' == current) {
                    state_ = 2305;
                    return true;
                }
                else if ('q' == current) {
                    state_ = 2311;
                    return true;
                }
                break;
            case 2305:
                if ('e' == current) {
                    state_ = 2306;
                    return true;
                }
                break;
            case 2306:
                if ('m' == current) {
                    state_ = 2307;
                    return true;
                }
                break;
            case 2307:
                if ('e' == current) {
                    state_ = 2308;
                    return true;
                }
                break;
            case 2308:
                if ('n' == current) {
                    state_ = 2309;
                    return true;
                }
                break;
            case 2309:
                if ('t' == current) {
                    state_ = 2310;
                    return true;
                }
                break;
            case 2310:
                // ReverseElement;
                if (';' == current) {
                    match_ = "\u220B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2311:
                if ('u' == current) {
                    state_ = 2312;
                    return true;
                }
                break;
            case 2312:
                if ('i' == current) {
                    state_ = 2313;
                    return true;
                }
                break;
            case 2313:
                if ('l' == current) {
                    state_ = 2314;
                    return true;
                }
                break;
            case 2314:
                if ('i' == current) {
                    state_ = 2315;
                    return true;
                }
                break;
            case 2315:
                if ('b' == current) {
                    state_ = 2316;
                    return true;
                }
                break;
            case 2316:
                if ('r' == current) {
                    state_ = 2317;
                    return true;
                }
                break;
            case 2317:
                if ('i' == current) {
                    state_ = 2318;
                    return true;
                }
                break;
            case 2318:
                if ('u' == current) {
                    state_ = 2319;
                    return true;
                }
                break;
            case 2319:
                if ('m' == current) {
                    state_ = 2320;
                    return true;
                }
                break;
            case 2320:
                // ReverseEquilibrium;
                if (';' == current) {
                    match_ = "\u21CB";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2321:
                if ('p' == current) {
                    state_ = 2322;
                    return true;
                }
                break;
            case 2322:
                if ('E' == current) {
                    state_ = 2323;
                    return true;
                }
                break;
            case 2323:
                if ('q' == current) {
                    state_ = 2324;
                    return true;
                }
                break;
            case 2324:
                if ('u' == current) {
                    state_ = 2325;
                    return true;
                }
                break;
            case 2325:
                if ('i' == current) {
                    state_ = 2326;
                    return true;
                }
                break;
            case 2326:
                if ('l' == current) {
                    state_ = 2327;
                    return true;
                }
                break;
            case 2327:
                if ('i' == current) {
                    state_ = 2328;
                    return true;
                }
                break;
            case 2328:
                if ('b' == current) {
                    state_ = 2329;
                    return true;
                }
                break;
            case 2329:
                if ('r' == current) {
                    state_ = 2330;
                    return true;
                }
                break;
            case 2330:
                if ('i' == current) {
                    state_ = 2331;
                    return true;
                }
                break;
            case 2331:
                if ('u' == current) {
                    state_ = 2332;
                    return true;
                }
                break;
            case 2332:
                if ('m' == current) {
                    state_ = 2333;
                    return true;
                }
                break;
            case 2333:
                // ReverseUpEquilibrium;
                if (';' == current) {
                    match_ = "\u296F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2334:
                if ('r' == current) {
                    state_ = 2335;
                    return true;
                }
                break;
            case 2335:
                // Rfr;
                if (';' == current) {
                    match_ = "\u211C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2336:
                if ('o' == current) {
                    state_ = 2337;
                    return true;
                }
                break;
            case 2337:
                // Rho;
                if (';' == current) {
                    match_ = "\u03A1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2338:
                if ('g' == current) {
                    state_ = 2339;
                    return true;
                }
                break;
            case 2339:
                if ('h' == current) {
                    state_ = 2340;
                    return true;
                }
                break;
            case 2340:
                if ('t' == current) {
                    state_ = 2341;
                    return true;
                }
                break;
            case 2341:
                switch (current) {
                    case 'A':
                        state_ = 2342;
                        return true;
                    case 'C':
                        state_ = 2370;
                        return true;
                    case 'D':
                        state_ = 2377;
                        return true;
                    case 'F':
                        state_ = 2410;
                        return true;
                    case 'T':
                        state_ = 2415;
                        return true;
                    case 'U':
                        state_ = 2444;
                        return true;
                    case 'V':
                        state_ = 2474;
                        return true;
                    case 'a':
                        state_ = 2483;
                        return true;
                }
                break;
            case 2342:
                if ('n' == current) {
                    state_ = 2343;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 2354;
                    return true;
                }
                break;
            case 2343:
                if ('g' == current) {
                    state_ = 2344;
                    return true;
                }
                break;
            case 2344:
                if ('l' == current) {
                    state_ = 2345;
                    return true;
                }
                break;
            case 2345:
                if ('e' == current) {
                    state_ = 2346;
                    return true;
                }
                break;
            case 2346:
                if ('B' == current) {
                    state_ = 2347;
                    return true;
                }
                break;
            case 2347:
                if ('r' == current) {
                    state_ = 2348;
                    return true;
                }
                break;
            case 2348:
                if ('a' == current) {
                    state_ = 2349;
                    return true;
                }
                break;
            case 2349:
                if ('c' == current) {
                    state_ = 2350;
                    return true;
                }
                break;
            case 2350:
                if ('k' == current) {
                    state_ = 2351;
                    return true;
                }
                break;
            case 2351:
                if ('e' == current) {
                    state_ = 2352;
                    return true;
                }
                break;
            case 2352:
                if ('t' == current) {
                    state_ = 2353;
                    return true;
                }
                break;
            case 2353:
                // RightAngleBracket;
                if (';' == current) {
                    match_ = "\u27E9";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2354:
                if ('r' == current) {
                    state_ = 2355;
                    return true;
                }
                break;
            case 2355:
                if ('o' == current) {
                    state_ = 2356;
                    return true;
                }
                break;
            case 2356:
                if ('w' == current) {
                    state_ = 2357;
                    return true;
                }
                break;
            case 2357:
                switch (current) {
                    case ';': // RightArrow;
                        match_ = "\u2192";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B':
                        state_ = 2358;
                        return true;
                    case 'L':
                        state_ = 2361;
                        return true;
                }
                break;
            case 2358:
                if ('a' == current) {
                    state_ = 2359;
                    return true;
                }
                break;
            case 2359:
                if ('r' == current) {
                    state_ = 2360;
                    return true;
                }
                break;
            case 2360:
                // RightArrowBar;
                if (';' == current) {
                    match_ = "\u21E5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2361:
                if ('e' == current) {
                    state_ = 2362;
                    return true;
                }
                break;
            case 2362:
                if ('f' == current) {
                    state_ = 2363;
                    return true;
                }
                break;
            case 2363:
                if ('t' == current) {
                    state_ = 2364;
                    return true;
                }
                break;
            case 2364:
                if ('A' == current) {
                    state_ = 2365;
                    return true;
                }
                break;
            case 2365:
                if ('r' == current) {
                    state_ = 2366;
                    return true;
                }
                break;
            case 2366:
                if ('r' == current) {
                    state_ = 2367;
                    return true;
                }
                break;
            case 2367:
                if ('o' == current) {
                    state_ = 2368;
                    return true;
                }
                break;
            case 2368:
                if ('w' == current) {
                    state_ = 2369;
                    return true;
                }
                break;
            case 2369:
                // RightArrowLeftArrow;
                if (';' == current) {
                    match_ = "\u21C4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2370:
                if ('e' == current) {
                    state_ = 2371;
                    return true;
                }
                break;
            case 2371:
                if ('i' == current) {
                    state_ = 2372;
                    return true;
                }
                break;
            case 2372:
                if ('l' == current) {
                    state_ = 2373;
                    return true;
                }
                break;
            case 2373:
                if ('i' == current) {
                    state_ = 2374;
                    return true;
                }
                break;
            case 2374:
                if ('n' == current) {
                    state_ = 2375;
                    return true;
                }
                break;
            case 2375:
                if ('g' == current) {
                    state_ = 2376;
                    return true;
                }
                break;
            case 2376:
                // RightCeiling;
                if (';' == current) {
                    match_ = "\u2309";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2377:
                if ('o' == current) {
                    state_ = 2378;
                    return true;
                }
                break;
            case 2378:
                if ('u' == current) {
                    state_ = 2379;
                    return true;
                }
                else if ('w' == current) {
                    state_ = 2390;
                    return true;
                }
                break;
            case 2379:
                if ('b' == current) {
                    state_ = 2380;
                    return true;
                }
                break;
            case 2380:
                if ('l' == current) {
                    state_ = 2381;
                    return true;
                }
                break;
            case 2381:
                if ('e' == current) {
                    state_ = 2382;
                    return true;
                }
                break;
            case 2382:
                if ('B' == current) {
                    state_ = 2383;
                    return true;
                }
                break;
            case 2383:
                if ('r' == current) {
                    state_ = 2384;
                    return true;
                }
                break;
            case 2384:
                if ('a' == current) {
                    state_ = 2385;
                    return true;
                }
                break;
            case 2385:
                if ('c' == current) {
                    state_ = 2386;
                    return true;
                }
                break;
            case 2386:
                if ('k' == current) {
                    state_ = 2387;
                    return true;
                }
                break;
            case 2387:
                if ('e' == current) {
                    state_ = 2388;
                    return true;
                }
                break;
            case 2388:
                if ('t' == current) {
                    state_ = 2389;
                    return true;
                }
                break;
            case 2389:
                // RightDoubleBracket;
                if (';' == current) {
                    match_ = "\u27E7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2390:
                if ('n' == current) {
                    state_ = 2391;
                    return true;
                }
                break;
            case 2391:
                if ('T' == current) {
                    state_ = 2392;
                    return true;
                }
                else if ('V' == current) {
                    state_ = 2401;
                    return true;
                }
                break;
            case 2392:
                if ('e' == current) {
                    state_ = 2393;
                    return true;
                }
                break;
            case 2393:
                if ('e' == current) {
                    state_ = 2394;
                    return true;
                }
                break;
            case 2394:
                if ('V' == current) {
                    state_ = 2395;
                    return true;
                }
                break;
            case 2395:
                if ('e' == current) {
                    state_ = 2396;
                    return true;
                }
                break;
            case 2396:
                if ('c' == current) {
                    state_ = 2397;
                    return true;
                }
                break;
            case 2397:
                if ('t' == current) {
                    state_ = 2398;
                    return true;
                }
                break;
            case 2398:
                if ('o' == current) {
                    state_ = 2399;
                    return true;
                }
                break;
            case 2399:
                if ('r' == current) {
                    state_ = 2400;
                    return true;
                }
                break;
            case 2400:
                // RightDownTeeVector;
                if (';' == current) {
                    match_ = "\u295D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2401:
                if ('e' == current) {
                    state_ = 2402;
                    return true;
                }
                break;
            case 2402:
                if ('c' == current) {
                    state_ = 2403;
                    return true;
                }
                break;
            case 2403:
                if ('t' == current) {
                    state_ = 2404;
                    return true;
                }
                break;
            case 2404:
                if ('o' == current) {
                    state_ = 2405;
                    return true;
                }
                break;
            case 2405:
                if ('r' == current) {
                    state_ = 2406;
                    return true;
                }
                break;
            case 2406:
                // RightDownVector;
                if (';' == current) {
                    match_ = "\u21C2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('B' == current) {
                    state_ = 2407;
                    return true;
                }
                break;
            case 2407:
                if ('a' == current) {
                    state_ = 2408;
                    return true;
                }
                break;
            case 2408:
                if ('r' == current) {
                    state_ = 2409;
                    return true;
                }
                break;
            case 2409:
                // RightDownVectorBar;
                if (';' == current) {
                    match_ = "\u2955";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2410:
                if ('l' == current) {
                    state_ = 2411;
                    return true;
                }
                break;
            case 2411:
                if ('o' == current) {
                    state_ = 2412;
                    return true;
                }
                break;
            case 2412:
                if ('o' == current) {
                    state_ = 2413;
                    return true;
                }
                break;
            case 2413:
                if ('r' == current) {
                    state_ = 2414;
                    return true;
                }
                break;
            case 2414:
                // RightFloor;
                if (';' == current) {
                    match_ = "\u230B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2415:
                if ('e' == current) {
                    state_ = 2416;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 2429;
                    return true;
                }
                break;
            case 2416:
                if ('e' == current) {
                    state_ = 2417;
                    return true;
                }
                break;
            case 2417:
                switch (current) {
                    case ';': // RightTee;
                        match_ = "\u22A2";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'A':
                        state_ = 2418;
                        return true;
                    case 'V':
                        state_ = 2423;
                        return true;
                }
                break;
            case 2418:
                if ('r' == current) {
                    state_ = 2419;
                    return true;
                }
                break;
            case 2419:
                if ('r' == current) {
                    state_ = 2420;
                    return true;
                }
                break;
            case 2420:
                if ('o' == current) {
                    state_ = 2421;
                    return true;
                }
                break;
            case 2421:
                if ('w' == current) {
                    state_ = 2422;
                    return true;
                }
                break;
            case 2422:
                // RightTeeArrow;
                if (';' == current) {
                    match_ = "\u21A6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2423:
                if ('e' == current) {
                    state_ = 2424;
                    return true;
                }
                break;
            case 2424:
                if ('c' == current) {
                    state_ = 2425;
                    return true;
                }
                break;
            case 2425:
                if ('t' == current) {
                    state_ = 2426;
                    return true;
                }
                break;
            case 2426:
                if ('o' == current) {
                    state_ = 2427;
                    return true;
                }
                break;
            case 2427:
                if ('r' == current) {
                    state_ = 2428;
                    return true;
                }
                break;
            case 2428:
                // RightTeeVector;
                if (';' == current) {
                    match_ = "\u295B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2429:
                if ('i' == current) {
                    state_ = 2430;
                    return true;
                }
                break;
            case 2430:
                if ('a' == current) {
                    state_ = 2431;
                    return true;
                }
                break;
            case 2431:
                if ('n' == current) {
                    state_ = 2432;
                    return true;
                }
                break;
            case 2432:
                if ('g' == current) {
                    state_ = 2433;
                    return true;
                }
                break;
            case 2433:
                if ('l' == current) {
                    state_ = 2434;
                    return true;
                }
                break;
            case 2434:
                if ('e' == current) {
                    state_ = 2435;
                    return true;
                }
                break;
            case 2435:
                switch (current) {
                    case ';': // RightTriangle;
                        match_ = "\u22B3";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B':
                        state_ = 2436;
                        return true;
                    case 'E':
                        state_ = 2439;
                        return true;
                }
                break;
            case 2436:
                if ('a' == current) {
                    state_ = 2437;
                    return true;
                }
                break;
            case 2437:
                if ('r' == current) {
                    state_ = 2438;
                    return true;
                }
                break;
            case 2438:
                // RightTriangleBar;
                if (';' == current) {
                    match_ = "\u29D0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2439:
                if ('q' == current) {
                    state_ = 2440;
                    return true;
                }
                break;
            case 2440:
                if ('u' == current) {
                    state_ = 2441;
                    return true;
                }
                break;
            case 2441:
                if ('a' == current) {
                    state_ = 2442;
                    return true;
                }
                break;
            case 2442:
                if ('l' == current) {
                    state_ = 2443;
                    return true;
                }
                break;
            case 2443:
                // RightTriangleEqual;
                if (';' == current) {
                    match_ = "\u22B5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2444:
                if ('p' == current) {
                    state_ = 2445;
                    return true;
                }
                break;
            case 2445:
                switch (current) {
                    case 'D':
                        state_ = 2446;
                        return true;
                    case 'T':
                        state_ = 2456;
                        return true;
                    case 'V':
                        state_ = 2465;
                        return true;
                }
                break;
            case 2446:
                if ('o' == current) {
                    state_ = 2447;
                    return true;
                }
                break;
            case 2447:
                if ('w' == current) {
                    state_ = 2448;
                    return true;
                }
                break;
            case 2448:
                if ('n' == current) {
                    state_ = 2449;
                    return true;
                }
                break;
            case 2449:
                if ('V' == current) {
                    state_ = 2450;
                    return true;
                }
                break;
            case 2450:
                if ('e' == current) {
                    state_ = 2451;
                    return true;
                }
                break;
            case 2451:
                if ('c' == current) {
                    state_ = 2452;
                    return true;
                }
                break;
            case 2452:
                if ('t' == current) {
                    state_ = 2453;
                    return true;
                }
                break;
            case 2453:
                if ('o' == current) {
                    state_ = 2454;
                    return true;
                }
                break;
            case 2454:
                if ('r' == current) {
                    state_ = 2455;
                    return true;
                }
                break;
            case 2455:
                // RightUpDownVector;
                if (';' == current) {
                    match_ = "\u294F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2456:
                if ('e' == current) {
                    state_ = 2457;
                    return true;
                }
                break;
            case 2457:
                if ('e' == current) {
                    state_ = 2458;
                    return true;
                }
                break;
            case 2458:
                if ('V' == current) {
                    state_ = 2459;
                    return true;
                }
                break;
            case 2459:
                if ('e' == current) {
                    state_ = 2460;
                    return true;
                }
                break;
            case 2460:
                if ('c' == current) {
                    state_ = 2461;
                    return true;
                }
                break;
            case 2461:
                if ('t' == current) {
                    state_ = 2462;
                    return true;
                }
                break;
            case 2462:
                if ('o' == current) {
                    state_ = 2463;
                    return true;
                }
                break;
            case 2463:
                if ('r' == current) {
                    state_ = 2464;
                    return true;
                }
                break;
            case 2464:
                // RightUpTeeVector;
                if (';' == current) {
                    match_ = "\u295C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2465:
                if ('e' == current) {
                    state_ = 2466;
                    return true;
                }
                break;
            case 2466:
                if ('c' == current) {
                    state_ = 2467;
                    return true;
                }
                break;
            case 2467:
                if ('t' == current) {
                    state_ = 2468;
                    return true;
                }
                break;
            case 2468:
                if ('o' == current) {
                    state_ = 2469;
                    return true;
                }
                break;
            case 2469:
                if ('r' == current) {
                    state_ = 2470;
                    return true;
                }
                break;
            case 2470:
                // RightUpVector;
                if (';' == current) {
                    match_ = "\u21BE";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('B' == current) {
                    state_ = 2471;
                    return true;
                }
                break;
            case 2471:
                if ('a' == current) {
                    state_ = 2472;
                    return true;
                }
                break;
            case 2472:
                if ('r' == current) {
                    state_ = 2473;
                    return true;
                }
                break;
            case 2473:
                // RightUpVectorBar;
                if (';' == current) {
                    match_ = "\u2954";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2474:
                if ('e' == current) {
                    state_ = 2475;
                    return true;
                }
                break;
            case 2475:
                if ('c' == current) {
                    state_ = 2476;
                    return true;
                }
                break;
            case 2476:
                if ('t' == current) {
                    state_ = 2477;
                    return true;
                }
                break;
            case 2477:
                if ('o' == current) {
                    state_ = 2478;
                    return true;
                }
                break;
            case 2478:
                if ('r' == current) {
                    state_ = 2479;
                    return true;
                }
                break;
            case 2479:
                // RightVector;
                if (';' == current) {
                    match_ = "\u21C0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('B' == current) {
                    state_ = 2480;
                    return true;
                }
                break;
            case 2480:
                if ('a' == current) {
                    state_ = 2481;
                    return true;
                }
                break;
            case 2481:
                if ('r' == current) {
                    state_ = 2482;
                    return true;
                }
                break;
            case 2482:
                // RightVectorBar;
                if (';' == current) {
                    match_ = "\u2953";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2483:
                if ('r' == current) {
                    state_ = 2484;
                    return true;
                }
                break;
            case 2484:
                if ('r' == current) {
                    state_ = 2485;
                    return true;
                }
                break;
            case 2485:
                if ('o' == current) {
                    state_ = 2486;
                    return true;
                }
                break;
            case 2486:
                if ('w' == current) {
                    state_ = 2487;
                    return true;
                }
                break;
            case 2487:
                // Rightarrow;
                if (';' == current) {
                    match_ = "\u21D2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2488:
                if ('p' == current) {
                    state_ = 2489;
                    return true;
                }
                else if ('u' == current) {
                    state_ = 2491;
                    return true;
                }
                break;
            case 2489:
                if ('f' == current) {
                    state_ = 2490;
                    return true;
                }
                break;
            case 2490:
                // Ropf;
                if (';' == current) {
                    match_ = "\u211D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2491:
                if ('n' == current) {
                    state_ = 2492;
                    return true;
                }
                break;
            case 2492:
                if ('d' == current) {
                    state_ = 2493;
                    return true;
                }
                break;
            case 2493:
                if ('I' == current) {
                    state_ = 2494;
                    return true;
                }
                break;
            case 2494:
                if ('m' == current) {
                    state_ = 2495;
                    return true;
                }
                break;
            case 2495:
                if ('p' == current) {
                    state_ = 2496;
                    return true;
                }
                break;
            case 2496:
                if ('l' == current) {
                    state_ = 2497;
                    return true;
                }
                break;
            case 2497:
                if ('i' == current) {
                    state_ = 2498;
                    return true;
                }
                break;
            case 2498:
                if ('e' == current) {
                    state_ = 2499;
                    return true;
                }
                break;
            case 2499:
                if ('s' == current) {
                    state_ = 2500;
                    return true;
                }
                break;
            case 2500:
                // RoundImplies;
                if (';' == current) {
                    match_ = "\u2970";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2501:
                if ('i' == current) {
                    state_ = 2502;
                    return true;
                }
                break;
            case 2502:
                if ('g' == current) {
                    state_ = 2503;
                    return true;
                }
                break;
            case 2503:
                if ('h' == current) {
                    state_ = 2504;
                    return true;
                }
                break;
            case 2504:
                if ('t' == current) {
                    state_ = 2505;
                    return true;
                }
                break;
            case 2505:
                if ('a' == current) {
                    state_ = 2506;
                    return true;
                }
                break;
            case 2506:
                if ('r' == current) {
                    state_ = 2507;
                    return true;
                }
                break;
            case 2507:
                if ('r' == current) {
                    state_ = 2508;
                    return true;
                }
                break;
            case 2508:
                if ('o' == current) {
                    state_ = 2509;
                    return true;
                }
                break;
            case 2509:
                if ('w' == current) {
                    state_ = 2510;
                    return true;
                }
                break;
            case 2510:
                // Rrightarrow;
                if (';' == current) {
                    match_ = "\u21DB";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2511:
                if ('c' == current) {
                    state_ = 2512;
                    return true;
                }
                else if ('h' == current) {
                    state_ = 2514;
                    return true;
                }
                break;
            case 2512:
                if ('r' == current) {
                    state_ = 2513;
                    return true;
                }
                break;
            case 2513:
                // Rscr;
                if (';' == current) {
                    match_ = "\u211B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2514:
                // Rsh;
                if (';' == current) {
                    match_ = "\u21B1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2515:
                if ('l' == current) {
                    state_ = 2516;
                    return true;
                }
                break;
            case 2516:
                if ('e' == current) {
                    state_ = 2517;
                    return true;
                }
                break;
            case 2517:
                if ('D' == current) {
                    state_ = 2518;
                    return true;
                }
                break;
            case 2518:
                if ('e' == current) {
                    state_ = 2519;
                    return true;
                }
                break;
            case 2519:
                if ('l' == current) {
                    state_ = 2520;
                    return true;
                }
                break;
            case 2520:
                if ('a' == current) {
                    state_ = 2521;
                    return true;
                }
                break;
            case 2521:
                if ('y' == current) {
                    state_ = 2522;
                    return true;
                }
                break;
            case 2522:
                if ('e' == current) {
                    state_ = 2523;
                    return true;
                }
                break;
            case 2523:
                if ('d' == current) {
                    state_ = 2524;
                    return true;
                }
                break;
            case 2524:
                // RuleDelayed;
                if (';' == current) {
                    match_ = "\u29F4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2525:
                switch (current) {
                    case 'H':
                        state_ = 2526;
                        return true;
                    case 'O':
                        state_ = 2533;
                        return true;
                    case 'a':
                        state_ = 2538;
                        return true;
                    case 'c':
                        state_ = 2543;
                        return true;
                    case 'f':
                        state_ = 2556;
                        return true;
                    case 'h':
                        state_ = 2558;
                        return true;
                    case 'i':
                        state_ = 2597;
                        return true;
                    case 'm':
                        state_ = 2601;
                        return true;
                    case 'o':
                        state_ = 2611;
                        return true;
                    case 'q':
                        state_ = 2614;
                        return true;
                    case 's':
                        state_ = 2660;
                        return true;
                    case 't':
                        state_ = 2663;
                        return true;
                    case 'u':
                        state_ = 2666;
                        return true;
                }
                break;
            case 2526:
                if ('C' == current) {
                    state_ = 2527;
                    return true;
                }
                else if ('c' == current) {
                    state_ = 2531;
                    return true;
                }
                break;
            case 2527:
                if ('H' == current) {
                    state_ = 2528;
                    return true;
                }
                break;
            case 2528:
                if ('c' == current) {
                    state_ = 2529;
                    return true;
                }
                break;
            case 2529:
                if ('y' == current) {
                    state_ = 2530;
                    return true;
                }
                break;
            case 2530:
                // SHCHcy;
                if (';' == current) {
                    match_ = "\u0429";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2531:
                if ('y' == current) {
                    state_ = 2532;
                    return true;
                }
                break;
            case 2532:
                // SHcy;
                if (';' == current) {
                    match_ = "\u0428";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2533:
                if ('F' == current) {
                    state_ = 2534;
                    return true;
                }
                break;
            case 2534:
                if ('T' == current) {
                    state_ = 2535;
                    return true;
                }
                break;
            case 2535:
                if ('c' == current) {
                    state_ = 2536;
                    return true;
                }
                break;
            case 2536:
                if ('y' == current) {
                    state_ = 2537;
                    return true;
                }
                break;
            case 2537:
                // SOFTcy;
                if (';' == current) {
                    match_ = "\u042C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2538:
                if ('c' == current) {
                    state_ = 2539;
                    return true;
                }
                break;
            case 2539:
                if ('u' == current) {
                    state_ = 2540;
                    return true;
                }
                break;
            case 2540:
                if ('t' == current) {
                    state_ = 2541;
                    return true;
                }
                break;
            case 2541:
                if ('e' == current) {
                    state_ = 2542;
                    return true;
                }
                break;
            case 2542:
                // Sacute;
                if (';' == current) {
                    match_ = "\u015A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2543:
                switch (current) {
                    case ';': // Sc;
                        match_ = "\u2ABC";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'a':
                        state_ = 2544;
                        return true;
                    case 'e':
                        state_ = 2548;
                        return true;
                    case 'i':
                        state_ = 2552;
                        return true;
                    case 'y':
                        state_ = 2555;
                        return true;
                }
                break;
            case 2544:
                if ('r' == current) {
                    state_ = 2545;
                    return true;
                }
                break;
            case 2545:
                if ('o' == current) {
                    state_ = 2546;
                    return true;
                }
                break;
            case 2546:
                if ('n' == current) {
                    state_ = 2547;
                    return true;
                }
                break;
            case 2547:
                // Scaron;
                if (';' == current) {
                    match_ = "\u0160";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2548:
                if ('d' == current) {
                    state_ = 2549;
                    return true;
                }
                break;
            case 2549:
                if ('i' == current) {
                    state_ = 2550;
                    return true;
                }
                break;
            case 2550:
                if ('l' == current) {
                    state_ = 2551;
                    return true;
                }
                break;
            case 2551:
                // Scedil;
                if (';' == current) {
                    match_ = "\u015E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2552:
                if ('r' == current) {
                    state_ = 2553;
                    return true;
                }
                break;
            case 2553:
                if ('c' == current) {
                    state_ = 2554;
                    return true;
                }
                break;
            case 2554:
                // Scirc;
                if (';' == current) {
                    match_ = "\u015C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2555:
                // Scy;
                if (';' == current) {
                    match_ = "\u0421";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2556:
                if ('r' == current) {
                    state_ = 2557;
                    return true;
                }
                break;
            case 2557:
                // Sfr;
                if (';' == current) {
                    match_ = "\uD835\uDD16";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2558:
                if ('o' == current) {
                    state_ = 2559;
                    return true;
                }
                break;
            case 2559:
                if ('r' == current) {
                    state_ = 2560;
                    return true;
                }
                break;
            case 2560:
                if ('t' == current) {
                    state_ = 2561;
                    return true;
                }
                break;
            case 2561:
                switch (current) {
                    case 'D':
                        state_ = 2562;
                        return true;
                    case 'L':
                        state_ = 2571;
                        return true;
                    case 'R':
                        state_ = 2580;
                        return true;
                    case 'U':
                        state_ = 2590;
                        return true;
                }
                break;
            case 2562:
                if ('o' == current) {
                    state_ = 2563;
                    return true;
                }
                break;
            case 2563:
                if ('w' == current) {
                    state_ = 2564;
                    return true;
                }
                break;
            case 2564:
                if ('n' == current) {
                    state_ = 2565;
                    return true;
                }
                break;
            case 2565:
                if ('A' == current) {
                    state_ = 2566;
                    return true;
                }
                break;
            case 2566:
                if ('r' == current) {
                    state_ = 2567;
                    return true;
                }
                break;
            case 2567:
                if ('r' == current) {
                    state_ = 2568;
                    return true;
                }
                break;
            case 2568:
                if ('o' == current) {
                    state_ = 2569;
                    return true;
                }
                break;
            case 2569:
                if ('w' == current) {
                    state_ = 2570;
                    return true;
                }
                break;
            case 2570:
                // ShortDownArrow;
                if (';' == current) {
                    match_ = "\u2193";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2571:
                if ('e' == current) {
                    state_ = 2572;
                    return true;
                }
                break;
            case 2572:
                if ('f' == current) {
                    state_ = 2573;
                    return true;
                }
                break;
            case 2573:
                if ('t' == current) {
                    state_ = 2574;
                    return true;
                }
                break;
            case 2574:
                if ('A' == current) {
                    state_ = 2575;
                    return true;
                }
                break;
            case 2575:
                if ('r' == current) {
                    state_ = 2576;
                    return true;
                }
                break;
            case 2576:
                if ('r' == current) {
                    state_ = 2577;
                    return true;
                }
                break;
            case 2577:
                if ('o' == current) {
                    state_ = 2578;
                    return true;
                }
                break;
            case 2578:
                if ('w' == current) {
                    state_ = 2579;
                    return true;
                }
                break;
            case 2579:
                // ShortLeftArrow;
                if (';' == current) {
                    match_ = "\u2190";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2580:
                if ('i' == current) {
                    state_ = 2581;
                    return true;
                }
                break;
            case 2581:
                if ('g' == current) {
                    state_ = 2582;
                    return true;
                }
                break;
            case 2582:
                if ('h' == current) {
                    state_ = 2583;
                    return true;
                }
                break;
            case 2583:
                if ('t' == current) {
                    state_ = 2584;
                    return true;
                }
                break;
            case 2584:
                if ('A' == current) {
                    state_ = 2585;
                    return true;
                }
                break;
            case 2585:
                if ('r' == current) {
                    state_ = 2586;
                    return true;
                }
                break;
            case 2586:
                if ('r' == current) {
                    state_ = 2587;
                    return true;
                }
                break;
            case 2587:
                if ('o' == current) {
                    state_ = 2588;
                    return true;
                }
                break;
            case 2588:
                if ('w' == current) {
                    state_ = 2589;
                    return true;
                }
                break;
            case 2589:
                // ShortRightArrow;
                if (';' == current) {
                    match_ = "\u2192";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2590:
                if ('p' == current) {
                    state_ = 2591;
                    return true;
                }
                break;
            case 2591:
                if ('A' == current) {
                    state_ = 2592;
                    return true;
                }
                break;
            case 2592:
                if ('r' == current) {
                    state_ = 2593;
                    return true;
                }
                break;
            case 2593:
                if ('r' == current) {
                    state_ = 2594;
                    return true;
                }
                break;
            case 2594:
                if ('o' == current) {
                    state_ = 2595;
                    return true;
                }
                break;
            case 2595:
                if ('w' == current) {
                    state_ = 2596;
                    return true;
                }
                break;
            case 2596:
                // ShortUpArrow;
                if (';' == current) {
                    match_ = "\u2191";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2597:
                if ('g' == current) {
                    state_ = 2598;
                    return true;
                }
                break;
            case 2598:
                if ('m' == current) {
                    state_ = 2599;
                    return true;
                }
                break;
            case 2599:
                if ('a' == current) {
                    state_ = 2600;
                    return true;
                }
                break;
            case 2600:
                // Sigma;
                if (';' == current) {
                    match_ = "\u03A3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2601:
                if ('a' == current) {
                    state_ = 2602;
                    return true;
                }
                break;
            case 2602:
                if ('l' == current) {
                    state_ = 2603;
                    return true;
                }
                break;
            case 2603:
                if ('l' == current) {
                    state_ = 2604;
                    return true;
                }
                break;
            case 2604:
                if ('C' == current) {
                    state_ = 2605;
                    return true;
                }
                break;
            case 2605:
                if ('i' == current) {
                    state_ = 2606;
                    return true;
                }
                break;
            case 2606:
                if ('r' == current) {
                    state_ = 2607;
                    return true;
                }
                break;
            case 2607:
                if ('c' == current) {
                    state_ = 2608;
                    return true;
                }
                break;
            case 2608:
                if ('l' == current) {
                    state_ = 2609;
                    return true;
                }
                break;
            case 2609:
                if ('e' == current) {
                    state_ = 2610;
                    return true;
                }
                break;
            case 2610:
                // SmallCircle;
                if (';' == current) {
                    match_ = "\u2218";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2611:
                if ('p' == current) {
                    state_ = 2612;
                    return true;
                }
                break;
            case 2612:
                if ('f' == current) {
                    state_ = 2613;
                    return true;
                }
                break;
            case 2613:
                // Sopf;
                if (';' == current) {
                    match_ = "\uD835\uDD4A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2614:
                if ('r' == current) {
                    state_ = 2615;
                    return true;
                }
                else if ('u' == current) {
                    state_ = 2617;
                    return true;
                }
                break;
            case 2615:
                if ('t' == current) {
                    state_ = 2616;
                    return true;
                }
                break;
            case 2616:
                // Sqrt;
                if (';' == current) {
                    match_ = "\u221A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2617:
                if ('a' == current) {
                    state_ = 2618;
                    return true;
                }
                break;
            case 2618:
                if ('r' == current) {
                    state_ = 2619;
                    return true;
                }
                break;
            case 2619:
                if ('e' == current) {
                    state_ = 2620;
                    return true;
                }
                break;
            case 2620:
                switch (current) {
                    case ';': // Square;
                        match_ = "\u25A1";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'I':
                        state_ = 2621;
                        return true;
                    case 'S':
                        state_ = 2633;
                        return true;
                    case 'U':
                        state_ = 2655;
                        return true;
                }
                break;
            case 2621:
                if ('n' == current) {
                    state_ = 2622;
                    return true;
                }
                break;
            case 2622:
                if ('t' == current) {
                    state_ = 2623;
                    return true;
                }
                break;
            case 2623:
                if ('e' == current) {
                    state_ = 2624;
                    return true;
                }
                break;
            case 2624:
                if ('r' == current) {
                    state_ = 2625;
                    return true;
                }
                break;
            case 2625:
                if ('s' == current) {
                    state_ = 2626;
                    return true;
                }
                break;
            case 2626:
                if ('e' == current) {
                    state_ = 2627;
                    return true;
                }
                break;
            case 2627:
                if ('c' == current) {
                    state_ = 2628;
                    return true;
                }
                break;
            case 2628:
                if ('t' == current) {
                    state_ = 2629;
                    return true;
                }
                break;
            case 2629:
                if ('i' == current) {
                    state_ = 2630;
                    return true;
                }
                break;
            case 2630:
                if ('o' == current) {
                    state_ = 2631;
                    return true;
                }
                break;
            case 2631:
                if ('n' == current) {
                    state_ = 2632;
                    return true;
                }
                break;
            case 2632:
                // SquareIntersection;
                if (';' == current) {
                    match_ = "\u2293";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2633:
                if ('u' == current) {
                    state_ = 2634;
                    return true;
                }
                break;
            case 2634:
                if ('b' == current) {
                    state_ = 2635;
                    return true;
                }
                else if ('p' == current) {
                    state_ = 2644;
                    return true;
                }
                break;
            case 2635:
                if ('s' == current) {
                    state_ = 2636;
                    return true;
                }
                break;
            case 2636:
                if ('e' == current) {
                    state_ = 2637;
                    return true;
                }
                break;
            case 2637:
                if ('t' == current) {
                    state_ = 2638;
                    return true;
                }
                break;
            case 2638:
                // SquareSubset;
                if (';' == current) {
                    match_ = "\u228F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('E' == current) {
                    state_ = 2639;
                    return true;
                }
                break;
            case 2639:
                if ('q' == current) {
                    state_ = 2640;
                    return true;
                }
                break;
            case 2640:
                if ('u' == current) {
                    state_ = 2641;
                    return true;
                }
                break;
            case 2641:
                if ('a' == current) {
                    state_ = 2642;
                    return true;
                }
                break;
            case 2642:
                if ('l' == current) {
                    state_ = 2643;
                    return true;
                }
                break;
            case 2643:
                // SquareSubsetEqual;
                if (';' == current) {
                    match_ = "\u2291";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2644:
                if ('e' == current) {
                    state_ = 2645;
                    return true;
                }
                break;
            case 2645:
                if ('r' == current) {
                    state_ = 2646;
                    return true;
                }
                break;
            case 2646:
                if ('s' == current) {
                    state_ = 2647;
                    return true;
                }
                break;
            case 2647:
                if ('e' == current) {
                    state_ = 2648;
                    return true;
                }
                break;
            case 2648:
                if ('t' == current) {
                    state_ = 2649;
                    return true;
                }
                break;
            case 2649:
                // SquareSuperset;
                if (';' == current) {
                    match_ = "\u2290";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('E' == current) {
                    state_ = 2650;
                    return true;
                }
                break;
            case 2650:
                if ('q' == current) {
                    state_ = 2651;
                    return true;
                }
                break;
            case 2651:
                if ('u' == current) {
                    state_ = 2652;
                    return true;
                }
                break;
            case 2652:
                if ('a' == current) {
                    state_ = 2653;
                    return true;
                }
                break;
            case 2653:
                if ('l' == current) {
                    state_ = 2654;
                    return true;
                }
                break;
            case 2654:
                // SquareSupersetEqual;
                if (';' == current) {
                    match_ = "\u2292";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2655:
                if ('n' == current) {
                    state_ = 2656;
                    return true;
                }
                break;
            case 2656:
                if ('i' == current) {
                    state_ = 2657;
                    return true;
                }
                break;
            case 2657:
                if ('o' == current) {
                    state_ = 2658;
                    return true;
                }
                break;
            case 2658:
                if ('n' == current) {
                    state_ = 2659;
                    return true;
                }
                break;
            case 2659:
                // SquareUnion;
                if (';' == current) {
                    match_ = "\u2294";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2660:
                if ('c' == current) {
                    state_ = 2661;
                    return true;
                }
                break;
            case 2661:
                if ('r' == current) {
                    state_ = 2662;
                    return true;
                }
                break;
            case 2662:
                // Sscr;
                if (';' == current) {
                    match_ = "\uD835\uDCAE";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2663:
                if ('a' == current) {
                    state_ = 2664;
                    return true;
                }
                break;
            case 2664:
                if ('r' == current) {
                    state_ = 2665;
                    return true;
                }
                break;
            case 2665:
                // Star;
                if (';' == current) {
                    match_ = "\u22C6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2666:
                switch (current) {
                    case 'b':
                        state_ = 2667;
                        return true;
                    case 'c':
                        state_ = 2676;
                        return true;
                    case 'm':
                        state_ = 2707;
                        return true;
                    case 'p':
                        state_ = 2708;
                        return true;
                }
                break;
            case 2667:
                // Sub;
                if (';' == current) {
                    match_ = "\u22D0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('s' == current) {
                    state_ = 2668;
                    return true;
                }
                break;
            case 2668:
                if ('e' == current) {
                    state_ = 2669;
                    return true;
                }
                break;
            case 2669:
                if ('t' == current) {
                    state_ = 2670;
                    return true;
                }
                break;
            case 2670:
                // Subset;
                if (';' == current) {
                    match_ = "\u22D0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('E' == current) {
                    state_ = 2671;
                    return true;
                }
                break;
            case 2671:
                if ('q' == current) {
                    state_ = 2672;
                    return true;
                }
                break;
            case 2672:
                if ('u' == current) {
                    state_ = 2673;
                    return true;
                }
                break;
            case 2673:
                if ('a' == current) {
                    state_ = 2674;
                    return true;
                }
                break;
            case 2674:
                if ('l' == current) {
                    state_ = 2675;
                    return true;
                }
                break;
            case 2675:
                // SubsetEqual;
                if (';' == current) {
                    match_ = "\u2286";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2676:
                if ('c' == current) {
                    state_ = 2677;
                    return true;
                }
                else if ('h' == current) {
                    state_ = 2702;
                    return true;
                }
                break;
            case 2677:
                if ('e' == current) {
                    state_ = 2678;
                    return true;
                }
                break;
            case 2678:
                if ('e' == current) {
                    state_ = 2679;
                    return true;
                }
                break;
            case 2679:
                if ('d' == current) {
                    state_ = 2680;
                    return true;
                }
                break;
            case 2680:
                if ('s' == current) {
                    state_ = 2681;
                    return true;
                }
                break;
            case 2681:
                switch (current) {
                    case ';': // Succeeds;
                        match_ = "\u227B";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E':
                        state_ = 2682;
                        return true;
                    case 'S':
                        state_ = 2687;
                        return true;
                    case 'T':
                        state_ = 2697;
                        return true;
                }
                break;
            case 2682:
                if ('q' == current) {
                    state_ = 2683;
                    return true;
                }
                break;
            case 2683:
                if ('u' == current) {
                    state_ = 2684;
                    return true;
                }
                break;
            case 2684:
                if ('a' == current) {
                    state_ = 2685;
                    return true;
                }
                break;
            case 2685:
                if ('l' == current) {
                    state_ = 2686;
                    return true;
                }
                break;
            case 2686:
                // SucceedsEqual;
                if (';' == current) {
                    match_ = "\u2AB0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2687:
                if ('l' == current) {
                    state_ = 2688;
                    return true;
                }
                break;
            case 2688:
                if ('a' == current) {
                    state_ = 2689;
                    return true;
                }
                break;
            case 2689:
                if ('n' == current) {
                    state_ = 2690;
                    return true;
                }
                break;
            case 2690:
                if ('t' == current) {
                    state_ = 2691;
                    return true;
                }
                break;
            case 2691:
                if ('E' == current) {
                    state_ = 2692;
                    return true;
                }
                break;
            case 2692:
                if ('q' == current) {
                    state_ = 2693;
                    return true;
                }
                break;
            case 2693:
                if ('u' == current) {
                    state_ = 2694;
                    return true;
                }
                break;
            case 2694:
                if ('a' == current) {
                    state_ = 2695;
                    return true;
                }
                break;
            case 2695:
                if ('l' == current) {
                    state_ = 2696;
                    return true;
                }
                break;
            case 2696:
                // SucceedsSlantEqual;
                if (';' == current) {
                    match_ = "\u227D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2697:
                if ('i' == current) {
                    state_ = 2698;
                    return true;
                }
                break;
            case 2698:
                if ('l' == current) {
                    state_ = 2699;
                    return true;
                }
                break;
            case 2699:
                if ('d' == current) {
                    state_ = 2700;
                    return true;
                }
                break;
            case 2700:
                if ('e' == current) {
                    state_ = 2701;
                    return true;
                }
                break;
            case 2701:
                // SucceedsTilde;
                if (';' == current) {
                    match_ = "\u227F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2702:
                if ('T' == current) {
                    state_ = 2703;
                    return true;
                }
                break;
            case 2703:
                if ('h' == current) {
                    state_ = 2704;
                    return true;
                }
                break;
            case 2704:
                if ('a' == current) {
                    state_ = 2705;
                    return true;
                }
                break;
            case 2705:
                if ('t' == current) {
                    state_ = 2706;
                    return true;
                }
                break;
            case 2706:
                // SuchThat;
                if (';' == current) {
                    match_ = "\u220B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2707:
                // Sum;
                if (';' == current) {
                    match_ = "\u2211";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2708:
                switch (current) {
                    case ';': // Sup;
                        match_ = "\u22D1";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e':
                        state_ = 2709;
                        return true;
                    case 's':
                        state_ = 2719;
                        return true;
                }
                break;
            case 2709:
                if ('r' == current) {
                    state_ = 2710;
                    return true;
                }
                break;
            case 2710:
                if ('s' == current) {
                    state_ = 2711;
                    return true;
                }
                break;
            case 2711:
                if ('e' == current) {
                    state_ = 2712;
                    return true;
                }
                break;
            case 2712:
                if ('t' == current) {
                    state_ = 2713;
                    return true;
                }
                break;
            case 2713:
                // Superset;
                if (';' == current) {
                    match_ = "\u2283";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('E' == current) {
                    state_ = 2714;
                    return true;
                }
                break;
            case 2714:
                if ('q' == current) {
                    state_ = 2715;
                    return true;
                }
                break;
            case 2715:
                if ('u' == current) {
                    state_ = 2716;
                    return true;
                }
                break;
            case 2716:
                if ('a' == current) {
                    state_ = 2717;
                    return true;
                }
                break;
            case 2717:
                if ('l' == current) {
                    state_ = 2718;
                    return true;
                }
                break;
            case 2718:
                // SupersetEqual;
                if (';' == current) {
                    match_ = "\u2287";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2719:
                if ('e' == current) {
                    state_ = 2720;
                    return true;
                }
                break;
            case 2720:
                if ('t' == current) {
                    state_ = 2721;
                    return true;
                }
                break;
            case 2721:
                // Supset;
                if (';' == current) {
                    match_ = "\u22D1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2722:
                switch (current) {
                    case 'H':
                        state_ = 2723;
                        return true;
                    case 'R':
                        state_ = 2727;
                        return true;
                    case 'S':
                        state_ = 2731;
                        return true;
                    case 'a':
                        state_ = 2737;
                        return true;
                    case 'c':
                        state_ = 2740;
                        return true;
                    case 'f':
                        state_ = 2750;
                        return true;
                    case 'h':
                        state_ = 2752;
                        return true;
                    case 'i':
                        state_ = 2776;
                        return true;
                    case 'o':
                        state_ = 2799;
                        return true;
                    case 'r':
                        state_ = 2802;
                        return true;
                    case 's':
                        state_ = 2810;
                        return true;
                }
                break;
            case 2723:
                if ('O' == current) {
                    state_ = 2724;
                    return true;
                }
                break;
            case 2724:
                if ('R' == current) {
                    state_ = 2725;
                    return true;
                }
                break;
            case 2725:
                // THORN
                if ('N' == current) {
                    match_ = "\u00DE";
                    matchLength_ = consumedCount_;
                    state_ = 2726;
                    return true;
                }
                break;
            case 2726:
                // THORN;
                if (';' == current) {
                    match_ = "\u00DE";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2727:
                if ('A' == current) {
                    state_ = 2728;
                    return true;
                }
                break;
            case 2728:
                if ('D' == current) {
                    state_ = 2729;
                    return true;
                }
                break;
            case 2729:
                if ('E' == current) {
                    state_ = 2730;
                    return true;
                }
                break;
            case 2730:
                // TRADE;
                if (';' == current) {
                    match_ = "\u2122";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2731:
                if ('H' == current) {
                    state_ = 2732;
                    return true;
                }
                else if ('c' == current) {
                    state_ = 2735;
                    return true;
                }
                break;
            case 2732:
                if ('c' == current) {
                    state_ = 2733;
                    return true;
                }
                break;
            case 2733:
                if ('y' == current) {
                    state_ = 2734;
                    return true;
                }
                break;
            case 2734:
                // TSHcy;
                if (';' == current) {
                    match_ = "\u040B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2735:
                if ('y' == current) {
                    state_ = 2736;
                    return true;
                }
                break;
            case 2736:
                // TScy;
                if (';' == current) {
                    match_ = "\u0426";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2737:
                if ('b' == current) {
                    state_ = 2738;
                    return true;
                }
                else if ('u' == current) {
                    state_ = 2739;
                    return true;
                }
                break;
            case 2738:
                // Tab;
                if (';' == current) {
                    match_ = "\u0009";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2739:
                // Tau;
                if (';' == current) {
                    match_ = "\u03A4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2740:
                switch (current) {
                    case 'a':
                        state_ = 2741;
                        return true;
                    case 'e':
                        state_ = 2745;
                        return true;
                    case 'y':
                        state_ = 2749;
                        return true;
                }
                break;
            case 2741:
                if ('r' == current) {
                    state_ = 2742;
                    return true;
                }
                break;
            case 2742:
                if ('o' == current) {
                    state_ = 2743;
                    return true;
                }
                break;
            case 2743:
                if ('n' == current) {
                    state_ = 2744;
                    return true;
                }
                break;
            case 2744:
                // Tcaron;
                if (';' == current) {
                    match_ = "\u0164";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2745:
                if ('d' == current) {
                    state_ = 2746;
                    return true;
                }
                break;
            case 2746:
                if ('i' == current) {
                    state_ = 2747;
                    return true;
                }
                break;
            case 2747:
                if ('l' == current) {
                    state_ = 2748;
                    return true;
                }
                break;
            case 2748:
                // Tcedil;
                if (';' == current) {
                    match_ = "\u0162";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2749:
                // Tcy;
                if (';' == current) {
                    match_ = "\u0422";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2750:
                if ('r' == current) {
                    state_ = 2751;
                    return true;
                }
                break;
            case 2751:
                // Tfr;
                if (';' == current) {
                    match_ = "\uD835\uDD17";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2752:
                if ('e' == current) {
                    state_ = 2753;
                    return true;
                }
                else if ('i' == current) {
                    state_ = 2762;
                    return true;
                }
                break;
            case 2753:
                if ('r' == current) {
                    state_ = 2754;
                    return true;
                }
                else if ('t' == current) {
                    state_ = 2760;
                    return true;
                }
                break;
            case 2754:
                if ('e' == current) {
                    state_ = 2755;
                    return true;
                }
                break;
            case 2755:
                if ('f' == current) {
                    state_ = 2756;
                    return true;
                }
                break;
            case 2756:
                if ('o' == current) {
                    state_ = 2757;
                    return true;
                }
                break;
            case 2757:
                if ('r' == current) {
                    state_ = 2758;
                    return true;
                }
                break;
            case 2758:
                if ('e' == current) {
                    state_ = 2759;
                    return true;
                }
                break;
            case 2759:
                // Therefore;
                if (';' == current) {
                    match_ = "\u2234";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2760:
                if ('a' == current) {
                    state_ = 2761;
                    return true;
                }
                break;
            case 2761:
                // Theta;
                if (';' == current) {
                    match_ = "\u0398";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2762:
                if ('c' == current) {
                    state_ = 2763;
                    return true;
                }
                else if ('n' == current) {
                    state_ = 2770;
                    return true;
                }
                break;
            case 2763:
                if ('k' == current) {
                    state_ = 2764;
                    return true;
                }
                break;
            case 2764:
                if ('S' == current) {
                    state_ = 2765;
                    return true;
                }
                break;
            case 2765:
                if ('p' == current) {
                    state_ = 2766;
                    return true;
                }
                break;
            case 2766:
                if ('a' == current) {
                    state_ = 2767;
                    return true;
                }
                break;
            case 2767:
                if ('c' == current) {
                    state_ = 2768;
                    return true;
                }
                break;
            case 2768:
                if ('e' == current) {
                    state_ = 2769;
                    return true;
                }
                break;
            case 2769:
                // ThickSpace;
                if (';' == current) {
                    match_ = "\u205F\u200A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2770:
                if ('S' == current) {
                    state_ = 2771;
                    return true;
                }
                break;
            case 2771:
                if ('p' == current) {
                    state_ = 2772;
                    return true;
                }
                break;
            case 2772:
                if ('a' == current) {
                    state_ = 2773;
                    return true;
                }
                break;
            case 2773:
                if ('c' == current) {
                    state_ = 2774;
                    return true;
                }
                break;
            case 2774:
                if ('e' == current) {
                    state_ = 2775;
                    return true;
                }
                break;
            case 2775:
                // ThinSpace;
                if (';' == current) {
                    match_ = "\u2009";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2776:
                if ('l' == current) {
                    state_ = 2777;
                    return true;
                }
                break;
            case 2777:
                if ('d' == current) {
                    state_ = 2778;
                    return true;
                }
                break;
            case 2778:
                if ('e' == current) {
                    state_ = 2779;
                    return true;
                }
                break;
            case 2779:
                switch (current) {
                    case ';': // Tilde;
                        match_ = "\u223C";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E':
                        state_ = 2780;
                        return true;
                    case 'F':
                        state_ = 2785;
                        return true;
                    case 'T':
                        state_ = 2794;
                        return true;
                }
                break;
            case 2780:
                if ('q' == current) {
                    state_ = 2781;
                    return true;
                }
                break;
            case 2781:
                if ('u' == current) {
                    state_ = 2782;
                    return true;
                }
                break;
            case 2782:
                if ('a' == current) {
                    state_ = 2783;
                    return true;
                }
                break;
            case 2783:
                if ('l' == current) {
                    state_ = 2784;
                    return true;
                }
                break;
            case 2784:
                // TildeEqual;
                if (';' == current) {
                    match_ = "\u2243";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2785:
                if ('u' == current) {
                    state_ = 2786;
                    return true;
                }
                break;
            case 2786:
                if ('l' == current) {
                    state_ = 2787;
                    return true;
                }
                break;
            case 2787:
                if ('l' == current) {
                    state_ = 2788;
                    return true;
                }
                break;
            case 2788:
                if ('E' == current) {
                    state_ = 2789;
                    return true;
                }
                break;
            case 2789:
                if ('q' == current) {
                    state_ = 2790;
                    return true;
                }
                break;
            case 2790:
                if ('u' == current) {
                    state_ = 2791;
                    return true;
                }
                break;
            case 2791:
                if ('a' == current) {
                    state_ = 2792;
                    return true;
                }
                break;
            case 2792:
                if ('l' == current) {
                    state_ = 2793;
                    return true;
                }
                break;
            case 2793:
                // TildeFullEqual;
                if (';' == current) {
                    match_ = "\u2245";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2794:
                if ('i' == current) {
                    state_ = 2795;
                    return true;
                }
                break;
            case 2795:
                if ('l' == current) {
                    state_ = 2796;
                    return true;
                }
                break;
            case 2796:
                if ('d' == current) {
                    state_ = 2797;
                    return true;
                }
                break;
            case 2797:
                if ('e' == current) {
                    state_ = 2798;
                    return true;
                }
                break;
            case 2798:
                // TildeTilde;
                if (';' == current) {
                    match_ = "\u2248";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2799:
                if ('p' == current) {
                    state_ = 2800;
                    return true;
                }
                break;
            case 2800:
                if ('f' == current) {
                    state_ = 2801;
                    return true;
                }
                break;
            case 2801:
                // Topf;
                if (';' == current) {
                    match_ = "\uD835\uDD4B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2802:
                if ('i' == current) {
                    state_ = 2803;
                    return true;
                }
                break;
            case 2803:
                if ('p' == current) {
                    state_ = 2804;
                    return true;
                }
                break;
            case 2804:
                if ('l' == current) {
                    state_ = 2805;
                    return true;
                }
                break;
            case 2805:
                if ('e' == current) {
                    state_ = 2806;
                    return true;
                }
                break;
            case 2806:
                if ('D' == current) {
                    state_ = 2807;
                    return true;
                }
                break;
            case 2807:
                if ('o' == current) {
                    state_ = 2808;
                    return true;
                }
                break;
            case 2808:
                if ('t' == current) {
                    state_ = 2809;
                    return true;
                }
                break;
            case 2809:
                // TripleDot;
                if (';' == current) {
                    match_ = "\u20DB";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2810:
                if ('c' == current) {
                    state_ = 2811;
                    return true;
                }
                else if ('t' == current) {
                    state_ = 2813;
                    return true;
                }
                break;
            case 2811:
                if ('r' == current) {
                    state_ = 2812;
                    return true;
                }
                break;
            case 2812:
                // Tscr;
                if (';' == current) {
                    match_ = "\uD835\uDCAF";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2813:
                if ('r' == current) {
                    state_ = 2814;
                    return true;
                }
                break;
            case 2814:
                if ('o' == current) {
                    state_ = 2815;
                    return true;
                }
                break;
            case 2815:
                if ('k' == current) {
                    state_ = 2816;
                    return true;
                }
                break;
            case 2816:
                // Tstrok;
                if (';' == current) {
                    match_ = "\u0166";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2817:
                switch (current) {
                    case 'a':
                        state_ = 2818;
                        return true;
                    case 'b':
                        state_ = 2829;
                        return true;
                    case 'c':
                        state_ = 2836;
                        return true;
                    case 'd':
                        state_ = 2841;
                        return true;
                    case 'f':
                        state_ = 2846;
                        return true;
                    case 'g':
                        state_ = 2848;
                        return true;
                    case 'm':
                        state_ = 2853;
                        return true;
                    case 'n':
                        state_ = 2857;
                        return true;
                    case 'o':
                        state_ = 2889;
                        return true;
                    case 'p':
                        state_ = 2895;
                        return true;
                    case 'r':
                        state_ = 2982;
                        return true;
                    case 's':
                        state_ = 2986;
                        return true;
                    case 't':
                        state_ = 2989;
                        return true;
                    case 'u':
                        state_ = 2994;
                        return true;
                }
                break;
            case 2818:
                if ('c' == current) {
                    state_ = 2819;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 2823;
                    return true;
                }
                break;
            case 2819:
                if ('u' == current) {
                    state_ = 2820;
                    return true;
                }
                break;
            case 2820:
                if ('t' == current) {
                    state_ = 2821;
                    return true;
                }
                break;
            case 2821:
                // Uacute
                if ('e' == current) {
                    match_ = "\u00DA";
                    matchLength_ = consumedCount_;
                    state_ = 2822;
                    return true;
                }
                break;
            case 2822:
                // Uacute;
                if (';' == current) {
                    match_ = "\u00DA";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2823:
                if ('r' == current) {
                    state_ = 2824;
                    return true;
                }
                break;
            case 2824:
                // Uarr;
                if (';' == current) {
                    match_ = "\u219F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('o' == current) {
                    state_ = 2825;
                    return true;
                }
                break;
            case 2825:
                if ('c' == current) {
                    state_ = 2826;
                    return true;
                }
                break;
            case 2826:
                if ('i' == current) {
                    state_ = 2827;
                    return true;
                }
                break;
            case 2827:
                if ('r' == current) {
                    state_ = 2828;
                    return true;
                }
                break;
            case 2828:
                // Uarrocir;
                if (';' == current) {
                    match_ = "\u2949";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2829:
                if ('r' == current) {
                    state_ = 2830;
                    return true;
                }
                break;
            case 2830:
                if ('c' == current) {
                    state_ = 2831;
                    return true;
                }
                else if ('e' == current) {
                    state_ = 2833;
                    return true;
                }
                break;
            case 2831:
                if ('y' == current) {
                    state_ = 2832;
                    return true;
                }
                break;
            case 2832:
                // Ubrcy;
                if (';' == current) {
                    match_ = "\u040E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2833:
                if ('v' == current) {
                    state_ = 2834;
                    return true;
                }
                break;
            case 2834:
                if ('e' == current) {
                    state_ = 2835;
                    return true;
                }
                break;
            case 2835:
                // Ubreve;
                if (';' == current) {
                    match_ = "\u016C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2836:
                if ('i' == current) {
                    state_ = 2837;
                    return true;
                }
                else if ('y' == current) {
                    state_ = 2840;
                    return true;
                }
                break;
            case 2837:
                if ('r' == current) {
                    state_ = 2838;
                    return true;
                }
                break;
            case 2838:
                // Ucirc
                if ('c' == current) {
                    match_ = "\u00DB";
                    matchLength_ = consumedCount_;
                    state_ = 2839;
                    return true;
                }
                break;
            case 2839:
                // Ucirc;
                if (';' == current) {
                    match_ = "\u00DB";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2840:
                // Ucy;
                if (';' == current) {
                    match_ = "\u0423";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2841:
                if ('b' == current) {
                    state_ = 2842;
                    return true;
                }
                break;
            case 2842:
                if ('l' == current) {
                    state_ = 2843;
                    return true;
                }
                break;
            case 2843:
                if ('a' == current) {
                    state_ = 2844;
                    return true;
                }
                break;
            case 2844:
                if ('c' == current) {
                    state_ = 2845;
                    return true;
                }
                break;
            case 2845:
                // Udblac;
                if (';' == current) {
                    match_ = "\u0170";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2846:
                if ('r' == current) {
                    state_ = 2847;
                    return true;
                }
                break;
            case 2847:
                // Ufr;
                if (';' == current) {
                    match_ = "\uD835\uDD18";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2848:
                if ('r' == current) {
                    state_ = 2849;
                    return true;
                }
                break;
            case 2849:
                if ('a' == current) {
                    state_ = 2850;
                    return true;
                }
                break;
            case 2850:
                if ('v' == current) {
                    state_ = 2851;
                    return true;
                }
                break;
            case 2851:
                // Ugrave
                if ('e' == current) {
                    match_ = "\u00D9";
                    matchLength_ = consumedCount_;
                    state_ = 2852;
                    return true;
                }
                break;
            case 2852:
                // Ugrave;
                if (';' == current) {
                    match_ = "\u00D9";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2853:
                if ('a' == current) {
                    state_ = 2854;
                    return true;
                }
                break;
            case 2854:
                if ('c' == current) {
                    state_ = 2855;
                    return true;
                }
                break;
            case 2855:
                if ('r' == current) {
                    state_ = 2856;
                    return true;
                }
                break;
            case 2856:
                // Umacr;
                if (';' == current) {
                    match_ = "\u016A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2857:
                if ('d' == current) {
                    state_ = 2858;
                    return true;
                }
                else if ('i' == current) {
                    state_ = 2882;
                    return true;
                }
                break;
            case 2858:
                if ('e' == current) {
                    state_ = 2859;
                    return true;
                }
                break;
            case 2859:
                if ('r' == current) {
                    state_ = 2860;
                    return true;
                }
                break;
            case 2860:
                if ('B' == current) {
                    state_ = 2861;
                    return true;
                }
                else if ('P' == current) {
                    state_ = 2871;
                    return true;
                }
                break;
            case 2861:
                if ('a' == current) {
                    state_ = 2862;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 2864;
                    return true;
                }
                break;
            case 2862:
                if ('r' == current) {
                    state_ = 2863;
                    return true;
                }
                break;
            case 2863:
                // UnderBar;
                if (';' == current) {
                    match_ = "_";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2864:
                if ('a' == current) {
                    state_ = 2865;
                    return true;
                }
                break;
            case 2865:
                if ('c' == current) {
                    state_ = 2866;
                    return true;
                }
                break;
            case 2866:
                if ('e' == current) {
                    state_ = 2867;
                    return true;
                }
                else if ('k' == current) {
                    state_ = 2868;
                    return true;
                }
                break;
            case 2867:
                // UnderBrace;
                if (';' == current) {
                    match_ = "\u23DF";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2868:
                if ('e' == current) {
                    state_ = 2869;
                    return true;
                }
                break;
            case 2869:
                if ('t' == current) {
                    state_ = 2870;
                    return true;
                }
                break;
            case 2870:
                // UnderBracket;
                if (';' == current) {
                    match_ = "\u23B5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2871:
                if ('a' == current) {
                    state_ = 2872;
                    return true;
                }
                break;
            case 2872:
                if ('r' == current) {
                    state_ = 2873;
                    return true;
                }
                break;
            case 2873:
                if ('e' == current) {
                    state_ = 2874;
                    return true;
                }
                break;
            case 2874:
                if ('n' == current) {
                    state_ = 2875;
                    return true;
                }
                break;
            case 2875:
                if ('t' == current) {
                    state_ = 2876;
                    return true;
                }
                break;
            case 2876:
                if ('h' == current) {
                    state_ = 2877;
                    return true;
                }
                break;
            case 2877:
                if ('e' == current) {
                    state_ = 2878;
                    return true;
                }
                break;
            case 2878:
                if ('s' == current) {
                    state_ = 2879;
                    return true;
                }
                break;
            case 2879:
                if ('i' == current) {
                    state_ = 2880;
                    return true;
                }
                break;
            case 2880:
                if ('s' == current) {
                    state_ = 2881;
                    return true;
                }
                break;
            case 2881:
                // UnderParenthesis;
                if (';' == current) {
                    match_ = "\u23DD";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2882:
                if ('o' == current) {
                    state_ = 2883;
                    return true;
                }
                break;
            case 2883:
                if ('n' == current) {
                    state_ = 2884;
                    return true;
                }
                break;
            case 2884:
                // Union;
                if (';' == current) {
                    match_ = "\u22C3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('P' == current) {
                    state_ = 2885;
                    return true;
                }
                break;
            case 2885:
                if ('l' == current) {
                    state_ = 2886;
                    return true;
                }
                break;
            case 2886:
                if ('u' == current) {
                    state_ = 2887;
                    return true;
                }
                break;
            case 2887:
                if ('s' == current) {
                    state_ = 2888;
                    return true;
                }
                break;
            case 2888:
                // UnionPlus;
                if (';' == current) {
                    match_ = "\u228E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2889:
                if ('g' == current) {
                    state_ = 2890;
                    return true;
                }
                else if ('p' == current) {
                    state_ = 2893;
                    return true;
                }
                break;
            case 2890:
                if ('o' == current) {
                    state_ = 2891;
                    return true;
                }
                break;
            case 2891:
                if ('n' == current) {
                    state_ = 2892;
                    return true;
                }
                break;
            case 2892:
                // Uogon;
                if (';' == current) {
                    match_ = "\u0172";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2893:
                if ('f' == current) {
                    state_ = 2894;
                    return true;
                }
                break;
            case 2894:
                // Uopf;
                if (';' == current) {
                    match_ = "\uD835\uDD4C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2895:
                switch (current) {
                    case 'A':
                        state_ = 2896;
                        return true;
                    case 'D':
                        state_ = 2913;
                        return true;
                    case 'E':
                        state_ = 2922;
                        return true;
                    case 'T':
                        state_ = 2933;
                        return true;
                    case 'a':
                        state_ = 2941;
                        return true;
                    case 'd':
                        state_ = 2946;
                        return true;
                    case 'p':
                        state_ = 2955;
                        return true;
                    case 's':
                        state_ = 2977;
                        return true;
                }
                break;
            case 2896:
                if ('r' == current) {
                    state_ = 2897;
                    return true;
                }
                break;
            case 2897:
                if ('r' == current) {
                    state_ = 2898;
                    return true;
                }
                break;
            case 2898:
                if ('o' == current) {
                    state_ = 2899;
                    return true;
                }
                break;
            case 2899:
                if ('w' == current) {
                    state_ = 2900;
                    return true;
                }
                break;
            case 2900:
                switch (current) {
                    case ';': // UpArrow;
                        match_ = "\u2191";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'B':
                        state_ = 2901;
                        return true;
                    case 'D':
                        state_ = 2904;
                        return true;
                }
                break;
            case 2901:
                if ('a' == current) {
                    state_ = 2902;
                    return true;
                }
                break;
            case 2902:
                if ('r' == current) {
                    state_ = 2903;
                    return true;
                }
                break;
            case 2903:
                // UpArrowBar;
                if (';' == current) {
                    match_ = "\u2912";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2904:
                if ('o' == current) {
                    state_ = 2905;
                    return true;
                }
                break;
            case 2905:
                if ('w' == current) {
                    state_ = 2906;
                    return true;
                }
                break;
            case 2906:
                if ('n' == current) {
                    state_ = 2907;
                    return true;
                }
                break;
            case 2907:
                if ('A' == current) {
                    state_ = 2908;
                    return true;
                }
                break;
            case 2908:
                if ('r' == current) {
                    state_ = 2909;
                    return true;
                }
                break;
            case 2909:
                if ('r' == current) {
                    state_ = 2910;
                    return true;
                }
                break;
            case 2910:
                if ('o' == current) {
                    state_ = 2911;
                    return true;
                }
                break;
            case 2911:
                if ('w' == current) {
                    state_ = 2912;
                    return true;
                }
                break;
            case 2912:
                // UpArrowDownArrow;
                if (';' == current) {
                    match_ = "\u21C5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2913:
                if ('o' == current) {
                    state_ = 2914;
                    return true;
                }
                break;
            case 2914:
                if ('w' == current) {
                    state_ = 2915;
                    return true;
                }
                break;
            case 2915:
                if ('n' == current) {
                    state_ = 2916;
                    return true;
                }
                break;
            case 2916:
                if ('A' == current) {
                    state_ = 2917;
                    return true;
                }
                break;
            case 2917:
                if ('r' == current) {
                    state_ = 2918;
                    return true;
                }
                break;
            case 2918:
                if ('r' == current) {
                    state_ = 2919;
                    return true;
                }
                break;
            case 2919:
                if ('o' == current) {
                    state_ = 2920;
                    return true;
                }
                break;
            case 2920:
                if ('w' == current) {
                    state_ = 2921;
                    return true;
                }
                break;
            case 2921:
                // UpDownArrow;
                if (';' == current) {
                    match_ = "\u2195";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2922:
                if ('q' == current) {
                    state_ = 2923;
                    return true;
                }
                break;
            case 2923:
                if ('u' == current) {
                    state_ = 2924;
                    return true;
                }
                break;
            case 2924:
                if ('i' == current) {
                    state_ = 2925;
                    return true;
                }
                break;
            case 2925:
                if ('l' == current) {
                    state_ = 2926;
                    return true;
                }
                break;
            case 2926:
                if ('i' == current) {
                    state_ = 2927;
                    return true;
                }
                break;
            case 2927:
                if ('b' == current) {
                    state_ = 2928;
                    return true;
                }
                break;
            case 2928:
                if ('r' == current) {
                    state_ = 2929;
                    return true;
                }
                break;
            case 2929:
                if ('i' == current) {
                    state_ = 2930;
                    return true;
                }
                break;
            case 2930:
                if ('u' == current) {
                    state_ = 2931;
                    return true;
                }
                break;
            case 2931:
                if ('m' == current) {
                    state_ = 2932;
                    return true;
                }
                break;
            case 2932:
                // UpEquilibrium;
                if (';' == current) {
                    match_ = "\u296E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2933:
                if ('e' == current) {
                    state_ = 2934;
                    return true;
                }
                break;
            case 2934:
                if ('e' == current) {
                    state_ = 2935;
                    return true;
                }
                break;
            case 2935:
                // UpTee;
                if (';' == current) {
                    match_ = "\u22A5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('A' == current) {
                    state_ = 2936;
                    return true;
                }
                break;
            case 2936:
                if ('r' == current) {
                    state_ = 2937;
                    return true;
                }
                break;
            case 2937:
                if ('r' == current) {
                    state_ = 2938;
                    return true;
                }
                break;
            case 2938:
                if ('o' == current) {
                    state_ = 2939;
                    return true;
                }
                break;
            case 2939:
                if ('w' == current) {
                    state_ = 2940;
                    return true;
                }
                break;
            case 2940:
                // UpTeeArrow;
                if (';' == current) {
                    match_ = "\u21A5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2941:
                if ('r' == current) {
                    state_ = 2942;
                    return true;
                }
                break;
            case 2942:
                if ('r' == current) {
                    state_ = 2943;
                    return true;
                }
                break;
            case 2943:
                if ('o' == current) {
                    state_ = 2944;
                    return true;
                }
                break;
            case 2944:
                if ('w' == current) {
                    state_ = 2945;
                    return true;
                }
                break;
            case 2945:
                // Uparrow;
                if (';' == current) {
                    match_ = "\u21D1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2946:
                if ('o' == current) {
                    state_ = 2947;
                    return true;
                }
                break;
            case 2947:
                if ('w' == current) {
                    state_ = 2948;
                    return true;
                }
                break;
            case 2948:
                if ('n' == current) {
                    state_ = 2949;
                    return true;
                }
                break;
            case 2949:
                if ('a' == current) {
                    state_ = 2950;
                    return true;
                }
                break;
            case 2950:
                if ('r' == current) {
                    state_ = 2951;
                    return true;
                }
                break;
            case 2951:
                if ('r' == current) {
                    state_ = 2952;
                    return true;
                }
                break;
            case 2952:
                if ('o' == current) {
                    state_ = 2953;
                    return true;
                }
                break;
            case 2953:
                if ('w' == current) {
                    state_ = 2954;
                    return true;
                }
                break;
            case 2954:
                // Updownarrow;
                if (';' == current) {
                    match_ = "\u21D5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2955:
                if ('e' == current) {
                    state_ = 2956;
                    return true;
                }
                break;
            case 2956:
                if ('r' == current) {
                    state_ = 2957;
                    return true;
                }
                break;
            case 2957:
                if ('L' == current) {
                    state_ = 2958;
                    return true;
                }
                else if ('R' == current) {
                    state_ = 2967;
                    return true;
                }
                break;
            case 2958:
                if ('e' == current) {
                    state_ = 2959;
                    return true;
                }
                break;
            case 2959:
                if ('f' == current) {
                    state_ = 2960;
                    return true;
                }
                break;
            case 2960:
                if ('t' == current) {
                    state_ = 2961;
                    return true;
                }
                break;
            case 2961:
                if ('A' == current) {
                    state_ = 2962;
                    return true;
                }
                break;
            case 2962:
                if ('r' == current) {
                    state_ = 2963;
                    return true;
                }
                break;
            case 2963:
                if ('r' == current) {
                    state_ = 2964;
                    return true;
                }
                break;
            case 2964:
                if ('o' == current) {
                    state_ = 2965;
                    return true;
                }
                break;
            case 2965:
                if ('w' == current) {
                    state_ = 2966;
                    return true;
                }
                break;
            case 2966:
                // UpperLeftArrow;
                if (';' == current) {
                    match_ = "\u2196";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2967:
                if ('i' == current) {
                    state_ = 2968;
                    return true;
                }
                break;
            case 2968:
                if ('g' == current) {
                    state_ = 2969;
                    return true;
                }
                break;
            case 2969:
                if ('h' == current) {
                    state_ = 2970;
                    return true;
                }
                break;
            case 2970:
                if ('t' == current) {
                    state_ = 2971;
                    return true;
                }
                break;
            case 2971:
                if ('A' == current) {
                    state_ = 2972;
                    return true;
                }
                break;
            case 2972:
                if ('r' == current) {
                    state_ = 2973;
                    return true;
                }
                break;
            case 2973:
                if ('r' == current) {
                    state_ = 2974;
                    return true;
                }
                break;
            case 2974:
                if ('o' == current) {
                    state_ = 2975;
                    return true;
                }
                break;
            case 2975:
                if ('w' == current) {
                    state_ = 2976;
                    return true;
                }
                break;
            case 2976:
                // UpperRightArrow;
                if (';' == current) {
                    match_ = "\u2197";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2977:
                if ('i' == current) {
                    state_ = 2978;
                    return true;
                }
                break;
            case 2978:
                // Upsi;
                if (';' == current) {
                    match_ = "\u03D2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('l' == current) {
                    state_ = 2979;
                    return true;
                }
                break;
            case 2979:
                if ('o' == current) {
                    state_ = 2980;
                    return true;
                }
                break;
            case 2980:
                if ('n' == current) {
                    state_ = 2981;
                    return true;
                }
                break;
            case 2981:
                // Upsilon;
                if (';' == current) {
                    match_ = "\u03A5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2982:
                if ('i' == current) {
                    state_ = 2983;
                    return true;
                }
                break;
            case 2983:
                if ('n' == current) {
                    state_ = 2984;
                    return true;
                }
                break;
            case 2984:
                if ('g' == current) {
                    state_ = 2985;
                    return true;
                }
                break;
            case 2985:
                // Uring;
                if (';' == current) {
                    match_ = "\u016E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2986:
                if ('c' == current) {
                    state_ = 2987;
                    return true;
                }
                break;
            case 2987:
                if ('r' == current) {
                    state_ = 2988;
                    return true;
                }
                break;
            case 2988:
                // Uscr;
                if (';' == current) {
                    match_ = "\uD835\uDCB0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2989:
                if ('i' == current) {
                    state_ = 2990;
                    return true;
                }
                break;
            case 2990:
                if ('l' == current) {
                    state_ = 2991;
                    return true;
                }
                break;
            case 2991:
                if ('d' == current) {
                    state_ = 2992;
                    return true;
                }
                break;
            case 2992:
                if ('e' == current) {
                    state_ = 2993;
                    return true;
                }
                break;
            case 2993:
                // Utilde;
                if (';' == current) {
                    match_ = "\u0168";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2994:
                if ('m' == current) {
                    state_ = 2995;
                    return true;
                }
                break;
            case 2995:
                // Uuml
                if ('l' == current) {
                    match_ = "\u00DC";
                    matchLength_ = consumedCount_;
                    state_ = 2996;
                    return true;
                }
                break;
            case 2996:
                // Uuml;
                if (';' == current) {
                    match_ = "\u00DC";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 2997:
                switch (current) {
                    case 'D':
                        state_ = 2998;
                        return true;
                    case 'b':
                        state_ = 3002;
                        return true;
                    case 'c':
                        state_ = 3005;
                        return true;
                    case 'd':
                        state_ = 3007;
                        return true;
                    case 'e':
                        state_ = 3012;
                        return true;
                    case 'f':
                        state_ = 3054;
                        return true;
                    case 'o':
                        state_ = 3056;
                        return true;
                    case 's':
                        state_ = 3059;
                        return true;
                    case 'v':
                        state_ = 3062;
                        return true;
                }
                break;
            case 2998:
                if ('a' == current) {
                    state_ = 2999;
                    return true;
                }
                break;
            case 2999:
                if ('s' == current) {
                    state_ = 3000;
                    return true;
                }
                break;
        }
        return false;
    }

    private boolean parse4(final int current) {
        consumedCount_++;
        switch (state_) {
            case 3000:
                if ('h' == current) {
                    state_ = 3001;
                    return true;
                }
                break;
            case 3001:
                // VDash;
                if (';' == current) {
                    match_ = "\u22AB";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3002:
                if ('a' == current) {
                    state_ = 3003;
                    return true;
                }
                break;
            case 3003:
                if ('r' == current) {
                    state_ = 3004;
                    return true;
                }
                break;
            case 3004:
                // Vbar;
                if (';' == current) {
                    match_ = "\u2AEB";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3005:
                if ('y' == current) {
                    state_ = 3006;
                    return true;
                }
                break;
            case 3006:
                // Vcy;
                if (';' == current) {
                    match_ = "\u0412";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3007:
                if ('a' == current) {
                    state_ = 3008;
                    return true;
                }
                break;
            case 3008:
                if ('s' == current) {
                    state_ = 3009;
                    return true;
                }
                break;
            case 3009:
                if ('h' == current) {
                    state_ = 3010;
                    return true;
                }
                break;
            case 3010:
                // Vdash;
                if (';' == current) {
                    match_ = "\u22A9";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('l' == current) {
                    state_ = 3011;
                    return true;
                }
                break;
            case 3011:
                // Vdashl;
                if (';' == current) {
                    match_ = "\u2AE6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3012:
                if ('e' == current) {
                    state_ = 3013;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 3014;
                    return true;
                }
                break;
            case 3013:
                // Vee;
                if (';' == current) {
                    match_ = "\u22C1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3014:
                switch (current) {
                    case 'b':
                        state_ = 3015;
                        return true;
                    case 't':
                        state_ = 3018;
                        return true;
                    case 'y':
                        state_ = 3044;
                        return true;
                }
                break;
            case 3015:
                if ('a' == current) {
                    state_ = 3016;
                    return true;
                }
                break;
            case 3016:
                if ('r' == current) {
                    state_ = 3017;
                    return true;
                }
                break;
            case 3017:
                // Verbar;
                if (';' == current) {
                    match_ = "\u2016";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3018:
                // Vert;
                if (';' == current) {
                    match_ = "\u2016";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('i' == current) {
                    state_ = 3019;
                    return true;
                }
                break;
            case 3019:
                if ('c' == current) {
                    state_ = 3020;
                    return true;
                }
                break;
            case 3020:
                if ('a' == current) {
                    state_ = 3021;
                    return true;
                }
                break;
            case 3021:
                if ('l' == current) {
                    state_ = 3022;
                    return true;
                }
                break;
            case 3022:
                switch (current) {
                    case 'B':
                        state_ = 3023;
                        return true;
                    case 'L':
                        state_ = 3026;
                        return true;
                    case 'S':
                        state_ = 3030;
                        return true;
                    case 'T':
                        state_ = 3039;
                        return true;
                }
                break;
            case 3023:
                if ('a' == current) {
                    state_ = 3024;
                    return true;
                }
                break;
            case 3024:
                if ('r' == current) {
                    state_ = 3025;
                    return true;
                }
                break;
            case 3025:
                // VerticalBar;
                if (';' == current) {
                    match_ = "\u2223";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3026:
                if ('i' == current) {
                    state_ = 3027;
                    return true;
                }
                break;
            case 3027:
                if ('n' == current) {
                    state_ = 3028;
                    return true;
                }
                break;
            case 3028:
                if ('e' == current) {
                    state_ = 3029;
                    return true;
                }
                break;
            case 3029:
                // VerticalLine;
                if (';' == current) {
                    match_ = "|";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3030:
                if ('e' == current) {
                    state_ = 3031;
                    return true;
                }
                break;
            case 3031:
                if ('p' == current) {
                    state_ = 3032;
                    return true;
                }
                break;
            case 3032:
                if ('a' == current) {
                    state_ = 3033;
                    return true;
                }
                break;
            case 3033:
                if ('r' == current) {
                    state_ = 3034;
                    return true;
                }
                break;
            case 3034:
                if ('a' == current) {
                    state_ = 3035;
                    return true;
                }
                break;
            case 3035:
                if ('t' == current) {
                    state_ = 3036;
                    return true;
                }
                break;
            case 3036:
                if ('o' == current) {
                    state_ = 3037;
                    return true;
                }
                break;
            case 3037:
                if ('r' == current) {
                    state_ = 3038;
                    return true;
                }
                break;
            case 3038:
                // VerticalSeparator;
                if (';' == current) {
                    match_ = "\u2758";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3039:
                if ('i' == current) {
                    state_ = 3040;
                    return true;
                }
                break;
            case 3040:
                if ('l' == current) {
                    state_ = 3041;
                    return true;
                }
                break;
            case 3041:
                if ('d' == current) {
                    state_ = 3042;
                    return true;
                }
                break;
            case 3042:
                if ('e' == current) {
                    state_ = 3043;
                    return true;
                }
                break;
            case 3043:
                // VerticalTilde;
                if (';' == current) {
                    match_ = "\u2240";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3044:
                if ('T' == current) {
                    state_ = 3045;
                    return true;
                }
                break;
            case 3045:
                if ('h' == current) {
                    state_ = 3046;
                    return true;
                }
                break;
            case 3046:
                if ('i' == current) {
                    state_ = 3047;
                    return true;
                }
                break;
            case 3047:
                if ('n' == current) {
                    state_ = 3048;
                    return true;
                }
                break;
            case 3048:
                if ('S' == current) {
                    state_ = 3049;
                    return true;
                }
                break;
            case 3049:
                if ('p' == current) {
                    state_ = 3050;
                    return true;
                }
                break;
            case 3050:
                if ('a' == current) {
                    state_ = 3051;
                    return true;
                }
                break;
            case 3051:
                if ('c' == current) {
                    state_ = 3052;
                    return true;
                }
                break;
            case 3052:
                if ('e' == current) {
                    state_ = 3053;
                    return true;
                }
                break;
            case 3053:
                // VeryThinSpace;
                if (';' == current) {
                    match_ = "\u200A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3054:
                if ('r' == current) {
                    state_ = 3055;
                    return true;
                }
                break;
            case 3055:
                // Vfr;
                if (';' == current) {
                    match_ = "\uD835\uDD19";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3056:
                if ('p' == current) {
                    state_ = 3057;
                    return true;
                }
                break;
            case 3057:
                if ('f' == current) {
                    state_ = 3058;
                    return true;
                }
                break;
            case 3058:
                // Vopf;
                if (';' == current) {
                    match_ = "\uD835\uDD4D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3059:
                if ('c' == current) {
                    state_ = 3060;
                    return true;
                }
                break;
            case 3060:
                if ('r' == current) {
                    state_ = 3061;
                    return true;
                }
                break;
            case 3061:
                // Vscr;
                if (';' == current) {
                    match_ = "\uD835\uDCB1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3062:
                if ('d' == current) {
                    state_ = 3063;
                    return true;
                }
                break;
            case 3063:
                if ('a' == current) {
                    state_ = 3064;
                    return true;
                }
                break;
            case 3064:
                if ('s' == current) {
                    state_ = 3065;
                    return true;
                }
                break;
            case 3065:
                if ('h' == current) {
                    state_ = 3066;
                    return true;
                }
                break;
            case 3066:
                // Vvdash;
                if (';' == current) {
                    match_ = "\u22AA";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3067:
                switch (current) {
                    case 'c':
                        state_ = 3068;
                        return true;
                    case 'e':
                        state_ = 3072;
                        return true;
                    case 'f':
                        state_ = 3076;
                        return true;
                    case 'o':
                        state_ = 3078;
                        return true;
                    case 's':
                        state_ = 3081;
                        return true;
                }
                break;
            case 3068:
                if ('i' == current) {
                    state_ = 3069;
                    return true;
                }
                break;
            case 3069:
                if ('r' == current) {
                    state_ = 3070;
                    return true;
                }
                break;
            case 3070:
                if ('c' == current) {
                    state_ = 3071;
                    return true;
                }
                break;
            case 3071:
                // Wcirc;
                if (';' == current) {
                    match_ = "\u0174";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3072:
                if ('d' == current) {
                    state_ = 3073;
                    return true;
                }
                break;
            case 3073:
                if ('g' == current) {
                    state_ = 3074;
                    return true;
                }
                break;
            case 3074:
                if ('e' == current) {
                    state_ = 3075;
                    return true;
                }
                break;
            case 3075:
                // Wedge;
                if (';' == current) {
                    match_ = "\u22C0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3076:
                if ('r' == current) {
                    state_ = 3077;
                    return true;
                }
                break;
            case 3077:
                // Wfr;
                if (';' == current) {
                    match_ = "\uD835\uDD1A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3078:
                if ('p' == current) {
                    state_ = 3079;
                    return true;
                }
                break;
            case 3079:
                if ('f' == current) {
                    state_ = 3080;
                    return true;
                }
                break;
            case 3080:
                // Wopf;
                if (';' == current) {
                    match_ = "\uD835\uDD4E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3081:
                if ('c' == current) {
                    state_ = 3082;
                    return true;
                }
                break;
            case 3082:
                if ('r' == current) {
                    state_ = 3083;
                    return true;
                }
                break;
            case 3083:
                // Wscr;
                if (';' == current) {
                    match_ = "\uD835\uDCB2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3084:
                switch (current) {
                    case 'f':
                        state_ = 3085;
                        return true;
                    case 'i':
                        state_ = 3087;
                        return true;
                    case 'o':
                        state_ = 3088;
                        return true;
                    case 's':
                        state_ = 3091;
                        return true;
                }
                break;
            case 3085:
                if ('r' == current) {
                    state_ = 3086;
                    return true;
                }
                break;
            case 3086:
                // Xfr;
                if (';' == current) {
                    match_ = "\uD835\uDD1B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3087:
                // Xi;
                if (';' == current) {
                    match_ = "\u039E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3088:
                if ('p' == current) {
                    state_ = 3089;
                    return true;
                }
                break;
            case 3089:
                if ('f' == current) {
                    state_ = 3090;
                    return true;
                }
                break;
            case 3090:
                // Xopf;
                if (';' == current) {
                    match_ = "\uD835\uDD4F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3091:
                if ('c' == current) {
                    state_ = 3092;
                    return true;
                }
                break;
            case 3092:
                if ('r' == current) {
                    state_ = 3093;
                    return true;
                }
                break;
            case 3093:
                // Xscr;
                if (';' == current) {
                    match_ = "\uD835\uDCB3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3094:
                switch (current) {
                    case 'A':
                        state_ = 3095;
                        return true;
                    case 'I':
                        state_ = 3098;
                        return true;
                    case 'U':
                        state_ = 3101;
                        return true;
                    case 'a':
                        state_ = 3104;
                        return true;
                    case 'c':
                        state_ = 3109;
                        return true;
                    case 'f':
                        state_ = 3114;
                        return true;
                    case 'o':
                        state_ = 3116;
                        return true;
                    case 's':
                        state_ = 3119;
                        return true;
                    case 'u':
                        state_ = 3122;
                        return true;
                }
                break;
            case 3095:
                if ('c' == current) {
                    state_ = 3096;
                    return true;
                }
                break;
            case 3096:
                if ('y' == current) {
                    state_ = 3097;
                    return true;
                }
                break;
            case 3097:
                // YAcy;
                if (';' == current) {
                    match_ = "\u042F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3098:
                if ('c' == current) {
                    state_ = 3099;
                    return true;
                }
                break;
            case 3099:
                if ('y' == current) {
                    state_ = 3100;
                    return true;
                }
                break;
            case 3100:
                // YIcy;
                if (';' == current) {
                    match_ = "\u0407";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3101:
                if ('c' == current) {
                    state_ = 3102;
                    return true;
                }
                break;
            case 3102:
                if ('y' == current) {
                    state_ = 3103;
                    return true;
                }
                break;
            case 3103:
                // YUcy;
                if (';' == current) {
                    match_ = "\u042E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3104:
                if ('c' == current) {
                    state_ = 3105;
                    return true;
                }
                break;
            case 3105:
                if ('u' == current) {
                    state_ = 3106;
                    return true;
                }
                break;
            case 3106:
                if ('t' == current) {
                    state_ = 3107;
                    return true;
                }
                break;
            case 3107:
                // Yacute
                if ('e' == current) {
                    match_ = "\u00DD";
                    matchLength_ = consumedCount_;
                    state_ = 3108;
                    return true;
                }
                break;
            case 3108:
                // Yacute;
                if (';' == current) {
                    match_ = "\u00DD";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3109:
                if ('i' == current) {
                    state_ = 3110;
                    return true;
                }
                else if ('y' == current) {
                    state_ = 3113;
                    return true;
                }
                break;
            case 3110:
                if ('r' == current) {
                    state_ = 3111;
                    return true;
                }
                break;
            case 3111:
                if ('c' == current) {
                    state_ = 3112;
                    return true;
                }
                break;
            case 3112:
                // Ycirc;
                if (';' == current) {
                    match_ = "\u0176";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3113:
                // Ycy;
                if (';' == current) {
                    match_ = "\u042B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3114:
                if ('r' == current) {
                    state_ = 3115;
                    return true;
                }
                break;
            case 3115:
                // Yfr;
                if (';' == current) {
                    match_ = "\uD835\uDD1C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3116:
                if ('p' == current) {
                    state_ = 3117;
                    return true;
                }
                break;
            case 3117:
                if ('f' == current) {
                    state_ = 3118;
                    return true;
                }
                break;
            case 3118:
                // Yopf;
                if (';' == current) {
                    match_ = "\uD835\uDD50";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3119:
                if ('c' == current) {
                    state_ = 3120;
                    return true;
                }
                break;
            case 3120:
                if ('r' == current) {
                    state_ = 3121;
                    return true;
                }
                break;
            case 3121:
                // Yscr;
                if (';' == current) {
                    match_ = "\uD835\uDCB4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3122:
                if ('m' == current) {
                    state_ = 3123;
                    return true;
                }
                break;
            case 3123:
                if ('l' == current) {
                    state_ = 3124;
                    return true;
                }
                break;
            case 3124:
                // Yuml;
                if (';' == current) {
                    match_ = "\u0178";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3125:
                switch (current) {
                    case 'H':
                        state_ = 3126;
                        return true;
                    case 'a':
                        state_ = 3129;
                        return true;
                    case 'c':
                        state_ = 3134;
                        return true;
                    case 'd':
                        state_ = 3140;
                        return true;
                    case 'e':
                        state_ = 3143;
                        return true;
                    case 'f':
                        state_ = 3158;
                        return true;
                    case 'o':
                        state_ = 3160;
                        return true;
                    case 's':
                        state_ = 3163;
                        return true;
                }
                break;
            case 3126:
                if ('c' == current) {
                    state_ = 3127;
                    return true;
                }
                break;
            case 3127:
                if ('y' == current) {
                    state_ = 3128;
                    return true;
                }
                break;
            case 3128:
                // ZHcy;
                if (';' == current) {
                    match_ = "\u0416";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3129:
                if ('c' == current) {
                    state_ = 3130;
                    return true;
                }
                break;
            case 3130:
                if ('u' == current) {
                    state_ = 3131;
                    return true;
                }
                break;
            case 3131:
                if ('t' == current) {
                    state_ = 3132;
                    return true;
                }
                break;
            case 3132:
                if ('e' == current) {
                    state_ = 3133;
                    return true;
                }
                break;
            case 3133:
                // Zacute;
                if (';' == current) {
                    match_ = "\u0179";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3134:
                if ('a' == current) {
                    state_ = 3135;
                    return true;
                }
                else if ('y' == current) {
                    state_ = 3139;
                    return true;
                }
                break;
            case 3135:
                if ('r' == current) {
                    state_ = 3136;
                    return true;
                }
                break;
            case 3136:
                if ('o' == current) {
                    state_ = 3137;
                    return true;
                }
                break;
            case 3137:
                if ('n' == current) {
                    state_ = 3138;
                    return true;
                }
                break;
            case 3138:
                // Zcaron;
                if (';' == current) {
                    match_ = "\u017D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3139:
                // Zcy;
                if (';' == current) {
                    match_ = "\u0417";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3140:
                if ('o' == current) {
                    state_ = 3141;
                    return true;
                }
                break;
            case 3141:
                if ('t' == current) {
                    state_ = 3142;
                    return true;
                }
                break;
            case 3142:
                // Zdot;
                if (';' == current) {
                    match_ = "\u017B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3143:
                if ('r' == current) {
                    state_ = 3144;
                    return true;
                }
                else if ('t' == current) {
                    state_ = 3156;
                    return true;
                }
                break;
            case 3144:
                if ('o' == current) {
                    state_ = 3145;
                    return true;
                }
                break;
            case 3145:
                if ('W' == current) {
                    state_ = 3146;
                    return true;
                }
                break;
            case 3146:
                if ('i' == current) {
                    state_ = 3147;
                    return true;
                }
                break;
            case 3147:
                if ('d' == current) {
                    state_ = 3148;
                    return true;
                }
                break;
            case 3148:
                if ('t' == current) {
                    state_ = 3149;
                    return true;
                }
                break;
            case 3149:
                if ('h' == current) {
                    state_ = 3150;
                    return true;
                }
                break;
            case 3150:
                if ('S' == current) {
                    state_ = 3151;
                    return true;
                }
                break;
            case 3151:
                if ('p' == current) {
                    state_ = 3152;
                    return true;
                }
                break;
            case 3152:
                if ('a' == current) {
                    state_ = 3153;
                    return true;
                }
                break;
            case 3153:
                if ('c' == current) {
                    state_ = 3154;
                    return true;
                }
                break;
            case 3154:
                if ('e' == current) {
                    state_ = 3155;
                    return true;
                }
                break;
            case 3155:
                // ZeroWidthSpace;
                if (';' == current) {
                    match_ = "\u200B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3156:
                if ('a' == current) {
                    state_ = 3157;
                    return true;
                }
                break;
            case 3157:
                // Zeta;
                if (';' == current) {
                    match_ = "\u0396";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3158:
                if ('r' == current) {
                    state_ = 3159;
                    return true;
                }
                break;
            case 3159:
                // Zfr;
                if (';' == current) {
                    match_ = "\u2128";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3160:
                if ('p' == current) {
                    state_ = 3161;
                    return true;
                }
                break;
            case 3161:
                if ('f' == current) {
                    state_ = 3162;
                    return true;
                }
                break;
            case 3162:
                // Zopf;
                if (';' == current) {
                    match_ = "\u2124";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3163:
                if ('c' == current) {
                    state_ = 3164;
                    return true;
                }
                break;
            case 3164:
                if ('r' == current) {
                    state_ = 3165;
                    return true;
                }
                break;
            case 3165:
                // Zscr;
                if (';' == current) {
                    match_ = "\uD835\uDCB5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3166:
                switch (current) {
                    case 'a':
                        state_ = 3167;
                        return true;
                    case 'b':
                        state_ = 3172;
                        return true;
                    case 'c':
                        state_ = 3177;
                        return true;
                    case 'e':
                        state_ = 3187;
                        return true;
                    case 'f':
                        state_ = 3191;
                        return true;
                    case 'g':
                        state_ = 3193;
                        return true;
                    case 'l':
                        state_ = 3198;
                        return true;
                    case 'm':
                        state_ = 3209;
                        return true;
                    case 'n':
                        state_ = 3216;
                        return true;
                    case 'o':
                        state_ = 3257;
                        return true;
                    case 'p':
                        state_ = 3263;
                        return true;
                    case 'r':
                        state_ = 3280;
                        return true;
                    case 's':
                        state_ = 3284;
                        return true;
                    case 't':
                        state_ = 3293;
                        return true;
                    case 'u':
                        state_ = 3298;
                        return true;
                    case 'w':
                        state_ = 3301;
                        return true;
                }
                break;
            case 3167:
                if ('c' == current) {
                    state_ = 3168;
                    return true;
                }
                break;
            case 3168:
                if ('u' == current) {
                    state_ = 3169;
                    return true;
                }
                break;
            case 3169:
                if ('t' == current) {
                    state_ = 3170;
                    return true;
                }
                break;
            case 3170:
                // aacute
                if ('e' == current) {
                    match_ = "\u00E1";
                    matchLength_ = consumedCount_;
                    state_ = 3171;
                    return true;
                }
                break;
            case 3171:
                // aacute;
                if (';' == current) {
                    match_ = "\u00E1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3172:
                if ('r' == current) {
                    state_ = 3173;
                    return true;
                }
                break;
            case 3173:
                if ('e' == current) {
                    state_ = 3174;
                    return true;
                }
                break;
            case 3174:
                if ('v' == current) {
                    state_ = 3175;
                    return true;
                }
                break;
            case 3175:
                if ('e' == current) {
                    state_ = 3176;
                    return true;
                }
                break;
            case 3176:
                // abreve;
                if (';' == current) {
                    match_ = "\u0103";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3177:
                switch (current) {
                    case ';': // ac;
                        match_ = "\u223E";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E':
                        state_ = 3178;
                        return true;
                    case 'd':
                        state_ = 3179;
                        return true;
                    case 'i':
                        state_ = 3180;
                        return true;
                    case 'u':
                        state_ = 3183;
                        return true;
                    case 'y':
                        state_ = 3186;
                        return true;
                }
                break;
            case 3178:
                // acE;
                if (';' == current) {
                    match_ = "\u223E\u0333";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3179:
                // acd;
                if (';' == current) {
                    match_ = "\u223F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3180:
                if ('r' == current) {
                    state_ = 3181;
                    return true;
                }
                break;
            case 3181:
                // acirc
                if ('c' == current) {
                    match_ = "\u00E2";
                    matchLength_ = consumedCount_;
                    state_ = 3182;
                    return true;
                }
                break;
            case 3182:
                // acirc;
                if (';' == current) {
                    match_ = "\u00E2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3183:
                if ('t' == current) {
                    state_ = 3184;
                    return true;
                }
                break;
            case 3184:
                // acute
                if ('e' == current) {
                    match_ = "\u00B4";
                    matchLength_ = consumedCount_;
                    state_ = 3185;
                    return true;
                }
                break;
            case 3185:
                // acute;
                if (';' == current) {
                    match_ = "\u00B4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3186:
                // acy;
                if (';' == current) {
                    match_ = "\u0430";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3187:
                if ('l' == current) {
                    state_ = 3188;
                    return true;
                }
                break;
            case 3188:
                if ('i' == current) {
                    state_ = 3189;
                    return true;
                }
                break;
            case 3189:
                // aelig
                if ('g' == current) {
                    match_ = "\u00E6";
                    matchLength_ = consumedCount_;
                    state_ = 3190;
                    return true;
                }
                break;
            case 3190:
                // aelig;
                if (';' == current) {
                    match_ = "\u00E6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3191:
                // af;
                if (';' == current) {
                    match_ = "\u2061";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('r' == current) {
                    state_ = 3192;
                    return true;
                }
                break;
            case 3192:
                // afr;
                if (';' == current) {
                    match_ = "\uD835\uDD1E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3193:
                if ('r' == current) {
                    state_ = 3194;
                    return true;
                }
                break;
            case 3194:
                if ('a' == current) {
                    state_ = 3195;
                    return true;
                }
                break;
            case 3195:
                if ('v' == current) {
                    state_ = 3196;
                    return true;
                }
                break;
            case 3196:
                // agrave
                if ('e' == current) {
                    match_ = "\u00E0";
                    matchLength_ = consumedCount_;
                    state_ = 3197;
                    return true;
                }
                break;
            case 3197:
                // agrave;
                if (';' == current) {
                    match_ = "\u00E0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3198:
                if ('e' == current) {
                    state_ = 3199;
                    return true;
                }
                else if ('p' == current) {
                    state_ = 3206;
                    return true;
                }
                break;
            case 3199:
                if ('f' == current) {
                    state_ = 3200;
                    return true;
                }
                else if ('p' == current) {
                    state_ = 3204;
                    return true;
                }
                break;
            case 3200:
                if ('s' == current) {
                    state_ = 3201;
                    return true;
                }
                break;
            case 3201:
                if ('y' == current) {
                    state_ = 3202;
                    return true;
                }
                break;
            case 3202:
                if ('m' == current) {
                    state_ = 3203;
                    return true;
                }
                break;
            case 3203:
                // alefsym;
                if (';' == current) {
                    match_ = "\u2135";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3204:
                if ('h' == current) {
                    state_ = 3205;
                    return true;
                }
                break;
            case 3205:
                // aleph;
                if (';' == current) {
                    match_ = "\u2135";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3206:
                if ('h' == current) {
                    state_ = 3207;
                    return true;
                }
                break;
            case 3207:
                if ('a' == current) {
                    state_ = 3208;
                    return true;
                }
                break;
            case 3208:
                // alpha;
                if (';' == current) {
                    match_ = "\u03B1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3209:
                if ('a' == current) {
                    state_ = 3210;
                    return true;
                }
                // amp
                else if ('p' == current) {
                    match_ = "&";
                    matchLength_ = consumedCount_;
                    state_ = 3215;
                    return true;
                }
                break;
            case 3210:
                if ('c' == current) {
                    state_ = 3211;
                    return true;
                }
                else if ('l' == current) {
                    state_ = 3213;
                    return true;
                }
                break;
            case 3211:
                if ('r' == current) {
                    state_ = 3212;
                    return true;
                }
                break;
            case 3212:
                // amacr;
                if (';' == current) {
                    match_ = "\u0101";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3213:
                if ('g' == current) {
                    state_ = 3214;
                    return true;
                }
                break;
            case 3214:
                // amalg;
                if (';' == current) {
                    match_ = "\u2A3F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3215:
                // amp;
                if (';' == current) {
                    match_ = "&";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3216:
                if ('d' == current) {
                    state_ = 3217;
                    return true;
                }
                else if ('g' == current) {
                    state_ = 3228;
                    return true;
                }
                break;
            case 3217:
                switch (current) {
                    case ';': // and;
                        match_ = "\u2227";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'a':
                        state_ = 3218;
                        return true;
                    case 'd':
                        state_ = 3221;
                        return true;
                    case 's':
                        state_ = 3222;
                        return true;
                    case 'v':
                        state_ = 3227;
                        return true;
                }
                break;
            case 3218:
                if ('n' == current) {
                    state_ = 3219;
                    return true;
                }
                break;
            case 3219:
                if ('d' == current) {
                    state_ = 3220;
                    return true;
                }
                break;
            case 3220:
                // andand;
                if (';' == current) {
                    match_ = "\u2A55";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3221:
                // andd;
                if (';' == current) {
                    match_ = "\u2A5C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3222:
                if ('l' == current) {
                    state_ = 3223;
                    return true;
                }
                break;
            case 3223:
                if ('o' == current) {
                    state_ = 3224;
                    return true;
                }
                break;
            case 3224:
                if ('p' == current) {
                    state_ = 3225;
                    return true;
                }
                break;
            case 3225:
                if ('e' == current) {
                    state_ = 3226;
                    return true;
                }
                break;
            case 3226:
                // andslope;
                if (';' == current) {
                    match_ = "\u2A58";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3227:
                // andv;
                if (';' == current) {
                    match_ = "\u2A5A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3228:
                switch (current) {
                    case ';': // ang;
                        match_ = "\u2220";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e':
                        state_ = 3229;
                        return true;
                    case 'l':
                        state_ = 3230;
                        return true;
                    case 'm':
                        state_ = 3232;
                        return true;
                    case 'r':
                        state_ = 3244;
                        return true;
                    case 's':
                        state_ = 3249;
                        return true;
                    case 'z':
                        state_ = 3253;
                        return true;
                }
                break;
            case 3229:
                // ange;
                if (';' == current) {
                    match_ = "\u29A4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3230:
                if ('e' == current) {
                    state_ = 3231;
                    return true;
                }
                break;
            case 3231:
                // angle;
                if (';' == current) {
                    match_ = "\u2220";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3232:
                if ('s' == current) {
                    state_ = 3233;
                    return true;
                }
                break;
            case 3233:
                if ('d' == current) {
                    state_ = 3234;
                    return true;
                }
                break;
            case 3234:
                // angmsd;
                if (';' == current) {
                    match_ = "\u2221";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('a' == current) {
                    state_ = 3235;
                    return true;
                }
                break;
            case 3235:
                switch (current) {
                    case 'a':
                        state_ = 3236;
                        return true;
                    case 'b':
                        state_ = 3237;
                        return true;
                    case 'c':
                        state_ = 3238;
                        return true;
                    case 'd':
                        state_ = 3239;
                        return true;
                    case 'e':
                        state_ = 3240;
                        return true;
                    case 'f':
                        state_ = 3241;
                        return true;
                    case 'g':
                        state_ = 3242;
                        return true;
                    case 'h':
                        state_ = 3243;
                        return true;
                }
                break;
            case 3236:
                // angmsdaa;
                if (';' == current) {
                    match_ = "\u29A8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3237:
                // angmsdab;
                if (';' == current) {
                    match_ = "\u29A9";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3238:
                // angmsdac;
                if (';' == current) {
                    match_ = "\u29AA";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3239:
                // angmsdad;
                if (';' == current) {
                    match_ = "\u29AB";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3240:
                // angmsdae;
                if (';' == current) {
                    match_ = "\u29AC";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3241:
                // angmsdaf;
                if (';' == current) {
                    match_ = "\u29AD";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3242:
                // angmsdag;
                if (';' == current) {
                    match_ = "\u29AE";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3243:
                // angmsdah;
                if (';' == current) {
                    match_ = "\u29AF";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3244:
                if ('t' == current) {
                    state_ = 3245;
                    return true;
                }
                break;
            case 3245:
                // angrt;
                if (';' == current) {
                    match_ = "\u221F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('v' == current) {
                    state_ = 3246;
                    return true;
                }
                break;
            case 3246:
                if ('b' == current) {
                    state_ = 3247;
                    return true;
                }
                break;
            case 3247:
                // angrtvb;
                if (';' == current) {
                    match_ = "\u22BE";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('d' == current) {
                    state_ = 3248;
                    return true;
                }
                break;
            case 3248:
                // angrtvbd;
                if (';' == current) {
                    match_ = "\u299D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3249:
                if ('p' == current) {
                    state_ = 3250;
                    return true;
                }
                else if ('t' == current) {
                    state_ = 3252;
                    return true;
                }
                break;
            case 3250:
                if ('h' == current) {
                    state_ = 3251;
                    return true;
                }
                break;
            case 3251:
                // angsph;
                if (';' == current) {
                    match_ = "\u2222";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3252:
                // angst;
                if (';' == current) {
                    match_ = "\u00C5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3253:
                if ('a' == current) {
                    state_ = 3254;
                    return true;
                }
                break;
            case 3254:
                if ('r' == current) {
                    state_ = 3255;
                    return true;
                }
                break;
            case 3255:
                if ('r' == current) {
                    state_ = 3256;
                    return true;
                }
                break;
            case 3256:
                // angzarr;
                if (';' == current) {
                    match_ = "\u237C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3257:
                if ('g' == current) {
                    state_ = 3258;
                    return true;
                }
                else if ('p' == current) {
                    state_ = 3261;
                    return true;
                }
                break;
            case 3258:
                if ('o' == current) {
                    state_ = 3259;
                    return true;
                }
                break;
            case 3259:
                if ('n' == current) {
                    state_ = 3260;
                    return true;
                }
                break;
            case 3260:
                // aogon;
                if (';' == current) {
                    match_ = "\u0105";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3261:
                if ('f' == current) {
                    state_ = 3262;
                    return true;
                }
                break;
            case 3262:
                // aopf;
                if (';' == current) {
                    match_ = "\uD835\uDD52";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3263:
                switch (current) {
                    case ';': // ap;
                        match_ = "\u2248";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E':
                        state_ = 3264;
                        return true;
                    case 'a':
                        state_ = 3265;
                        return true;
                    case 'e':
                        state_ = 3269;
                        return true;
                    case 'i':
                        state_ = 3270;
                        return true;
                    case 'o':
                        state_ = 3272;
                        return true;
                    case 'p':
                        state_ = 3274;
                        return true;
                }
                break;
            case 3264:
                // apE;
                if (';' == current) {
                    match_ = "\u2A70";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3265:
                if ('c' == current) {
                    state_ = 3266;
                    return true;
                }
                break;
            case 3266:
                if ('i' == current) {
                    state_ = 3267;
                    return true;
                }
                break;
            case 3267:
                if ('r' == current) {
                    state_ = 3268;
                    return true;
                }
                break;
            case 3268:
                // apacir;
                if (';' == current) {
                    match_ = "\u2A6F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3269:
                // ape;
                if (';' == current) {
                    match_ = "\u224A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3270:
                if ('d' == current) {
                    state_ = 3271;
                    return true;
                }
                break;
            case 3271:
                // apid;
                if (';' == current) {
                    match_ = "\u224B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3272:
                if ('s' == current) {
                    state_ = 3273;
                    return true;
                }
                break;
            case 3273:
                // apos;
                if (';' == current) {
                    match_ = "'";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3274:
                if ('r' == current) {
                    state_ = 3275;
                    return true;
                }
                break;
            case 3275:
                if ('o' == current) {
                    state_ = 3276;
                    return true;
                }
                break;
            case 3276:
                if ('x' == current) {
                    state_ = 3277;
                    return true;
                }
                break;
            case 3277:
                // approx;
                if (';' == current) {
                    match_ = "\u2248";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state_ = 3278;
                    return true;
                }
                break;
            case 3278:
                if ('q' == current) {
                    state_ = 3279;
                    return true;
                }
                break;
            case 3279:
                // approxeq;
                if (';' == current) {
                    match_ = "\u224A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3280:
                if ('i' == current) {
                    state_ = 3281;
                    return true;
                }
                break;
            case 3281:
                if ('n' == current) {
                    state_ = 3282;
                    return true;
                }
                break;
            case 3282:
                // aring
                if ('g' == current) {
                    match_ = "\u00E5";
                    matchLength_ = consumedCount_;
                    state_ = 3283;
                    return true;
                }
                break;
            case 3283:
                // aring;
                if (';' == current) {
                    match_ = "\u00E5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3284:
                switch (current) {
                    case 'c':
                        state_ = 3285;
                        return true;
                    case 't':
                        state_ = 3287;
                        return true;
                    case 'y':
                        state_ = 3288;
                        return true;
                }
                break;
            case 3285:
                if ('r' == current) {
                    state_ = 3286;
                    return true;
                }
                break;
            case 3286:
                // ascr;
                if (';' == current) {
                    match_ = "\uD835\uDCB6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3287:
                // ast;
                if (';' == current) {
                    match_ = "*";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3288:
                if ('m' == current) {
                    state_ = 3289;
                    return true;
                }
                break;
            case 3289:
                if ('p' == current) {
                    state_ = 3290;
                    return true;
                }
                break;
            case 3290:
                // asymp;
                if (';' == current) {
                    match_ = "\u2248";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state_ = 3291;
                    return true;
                }
                break;
            case 3291:
                if ('q' == current) {
                    state_ = 3292;
                    return true;
                }
                break;
            case 3292:
                // asympeq;
                if (';' == current) {
                    match_ = "\u224D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3293:
                if ('i' == current) {
                    state_ = 3294;
                    return true;
                }
                break;
            case 3294:
                if ('l' == current) {
                    state_ = 3295;
                    return true;
                }
                break;
            case 3295:
                if ('d' == current) {
                    state_ = 3296;
                    return true;
                }
                break;
            case 3296:
                // atilde
                if ('e' == current) {
                    match_ = "\u00E3";
                    matchLength_ = consumedCount_;
                    state_ = 3297;
                    return true;
                }
                break;
            case 3297:
                // atilde;
                if (';' == current) {
                    match_ = "\u00E3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3298:
                if ('m' == current) {
                    state_ = 3299;
                    return true;
                }
                break;
            case 3299:
                // auml
                if ('l' == current) {
                    match_ = "\u00E4";
                    matchLength_ = consumedCount_;
                    state_ = 3300;
                    return true;
                }
                break;
            case 3300:
                // auml;
                if (';' == current) {
                    match_ = "\u00E4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3301:
                if ('c' == current) {
                    state_ = 3302;
                    return true;
                }
                else if ('i' == current) {
                    state_ = 3308;
                    return true;
                }
                break;
            case 3302:
                if ('o' == current) {
                    state_ = 3303;
                    return true;
                }
                break;
            case 3303:
                if ('n' == current) {
                    state_ = 3304;
                    return true;
                }
                break;
            case 3304:
                if ('i' == current) {
                    state_ = 3305;
                    return true;
                }
                break;
            case 3305:
                if ('n' == current) {
                    state_ = 3306;
                    return true;
                }
                break;
            case 3306:
                if ('t' == current) {
                    state_ = 3307;
                    return true;
                }
                break;
            case 3307:
                // awconint;
                if (';' == current) {
                    match_ = "\u2233";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3308:
                if ('n' == current) {
                    state_ = 3309;
                    return true;
                }
                break;
            case 3309:
                if ('t' == current) {
                    state_ = 3310;
                    return true;
                }
                break;
            case 3310:
                // awint;
                if (';' == current) {
                    match_ = "\u2A11";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3311:
                switch (current) {
                    case 'N':
                        state_ = 3312;
                        return true;
                    case 'a':
                        state_ = 3315;
                        return true;
                    case 'b':
                        state_ = 3348;
                        return true;
                    case 'c':
                        state_ = 3355;
                        return true;
                    case 'd':
                        state_ = 3360;
                        return true;
                    case 'e':
                        state_ = 3364;
                        return true;
                    case 'f':
                        state_ = 3389;
                        return true;
                    case 'i':
                        state_ = 3391;
                        return true;
                    case 'k':
                        state_ = 3449;
                        return true;
                    case 'l':
                        state_ = 3454;
                        return true;
                    case 'n':
                        state_ = 3503;
                        return true;
                    case 'o':
                        state_ = 3511;
                        return true;
                    case 'p':
                        state_ = 3584;
                        return true;
                    case 'r':
                        state_ = 3589;
                        return true;
                    case 's':
                        state_ = 3597;
                        return true;
                    case 'u':
                        state_ = 3613;
                        return true;
                }
                break;
            case 3312:
                if ('o' == current) {
                    state_ = 3313;
                    return true;
                }
                break;
            case 3313:
                if ('t' == current) {
                    state_ = 3314;
                    return true;
                }
                break;
            case 3314:
                // bNot;
                if (';' == current) {
                    match_ = "\u2AED";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3315:
                if ('c' == current) {
                    state_ = 3316;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 3339;
                    return true;
                }
                break;
            case 3316:
                if ('k' == current) {
                    state_ = 3317;
                    return true;
                }
                break;
            case 3317:
                switch (current) {
                    case 'c':
                        state_ = 3318;
                        return true;
                    case 'e':
                        state_ = 3322;
                        return true;
                    case 'p':
                        state_ = 3329;
                        return true;
                    case 's':
                        state_ = 3334;
                        return true;
                }
                break;
            case 3318:
                if ('o' == current) {
                    state_ = 3319;
                    return true;
                }
                break;
            case 3319:
                if ('n' == current) {
                    state_ = 3320;
                    return true;
                }
                break;
            case 3320:
                if ('g' == current) {
                    state_ = 3321;
                    return true;
                }
                break;
            case 3321:
                // backcong;
                if (';' == current) {
                    match_ = "\u224C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3322:
                if ('p' == current) {
                    state_ = 3323;
                    return true;
                }
                break;
            case 3323:
                if ('s' == current) {
                    state_ = 3324;
                    return true;
                }
                break;
            case 3324:
                if ('i' == current) {
                    state_ = 3325;
                    return true;
                }
                break;
            case 3325:
                if ('l' == current) {
                    state_ = 3326;
                    return true;
                }
                break;
            case 3326:
                if ('o' == current) {
                    state_ = 3327;
                    return true;
                }
                break;
            case 3327:
                if ('n' == current) {
                    state_ = 3328;
                    return true;
                }
                break;
            case 3328:
                // backepsilon;
                if (';' == current) {
                    match_ = "\u03F6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3329:
                if ('r' == current) {
                    state_ = 3330;
                    return true;
                }
                break;
            case 3330:
                if ('i' == current) {
                    state_ = 3331;
                    return true;
                }
                break;
            case 3331:
                if ('m' == current) {
                    state_ = 3332;
                    return true;
                }
                break;
            case 3332:
                if ('e' == current) {
                    state_ = 3333;
                    return true;
                }
                break;
            case 3333:
                // backprime;
                if (';' == current) {
                    match_ = "\u2035";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3334:
                if ('i' == current) {
                    state_ = 3335;
                    return true;
                }
                break;
            case 3335:
                if ('m' == current) {
                    state_ = 3336;
                    return true;
                }
                break;
            case 3336:
                // backsim;
                if (';' == current) {
                    match_ = "\u223D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state_ = 3337;
                    return true;
                }
                break;
            case 3337:
                if ('q' == current) {
                    state_ = 3338;
                    return true;
                }
                break;
            case 3338:
                // backsimeq;
                if (';' == current) {
                    match_ = "\u22CD";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3339:
                if ('v' == current) {
                    state_ = 3340;
                    return true;
                }
                else if ('w' == current) {
                    state_ = 3343;
                    return true;
                }
                break;
            case 3340:
                if ('e' == current) {
                    state_ = 3341;
                    return true;
                }
                break;
            case 3341:
                if ('e' == current) {
                    state_ = 3342;
                    return true;
                }
                break;
            case 3342:
                // barvee;
                if (';' == current) {
                    match_ = "\u22BD";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3343:
                if ('e' == current) {
                    state_ = 3344;
                    return true;
                }
                break;
            case 3344:
                if ('d' == current) {
                    state_ = 3345;
                    return true;
                }
                break;
            case 3345:
                // barwed;
                if (';' == current) {
                    match_ = "\u2305";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('g' == current) {
                    state_ = 3346;
                    return true;
                }
                break;
            case 3346:
                if ('e' == current) {
                    state_ = 3347;
                    return true;
                }
                break;
            case 3347:
                // barwedge;
                if (';' == current) {
                    match_ = "\u2305";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3348:
                if ('r' == current) {
                    state_ = 3349;
                    return true;
                }
                break;
            case 3349:
                if ('k' == current) {
                    state_ = 3350;
                    return true;
                }
                break;
            case 3350:
                // bbrk;
                if (';' == current) {
                    match_ = "\u23B5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('t' == current) {
                    state_ = 3351;
                    return true;
                }
                break;
            case 3351:
                if ('b' == current) {
                    state_ = 3352;
                    return true;
                }
                break;
            case 3352:
                if ('r' == current) {
                    state_ = 3353;
                    return true;
                }
                break;
            case 3353:
                if ('k' == current) {
                    state_ = 3354;
                    return true;
                }
                break;
            case 3354:
                // bbrktbrk;
                if (';' == current) {
                    match_ = "\u23B6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3355:
                if ('o' == current) {
                    state_ = 3356;
                    return true;
                }
                else if ('y' == current) {
                    state_ = 3359;
                    return true;
                }
                break;
            case 3356:
                if ('n' == current) {
                    state_ = 3357;
                    return true;
                }
                break;
            case 3357:
                if ('g' == current) {
                    state_ = 3358;
                    return true;
                }
                break;
            case 3358:
                // bcong;
                if (';' == current) {
                    match_ = "\u224C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3359:
                // bcy;
                if (';' == current) {
                    match_ = "\u0431";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3360:
                if ('q' == current) {
                    state_ = 3361;
                    return true;
                }
                break;
            case 3361:
                if ('u' == current) {
                    state_ = 3362;
                    return true;
                }
                break;
            case 3362:
                if ('o' == current) {
                    state_ = 3363;
                    return true;
                }
                break;
            case 3363:
                // bdquo;
                if (';' == current) {
                    match_ = "\u201E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3364:
                switch (current) {
                    case 'c':
                        state_ = 3365;
                        return true;
                    case 'm':
                        state_ = 3370;
                        return true;
                    case 'p':
                        state_ = 3375;
                        return true;
                    case 'r':
                        state_ = 3378;
                        return true;
                    case 't':
                        state_ = 3382;
                        return true;
                }
                break;
            case 3365:
                if ('a' == current) {
                    state_ = 3366;
                    return true;
                }
                break;
            case 3366:
                if ('u' == current) {
                    state_ = 3367;
                    return true;
                }
                break;
            case 3367:
                if ('s' == current) {
                    state_ = 3368;
                    return true;
                }
                break;
            case 3368:
                // becaus;
                if (';' == current) {
                    match_ = "\u2235";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state_ = 3369;
                    return true;
                }
                break;
            case 3369:
                // because;
                if (';' == current) {
                    match_ = "\u2235";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3370:
                if ('p' == current) {
                    state_ = 3371;
                    return true;
                }
                break;
            case 3371:
                if ('t' == current) {
                    state_ = 3372;
                    return true;
                }
                break;
            case 3372:
                if ('y' == current) {
                    state_ = 3373;
                    return true;
                }
                break;
            case 3373:
                if ('v' == current) {
                    state_ = 3374;
                    return true;
                }
                break;
            case 3374:
                // bemptyv;
                if (';' == current) {
                    match_ = "\u29B0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3375:
                if ('s' == current) {
                    state_ = 3376;
                    return true;
                }
                break;
            case 3376:
                if ('i' == current) {
                    state_ = 3377;
                    return true;
                }
                break;
            case 3377:
                // bepsi;
                if (';' == current) {
                    match_ = "\u03F6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3378:
                if ('n' == current) {
                    state_ = 3379;
                    return true;
                }
                break;
            case 3379:
                if ('o' == current) {
                    state_ = 3380;
                    return true;
                }
                break;
            case 3380:
                if ('u' == current) {
                    state_ = 3381;
                    return true;
                }
                break;
            case 3381:
                // bernou;
                if (';' == current) {
                    match_ = "\u212C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3382:
                switch (current) {
                    case 'a':
                        state_ = 3383;
                        return true;
                    case 'h':
                        state_ = 3384;
                        return true;
                    case 'w':
                        state_ = 3385;
                        return true;
                }
                break;
            case 3383:
                // beta;
                if (';' == current) {
                    match_ = "\u03B2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3384:
                // beth;
                if (';' == current) {
                    match_ = "\u2136";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3385:
                if ('e' == current) {
                    state_ = 3386;
                    return true;
                }
                break;
            case 3386:
                if ('e' == current) {
                    state_ = 3387;
                    return true;
                }
                break;
            case 3387:
                if ('n' == current) {
                    state_ = 3388;
                    return true;
                }
                break;
            case 3388:
                // between;
                if (';' == current) {
                    match_ = "\u226C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3389:
                if ('r' == current) {
                    state_ = 3390;
                    return true;
                }
                break;
            case 3390:
                // bfr;
                if (';' == current) {
                    match_ = "\uD835\uDD1F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3391:
                if ('g' == current) {
                    state_ = 3392;
                    return true;
                }
                break;
            case 3392:
                switch (current) {
                    case 'c':
                        state_ = 3393;
                        return true;
                    case 'o':
                        state_ = 3401;
                        return true;
                    case 's':
                        state_ = 3414;
                        return true;
                    case 't':
                        state_ = 3422;
                        return true;
                    case 'u':
                        state_ = 3436;
                        return true;
                    case 'v':
                        state_ = 3441;
                        return true;
                    case 'w':
                        state_ = 3444;
                        return true;
                }
                break;
            case 3393:
                switch (current) {
                    case 'a':
                        state_ = 3394;
                        return true;
                    case 'i':
                        state_ = 3396;
                        return true;
                    case 'u':
                        state_ = 3399;
                        return true;
                }
                break;
            case 3394:
                if ('p' == current) {
                    state_ = 3395;
                    return true;
                }
                break;
            case 3395:
                // bigcap;
                if (';' == current) {
                    match_ = "\u22C2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3396:
                if ('r' == current) {
                    state_ = 3397;
                    return true;
                }
                break;
            case 3397:
                if ('c' == current) {
                    state_ = 3398;
                    return true;
                }
                break;
            case 3398:
                // bigcirc;
                if (';' == current) {
                    match_ = "\u25EF";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3399:
                if ('p' == current) {
                    state_ = 3400;
                    return true;
                }
                break;
            case 3400:
                // bigcup;
                if (';' == current) {
                    match_ = "\u22C3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3401:
                switch (current) {
                    case 'd':
                        state_ = 3402;
                        return true;
                    case 'p':
                        state_ = 3405;
                        return true;
                    case 't':
                        state_ = 3409;
                        return true;
                }
                break;
            case 3402:
                if ('o' == current) {
                    state_ = 3403;
                    return true;
                }
                break;
            case 3403:
                if ('t' == current) {
                    state_ = 3404;
                    return true;
                }
                break;
            case 3404:
                // bigodot;
                if (';' == current) {
                    match_ = "\u2A00";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3405:
                if ('l' == current) {
                    state_ = 3406;
                    return true;
                }
                break;
            case 3406:
                if ('u' == current) {
                    state_ = 3407;
                    return true;
                }
                break;
            case 3407:
                if ('s' == current) {
                    state_ = 3408;
                    return true;
                }
                break;
            case 3408:
                // bigoplus;
                if (';' == current) {
                    match_ = "\u2A01";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3409:
                if ('i' == current) {
                    state_ = 3410;
                    return true;
                }
                break;
            case 3410:
                if ('m' == current) {
                    state_ = 3411;
                    return true;
                }
                break;
            case 3411:
                if ('e' == current) {
                    state_ = 3412;
                    return true;
                }
                break;
            case 3412:
                if ('s' == current) {
                    state_ = 3413;
                    return true;
                }
                break;
            case 3413:
                // bigotimes;
                if (';' == current) {
                    match_ = "\u2A02";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3414:
                if ('q' == current) {
                    state_ = 3415;
                    return true;
                }
                else if ('t' == current) {
                    state_ = 3419;
                    return true;
                }
                break;
            case 3415:
                if ('c' == current) {
                    state_ = 3416;
                    return true;
                }
                break;
            case 3416:
                if ('u' == current) {
                    state_ = 3417;
                    return true;
                }
                break;
            case 3417:
                if ('p' == current) {
                    state_ = 3418;
                    return true;
                }
                break;
            case 3418:
                // bigsqcup;
                if (';' == current) {
                    match_ = "\u2A06";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3419:
                if ('a' == current) {
                    state_ = 3420;
                    return true;
                }
                break;
            case 3420:
                if ('r' == current) {
                    state_ = 3421;
                    return true;
                }
                break;
            case 3421:
                // bigstar;
                if (';' == current) {
                    match_ = "\u2605";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3422:
                if ('r' == current) {
                    state_ = 3423;
                    return true;
                }
                break;
            case 3423:
                if ('i' == current) {
                    state_ = 3424;
                    return true;
                }
                break;
            case 3424:
                if ('a' == current) {
                    state_ = 3425;
                    return true;
                }
                break;
            case 3425:
                if ('n' == current) {
                    state_ = 3426;
                    return true;
                }
                break;
            case 3426:
                if ('g' == current) {
                    state_ = 3427;
                    return true;
                }
                break;
            case 3427:
                if ('l' == current) {
                    state_ = 3428;
                    return true;
                }
                break;
            case 3428:
                if ('e' == current) {
                    state_ = 3429;
                    return true;
                }
                break;
            case 3429:
                if ('d' == current) {
                    state_ = 3430;
                    return true;
                }
                else if ('u' == current) {
                    state_ = 3434;
                    return true;
                }
                break;
            case 3430:
                if ('o' == current) {
                    state_ = 3431;
                    return true;
                }
                break;
            case 3431:
                if ('w' == current) {
                    state_ = 3432;
                    return true;
                }
                break;
            case 3432:
                if ('n' == current) {
                    state_ = 3433;
                    return true;
                }
                break;
            case 3433:
                // bigtriangledown;
                if (';' == current) {
                    match_ = "\u25BD";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3434:
                if ('p' == current) {
                    state_ = 3435;
                    return true;
                }
                break;
            case 3435:
                // bigtriangleup;
                if (';' == current) {
                    match_ = "\u25B3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3436:
                if ('p' == current) {
                    state_ = 3437;
                    return true;
                }
                break;
            case 3437:
                if ('l' == current) {
                    state_ = 3438;
                    return true;
                }
                break;
            case 3438:
                if ('u' == current) {
                    state_ = 3439;
                    return true;
                }
                break;
            case 3439:
                if ('s' == current) {
                    state_ = 3440;
                    return true;
                }
                break;
            case 3440:
                // biguplus;
                if (';' == current) {
                    match_ = "\u2A04";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3441:
                if ('e' == current) {
                    state_ = 3442;
                    return true;
                }
                break;
            case 3442:
                if ('e' == current) {
                    state_ = 3443;
                    return true;
                }
                break;
            case 3443:
                // bigvee;
                if (';' == current) {
                    match_ = "\u22C1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3444:
                if ('e' == current) {
                    state_ = 3445;
                    return true;
                }
                break;
            case 3445:
                if ('d' == current) {
                    state_ = 3446;
                    return true;
                }
                break;
            case 3446:
                if ('g' == current) {
                    state_ = 3447;
                    return true;
                }
                break;
            case 3447:
                if ('e' == current) {
                    state_ = 3448;
                    return true;
                }
                break;
            case 3448:
                // bigwedge;
                if (';' == current) {
                    match_ = "\u22C0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3449:
                if ('a' == current) {
                    state_ = 3450;
                    return true;
                }
                break;
            case 3450:
                if ('r' == current) {
                    state_ = 3451;
                    return true;
                }
                break;
            case 3451:
                if ('o' == current) {
                    state_ = 3452;
                    return true;
                }
                break;
            case 3452:
                if ('w' == current) {
                    state_ = 3453;
                    return true;
                }
                break;
            case 3453:
                // bkarow;
                if (';' == current) {
                    match_ = "\u290D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3454:
                switch (current) {
                    case 'a':
                        state_ = 3455;
                        return true;
                    case 'k':
                        state_ = 3494;
                        return true;
                    case 'o':
                        state_ = 3500;
                        return true;
                }
                break;
            case 3455:
                if ('c' == current) {
                    state_ = 3456;
                    return true;
                }
                else if ('n' == current) {
                    state_ = 3492;
                    return true;
                }
                break;
            case 3456:
                if ('k' == current) {
                    state_ = 3457;
                    return true;
                }
                break;
            case 3457:
                switch (current) {
                    case 'l':
                        state_ = 3458;
                        return true;
                    case 's':
                        state_ = 3465;
                        return true;
                    case 't':
                        state_ = 3471;
                        return true;
                }
                break;
            case 3458:
                if ('o' == current) {
                    state_ = 3459;
                    return true;
                }
                break;
            case 3459:
                if ('z' == current) {
                    state_ = 3460;
                    return true;
                }
                break;
            case 3460:
                if ('e' == current) {
                    state_ = 3461;
                    return true;
                }
                break;
            case 3461:
                if ('n' == current) {
                    state_ = 3462;
                    return true;
                }
                break;
            case 3462:
                if ('g' == current) {
                    state_ = 3463;
                    return true;
                }
                break;
            case 3463:
                if ('e' == current) {
                    state_ = 3464;
                    return true;
                }
                break;
            case 3464:
                // blacklozenge;
                if (';' == current) {
                    match_ = "\u29EB";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3465:
                if ('q' == current) {
                    state_ = 3466;
                    return true;
                }
                break;
            case 3466:
                if ('u' == current) {
                    state_ = 3467;
                    return true;
                }
                break;
            case 3467:
                if ('a' == current) {
                    state_ = 3468;
                    return true;
                }
                break;
            case 3468:
                if ('r' == current) {
                    state_ = 3469;
                    return true;
                }
                break;
            case 3469:
                if ('e' == current) {
                    state_ = 3470;
                    return true;
                }
                break;
            case 3470:
                // blacksquare;
                if (';' == current) {
                    match_ = "\u25AA";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3471:
                if ('r' == current) {
                    state_ = 3472;
                    return true;
                }
                break;
            case 3472:
                if ('i' == current) {
                    state_ = 3473;
                    return true;
                }
                break;
            case 3473:
                if ('a' == current) {
                    state_ = 3474;
                    return true;
                }
                break;
            case 3474:
                if ('n' == current) {
                    state_ = 3475;
                    return true;
                }
                break;
            case 3475:
                if ('g' == current) {
                    state_ = 3476;
                    return true;
                }
                break;
            case 3476:
                if ('l' == current) {
                    state_ = 3477;
                    return true;
                }
                break;
            case 3477:
                if ('e' == current) {
                    state_ = 3478;
                    return true;
                }
                break;
            case 3478:
                switch (current) {
                    case ';': // blacktriangle;
                        match_ = "\u25B4";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'd':
                        state_ = 3479;
                        return true;
                    case 'l':
                        state_ = 3483;
                        return true;
                    case 'r':
                        state_ = 3487;
                        return true;
                }
                break;
            case 3479:
                if ('o' == current) {
                    state_ = 3480;
                    return true;
                }
                break;
            case 3480:
                if ('w' == current) {
                    state_ = 3481;
                    return true;
                }
                break;
            case 3481:
                if ('n' == current) {
                    state_ = 3482;
                    return true;
                }
                break;
            case 3482:
                // blacktriangledown;
                if (';' == current) {
                    match_ = "\u25BE";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3483:
                if ('e' == current) {
                    state_ = 3484;
                    return true;
                }
                break;
            case 3484:
                if ('f' == current) {
                    state_ = 3485;
                    return true;
                }
                break;
            case 3485:
                if ('t' == current) {
                    state_ = 3486;
                    return true;
                }
                break;
            case 3486:
                // blacktriangleleft;
                if (';' == current) {
                    match_ = "\u25C2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3487:
                if ('i' == current) {
                    state_ = 3488;
                    return true;
                }
                break;
            case 3488:
                if ('g' == current) {
                    state_ = 3489;
                    return true;
                }
                break;
            case 3489:
                if ('h' == current) {
                    state_ = 3490;
                    return true;
                }
                break;
            case 3490:
                if ('t' == current) {
                    state_ = 3491;
                    return true;
                }
                break;
            case 3491:
                // blacktriangleright;
                if (';' == current) {
                    match_ = "\u25B8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3492:
                if ('k' == current) {
                    state_ = 3493;
                    return true;
                }
                break;
            case 3493:
                // blank;
                if (';' == current) {
                    match_ = "\u2423";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3494:
                if ('1' == current) {
                    state_ = 3495;
                    return true;
                }
                else if ('3' == current) {
                    state_ = 3498;
                    return true;
                }
                break;
            case 3495:
                if ('2' == current) {
                    state_ = 3496;
                    return true;
                }
                else if ('4' == current) {
                    state_ = 3497;
                    return true;
                }
                break;
            case 3496:
                // blk12;
                if (';' == current) {
                    match_ = "\u2592";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3497:
                // blk14;
                if (';' == current) {
                    match_ = "\u2591";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3498:
                if ('4' == current) {
                    state_ = 3499;
                    return true;
                }
                break;
            case 3499:
                // blk34;
                if (';' == current) {
                    match_ = "\u2593";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3500:
                if ('c' == current) {
                    state_ = 3501;
                    return true;
                }
                break;
            case 3501:
                if ('k' == current) {
                    state_ = 3502;
                    return true;
                }
                break;
            case 3502:
                // block;
                if (';' == current) {
                    match_ = "\u2588";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3503:
                if ('e' == current) {
                    state_ = 3504;
                    return true;
                }
                else if ('o' == current) {
                    state_ = 3509;
                    return true;
                }
                break;
            case 3504:
                // bne;
                if (';' == current) {
                    match_ = "=\u20E5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('q' == current) {
                    state_ = 3505;
                    return true;
                }
                break;
            case 3505:
                if ('u' == current) {
                    state_ = 3506;
                    return true;
                }
                break;
            case 3506:
                if ('i' == current) {
                    state_ = 3507;
                    return true;
                }
                break;
            case 3507:
                if ('v' == current) {
                    state_ = 3508;
                    return true;
                }
                break;
            case 3508:
                // bnequiv;
                if (';' == current) {
                    match_ = "\u2261\u20E5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3509:
                if ('t' == current) {
                    state_ = 3510;
                    return true;
                }
                break;
            case 3510:
                // bnot;
                if (';' == current) {
                    match_ = "\u2310";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3511:
                switch (current) {
                    case 'p':
                        state_ = 3512;
                        return true;
                    case 't':
                        state_ = 3514;
                        return true;
                    case 'w':
                        state_ = 3518;
                        return true;
                    case 'x':
                        state_ = 3522;
                        return true;
                }
                break;
            case 3512:
                if ('f' == current) {
                    state_ = 3513;
                    return true;
                }
                break;
            case 3513:
                // bopf;
                if (';' == current) {
                    match_ = "\uD835\uDD53";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3514:
                // bot;
                if (';' == current) {
                    match_ = "\u22A5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('t' == current) {
                    state_ = 3515;
                    return true;
                }
                break;
            case 3515:
                if ('o' == current) {
                    state_ = 3516;
                    return true;
                }
                break;
            case 3516:
                if ('m' == current) {
                    state_ = 3517;
                    return true;
                }
                break;
            case 3517:
                // bottom;
                if (';' == current) {
                    match_ = "\u22A5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3518:
                if ('t' == current) {
                    state_ = 3519;
                    return true;
                }
                break;
            case 3519:
                if ('i' == current) {
                    state_ = 3520;
                    return true;
                }
                break;
            case 3520:
                if ('e' == current) {
                    state_ = 3521;
                    return true;
                }
                break;
            case 3521:
                // bowtie;
                if (';' == current) {
                    match_ = "\u22C8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3522:
                switch (current) {
                    case 'D':
                        state_ = 3523;
                        return true;
                    case 'H':
                        state_ = 3528;
                        return true;
                    case 'U':
                        state_ = 3533;
                        return true;
                    case 'V':
                        state_ = 3538;
                        return true;
                    case 'b':
                        state_ = 3545;
                        return true;
                    case 'd':
                        state_ = 3548;
                        return true;
                    case 'h':
                        state_ = 3553;
                        return true;
                    case 'm':
                        state_ = 3558;
                        return true;
                    case 'p':
                        state_ = 3563;
                        return true;
                    case 't':
                        state_ = 3567;
                        return true;
                    case 'u':
                        state_ = 3572;
                        return true;
                    case 'v':
                        state_ = 3577;
                        return true;
                }
                break;
            case 3523:
                switch (current) {
                    case 'L':
                        state_ = 3524;
                        return true;
                    case 'R':
                        state_ = 3525;
                        return true;
                    case 'l':
                        state_ = 3526;
                        return true;
                    case 'r':
                        state_ = 3527;
                        return true;
                }
                break;
            case 3524:
                // boxDL;
                if (';' == current) {
                    match_ = "\u2557";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3525:
                // boxDR;
                if (';' == current) {
                    match_ = "\u2554";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3526:
                // boxDl;
                if (';' == current) {
                    match_ = "\u2556";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3527:
                // boxDr;
                if (';' == current) {
                    match_ = "\u2553";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3528:
                switch (current) {
                    case ';': // boxH;
                        match_ = "\u2550";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'D':
                        state_ = 3529;
                        return true;
                    case 'U':
                        state_ = 3530;
                        return true;
                    case 'd':
                        state_ = 3531;
                        return true;
                    case 'u':
                        state_ = 3532;
                        return true;
                }
                break;
            case 3529:
                // boxHD;
                if (';' == current) {
                    match_ = "\u2566";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3530:
                // boxHU;
                if (';' == current) {
                    match_ = "\u2569";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3531:
                // boxHd;
                if (';' == current) {
                    match_ = "\u2564";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3532:
                // boxHu;
                if (';' == current) {
                    match_ = "\u2567";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3533:
                switch (current) {
                    case 'L':
                        state_ = 3534;
                        return true;
                    case 'R':
                        state_ = 3535;
                        return true;
                    case 'l':
                        state_ = 3536;
                        return true;
                    case 'r':
                        state_ = 3537;
                        return true;
                }
                break;
            case 3534:
                // boxUL;
                if (';' == current) {
                    match_ = "\u255D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3535:
                // boxUR;
                if (';' == current) {
                    match_ = "\u255A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3536:
                // boxUl;
                if (';' == current) {
                    match_ = "\u255C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3537:
                // boxUr;
                if (';' == current) {
                    match_ = "\u2559";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3538:
                switch (current) {
                    case ';': // boxV;
                        match_ = "\u2551";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'H':
                        state_ = 3539;
                        return true;
                    case 'L':
                        state_ = 3540;
                        return true;
                    case 'R':
                        state_ = 3541;
                        return true;
                    case 'h':
                        state_ = 3542;
                        return true;
                    case 'l':
                        state_ = 3543;
                        return true;
                    case 'r':
                        state_ = 3544;
                        return true;
                }
                break;
            case 3539:
                // boxVH;
                if (';' == current) {
                    match_ = "\u256C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3540:
                // boxVL;
                if (';' == current) {
                    match_ = "\u2563";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3541:
                // boxVR;
                if (';' == current) {
                    match_ = "\u2560";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3542:
                // boxVh;
                if (';' == current) {
                    match_ = "\u256B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3543:
                // boxVl;
                if (';' == current) {
                    match_ = "\u2562";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3544:
                // boxVr;
                if (';' == current) {
                    match_ = "\u255F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3545:
                if ('o' == current) {
                    state_ = 3546;
                    return true;
                }
                break;
            case 3546:
                if ('x' == current) {
                    state_ = 3547;
                    return true;
                }
                break;
            case 3547:
                // boxbox;
                if (';' == current) {
                    match_ = "\u29C9";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3548:
                switch (current) {
                    case 'L':
                        state_ = 3549;
                        return true;
                    case 'R':
                        state_ = 3550;
                        return true;
                    case 'l':
                        state_ = 3551;
                        return true;
                    case 'r':
                        state_ = 3552;
                        return true;
                }
                break;
            case 3549:
                // boxdL;
                if (';' == current) {
                    match_ = "\u2555";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3550:
                // boxdR;
                if (';' == current) {
                    match_ = "\u2552";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3551:
                // boxdl;
                if (';' == current) {
                    match_ = "\u2510";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3552:
                // boxdr;
                if (';' == current) {
                    match_ = "\u250C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3553:
                switch (current) {
                    case ';': // boxh;
                        match_ = "\u2500";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'D':
                        state_ = 3554;
                        return true;
                    case 'U':
                        state_ = 3555;
                        return true;
                    case 'd':
                        state_ = 3556;
                        return true;
                    case 'u':
                        state_ = 3557;
                        return true;
                }
                break;
            case 3554:
                // boxhD;
                if (';' == current) {
                    match_ = "\u2565";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3555:
                // boxhU;
                if (';' == current) {
                    match_ = "\u2568";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3556:
                // boxhd;
                if (';' == current) {
                    match_ = "\u252C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3557:
                // boxhu;
                if (';' == current) {
                    match_ = "\u2534";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3558:
                if ('i' == current) {
                    state_ = 3559;
                    return true;
                }
                break;
            case 3559:
                if ('n' == current) {
                    state_ = 3560;
                    return true;
                }
                break;
            case 3560:
                if ('u' == current) {
                    state_ = 3561;
                    return true;
                }
                break;
            case 3561:
                if ('s' == current) {
                    state_ = 3562;
                    return true;
                }
                break;
            case 3562:
                // boxminus;
                if (';' == current) {
                    match_ = "\u229F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3563:
                if ('l' == current) {
                    state_ = 3564;
                    return true;
                }
                break;
            case 3564:
                if ('u' == current) {
                    state_ = 3565;
                    return true;
                }
                break;
            case 3565:
                if ('s' == current) {
                    state_ = 3566;
                    return true;
                }
                break;
            case 3566:
                // boxplus;
                if (';' == current) {
                    match_ = "\u229E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3567:
                if ('i' == current) {
                    state_ = 3568;
                    return true;
                }
                break;
            case 3568:
                if ('m' == current) {
                    state_ = 3569;
                    return true;
                }
                break;
            case 3569:
                if ('e' == current) {
                    state_ = 3570;
                    return true;
                }
                break;
            case 3570:
                if ('s' == current) {
                    state_ = 3571;
                    return true;
                }
                break;
            case 3571:
                // boxtimes;
                if (';' == current) {
                    match_ = "\u22A0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3572:
                switch (current) {
                    case 'L':
                        state_ = 3573;
                        return true;
                    case 'R':
                        state_ = 3574;
                        return true;
                    case 'l':
                        state_ = 3575;
                        return true;
                    case 'r':
                        state_ = 3576;
                        return true;
                }
                break;
            case 3573:
                // boxuL;
                if (';' == current) {
                    match_ = "\u255B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3574:
                // boxuR;
                if (';' == current) {
                    match_ = "\u2558";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3575:
                // boxul;
                if (';' == current) {
                    match_ = "\u2518";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3576:
                // boxur;
                if (';' == current) {
                    match_ = "\u2514";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3577:
                switch (current) {
                    case ';': // boxv;
                        match_ = "\u2502";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'H':
                        state_ = 3578;
                        return true;
                    case 'L':
                        state_ = 3579;
                        return true;
                    case 'R':
                        state_ = 3580;
                        return true;
                    case 'h':
                        state_ = 3581;
                        return true;
                    case 'l':
                        state_ = 3582;
                        return true;
                    case 'r':
                        state_ = 3583;
                        return true;
                }
                break;
            case 3578:
                // boxvH;
                if (';' == current) {
                    match_ = "\u256A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3579:
                // boxvL;
                if (';' == current) {
                    match_ = "\u2561";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3580:
                // boxvR;
                if (';' == current) {
                    match_ = "\u255E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3581:
                // boxvh;
                if (';' == current) {
                    match_ = "\u253C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3582:
                // boxvl;
                if (';' == current) {
                    match_ = "\u2524";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3583:
                // boxvr;
                if (';' == current) {
                    match_ = "\u251C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3584:
                if ('r' == current) {
                    state_ = 3585;
                    return true;
                }
                break;
            case 3585:
                if ('i' == current) {
                    state_ = 3586;
                    return true;
                }
                break;
            case 3586:
                if ('m' == current) {
                    state_ = 3587;
                    return true;
                }
                break;
            case 3587:
                if ('e' == current) {
                    state_ = 3588;
                    return true;
                }
                break;
            case 3588:
                // bprime;
                if (';' == current) {
                    match_ = "\u2035";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3589:
                if ('e' == current) {
                    state_ = 3590;
                    return true;
                }
                else if ('v' == current) {
                    state_ = 3593;
                    return true;
                }
                break;
            case 3590:
                if ('v' == current) {
                    state_ = 3591;
                    return true;
                }
                break;
            case 3591:
                if ('e' == current) {
                    state_ = 3592;
                    return true;
                }
                break;
            case 3592:
                // breve;
                if (';' == current) {
                    match_ = "\u02D8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3593:
                if ('b' == current) {
                    state_ = 3594;
                    return true;
                }
                break;
            case 3594:
                if ('a' == current) {
                    state_ = 3595;
                    return true;
                }
                break;
            case 3595:
                // brvbar
                if ('r' == current) {
                    match_ = "\u00A6";
                    matchLength_ = consumedCount_;
                    state_ = 3596;
                    return true;
                }
                break;
            case 3596:
                // brvbar;
                if (';' == current) {
                    match_ = "\u00A6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3597:
                switch (current) {
                    case 'c':
                        state_ = 3598;
                        return true;
                    case 'e':
                        state_ = 3600;
                        return true;
                    case 'i':
                        state_ = 3603;
                        return true;
                    case 'o':
                        state_ = 3606;
                        return true;
                }
                break;
            case 3598:
                if ('r' == current) {
                    state_ = 3599;
                    return true;
                }
                break;
            case 3599:
                // bscr;
                if (';' == current) {
                    match_ = "\uD835\uDCB7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3600:
                if ('m' == current) {
                    state_ = 3601;
                    return true;
                }
                break;
            case 3601:
                if ('i' == current) {
                    state_ = 3602;
                    return true;
                }
                break;
            case 3602:
                // bsemi;
                if (';' == current) {
                    match_ = "\u204F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3603:
                if ('m' == current) {
                    state_ = 3604;
                    return true;
                }
                break;
            case 3604:
                // bsim;
                if (';' == current) {
                    match_ = "\u223D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state_ = 3605;
                    return true;
                }
                break;
            case 3605:
                // bsime;
                if (';' == current) {
                    match_ = "\u22CD";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3606:
                if ('l' == current) {
                    state_ = 3607;
                    return true;
                }
                break;
            case 3607:
                switch (current) {
                    case ';': // bsol;
                        match_ = "\\";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'b':
                        state_ = 3608;
                        return true;
                    case 'h':
                        state_ = 3609;
                        return true;
                }
                break;
            case 3608:
                // bsolb;
                if (';' == current) {
                    match_ = "\u29C5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3609:
                if ('s' == current) {
                    state_ = 3610;
                    return true;
                }
                break;
            case 3610:
                if ('u' == current) {
                    state_ = 3611;
                    return true;
                }
                break;
            case 3611:
                if ('b' == current) {
                    state_ = 3612;
                    return true;
                }
                break;
            case 3612:
                // bsolhsub;
                if (';' == current) {
                    match_ = "\u27C8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3613:
                if ('l' == current) {
                    state_ = 3614;
                    return true;
                }
                else if ('m' == current) {
                    state_ = 3618;
                    return true;
                }
                break;
            case 3614:
                if ('l' == current) {
                    state_ = 3615;
                    return true;
                }
                break;
            case 3615:
                // bull;
                if (';' == current) {
                    match_ = "\u2022";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state_ = 3616;
                    return true;
                }
                break;
            case 3616:
                if ('t' == current) {
                    state_ = 3617;
                    return true;
                }
                break;
            case 3617:
                // bullet;
                if (';' == current) {
                    match_ = "\u2022";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3618:
                if ('p' == current) {
                    state_ = 3619;
                    return true;
                }
                break;
            case 3619:
                switch (current) {
                    case ';': // bump;
                        match_ = "\u224E";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E':
                        state_ = 3620;
                        return true;
                    case 'e':
                        state_ = 3621;
                        return true;
                }
                break;
            case 3620:
                // bumpE;
                if (';' == current) {
                    match_ = "\u2AAE";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3621:
                // bumpe;
                if (';' == current) {
                    match_ = "\u224F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('q' == current) {
                    state_ = 3622;
                    return true;
                }
                break;
            case 3622:
                // bumpeq;
                if (';' == current) {
                    match_ = "\u224F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3623:
                switch (current) {
                    case 'a':
                        state_ = 3624;
                        return true;
                    case 'c':
                        state_ = 3652;
                        return true;
                    case 'd':
                        state_ = 3671;
                        return true;
                    case 'e':
                        state_ = 3674;
                        return true;
                    case 'f':
                        state_ = 3690;
                        return true;
                    case 'h':
                        state_ = 3692;
                        return true;
                    case 'i':
                        state_ = 3703;
                        return true;
                    case 'l':
                        state_ = 3752;
                        return true;
                    case 'o':
                        state_ = 3759;
                        return true;
                    case 'r':
                        state_ = 3797;
                        return true;
                    case 's':
                        state_ = 3804;
                        return true;
                    case 't':
                        state_ = 3812;
                        return true;
                    case 'u':
                        state_ = 3816;
                        return true;
                    case 'w':
                        state_ = 3900;
                        return true;
                    case 'y':
                        state_ = 3910;
                        return true;
                }
                break;
            case 3624:
                switch (current) {
                    case 'c':
                        state_ = 3625;
                        return true;
                    case 'p':
                        state_ = 3629;
                        return true;
                    case 'r':
                        state_ = 3647;
                        return true;
                }
                break;
            case 3625:
                if ('u' == current) {
                    state_ = 3626;
                    return true;
                }
                break;
            case 3626:
                if ('t' == current) {
                    state_ = 3627;
                    return true;
                }
                break;
            case 3627:
                if ('e' == current) {
                    state_ = 3628;
                    return true;
                }
                break;
            case 3628:
                // cacute;
                if (';' == current) {
                    match_ = "\u0107";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3629:
                switch (current) {
                    case ';': // cap;
                        match_ = "\u2229";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'a':
                        state_ = 3630;
                        return true;
                    case 'b':
                        state_ = 3633;
                        return true;
                    case 'c':
                        state_ = 3638;
                        return true;
                    case 'd':
                        state_ = 3643;
                        return true;
                    case 's':
                        state_ = 3646;
                        return true;
                }
                break;
            case 3630:
                if ('n' == current) {
                    state_ = 3631;
                    return true;
                }
                break;
            case 3631:
                if ('d' == current) {
                    state_ = 3632;
                    return true;
                }
                break;
            case 3632:
                // capand;
                if (';' == current) {
                    match_ = "\u2A44";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3633:
                if ('r' == current) {
                    state_ = 3634;
                    return true;
                }
                break;
            case 3634:
                if ('c' == current) {
                    state_ = 3635;
                    return true;
                }
                break;
            case 3635:
                if ('u' == current) {
                    state_ = 3636;
                    return true;
                }
                break;
            case 3636:
                if ('p' == current) {
                    state_ = 3637;
                    return true;
                }
                break;
            case 3637:
                // capbrcup;
                if (';' == current) {
                    match_ = "\u2A49";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3638:
                if ('a' == current) {
                    state_ = 3639;
                    return true;
                }
                else if ('u' == current) {
                    state_ = 3641;
                    return true;
                }
                break;
            case 3639:
                if ('p' == current) {
                    state_ = 3640;
                    return true;
                }
                break;
            case 3640:
                // capcap;
                if (';' == current) {
                    match_ = "\u2A4B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3641:
                if ('p' == current) {
                    state_ = 3642;
                    return true;
                }
                break;
            case 3642:
                // capcup;
                if (';' == current) {
                    match_ = "\u2A47";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3643:
                if ('o' == current) {
                    state_ = 3644;
                    return true;
                }
                break;
            case 3644:
                if ('t' == current) {
                    state_ = 3645;
                    return true;
                }
                break;
            case 3645:
                // capdot;
                if (';' == current) {
                    match_ = "\u2A40";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3646:
                // caps;
                if (';' == current) {
                    match_ = "\u2229\uFE00";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3647:
                if ('e' == current) {
                    state_ = 3648;
                    return true;
                }
                else if ('o' == current) {
                    state_ = 3650;
                    return true;
                }
                break;
            case 3648:
                if ('t' == current) {
                    state_ = 3649;
                    return true;
                }
                break;
            case 3649:
                // caret;
                if (';' == current) {
                    match_ = "\u2041";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3650:
                if ('n' == current) {
                    state_ = 3651;
                    return true;
                }
                break;
            case 3651:
                // caron;
                if (';' == current) {
                    match_ = "\u02C7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3652:
                switch (current) {
                    case 'a':
                        state_ = 3653;
                        return true;
                    case 'e':
                        state_ = 3659;
                        return true;
                    case 'i':
                        state_ = 3663;
                        return true;
                    case 'u':
                        state_ = 3666;
                        return true;
                }
                break;
            case 3653:
                if ('p' == current) {
                    state_ = 3654;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 3656;
                    return true;
                }
                break;
            case 3654:
                if ('s' == current) {
                    state_ = 3655;
                    return true;
                }
                break;
            case 3655:
                // ccaps;
                if (';' == current) {
                    match_ = "\u2A4D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3656:
                if ('o' == current) {
                    state_ = 3657;
                    return true;
                }
                break;
            case 3657:
                if ('n' == current) {
                    state_ = 3658;
                    return true;
                }
                break;
            case 3658:
                // ccaron;
                if (';' == current) {
                    match_ = "\u010D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3659:
                if ('d' == current) {
                    state_ = 3660;
                    return true;
                }
                break;
            case 3660:
                if ('i' == current) {
                    state_ = 3661;
                    return true;
                }
                break;
            case 3661:
                // ccedil
                if ('l' == current) {
                    match_ = "\u00E7";
                    matchLength_ = consumedCount_;
                    state_ = 3662;
                    return true;
                }
                break;
            case 3662:
                // ccedil;
                if (';' == current) {
                    match_ = "\u00E7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3663:
                if ('r' == current) {
                    state_ = 3664;
                    return true;
                }
                break;
            case 3664:
                if ('c' == current) {
                    state_ = 3665;
                    return true;
                }
                break;
            case 3665:
                // ccirc;
                if (';' == current) {
                    match_ = "\u0109";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3666:
                if ('p' == current) {
                    state_ = 3667;
                    return true;
                }
                break;
            case 3667:
                if ('s' == current) {
                    state_ = 3668;
                    return true;
                }
                break;
            case 3668:
                // ccups;
                if (';' == current) {
                    match_ = "\u2A4C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('s' == current) {
                    state_ = 3669;
                    return true;
                }
                break;
            case 3669:
                if ('m' == current) {
                    state_ = 3670;
                    return true;
                }
                break;
            case 3670:
                // ccupssm;
                if (';' == current) {
                    match_ = "\u2A50";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3671:
                if ('o' == current) {
                    state_ = 3672;
                    return true;
                }
                break;
            case 3672:
                if ('t' == current) {
                    state_ = 3673;
                    return true;
                }
                break;
            case 3673:
                // cdot;
                if (';' == current) {
                    match_ = "\u010B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3674:
                switch (current) {
                    case 'd':
                        state_ = 3675;
                        return true;
                    case 'm':
                        state_ = 3678;
                        return true;
                    case 'n':
                        state_ = 3683;
                        return true;
                }
                break;
            case 3675:
                if ('i' == current) {
                    state_ = 3676;
                    return true;
                }
                break;
            case 3676:
                // cedil
                if ('l' == current) {
                    match_ = "\u00B8";
                    matchLength_ = consumedCount_;
                    state_ = 3677;
                    return true;
                }
                break;
            case 3677:
                // cedil;
                if (';' == current) {
                    match_ = "\u00B8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3678:
                if ('p' == current) {
                    state_ = 3679;
                    return true;
                }
                break;
            case 3679:
                if ('t' == current) {
                    state_ = 3680;
                    return true;
                }
                break;
            case 3680:
                if ('y' == current) {
                    state_ = 3681;
                    return true;
                }
                break;
            case 3681:
                if ('v' == current) {
                    state_ = 3682;
                    return true;
                }
                break;
            case 3682:
                // cemptyv;
                if (';' == current) {
                    match_ = "\u29B2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3683:
                // cent
                if ('t' == current) {
                    match_ = "\u00A2";
                    matchLength_ = consumedCount_;
                    state_ = 3684;
                    return true;
                }
                break;
            case 3684:
                // cent;
                if (';' == current) {
                    match_ = "\u00A2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state_ = 3685;
                    return true;
                }
                break;
            case 3685:
                if ('r' == current) {
                    state_ = 3686;
                    return true;
                }
                break;
            case 3686:
                if ('d' == current) {
                    state_ = 3687;
                    return true;
                }
                break;
            case 3687:
                if ('o' == current) {
                    state_ = 3688;
                    return true;
                }
                break;
            case 3688:
                if ('t' == current) {
                    state_ = 3689;
                    return true;
                }
                break;
            case 3689:
                // centerdot;
                if (';' == current) {
                    match_ = "\u00B7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3690:
                if ('r' == current) {
                    state_ = 3691;
                    return true;
                }
                break;
            case 3691:
                // cfr;
                if (';' == current) {
                    match_ = "\uD835\uDD20";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3692:
                switch (current) {
                    case 'c':
                        state_ = 3693;
                        return true;
                    case 'e':
                        state_ = 3695;
                        return true;
                    case 'i':
                        state_ = 3702;
                        return true;
                }
                break;
            case 3693:
                if ('y' == current) {
                    state_ = 3694;
                    return true;
                }
                break;
            case 3694:
                // chcy;
                if (';' == current) {
                    match_ = "\u0447";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3695:
                if ('c' == current) {
                    state_ = 3696;
                    return true;
                }
                break;
            case 3696:
                if ('k' == current) {
                    state_ = 3697;
                    return true;
                }
                break;
            case 3697:
                // check;
                if (';' == current) {
                    match_ = "\u2713";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('m' == current) {
                    state_ = 3698;
                    return true;
                }
                break;
            case 3698:
                if ('a' == current) {
                    state_ = 3699;
                    return true;
                }
                break;
            case 3699:
                if ('r' == current) {
                    state_ = 3700;
                    return true;
                }
                break;
            case 3700:
                if ('k' == current) {
                    state_ = 3701;
                    return true;
                }
                break;
            case 3701:
                // checkmark;
                if (';' == current) {
                    match_ = "\u2713";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3702:
                // chi;
                if (';' == current) {
                    match_ = "\u03C7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3703:
                if ('r' == current) {
                    state_ = 3704;
                    return true;
                }
                break;
            case 3704:
                switch (current) {
                    case ';': // cir;
                        match_ = "\u25CB";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'E':
                        state_ = 3705;
                        return true;
                    case 'c':
                        state_ = 3706;
                        return true;
                    case 'e':
                        state_ = 3739;
                        return true;
                    case 'f':
                        state_ = 3740;
                        return true;
                    case 'm':
                        state_ = 3745;
                        return true;
                    case 's':
                        state_ = 3748;
                        return true;
                }
                break;
            case 3705:
                // cirE;
                if (';' == current) {
                    match_ = "\u29C3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3706:
                switch (current) {
                    case ';': // circ;
                        match_ = "\u02C6";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e':
                        state_ = 3707;
                        return true;
                    case 'l':
                        state_ = 3709;
                        return true;
                }
                break;
            case 3707:
                if ('q' == current) {
                    state_ = 3708;
                    return true;
                }
                break;
            case 3708:
                // circeq;
                if (';' == current) {
                    match_ = "\u2257";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3709:
                if ('e' == current) {
                    state_ = 3710;
                    return true;
                }
                break;
            case 3710:
                if ('a' == current) {
                    state_ = 3711;
                    return true;
                }
                else if ('d' == current) {
                    state_ = 3725;
                    return true;
                }
                break;
            case 3711:
                if ('r' == current) {
                    state_ = 3712;
                    return true;
                }
                break;
            case 3712:
                if ('r' == current) {
                    state_ = 3713;
                    return true;
                }
                break;
            case 3713:
                if ('o' == current) {
                    state_ = 3714;
                    return true;
                }
                break;
            case 3714:
                if ('w' == current) {
                    state_ = 3715;
                    return true;
                }
                break;
            case 3715:
                if ('l' == current) {
                    state_ = 3716;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 3720;
                    return true;
                }
                break;
            case 3716:
                if ('e' == current) {
                    state_ = 3717;
                    return true;
                }
                break;
            case 3717:
                if ('f' == current) {
                    state_ = 3718;
                    return true;
                }
                break;
            case 3718:
                if ('t' == current) {
                    state_ = 3719;
                    return true;
                }
                break;
            case 3719:
                // circlearrowleft;
                if (';' == current) {
                    match_ = "\u21BA";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3720:
                if ('i' == current) {
                    state_ = 3721;
                    return true;
                }
                break;
            case 3721:
                if ('g' == current) {
                    state_ = 3722;
                    return true;
                }
                break;
            case 3722:
                if ('h' == current) {
                    state_ = 3723;
                    return true;
                }
                break;
            case 3723:
                if ('t' == current) {
                    state_ = 3724;
                    return true;
                }
                break;
            case 3724:
                // circlearrowright;
                if (';' == current) {
                    match_ = "\u21BB";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3725:
                switch (current) {
                    case 'R':
                        state_ = 3726;
                        return true;
                    case 'S':
                        state_ = 3727;
                        return true;
                    case 'a':
                        state_ = 3728;
                        return true;
                    case 'c':
                        state_ = 3731;
                        return true;
                    case 'd':
                        state_ = 3735;
                        return true;
                }
                break;
            case 3726:
                // circledR;
                if (';' == current) {
                    match_ = "\u00AE";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3727:
                // circledS;
                if (';' == current) {
                    match_ = "\u24C8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3728:
                if ('s' == current) {
                    state_ = 3729;
                    return true;
                }
                break;
            case 3729:
                if ('t' == current) {
                    state_ = 3730;
                    return true;
                }
                break;
            case 3730:
                // circledast;
                if (';' == current) {
                    match_ = "\u229B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3731:
                if ('i' == current) {
                    state_ = 3732;
                    return true;
                }
                break;
            case 3732:
                if ('r' == current) {
                    state_ = 3733;
                    return true;
                }
                break;
            case 3733:
                if ('c' == current) {
                    state_ = 3734;
                    return true;
                }
                break;
            case 3734:
                // circledcirc;
                if (';' == current) {
                    match_ = "\u229A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3735:
                if ('a' == current) {
                    state_ = 3736;
                    return true;
                }
                break;
            case 3736:
                if ('s' == current) {
                    state_ = 3737;
                    return true;
                }
                break;
            case 3737:
                if ('h' == current) {
                    state_ = 3738;
                    return true;
                }
                break;
            case 3738:
                // circleddash;
                if (';' == current) {
                    match_ = "\u229D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3739:
                // cire;
                if (';' == current) {
                    match_ = "\u2257";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3740:
                if ('n' == current) {
                    state_ = 3741;
                    return true;
                }
                break;
            case 3741:
                if ('i' == current) {
                    state_ = 3742;
                    return true;
                }
                break;
            case 3742:
                if ('n' == current) {
                    state_ = 3743;
                    return true;
                }
                break;
            case 3743:
                if ('t' == current) {
                    state_ = 3744;
                    return true;
                }
                break;
            case 3744:
                // cirfnint;
                if (';' == current) {
                    match_ = "\u2A10";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3745:
                if ('i' == current) {
                    state_ = 3746;
                    return true;
                }
                break;
            case 3746:
                if ('d' == current) {
                    state_ = 3747;
                    return true;
                }
                break;
            case 3747:
                // cirmid;
                if (';' == current) {
                    match_ = "\u2AEF";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3748:
                if ('c' == current) {
                    state_ = 3749;
                    return true;
                }
                break;
            case 3749:
                if ('i' == current) {
                    state_ = 3750;
                    return true;
                }
                break;
            case 3750:
                if ('r' == current) {
                    state_ = 3751;
                    return true;
                }
                break;
            case 3751:
                // cirscir;
                if (';' == current) {
                    match_ = "\u29C2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3752:
                if ('u' == current) {
                    state_ = 3753;
                    return true;
                }
                break;
            case 3753:
                if ('b' == current) {
                    state_ = 3754;
                    return true;
                }
                break;
            case 3754:
                if ('s' == current) {
                    state_ = 3755;
                    return true;
                }
                break;
            case 3755:
                // clubs;
                if (';' == current) {
                    match_ = "\u2663";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('u' == current) {
                    state_ = 3756;
                    return true;
                }
                break;
            case 3756:
                if ('i' == current) {
                    state_ = 3757;
                    return true;
                }
                break;
            case 3757:
                if ('t' == current) {
                    state_ = 3758;
                    return true;
                }
                break;
            case 3758:
                // clubsuit;
                if (';' == current) {
                    match_ = "\u2663";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3759:
                switch (current) {
                    case 'l':
                        state_ = 3760;
                        return true;
                    case 'm':
                        state_ = 3765;
                        return true;
                    case 'n':
                        state_ = 3781;
                        return true;
                    case 'p':
                        state_ = 3789;
                        return true;
                }
                break;
            case 3760:
                if ('o' == current) {
                    state_ = 3761;
                    return true;
                }
                break;
            case 3761:
                if ('n' == current) {
                    state_ = 3762;
                    return true;
                }
                break;
            case 3762:
                // colon;
                if (';' == current) {
                    match_ = ":";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state_ = 3763;
                    return true;
                }
                break;
            case 3763:
                // colone;
                if (';' == current) {
                    match_ = "\u2254";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('q' == current) {
                    state_ = 3764;
                    return true;
                }
                break;
            case 3764:
                // coloneq;
                if (';' == current) {
                    match_ = "\u2254";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3765:
                if ('m' == current) {
                    state_ = 3766;
                    return true;
                }
                else if ('p' == current) {
                    state_ = 3769;
                    return true;
                }
                break;
            case 3766:
                if ('a' == current) {
                    state_ = 3767;
                    return true;
                }
                break;
            case 3767:
                // comma;
                if (';' == current) {
                    match_ = ",";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('t' == current) {
                    state_ = 3768;
                    return true;
                }
                break;
            case 3768:
                // commat;
                if (';' == current) {
                    match_ = "@";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3769:
                switch (current) {
                    case ';': // comp;
                        match_ = "\u2201";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'f':
                        state_ = 3770;
                        return true;
                    case 'l':
                        state_ = 3772;
                        return true;
                }
                break;
            case 3770:
                if ('n' == current) {
                    state_ = 3771;
                    return true;
                }
                break;
            case 3771:
                // compfn;
                if (';' == current) {
                    match_ = "\u2218";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3772:
                if ('e' == current) {
                    state_ = 3773;
                    return true;
                }
                break;
            case 3773:
                if ('m' == current) {
                    state_ = 3774;
                    return true;
                }
                else if ('x' == current) {
                    state_ = 3778;
                    return true;
                }
                break;
            case 3774:
                if ('e' == current) {
                    state_ = 3775;
                    return true;
                }
                break;
            case 3775:
                if ('n' == current) {
                    state_ = 3776;
                    return true;
                }
                break;
            case 3776:
                if ('t' == current) {
                    state_ = 3777;
                    return true;
                }
                break;
            case 3777:
                // complement;
                if (';' == current) {
                    match_ = "\u2201";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3778:
                if ('e' == current) {
                    state_ = 3779;
                    return true;
                }
                break;
            case 3779:
                if ('s' == current) {
                    state_ = 3780;
                    return true;
                }
                break;
            case 3780:
                // complexes;
                if (';' == current) {
                    match_ = "\u2102";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3781:
                if ('g' == current) {
                    state_ = 3782;
                    return true;
                }
                else if ('i' == current) {
                    state_ = 3786;
                    return true;
                }
                break;
            case 3782:
                // cong;
                if (';' == current) {
                    match_ = "\u2245";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('d' == current) {
                    state_ = 3783;
                    return true;
                }
                break;
            case 3783:
                if ('o' == current) {
                    state_ = 3784;
                    return true;
                }
                break;
            case 3784:
                if ('t' == current) {
                    state_ = 3785;
                    return true;
                }
                break;
            case 3785:
                // congdot;
                if (';' == current) {
                    match_ = "\u2A6D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3786:
                if ('n' == current) {
                    state_ = 3787;
                    return true;
                }
                break;
            case 3787:
                if ('t' == current) {
                    state_ = 3788;
                    return true;
                }
                break;
            case 3788:
                // conint;
                if (';' == current) {
                    match_ = "\u222E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3789:
                switch (current) {
                    case 'f':
                        state_ = 3790;
                        return true;
                    case 'r':
                        state_ = 3791;
                        return true;
                    case 'y': // copy
                        match_ = "\u00A9";
                        matchLength_ = consumedCount_;
                        state_ = 3794;
                        return true;
                }
                break;
            case 3790:
                // copf;
                if (';' == current) {
                    match_ = "\uD835\uDD54";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3791:
                if ('o' == current) {
                    state_ = 3792;
                    return true;
                }
                break;
            case 3792:
                if ('d' == current) {
                    state_ = 3793;
                    return true;
                }
                break;
            case 3793:
                // coprod;
                if (';' == current) {
                    match_ = "\u2210";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3794:
                // copy;
                if (';' == current) {
                    match_ = "\u00A9";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('s' == current) {
                    state_ = 3795;
                    return true;
                }
                break;
            case 3795:
                if ('r' == current) {
                    state_ = 3796;
                    return true;
                }
                break;
            case 3796:
                // copysr;
                if (';' == current) {
                    match_ = "\u2117";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3797:
                if ('a' == current) {
                    state_ = 3798;
                    return true;
                }
                else if ('o' == current) {
                    state_ = 3801;
                    return true;
                }
                break;
            case 3798:
                if ('r' == current) {
                    state_ = 3799;
                    return true;
                }
                break;
            case 3799:
                if ('r' == current) {
                    state_ = 3800;
                    return true;
                }
                break;
            case 3800:
                // crarr;
                if (';' == current) {
                    match_ = "\u21B5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3801:
                if ('s' == current) {
                    state_ = 3802;
                    return true;
                }
                break;
            case 3802:
                if ('s' == current) {
                    state_ = 3803;
                    return true;
                }
                break;
            case 3803:
                // cross;
                if (';' == current) {
                    match_ = "\u2717";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3804:
                if ('c' == current) {
                    state_ = 3805;
                    return true;
                }
                else if ('u' == current) {
                    state_ = 3807;
                    return true;
                }
                break;
            case 3805:
                if ('r' == current) {
                    state_ = 3806;
                    return true;
                }
                break;
            case 3806:
                // cscr;
                if (';' == current) {
                    match_ = "\uD835\uDCB8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3807:
                if ('b' == current) {
                    state_ = 3808;
                    return true;
                }
                else if ('p' == current) {
                    state_ = 3810;
                    return true;
                }
                break;
            case 3808:
                // csub;
                if (';' == current) {
                    match_ = "\u2ACF";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state_ = 3809;
                    return true;
                }
                break;
            case 3809:
                // csube;
                if (';' == current) {
                    match_ = "\u2AD1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3810:
                // csup;
                if (';' == current) {
                    match_ = "\u2AD0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('e' == current) {
                    state_ = 3811;
                    return true;
                }
                break;
            case 3811:
                // csupe;
                if (';' == current) {
                    match_ = "\u2AD2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3812:
                if ('d' == current) {
                    state_ = 3813;
                    return true;
                }
                break;
            case 3813:
                if ('o' == current) {
                    state_ = 3814;
                    return true;
                }
                break;
            case 3814:
                if ('t' == current) {
                    state_ = 3815;
                    return true;
                }
                break;
            case 3815:
                // ctdot;
                if (';' == current) {
                    match_ = "\u22EF";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3816:
                switch (current) {
                    case 'd':
                        state_ = 3817;
                        return true;
                    case 'e':
                        state_ = 3823;
                        return true;
                    case 'l':
                        state_ = 3828;
                        return true;
                    case 'p':
                        state_ = 3833;
                        return true;
                    case 'r':
                        state_ = 3850;
                        return true;
                    case 'v':
                        state_ = 3894;
                        return true;
                    case 'w':
                        state_ = 3897;
                        return true;
                }
                break;
            case 3817:
                if ('a' == current) {
                    state_ = 3818;
                    return true;
                }
                break;
            case 3818:
                if ('r' == current) {
                    state_ = 3819;
                    return true;
                }
                break;
            case 3819:
                if ('r' == current) {
                    state_ = 3820;
                    return true;
                }
                break;
            case 3820:
                if ('l' == current) {
                    state_ = 3821;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 3822;
                    return true;
                }
                break;
            case 3821:
                // cudarrl;
                if (';' == current) {
                    match_ = "\u2938";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3822:
                // cudarrr;
                if (';' == current) {
                    match_ = "\u2935";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3823:
                if ('p' == current) {
                    state_ = 3824;
                    return true;
                }
                else if ('s' == current) {
                    state_ = 3826;
                    return true;
                }
                break;
            case 3824:
                if ('r' == current) {
                    state_ = 3825;
                    return true;
                }
                break;
            case 3825:
                // cuepr;
                if (';' == current) {
                    match_ = "\u22DE";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3826:
                if ('c' == current) {
                    state_ = 3827;
                    return true;
                }
                break;
            case 3827:
                // cuesc;
                if (';' == current) {
                    match_ = "\u22DF";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3828:
                if ('a' == current) {
                    state_ = 3829;
                    return true;
                }
                break;
            case 3829:
                if ('r' == current) {
                    state_ = 3830;
                    return true;
                }
                break;
            case 3830:
                if ('r' == current) {
                    state_ = 3831;
                    return true;
                }
                break;
            case 3831:
                // cularr;
                if (';' == current) {
                    match_ = "\u21B6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('p' == current) {
                    state_ = 3832;
                    return true;
                }
                break;
            case 3832:
                // cularrp;
                if (';' == current) {
                    match_ = "\u293D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3833:
                switch (current) {
                    case ';': // cup;
                        match_ = "\u222A";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'b':
                        state_ = 3834;
                        return true;
                    case 'c':
                        state_ = 3839;
                        return true;
                    case 'd':
                        state_ = 3844;
                        return true;
                    case 'o':
                        state_ = 3847;
                        return true;
                    case 's':
                        state_ = 3849;
                        return true;
                }
                break;
            case 3834:
                if ('r' == current) {
                    state_ = 3835;
                    return true;
                }
                break;
            case 3835:
                if ('c' == current) {
                    state_ = 3836;
                    return true;
                }
                break;
            case 3836:
                if ('a' == current) {
                    state_ = 3837;
                    return true;
                }
                break;
            case 3837:
                if ('p' == current) {
                    state_ = 3838;
                    return true;
                }
                break;
            case 3838:
                // cupbrcap;
                if (';' == current) {
                    match_ = "\u2A48";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3839:
                if ('a' == current) {
                    state_ = 3840;
                    return true;
                }
                else if ('u' == current) {
                    state_ = 3842;
                    return true;
                }
                break;
            case 3840:
                if ('p' == current) {
                    state_ = 3841;
                    return true;
                }
                break;
            case 3841:
                // cupcap;
                if (';' == current) {
                    match_ = "\u2A46";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3842:
                if ('p' == current) {
                    state_ = 3843;
                    return true;
                }
                break;
            case 3843:
                // cupcup;
                if (';' == current) {
                    match_ = "\u2A4A";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3844:
                if ('o' == current) {
                    state_ = 3845;
                    return true;
                }
                break;
            case 3845:
                if ('t' == current) {
                    state_ = 3846;
                    return true;
                }
                break;
            case 3846:
                // cupdot;
                if (';' == current) {
                    match_ = "\u228D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3847:
                if ('r' == current) {
                    state_ = 3848;
                    return true;
                }
                break;
            case 3848:
                // cupor;
                if (';' == current) {
                    match_ = "\u2A45";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3849:
                // cups;
                if (';' == current) {
                    match_ = "\u222A\uFE00";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3850:
                switch (current) {
                    case 'a':
                        state_ = 3851;
                        return true;
                    case 'l':
                        state_ = 3855;
                        return true;
                    case 'r':
                        state_ = 3875;
                        return true;
                    case 'v':
                        state_ = 3878;
                        return true;
                }
                break;
            case 3851:
                if ('r' == current) {
                    state_ = 3852;
                    return true;
                }
                break;
            case 3852:
                if ('r' == current) {
                    state_ = 3853;
                    return true;
                }
                break;
            case 3853:
                // curarr;
                if (';' == current) {
                    match_ = "\u21B7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('m' == current) {
                    state_ = 3854;
                    return true;
                }
                break;
            case 3854:
                // curarrm;
                if (';' == current) {
                    match_ = "\u293C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3855:
                if ('y' == current) {
                    state_ = 3856;
                    return true;
                }
                break;
            case 3856:
                switch (current) {
                    case 'e':
                        state_ = 3857;
                        return true;
                    case 'v':
                        state_ = 3867;
                        return true;
                    case 'w':
                        state_ = 3870;
                        return true;
                }
                break;
            case 3857:
                if ('q' == current) {
                    state_ = 3858;
                    return true;
                }
                break;
            case 3858:
                if ('p' == current) {
                    state_ = 3859;
                    return true;
                }
                else if ('s' == current) {
                    state_ = 3863;
                    return true;
                }
                break;
            case 3859:
                if ('r' == current) {
                    state_ = 3860;
                    return true;
                }
                break;
            case 3860:
                if ('e' == current) {
                    state_ = 3861;
                    return true;
                }
                break;
            case 3861:
                if ('c' == current) {
                    state_ = 3862;
                    return true;
                }
                break;
            case 3862:
                // curlyeqprec;
                if (';' == current) {
                    match_ = "\u22DE";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3863:
                if ('u' == current) {
                    state_ = 3864;
                    return true;
                }
                break;
            case 3864:
                if ('c' == current) {
                    state_ = 3865;
                    return true;
                }
                break;
            case 3865:
                if ('c' == current) {
                    state_ = 3866;
                    return true;
                }
                break;
            case 3866:
                // curlyeqsucc;
                if (';' == current) {
                    match_ = "\u22DF";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3867:
                if ('e' == current) {
                    state_ = 3868;
                    return true;
                }
                break;
            case 3868:
                if ('e' == current) {
                    state_ = 3869;
                    return true;
                }
                break;
            case 3869:
                // curlyvee;
                if (';' == current) {
                    match_ = "\u22CE";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3870:
                if ('e' == current) {
                    state_ = 3871;
                    return true;
                }
                break;
            case 3871:
                if ('d' == current) {
                    state_ = 3872;
                    return true;
                }
                break;
            case 3872:
                if ('g' == current) {
                    state_ = 3873;
                    return true;
                }
                break;
            case 3873:
                if ('e' == current) {
                    state_ = 3874;
                    return true;
                }
                break;
            case 3874:
                // curlywedge;
                if (';' == current) {
                    match_ = "\u22CF";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3875:
                if ('e' == current) {
                    state_ = 3876;
                    return true;
                }
                break;
            case 3876:
                // curren
                if ('n' == current) {
                    match_ = "\u00A4";
                    matchLength_ = consumedCount_;
                    state_ = 3877;
                    return true;
                }
                break;
            case 3877:
                // curren;
                if (';' == current) {
                    match_ = "\u00A4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3878:
                if ('e' == current) {
                    state_ = 3879;
                    return true;
                }
                break;
            case 3879:
                if ('a' == current) {
                    state_ = 3880;
                    return true;
                }
                break;
            case 3880:
                if ('r' == current) {
                    state_ = 3881;
                    return true;
                }
                break;
            case 3881:
                if ('r' == current) {
                    state_ = 3882;
                    return true;
                }
                break;
            case 3882:
                if ('o' == current) {
                    state_ = 3883;
                    return true;
                }
                break;
            case 3883:
                if ('w' == current) {
                    state_ = 3884;
                    return true;
                }
                break;
            case 3884:
                if ('l' == current) {
                    state_ = 3885;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 3889;
                    return true;
                }
                break;
            case 3885:
                if ('e' == current) {
                    state_ = 3886;
                    return true;
                }
                break;
            case 3886:
                if ('f' == current) {
                    state_ = 3887;
                    return true;
                }
                break;
            case 3887:
                if ('t' == current) {
                    state_ = 3888;
                    return true;
                }
                break;
            case 3888:
                // curvearrowleft;
                if (';' == current) {
                    match_ = "\u21B6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3889:
                if ('i' == current) {
                    state_ = 3890;
                    return true;
                }
                break;
            case 3890:
                if ('g' == current) {
                    state_ = 3891;
                    return true;
                }
                break;
            case 3891:
                if ('h' == current) {
                    state_ = 3892;
                    return true;
                }
                break;
            case 3892:
                if ('t' == current) {
                    state_ = 3893;
                    return true;
                }
                break;
            case 3893:
                // curvearrowright;
                if (';' == current) {
                    match_ = "\u21B7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3894:
                if ('e' == current) {
                    state_ = 3895;
                    return true;
                }
                break;
            case 3895:
                if ('e' == current) {
                    state_ = 3896;
                    return true;
                }
                break;
            case 3896:
                // cuvee;
                if (';' == current) {
                    match_ = "\u22CE";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3897:
                if ('e' == current) {
                    state_ = 3898;
                    return true;
                }
                break;
            case 3898:
                if ('d' == current) {
                    state_ = 3899;
                    return true;
                }
                break;
            case 3899:
                // cuwed;
                if (';' == current) {
                    match_ = "\u22CF";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3900:
                if ('c' == current) {
                    state_ = 3901;
                    return true;
                }
                else if ('i' == current) {
                    state_ = 3907;
                    return true;
                }
                break;
            case 3901:
                if ('o' == current) {
                    state_ = 3902;
                    return true;
                }
                break;
            case 3902:
                if ('n' == current) {
                    state_ = 3903;
                    return true;
                }
                break;
            case 3903:
                if ('i' == current) {
                    state_ = 3904;
                    return true;
                }
                break;
            case 3904:
                if ('n' == current) {
                    state_ = 3905;
                    return true;
                }
                break;
            case 3905:
                if ('t' == current) {
                    state_ = 3906;
                    return true;
                }
                break;
            case 3906:
                // cwconint;
                if (';' == current) {
                    match_ = "\u2232";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3907:
                if ('n' == current) {
                    state_ = 3908;
                    return true;
                }
                break;
            case 3908:
                if ('t' == current) {
                    state_ = 3909;
                    return true;
                }
                break;
            case 3909:
                // cwint;
                if (';' == current) {
                    match_ = "\u2231";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3910:
                if ('l' == current) {
                    state_ = 3911;
                    return true;
                }
                break;
            case 3911:
                if ('c' == current) {
                    state_ = 3912;
                    return true;
                }
                break;
            case 3912:
                if ('t' == current) {
                    state_ = 3913;
                    return true;
                }
                break;
            case 3913:
                if ('y' == current) {
                    state_ = 3914;
                    return true;
                }
                break;
            case 3914:
                // cylcty;
                if (';' == current) {
                    match_ = "\u232D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3915:
                switch (current) {
                    case 'A':
                        state_ = 3916;
                        return true;
                    case 'H':
                        state_ = 3919;
                        return true;
                    case 'a':
                        state_ = 3922;
                        return true;
                    case 'b':
                        state_ = 3936;
                        return true;
                    case 'c':
                        state_ = 3945;
                        return true;
                    case 'd':
                        state_ = 3951;
                        return true;
                    case 'e':
                        state_ = 3964;
                        return true;
                    case 'f':
                        state_ = 3974;
                        return true;
                    case 'h':
                        state_ = 3980;
                        return true;
                    case 'i':
                        state_ = 3985;
                        return true;
                    case 'j':
                        state_ = 4019;
                        return true;
                    case 'l':
                        state_ = 4022;
                        return true;
                    case 'o':
                        state_ = 4030;
                        return true;
                    case 'r':
                        state_ = 4103;
                        return true;
                    case 's':
                        state_ = 4117;
                        return true;
                    case 't':
                        state_ = 4127;
                        return true;
                    case 'u':
                        state_ = 4134;
                        return true;
                    case 'w':
                        state_ = 4141;
                        return true;
                    case 'z':
                        state_ = 4147;
                        return true;
                }
                break;
            case 3916:
                if ('r' == current) {
                    state_ = 3917;
                    return true;
                }
                break;
            case 3917:
                if ('r' == current) {
                    state_ = 3918;
                    return true;
                }
                break;
            case 3918:
                // dArr;
                if (';' == current) {
                    match_ = "\u21D3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3919:
                if ('a' == current) {
                    state_ = 3920;
                    return true;
                }
                break;
            case 3920:
                if ('r' == current) {
                    state_ = 3921;
                    return true;
                }
                break;
            case 3921:
                // dHar;
                if (';' == current) {
                    match_ = "\u2965";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3922:
                switch (current) {
                    case 'g':
                        state_ = 3923;
                        return true;
                    case 'l':
                        state_ = 3927;
                        return true;
                    case 'r':
                        state_ = 3931;
                        return true;
                    case 's':
                        state_ = 3933;
                        return true;
                }
                break;
            case 3923:
                if ('g' == current) {
                    state_ = 3924;
                    return true;
                }
                break;
            case 3924:
                if ('e' == current) {
                    state_ = 3925;
                    return true;
                }
                break;
            case 3925:
                if ('r' == current) {
                    state_ = 3926;
                    return true;
                }
                break;
            case 3926:
                // dagger;
                if (';' == current) {
                    match_ = "\u2020";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3927:
                if ('e' == current) {
                    state_ = 3928;
                    return true;
                }
                break;
            case 3928:
                if ('t' == current) {
                    state_ = 3929;
                    return true;
                }
                break;
            case 3929:
                if ('h' == current) {
                    state_ = 3930;
                    return true;
                }
                break;
            case 3930:
                // daleth;
                if (';' == current) {
                    match_ = "\u2138";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3931:
                if ('r' == current) {
                    state_ = 3932;
                    return true;
                }
                break;
            case 3932:
                // darr;
                if (';' == current) {
                    match_ = "\u2193";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3933:
                if ('h' == current) {
                    state_ = 3934;
                    return true;
                }
                break;
            case 3934:
                // dash;
                if (';' == current) {
                    match_ = "\u2010";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('v' == current) {
                    state_ = 3935;
                    return true;
                }
                break;
            case 3935:
                // dashv;
                if (';' == current) {
                    match_ = "\u22A3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3936:
                if ('k' == current) {
                    state_ = 3937;
                    return true;
                }
                else if ('l' == current) {
                    state_ = 3942;
                    return true;
                }
                break;
            case 3937:
                if ('a' == current) {
                    state_ = 3938;
                    return true;
                }
                break;
            case 3938:
                if ('r' == current) {
                    state_ = 3939;
                    return true;
                }
                break;
            case 3939:
                if ('o' == current) {
                    state_ = 3940;
                    return true;
                }
                break;
            case 3940:
                if ('w' == current) {
                    state_ = 3941;
                    return true;
                }
                break;
            case 3941:
                // dbkarow;
                if (';' == current) {
                    match_ = "\u290F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3942:
                if ('a' == current) {
                    state_ = 3943;
                    return true;
                }
                break;
            case 3943:
                if ('c' == current) {
                    state_ = 3944;
                    return true;
                }
                break;
            case 3944:
                // dblac;
                if (';' == current) {
                    match_ = "\u02DD";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3945:
                if ('a' == current) {
                    state_ = 3946;
                    return true;
                }
                else if ('y' == current) {
                    state_ = 3950;
                    return true;
                }
                break;
            case 3946:
                if ('r' == current) {
                    state_ = 3947;
                    return true;
                }
                break;
            case 3947:
                if ('o' == current) {
                    state_ = 3948;
                    return true;
                }
                break;
            case 3948:
                if ('n' == current) {
                    state_ = 3949;
                    return true;
                }
                break;
            case 3949:
                // dcaron;
                if (';' == current) {
                    match_ = "\u010F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3950:
                // dcy;
                if (';' == current) {
                    match_ = "\u0434";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3951:
                switch (current) {
                    case ';': // dd;
                        match_ = "\u2146";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'a':
                        state_ = 3952;
                        return true;
                    case 'o':
                        state_ = 3959;
                        return true;
                }
                break;
            case 3952:
                if ('g' == current) {
                    state_ = 3953;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 3957;
                    return true;
                }
                break;
            case 3953:
                if ('g' == current) {
                    state_ = 3954;
                    return true;
                }
                break;
            case 3954:
                if ('e' == current) {
                    state_ = 3955;
                    return true;
                }
                break;
            case 3955:
                if ('r' == current) {
                    state_ = 3956;
                    return true;
                }
                break;
            case 3956:
                // ddagger;
                if (';' == current) {
                    match_ = "\u2021";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3957:
                if ('r' == current) {
                    state_ = 3958;
                    return true;
                }
                break;
            case 3958:
                // ddarr;
                if (';' == current) {
                    match_ = "\u21CA";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3959:
                if ('t' == current) {
                    state_ = 3960;
                    return true;
                }
                break;
            case 3960:
                if ('s' == current) {
                    state_ = 3961;
                    return true;
                }
                break;
            case 3961:
                if ('e' == current) {
                    state_ = 3962;
                    return true;
                }
                break;
            case 3962:
                if ('q' == current) {
                    state_ = 3963;
                    return true;
                }
                break;
            case 3963:
                // ddotseq;
                if (';' == current) {
                    match_ = "\u2A77";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3964:
                switch (current) {
                    case 'g': // deg
                        match_ = "\u00B0";
                        matchLength_ = consumedCount_;
                        state_ = 3965;
                        return true;
                    case 'l':
                        state_ = 3966;
                        return true;
                    case 'm':
                        state_ = 3969;
                        return true;
                }
                break;
            case 3965:
                // deg;
                if (';' == current) {
                    match_ = "\u00B0";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3966:
                if ('t' == current) {
                    state_ = 3967;
                    return true;
                }
                break;
            case 3967:
                if ('a' == current) {
                    state_ = 3968;
                    return true;
                }
                break;
            case 3968:
                // delta;
                if (';' == current) {
                    match_ = "\u03B4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3969:
                if ('p' == current) {
                    state_ = 3970;
                    return true;
                }
                break;
            case 3970:
                if ('t' == current) {
                    state_ = 3971;
                    return true;
                }
                break;
            case 3971:
                if ('y' == current) {
                    state_ = 3972;
                    return true;
                }
                break;
            case 3972:
                if ('v' == current) {
                    state_ = 3973;
                    return true;
                }
                break;
            case 3973:
                // demptyv;
                if (';' == current) {
                    match_ = "\u29B1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3974:
                if ('i' == current) {
                    state_ = 3975;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 3979;
                    return true;
                }
                break;
            case 3975:
                if ('s' == current) {
                    state_ = 3976;
                    return true;
                }
                break;
            case 3976:
                if ('h' == current) {
                    state_ = 3977;
                    return true;
                }
                break;
            case 3977:
                if ('t' == current) {
                    state_ = 3978;
                    return true;
                }
                break;
            case 3978:
                // dfisht;
                if (';' == current) {
                    match_ = "\u297F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3979:
                // dfr;
                if (';' == current) {
                    match_ = "\uD835\uDD21";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3980:
                if ('a' == current) {
                    state_ = 3981;
                    return true;
                }
                break;
            case 3981:
                if ('r' == current) {
                    state_ = 3982;
                    return true;
                }
                break;
            case 3982:
                if ('l' == current) {
                    state_ = 3983;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 3984;
                    return true;
                }
                break;
            case 3983:
                // dharl;
                if (';' == current) {
                    match_ = "\u21C3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3984:
                // dharr;
                if (';' == current) {
                    match_ = "\u21C2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3985:
                switch (current) {
                    case 'a':
                        state_ = 3986;
                        return true;
                    case 'e':
                        state_ = 3996;
                        return true;
                    case 'g':
                        state_ = 3997;
                        return true;
                    case 's':
                        state_ = 4002;
                        return true;
                    case 'v':
                        state_ = 4005;
                        return true;
                }
                break;
            case 3986:
                if ('m' == current) {
                    state_ = 3987;
                    return true;
                }
                break;
            case 3987:
                switch (current) {
                    case ';': // diam;
                        match_ = "\u22C4";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'o':
                        state_ = 3988;
                        return true;
                    case 's':
                        state_ = 3995;
                        return true;
                }
                break;
            case 3988:
                if ('n' == current) {
                    state_ = 3989;
                    return true;
                }
                break;
            case 3989:
                if ('d' == current) {
                    state_ = 3990;
                    return true;
                }
                break;
            case 3990:
                // diamond;
                if (';' == current) {
                    match_ = "\u22C4";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('s' == current) {
                    state_ = 3991;
                    return true;
                }
                break;
            case 3991:
                if ('u' == current) {
                    state_ = 3992;
                    return true;
                }
                break;
            case 3992:
                if ('i' == current) {
                    state_ = 3993;
                    return true;
                }
                break;
            case 3993:
                if ('t' == current) {
                    state_ = 3994;
                    return true;
                }
                break;
            case 3994:
                // diamondsuit;
                if (';' == current) {
                    match_ = "\u2666";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3995:
                // diams;
                if (';' == current) {
                    match_ = "\u2666";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3996:
                // die;
                if (';' == current) {
                    match_ = "\u00A8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 3997:
                if ('a' == current) {
                    state_ = 3998;
                    return true;
                }
                break;
            case 3998:
                if ('m' == current) {
                    state_ = 3999;
                    return true;
                }
                break;
            case 3999:
                if ('m' == current) {
                    state_ = 4000;
                    return true;
                }
                break;
        }
        return false;
    }

    private boolean parse5(final int current) {
        consumedCount_++;
        switch (state_) {
            case 4000:
                if ('a' == current) {
                    state_ = 4001;
                    return true;
                }
                break;
            case 4001:
                // digamma;
                if (';' == current) {
                    match_ = "\u03DD";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4002:
                if ('i' == current) {
                    state_ = 4003;
                    return true;
                }
                break;
            case 4003:
                if ('n' == current) {
                    state_ = 4004;
                    return true;
                }
                break;
            case 4004:
                // disin;
                if (';' == current) {
                    match_ = "\u22F2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4005:
                switch (current) {
                    case ';': // div;
                        match_ = "\u00F7";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'i':
                        state_ = 4006;
                        return true;
                    case 'o':
                        state_ = 4016;
                        return true;
                }
                break;
            case 4006:
                if ('d' == current) {
                    state_ = 4007;
                    return true;
                }
                break;
            case 4007:
                // divide
                if ('e' == current) {
                    match_ = "\u00F7";
                    matchLength_ = consumedCount_;
                    state_ = 4008;
                    return true;
                }
                break;
            case 4008:
                // divide;
                if (';' == current) {
                    match_ = "\u00F7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('o' == current) {
                    state_ = 4009;
                    return true;
                }
                break;
            case 4009:
                if ('n' == current) {
                    state_ = 4010;
                    return true;
                }
                break;
            case 4010:
                if ('t' == current) {
                    state_ = 4011;
                    return true;
                }
                break;
            case 4011:
                if ('i' == current) {
                    state_ = 4012;
                    return true;
                }
                break;
            case 4012:
                if ('m' == current) {
                    state_ = 4013;
                    return true;
                }
                break;
            case 4013:
                if ('e' == current) {
                    state_ = 4014;
                    return true;
                }
                break;
            case 4014:
                if ('s' == current) {
                    state_ = 4015;
                    return true;
                }
                break;
            case 4015:
                // divideontimes;
                if (';' == current) {
                    match_ = "\u22C7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4016:
                if ('n' == current) {
                    state_ = 4017;
                    return true;
                }
                break;
            case 4017:
                if ('x' == current) {
                    state_ = 4018;
                    return true;
                }
                break;
            case 4018:
                // divonx;
                if (';' == current) {
                    match_ = "\u22C7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4019:
                if ('c' == current) {
                    state_ = 4020;
                    return true;
                }
                break;
            case 4020:
                if ('y' == current) {
                    state_ = 4021;
                    return true;
                }
                break;
            case 4021:
                // djcy;
                if (';' == current) {
                    match_ = "\u0452";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4022:
                if ('c' == current) {
                    state_ = 4023;
                    return true;
                }
                break;
            case 4023:
                if ('o' == current) {
                    state_ = 4024;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 4027;
                    return true;
                }
                break;
            case 4024:
                if ('r' == current) {
                    state_ = 4025;
                    return true;
                }
                break;
            case 4025:
                if ('n' == current) {
                    state_ = 4026;
                    return true;
                }
                break;
            case 4026:
                // dlcorn;
                if (';' == current) {
                    match_ = "\u231E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4027:
                if ('o' == current) {
                    state_ = 4028;
                    return true;
                }
                break;
            case 4028:
                if ('p' == current) {
                    state_ = 4029;
                    return true;
                }
                break;
            case 4029:
                // dlcrop;
                if (';' == current) {
                    match_ = "\u230D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4030:
                switch (current) {
                    case 'l':
                        state_ = 4031;
                        return true;
                    case 'p':
                        state_ = 4035;
                        return true;
                    case 't':
                        state_ = 4037;
                        return true;
                    case 'u':
                        state_ = 4058;
                        return true;
                    case 'w':
                        state_ = 4070;
                        return true;
                }
                break;
            case 4031:
                if ('l' == current) {
                    state_ = 4032;
                    return true;
                }
                break;
            case 4032:
                if ('a' == current) {
                    state_ = 4033;
                    return true;
                }
                break;
            case 4033:
                if ('r' == current) {
                    state_ = 4034;
                    return true;
                }
                break;
            case 4034:
                // dollar;
                if (';' == current) {
                    match_ = "$";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4035:
                if ('f' == current) {
                    state_ = 4036;
                    return true;
                }
                break;
            case 4036:
                // dopf;
                if (';' == current) {
                    match_ = "\uD835\uDD55";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4037:
                switch (current) {
                    case ';': // dot;
                        match_ = "\u02D9";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'e':
                        state_ = 4038;
                        return true;
                    case 'm':
                        state_ = 4043;
                        return true;
                    case 'p':
                        state_ = 4048;
                        return true;
                    case 's':
                        state_ = 4052;
                        return true;
                }
                break;
            case 4038:
                if ('q' == current) {
                    state_ = 4039;
                    return true;
                }
                break;
            case 4039:
                // doteq;
                if (';' == current) {
                    match_ = "\u2250";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('d' == current) {
                    state_ = 4040;
                    return true;
                }
                break;
            case 4040:
                if ('o' == current) {
                    state_ = 4041;
                    return true;
                }
                break;
            case 4041:
                if ('t' == current) {
                    state_ = 4042;
                    return true;
                }
                break;
            case 4042:
                // doteqdot;
                if (';' == current) {
                    match_ = "\u2251";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4043:
                if ('i' == current) {
                    state_ = 4044;
                    return true;
                }
                break;
            case 4044:
                if ('n' == current) {
                    state_ = 4045;
                    return true;
                }
                break;
            case 4045:
                if ('u' == current) {
                    state_ = 4046;
                    return true;
                }
                break;
            case 4046:
                if ('s' == current) {
                    state_ = 4047;
                    return true;
                }
                break;
            case 4047:
                // dotminus;
                if (';' == current) {
                    match_ = "\u2238";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4048:
                if ('l' == current) {
                    state_ = 4049;
                    return true;
                }
                break;
            case 4049:
                if ('u' == current) {
                    state_ = 4050;
                    return true;
                }
                break;
            case 4050:
                if ('s' == current) {
                    state_ = 4051;
                    return true;
                }
                break;
            case 4051:
                // dotplus;
                if (';' == current) {
                    match_ = "\u2214";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4052:
                if ('q' == current) {
                    state_ = 4053;
                    return true;
                }
                break;
            case 4053:
                if ('u' == current) {
                    state_ = 4054;
                    return true;
                }
                break;
            case 4054:
                if ('a' == current) {
                    state_ = 4055;
                    return true;
                }
                break;
            case 4055:
                if ('r' == current) {
                    state_ = 4056;
                    return true;
                }
                break;
            case 4056:
                if ('e' == current) {
                    state_ = 4057;
                    return true;
                }
                break;
            case 4057:
                // dotsquare;
                if (';' == current) {
                    match_ = "\u22A1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4058:
                if ('b' == current) {
                    state_ = 4059;
                    return true;
                }
                break;
            case 4059:
                if ('l' == current) {
                    state_ = 4060;
                    return true;
                }
                break;
            case 4060:
                if ('e' == current) {
                    state_ = 4061;
                    return true;
                }
                break;
            case 4061:
                if ('b' == current) {
                    state_ = 4062;
                    return true;
                }
                break;
            case 4062:
                if ('a' == current) {
                    state_ = 4063;
                    return true;
                }
                break;
            case 4063:
                if ('r' == current) {
                    state_ = 4064;
                    return true;
                }
                break;
            case 4064:
                if ('w' == current) {
                    state_ = 4065;
                    return true;
                }
                break;
            case 4065:
                if ('e' == current) {
                    state_ = 4066;
                    return true;
                }
                break;
            case 4066:
                if ('d' == current) {
                    state_ = 4067;
                    return true;
                }
                break;
            case 4067:
                if ('g' == current) {
                    state_ = 4068;
                    return true;
                }
                break;
            case 4068:
                if ('e' == current) {
                    state_ = 4069;
                    return true;
                }
                break;
            case 4069:
                // doublebarwedge;
                if (';' == current) {
                    match_ = "\u2306";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4070:
                if ('n' == current) {
                    state_ = 4071;
                    return true;
                }
                break;
            case 4071:
                switch (current) {
                    case 'a':
                        state_ = 4072;
                        return true;
                    case 'd':
                        state_ = 4077;
                        return true;
                    case 'h':
                        state_ = 4087;
                        return true;
                }
                break;
            case 4072:
                if ('r' == current) {
                    state_ = 4073;
                    return true;
                }
                break;
            case 4073:
                if ('r' == current) {
                    state_ = 4074;
                    return true;
                }
                break;
            case 4074:
                if ('o' == current) {
                    state_ = 4075;
                    return true;
                }
                break;
            case 4075:
                if ('w' == current) {
                    state_ = 4076;
                    return true;
                }
                break;
            case 4076:
                // downarrow;
                if (';' == current) {
                    match_ = "\u2193";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4077:
                if ('o' == current) {
                    state_ = 4078;
                    return true;
                }
                break;
            case 4078:
                if ('w' == current) {
                    state_ = 4079;
                    return true;
                }
                break;
            case 4079:
                if ('n' == current) {
                    state_ = 4080;
                    return true;
                }
                break;
            case 4080:
                if ('a' == current) {
                    state_ = 4081;
                    return true;
                }
                break;
            case 4081:
                if ('r' == current) {
                    state_ = 4082;
                    return true;
                }
                break;
            case 4082:
                if ('r' == current) {
                    state_ = 4083;
                    return true;
                }
                break;
            case 4083:
                if ('o' == current) {
                    state_ = 4084;
                    return true;
                }
                break;
            case 4084:
                if ('w' == current) {
                    state_ = 4085;
                    return true;
                }
                break;
            case 4085:
                if ('s' == current) {
                    state_ = 4086;
                    return true;
                }
                break;
            case 4086:
                // downdownarrows;
                if (';' == current) {
                    match_ = "\u21CA";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4087:
                if ('a' == current) {
                    state_ = 4088;
                    return true;
                }
                break;
            case 4088:
                if ('r' == current) {
                    state_ = 4089;
                    return true;
                }
                break;
            case 4089:
                if ('p' == current) {
                    state_ = 4090;
                    return true;
                }
                break;
            case 4090:
                if ('o' == current) {
                    state_ = 4091;
                    return true;
                }
                break;
            case 4091:
                if ('o' == current) {
                    state_ = 4092;
                    return true;
                }
                break;
            case 4092:
                if ('n' == current) {
                    state_ = 4093;
                    return true;
                }
                break;
            case 4093:
                if ('l' == current) {
                    state_ = 4094;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 4098;
                    return true;
                }
                break;
            case 4094:
                if ('e' == current) {
                    state_ = 4095;
                    return true;
                }
                break;
            case 4095:
                if ('f' == current) {
                    state_ = 4096;
                    return true;
                }
                break;
            case 4096:
                if ('t' == current) {
                    state_ = 4097;
                    return true;
                }
                break;
            case 4097:
                // downharpoonleft;
                if (';' == current) {
                    match_ = "\u21C3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4098:
                if ('i' == current) {
                    state_ = 4099;
                    return true;
                }
                break;
            case 4099:
                if ('g' == current) {
                    state_ = 4100;
                    return true;
                }
                break;
            case 4100:
                if ('h' == current) {
                    state_ = 4101;
                    return true;
                }
                break;
            case 4101:
                if ('t' == current) {
                    state_ = 4102;
                    return true;
                }
                break;
            case 4102:
                // downharpoonright;
                if (';' == current) {
                    match_ = "\u21C2";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4103:
                if ('b' == current) {
                    state_ = 4104;
                    return true;
                }
                else if ('c' == current) {
                    state_ = 4110;
                    return true;
                }
                break;
            case 4104:
                if ('k' == current) {
                    state_ = 4105;
                    return true;
                }
                break;
            case 4105:
                if ('a' == current) {
                    state_ = 4106;
                    return true;
                }
                break;
            case 4106:
                if ('r' == current) {
                    state_ = 4107;
                    return true;
                }
                break;
            case 4107:
                if ('o' == current) {
                    state_ = 4108;
                    return true;
                }
                break;
            case 4108:
                if ('w' == current) {
                    state_ = 4109;
                    return true;
                }
                break;
            case 4109:
                // drbkarow;
                if (';' == current) {
                    match_ = "\u2910";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4110:
                if ('o' == current) {
                    state_ = 4111;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 4114;
                    return true;
                }
                break;
            case 4111:
                if ('r' == current) {
                    state_ = 4112;
                    return true;
                }
                break;
            case 4112:
                if ('n' == current) {
                    state_ = 4113;
                    return true;
                }
                break;
            case 4113:
                // drcorn;
                if (';' == current) {
                    match_ = "\u231F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4114:
                if ('o' == current) {
                    state_ = 4115;
                    return true;
                }
                break;
            case 4115:
                if ('p' == current) {
                    state_ = 4116;
                    return true;
                }
                break;
            case 4116:
                // drcrop;
                if (';' == current) {
                    match_ = "\u230C";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4117:
                switch (current) {
                    case 'c':
                        state_ = 4118;
                        return true;
                    case 'o':
                        state_ = 4121;
                        return true;
                    case 't':
                        state_ = 4123;
                        return true;
                }
                break;
            case 4118:
                if ('r' == current) {
                    state_ = 4119;
                    return true;
                }
                else if ('y' == current) {
                    state_ = 4120;
                    return true;
                }
                break;
            case 4119:
                // dscr;
                if (';' == current) {
                    match_ = "\uD835\uDCB9";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4120:
                // dscy;
                if (';' == current) {
                    match_ = "\u0455";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4121:
                if ('l' == current) {
                    state_ = 4122;
                    return true;
                }
                break;
            case 4122:
                // dsol;
                if (';' == current) {
                    match_ = "\u29F6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4123:
                if ('r' == current) {
                    state_ = 4124;
                    return true;
                }
                break;
            case 4124:
                if ('o' == current) {
                    state_ = 4125;
                    return true;
                }
                break;
            case 4125:
                if ('k' == current) {
                    state_ = 4126;
                    return true;
                }
                break;
            case 4126:
                // dstrok;
                if (';' == current) {
                    match_ = "\u0111";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4127:
                if ('d' == current) {
                    state_ = 4128;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 4131;
                    return true;
                }
                break;
            case 4128:
                if ('o' == current) {
                    state_ = 4129;
                    return true;
                }
                break;
            case 4129:
                if ('t' == current) {
                    state_ = 4130;
                    return true;
                }
                break;
            case 4130:
                // dtdot;
                if (';' == current) {
                    match_ = "\u22F1";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4131:
                if ('i' == current) {
                    state_ = 4132;
                    return true;
                }
                break;
            case 4132:
                // dtri;
                if (';' == current) {
                    match_ = "\u25BF";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('f' == current) {
                    state_ = 4133;
                    return true;
                }
                break;
            case 4133:
                // dtrif;
                if (';' == current) {
                    match_ = "\u25BE";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4134:
                if ('a' == current) {
                    state_ = 4135;
                    return true;
                }
                else if ('h' == current) {
                    state_ = 4138;
                    return true;
                }
                break;
            case 4135:
                if ('r' == current) {
                    state_ = 4136;
                    return true;
                }
                break;
            case 4136:
                if ('r' == current) {
                    state_ = 4137;
                    return true;
                }
                break;
            case 4137:
                // duarr;
                if (';' == current) {
                    match_ = "\u21F5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4138:
                if ('a' == current) {
                    state_ = 4139;
                    return true;
                }
                break;
            case 4139:
                if ('r' == current) {
                    state_ = 4140;
                    return true;
                }
                break;
            case 4140:
                // duhar;
                if (';' == current) {
                    match_ = "\u296F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4141:
                if ('a' == current) {
                    state_ = 4142;
                    return true;
                }
                break;
            case 4142:
                if ('n' == current) {
                    state_ = 4143;
                    return true;
                }
                break;
            case 4143:
                if ('g' == current) {
                    state_ = 4144;
                    return true;
                }
                break;
            case 4144:
                if ('l' == current) {
                    state_ = 4145;
                    return true;
                }
                break;
            case 4145:
                if ('e' == current) {
                    state_ = 4146;
                    return true;
                }
                break;
            case 4146:
                // dwangle;
                if (';' == current) {
                    match_ = "\u29A6";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4147:
                if ('c' == current) {
                    state_ = 4148;
                    return true;
                }
                else if ('i' == current) {
                    state_ = 4150;
                    return true;
                }
                break;
            case 4148:
                if ('y' == current) {
                    state_ = 4149;
                    return true;
                }
                break;
            case 4149:
                // dzcy;
                if (';' == current) {
                    match_ = "\u045F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4150:
                if ('g' == current) {
                    state_ = 4151;
                    return true;
                }
                break;
            case 4151:
                if ('r' == current) {
                    state_ = 4152;
                    return true;
                }
                break;
            case 4152:
                if ('a' == current) {
                    state_ = 4153;
                    return true;
                }
                break;
            case 4153:
                if ('r' == current) {
                    state_ = 4154;
                    return true;
                }
                break;
            case 4154:
                if ('r' == current) {
                    state_ = 4155;
                    return true;
                }
                break;
            case 4155:
                // dzigrarr;
                if (';' == current) {
                    match_ = "\u27FF";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4156:
                switch (current) {
                    case 'D':
                        state_ = 4157;
                        return true;
                    case 'a':
                        state_ = 4163;
                        return true;
                    case 'c':
                        state_ = 4172;
                        return true;
                    case 'd':
                        state_ = 4185;
                        return true;
                    case 'e':
                        state_ = 4188;
                        return true;
                    case 'f':
                        state_ = 4189;
                        return true;
                    case 'g':
                        state_ = 4194;
                        return true;
                    case 'l':
                        state_ = 4203;
                        return true;
                    case 'm':
                        state_ = 4215;
                        return true;
                    case 'n':
                        state_ = 4231;
                        return true;
                    case 'o':
                        state_ = 4235;
                        return true;
                    case 'p':
                        state_ = 4241;
                        return true;
                    case 'q':
                        state_ = 4255;
                        return true;
                    case 'r':
                        state_ = 4295;
                        return true;
                    case 's':
                        state_ = 4302;
                        return true;
                    case 't':
                        state_ = 4310;
                        return true;
                    case 'u':
                        state_ = 4313;
                        return true;
                    case 'x':
                        state_ = 4318;
                        return true;
                }
                break;
            case 4157:
                if ('D' == current) {
                    state_ = 4158;
                    return true;
                }
                else if ('o' == current) {
                    state_ = 4161;
                    return true;
                }
                break;
            case 4158:
                if ('o' == current) {
                    state_ = 4159;
                    return true;
                }
                break;
            case 4159:
                if ('t' == current) {
                    state_ = 4160;
                    return true;
                }
                break;
            case 4160:
                // eDDot;
                if (';' == current) {
                    match_ = "\u2A77";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4161:
                if ('t' == current) {
                    state_ = 4162;
                    return true;
                }
                break;
            case 4162:
                // eDot;
                if (';' == current) {
                    match_ = "\u2251";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4163:
                if ('c' == current) {
                    state_ = 4164;
                    return true;
                }
                else if ('s' == current) {
                    state_ = 4168;
                    return true;
                }
                break;
            case 4164:
                if ('u' == current) {
                    state_ = 4165;
                    return true;
                }
                break;
            case 4165:
                if ('t' == current) {
                    state_ = 4166;
                    return true;
                }
                break;
            case 4166:
                // eacute
                if ('e' == current) {
                    match_ = "\u00E9";
                    matchLength_ = consumedCount_;
                    state_ = 4167;
                    return true;
                }
                break;
            case 4167:
                // eacute;
                if (';' == current) {
                    match_ = "\u00E9";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4168:
                if ('t' == current) {
                    state_ = 4169;
                    return true;
                }
                break;
            case 4169:
                if ('e' == current) {
                    state_ = 4170;
                    return true;
                }
                break;
            case 4170:
                if ('r' == current) {
                    state_ = 4171;
                    return true;
                }
                break;
            case 4171:
                // easter;
                if (';' == current) {
                    match_ = "\u2A6E";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4172:
                switch (current) {
                    case 'a':
                        state_ = 4173;
                        return true;
                    case 'i':
                        state_ = 4177;
                        return true;
                    case 'o':
                        state_ = 4180;
                        return true;
                    case 'y':
                        state_ = 4184;
                        return true;
                }
                break;
            case 4173:
                if ('r' == current) {
                    state_ = 4174;
                    return true;
                }
                break;
            case 4174:
                if ('o' == current) {
                    state_ = 4175;
                    return true;
                }
                break;
            case 4175:
                if ('n' == current) {
                    state_ = 4176;
                    return true;
                }
                break;
            case 4176:
                // ecaron;
                if (';' == current) {
                    match_ = "\u011B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4177:
                if ('r' == current) {
                    state_ = 4178;
                    return true;
                }
                break;
            case 4178:
                // ecir;
                if (';' == current) {
                    match_ = "\u2256";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                // ecirc
                else if ('c' == current) {
                    match_ = "\u00EA";
                    matchLength_ = consumedCount_;
                    state_ = 4179;
                    return true;
                }
                break;
            case 4179:
                // ecirc;
                if (';' == current) {
                    match_ = "\u00EA";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4180:
                if ('l' == current) {
                    state_ = 4181;
                    return true;
                }
                break;
            case 4181:
                if ('o' == current) {
                    state_ = 4182;
                    return true;
                }
                break;
            case 4182:
                if ('n' == current) {
                    state_ = 4183;
                    return true;
                }
                break;
            case 4183:
                // ecolon;
                if (';' == current) {
                    match_ = "\u2255";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4184:
                // ecy;
                if (';' == current) {
                    match_ = "\u044D";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4185:
                if ('o' == current) {
                    state_ = 4186;
                    return true;
                }
                break;
            case 4186:
                if ('t' == current) {
                    state_ = 4187;
                    return true;
                }
                break;
            case 4187:
                // edot;
                if (';' == current) {
                    match_ = "\u0117";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4188:
                // ee;
                if (';' == current) {
                    match_ = "\u2147";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4189:
                if ('D' == current) {
                    state_ = 4190;
                    return true;
                }
                else if ('r' == current) {
                    state_ = 4193;
                    return true;
                }
                break;
            case 4190:
                if ('o' == current) {
                    state_ = 4191;
                    return true;
                }
                break;
            case 4191:
                if ('t' == current) {
                    state_ = 4192;
                    return true;
                }
                break;
            case 4192:
                // efDot;
                if (';' == current) {
                    match_ = "\u2252";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4193:
                // efr;
                if (';' == current) {
                    match_ = "\uD835\uDD22";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4194:
                switch (current) {
                    case ';': // eg;
                        match_ = "\u2A9A";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'r':
                        state_ = 4195;
                        return true;
                    case 's':
                        state_ = 4199;
                        return true;
                }
                break;
            case 4195:
                if ('a' == current) {
                    state_ = 4196;
                    return true;
                }
                break;
            case 4196:
                if ('v' == current) {
                    state_ = 4197;
                    return true;
                }
                break;
            case 4197:
                // egrave
                if ('e' == current) {
                    match_ = "\u00E8";
                    matchLength_ = consumedCount_;
                    state_ = 4198;
                    return true;
                }
                break;
            case 4198:
                // egrave;
                if (';' == current) {
                    match_ = "\u00E8";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4199:
                // egs;
                if (';' == current) {
                    match_ = "\u2A96";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('d' == current) {
                    state_ = 4200;
                    return true;
                }
                break;
            case 4200:
                if ('o' == current) {
                    state_ = 4201;
                    return true;
                }
                break;
            case 4201:
                if ('t' == current) {
                    state_ = 4202;
                    return true;
                }
                break;
            case 4202:
                // egsdot;
                if (';' == current) {
                    match_ = "\u2A98";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4203:
                switch (current) {
                    case ';': // el;
                        match_ = "\u2A99";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'i':
                        state_ = 4204;
                        return true;
                    case 'l':
                        state_ = 4210;
                        return true;
                    case 's':
                        state_ = 4211;
                        return true;
                }
                break;
            case 4204:
                if ('n' == current) {
                    state_ = 4205;
                    return true;
                }
                break;
            case 4205:
                if ('t' == current) {
                    state_ = 4206;
                    return true;
                }
                break;
            case 4206:
                if ('e' == current) {
                    state_ = 4207;
                    return true;
                }
                break;
            case 4207:
                if ('r' == current) {
                    state_ = 4208;
                    return true;
                }
                break;
            case 4208:
                if ('s' == current) {
                    state_ = 4209;
                    return true;
                }
                break;
            case 4209:
                // elinters;
                if (';' == current) {
                    match_ = "\u23E7";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4210:
                // ell;
                if (';' == current) {
                    match_ = "\u2113";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4211:
                // els;
                if (';' == current) {
                    match_ = "\u2A95";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('d' == current) {
                    state_ = 4212;
                    return true;
                }
                break;
            case 4212:
                if ('o' == current) {
                    state_ = 4213;
                    return true;
                }
                break;
            case 4213:
                if ('t' == current) {
                    state_ = 4214;
                    return true;
                }
                break;
            case 4214:
                // elsdot;
                if (';' == current) {
                    match_ = "\u2A97";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4215:
                switch (current) {
                    case 'a':
                        state_ = 4216;
                        return true;
                    case 'p':
                        state_ = 4219;
                        return true;
                    case 's':
                        state_ = 4226;
                        return true;
                }
                break;
            case 4216:
                if ('c' == current) {
                    state_ = 4217;
                    return true;
                }
                break;
            case 4217:
                if ('r' == current) {
                    state_ = 4218;
                    return true;
                }
                break;
            case 4218:
                // emacr;
                if (';' == current) {
                    match_ = "\u0113";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4219:
                if ('t' == current) {
                    state_ = 4220;
                    return true;
                }
                break;
            case 4220:
                if ('y' == current) {
                    state_ = 4221;
                    return true;
                }
                break;
            case 4221:
                switch (current) {
                    case ';': // empty;
                        match_ = "\u2205";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 's':
                        state_ = 4222;
                        return true;
                    case 'v':
                        state_ = 4225;
                        return true;
                }
                break;
            case 4222:
                if ('e' == current) {
                    state_ = 4223;
                    return true;
                }
                break;
            case 4223:
                if ('t' == current) {
                    state_ = 4224;
                    return true;
                }
                break;
            case 4224:
                // emptyset;
                if (';' == current) {
                    match_ = "\u2205";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4225:
                // emptyv;
                if (';' == current) {
                    match_ = "\u2205";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4226:
                if ('p' == current) {
                    state_ = 4227;
                    return true;
                }
                break;
            case 4227:
                if ('1' == current) {
                    state_ = 4228;
                    return true;
                }
                // emsp;
                else if (';' == current) {
                    match_ = "\u2003";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4228:
                if ('3' == current) {
                    state_ = 4229;
                    return true;
                }
                else if ('4' == current) {
                    state_ = 4230;
                    return true;
                }
                break;
            case 4229:
                // emsp13;
                if (';' == current) {
                    match_ = "\u2004";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4230:
                // emsp14;
                if (';' == current) {
                    match_ = "\u2005";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4231:
                if ('g' == current) {
                    state_ = 4232;
                    return true;
                }
                else if ('s' == current) {
                    state_ = 4233;
                    return true;
                }
                break;
            case 4232:
                // eng;
                if (';' == current) {
                    match_ = "\u014B";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4233:
                if ('p' == current) {
                    state_ = 4234;
                    return true;
                }
                break;
            case 4234:
                // ensp;
                if (';' == current) {
                    match_ = "\u2002";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4235:
                if ('g' == current) {
                    state_ = 4236;
                    return true;
                }
                else if ('p' == current) {
                    state_ = 4239;
                    return true;
                }
                break;
            case 4236:
                if ('o' == current) {
                    state_ = 4237;
                    return true;
                }
                break;
            case 4237:
                if ('n' == current) {
                    state_ = 4238;
                    return true;
                }
                break;
            case 4238:
                // eogon;
                if (';' == current) {
                    match_ = "\u0119";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4239:
                if ('f' == current) {
                    state_ = 4240;
                    return true;
                }
                break;
            case 4240:
                // eopf;
                if (';' == current) {
                    match_ = "\uD835\uDD56";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4241:
                switch (current) {
                    case 'a':
                        state_ = 4242;
                        return true;
                    case 'l':
                        state_ = 4246;
                        return true;
                    case 's':
                        state_ = 4249;
                        return true;
                }
                break;
            case 4242:
                if ('r' == current) {
                    state_ = 4243;
                    return true;
                }
                break;
            case 4243:
                // epar;
                if (';' == current) {
                    match_ = "\u22D5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('s' == current) {
                    state_ = 4244;
                    return true;
                }
                break;
            case 4244:
                if ('l' == current) {
                    state_ = 4245;
                    return true;
                }
                break;
            case 4245:
                // eparsl;
                if (';' == current) {
                    match_ = "\u29E3";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4246:
                if ('u' == current) {
                    state_ = 4247;
                    return true;
                }
                break;
            case 4247:
                if ('s' == current) {
                    state_ = 4248;
                    return true;
                }
                break;
            case 4248:
                // eplus;
                if (';' == current) {
                    match_ = "\u2A71";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4249:
                if ('i' == current) {
                    state_ = 4250;
                    return true;
                }
                break;
            case 4250:
                switch (current) {
                    case ';': // epsi;
                        match_ = "\u03B5";
                        matchLength_ = consumedCount_;
                        state_ = STATE_ENDS_WITH_SEMICOLON;
                        return false;
                    case 'l':
                        state_ = 4251;
                        return true;
                    case 'v':
                        state_ = 4254;
                        return true;
                }
                break;
            case 4251:
                if ('o' == current) {
                    state_ = 4252;
                    return true;
                }
                break;
            case 4252:
                if ('n' == current) {
                    state_ = 4253;
                    return true;
                }
                break;
            case 4253:
                // epsilon;
                if (';' == current) {
                    match_ = "\u03B5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4254:
                // epsiv;
                if (';' == current) {
                    match_ = "\u03F5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4255:
                switch (current) {
                    case 'c':
                        state_ = 4256;
                        return true;
                    case 's':
                        state_ = 4264;
                        return true;
                    case 'u':
                        state_ = 4278;
                        return true;
                    case 'v':
                        state_ = 4289;
                        return true;
                }
                break;
            case 4256:
                if ('i' == current) {
                    state_ = 4257;
                    return true;
                }
                else if ('o' == current) {
                    state_ = 4260;
                    return true;
                }
                break;
            case 4257:
                if ('r' == current) {
                    state_ = 4258;
                    return true;
                }
                break;
            case 4258:
                if ('c' == current) {
                    state_ = 4259;
                    return true;
                }
                break;
            case 4259:
                // eqcirc;
                if (';' == current) {
                    match_ = "\u2256";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4260:
                if ('l' == current) {
                    state_ = 4261;
                    return true;
                }
                break;
            case 4261:
                if ('o' == current) {
                    state_ = 4262;
                    return true;
                }
                break;
            case 4262:
                if ('n' == current) {
                    state_ = 4263;
                    return true;
                }
                break;
            case 4263:
                // eqcolon;
                if (';' == current) {
                    match_ = "\u2255";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4264:
                if ('i' == current) {
                    state_ = 4265;
                    return true;
                }
                else if ('l' == current) {
                    state_ = 4267;
                    return true;
                }
                break;
            case 4265:
                if ('m' == current) {
                    state_ = 4266;
                    return true;
                }
                break;
            case 4266:
                // eqsim;
                if (';' == current) {
                    match_ = "\u2242";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4267:
                if ('a' == current) {
                    state_ = 4268;
                    return true;
                }
                break;
            case 4268:
                if ('n' == current) {
                    state_ = 4269;
                    return true;
                }
                break;
            case 4269:
                if ('t' == current) {
                    state_ = 4270;
                    return true;
                }
                break;
            case 4270:
                if ('g' == current) {
                    state_ = 4271;
                    return true;
                }
                else if ('l' == current) {
                    state_ = 4274;
                    return true;
                }
                break;
            case 4271:
                if ('t' == current) {
                    state_ = 4272;
                    return true;
                }
                break;
            case 4272:
                if ('r' == current) {
                    state_ = 4273;
                    return true;
                }
                break;
            case 4273:
                // eqslantgtr;
                if (';' == current) {
                    match_ = "\u2A96";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4274:
                if ('e' == current) {
                    state_ = 4275;
                    return true;
                }
                break;
            case 4275:
                if ('s' == current) {
                    state_ = 4276;
                    return true;
                }
                break;
            case 4276:
                if ('s' == current) {
                    state_ = 4277;
                    return true;
                }
                break;
            case 4277:
                // eqslantless;
                if (';' == current) {
                    match_ = "\u2A95";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4278:
                switch (current) {
                    case 'a':
                        state_ = 4279;
                        return true;
                    case 'e':
                        state_ = 4282;
                        return true;
                    case 'i':
                        state_ = 4285;
                        return true;
                }
                break;
            case 4279:
                if ('l' == current) {
                    state_ = 4280;
                    return true;
                }
                break;
            case 4280:
                if ('s' == current) {
                    state_ = 4281;
                    return true;
                }
                break;
            case 4281:
                // equals;
                if (';' == current) {
                    match_ = "=";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4282:
                if ('s' == current) {
                    state_ = 4283;
                    return true;
                }
                break;
            case 4283:
                if ('t' == current) {
                    state_ = 4284;
                    return true;
                }
                break;
            case 4284:
                // equest;
                if (';' == current) {
                    match_ = "\u225F";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4285:
                if ('v' == current) {
                    state_ = 4286;
                    return true;
                }
                break;
            case 4286:
                // equiv;
                if (';' == current) {
                    match_ = "\u2261";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                else if ('D' == current) {
                    state_ = 4287;
                    return true;
                }
                break;
            case 4287:
                if ('D' == current) {
                    state_ = 4288;
                    return true;
                }
                break;
            case 4288:
                // equivDD;
                if (';' == current) {
                    match_ = "\u2A78";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4289:
                if ('p' == current) {
                    state_ = 4290;
                    return true;
                }
                break;
            case 4290:
                if ('a' == current) {
                    state_ = 4291;
                    return true;
                }
                break;
            case 4291:
                if ('r' == current) {
                    state_ = 4292;
                    return true;
                }
                break;
            case 4292:
                if ('s' == current) {
                    state_ = 4293;
                    return true;
                }
                break;
            case 4293:
                if ('l' == current) {
                    state_ = 4294;
                    return true;
                }
                break;
            case 4294:
                // eqvparsl;
                if (';' == current) {
                    match_ = "\u29E5";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4295:
                if ('D' == current) {
                    state_ = 4296;
                    return true;
                }
                else if ('a' == current) {
                    state_ = 4299;
                    return true;
                }
                break;
            case 4296:
                if ('o' == current) {
                    state_ = 4297;
                    return true;
                }
                break;
            case 4297:
                if ('t' == current) {
                    state_ = 4298;
                    return true;
                }
                break;
            case 4298:
                // erDot;
                if (';' == current) {
                    match_ = "\u2253";
                    matchLength_ = consumedCount_;
                    state_ = STATE_ENDS_WITH_SEMICOLON;
                    return false;
                }
                break;
            case 4299:
                if ('r' == current) {
                    state_ = 4300;
                    return true;
                }
                break;
            