/*
 * SPDX-License-Identifier: Apache-2.0
 * Copyright Red Hat Inc. and Hibernate Authors
 */
package org.hibernate.validator.internal.engine.valuecontext;

import java.lang.reflect.TypeVariable;

import org.hibernate.validator.internal.engine.groups.Group;
import org.hibernate.validator.internal.engine.path.MutablePath;
import org.hibernate.validator.internal.engine.valueextraction.AnnotatedObject;
import org.hibernate.validator.internal.engine.valueextraction.ArrayElement;
import org.hibernate.validator.internal.metadata.core.MetaConstraint;
import org.hibernate.validator.internal.metadata.facets.Cascadable;
import org.hibernate.validator.internal.metadata.facets.Validatable;
import org.hibernate.validator.internal.metadata.location.ConstraintLocation;
import org.hibernate.validator.internal.metadata.location.ConstraintLocation.ConstraintLocationKind;
import org.hibernate.validator.internal.util.ExecutableParameterNameProvider;
import org.hibernate.validator.internal.util.TypeVariables;

/**
 * An instance of this class is used to collect all the relevant information for validating a single class, property or
 * method invocation.
 *
 * @author Hardy Ferentschik
 * @author Gunnar Morling
 * @author Guillaume Smet
 */
public abstract sealed class ValueContext<T, V> permits BeanValueContext, ExecutableValueContext {

	protected final ValueContext<?, ?> parentContext;

	private final ExecutableParameterNameProvider parameterNameProvider;

	/**
	 * The current bean which gets validated. This is the bean hosting the constraints which get validated.
	 */
	protected Object currentBean;

	/**
	 * The current property path we are validating.
	 */
	protected MutablePath propertyPath;

	/**
	 * The current group we are validating.
	 */
	protected Class<?> currentGroup;

	/**
	 * The previous group we were validating.
	 */
	protected Class<?> previousGroup;

	/**
	 * The value which gets currently evaluated.
	 */
	private V currentValue;

	protected Validatable currentValidatable;

	/**
	 * The {@code ConstraintLocationKind} the constraint was defined on
	 */
	private ConstraintLocationKind constraintLocationKind;

	ValueContext(ValueContext<?, ?> parentContext, ExecutableParameterNameProvider parameterNameProvider, T currentBean, Validatable validatable, MutablePath propertyPath) {
		this.parentContext = parentContext;
		this.parameterNameProvider = parameterNameProvider;
		this.currentBean = currentBean;
		this.currentValidatable = validatable;
		this.propertyPath = propertyPath;
	}

	public final MutablePath getPropertyPath() {
		return propertyPath;
	}

	public final Class<?> getCurrentGroup() {
		return currentGroup;
	}

	@SuppressWarnings("unchecked")
	public final T getCurrentBean() {
		return (T) currentBean;
	}

	public Validatable getCurrentValidatable() {
		return currentValidatable;
	}

	/**
	 * Returns the current value to be validated.
	 */
	public final Object getCurrentValidatedValue() {
		return currentValue;
	}

	public final void appendNode(Cascadable node) {
		MutablePath newPath = MutablePath.createCopy( propertyPath );
		node.appendTo( newPath );
		propertyPath = newPath;
	}

	public final void appendNode(ConstraintLocation location) {
		MutablePath newPath = MutablePath.createCopy( propertyPath );
		location.appendTo( parameterNameProvider, newPath );
		propertyPath = newPath;
	}

	public final void appendTypeParameterNode(String nodeName) {
		if ( propertyPath.needToAddContainerElementNode( nodeName ) ) {
			MutablePath newPath = MutablePath.createCopy( propertyPath );
			newPath.addContainerElementNode( nodeName );
			propertyPath = newPath;
		}
	}

	public void appendEmptyNode() {
		MutablePath newPath = MutablePath.createCopy( propertyPath );
		newPath.addEmptyNode();
		propertyPath = newPath;
	}

	public final void updateNode(ConstraintLocation location) {
		location.applyTo( parameterNameProvider, propertyPath );
	}

	public final void markCurrentPropertyAsIterable() {
		propertyPath.makeLeafNodeIterable();
	}

	public final void markCurrentPropertyAsIterableAndSetKey(Object key) {
		propertyPath.makeLeafNodeIterableAndSetMapKey( key );
	}

	public final void markCurrentPropertyAsIterableAndSetIndex(Integer index) {
		propertyPath.makeLeafNodeIterableAndSetIndex( index );
	}

	/**
	 * Sets the container element information.
	 *
	 * @param containerClass the class of the container
	 * @param typeParameterIndex the index of the actual type parameter
	 *
	 * @see TypeVariables#getContainerClass(TypeVariable)
	 * @see TypeVariables#getActualTypeParameter(TypeVariable)
	 * @see AnnotatedObject
	 * @see ArrayElement
	 */
	public final void setTypeParameter(Class<?> containerClass, Integer typeParameterIndex) {
		if ( containerClass == null ) {
			return;
		}

		propertyPath.setLeafNodeTypeParameter( containerClass, typeParameterIndex );
	}

	public final void setCurrentGroup(Class<?> currentGroup) {
		if ( this.previousGroup != this.currentGroup ) {
			this.previousGroup = this.currentGroup;
		}
		this.currentGroup = currentGroup;
	}

	public abstract void markCurrentGroupAsProcessed();

	protected abstract boolean isProcessedForGroup(Class<?> group);

	public abstract void markConstraintProcessed(MetaConstraint<?> metaConstraint);

	public abstract boolean hasMetaConstraintBeenProcessed(MetaConstraint<?> metaConstraint);

	public final void setCurrentValidatedValue(V currentValue) {
		propertyPath.setLeafNodeValueIfRequired( currentValue );
		this.currentValue = currentValue;
	}

	public final boolean validatingDefault() {
		return getCurrentGroup() != null && Group.isDefaultGroup( getCurrentGroup() );
	}

	public final ConstraintLocationKind getConstraintLocationKind() {
		return constraintLocationKind;
	}

	public final void setConstraintLocationKind(ConstraintLocationKind constraintLocationKind) {
		this.constraintLocationKind = constraintLocationKind;
	}

	public final ValueState<V> getCurrentValueState() {
		return new ValueState<V>( propertyPath, currentValue );
	}

	public final void resetValueState(ValueState<V> valueState) {
		this.propertyPath = valueState.getPropertyPath();
		this.currentValue = valueState.getCurrentValue();
	}

	@Override
	public String toString() {
		final StringBuilder sb = new StringBuilder();
		sb.append( "ValueContext" );
		sb.append( "{currentBean=" ).append( currentBean );
		sb.append( ", propertyPath=" ).append( propertyPath );
		sb.append( ", currentGroup=" ).append( currentGroup );
		sb.append( ", currentValue=" ).append( currentValue );
		sb.append( ", constraintLocationKind=" ).append( constraintLocationKind );
		sb.append( '}' );
		return sb.toString();
	}

	public Object getValue(Object parent, ConstraintLocation location) {
		// TODO: For BVAL-214 we'd get the value from a map or another alternative structure instead
		return location.getValue( parent );
	}

	public abstract boolean isBeanAlreadyValidated(Object value, Class<?> group);

	public static class ValueState<V> {

		private final MutablePath propertyPath;

		private final V currentValue;

		ValueState(MutablePath propertyPath, V currentValue) {
			this.propertyPath = propertyPath;
			this.currentValue = currentValue;
		}

		public MutablePath getPropertyPath() {
			return propertyPath;
		}

		public V getCurrentValue() {
			return currentValue;
		}
	}
}
