/*
 * This file is part of the GeoLatte project.
 *
 *     GeoLatte is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     GeoLatte is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU Lesser General Public License for more details.
 *
 *     You should have received a copy of the GNU Lesser General Public License
 *     along with GeoLatte.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (C) 2010 - 2011 and Ownership of code is shared by:
 * Qmino bvba - Romeinsestraat 18 - 3001 Heverlee  (http://www.qmino.com)
 * Geovise bvba - Generaal Eisenhowerlei 9 - 2140 Antwerpen (http://www.geovise.com)
 */

package org.geolatte.geom.crs;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * A parameter in the <code>Projection</code> of a <code>ProjectedCoordinateReferenceSystem</code>.
 *
 * @author Karel Maesen, Geovise BVBA
 *         creation-date: 8/2/11
 */
public class CrsParameter {

    private final String name;
    private final double value;

    public static Map<String, CrsParameter> toMap(List<CrsParameter> params) {
        Map<String, CrsParameter> map = new HashMap<>();
        for ( CrsParameter p : params) {
            map.put(p.getName(), p);
        }
        return map;
    }

    /**
     * Constructs an instance with the given parameter name and value.
     * @param name parameter name
     * @param value parameter value
     */
    public CrsParameter(String name, double value) {
        this.name = name;
        this.value = value;
    }

    /**
     * Returns the name of this parameter
     *
     * @return
     */
    public String getName() {
        return name;
    }

    /**
     * Returns value of this parameter.
     * @return
     */
    public double getValue() {
        return value;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        CrsParameter that = (CrsParameter) o;

        if (Double.compare(that.value, value) != 0) return false;
        if (name != null ? !name.equals(that.name) : that.name != null) return false;

        return true;
    }

    @Override
    public int hashCode() {
        int result;
        long temp;
        result = name != null ? name.hashCode() : 0;
        temp = value != +0.0d ? Double.doubleToLongBits(value) : 0L;
        result = 31 * result + (int) (temp ^ (temp >>> 32));
        return result;
    }

    @Override
    public String toString() {
        return "CrsParameter{" +
                "name='" + name + '\'' +
                ", value=" + value +
                '}';
    }
}
