/*
 *  ******************************************************************************
 *  *
 *  *
 *  * This program and the accompanying materials are made available under the
 *  * terms of the Apache License, Version 2.0 which is available at
 *  * https://www.apache.org/licenses/LICENSE-2.0.
 *  *
 *  *  See the NOTICE file distributed with this work for additional
 *  *  information regarding copyright ownership.
 *  * Unless required by applicable law or agreed to in writing, software
 *  * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 *  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 *  * License for the specific language governing permissions and limitations
 *  * under the License.
 *  *
 *  * SPDX-License-Identifier: Apache-2.0
 *  *****************************************************************************
 */

package org.deeplearning4j.nn.conf.graph;

import lombok.Data;
import lombok.val;
import org.deeplearning4j.nn.conf.inputs.InputType;
import org.deeplearning4j.nn.conf.inputs.InvalidInputTypeException;
import org.deeplearning4j.nn.conf.layers.InputTypeUtil;
import org.deeplearning4j.nn.conf.memory.LayerMemoryReport;
import org.deeplearning4j.nn.conf.memory.MemoryReport;
import org.deeplearning4j.nn.graph.ComputationGraph;
import org.nd4j.linalg.api.buffer.DataType;
import org.nd4j.linalg.api.ndarray.INDArray;
import org.nd4j.shade.jackson.annotation.JsonProperty;

@Data
public class ElementWiseVertex extends GraphVertex {

    /**
     * @param op The operation to perform on the inputs
     */
    public ElementWiseVertex(@JsonProperty("op") Op op) {
        this.op = op;
    }

    public enum Op {
        Add, Subtract, Product, Average, Max
    }

    protected Op op;

    @Override
    public ElementWiseVertex clone() {
        return new ElementWiseVertex(op);
    }

    @Override
    public boolean equals(Object o) {
        if (!(o instanceof ElementWiseVertex))
            return false;
        return ((ElementWiseVertex) o).op == op;
    }

    @Override
    public int hashCode() {
        return op.hashCode();
    }

    @Override
    public long numParams(boolean backprop) {
        return 0;
    }

    @Override
    public int minVertexInputs() {
        return 2;
    }

    @Override
    public int maxVertexInputs() {
        switch (op) {
            case Add:
            case Average:
            case Product:
            case Max:
                //No upper bound
                return Integer.MAX_VALUE;
            case Subtract:
                return 2;
            default:
                throw new UnsupportedOperationException("Unknown op: " + op);
        }
    }

    @Override
    public org.deeplearning4j.nn.graph.vertex.GraphVertex instantiate(ComputationGraph graph, String name, int idx,
                                                                      INDArray paramsView, boolean initializeParams, DataType networkDatatype) {
        org.deeplearning4j.nn.graph.vertex.impl.ElementWiseVertex.Op op;
        switch (this.op) {
            case Add:
                op = org.deeplearning4j.nn.graph.vertex.impl.ElementWiseVertex.Op.Add;
                break;
            case Average:
                op = org.deeplearning4j.nn.graph.vertex.impl.ElementWiseVertex.Op.Average;
                break;
            case Subtract:
                op = org.deeplearning4j.nn.graph.vertex.impl.ElementWiseVertex.Op.Subtract;
                break;
            case Product:
                op = org.deeplearning4j.nn.graph.vertex.impl.ElementWiseVertex.Op.Product;
                break;
            case Max:
                op = org.deeplearning4j.nn.graph.vertex.impl.ElementWiseVertex.Op.Max;
                break;
            default:
                throw new RuntimeException();
        }
        return new org.deeplearning4j.nn.graph.vertex.impl.ElementWiseVertex(graph, name, idx, op, networkDatatype);
    }

    @Override
    public InputType getOutputType(int layerIndex, InputType... vertexInputs) throws InvalidInputTypeException {
        if (vertexInputs.length == 1)
            return vertexInputs[0];
        InputTypeUtil.convertMultipleTypes(vertexInputs);

        InputType first = vertexInputs[0];
        if (first.getType() != InputType.Type.CNN) {
            //FF, RNN or flat CNN data inputs
            for (int i = 1; i < vertexInputs.length; i++) {
                if (vertexInputs[i].getType() != first.getType()) {
                    throw new InvalidInputTypeException(
                            "Invalid input: ElementWise vertex cannot process activations of different types:"
                                    + " first type = " + first.getType() + ", input type " + (i + 1)
                                    + " = " + vertexInputs[i].getType());
                }
            }
        } else {
            //CNN inputs... also check that the channels, width and heights match:
            InputType.InputTypeConvolutional firstConv = (InputType.InputTypeConvolutional) first;

            val fd = firstConv.getChannels();
            val fw = firstConv.getWidth();
            val fh = firstConv.getHeight();

            for (int i = 1; i < vertexInputs.length; i++) {
                if (vertexInputs[i].getType() != InputType.Type.CNN) {
                    throw new InvalidInputTypeException(
                            "Invalid input: ElementWise vertex cannot process activations of different types:"
                                    + " first type = " + InputType.Type.CNN + ", input type " + (i + 1)
                                    + " = " + vertexInputs[i].getType());
                }

                InputType.InputTypeConvolutional otherConv = (InputType.InputTypeConvolutional) vertexInputs[i];

                val od = otherConv.getChannels();
                val ow = otherConv.getWidth();
                val oh = otherConv.getHeight();

                if (fd != od || fw != ow || fh != oh) {
                    throw new InvalidInputTypeException(
                            "Invalid input: ElementWise vertex cannot process CNN activations of different sizes:"
                                    + "first [channels,width,height] = [" + fd + "," + fw + "," + fh
                                    + "], input " + i + " = [" + od + "," + ow + "," + oh + "]");
                }
            }
        }

        if(vertexInputs.length < 2)
            return vertexInputs[0];

        if(first.getType() == InputType.Type.FF) {
            //could be 1s and a higher value. broadcast to the higher value where possible
            InputType.InputTypeFeedForward maxInputType = null;
            for(int i = 0 ; i < vertexInputs.length; i++) {
                InputType.InputTypeFeedForward feedForward = (InputType.InputTypeFeedForward) vertexInputs[i];
                if(maxInputType == null)
                    maxInputType = feedForward;
                else {
                    if(maxInputType.getSize() < feedForward.getSize()) {
                        maxInputType = feedForward;
                    }
                }
            }

            return maxInputType;
        } else if(first.getType() == InputType.Type.CNNFlat) {
            //could be 1s and a higher value. broadcast to the higher value where possible
            InputType.InputTypeConvolutionalFlat maxInputType = null;
            for(int i = 0 ; i < vertexInputs.length; i++) {
                InputType.InputTypeConvolutionalFlat feedForward = (InputType.InputTypeConvolutionalFlat) vertexInputs[i];
                if(maxInputType == null)
                    maxInputType = feedForward;
                else {
                    if(maxInputType.getFlattenedSize() < feedForward.getFlattenedSize()) {
                        maxInputType = feedForward;
                    }
                }
            }

            return maxInputType;
        } else if(first.getType() == InputType.Type.RNN) {
            //could be 1s and a higher value. broadcast to the higher value where possible
            InputType.InputTypeRecurrent maxInputType = null;
            for(int i = 0 ; i < vertexInputs.length; i++) {
                InputType.InputTypeRecurrent feedForward = (InputType.InputTypeRecurrent) vertexInputs[i];
                if(maxInputType == null)
                    maxInputType = feedForward;
                else {
                    if(maxInputType.getTimeSeriesLength() < feedForward.getTimeSeriesLength()) {
                        maxInputType = feedForward;
                    }
                }
            }

            return maxInputType;
        }


        return first; //Same output shape/size as
    }

    @Override
    public MemoryReport getMemoryReport(InputType... inputTypes) {
        //No working memory in addition to output activations
        return new LayerMemoryReport.Builder(null, ElementWiseVertex.class, inputTypes[0], inputTypes[0])
                .standardMemory(0, 0).workingMemory(0, 0, 0, 0).cacheMemory(0, 0).build();
    }
}
