/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.monitoring;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.cassandra.concurrent.ScheduledExecutors;
import org.apache.cassandra.db.monitoring.ApproximateTime;
import org.apache.cassandra.db.monitoring.Monitorable;
import org.cassandraunit.shaded.com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MonitoringTask {
    private static final String LINE_SEPARATOR = System.getProperty("line.separator");
    private static final Logger logger = LoggerFactory.getLogger(MonitoringTask.class);
    private static final int REPORT_INTERVAL_MS = Math.max(0, Integer.valueOf(System.getProperty("cassandra.monitoring_report_interval_ms", "5000")));
    private static final int MAX_OPERATIONS = Integer.valueOf(System.getProperty("cassandra.monitoring_max_operations", "50"));
    @VisibleForTesting
    static MonitoringTask instance = MonitoringTask.make(REPORT_INTERVAL_MS, MAX_OPERATIONS);
    private final int maxOperations;
    private final ScheduledFuture<?> reportingTask;
    private final BlockingQueue<FailedOperation> operationsQueue;
    private final AtomicLong numDroppedOperations;
    private long lastLogTime;

    @VisibleForTesting
    static MonitoringTask make(int reportIntervalMillis, int maxTimedoutOperations) {
        if (instance != null) {
            instance.cancel();
            instance = null;
        }
        return new MonitoringTask(reportIntervalMillis, maxTimedoutOperations);
    }

    private MonitoringTask(int reportIntervalMillis, int maxOperations) {
        this.maxOperations = maxOperations;
        this.operationsQueue = maxOperations > 0 ? new ArrayBlockingQueue(maxOperations) : new LinkedBlockingQueue();
        this.numDroppedOperations = new AtomicLong();
        this.lastLogTime = ApproximateTime.currentTimeMillis();
        logger.info("Scheduling monitoring task with report interval of {} ms, max operations {}", (Object)reportIntervalMillis, (Object)maxOperations);
        this.reportingTask = ScheduledExecutors.scheduledTasks.scheduleWithFixedDelay(() -> this.logFailedOperations(ApproximateTime.currentTimeMillis()), reportIntervalMillis, reportIntervalMillis, TimeUnit.MILLISECONDS);
    }

    public void cancel() {
        this.reportingTask.cancel(false);
    }

    public static void addFailedOperation(Monitorable operation, long now) {
        instance.innerAddFailedOperation(operation, now);
    }

    private void innerAddFailedOperation(Monitorable operation, long now) {
        if (this.maxOperations == 0) {
            return;
        }
        if (!this.operationsQueue.offer(new FailedOperation(operation, now))) {
            this.numDroppedOperations.incrementAndGet();
        }
    }

    @VisibleForTesting
    FailedOperations aggregateFailedOperations() {
        FailedOperation failedOperation;
        HashMap<String, FailedOperation> operations = new HashMap<String, FailedOperation>();
        while ((failedOperation = (FailedOperation)this.operationsQueue.poll()) != null) {
            FailedOperation existing = (FailedOperation)operations.get(failedOperation.name());
            if (existing != null) {
                existing.addTimeout(failedOperation);
                continue;
            }
            operations.put(failedOperation.name(), failedOperation);
        }
        return new FailedOperations(operations, this.numDroppedOperations.getAndSet(0L));
    }

    @VisibleForTesting
    List<String> getFailedOperations() {
        FailedOperations failedOperations = this.aggregateFailedOperations();
        String ret = failedOperations.getLogMessage();
        this.lastLogTime = ApproximateTime.currentTimeMillis();
        return ret.isEmpty() ? Collections.emptyList() : Arrays.asList(ret.split("\n"));
    }

    @VisibleForTesting
    void logFailedOperations(long now) {
        FailedOperations failedOperations = this.aggregateFailedOperations();
        if (!failedOperations.isEmpty()) {
            long elapsed = now - this.lastLogTime;
            logger.warn("{} operations timed out in the last {} msecs, operation list available at debug log level", (Object)failedOperations.num(), (Object)elapsed);
            if (logger.isDebugEnabled()) {
                logger.debug("{} operations timed out in the last {} msecs:{}{}", new Object[]{failedOperations.num(), elapsed, LINE_SEPARATOR, failedOperations.getLogMessage()});
            }
        }
        this.lastLogTime = now;
    }

    private static final class FailedOperation {
        public final Monitorable operation;
        public int numTimeouts;
        public long totalTime;
        public long maxTime;
        public long minTime;
        private String name;

        FailedOperation(Monitorable operation, long failedAt) {
            this.operation = operation;
            this.numTimeouts = 1;
            this.minTime = this.totalTime = failedAt - operation.constructionTime().timestamp;
            this.maxTime = this.totalTime;
        }

        public String name() {
            if (this.name == null) {
                this.name = this.operation.name();
            }
            return this.name;
        }

        void addTimeout(FailedOperation operation) {
            ++this.numTimeouts;
            this.totalTime += operation.totalTime;
            this.maxTime = Math.max(this.maxTime, operation.maxTime);
            this.minTime = Math.min(this.minTime, operation.minTime);
        }

        public String getLogMessage() {
            if (this.numTimeouts == 1) {
                return String.format("%s: total time %d msec - timeout %d %s", this.name(), this.totalTime, this.operation.timeout(), this.operation.constructionTime().isCrossNode ? "msec/cross-node" : "msec");
            }
            return String.format("%s (timed out %d times): total time avg/min/max %d/%d/%d msec - timeout %d %s", this.name(), this.numTimeouts, this.totalTime / (long)this.numTimeouts, this.minTime, this.maxTime, this.operation.timeout(), this.operation.constructionTime().isCrossNode ? "msec/cross-node" : "msec");
        }
    }

    private static final class FailedOperations {
        public final Map<String, FailedOperation> operations;
        public final long numDropped;

        FailedOperations(Map<String, FailedOperation> operations, long numDropped) {
            this.operations = operations;
            this.numDropped = numDropped;
        }

        public boolean isEmpty() {
            return this.operations.isEmpty() && this.numDropped == 0L;
        }

        public long num() {
            return (long)this.operations.size() + this.numDropped;
        }

        public String getLogMessage() {
            if (this.isEmpty()) {
                return "";
            }
            StringBuilder ret = new StringBuilder();
            this.operations.values().forEach(o -> FailedOperations.addOperation(ret, o));
            if (this.numDropped > 0L) {
                ret.append(LINE_SEPARATOR).append("... (").append(this.numDropped).append(" were dropped)");
            }
            return ret.toString();
        }

        private static void addOperation(StringBuilder ret, FailedOperation operation) {
            if (ret.length() > 0) {
                ret.append(LINE_SEPARATOR);
            }
            ret.append(operation.getLogMessage());
        }
    }
}

