/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.io.nativeio;

import java.io.Closeable;
import java.io.File;
import java.io.FileDescriptor;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.spi.AbstractInterruptibleChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Random;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.commons.io.FileUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.LocalFileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.io.nativeio.Errno;
import org.apache.hadoop.io.nativeio.NativeIO;
import org.apache.hadoop.io.nativeio.NativeIOException;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.test.GenericTestUtils;
import org.apache.hadoop.util.NativeCodeLoader;
import org.apache.hadoop.util.Time;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TestNativeIO {
    static final Logger LOG = LoggerFactory.getLogger(TestNativeIO.class);
    static final File TEST_DIR = GenericTestUtils.getTestDir("testnativeio");

    @Before
    public void checkLoaded() {
        Assume.assumeTrue((boolean)NativeCodeLoader.isNativeCodeLoaded());
    }

    @Before
    public void setupTestDir() {
        FileUtil.fullyDelete((File)TEST_DIR);
        TEST_DIR.mkdirs();
    }

    @Test(timeout=30000L)
    public void testFstat() throws Exception {
        FileOutputStream fos = new FileOutputStream(new File(TEST_DIR, "testfstat"));
        NativeIO.POSIX.Stat stat = NativeIO.POSIX.getFstat((FileDescriptor)fos.getFD());
        fos.close();
        LOG.info("Stat: " + String.valueOf(stat));
        String owner = stat.getOwner();
        String expectedOwner = System.getProperty("user.name");
        if (Path.WINDOWS) {
            UserGroupInformation ugi = UserGroupInformation.createRemoteUser((String)expectedOwner);
            String adminsGroupString = "Administrators";
            if (Arrays.asList(ugi.getGroupNames()).contains("Administrators")) {
                expectedOwner = "Administrators";
            }
        }
        Assert.assertEquals((Object)expectedOwner, (Object)owner);
        Assert.assertNotNull((Object)stat.getGroup());
        Assert.assertTrue((!stat.getGroup().isEmpty() ? 1 : 0) != 0);
        Assert.assertEquals((String)"Stat mode field should indicate a regular file", (long)NativeIO.POSIX.Stat.S_IFREG, (long)(stat.getMode() & NativeIO.POSIX.Stat.S_IFMT));
    }

    @Test(timeout=30000L)
    public void testMultiThreadedFstat() throws Exception {
        if (Path.WINDOWS) {
            return;
        }
        final FileOutputStream fos = new FileOutputStream(new File(TEST_DIR, "testfstat"));
        final AtomicReference thrown = new AtomicReference();
        ArrayList<1> statters = new ArrayList<1>();
        for (int i = 0; i < 10; ++i) {
            Thread thread = new Thread(){

                @Override
                public void run() {
                    long et = Time.now() + 5000L;
                    while (Time.now() < et) {
                        try {
                            NativeIO.POSIX.Stat stat = NativeIO.POSIX.getFstat((FileDescriptor)fos.getFD());
                            Assert.assertEquals((Object)System.getProperty("user.name"), (Object)stat.getOwner());
                            Assert.assertNotNull((Object)stat.getGroup());
                            Assert.assertTrue((!stat.getGroup().isEmpty() ? 1 : 0) != 0);
                            Assert.assertEquals((String)"Stat mode field should indicate a regular file", (long)NativeIO.POSIX.Stat.S_IFREG, (long)(stat.getMode() & NativeIO.POSIX.Stat.S_IFMT));
                        }
                        catch (Throwable t) {
                            thrown.set(t);
                        }
                    }
                }
            };
            statters.add(thread);
            thread.start();
        }
        for (Thread thread : statters) {
            thread.join();
        }
        fos.close();
        if (thrown.get() != null) {
            throw new RuntimeException((Throwable)thrown.get());
        }
    }

    @Test(timeout=30000L)
    public void testFstatClosedFd() throws Exception {
        FileOutputStream fos = new FileOutputStream(new File(TEST_DIR, "testfstat2"));
        fos.close();
        try {
            NativeIO.POSIX.Stat stat = NativeIO.POSIX.getFstat((FileDescriptor)fos.getFD());
        }
        catch (NativeIOException nioe) {
            LOG.info("Got expected exception", (Throwable)nioe);
            Assert.assertEquals((Object)Errno.EBADF, (Object)nioe.getErrno());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=30000L)
    public void testSetFilePointer() throws Exception {
        if (!Path.WINDOWS) {
            return;
        }
        LOG.info("Set a file pointer on Windows");
        try {
            File testfile = new File(TEST_DIR, "testSetFilePointer");
            Assert.assertTrue((String)"Create test subject", (testfile.exists() || testfile.createNewFile() ? 1 : 0) != 0);
            try (FileWriter writer = new FileWriter(testfile);){
                for (int i = 0; i < 200; ++i) {
                    if (i < 100) {
                        writer.write(97);
                        continue;
                    }
                    writer.write(98);
                }
                writer.flush();
            }
            FileDescriptor fd = NativeIO.Windows.createFile((String)testfile.getCanonicalPath(), (long)0x80000000L, (long)7L, (long)3L);
            NativeIO.Windows.setFilePointer((FileDescriptor)fd, (long)120L, (long)0L);
            try (FileReader reader = new FileReader(fd);){
                int c = reader.read();
                Assert.assertTrue((String)("Unexpected character: " + c), (c == 98 ? 1 : 0) != 0);
            }
        }
        catch (Exception e) {
            Assert.fail((String)("Got unexpected exception: " + e.getMessage()));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=30000L)
    public void testCreateFile() throws Exception {
        if (!Path.WINDOWS) {
            return;
        }
        LOG.info("Open a file on Windows with SHARE_DELETE shared mode");
        try {
            File testfile = new File(TEST_DIR, "testCreateFile");
            Assert.assertTrue((String)"Create test subject", (testfile.exists() || testfile.createNewFile() ? 1 : 0) != 0);
            FileDescriptor fd = NativeIO.Windows.createFile((String)testfile.getCanonicalPath(), (long)0x80000000L, (long)7L, (long)3L);
            try (FileInputStream fin = new FileInputStream(fd);){
                fin.read();
                File newfile = new File(TEST_DIR, "testRenamedFile");
                boolean renamed = testfile.renameTo(newfile);
                Assert.assertTrue((String)"Rename failed.", (boolean)renamed);
                fin.read();
            }
        }
        catch (Exception e) {
            Assert.fail((String)("Got unexpected exception: " + e.getMessage()));
        }
    }

    @Test(timeout=30000L)
    public void testAccess() throws Exception {
        if (!Path.WINDOWS) {
            return;
        }
        File testFile = new File(TEST_DIR, "testfileaccess");
        Assert.assertTrue((boolean)testFile.createNewFile());
        FileUtil.setReadable((File)testFile, (boolean)false);
        Assert.assertFalse((boolean)NativeIO.Windows.access((String)testFile.getAbsolutePath(), (NativeIO.Windows.AccessRight)NativeIO.Windows.AccessRight.ACCESS_READ));
        FileUtil.setReadable((File)testFile, (boolean)true);
        Assert.assertTrue((boolean)NativeIO.Windows.access((String)testFile.getAbsolutePath(), (NativeIO.Windows.AccessRight)NativeIO.Windows.AccessRight.ACCESS_READ));
        FileUtil.setWritable((File)testFile, (boolean)false);
        Assert.assertFalse((boolean)NativeIO.Windows.access((String)testFile.getAbsolutePath(), (NativeIO.Windows.AccessRight)NativeIO.Windows.AccessRight.ACCESS_WRITE));
        FileUtil.setWritable((File)testFile, (boolean)true);
        Assert.assertTrue((boolean)NativeIO.Windows.access((String)testFile.getAbsolutePath(), (NativeIO.Windows.AccessRight)NativeIO.Windows.AccessRight.ACCESS_WRITE));
        FileUtil.setExecutable((File)testFile, (boolean)false);
        Assert.assertFalse((boolean)NativeIO.Windows.access((String)testFile.getAbsolutePath(), (NativeIO.Windows.AccessRight)NativeIO.Windows.AccessRight.ACCESS_EXECUTE));
        FileUtil.setExecutable((File)testFile, (boolean)true);
        Assert.assertTrue((boolean)NativeIO.Windows.access((String)testFile.getAbsolutePath(), (NativeIO.Windows.AccessRight)NativeIO.Windows.AccessRight.ACCESS_EXECUTE));
        String testFileRelativePath = "";
        for (int i = 0; i < 15; ++i) {
            testFileRelativePath = testFileRelativePath + "testfileaccessfolder\\";
        }
        testFileRelativePath = testFileRelativePath + "testfileaccess";
        testFile = new File(TEST_DIR, testFileRelativePath);
        Assert.assertTrue((boolean)testFile.getParentFile().mkdirs());
        Assert.assertTrue((boolean)testFile.createNewFile());
        FileUtil.setReadable((File)testFile, (boolean)false);
        Assert.assertFalse((boolean)NativeIO.Windows.access((String)testFile.getAbsolutePath(), (NativeIO.Windows.AccessRight)NativeIO.Windows.AccessRight.ACCESS_READ));
        FileUtil.setReadable((File)testFile, (boolean)true);
        Assert.assertTrue((boolean)NativeIO.Windows.access((String)testFile.getAbsolutePath(), (NativeIO.Windows.AccessRight)NativeIO.Windows.AccessRight.ACCESS_READ));
        FileUtil.setWritable((File)testFile, (boolean)false);
        Assert.assertFalse((boolean)NativeIO.Windows.access((String)testFile.getAbsolutePath(), (NativeIO.Windows.AccessRight)NativeIO.Windows.AccessRight.ACCESS_WRITE));
        FileUtil.setWritable((File)testFile, (boolean)true);
        Assert.assertTrue((boolean)NativeIO.Windows.access((String)testFile.getAbsolutePath(), (NativeIO.Windows.AccessRight)NativeIO.Windows.AccessRight.ACCESS_WRITE));
        FileUtil.setExecutable((File)testFile, (boolean)false);
        Assert.assertFalse((boolean)NativeIO.Windows.access((String)testFile.getAbsolutePath(), (NativeIO.Windows.AccessRight)NativeIO.Windows.AccessRight.ACCESS_EXECUTE));
        FileUtil.setExecutable((File)testFile, (boolean)true);
        Assert.assertTrue((boolean)NativeIO.Windows.access((String)testFile.getAbsolutePath(), (NativeIO.Windows.AccessRight)NativeIO.Windows.AccessRight.ACCESS_EXECUTE));
    }

    @Test(timeout=30000L)
    public void testOpenMissingWithoutCreate() throws Exception {
        if (Path.WINDOWS) {
            return;
        }
        LOG.info("Open a missing file without O_CREAT and it should fail");
        try {
            FileDescriptor fd = NativeIO.POSIX.open((String)new File(TEST_DIR, "doesntexist").getAbsolutePath(), (int)NativeIO.POSIX.O_WRONLY, (int)448);
            Assert.fail((String)"Able to open a new file without O_CREAT");
        }
        catch (NativeIOException nioe) {
            LOG.info("Got expected exception", (Throwable)nioe);
            Assert.assertEquals((Object)Errno.ENOENT, (Object)nioe.getErrno());
        }
    }

    @Test(timeout=30000L)
    public void testOpenWithCreate() throws Exception {
        if (Path.WINDOWS) {
            return;
        }
        LOG.info("Test creating a file with O_CREAT");
        FileDescriptor fd = NativeIO.POSIX.open((String)new File(TEST_DIR, "testWorkingOpen").getAbsolutePath(), (int)(NativeIO.POSIX.O_WRONLY | NativeIO.POSIX.O_CREAT), (int)448);
        Assert.assertNotNull((Object)true);
        Assert.assertTrue((boolean)fd.valid());
        FileOutputStream fos = new FileOutputStream(fd);
        fos.write("foo".getBytes());
        fos.close();
        Assert.assertFalse((boolean)fd.valid());
        LOG.info("Test exclusive create");
        try {
            fd = NativeIO.POSIX.open((String)new File(TEST_DIR, "testWorkingOpen").getAbsolutePath(), (int)(NativeIO.POSIX.O_WRONLY | NativeIO.POSIX.O_CREAT | NativeIO.POSIX.O_EXCL), (int)448);
            Assert.fail((String)"Was able to create existing file with O_EXCL");
        }
        catch (NativeIOException nioe) {
            LOG.info("Got expected exception for failed exclusive create", (Throwable)nioe);
            Assert.assertEquals((Object)Errno.EEXIST, (Object)nioe.getErrno());
        }
    }

    @Test(timeout=30000L)
    public void testFDDoesntLeak() throws IOException {
        if (Path.WINDOWS) {
            return;
        }
        for (int i = 0; i < 10000; ++i) {
            FileDescriptor fd = NativeIO.POSIX.open((String)new File(TEST_DIR, "testNoFdLeak").getAbsolutePath(), (int)(NativeIO.POSIX.O_WRONLY | NativeIO.POSIX.O_CREAT), (int)448);
            Assert.assertNotNull((Object)true);
            Assert.assertTrue((boolean)fd.valid());
            FileOutputStream fos = new FileOutputStream(fd);
            fos.write("foo".getBytes());
            fos.close();
        }
    }

    @Test(timeout=30000L)
    public void testChmod() throws Exception {
        if (Path.WINDOWS) {
            return;
        }
        try {
            NativeIO.POSIX.chmod((String)"/this/file/doesnt/exist", (int)777);
            Assert.fail((String)"Chmod of non-existent file didn't fail");
        }
        catch (NativeIOException nioe) {
            Assert.assertEquals((Object)Errno.ENOENT, (Object)nioe.getErrno());
        }
        File toChmod = new File(TEST_DIR, "testChmod");
        Assert.assertTrue((String)"Create test subject", (toChmod.exists() || toChmod.mkdir() ? 1 : 0) != 0);
        NativeIO.POSIX.chmod((String)toChmod.getAbsolutePath(), (int)511);
        this.assertPermissions(toChmod, 511);
        NativeIO.POSIX.chmod((String)toChmod.getAbsolutePath(), (int)0);
        this.assertPermissions(toChmod, 0);
        NativeIO.POSIX.chmod((String)toChmod.getAbsolutePath(), (int)420);
        this.assertPermissions(toChmod, 420);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=30000L)
    public void testPosixFadvise() throws Exception {
        if (Path.WINDOWS) {
            return;
        }
        try (FileInputStream fis = new FileInputStream("/dev/zero");){
            NativeIO.POSIX.posix_fadvise((FileDescriptor)fis.getFD(), (long)0L, (long)0L, (int)NativeIO.POSIX.POSIX_FADV_SEQUENTIAL);
        }
        try {
            NativeIO.POSIX.posix_fadvise((FileDescriptor)fis.getFD(), (long)0L, (long)1024L, (int)NativeIO.POSIX.POSIX_FADV_SEQUENTIAL);
            Assert.fail((String)"Did not throw on bad file");
        }
        catch (NativeIOException nioe) {
            Assert.assertEquals((Object)Errno.EBADF, (Object)nioe.getErrno());
        }
        try {
            NativeIO.POSIX.posix_fadvise(null, (long)0L, (long)1024L, (int)NativeIO.POSIX.POSIX_FADV_SEQUENTIAL);
            Assert.fail((String)"Did not throw on null file");
        }
        catch (NullPointerException nullPointerException) {
            // empty catch block
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=30000L)
    public void testSyncFileRange() throws Exception {
        try (FileOutputStream fos = new FileOutputStream(new File(TEST_DIR, "testSyncFileRange"));){
            fos.write("foo".getBytes());
            NativeIO.POSIX.sync_file_range((FileDescriptor)fos.getFD(), (long)0L, (long)1024L, (int)NativeIO.POSIX.SYNC_FILE_RANGE_WRITE);
        }
        try {
            NativeIO.POSIX.sync_file_range((FileDescriptor)fos.getFD(), (long)0L, (long)1024L, (int)NativeIO.POSIX.SYNC_FILE_RANGE_WRITE);
            Assert.fail((String)"Did not throw on bad file");
        }
        catch (NativeIOException nioe) {
            Assert.assertEquals((Object)Errno.EBADF, (Object)nioe.getErrno());
        }
    }

    private void assertPermissions(File f, int expected) throws IOException {
        LocalFileSystem localfs = FileSystem.getLocal((Configuration)new Configuration());
        FsPermission perms = localfs.getFileStatus(new Path(f.getAbsolutePath())).getPermission();
        Assert.assertEquals((long)expected, (long)perms.toShort());
    }

    @Test(timeout=30000L)
    public void testGetUserName() throws IOException {
        if (Path.WINDOWS) {
            return;
        }
        Assert.assertFalse((boolean)NativeIO.POSIX.getUserName((int)0).isEmpty());
    }

    @Test(timeout=30000L)
    public void testGetGroupName() throws IOException {
        if (Path.WINDOWS) {
            return;
        }
        Assert.assertFalse((boolean)NativeIO.POSIX.getGroupName((int)0).isEmpty());
    }

    @Test(timeout=30000L)
    public void testRenameTo() throws Exception {
        File TEST_DIR = GenericTestUtils.getTestDir("renameTest");
        Assume.assumeTrue((boolean)TEST_DIR.mkdirs());
        File nonExistentFile = new File(TEST_DIR, "nonexistent");
        File targetFile = new File(TEST_DIR, "target");
        try {
            NativeIO.renameTo((File)nonExistentFile, (File)targetFile);
            Assert.fail();
        }
        catch (NativeIOException e) {
            if (Path.WINDOWS) {
                Assert.assertEquals((Object)String.format("The system cannot find the file specified.%n", new Object[0]), (Object)e.getMessage());
            }
            Assert.assertEquals((Object)Errno.ENOENT, (Object)e.getErrno());
        }
        File sourceFile = new File(TEST_DIR, "source");
        Assert.assertTrue((boolean)sourceFile.createNewFile());
        NativeIO.renameTo((File)sourceFile, (File)sourceFile);
        NativeIO.renameTo((File)sourceFile, (File)targetFile);
        sourceFile = new File(TEST_DIR, "source");
        Assert.assertTrue((boolean)sourceFile.createNewFile());
        File badTarget = new File(targetFile, "subdir");
        try {
            NativeIO.renameTo((File)sourceFile, (File)badTarget);
            Assert.fail();
        }
        catch (NativeIOException e) {
            if (Path.WINDOWS) {
                Assert.assertEquals((Object)String.format("The parameter is incorrect.%n", new Object[0]), (Object)e.getMessage());
            }
            Assert.assertEquals((Object)Errno.ENOTDIR, (Object)e.getErrno());
        }
        Assert.assertTrue((boolean)targetFile.exists());
        NativeIO.renameTo((File)sourceFile, (File)targetFile);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=10000L)
    public void testMlock() throws Exception {
        Assume.assumeTrue((boolean)NativeIO.isAvailable());
        File TEST_FILE = GenericTestUtils.getTestDir("testMlockFile");
        int BUF_LEN = 12289;
        byte[] buf = new byte[12289];
        int bufSum = 0;
        for (int i = 0; i < buf.length; ++i) {
            buf[i] = (byte)(i % 60);
            bufSum += buf[i];
        }
        try (FileOutputStream fos = new FileOutputStream(TEST_FILE);){
            fos.write(buf);
            fos.getChannel().force(true);
        }
        FileInputStream fis = null;
        AbstractInterruptibleChannel channel = null;
        try {
            fis = new FileInputStream(TEST_FILE);
            channel = fis.getChannel();
            long fileSize = ((FileChannel)channel).size();
            MappedByteBuffer mapbuf = ((FileChannel)channel).map(FileChannel.MapMode.READ_ONLY, 0L, fileSize);
            NativeIO.POSIX.mlock((ByteBuffer)mapbuf, (long)fileSize);
            int sum = 0;
            int i = 0;
            while ((long)i < fileSize) {
                sum += mapbuf.get(i);
                ++i;
            }
            Assert.assertEquals((String)"Expected sums to be equal", (long)bufSum, (long)sum);
            NativeIO.POSIX.munmap((MappedByteBuffer)mapbuf);
        }
        finally {
            if (channel != null) {
                channel.close();
            }
            if (fis != null) {
                fis.close();
            }
        }
    }

    @Test(timeout=10000L)
    public void testGetMemlockLimit() throws Exception {
        Assume.assumeTrue((boolean)NativeIO.isAvailable());
        NativeIO.getMemlockLimit();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test(timeout=30000L)
    public void testCopyFileUnbuffered() throws Exception {
        String METHOD_NAME = GenericTestUtils.getMethodName();
        File srcFile = new File(TEST_DIR, METHOD_NAME + ".src.dat");
        File dstFile = new File(TEST_DIR, METHOD_NAME + ".dst.dat");
        int fileSize = 0x8000000;
        int SEED = 48879;
        int batchSize = 4096;
        int numBatches = 32768;
        Random rb = new Random(48879L);
        FileChannel channel = null;
        RandomAccessFile raSrcFile = null;
        try {
            raSrcFile = new RandomAccessFile(srcFile, "rw");
            channel = raSrcFile.getChannel();
            byte[] bytesToWrite = new byte[4096];
            MappedByteBuffer mapBuf = channel.map(FileChannel.MapMode.READ_WRITE, 0L, 0x8000000L);
            for (int i = 0; i < 32768; ++i) {
                rb.nextBytes(bytesToWrite);
                mapBuf.put(bytesToWrite);
            }
            NativeIO.copyFileUnbuffered((File)srcFile, (File)dstFile);
            Assert.assertEquals((long)srcFile.length(), (long)dstFile.length());
        }
        catch (Throwable throwable) {
            IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{channel});
            IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{raSrcFile});
            FileUtils.deleteQuietly((File)TEST_DIR);
            throw throwable;
        }
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{channel});
        IOUtils.cleanupWithLogger((Logger)LOG, (Closeable[])new Closeable[]{raSrcFile});
        FileUtils.deleteQuietly((File)TEST_DIR);
    }

    @Test(timeout=10000L)
    public void testNativePosixConsts() {
        Assume.assumeTrue((String)"Native POSIX constants not required for Windows", (!Path.WINDOWS ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native 0_RDONLY const not set", (NativeIO.POSIX.O_RDONLY >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native 0_WRONLY const not set", (NativeIO.POSIX.O_WRONLY >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native 0_RDWR const not set", (NativeIO.POSIX.O_RDWR >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native 0_CREAT const not set", (NativeIO.POSIX.O_CREAT >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native 0_EXCL const not set", (NativeIO.POSIX.O_EXCL >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native 0_NOCTTY const not set", (NativeIO.POSIX.O_NOCTTY >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native 0_TRUNC const not set", (NativeIO.POSIX.O_TRUNC >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native 0_APPEND const not set", (NativeIO.POSIX.O_APPEND >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native 0_NONBLOCK const not set", (NativeIO.POSIX.O_NONBLOCK >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native 0_SYNC const not set", (NativeIO.POSIX.O_SYNC >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native S_IFMT const not set", (NativeIO.POSIX.Stat.S_IFMT >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native S_IFIFO const not set", (NativeIO.POSIX.Stat.S_IFIFO >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native S_IFCHR const not set", (NativeIO.POSIX.Stat.S_IFCHR >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native S_IFDIR const not set", (NativeIO.POSIX.Stat.S_IFDIR >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native S_IFBLK const not set", (NativeIO.POSIX.Stat.S_IFBLK >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native S_IFREG const not set", (NativeIO.POSIX.Stat.S_IFREG >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native S_IFLNK const not set", (NativeIO.POSIX.Stat.S_IFLNK >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native S_IFSOCK const not set", (NativeIO.POSIX.Stat.S_IFSOCK >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native S_ISUID const not set", (NativeIO.POSIX.Stat.S_ISUID >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native S_ISGID const not set", (NativeIO.POSIX.Stat.S_ISGID >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native S_ISVTX const not set", (NativeIO.POSIX.Stat.S_ISVTX >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native S_IRUSR const not set", (NativeIO.POSIX.Stat.S_IRUSR >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native S_IWUSR const not set", (NativeIO.POSIX.Stat.S_IWUSR >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native S_IXUSR const not set", (NativeIO.POSIX.Stat.S_IXUSR >= 0 ? 1 : 0) != 0);
    }

    @Test(timeout=10000L)
    public void testNativeFadviseConsts() {
        Assume.assumeTrue((String)"Fadvise constants not supported", (boolean)NativeIO.POSIX.fadvisePossible);
        Assert.assertTrue((String)"Native POSIX_FADV_NORMAL const not set", (NativeIO.POSIX.POSIX_FADV_NORMAL >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native POSIX_FADV_RANDOM const not set", (NativeIO.POSIX.POSIX_FADV_RANDOM >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native POSIX_FADV_SEQUENTIAL const not set", (NativeIO.POSIX.POSIX_FADV_SEQUENTIAL >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native POSIX_FADV_WILLNEED const not set", (NativeIO.POSIX.POSIX_FADV_WILLNEED >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native POSIX_FADV_DONTNEED const not set", (NativeIO.POSIX.POSIX_FADV_DONTNEED >= 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"Native POSIX_FADV_NOREUSE const not set", (NativeIO.POSIX.POSIX_FADV_NOREUSE >= 0 ? 1 : 0) != 0);
    }
}

