/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.io.network.partition;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import org.apache.flink.runtime.io.network.buffer.Buffer;
import org.apache.flink.runtime.io.network.buffer.BufferBuilderTestUtils;
import org.apache.flink.runtime.io.network.buffer.BufferCompressor;
import org.apache.flink.runtime.io.network.buffer.BufferDecompressor;
import org.apache.flink.runtime.io.network.partition.BoundedData;
import org.apache.flink.runtime.io.network.partition.PageSizeUtil;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.junit.Assert;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(value=Parameterized.class)
public abstract class BoundedDataTestBase {
    @ClassRule
    public static final TemporaryFolder TMP_FOLDER = new TemporaryFolder();
    protected static final int BUFFER_SIZE = 0x100000;
    private static final String COMPRESSION_CODEC = "LZ4";
    private static final BufferCompressor COMPRESSOR = new BufferCompressor(0x100000, "LZ4");
    private static final BufferDecompressor DECOMPRESSOR = new BufferDecompressor(0x100000, "LZ4");
    @Parameterized.Parameter
    public static boolean compressionEnabled;

    @Parameterized.Parameters(name="compressionEnabled = {0}")
    public static Boolean[] compressionEnabled() {
        return new Boolean[]{false, true};
    }

    protected abstract boolean isRegionBased();

    protected abstract BoundedData createBoundedData(Path var1) throws IOException;

    protected abstract BoundedData createBoundedDataWithRegion(Path var1, int var2) throws IOException;

    protected BoundedData createBoundedData() throws IOException {
        return this.createBoundedData(BoundedDataTestBase.createTempPath());
    }

    private BoundedData createBoundedDataWithRegion(int regionSize) throws IOException {
        return this.createBoundedDataWithRegion(BoundedDataTestBase.createTempPath(), regionSize);
    }

    @Test
    public void testWriteAndReadData() throws Exception {
        try (BoundedData bd = this.createBoundedData();){
            this.testWriteAndReadData(bd);
        }
    }

    @Test
    public void testWriteAndReadDataAcrossRegions() throws Exception {
        if (!this.isRegionBased()) {
            return;
        }
        try (BoundedData bd = this.createBoundedDataWithRegion(1276347);){
            this.testWriteAndReadData(bd);
        }
    }

    private void testWriteAndReadData(BoundedData bd) throws Exception {
        int numInts = 10000000;
        int numBuffers = BoundedDataTestBase.writeInts(bd, 10000000);
        bd.finishWrite();
        BoundedDataTestBase.readInts(bd.createReader(), numBuffers, 10000000);
    }

    @Test
    public void returnNullAfterEmpty() throws Exception {
        try (BoundedData bd = this.createBoundedData();){
            bd.finishWrite();
            BoundedData.Reader reader = bd.createReader();
            Assert.assertNull((Object)reader.nextBuffer());
            Assert.assertNull((Object)reader.nextBuffer());
            Assert.assertNull((Object)reader.nextBuffer());
        }
    }

    @Test
    public void testDeleteFileOnClose() throws Exception {
        Path path = BoundedDataTestBase.createTempPath();
        BoundedData bd = this.createBoundedData(path);
        Assert.assertTrue((boolean)Files.exists(path, new LinkOption[0]));
        bd.close();
        Assert.assertFalse((boolean)Files.exists(path, new LinkOption[0]));
    }

    @Test
    public void testGetSizeSingleRegion() throws Exception {
        try (BoundedData bd = this.createBoundedData();){
            BoundedDataTestBase.testGetSize(bd, 60787, 76687);
        }
    }

    @Test
    public void testGetSizeMultipleRegions() throws Exception {
        if (!this.isRegionBased()) {
            return;
        }
        int pageSize = PageSizeUtil.getSystemPageSizeOrConservativeMultiple();
        try (BoundedData bd = this.createBoundedDataWithRegion(pageSize);){
            BoundedDataTestBase.testGetSize(bd, pageSize / 3, pageSize - 8);
        }
    }

    private static void testGetSize(BoundedData bd, int bufferSize1, int bufferSize2) throws Exception {
        int expectedSize1 = bufferSize1 + 8;
        int expectedSizeFinal = bufferSize1 + bufferSize2 + 16;
        bd.writeBuffer(BufferBuilderTestUtils.buildSomeBuffer(bufferSize1));
        Assert.assertEquals((long)expectedSize1, (long)bd.getSize());
        bd.writeBuffer(BufferBuilderTestUtils.buildSomeBuffer(bufferSize2));
        Assert.assertEquals((long)expectedSizeFinal, (long)bd.getSize());
        bd.finishWrite();
        Assert.assertEquals((long)expectedSizeFinal, (long)bd.getSize());
    }

    private static int writeInts(BoundedData bd, int numInts) throws IOException {
        int numIntsInBuffer = 262144;
        int numBuffers = 0;
        for (int nextValue = 0; nextValue < numInts; nextValue += 262144) {
            Buffer buffer = BufferBuilderTestUtils.buildBufferWithAscendingInts(0x100000, 262144, nextValue);
            if (compressionEnabled) {
                bd.writeBuffer(COMPRESSOR.compressToIntermediateBuffer(buffer));
            } else {
                bd.writeBuffer(buffer);
            }
            ++numBuffers;
        }
        return numBuffers;
    }

    private static void readInts(BoundedData.Reader reader, int numBuffersExpected, int numInts) throws IOException {
        Buffer b;
        int nextValue = 0;
        int numBuffers = 0;
        while ((b = reader.nextBuffer()) != null) {
            int numIntsInBuffer = b.getSize() / 4;
            if (compressionEnabled && b.isCompressed()) {
                Buffer decompressedBuffer = DECOMPRESSOR.decompressToIntermediateBuffer(b);
                BufferBuilderTestUtils.validateBufferWithAscendingInts(decompressedBuffer, numIntsInBuffer, nextValue);
            } else {
                BufferBuilderTestUtils.validateBufferWithAscendingInts(b, numIntsInBuffer, nextValue);
            }
            nextValue += numIntsInBuffer;
            ++numBuffers;
            b.recycleBuffer();
        }
        Assert.assertEquals((long)numBuffersExpected, (long)numBuffers);
        Assert.assertThat((Object)nextValue, (Matcher)Matchers.greaterThanOrEqualTo((Comparable)Integer.valueOf(numInts)));
    }

    private static Path createTempPath() throws IOException {
        return new File(TMP_FOLDER.newFolder(), "subpartitiondata").toPath();
    }
}

