/*
 * Decompiled with CFR 0.152.
 */
package org.apache.falcon.entity.parser;

import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.TimeZone;
import org.apache.commons.lang3.StringUtils;
import org.apache.falcon.FalconException;
import org.apache.falcon.catalog.CatalogServiceFactory;
import org.apache.falcon.entity.CatalogStorage;
import org.apache.falcon.entity.ClusterHelper;
import org.apache.falcon.entity.EntityUtil;
import org.apache.falcon.entity.FeedHelper;
import org.apache.falcon.entity.FileSystemStorage;
import org.apache.falcon.entity.Storage;
import org.apache.falcon.entity.parser.CrossEntityValidations;
import org.apache.falcon.entity.parser.EntityParser;
import org.apache.falcon.entity.parser.ValidationException;
import org.apache.falcon.entity.store.ConfigurationStore;
import org.apache.falcon.entity.v0.AccessControlList;
import org.apache.falcon.entity.v0.Entity;
import org.apache.falcon.entity.v0.EntityGraph;
import org.apache.falcon.entity.v0.EntityType;
import org.apache.falcon.entity.v0.Frequency;
import org.apache.falcon.entity.v0.feed.ACL;
import org.apache.falcon.entity.v0.feed.ClusterType;
import org.apache.falcon.entity.v0.feed.Feed;
import org.apache.falcon.entity.v0.feed.Location;
import org.apache.falcon.entity.v0.feed.LocationType;
import org.apache.falcon.entity.v0.feed.Properties;
import org.apache.falcon.entity.v0.feed.Property;
import org.apache.falcon.entity.v0.feed.Sla;
import org.apache.falcon.entity.v0.process.Cluster;
import org.apache.falcon.entity.v0.process.Input;
import org.apache.falcon.entity.v0.process.Output;
import org.apache.falcon.entity.v0.process.Process;
import org.apache.falcon.expression.ExpressionHelper;
import org.apache.falcon.group.FeedGroup;
import org.apache.falcon.group.FeedGroupMap;
import org.apache.falcon.service.LifecyclePolicyMap;
import org.apache.falcon.util.DateUtil;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.security.authorize.AuthorizationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FeedEntityParser
extends EntityParser<Feed> {
    private static final Logger LOG = LoggerFactory.getLogger(FeedEntityParser.class);

    public FeedEntityParser() {
        super(EntityType.FEED);
    }

    @Override
    public void validate(Feed feed) throws FalconException {
        if (feed.getTimezone() == null) {
            feed.setTimezone(TimeZone.getTimeZone("UTC"));
        }
        if (feed.getClusters() == null) {
            throw new ValidationException("Feed should have at least one cluster");
        }
        this.validateLifecycle(feed);
        this.validateACL(feed);
        for (org.apache.falcon.entity.v0.feed.Cluster cluster : feed.getClusters().getClusters()) {
            this.validateEntityExists(EntityType.CLUSTER, cluster.getName());
            if (cluster.getValidity().getEnd() == null) {
                cluster.getValidity().setEnd(DateUtil.NEVER);
            }
            this.validateClusterValidity(cluster.getValidity().getStart(), cluster.getValidity().getEnd(), cluster.getName());
            this.validateClusterHasRegistry(feed, cluster);
            this.validateFeedCutOffPeriod(feed, cluster);
        }
        this.validateFeedStorage(feed);
        this.validateFeedPath(feed);
        this.validateFeedPartitionExpression(feed);
        this.validateFeedGroups(feed);
        this.validateFeedSLA(feed);
        this.validateProperties(feed);
        Feed oldFeed = (Feed)ConfigurationStore.get().get(EntityType.FEED, feed.getName());
        if (oldFeed == null) {
            return;
        }
        EntityGraph graph = EntityGraph.get();
        Set<Entity> referenced = graph.getDependents((Entity)oldFeed);
        Set<Process> processes = this.findProcesses(referenced);
        if (processes.isEmpty()) {
            return;
        }
        this.ensureValidityFor(feed, processes);
    }

    private void validateLifecycle(Feed feed) throws FalconException {
        LifecyclePolicyMap map = LifecyclePolicyMap.get();
        for (org.apache.falcon.entity.v0.feed.Cluster cluster : feed.getClusters().getClusters()) {
            if (!FeedHelper.isLifecycleEnabled(feed, cluster.getName())) continue;
            if (FeedHelper.getRetentionStage(feed, cluster.getName()) == null) {
                throw new ValidationException("Retention is a mandatory stage, didn't find it for cluster: " + cluster.getName());
            }
            this.validateRetentionFrequency(feed, cluster.getName());
            for (String policyName : FeedHelper.getPolicies(feed, cluster.getName())) {
                map.get(policyName).validate(feed, cluster.getName());
            }
        }
    }

    private void validateRetentionFrequency(Feed feed, String clusterName) throws FalconException {
        Frequency retentionFrequency = FeedHelper.getRetentionFrequency(feed, clusterName);
        Frequency feedFrequency = feed.getFrequency();
        if (DateUtil.getFrequencyInMillis(retentionFrequency) < DateUtil.getFrequencyInMillis(feedFrequency)) {
            throw new ValidationException("Retention can not be more frequent than data availability.");
        }
    }

    private Set<Process> findProcesses(Set<Entity> referenced) {
        HashSet<Process> processes = new HashSet<Process>();
        for (Entity entity : referenced) {
            if (entity.getEntityType() != EntityType.PROCESS) continue;
            processes.add((Process)entity);
        }
        return processes;
    }

    private void validateFeedSLA(Feed feed) throws FalconException {
        for (org.apache.falcon.entity.v0.feed.Cluster cluster : feed.getClusters().getClusters()) {
            Sla clusterSla = FeedHelper.getSLA(cluster, feed);
            if (clusterSla == null) continue;
            Frequency slaLowExpression = clusterSla.getSlaLow();
            ExpressionHelper evaluator = ExpressionHelper.get();
            ExpressionHelper.setReferenceDate(new Date());
            Date slaLow = new Date(evaluator.evaluate(slaLowExpression.toString(), Long.class));
            Frequency slaHighExpression = clusterSla.getSlaHigh();
            Date slaHigh = new Date(evaluator.evaluate(slaHighExpression.toString(), Long.class));
            if (slaLow.after(slaHigh)) {
                throw new ValidationException("slaLow of Feed: " + slaLowExpression + "is greater than slaHigh: " + slaHighExpression + " for cluster: " + cluster.getName());
            }
            Frequency retentionExpression = cluster.getRetention().getLimit();
            Date retention = new Date(evaluator.evaluate(retentionExpression.toString(), Long.class));
            if (!slaHigh.after(retention)) continue;
            throw new ValidationException("slaHigh of Feed: " + slaHighExpression + " is greater than retention of the feed: " + retentionExpression + " for cluster: " + cluster.getName());
        }
    }

    private void validateFeedGroups(Feed feed) throws FalconException {
        String[] groupNames = feed.getGroups() != null ? feed.getGroups().split(",") : new String[]{};
        Storage storage = FeedHelper.createStorage(feed);
        String defaultPath = storage.getUriTemplate(LocationType.DATA);
        for (org.apache.falcon.entity.v0.feed.Cluster cluster : feed.getClusters().getClusters()) {
            String uriTemplate = FeedHelper.createStorage(cluster, feed).getUriTemplate(LocationType.DATA);
            if (FeedGroup.getDatePattern(uriTemplate).equals(FeedGroup.getDatePattern(defaultPath))) continue;
            throw new ValidationException("Feeds default path pattern: " + storage.getUriTemplate(LocationType.DATA) + ", does not match with cluster: " + cluster.getName() + " path pattern: " + uriTemplate);
        }
        for (String groupName : groupNames) {
            FeedGroup group = FeedGroupMap.get().getGroupsMapping().get(groupName);
            if (group == null || group.canContainFeed(feed)) continue;
            throw new ValidationException("Feed " + feed.getName() + "'s frequency: " + feed.getFrequency().toString() + ", path pattern: " + storage + " does not match with group: " + group.getName() + "'s frequency: " + group.getFrequency() + ", date pattern: " + group.getDatePattern());
        }
    }

    private void ensureValidityFor(Feed newFeed, Set<Process> processes) throws FalconException {
        for (Process process : processes) {
            try {
                this.ensureValidityFor(newFeed, process);
            }
            catch (FalconException e) {
                throw new ValidationException("Process " + process.getName() + " is not compatible " + "with changes to feed " + newFeed.getName(), e);
            }
        }
    }

    private void ensureValidityFor(Feed newFeed, Process process) throws FalconException {
        for (Cluster cluster : process.getClusters().getClusters()) {
            String clusterName = cluster.getName();
            if (process.getInputs() != null) {
                for (Input input : process.getInputs().getInputs()) {
                    if (!input.getFeed().equals(newFeed.getName())) continue;
                    CrossEntityValidations.validateFeedDefinedForCluster(newFeed, clusterName);
                    CrossEntityValidations.validateFeedRetentionPeriod(input.getStart(), newFeed, clusterName);
                    CrossEntityValidations.validateInstanceRange(process, input, newFeed);
                    this.validateInputPartition(newFeed, input);
                }
            }
            if (process.getOutputs() != null) {
                for (Output output : process.getOutputs().getOutputs()) {
                    if (!output.getFeed().equals(newFeed.getName())) continue;
                    CrossEntityValidations.validateFeedDefinedForCluster(newFeed, clusterName);
                    CrossEntityValidations.validateInstance(process, output, newFeed);
                }
            }
            LOG.debug("Verified and found {} to be valid for new definition of {}", (Object)process.getName(), (Object)newFeed.getName());
        }
    }

    private void validateInputPartition(Feed newFeed, Input input) throws FalconException {
        if (input.getPartition() == null) {
            return;
        }
        Storage.TYPE baseFeedStorageType = FeedHelper.getStorageType(newFeed);
        if (baseFeedStorageType == Storage.TYPE.FILESYSTEM) {
            CrossEntityValidations.validateInputPartition(input, newFeed);
        } else if (baseFeedStorageType == Storage.TYPE.TABLE) {
            throw new ValidationException("Input partitions are not supported for table storage: " + input.getName());
        }
    }

    private void validateClusterValidity(Date start, Date end, String clusterName) throws FalconException {
        try {
            if (start.after(end)) {
                throw new ValidationException("Feed start time: " + start + " cannot be after feed end time: " + end + " for cluster: " + clusterName);
            }
        }
        catch (ValidationException e) {
            throw new ValidationException(e);
        }
        catch (Exception e) {
            throw new FalconException(e);
        }
    }

    private void validateFeedCutOffPeriod(Feed feed, org.apache.falcon.entity.v0.feed.Cluster cluster) throws FalconException {
        ExpressionHelper evaluator = ExpressionHelper.get();
        String feedRetention = cluster.getRetention().getLimit().toString();
        long retentionPeriod = evaluator.evaluate(feedRetention, Long.class);
        if (feed.getLateArrival() == null) {
            LOG.debug("Feed's late arrival cut-off not set");
            return;
        }
        String feedCutoff = feed.getLateArrival().getCutOff().toString();
        long feedCutOffPeriod = evaluator.evaluate(feedCutoff, Long.class);
        if (retentionPeriod < feedCutOffPeriod) {
            throw new ValidationException("Feed's retention limit: " + feedRetention + " of referenced cluster " + cluster.getName() + " should be more than feed's late arrival cut-off period: " + feedCutoff + " for feed: " + feed.getName());
        }
    }

    private void validateFeedPartitionExpression(Feed feed) throws FalconException {
        int numSourceClusters = 0;
        int numTrgClusters = 0;
        HashSet<String> clusters = new HashSet<String>();
        for (org.apache.falcon.entity.v0.feed.Cluster cl : feed.getClusters().getClusters()) {
            if (!clusters.add(cl.getName())) {
                throw new ValidationException("Cluster: " + cl.getName() + " is defined more than once for feed: " + feed.getName());
            }
            if (cl.getType() == ClusterType.SOURCE) {
                ++numSourceClusters;
                continue;
            }
            if (cl.getType() != ClusterType.TARGET) continue;
            ++numTrgClusters;
        }
        if (numTrgClusters >= 1 && numSourceClusters == 0) {
            throw new ValidationException("Feed: " + feed.getName() + " should have atleast one source cluster defined");
        }
        int feedParts = feed.getPartitions() != null ? feed.getPartitions().getPartitions().size() : 0;
        for (org.apache.falcon.entity.v0.feed.Cluster cluster : feed.getClusters().getClusters()) {
            if (cluster.getType() == ClusterType.SOURCE && numSourceClusters > 1 && numTrgClusters >= 1) {
                String part = FeedHelper.normalizePartitionExpression(cluster.getPartition());
                if (StringUtils.split((String)part, (char)'/').length == 0) {
                    throw new ValidationException("Partition expression has to be specified for cluster " + cluster.getName() + " as there are more than one source clusters");
                }
                this.validateClusterExpDefined(cluster);
                continue;
            }
            if (cluster.getType() != ClusterType.TARGET) continue;
            for (org.apache.falcon.entity.v0.feed.Cluster src : feed.getClusters().getClusters()) {
                String part;
                int numParts;
                if (src.getType() != ClusterType.SOURCE || (numParts = StringUtils.split((String)(part = FeedHelper.normalizePartitionExpression(src.getPartition(), cluster.getPartition())), (char)'/').length) <= feedParts) continue;
                throw new ValidationException("Partition for " + src.getName() + " and " + cluster.getName() + "clusters is more than the number of partitions defined in feed");
            }
            if (numTrgClusters <= 1 || numSourceClusters < 1) continue;
            this.validateClusterExpDefined(cluster);
        }
    }

    private void validateClusterExpDefined(org.apache.falcon.entity.v0.feed.Cluster cl) throws FalconException {
        String part;
        if (cl.getPartition() == null) {
            return;
        }
        org.apache.falcon.entity.v0.cluster.Cluster cluster = (org.apache.falcon.entity.v0.cluster.Cluster)EntityUtil.getEntity(EntityType.CLUSTER, cl.getName());
        if (FeedHelper.evaluateClusterExp(cluster, part = FeedHelper.normalizePartitionExpression(cl.getPartition())).equals(part)) {
            throw new ValidationException("Alteast one of the partition tags has to be a cluster expression for cluster " + cl.getName());
        }
    }

    private void validateFeedStorage(Feed feed) throws FalconException {
        Storage.TYPE baseFeedStorageType = FeedHelper.getStorageType(feed);
        this.validateMultipleSourcesExist(feed, baseFeedStorageType);
        this.validateUniformStorageType(feed, baseFeedStorageType);
        this.validatePartitions(feed, baseFeedStorageType);
        this.validateStorageExists(feed);
    }

    private void validateMultipleSourcesExist(Feed feed, Storage.TYPE baseFeedStorageType) throws FalconException {
        if (baseFeedStorageType == Storage.TYPE.FILESYSTEM) {
            return;
        }
        int numberOfSourceClusters = 0;
        for (org.apache.falcon.entity.v0.feed.Cluster cluster : feed.getClusters().getClusters()) {
            if (cluster.getType() != ClusterType.SOURCE) continue;
            ++numberOfSourceClusters;
        }
        if (numberOfSourceClusters > 1) {
            throw new ValidationException("Multiple sources are not supported for feed with table storage: " + feed.getName());
        }
    }

    private void validateUniformStorageType(Feed feed, Storage.TYPE feedStorageType) throws FalconException {
        for (org.apache.falcon.entity.v0.feed.Cluster cluster : feed.getClusters().getClusters()) {
            Storage.TYPE feedClusterStorageType = FeedHelper.getStorageType(feed, cluster);
            if (feedStorageType == feedClusterStorageType) continue;
            throw new ValidationException("The storage type is not uniform for cluster: " + cluster.getName());
        }
    }

    private void validateClusterHasRegistry(Feed feed, org.apache.falcon.entity.v0.feed.Cluster cluster) throws FalconException {
        Storage.TYPE feedClusterStorageType = FeedHelper.getStorageType(feed, cluster);
        if (feedClusterStorageType != Storage.TYPE.TABLE) {
            return;
        }
        org.apache.falcon.entity.v0.cluster.Cluster clusterEntity = (org.apache.falcon.entity.v0.cluster.Cluster)EntityUtil.getEntity(EntityType.CLUSTER, cluster.getName());
        if (ClusterHelper.getRegistryEndPoint(clusterEntity) == null) {
            throw new ValidationException("Cluster should have registry interface defined: " + clusterEntity.getName());
        }
    }

    private void validatePartitions(Feed feed, Storage.TYPE storageType) throws FalconException {
        if (storageType == Storage.TYPE.TABLE && feed.getPartitions() != null) {
            throw new ValidationException("Partitions are not supported for feeds with table storage. It should be defined as part of the table URI. " + feed.getName());
        }
    }

    private void validateStorageExists(Feed feed) throws FalconException {
        StringBuilder buffer = new StringBuilder();
        for (org.apache.falcon.entity.v0.feed.Cluster cluster : feed.getClusters().getClusters()) {
            Storage storage;
            org.apache.falcon.entity.v0.cluster.Cluster clusterEntity = (org.apache.falcon.entity.v0.cluster.Cluster)EntityUtil.getEntity(EntityType.CLUSTER, cluster.getName());
            if (!EntityUtil.responsibleFor(clusterEntity.getColo()) || (storage = FeedHelper.createStorage(cluster, feed)).getType() == Storage.TYPE.FILESYSTEM) continue;
            CatalogStorage catalogStorage = (CatalogStorage)storage;
            Configuration clusterConf = ClusterHelper.getConfiguration(clusterEntity);
            if (CatalogServiceFactory.getCatalogService().tableExists(clusterConf, catalogStorage.getCatalogUrl(), catalogStorage.getDatabase(), catalogStorage.getTable())) continue;
            buffer.append("Table [").append(catalogStorage.getTable()).append("] does not exist for feed: ").append(feed.getName()).append(" in cluster: ").append(cluster.getName());
        }
        if (buffer.length() > 0) {
            throw new ValidationException(buffer.toString());
        }
    }

    protected void validateACL(Feed feed) throws FalconException {
        if (this.isAuthorizationDisabled) {
            return;
        }
        ACL feedACL = feed.getACL();
        this.validateACLOwnerAndGroup((AccessControlList)feedACL);
        try {
            this.authorize(feed.getName(), (AccessControlList)feedACL);
        }
        catch (AuthorizationException e) {
            throw new ValidationException((Exception)((Object)e));
        }
        for (org.apache.falcon.entity.v0.feed.Cluster cluster : feed.getClusters().getClusters()) {
            org.apache.falcon.entity.v0.cluster.Cluster clusterEntity = (org.apache.falcon.entity.v0.cluster.Cluster)EntityUtil.getEntity(EntityType.CLUSTER, cluster.getName());
            if (!EntityUtil.responsibleFor(clusterEntity.getColo())) continue;
            Storage storage = FeedHelper.createStorage(cluster, feed);
            try {
                storage.validateACL((AccessControlList)feedACL);
            }
            catch (FalconException e) {
                throw new ValidationException(e);
            }
        }
    }

    protected void validateProperties(Feed feed) throws ValidationException {
        Properties properties = feed.getProperties();
        if (properties == null) {
            return;
        }
        List propertyList = feed.getProperties().getProperties();
        HashSet<String> propertyKeys = new HashSet<String>();
        for (Property prop : propertyList) {
            if (StringUtils.isBlank((CharSequence)prop.getName())) {
                throw new ValidationException("Property name and value cannot be empty for Feed : " + feed.getName());
            }
            if (propertyKeys.add(prop.getName())) continue;
            throw new ValidationException("Multiple properties with same name found for Feed : " + feed.getName());
        }
    }

    private void validateFeedPath(Feed feed) throws FalconException {
        if (FeedHelper.getStorageType(feed) == Storage.TYPE.TABLE) {
            return;
        }
        for (org.apache.falcon.entity.v0.feed.Cluster cluster : feed.getClusters().getClusters()) {
            List<Location> locations = FeedHelper.getLocations(cluster, feed);
            Location dataLocation = FileSystemStorage.getLocation(locations, LocationType.DATA);
            if (dataLocation != null) continue;
            throw new ValidationException(feed.getName() + " is a FileSystem based feed " + "but it doesn't contain location type - data in cluster " + cluster.getName());
        }
    }
}

