/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.base.Throwables;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ConcurrentHashMultiset;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Multiset;
import com.google.common.util.concurrent.RateLimiter;
import java.io.File;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.TimeUnit;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.management.openmbean.OpenDataException;
import javax.management.openmbean.TabularData;
import org.apache.cassandra.cache.AutoSavingCache;
import org.apache.cassandra.concurrent.DebuggableThreadPoolExecutor;
import org.apache.cassandra.concurrent.JMXEnabledThreadPoolExecutor;
import org.apache.cassandra.concurrent.NamedThreadFactory;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.Column;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.CounterColumn;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.OnDiskAtom;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.compaction.AbstractCompactedRow;
import org.apache.cassandra.db.compaction.AbstractCompactionStrategy;
import org.apache.cassandra.db.compaction.AbstractCompactionTask;
import org.apache.cassandra.db.compaction.CompactionController;
import org.apache.cassandra.db.compaction.CompactionInfo;
import org.apache.cassandra.db.compaction.CompactionInterruptedException;
import org.apache.cassandra.db.compaction.CompactionIterable;
import org.apache.cassandra.db.compaction.CompactionManagerMBean;
import org.apache.cassandra.db.compaction.ICompactionScanner;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.db.compaction.Scrubber;
import org.apache.cassandra.db.index.SecondaryIndexBuilder;
import org.apache.cassandra.dht.AbstractBounds;
import org.apache.cassandra.dht.Bounds;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.sstable.SSTableIdentityIterator;
import org.apache.cassandra.io.sstable.SSTableMetadata;
import org.apache.cassandra.io.sstable.SSTableReader;
import org.apache.cassandra.io.sstable.SSTableWriter;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.metrics.CompactionMetrics;
import org.apache.cassandra.repair.Validator;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.CounterId;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.Pair;
import org.apache.cassandra.utils.WrappedRunnable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompactionManager
implements CompactionManagerMBean {
    public static final String MBEAN_OBJECT_NAME = "org.apache.cassandra.db:type=CompactionManager";
    private static final Logger logger = LoggerFactory.getLogger(CompactionManager.class);
    public static final CompactionManager instance;
    public static final int NO_GC = Integer.MIN_VALUE;
    public static final int GC_ALL = Integer.MAX_VALUE;
    public static final ThreadLocal<Boolean> isCompactionManager;
    private final CompactionExecutor executor = new CompactionExecutor();
    private final CompactionExecutor validationExecutor = new ValidationExecutor();
    private static final CompactionExecutor cacheCleanupExecutor;
    private final CompactionMetrics metrics = new CompactionMetrics(this.executor, this.validationExecutor);
    private final Multiset<ColumnFamilyStore> compactingCF = ConcurrentHashMultiset.create();
    private final RateLimiter compactionRateLimiter = RateLimiter.create((double)Double.MAX_VALUE);

    public RateLimiter getRateLimiter() {
        double currentThroughput = (double)DatabaseDescriptor.getCompactionThroughputMbPerSec() * 1024.0 * 1024.0;
        if (currentThroughput == 0.0 || StorageService.instance.isBootstrapMode()) {
            currentThroughput = Double.MAX_VALUE;
        }
        if (this.compactionRateLimiter.getRate() != currentThroughput) {
            this.compactionRateLimiter.setRate(currentThroughput);
        }
        return this.compactionRateLimiter;
    }

    public List<Future<?>> submitBackground(ColumnFamilyStore cfs) {
        if (cfs.isAutoCompactionDisabled()) {
            logger.debug("Autocompaction is disabled");
            return Collections.emptyList();
        }
        int count = this.compactingCF.count((Object)cfs);
        if (count > 0 && this.executor.getActiveCount() >= this.executor.getMaximumPoolSize()) {
            logger.debug("Background compaction is still running for {}.{} ({} remaining). Skipping", new Object[]{cfs.keyspace.getName(), cfs.name, count});
            return Collections.emptyList();
        }
        logger.debug("Scheduling a background task check for {}.{} with {}", new Object[]{cfs.keyspace.getName(), cfs.name, cfs.getCompactionStrategy().getClass().getSimpleName()});
        ArrayList futures = new ArrayList();
        do {
            this.compactingCF.add((Object)cfs);
            futures.add(this.executor.submit(new BackgroundCompactionTask(cfs)));
        } while (this.executor.getActiveCount() + futures.size() < this.executor.getMaximumPoolSize());
        return futures;
    }

    public boolean isCompacting(Iterable<ColumnFamilyStore> cfses) {
        for (ColumnFamilyStore cfs : cfses) {
            if (cfs.getDataTracker().getCompacting().isEmpty()) continue;
            return true;
        }
        return false;
    }

    private void performAllSSTableOperation(final ColumnFamilyStore cfs, final AllSSTablesOperation operation) throws InterruptedException, ExecutionException {
        final Iterable<SSTableReader> sstables = cfs.markAllCompacting();
        if (sstables == null) {
            return;
        }
        Callable<Object> runnable = new Callable<Object>(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public Object call() throws IOException {
                try {
                    operation.perform(cfs, sstables);
                }
                finally {
                    cfs.getDataTracker().unmarkCompacting(sstables);
                }
                return this;
            }
        };
        this.executor.submit(runnable).get();
    }

    public void performScrub(ColumnFamilyStore cfStore, final boolean skipCorrupted) throws InterruptedException, ExecutionException {
        this.performAllSSTableOperation(cfStore, new AllSSTablesOperation(){

            @Override
            public void perform(ColumnFamilyStore store, Iterable<SSTableReader> sstables) throws IOException {
                CompactionManager.this.doScrub(store, sstables, skipCorrupted);
            }
        });
    }

    public void performSSTableRewrite(ColumnFamilyStore cfStore, final boolean excludeCurrentVersion) throws InterruptedException, ExecutionException {
        this.performAllSSTableOperation(cfStore, new AllSSTablesOperation(){

            @Override
            public void perform(ColumnFamilyStore cfs, Iterable<SSTableReader> sstables) {
                for (SSTableReader sstable : sstables) {
                    if (excludeCurrentVersion && sstable.descriptor.version.equals(Descriptor.Version.CURRENT)) continue;
                    AbstractCompactionTask task = cfs.getCompactionStrategy().getCompactionTask(Collections.singleton(sstable), Integer.MIN_VALUE, Long.MAX_VALUE);
                    task.setUserDefined(true);
                    task.setCompactionType(OperationType.UPGRADE_SSTABLES);
                    task.execute(CompactionManager.this.metrics);
                }
            }
        });
    }

    public void performCleanup(ColumnFamilyStore cfStore, final CounterId.OneShotRenewer renewer) throws InterruptedException, ExecutionException {
        this.performAllSSTableOperation(cfStore, new AllSSTablesOperation(){

            @Override
            public void perform(ColumnFamilyStore store, Iterable<SSTableReader> sstables) throws IOException {
                ArrayList sortedSSTables = Lists.newArrayList(sstables);
                Collections.sort(sortedSSTables, new SSTableReader.SizeComparator());
                CompactionManager.this.doCleanupCompaction(store, sortedSSTables, renewer);
            }
        });
    }

    public void performMaximal(ColumnFamilyStore cfStore) throws InterruptedException, ExecutionException {
        this.submitMaximal(cfStore, CompactionManager.getDefaultGcBefore(cfStore)).get();
    }

    public Future<?> submitMaximal(ColumnFamilyStore cfStore, int gcBefore) {
        final AbstractCompactionTask task = cfStore.getCompactionStrategy().getMaximalTask(gcBefore);
        WrappedRunnable runnable = new WrappedRunnable(){

            @Override
            protected void runMayThrow() throws IOException {
                if (task == null) {
                    return;
                }
                task.execute(CompactionManager.this.metrics);
            }
        };
        return this.executor.submit(runnable);
    }

    @Override
    public void forceUserDefinedCompaction(String dataFiles) {
        String[] filenames = dataFiles.split(",");
        ArrayListMultimap descriptors = ArrayListMultimap.create();
        for (String filename : filenames) {
            Descriptor desc = Descriptor.fromFilename(filename.trim());
            if (Schema.instance.getCFMetaData(desc) == null) {
                logger.warn("Schema does not exist for file {}. Skipping.", (Object)filename);
                continue;
            }
            File directory = new File(desc.ksname + File.separator + desc.cfname);
            Pair<Descriptor, String> p = Descriptor.fromFilename(directory, filename.trim());
            Pair<String, String> key = Pair.create(((Descriptor)p.left).ksname, ((Descriptor)p.left).cfname);
            descriptors.put(key, p.left);
        }
        ArrayList futures = new ArrayList();
        for (Pair key : descriptors.keySet()) {
            ColumnFamilyStore cfs = Keyspace.open((String)key.left).getColumnFamilyStore((String)key.right);
            futures.add(this.submitUserDefined(cfs, descriptors.get((Object)key), CompactionManager.getDefaultGcBefore(cfs)));
        }
        FBUtilities.waitOnFutures(futures);
    }

    public Future<?> submitUserDefined(final ColumnFamilyStore cfs, final Collection<Descriptor> dataFiles, final int gcBefore) {
        WrappedRunnable runnable = new WrappedRunnable(){

            @Override
            protected void runMayThrow() throws IOException {
                ArrayList<SSTableReader> sstables = new ArrayList<SSTableReader>(dataFiles.size());
                for (Descriptor desc : dataFiles) {
                    SSTableReader sstable = CompactionManager.this.lookupSSTable(cfs, desc);
                    if (sstable == null) {
                        logger.info("Will not compact {}: it is not an active sstable", (Object)desc);
                        continue;
                    }
                    sstables.add(sstable);
                }
                if (sstables.isEmpty()) {
                    logger.info("No files to compact for user defined compaction");
                } else {
                    AbstractCompactionTask task = cfs.getCompactionStrategy().getUserDefinedTask(sstables, gcBefore);
                    if (task != null) {
                        task.execute(CompactionManager.this.metrics);
                    }
                }
            }
        };
        return this.executor.submit(runnable);
    }

    private SSTableReader lookupSSTable(ColumnFamilyStore cfs, Descriptor descriptor) {
        for (SSTableReader sstable : cfs.getSSTables()) {
            if (!sstable.descriptor.toString().endsWith(descriptor.toString())) continue;
            return sstable;
        }
        return null;
    }

    public Future<Object> submitValidation(final ColumnFamilyStore cfStore, final Validator validator) {
        Callable<Object> callable = new Callable<Object>(){

            @Override
            public Object call() throws IOException {
                try {
                    CompactionManager.this.doValidationCompaction(cfStore, validator);
                }
                catch (Throwable e) {
                    validator.fail();
                    throw e;
                }
                return this;
            }
        };
        return this.validationExecutor.submit(callable);
    }

    public void disableAutoCompaction() {
        for (String ksname : Schema.instance.getNonSystemKeyspaces()) {
            for (ColumnFamilyStore cfs : Keyspace.open(ksname).getColumnFamilyStores()) {
                cfs.disableAutoCompaction();
            }
        }
    }

    private void doScrub(ColumnFamilyStore cfs, Iterable<SSTableReader> sstables, boolean skipCorrupted) throws IOException {
        assert (!cfs.isIndex());
        for (SSTableReader sstable : sstables) {
            this.scrubOne(cfs, sstable, skipCorrupted);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void scrubOne(ColumnFamilyStore cfs, SSTableReader sstable, boolean skipCorrupted) throws IOException {
        Scrubber scrubber = new Scrubber(cfs, sstable, skipCorrupted);
        CompactionInfo.Holder scrubInfo = scrubber.getScrubInfo();
        this.metrics.beginCompaction(scrubInfo);
        try {
            scrubber.scrub();
        }
        finally {
            scrubber.close();
            this.metrics.finishCompaction(scrubInfo);
        }
        if (scrubber.getNewInOrderSSTable() != null) {
            cfs.addSSTable(scrubber.getNewInOrderSSTable());
        }
        if (scrubber.getNewSSTable() == null) {
            cfs.markObsolete(Collections.singletonList(sstable), OperationType.SCRUB);
        } else {
            cfs.replaceCompactedSSTables(Collections.singletonList(sstable), Collections.singletonList(scrubber.getNewSSTable()), OperationType.SCRUB);
        }
    }

    static boolean needsCleanup(SSTableReader sstable, Collection<Range<Token>> ownedRanges) {
        assert (!ownedRanges.isEmpty());
        List sortedRanges = Range.normalize(ownedRanges);
        Range firstRange = sortedRanges.get(0);
        if (sstable.first.token.compareTo((Token)firstRange.left) <= 0) {
            return true;
        }
        for (int i = 0; i < sortedRanges.size(); ++i) {
            Range range = sortedRanges.get(i);
            if (((Token)range.right).isMinimum()) {
                return false;
            }
            DecoratedKey firstBeyondRange = sstable.firstKeyBeyond(((Token)range.right).maxKeyBound());
            if (firstBeyondRange == null) {
                return false;
            }
            if (i == sortedRanges.size() - 1) {
                return true;
            }
            Range<Token> nextRange = sortedRanges.get(i + 1);
            if (nextRange.contains(firstBeyondRange.token)) continue;
            return true;
        }
        return false;
    }

    private void doCleanupCompaction(ColumnFamilyStore cfs, Collection<SSTableReader> sstables, CounterId.OneShotRenewer renewer) throws IOException {
        assert (!cfs.isIndex());
        Keyspace keyspace = cfs.keyspace;
        Collection ranges = StorageService.instance.getLocalRanges(keyspace.getName());
        if (ranges.isEmpty()) {
            logger.info("Cleanup cannot run before a node has joined the ring");
            return;
        }
        boolean hasIndexes = cfs.indexManager.hasIndexes();
        CleanupStrategy cleanupStrategy = CleanupStrategy.get(cfs, ranges, renewer);
        for (SSTableReader sstable : sstables) {
            Set<SSTableReader> sstableAsSet = Collections.singleton(sstable);
            if (!hasIndexes && !((AbstractBounds)new Bounds<Token>(sstable.first.token, sstable.last.token)).intersects(ranges)) {
                cfs.replaceCompactedSSTables(sstableAsSet, Collections.emptyList(), OperationType.CLEANUP);
                continue;
            }
            if (!CompactionManager.needsCleanup(sstable, ranges)) {
                logger.debug("Skipping {} for cleanup; all rows should be kept", (Object)sstable);
                continue;
            }
            CompactionController controller = new CompactionController(cfs, sstableAsSet, CompactionManager.getDefaultGcBefore(cfs));
            long start = System.nanoTime();
            long totalkeysWritten = 0L;
            int expectedBloomFilterSize = Math.max(cfs.metadata.getIndexInterval(), (int)SSTableReader.getApproximateKeyCount(sstableAsSet, cfs.metadata));
            if (logger.isDebugEnabled()) {
                logger.debug("Expected bloom filter size : " + expectedBloomFilterSize);
            }
            logger.info("Cleaning up " + sstable);
            File compactionFileLocation = cfs.directories.getWriteableLocationAsFile(cfs.getExpectedCompactedFileSize(sstableAsSet, OperationType.CLEANUP));
            if (compactionFileLocation == null) {
                throw new IOException("disk full");
            }
            ICompactionScanner scanner = cleanupStrategy.getScanner(sstable, this.getRateLimiter());
            CleanupInfo ci = new CleanupInfo(sstable, scanner);
            this.metrics.beginCompaction(ci);
            SSTableWriter writer = CompactionManager.createWriter(cfs, compactionFileLocation, expectedBloomFilterSize, sstable);
            SSTableReader newSstable = null;
            try {
                while (scanner.hasNext()) {
                    AbstractCompactedRow compactedRow;
                    if (ci.isStopRequested()) {
                        throw new CompactionInterruptedException(ci.getCompactionInfo());
                    }
                    SSTableIdentityIterator row = (SSTableIdentityIterator)scanner.next();
                    if ((row = cleanupStrategy.cleanup(row)) == null || writer.append(compactedRow = controller.getCompactedRow(row)) == null) continue;
                    ++totalkeysWritten;
                }
                if (totalkeysWritten > 0L) {
                    newSstable = writer.closeAndOpenReader(sstable.maxDataAge);
                } else {
                    writer.abort();
                }
            }
            catch (Throwable e) {
                writer.abort();
                throw Throwables.propagate((Throwable)e);
            }
            finally {
                controller.close();
                scanner.close();
                this.metrics.finishCompaction(ci);
            }
            ArrayList<SSTableReader> results = new ArrayList<SSTableReader>(1);
            if (newSstable != null) {
                results.add(newSstable);
                String format = "Cleaned up to %s.  %,d to %,d (~%d%% of original) bytes for %,d keys.  Time: %,dms.";
                long dTime = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - start);
                long startsize = sstable.onDiskLength();
                long endsize = newSstable.onDiskLength();
                double ratio = (double)endsize / (double)startsize;
                logger.info(String.format(format, writer.getFilename(), startsize, endsize, (int)(ratio * 100.0), totalkeysWritten, dTime));
            }
            cfs.indexManager.flushIndexesBlocking();
            cfs.replaceCompactedSSTables(Arrays.asList(sstable), results, OperationType.CLEANUP);
        }
    }

    public static SSTableWriter createWriter(ColumnFamilyStore cfs, File compactionFileLocation, int expectedBloomFilterSize, SSTableReader sstable) {
        FileUtils.createDirectory(compactionFileLocation);
        return new SSTableWriter(cfs.getTempSSTablePath(compactionFileLocation), expectedBloomFilterSize, cfs.metadata, cfs.partitioner, SSTableMetadata.createCollector(Collections.singleton(sstable), cfs.metadata.comparator, sstable.getSSTableLevel()));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void doValidationCompaction(ColumnFamilyStore cfs, Validator validator) throws IOException {
        int gcBefore;
        Collection<SSTableReader> sstables;
        if (!cfs.isValid()) {
            return;
        }
        String snapshotName = validator.desc.sessionId.toString();
        boolean isSnapshotValidation = cfs.snapshotExists(snapshotName);
        if (isSnapshotValidation) {
            sstables = cfs.getSnapshotSSTableReader(snapshotName);
            gcBefore = cfs.gcBefore(cfs.getSnapshotCreationTime(snapshotName));
        } else {
            StorageService.instance.forceKeyspaceFlush(cfs.keyspace.getName(), cfs.name);
            sstables = cfs.markCurrentSSTablesReferenced();
            gcBefore = validator.gcBefore > 0 ? validator.gcBefore : CompactionManager.getDefaultGcBefore(cfs);
        }
        ValidationCompactionIterable ci = new ValidationCompactionIterable(cfs, sstables, validator.desc.range, gcBefore);
        Iterator iter = ci.iterator();
        this.metrics.beginCompaction(ci);
        try {
            validator.prepare(cfs);
            while (iter.hasNext()) {
                if (ci.isStopRequested()) {
                    throw new CompactionInterruptedException(ci.getCompactionInfo());
                }
                AbstractCompactedRow row = (AbstractCompactedRow)iter.next();
                validator.add(row);
            }
            validator.complete();
        }
        finally {
            iter.close();
            SSTableReader.releaseReferences(sstables);
            if (isSnapshotValidation) {
                cfs.clearSnapshot(snapshotName);
            }
            this.metrics.finishCompaction(ci);
        }
    }

    public Future<?> submitIndexBuild(final SecondaryIndexBuilder builder) {
        Runnable runnable = new Runnable(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                CompactionManager.this.metrics.beginCompaction(builder);
                try {
                    builder.build();
                }
                finally {
                    CompactionManager.this.metrics.finishCompaction(builder);
                }
            }
        };
        return this.executor.submit(runnable);
    }

    public Future<?> submitCacheWrite(final AutoSavingCache.Writer writer) {
        Runnable runnable = new Runnable(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                if (!AutoSavingCache.flushInProgress.add(writer.cacheType())) {
                    logger.debug("Cache flushing was already in progress: skipping {}", (Object)writer.getCompactionInfo());
                    return;
                }
                try {
                    CompactionManager.this.metrics.beginCompaction(writer);
                    try {
                        writer.saveCache();
                    }
                    finally {
                        CompactionManager.this.metrics.finishCompaction(writer);
                    }
                }
                finally {
                    AutoSavingCache.flushInProgress.remove((Object)writer.cacheType());
                }
            }
        };
        return this.executor.submit(runnable);
    }

    static int getDefaultGcBefore(ColumnFamilyStore cfs) {
        return cfs.isIndex() ? (int)(System.currentTimeMillis() / 1000L) : cfs.gcBefore(System.currentTimeMillis());
    }

    public int getActiveCompactions() {
        return CompactionMetrics.getCompactions().size();
    }

    @Override
    public List<Map<String, String>> getCompactions() {
        List<CompactionInfo.Holder> compactionHolders = CompactionMetrics.getCompactions();
        ArrayList<Map<String, String>> out = new ArrayList<Map<String, String>>(compactionHolders.size());
        for (CompactionInfo.Holder ci : compactionHolders) {
            out.add(ci.getCompactionInfo().asMap());
        }
        return out;
    }

    @Override
    public List<String> getCompactionSummary() {
        List<CompactionInfo.Holder> compactionHolders = CompactionMetrics.getCompactions();
        ArrayList<String> out = new ArrayList<String>(compactionHolders.size());
        for (CompactionInfo.Holder ci : compactionHolders) {
            out.add(ci.getCompactionInfo().toString());
        }
        return out;
    }

    @Override
    public TabularData getCompactionHistory() {
        try {
            return SystemKeyspace.getCompactionHistory();
        }
        catch (OpenDataException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public long getTotalBytesCompacted() {
        return this.metrics.bytesCompacted.count();
    }

    @Override
    public long getTotalCompactionsCompleted() {
        return this.metrics.totalCompactionsCompleted.count();
    }

    @Override
    public int getPendingTasks() {
        return (Integer)this.metrics.pendingTasks.value();
    }

    @Override
    public long getCompletedTasks() {
        return (Long)this.metrics.completedTasks.value();
    }

    @Override
    public void stopCompaction(String type) {
        OperationType operation = OperationType.valueOf(type);
        for (CompactionInfo.Holder holder : CompactionMetrics.getCompactions()) {
            if (holder.getCompactionInfo().getTaskType() != operation) continue;
            holder.stop();
        }
    }

    @Override
    public int getCoreCompactorThreads() {
        return this.executor.getCorePoolSize();
    }

    @Override
    public void setCoreCompactorThreads(int number) {
        this.executor.setCorePoolSize(number);
    }

    @Override
    public int getMaximumCompactorThreads() {
        return this.executor.getMaximumPoolSize();
    }

    @Override
    public void setMaximumCompactorThreads(int number) {
        this.executor.setMaximumPoolSize(number);
    }

    @Override
    public int getCoreValidationThreads() {
        return this.validationExecutor.getCorePoolSize();
    }

    @Override
    public void setCoreValidationThreads(int number) {
        this.validationExecutor.setCorePoolSize(number);
    }

    @Override
    public int getMaximumValidatorThreads() {
        return this.validationExecutor.getMaximumPoolSize();
    }

    @Override
    public void setMaximumValidatorThreads(int number) {
        this.validationExecutor.setMaximumPoolSize(number);
    }

    public void interruptCompactionFor(Iterable<CFMetaData> columnFamilies, boolean interruptValidation) {
        assert (columnFamilies != null);
        for (CompactionInfo.Holder compactionHolder : CompactionMetrics.getCompactions()) {
            CompactionInfo info = compactionHolder.getCompactionInfo();
            if (info.getTaskType() == OperationType.VALIDATION && !interruptValidation || !Iterables.contains(columnFamilies, (Object)info.getCFMetaData())) continue;
            compactionHolder.stop();
        }
    }

    static {
        isCompactionManager = new ThreadLocal<Boolean>(){

            @Override
            protected Boolean initialValue() {
                return false;
            }
        };
        instance = new CompactionManager();
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        try {
            mbs.registerMBean(instance, new ObjectName(MBEAN_OBJECT_NAME));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
        cacheCleanupExecutor = new CacheCleanupExecutor();
    }

    private static class CleanupInfo
    extends CompactionInfo.Holder {
        private final SSTableReader sstable;
        private final ICompactionScanner scanner;

        public CleanupInfo(SSTableReader sstable, ICompactionScanner scanner) {
            this.sstable = sstable;
            this.scanner = scanner;
        }

        @Override
        public CompactionInfo getCompactionInfo() {
            try {
                return new CompactionInfo(this.sstable.metadata, OperationType.CLEANUP, this.scanner.getCurrentPosition(), this.scanner.getLengthInBytes());
            }
            catch (Exception e) {
                throw new RuntimeException();
            }
        }
    }

    public static interface CompactionExecutorStatsCollector {
        public void beginCompaction(CompactionInfo.Holder var1);

        public void finishCompaction(CompactionInfo.Holder var1);
    }

    private static class CacheCleanupExecutor
    extends CompactionExecutor {
        public CacheCleanupExecutor() {
            super(1, "CacheCleanupExecutor");
        }
    }

    private static class ValidationExecutor
    extends CompactionExecutor {
        public ValidationExecutor() {
            super(1, Integer.MAX_VALUE, "ValidationExecutor", new SynchronousQueue<Runnable>());
        }
    }

    private static class CompactionExecutor
    extends JMXEnabledThreadPoolExecutor {
        protected CompactionExecutor(int minThreads, int maxThreads, String name, BlockingQueue<Runnable> queue) {
            super(minThreads, maxThreads, 60L, TimeUnit.SECONDS, queue, new NamedThreadFactory(name, 1), "internal");
        }

        private CompactionExecutor(int threadCount, String name) {
            this(threadCount, threadCount, name, new LinkedBlockingQueue<Runnable>());
        }

        public CompactionExecutor() {
            this(Math.max(1, DatabaseDescriptor.getConcurrentCompactors()), "CompactionExecutor");
        }

        @Override
        protected void beforeExecute(Thread t, Runnable r) {
            isCompactionManager.set(true);
            super.beforeExecute(t, r);
        }

        @Override
        public void afterExecute(Runnable r, Throwable t) {
            DebuggableThreadPoolExecutor.maybeResetTraceSessionWrapper(r);
            if (t == null) {
                t = DebuggableThreadPoolExecutor.extractThrowable(r);
            }
            if (t != null) {
                if (t instanceof CompactionInterruptedException) {
                    logger.info(t.getMessage());
                    logger.debug("Full interruption stack trace:", t);
                } else {
                    DebuggableThreadPoolExecutor.handleOrLog(t);
                }
            }
        }
    }

    private static class ValidationCompactionController
    extends CompactionController {
        public ValidationCompactionController(ColumnFamilyStore cfs, int gcBefore) {
            super(cfs, gcBefore);
        }

        @Override
        public boolean shouldPurge(DecoratedKey key, long delTimestamp) {
            return true;
        }
    }

    private static class ValidationCompactionIterable
    extends CompactionIterable {
        public ValidationCompactionIterable(ColumnFamilyStore cfs, Collection<SSTableReader> sstables, Range<Token> range, int gcBefore) {
            super(OperationType.VALIDATION, cfs.getCompactionStrategy().getScanners(sstables, range), new ValidationCompactionController(cfs, gcBefore));
        }
    }

    private static abstract class CleanupStrategy {
        private CleanupStrategy() {
        }

        public static CleanupStrategy get(ColumnFamilyStore cfs, Collection<Range<Token>> ranges, CounterId.OneShotRenewer renewer) {
            if (cfs.indexManager.hasIndexes() || cfs.metadata.getDefaultValidator().isCommutative()) {
                return new Full(cfs, ranges, renewer);
            }
            return new Bounded(cfs, ranges);
        }

        public abstract ICompactionScanner getScanner(SSTableReader var1, RateLimiter var2);

        public abstract SSTableIdentityIterator cleanup(SSTableIdentityIterator var1);

        private static final class Full
        extends CleanupStrategy {
            private final Collection<Range<Token>> ranges;
            private final ColumnFamilyStore cfs;
            private List<Column> indexedColumnsInRow;
            private final CounterId.OneShotRenewer renewer;

            public Full(ColumnFamilyStore cfs, Collection<Range<Token>> ranges, CounterId.OneShotRenewer renewer) {
                this.cfs = cfs;
                this.ranges = ranges;
                this.indexedColumnsInRow = null;
                this.renewer = renewer;
            }

            @Override
            public ICompactionScanner getScanner(SSTableReader sstable, RateLimiter limiter) {
                return sstable.getScanner(limiter);
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public SSTableIdentityIterator cleanup(SSTableIdentityIterator row) {
                if (Range.isInRanges(row.getKey().token, this.ranges)) {
                    return row;
                }
                this.cfs.invalidateCachedRow(row.getKey());
                if (this.indexedColumnsInRow != null) {
                    this.indexedColumnsInRow.clear();
                }
                while (row.hasNext()) {
                    OnDiskAtom column = row.next();
                    if (column instanceof CounterColumn) {
                        this.renewer.maybeRenew((CounterColumn)column);
                    }
                    if (!(column instanceof Column) || !this.cfs.indexManager.indexes((Column)column)) continue;
                    if (this.indexedColumnsInRow == null) {
                        this.indexedColumnsInRow = new ArrayList<Column>();
                    }
                    this.indexedColumnsInRow.add((Column)column);
                }
                if (this.indexedColumnsInRow != null && !this.indexedColumnsInRow.isEmpty()) {
                    Keyspace.switchLock.readLock().lock();
                    try {
                        this.cfs.indexManager.deleteFromIndexes(row.getKey(), this.indexedColumnsInRow);
                    }
                    finally {
                        Keyspace.switchLock.readLock().unlock();
                    }
                }
                return null;
            }
        }

        private static final class Bounded
        extends CleanupStrategy {
            private final Collection<Range<Token>> ranges;

            public Bounded(final ColumnFamilyStore cfs, Collection<Range<Token>> ranges) {
                this.ranges = ranges;
                cacheCleanupExecutor.submit(new Runnable(){

                    @Override
                    public void run() {
                        cfs.cleanupCache();
                    }
                });
            }

            @Override
            public ICompactionScanner getScanner(SSTableReader sstable, RateLimiter limiter) {
                return sstable.getScanner(this.ranges, limiter);
            }

            @Override
            public SSTableIdentityIterator cleanup(SSTableIdentityIterator row) {
                return row;
            }
        }
    }

    private static interface AllSSTablesOperation {
        public void perform(ColumnFamilyStore var1, Iterable<SSTableReader> var2) throws IOException;
    }

    class BackgroundCompactionTask
    implements Runnable {
        private final ColumnFamilyStore cfs;

        BackgroundCompactionTask(ColumnFamilyStore cfs) {
            this.cfs = cfs;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                logger.debug("Checking {}.{}", (Object)this.cfs.keyspace.getName(), (Object)this.cfs.name);
                if (!this.cfs.isValid()) {
                    logger.debug("Aborting compaction for dropped CF");
                    return;
                }
                AbstractCompactionStrategy strategy = this.cfs.getCompactionStrategy();
                AbstractCompactionTask task = strategy.getNextBackgroundTask(CompactionManager.getDefaultGcBefore(this.cfs));
                if (task == null) {
                    logger.debug("No tasks available");
                    return;
                }
                task.execute(CompactionManager.this.metrics);
            }
            finally {
                CompactionManager.this.compactingCF.remove((Object)this.cfs);
            }
            CompactionManager.this.submitBackground(this.cfs);
        }
    }
}

