/*
 * Decompiled with CFR 0.152.
 */
package alluxio.master.journal.ufs;

import alluxio.conf.PropertyKey;
import alluxio.conf.ServerConfiguration;
import alluxio.master.journal.ufs.UfsJournal;
import alluxio.master.journal.ufs.UfsJournalFile;
import alluxio.master.journal.ufs.UfsJournalSnapshot;
import alluxio.underfs.UnderFileSystem;
import alluxio.util.ThreadFactoryUtils;
import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.IOException;
import java.net.URI;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import javax.annotation.concurrent.ThreadSafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
final class UfsJournalGarbageCollector
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(UfsJournalGarbageCollector.class);
    private final ScheduledExecutorService mExecutor = Executors.newSingleThreadScheduledExecutor(ThreadFactoryUtils.build((String)"UfsJournalGarbageCollector-%d", (boolean)true));
    private final UfsJournal mJournal;
    private final UnderFileSystem mUfs;
    private ScheduledFuture<?> mGc;

    UfsJournalGarbageCollector(UfsJournal journal) {
        this.mJournal = (UfsJournal)Preconditions.checkNotNull((Object)journal, (Object)"journal");
        this.mUfs = this.mJournal.getUfs();
        this.mGc = this.mExecutor.scheduleAtFixedRate(this::gc, 1000L, ServerConfiguration.getMs(PropertyKey.MASTER_JOURNAL_GC_PERIOD_MS), TimeUnit.MILLISECONDS);
    }

    @Override
    public void close() {
        if (this.mGc != null) {
            this.mGc.cancel(true);
            this.mGc = null;
        }
        this.mExecutor.shutdown();
    }

    void gc() {
        UfsJournalSnapshot snapshot;
        try {
            snapshot = UfsJournalSnapshot.getSnapshot(this.mJournal);
        }
        catch (IOException e) {
            LOG.warn("Failed to get journal snapshot with error {}.", (Object)e.toString());
            return;
        }
        long checkpointSequenceNumber = 0L;
        List<UfsJournalFile> checkpoints = snapshot.getCheckpoints();
        if (!checkpoints.isEmpty()) {
            checkpointSequenceNumber = checkpoints.get(checkpoints.size() - 1).getEnd();
        }
        for (int i = 0; i < checkpoints.size() - 1; ++i) {
            if (i < checkpoints.size() - 2) {
                this.deleteNoException(checkpoints.get(i).getLocation());
            }
            this.gcFileIfStale(checkpoints.get(i), checkpointSequenceNumber);
        }
        for (UfsJournalFile log : snapshot.getLogs()) {
            this.gcFileIfStale(log, checkpointSequenceNumber);
        }
        for (UfsJournalFile tmpCheckpoint : snapshot.getTemporaryCheckpoints()) {
            this.gcFileIfStale(tmpCheckpoint, checkpointSequenceNumber);
        }
    }

    private void gcFileIfStale(UfsJournalFile file, long checkpointSequenceNumber) {
        long thresholdMs;
        Long lastModifiedTimeMs;
        if (file.getEnd() > checkpointSequenceNumber && !file.isTmpCheckpoint()) {
            return;
        }
        try {
            lastModifiedTimeMs = this.mUfs.getFileStatus(file.getLocation().toString()).getLastModifiedTime();
        }
        catch (IOException e) {
            LOG.warn("Failed to get the last modified time for {}.", (Object)file.getLocation());
            return;
        }
        if (lastModifiedTimeMs == null) {
            LOG.warn("Failed to get the last modified time for {}.", (Object)file.getLocation());
            return;
        }
        long l = thresholdMs = file.isTmpCheckpoint() ? ServerConfiguration.getMs(PropertyKey.MASTER_JOURNAL_TEMPORARY_FILE_GC_THRESHOLD_MS) : ServerConfiguration.getMs(PropertyKey.MASTER_JOURNAL_GC_THRESHOLD_MS);
        if (System.currentTimeMillis() - lastModifiedTimeMs > thresholdMs) {
            this.deleteNoException(file.getLocation());
        }
    }

    private void deleteNoException(URI location) {
        try {
            this.mUfs.deleteFile(location.toString());
            LOG.info("Garbage collected journal file {}.", (Object)location);
        }
        catch (IOException e) {
            LOG.warn("Failed to garbage collect journal file {}.", (Object)location);
        }
    }
}

