/*
 * Decompiled with CFR 0.152.
 */
package alluxio.master;

import alluxio.collections.ConcurrentHashSet;
import alluxio.conf.PropertyKey;
import alluxio.conf.ServerConfiguration;
import alluxio.exception.ExceptionMessage;
import alluxio.master.StateLockOptions;
import alluxio.resource.LockResource;
import alluxio.retry.RetryUtils;
import alluxio.util.ThreadFactoryUtils;
import alluxio.util.ThreadUtils;
import alluxio.util.logging.SamplingLogger;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StateLockManager {
    private static final Logger LOG = LoggerFactory.getLogger(StateLockManager.class);
    private static final SamplingLogger SAMPLING_LOG = new SamplingLogger(LOG, 30000L);
    private static final int READ_LOCK_COUNT_HIGH = 20000;
    private ReentrantReadWriteLock mStateLock = new ReentrantReadWriteLock(true);
    private Set<Thread> mSharedWaitersAndHolders;
    private ScheduledExecutorService mScheduler;
    private boolean mInterruptCycleEnabled;
    private long mInterruptCycleInterval;
    private Lock mInterruptCycleLock = new ReentrantLock(true);
    private volatile int mInterruptCycleRefCount = 0;
    private ScheduledFuture<?> mInterrupterFuture;
    private AtomicBoolean mInterruptCycleTicking = new AtomicBoolean(false);
    private long mForcedDurationMs;
    private long mExclusiveOnlyDeadlineMs = -1L;

    public StateLockManager() {
        this.mSharedWaitersAndHolders = new ConcurrentHashSet();
        this.mScheduler = Executors.newSingleThreadScheduledExecutor(ThreadFactoryUtils.build((String)"state-lock-manager-%d", (boolean)true));
        this.mInterruptCycleEnabled = ServerConfiguration.getBoolean(PropertyKey.MASTER_BACKUP_STATE_LOCK_INTERRUPT_CYCLE_ENABLED);
        this.mInterruptCycleInterval = ServerConfiguration.getMs(PropertyKey.MASTER_BACKUP_STATE_LOCK_INTERRUPT_CYCLE_INTERVAL);
        this.mForcedDurationMs = ServerConfiguration.getMs(PropertyKey.MASTER_BACKUP_STATE_LOCK_FORCED_DURATION);
        Preconditions.checkArgument((this.mInterruptCycleInterval > 0L ? 1 : 0) != 0, (Object)"Interrupt-cycle interval should be greater than 0.");
    }

    public void mastersStartedCallback() {
        if (this.mExclusiveOnlyDeadlineMs == -1L) {
            long exclusiveOnlyDurationMs = ServerConfiguration.getMs(PropertyKey.MASTER_BACKUP_STATE_LOCK_EXCLUSIVE_DURATION);
            this.mExclusiveOnlyDeadlineMs = System.currentTimeMillis() + exclusiveOnlyDurationMs;
            if (exclusiveOnlyDurationMs > 0L) {
                LOG.info("State-lock will remain in exclusive-only mode for {}ms until {}", (Object)exclusiveOnlyDurationMs, (Object)new Date(this.mExclusiveOnlyDeadlineMs).toString());
            }
        }
    }

    public LockResource lockShared() throws InterruptedException {
        long exclusiveOnlyRemainingMs;
        int readLockCount;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Thread-{} entered lockShared().", (Object)ThreadUtils.getCurrentThreadIdentifier());
        }
        if (LOG.isInfoEnabled() && (readLockCount = this.mStateLock.getReadLockCount()) > 20000) {
            SAMPLING_LOG.info("Read Lock Count Too High: {} {}", (Object)readLockCount, this.mSharedWaitersAndHolders);
        }
        if ((exclusiveOnlyRemainingMs = this.mExclusiveOnlyDeadlineMs - System.currentTimeMillis()) > 0L) {
            String safeModeMsg = String.format("Master still in exclusive-only phase (%dms remaining) for the state-lock. Please see documentation for %s.", exclusiveOnlyRemainingMs, "alluxio.master.backup.state.lock.exclusive.duration");
            throw new IllegalStateException(safeModeMsg);
        }
        this.mSharedWaitersAndHolders.add(Thread.currentThread());
        this.mStateLock.readLock().lockInterruptibly();
        return new LockResource((Lock)this.mStateLock.readLock(), false, false, () -> this.mSharedWaitersAndHolders.remove(Thread.currentThread()));
    }

    public LockResource lockExclusive(StateLockOptions lockOptions) throws TimeoutException, InterruptedException, IOException {
        return this.lockExclusive(lockOptions, null);
    }

    public LockResource lockExclusive(StateLockOptions lockOptions, RetryUtils.RunnableThrowsIOException beforeAttempt) throws TimeoutException, InterruptedException, IOException {
        LOG.debug("Thread-{} entered lockExclusive().", (Object)ThreadUtils.getCurrentThreadIdentifier());
        StateLockOptions.GraceMode graceMode = lockOptions.getGraceMode();
        boolean graceCycleEntered = false;
        boolean lockAcquired = false;
        long deadlineMs = System.currentTimeMillis() + lockOptions.getGraceCycleTimeoutMs();
        while (System.currentTimeMillis() < deadlineMs) {
            if (!graceCycleEntered) {
                graceCycleEntered = true;
                LOG.info("Thread-{} entered grace-cycle of try-sleep: {}ms-{}ms for the total of {}ms", new Object[]{ThreadUtils.getCurrentThreadIdentifier(), lockOptions.getGraceCycleTryMs(), lockOptions.getGraceCycleSleepMs(), lockOptions.getGraceCycleTimeoutMs()});
            }
            if (beforeAttempt != null) {
                beforeAttempt.run();
            }
            if (this.mStateLock.writeLock().tryLock(lockOptions.getGraceCycleTryMs(), TimeUnit.MILLISECONDS)) {
                lockAcquired = true;
                break;
            }
            long remainingWaitMs = deadlineMs - System.currentTimeMillis();
            if (remainingWaitMs <= 0L) continue;
            Thread.sleep(Math.min(lockOptions.getGraceCycleSleepMs(), remainingWaitMs));
        }
        if (lockAcquired) {
            LOG.info("Thread-{} acquired the lock within grace-cycle.", (Object)ThreadUtils.getCurrentThreadIdentifier());
            this.activateInterruptCycle();
        } else {
            if (graceMode == StateLockOptions.GraceMode.TIMEOUT) {
                throw new TimeoutException(ExceptionMessage.STATE_LOCK_TIMED_OUT.getMessage(new Object[]{lockOptions.getGraceCycleTimeoutMs()}));
            }
            this.activateInterruptCycle();
            LOG.info("Thread-{} forcing the lock with {} waiters/holders: {}", new Object[]{ThreadUtils.getCurrentThreadIdentifier(), this.mSharedWaitersAndHolders.size(), this.mSharedWaitersAndHolders.stream().map(th -> Long.toString(th.getId())).collect(Collectors.joining(","))});
            try {
                if (beforeAttempt != null) {
                    beforeAttempt.run();
                }
                if (!this.mStateLock.writeLock().tryLock(this.mForcedDurationMs, TimeUnit.MILLISECONDS)) {
                    throw new TimeoutException(ExceptionMessage.STATE_LOCK_TIMED_OUT.getMessage(new Object[]{lockOptions.getGraceCycleTimeoutMs() + this.mForcedDurationMs}));
                }
            }
            catch (Throwable throwable) {
                this.deactivateInterruptCycle();
                throw throwable;
            }
        }
        return new LockResource((Lock)this.mStateLock.writeLock(), false, false, () -> this.deactivateInterruptCycle());
    }

    public List<String> getSharedWaitersAndHolders() {
        ArrayList<String> result = new ArrayList<String>();
        for (Thread waiterOrHolder : this.mSharedWaitersAndHolders) {
            result.add(ThreadUtils.getThreadIdentifier((Thread)waiterOrHolder));
        }
        return result;
    }

    public boolean interruptCycleTicking() {
        return this.mInterruptCycleTicking.get();
    }

    private void activateInterruptCycle() {
        if (!this.mInterruptCycleEnabled) {
            return;
        }
        try (LockResource lr = new LockResource(this.mInterruptCycleLock);){
            if (this.mInterruptCycleRefCount++ > 0) {
                return;
            }
            LOG.info("Interrupt cycle activated.");
            this.mInterrupterFuture = this.mScheduler.scheduleAtFixedRate(this::waiterInterruptRoutine, this.mInterruptCycleInterval, this.mInterruptCycleInterval, TimeUnit.MILLISECONDS);
        }
    }

    private void deactivateInterruptCycle() {
        if (!this.mInterruptCycleEnabled) {
            return;
        }
        try (LockResource lr = new LockResource(this.mInterruptCycleLock);){
            Preconditions.checkArgument((this.mInterruptCycleRefCount > 0 ? 1 : 0) != 0);
            if (--this.mInterruptCycleRefCount > 0) {
                return;
            }
            this.mInterrupterFuture.cancel(true);
            this.mInterruptCycleTicking.set(false);
            LOG.info("Interrupt cycle deactivated.");
            this.mInterrupterFuture = null;
        }
    }

    private void waiterInterruptRoutine() {
        this.mInterruptCycleTicking.set(true);
        ArrayList<Thread> interruptedThreads = new ArrayList<Thread>(this.mSharedWaitersAndHolders.size());
        for (Thread th2 : this.mSharedWaitersAndHolders) {
            th2.interrupt();
            interruptedThreads.add(th2);
        }
        LOG.info("Interrupt-cycle interrupted {} waiters/holders: {}", (Object)interruptedThreads.size(), (Object)interruptedThreads.stream().map(th -> ThreadUtils.getThreadIdentifier((Thread)th)).collect(Collectors.joining(",")));
    }
}

