/*
	* Copyright (C) 2020-2021 Sebastiano Vigna
	*
	* Licensed under the Apache License, Version 2.0 (the "License");
	* you may not use this file except in compliance with the License.
	* You may obtain a copy of the License at
	*
	*     http://www.apache.org/licenses/LICENSE-2.0
	*
	* Unless required by applicable law or agreed to in writing, software
	* distributed under the License is distributed on an "AS IS" BASIS,
	* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
	* See the License for the specific language governing permissions and
	* limitations under the License.
	*/
package it.unimi.dsi.fastutil.doubles;
/**
 * A type-specific {@link it.unimi.dsi.fastutil.Pair Pair}; provides some
 * additional methods that use polymorphism to avoid (un)boxing.
 */
public interface DoubleObjectPair<V> extends it.unimi.dsi.fastutil.Pair<Double, V> {
	/**
	 * Returns the left element of this pair.
	 *
	 * @return the left element of this pair.
	 */
	public double leftDouble();
	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@SuppressWarnings("deprecation")
	@Deprecated
	@Override
	public default Double left() {
		return Double.valueOf(leftDouble());
	}
	/**
	 * Sets the left element of this pair (optional operation).
	 *
	 * @param l
	 *            a new value for the left element.
	 *
	 * @implSpec This implementation throws an
	 *           {@link UnsupportedOperationException}.
	 */
	public default DoubleObjectPair<V> left(final double l) {
		throw new UnsupportedOperationException();
	}
	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@SuppressWarnings("deprecation")
	@Deprecated
	@Override
	public default DoubleObjectPair<V> left(final Double l) {
		return left((l).doubleValue());
	}
	/**
	 * Returns the left element of this pair.
	 *
	 * @return the left element of this pair.
	 *
	 * @implSpec This implementation delegates to {@link #left()}.
	 *
	 */
	public default double firstDouble() {
		return leftDouble();
	}
	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@SuppressWarnings("deprecation")
	@Deprecated
	@Override
	public default Double first() {
		return Double.valueOf(firstDouble());
	}
	/**
	 * Sets the left element of this pair (optional operation).
	 *
	 * @param l
	 *            a new value for the left element.
	 *
	 * @implSpec This implementation delegates to {@link #left(Object)}.
	 */
	public default DoubleObjectPair<V> first(final double l) {
		return left(l);
	}
	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@SuppressWarnings("deprecation")
	@Deprecated
	@Override
	public default DoubleObjectPair<V> first(final Double l) {
		return first((l).doubleValue());
	}
	/**
	 * Returns the left element of this pair.
	 *
	 * @return the left element of this pair.
	 *
	 * @implSpec This implementation delegates to {@link #left()}.
	 *
	 */
	public default double keyDouble() {
		return firstDouble();
	}
	/**
	 * {@inheritDoc}
	 * 
	 * @deprecated Please use the corresponding type-specific method instead.
	 */
	@SuppressWarnings("deprecation")
	@Deprecated
	@Override
	public default Double key() {
		return Double.valueOf(keyDouble());
	}
	/**
	 * Sets the left element of this pair (optional operation).
	 *
	 * @param l
	 *            a new value for the left element.
	 *
	 * @implSpec This implementation delegates to {@link #left(Object)}.
	 */
	public default DoubleObjectPair<V> key(final double l) {
		return left(l);
	}
	@Override
	@Deprecated
	public default DoubleObjectPair<V> key(Double l) {
		return key((l).doubleValue());
	}
	/**
	 * Returns a new type-specific immutable {@link it.unimi.dsi.fastutil.Pair Pair}
	 * with given left and right value.
	 * 
	 * @param left
	 *            the left value.
	 * @param right
	 *            the right value.
	 */
	public static <V> DoubleObjectPair<V> of(final double left, final V right) {
		return new DoubleObjectImmutablePair<V>(left, right);
	}
}
