/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.reactivex.core.http;

import java.util.Map;
import io.reactivex.Observable;
import io.reactivex.Flowable;
import io.reactivex.Single;
import io.reactivex.Completable;
import io.reactivex.Maybe;
import io.vertx.reactivex.core.MultiMap;
import io.vertx.reactivex.core.buffer.Buffer;
import io.vertx.core.http.HttpVersion;
import io.vertx.reactivex.core.streams.WriteStream;
import io.vertx.core.http.HttpMethod;
import io.vertx.reactivex.core.streams.ReadStream;
import io.vertx.core.Handler;

/**
 * Represents a client-side HTTP request.
 * <p>
 * Instances are created by an {@link io.vertx.reactivex.core.http.HttpClient} instance, via one of the methods corresponding to the
 * specific HTTP methods, or the generic request methods. On creation the request will not have been written to the
 * wire.
 * <p>
 * Once a request has been obtained, headers can be set on it, and data can be written to its body if required. Once
 * you are ready to send the request, one of the {@link io.vertx.reactivex.core.http.HttpClientRequest#end} methods should be called.
 * <p>
 * Nothing is actually sent until the request has been internally assigned an HTTP connection.
 * <p>
 * The {@link io.vertx.reactivex.core.http.HttpClient} instance will return an instance of this class immediately, even if there are no HTTP
 * connections available in the pool. Any requests sent before a connection is assigned will be queued
 * internally and actually sent when an HTTP connection becomes available from the pool.
 * <p>
 * The headers of the request are queued for writing either when the {@link io.vertx.reactivex.core.http.HttpClientRequest#end} method is called, or, when the first
 * part of the body is written, whichever occurs first.
 * <p>
 * This class supports both chunked and non-chunked HTTP.
 * <p>
 * It implements {@link io.vertx.reactivex.core.streams.WriteStream} so it can be used with
 * {@link io.vertx.reactivex.core.streams.Pump} to pump data with flow control.
 * <p>
 * An example of using this class is as follows:
 * <p>
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.http.HttpClientRequest original} non RX-ified interface using Vert.x codegen.
 */

@io.vertx.lang.reactivex.RxGen(io.vertx.core.http.HttpClientRequest.class)
public class HttpClientRequest implements WriteStream<Buffer>,  ReadStream<HttpClientResponse> {

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    HttpClientRequest that = (HttpClientRequest) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  public static final io.vertx.lang.reactivex.TypeArg<HttpClientRequest> __TYPE_ARG = new io.vertx.lang.reactivex.TypeArg<>(
    obj -> new HttpClientRequest((io.vertx.core.http.HttpClientRequest) obj),
    HttpClientRequest::getDelegate
  );

  private final io.vertx.core.http.HttpClientRequest delegate;
  
  public HttpClientRequest(io.vertx.core.http.HttpClientRequest delegate) {
    this.delegate = delegate;
  }

  public io.vertx.core.http.HttpClientRequest getDelegate() {
    return delegate;
  }

  private io.reactivex.Observable<HttpClientResponse> observable;
  private io.reactivex.Flowable<HttpClientResponse> flowable;

  public synchronized io.reactivex.Observable<HttpClientResponse> toObservable() {
    if (observable == null) {
      java.util.function.Function<io.vertx.core.http.HttpClientResponse, HttpClientResponse> conv = HttpClientResponse::newInstance;
      observable = io.vertx.reactivex.ObservableHelper.toObservable(delegate, conv);
    }
    return observable;
  }

  public synchronized io.reactivex.Flowable<HttpClientResponse> toFlowable() {
    if (flowable == null) {
      java.util.function.Function<io.vertx.core.http.HttpClientResponse, HttpClientResponse> conv = HttpClientResponse::newInstance;
      flowable = io.vertx.reactivex.FlowableHelper.toFlowable(delegate, conv);
    }
    return flowable;
  }

  /**
   * This will return <code>true</code> if there are more bytes in the write queue than the value set using {@link io.vertx.reactivex.core.http.HttpClientRequest#setWriteQueueMaxSize}
   * @return true if write queue is full
   */
  public boolean writeQueueFull() { 
    boolean ret = delegate.writeQueueFull();
    return ret;
  }

  public HttpClientRequest exceptionHandler(Handler<Throwable> handler) { 
    delegate.exceptionHandler(handler);
    return this;
  }

  /**
   * @param data 
   * @return 
   */
  public HttpClientRequest write(Buffer data) { 
    delegate.write(data.getDelegate());
    return this;
  }

  public HttpClientRequest setWriteQueueMaxSize(int maxSize) { 
    delegate.setWriteQueueMaxSize(maxSize);
    return this;
  }

  public HttpClientRequest drainHandler(Handler<Void> handler) { 
    delegate.drainHandler(handler);
    return this;
  }

  public HttpClientRequest handler(Handler<HttpClientResponse> handler) { 
    delegate.handler(new Handler<io.vertx.core.http.HttpClientResponse>() {
      public void handle(io.vertx.core.http.HttpClientResponse event) {
        handler.handle(HttpClientResponse.newInstance(event));
      }
    });
    return this;
  }

  public HttpClientRequest pause() { 
    delegate.pause();
    return this;
  }

  public HttpClientRequest resume() { 
    delegate.resume();
    return this;
  }

  public HttpClientRequest endHandler(Handler<Void> endHandler) { 
    delegate.endHandler(endHandler);
    return this;
  }

  public HttpClientRequest setFollowRedirects(boolean followRedirects) { 
    delegate.setFollowRedirects(followRedirects);
    return this;
  }

  /**
   * If chunked is true then the request will be set into HTTP chunked mode
   * @param chunked true if chunked encoding
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClientRequest setChunked(boolean chunked) { 
    delegate.setChunked(chunked);
    return this;
  }

  /**
   * @return Is the request chunked?
   */
  public boolean isChunked() { 
    boolean ret = delegate.isChunked();
    return ret;
  }

  /**
   * The HTTP method for the request.
   * @return 
   */
  public HttpMethod method() { 
    HttpMethod ret = delegate.method();
    return ret;
  }

  /**
   * @return the raw value of the method this request sends
   */
  public String getRawMethod() { 
    String ret = delegate.getRawMethod();
    return ret;
  }

  /**
   * Set the value the method to send when the method  is used.
   * @param method the raw method
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClientRequest setRawMethod(String method) { 
    delegate.setRawMethod(method);
    return this;
  }

  /**
   * @return the absolute URI corresponding to the the HTTP request
   */
  public String absoluteURI() { 
    String ret = delegate.absoluteURI();
    return ret;
  }

  /**
   * @return The URI of the request.
   */
  public String uri() { 
    String ret = delegate.uri();
    return ret;
  }

  /**
   * @return The path part of the uri. For example /somepath/somemorepath/someresource.foo
   */
  public String path() { 
    String ret = delegate.path();
    return ret;
  }

  /**
   * @return the query part of the uri. For example someparam=32&amp;someotherparam=x
   */
  public String query() { 
    String ret = delegate.query();
    return ret;
  }

  /**
   * Set the request host.<p/>
   *
   * For HTTP/2 it sets the  pseudo header otherwise it sets the  header
   * @param host 
   * @return 
   */
  public HttpClientRequest setHost(String host) { 
    delegate.setHost(host);
    return this;
  }

  /**
   * @return the request host. For HTTP/2 it returns the  pseudo header otherwise it returns the  header
   */
  public String getHost() { 
    String ret = delegate.getHost();
    return ret;
  }

  /**
   * @return The HTTP headers
   */
  public MultiMap headers() { 
    if (cached_0 != null) {
      return cached_0;
    }
    MultiMap ret = MultiMap.newInstance(delegate.headers());
    cached_0 = ret;
    return ret;
  }

  /**
   * Put an HTTP header
   * @param name The header name
   * @param value The header value
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClientRequest putHeader(String name, String value) { 
    delegate.putHeader(name, value);
    return this;
  }

  /**
   * Write a {@link java.lang.String} to the request body, encoded as UTF-8.
   * @param chunk 
   * @return @return a reference to this, so the API can be used fluently
   */
  public HttpClientRequest write(String chunk) { 
    delegate.write(chunk);
    return this;
  }

  /**
   * Write a {@link java.lang.String} to the request body, encoded using the encoding <code>enc</code>.
   * @param chunk 
   * @param enc 
   * @return @return a reference to this, so the API can be used fluently
   */
  public HttpClientRequest write(String chunk, String enc) { 
    delegate.write(chunk, enc);
    return this;
  }

  /**
   * If you send an HTTP request with the header <code>Expect</code> set to the value <code>100-continue</code>
   * and the server responds with an interim HTTP response with a status code of <code>100</code> and a continue handler
   * has been set using this method, then the <code>handler</code> will be called.
   * <p>
   * You can then continue to write data to the request body and later end it. This is normally used in conjunction with
   * the {@link io.vertx.reactivex.core.http.HttpClientRequest#sendHead} method to force the request header to be written before the request has ended.
   * @param handler 
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClientRequest continueHandler(Handler<Void> handler) { 
    delegate.continueHandler(handler);
    return this;
  }

  /**
   * Forces the head of the request to be written before {@link io.vertx.reactivex.core.http.HttpClientRequest#end} is called on the request or any data is
   * written to it.
   * <p>
   * This is normally used to implement HTTP 100-continue handling, see  for
   * more information.
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClientRequest sendHead() { 
    delegate.sendHead();
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpClientRequest#sendHead} but with an handler after headers have been sent. The handler will be called with
   * the {@link io.vertx.reactivex.core.http.HttpVersion} if it can be determined or null otherwise.<p>
   * @param completionHandler 
   * @return 
   */
  public HttpClientRequest sendHead(Handler<HttpVersion> completionHandler) { 
    delegate.sendHead(completionHandler);
    return this;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but writes a String in UTF-8 encoding
   * @param chunk 
   */
  public void end(String chunk) { 
    delegate.end(chunk);
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but writes a String with the specified encoding
   * @param chunk 
   * @param enc 
   */
  public void end(String chunk, String enc) { 
    delegate.end(chunk, enc);
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but writes some data to the request body before ending. If the request is not chunked and
   * no other data has been written then the <code>Content-Length</code> header will be automatically set
   * @param chunk 
   */
  public void end(Buffer chunk) { 
    delegate.end(chunk.getDelegate());
  }

  /**
   * Ends the request. If no data has been written to the request body, and {@link io.vertx.reactivex.core.http.HttpClientRequest#sendHead} has not been called then
   * the actual request won't get written until this method gets called.
   * <p>
   * Once the request has ended, it cannot be used any more,
   */
  public void end() { 
    delegate.end();
  }

  /**
   * Set's the amount of time after which if the request does not return any data within the timeout period an
   * {@link java.util.concurrent.TimeoutException} will be passed to the exception handler (if provided) and
   * the request will be closed.
   * <p>
   * Calling this method more than once has the effect of canceling any existing timeout and starting
   * the timeout from scratch.
   * @param timeoutMs The quantity of time in milliseconds.
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClientRequest setTimeout(long timeoutMs) { 
    delegate.setTimeout(timeoutMs);
    return this;
  }

  /**
   * Set a push handler for this request.<p/>
   *
   * The handler is called when the client receives a <i>push promise</i> from the server. The handler can be called
   * multiple times, for each push promise.<p/>
   *
   * The handler is called with a <i>read-only</i> {@link io.vertx.reactivex.core.http.HttpClientRequest}, the following methods can be called:<p/>
   *
   * <ul>
   *   <li>{@link io.vertx.reactivex.core.http.HttpClientRequest#method}</li>
   *   <li>{@link io.vertx.reactivex.core.http.HttpClientRequest#uri}</li>
   *   <li>{@link io.vertx.reactivex.core.http.HttpClientRequest#headers}</li>
   *   <li>{@link io.vertx.reactivex.core.http.HttpClientRequest#getHost}</li>
   * </ul>
   *
   * In addition the handler should call the {@link io.vertx.reactivex.core.http.HttpClientRequest#handler} method to set an handler to
   * process the response.<p/>
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClientRequest pushHandler(Handler<HttpClientRequest> handler) { 
    delegate.pushHandler(new Handler<io.vertx.core.http.HttpClientRequest>() {
      public void handle(io.vertx.core.http.HttpClientRequest event) {
        handler.handle(HttpClientRequest.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Reset this stream with the error code <code>0</code>.
   * @return 
   */
  public boolean reset() { 
    boolean ret = delegate.reset();
    return ret;
  }

  /**
   * Reset this request:
   * <p/>
   * <ul>
   *   <li>for HTTP/2, this performs send an HTTP/2 reset frame with the specified error <code>code</code></li>
   *   <li>for HTTP/1.x, this closes the connection after the current in-flight requests are ended</li>
   * </ul>
   * <p/>
   * When the request has not yet been sent, the request will be aborted and false is returned as indicator.
   * <p/>
   * @param code the error code
   * @return true when reset has been performed
   */
  public boolean reset(long code) { 
    boolean ret = delegate.reset(code);
    return ret;
  }

  /**
   * @return the {@link io.vertx.reactivex.core.http.HttpConnection} associated with this request
   */
  public HttpConnection connection() { 
    if (cached_1 != null) {
      return cached_1;
    }
    HttpConnection ret = HttpConnection.newInstance(delegate.connection());
    cached_1 = ret;
    return ret;
  }

  /**
   * Set a connection handler called when an HTTP connection has been established.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClientRequest connectionHandler(Handler<HttpConnection> handler) { 
    delegate.connectionHandler(new Handler<io.vertx.core.http.HttpConnection>() {
      public void handle(io.vertx.core.http.HttpConnection event) {
        handler.handle(HttpConnection.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Write an HTTP/2 frame to the request, allowing to extend the HTTP/2 protocol.<p>
   *
   * The frame is sent immediatly and is not subject to flow control.<p>
   *
   * This method must be called after the request headers have been sent and only for the protocol HTTP/2.
   * The {@link io.vertx.reactivex.core.http.HttpClientRequest#sendHead} should be used for this purpose.
   * @param type the 8-bit frame type
   * @param flags the 8-bit frame flags
   * @param payload the frame payload
   * @return a reference to this, so the API can be used fluently
   */
  public HttpClientRequest writeCustomFrame(int type, int flags, Buffer payload) { 
    delegate.writeCustomFrame(type, flags, payload.getDelegate());
    return this;
  }

  /**
   * @return the id of the stream of this response,  when it is not yet determined, i.e the request has not been yet sent or it is not supported HTTP/1.x
   */
  public int streamId() { 
    int ret = delegate.streamId();
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpClientRequest#writeCustomFrame} but with an {@link io.vertx.reactivex.core.http.HttpFrame}.
   * @param frame the frame to write
   * @return 
   */
  public HttpClientRequest writeCustomFrame(HttpFrame frame) { 
    delegate.writeCustomFrame(frame.getDelegate());
    return this;
  }

  private MultiMap cached_0;
  private HttpConnection cached_1;

  public static  HttpClientRequest newInstance(io.vertx.core.http.HttpClientRequest arg) {
    return arg != null ? new HttpClientRequest(arg) : null;
  }
}
