/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.rxjava.core.http;

import rx.Observable;
import rx.Single;
import io.vertx.rx.java.RxHelper;
import io.vertx.rx.java.WriteStreamSubscriber;
import io.vertx.rx.java.SingleOnSubscribeAdapter;
import java.util.Map;
import java.util.Set;
import java.util.List;
import java.util.Iterator;
import java.util.function.Function;
import java.util.stream.Collectors;
import io.vertx.core.Handler;
import io.vertx.core.AsyncResult;
import io.vertx.core.json.JsonObject;
import io.vertx.core.json.JsonArray;
import io.vertx.lang.rx.RxGen;
import io.vertx.lang.rx.TypeArg;
import io.vertx.lang.rx.MappingIterator;

/**
 * Represents a client-side HTTP request.
 * <p>
 * Instances are created by an {@link io.vertx.rxjava.core.http.HttpClient} instance, via one of the methods corresponding to the
 * specific HTTP methods, or the generic request methods. On creation the request will not have been written to the
 * wire.
 * <p>
 * Once a request has been obtained, headers can be set on it, and data can be written to its body if required. Once
 * you are ready to send the request, one of the {@link io.vertx.rxjava.core.http.HttpClientRequest#end} methods should be called.
 * <p>
 * Nothing is actually sent until the request has been internally assigned an HTTP connection.
 * <p>
 * The {@link io.vertx.rxjava.core.http.HttpClient} instance will return an instance of this class immediately, even if there are no HTTP
 * connections available in the pool. Any requests sent before a connection is assigned will be queued
 * internally and actually sent when an HTTP connection becomes available from the pool.
 * <p>
 * The headers of the request are queued for writing either when the {@link io.vertx.rxjava.core.http.HttpClientRequest#end} method is called, or, when the first
 * part of the body is written, whichever occurs first.
 * <p>
 * This class supports both chunked and non-chunked HTTP.
 * <p>
 * It implements {@link io.vertx.rxjava.core.streams.WriteStream} so it can be used with
 * {@link io.vertx.rxjava.core.streams.Pipe} to pipe data with flow control.
 * <p>
 * An example of using this class is as follows:
 * <p>
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.http.HttpClientRequest original} non RX-ified interface using Vert.x codegen.
 */

@RxGen(io.vertx.core.http.HttpClientRequest.class)
public class HttpClientRequest implements io.vertx.rxjava.core.streams.WriteStream<io.vertx.rxjava.core.buffer.Buffer> {

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    HttpClientRequest that = (HttpClientRequest) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  public static final TypeArg<HttpClientRequest> __TYPE_ARG = new TypeArg<>(    obj -> new HttpClientRequest((io.vertx.core.http.HttpClientRequest) obj),
    HttpClientRequest::getDelegate
  );

  private final io.vertx.core.http.HttpClientRequest delegate;
  
  public HttpClientRequest(io.vertx.core.http.HttpClientRequest delegate) {
    this.delegate = delegate;
  }

  public HttpClientRequest(Object delegate) {
    this.delegate = (io.vertx.core.http.HttpClientRequest)delegate;
  }

  public io.vertx.core.http.HttpClientRequest getDelegate() {
    return delegate;
  }

  private WriteStreamSubscriber<io.vertx.rxjava.core.buffer.Buffer> subscriber;

  public synchronized WriteStreamSubscriber<io.vertx.rxjava.core.buffer.Buffer> toSubscriber() {
    if (subscriber == null) {
      Function<io.vertx.rxjava.core.buffer.Buffer, io.vertx.core.buffer.Buffer> conv = io.vertx.rxjava.core.buffer.Buffer::getDelegate;
      subscriber = RxHelper.toSubscriber(getDelegate(), conv);
    }
    return subscriber;
  }

  /**
   * Same as  but with an <code>handler</code> called when the operation completes
   * @param data 
   * @param handler 
   */
  public void write(io.vertx.rxjava.core.buffer.Buffer data, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.write(data.getDelegate(), handler);
  }

  /**
   * Same as  but with an <code>handler</code> called when the operation completes
   * @param data 
   */
  public void write(io.vertx.rxjava.core.buffer.Buffer data) {
    write(data, ar -> { });
  }

    /**
   * Same as  but with an <code>handler</code> called when the operation completes
   * @param data 
   * @return 
   */
  public rx.Single<java.lang.Void> rxWrite(io.vertx.rxjava.core.buffer.Buffer data) { 
    return Single.create(new SingleOnSubscribeAdapter<>(fut -> {
      write(data, fut);
    }));
  }

  /**
   * This will return <code>true</code> if there are more bytes in the write queue than the value set using {@link io.vertx.rxjava.core.http.HttpClientRequest#setWriteQueueMaxSize}
   * @return <code>true</code> if write queue is full
   */
  public boolean writeQueueFull() { 
    boolean ret = delegate.writeQueueFull();
    return ret;
  }

  public io.vertx.rxjava.core.http.HttpClientRequest exceptionHandler(io.vertx.core.Handler<java.lang.Throwable> handler) { 
    delegate.exceptionHandler(handler);
    return this;
  }

  public io.vertx.rxjava.core.http.HttpClientRequest setWriteQueueMaxSize(int maxSize) { 
    delegate.setWriteQueueMaxSize(maxSize);
    return this;
  }

  public io.vertx.rxjava.core.http.HttpClientRequest drainHandler(io.vertx.core.Handler<java.lang.Void> handler) { 
    delegate.drainHandler(handler);
    return this;
  }

  /**
   * Set the host value of the HTTP/1.1 <code>host</code> header or HTTP/2 <code>authority</code> pseudo header
   * <p>The initial value is the same than the server socket address host.
   * <p>Keep in mind that changing this value won't change the actual server socket address for this request.
   * @param host the host part of the HTTP/1.1 <code>host</code> header or HTTP/2 <code>authority</code> pseudo header
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.rxjava.core.http.HttpClientRequest setHost(java.lang.String host) { 
    delegate.setHost(host);
    return this;
  }

  /**
   * @return the host value of the HTTP/1.1 <code>host</code> header or HTTP/2 <code>authority</code> pseudo header
   */
  public java.lang.String getHost() { 
    java.lang.String ret = delegate.getHost();
    return ret;
  }

  /**
   * Set the port value of the HTTP/1.1 <code>host</code> header or HTTP/2 <code>authority</code> pseudo header
   *
   * <p> Keep in mind that this won't change the actual server socket address for this request.
   * <p>The initial value is the same than the server socket address port.
   * @param port the port part of the HTTP/1.1 <code>host</code> header or HTTP/2 <code>authority</code> pseudo header
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.rxjava.core.http.HttpClientRequest setPort(int port) { 
    delegate.setPort(port);
    return this;
  }

  /**
   * @return the port value of the HTTP/1.1 <code>host</code> header or HTTP/2 <code>authority</code> pseudo header
   */
  public int getPort() { 
    int ret = delegate.getPort();
    return ret;
  }

  /**
   * Set the request to follow HTTP redirects up to {@link io.vertx.core.http.HttpClientOptions}.
   * @param followRedirects <code>true</code> to follow HTTP redirects
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.rxjava.core.http.HttpClientRequest setFollowRedirects(boolean followRedirects) { 
    delegate.setFollowRedirects(followRedirects);
    return this;
  }

  /**
   * Set the max number of HTTP redirects this request will follow. The default is <code>0</code> which means
   * no redirects.
   * @param maxRedirects the number of HTTP redirect to follow
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.rxjava.core.http.HttpClientRequest setMaxRedirects(int maxRedirects) { 
    delegate.setMaxRedirects(maxRedirects);
    return this;
  }

  /**
   * If chunked is true then the request will be set into HTTP chunked mode
   * @param chunked true if chunked encoding
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.rxjava.core.http.HttpClientRequest setChunked(boolean chunked) { 
    delegate.setChunked(chunked);
    return this;
  }

  /**
   * @return Is the request chunked?
   */
  public boolean isChunked() { 
    boolean ret = delegate.isChunked();
    return ret;
  }

  /**
   * The HTTP method for the request.
   * @return 
   */
  public io.vertx.core.http.HttpMethod getMethod() { 
    io.vertx.core.http.HttpMethod ret = delegate.getMethod();
    return ret;
  }

  /**
   * Set the HTTP method for this request.
   * @param method the HTTP method
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.rxjava.core.http.HttpClientRequest setMethod(io.vertx.core.http.HttpMethod method) { 
    delegate.setMethod(method);
    return this;
  }

  /**
   * @return the absolute URI corresponding to the HTTP request
   */
  public java.lang.String absoluteURI() { 
    java.lang.String ret = delegate.absoluteURI();
    return ret;
  }

  /**
   * @return The URI of the request.
   */
  public java.lang.String getURI() { 
    java.lang.String ret = delegate.getURI();
    return ret;
  }

  /**
   * Set the request uri.
   * @param uri the request uri
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.rxjava.core.http.HttpClientRequest setURI(java.lang.String uri) { 
    delegate.setURI(uri);
    return this;
  }

  /**
   * @return The path part of the uri. For example /somepath/somemorepath/someresource.foo
   */
  public java.lang.String path() { 
    java.lang.String ret = delegate.path();
    return ret;
  }

  /**
   * @return the query part of the uri. For example someparam=32&amp;someotherparam=x
   */
  public java.lang.String query() { 
    java.lang.String ret = delegate.query();
    return ret;
  }

  /**
   * @return The HTTP headers
   */
  public io.vertx.rxjava.core.MultiMap headers() { 
    if (cached_0 != null) {
      return cached_0;
    }
    io.vertx.rxjava.core.MultiMap ret = io.vertx.rxjava.core.MultiMap.newInstance((io.vertx.core.MultiMap)delegate.headers());
    cached_0 = ret;
    return ret;
  }

  /**
   * Put an HTTP header
   * @param name The header name
   * @param value The header value
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.rxjava.core.http.HttpClientRequest putHeader(java.lang.String name, java.lang.String value) { 
    delegate.putHeader(name, value);
    return this;
  }

  /**
   * @return the HTTP version for this request
   */
  public io.vertx.core.http.HttpVersion version() { 
    io.vertx.core.http.HttpVersion ret = delegate.version();
    return ret;
  }

  /**
   * Same as {@link io.vertx.rxjava.core.http.HttpClientRequest#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param handler 
   */
  public void write(java.lang.String chunk, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.write(chunk, handler);
  }

  /**
   * Same as {@link io.vertx.rxjava.core.http.HttpClientRequest#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   */
  public void write(java.lang.String chunk) {
    write(chunk, ar -> { });
  }

    /**
   * Same as {@link io.vertx.rxjava.core.http.HttpClientRequest#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @return 
   */
  public rx.Single<java.lang.Void> rxWrite(java.lang.String chunk) { 
    return Single.create(new SingleOnSubscribeAdapter<>(fut -> {
      write(chunk, fut);
    }));
  }

  /**
   * Same as {@link io.vertx.rxjava.core.http.HttpClientRequest#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   * @param handler 
   */
  public void write(java.lang.String chunk, java.lang.String enc, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.write(chunk, enc, handler);
  }

  /**
   * Same as {@link io.vertx.rxjava.core.http.HttpClientRequest#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   */
  public void write(java.lang.String chunk, java.lang.String enc) {
    write(chunk, enc, ar -> { });
  }

    /**
   * Same as {@link io.vertx.rxjava.core.http.HttpClientRequest#write} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   * @return 
   */
  public rx.Single<java.lang.Void> rxWrite(java.lang.String chunk, java.lang.String enc) { 
    return Single.create(new SingleOnSubscribeAdapter<>(fut -> {
      write(chunk, enc, fut);
    }));
  }

  /**
   * If you send an HTTP request with the header <code>Expect</code> set to the value <code>100-continue</code>
   * and the server responds with an interim HTTP response with a status code of <code>100</code> and a Continue handler
   * has been set using this method, then the <code>handler</code> will be called.
   * <p>
   * You can then continue to write data to the request body and later end it. This is normally used in conjunction with
   * the {@link io.vertx.rxjava.core.http.HttpClientRequest#sendHead} method to force the request header to be written before the request has ended.
   * @param handler 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.rxjava.core.http.HttpClientRequest continueHandler(io.vertx.core.Handler<java.lang.Void> handler) { 
    delegate.continueHandler(handler);
    return this;
  }

  /**
   * If the server responds with an interim HTTP response with a status code of <code>103</code> and a Early Hints handler
   * has been set using this method, then the <code>handler</code> will be called.
   * @param handler 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.rxjava.core.http.HttpClientRequest earlyHintsHandler(io.vertx.core.Handler<io.vertx.rxjava.core.MultiMap> handler) { 
    delegate.earlyHintsHandler(new io.vertx.lang.rx.DelegatingHandler<>(handler, event -> io.vertx.rxjava.core.MultiMap.newInstance((io.vertx.core.MultiMap)event)));
    return this;
  }

  /**
   * Like {@link io.vertx.rxjava.core.http.HttpClientRequest#sendHead} but with an handler after headers have been sent. The handler will be called with
   * the {@link io.vertx.core.http.HttpVersion} if it can be determined or null otherwise.<p>
   * @param completionHandler 
   * @return 
   */
  public io.vertx.rxjava.core.http.HttpClientRequest sendHead(io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> completionHandler) { 
    delegate.sendHead(completionHandler);
    return this;
  }

  /**
   * Like {@link io.vertx.rxjava.core.http.HttpClientRequest#sendHead} but with an handler after headers have been sent. The handler will be called with
   * the {@link io.vertx.core.http.HttpVersion} if it can be determined or null otherwise.<p>
   * @return 
   */
  public io.vertx.rxjava.core.http.HttpClientRequest sendHead() {
    return 
sendHead(ar -> { });
  }

    /**
   * Like {@link io.vertx.rxjava.core.http.HttpClientRequest#sendHead} but with an handler after headers have been sent. The handler will be called with
   * the {@link io.vertx.core.http.HttpVersion} if it can be determined or null otherwise.<p>
   * @return 
   */
  public rx.Single<java.lang.Void> rxSendHead() { 
    return Single.create(new SingleOnSubscribeAdapter<>(fut -> {
      sendHead(fut);
    }));
  }

  /**
   * Create an HTTP tunnel to the server.
   *
   * <p> Send HTTP request headers to the server, then configures the transport to exchange
   * raw buffers when the server replies with an appropriate response:
   *
   * <ul>
   *   <li><code>200</code> for HTTP <code>CONNECT</code> method</li>
   *   <li><code>101</code> for HTTP/1.1 <code>GET</code> with <code>Upgrade</code> <code>connection</code> header</li>
   * </ul>
   *
   * <p> The <code>handler</code> is called after response headers are received.
   *
   * <p> Use {@link io.vertx.rxjava.core.http.HttpClientResponse#netSocket} to get a  for interacting
   * more conveniently with the server.
   *
   * <p> HTTP/1.1 pipe-lined requests are not supported.f
   * @param handler the response completion handler
   */
  public void connect(io.vertx.core.Handler<io.vertx.core.AsyncResult<io.vertx.rxjava.core.http.HttpClientResponse>> handler) { 
    delegate.connect(new io.vertx.lang.rx.DelegatingHandler<>(handler, ar -> ar.map(event -> io.vertx.rxjava.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)event))));
  }

  /**
   * Create an HTTP tunnel to the server.
   *
   * <p> Send HTTP request headers to the server, then configures the transport to exchange
   * raw buffers when the server replies with an appropriate response:
   *
   * <ul>
   *   <li><code>200</code> for HTTP <code>CONNECT</code> method</li>
   *   <li><code>101</code> for HTTP/1.1 <code>GET</code> with <code>Upgrade</code> <code>connection</code> header</li>
   * </ul>
   *
   * <p> The <code>handler</code> is called after response headers are received.
   *
   * <p> Use {@link io.vertx.rxjava.core.http.HttpClientResponse#netSocket} to get a  for interacting
   * more conveniently with the server.
   *
   * <p> HTTP/1.1 pipe-lined requests are not supported.f
   */
  public void connect() {
    connect(ar -> { });
  }

    /**
   * Create an HTTP tunnel to the server.
   *
   * <p> Send HTTP request headers to the server, then configures the transport to exchange
   * raw buffers when the server replies with an appropriate response:
   *
   * <ul>
   *   <li><code>200</code> for HTTP <code>CONNECT</code> method</li>
   *   <li><code>101</code> for HTTP/1.1 <code>GET</code> with <code>Upgrade</code> <code>connection</code> header</li>
   * </ul>
   *
   * <p> The <code>handler</code> is called after response headers are received.
   *
   * <p> Use {@link io.vertx.rxjava.core.http.HttpClientResponse#netSocket} to get a  for interacting
   * more conveniently with the server.
   *
   * <p> HTTP/1.1 pipe-lined requests are not supported.f
   * @return 
   */
  public rx.Single<io.vertx.rxjava.core.http.HttpClientResponse> rxConnect() { 
    return Single.create(new SingleOnSubscribeAdapter<>(fut -> {
      connect(fut);
    }));
  }

  /**
   * Set a callback for the associated {@link io.vertx.rxjava.core.http.HttpClientResponse}.
   *
   * <p> This method does not modify the current request being sent.
   * @param handler the completion handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.rxjava.core.http.HttpClientRequest response(io.vertx.core.Handler<io.vertx.core.AsyncResult<io.vertx.rxjava.core.http.HttpClientResponse>> handler) { 
    delegate.response(new io.vertx.lang.rx.DelegatingHandler<>(handler, ar -> ar.map(event -> io.vertx.rxjava.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)event))));
    return this;
  }

  /**
   * Set a callback for the associated {@link io.vertx.rxjava.core.http.HttpClientResponse}.
   *
   * <p> This method does not modify the current request being sent.
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.rxjava.core.http.HttpClientRequest response() {
    return 
response(ar -> { });
  }

    /**
   * Set a callback for the associated {@link io.vertx.rxjava.core.http.HttpClientResponse}.
   *
   * <p> This method does not modify the current request being sent.
   * @return a reference to this, so the API can be used fluently
   */
  public rx.Single<io.vertx.rxjava.core.http.HttpClientResponse> rxResponse() { 
    return Single.create(new SingleOnSubscribeAdapter<>(fut -> {
      response(fut);
    }));
  }

  /**
   * Send the request with an empty body.
   * @param handler the completion handler for the {@link io.vertx.rxjava.core.http.HttpClientResponse}
   */
  public void send(io.vertx.core.Handler<io.vertx.core.AsyncResult<io.vertx.rxjava.core.http.HttpClientResponse>> handler) { 
    delegate.send(new io.vertx.lang.rx.DelegatingHandler<>(handler, ar -> ar.map(event -> io.vertx.rxjava.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)event))));
  }

  /**
   * Send the request with an empty body.
   */
  public void send() {
    send(ar -> { });
  }

    /**
   * Send the request with an empty body.
   * @return 
   */
  public rx.Single<io.vertx.rxjava.core.http.HttpClientResponse> rxSend() { 
    return Single.create(new SingleOnSubscribeAdapter<>(fut -> {
      send(fut);
    }));
  }

  /**
   * Send the request with a string <code>body</code>.
   * @param body 
   * @param handler the completion handler for the {@link io.vertx.rxjava.core.http.HttpClientResponse}
   */
  public void send(java.lang.String body, io.vertx.core.Handler<io.vertx.core.AsyncResult<io.vertx.rxjava.core.http.HttpClientResponse>> handler) { 
    delegate.send(body, new io.vertx.lang.rx.DelegatingHandler<>(handler, ar -> ar.map(event -> io.vertx.rxjava.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)event))));
  }

  /**
   * Send the request with a string <code>body</code>.
   * @param body 
   */
  public void send(java.lang.String body) {
    send(body, ar -> { });
  }

    /**
   * Send the request with a string <code>body</code>.
   * @param body 
   * @return 
   */
  public rx.Single<io.vertx.rxjava.core.http.HttpClientResponse> rxSend(java.lang.String body) { 
    return Single.create(new SingleOnSubscribeAdapter<>(fut -> {
      send(body, fut);
    }));
  }

  /**
   * Send the request with a buffer <code>body</code>.
   * @param body 
   * @param handler the completion handler for the {@link io.vertx.rxjava.core.http.HttpClientResponse}
   */
  public void send(io.vertx.rxjava.core.buffer.Buffer body, io.vertx.core.Handler<io.vertx.core.AsyncResult<io.vertx.rxjava.core.http.HttpClientResponse>> handler) { 
    delegate.send(body.getDelegate(), new io.vertx.lang.rx.DelegatingHandler<>(handler, ar -> ar.map(event -> io.vertx.rxjava.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)event))));
  }

  /**
   * Send the request with a buffer <code>body</code>.
   * @param body 
   */
  public void send(io.vertx.rxjava.core.buffer.Buffer body) {
    send(body, ar -> { });
  }

    /**
   * Send the request with a buffer <code>body</code>.
   * @param body 
   * @return 
   */
  public rx.Single<io.vertx.rxjava.core.http.HttpClientResponse> rxSend(io.vertx.rxjava.core.buffer.Buffer body) { 
    return Single.create(new SingleOnSubscribeAdapter<>(fut -> {
      send(body, fut);
    }));
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.rxjava.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.rxjava.core.http.HttpHeaders}.
   * @param body 
   * @param handler the completion handler for the {@link io.vertx.rxjava.core.http.HttpClientResponse}
   */
  public void send(io.vertx.rxjava.core.streams.ReadStream<io.vertx.rxjava.core.buffer.Buffer> body, io.vertx.core.Handler<io.vertx.core.AsyncResult<io.vertx.rxjava.core.http.HttpClientResponse>> handler) { 
    delegate.send(body.getDelegate(), new io.vertx.lang.rx.DelegatingHandler<>(handler, ar -> ar.map(event -> io.vertx.rxjava.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)event))));
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.rxjava.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.rxjava.core.http.HttpHeaders}.
   * @param body 
   */
  public void send(io.vertx.rxjava.core.streams.ReadStream<io.vertx.rxjava.core.buffer.Buffer> body) {
    send(body, ar -> { });
  }

    /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.rxjava.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.rxjava.core.http.HttpHeaders}.
   * @param body 
   * @return 
   */
  public rx.Single<io.vertx.rxjava.core.http.HttpClientResponse> rxSend(io.vertx.rxjava.core.streams.ReadStream<io.vertx.rxjava.core.buffer.Buffer> body) { 
    return Single.create(new SingleOnSubscribeAdapter<>(fut -> {
      send(body, fut);
    }));
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.rxjava.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.rxjava.core.http.HttpHeaders}.
   * @param body 
   * @param handler the completion handler for the {@link io.vertx.rxjava.core.http.HttpClientResponse}
   */
  public void send(rx.Observable<io.vertx.rxjava.core.buffer.Buffer> body, io.vertx.core.Handler<io.vertx.core.AsyncResult<io.vertx.rxjava.core.http.HttpClientResponse>> handler) { 
    delegate.send(io.vertx.rx.java.ReadStreamSubscriber.asReadStream(body,obj -> (io.vertx.core.buffer.Buffer)obj.getDelegate()).resume(), new io.vertx.lang.rx.DelegatingHandler<>(handler, ar -> ar.map(event -> io.vertx.rxjava.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)event))));
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.rxjava.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.rxjava.core.http.HttpHeaders}.
   * @param body 
   */
  public void send(rx.Observable<io.vertx.rxjava.core.buffer.Buffer> body) {
    send(body, ar -> { });
  }

    /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.rxjava.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.rxjava.core.http.HttpHeaders}.
   * @param body 
   * @return 
   */
  public rx.Single<io.vertx.rxjava.core.http.HttpClientResponse> rxSend(rx.Observable<io.vertx.rxjava.core.buffer.Buffer> body) { 
    return Single.create(new SingleOnSubscribeAdapter<>(fut -> {
      send(body, fut);
    }));
  }

  /**
   * Same as {@link io.vertx.rxjava.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param handler 
   */
  public void end(java.lang.String chunk, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.end(chunk, handler);
  }

  /**
   * Same as {@link io.vertx.rxjava.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   */
  public void end(java.lang.String chunk) {
    end(chunk, ar -> { });
  }

    /**
   * Same as {@link io.vertx.rxjava.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @return 
   */
  public rx.Single<java.lang.Void> rxEnd(java.lang.String chunk) { 
    return Single.create(new SingleOnSubscribeAdapter<>(fut -> {
      end(chunk, fut);
    }));
  }

  /**
   * Same as {@link io.vertx.rxjava.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   * @param handler 
   */
  public void end(java.lang.String chunk, java.lang.String enc, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.end(chunk, enc, handler);
  }

  /**
   * Same as {@link io.vertx.rxjava.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   */
  public void end(java.lang.String chunk, java.lang.String enc) {
    end(chunk, enc, ar -> { });
  }

    /**
   * Same as {@link io.vertx.rxjava.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param enc 
   * @return 
   */
  public rx.Single<java.lang.Void> rxEnd(java.lang.String chunk, java.lang.String enc) { 
    return Single.create(new SingleOnSubscribeAdapter<>(fut -> {
      end(chunk, enc, fut);
    }));
  }

  /**
   * Same as {@link io.vertx.rxjava.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @param handler 
   */
  public void end(io.vertx.rxjava.core.buffer.Buffer chunk, io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.end(chunk.getDelegate(), handler);
  }

  /**
   * Same as {@link io.vertx.rxjava.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   */
  public void end(io.vertx.rxjava.core.buffer.Buffer chunk) {
    end(chunk, ar -> { });
  }

    /**
   * Same as {@link io.vertx.rxjava.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param chunk 
   * @return 
   */
  public rx.Single<java.lang.Void> rxEnd(io.vertx.rxjava.core.buffer.Buffer chunk) { 
    return Single.create(new SingleOnSubscribeAdapter<>(fut -> {
      end(chunk, fut);
    }));
  }

  /**
   * Same as {@link io.vertx.rxjava.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @param handler 
   */
  public void end(io.vertx.core.Handler<io.vertx.core.AsyncResult<java.lang.Void>> handler) { 
    delegate.end(handler);
  }

  /**
   * Same as {@link io.vertx.rxjava.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   */
  public void end() {
    end(ar -> { });
  }

    /**
   * Same as {@link io.vertx.rxjava.core.http.HttpClientRequest#end} but with an <code>handler</code> called when the operation completes
   * @return 
   */
  public rx.Single<java.lang.Void> rxEnd() { 
    return Single.create(new SingleOnSubscribeAdapter<>(fut -> {
      end(fut);
    }));
  }

  /**
   * Set's the amount of time after which if the request does not return any data within the timeout period an
   * {@link java.util.concurrent.TimeoutException} will be passed to the exception handler (if provided) and
   * the request will be closed.
   * <p>
   * Calling this method more than once has the effect of canceling any existing timeout and starting
   * the timeout from scratch.
   * @param timeoutMs The quantity of time in milliseconds.
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.rxjava.core.http.HttpClientRequest setTimeout(long timeoutMs) { 
    delegate.setTimeout(timeoutMs);
    return this;
  }

  /**
   * Set a push handler for this request.<p/>
   *
   * The handler is called when the client receives a <i>push promise</i> from the server. The handler can be called
   * multiple times, for each push promise.<p/>
   *
   * The handler is called with a <i>read-only</i> {@link io.vertx.rxjava.core.http.HttpClientRequest}, the following methods can be called:<p/>
   *
   * <ul>
   *   <li>{@link io.vertx.rxjava.core.http.HttpClientRequest#getMethod}</li>
   *   <li>{@link io.vertx.rxjava.core.http.HttpClientRequest#getURI}</li>
   *   <li>{@link io.vertx.rxjava.core.http.HttpClientRequest#headers}</li>
   * </ul>
   *
   * In addition the handler should call the {@link io.vertx.rxjava.core.http.HttpClientRequest#response} method to set an handler to
   * process the response.<p/>
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.rxjava.core.http.HttpClientRequest pushHandler(io.vertx.core.Handler<io.vertx.rxjava.core.http.HttpClientRequest> handler) { 
    delegate.pushHandler(new io.vertx.lang.rx.DelegatingHandler<>(handler, event -> io.vertx.rxjava.core.http.HttpClientRequest.newInstance((io.vertx.core.http.HttpClientRequest)event)));
    return this;
  }

  /**
   * Reset this stream with the error code <code>0</code>.
   * @return 
   */
  public boolean reset() { 
    boolean ret = delegate.reset();
    return ret;
  }

  /**
   * Reset this request:
   * <p/>
   * <ul>
   *   <li>for HTTP/2, this performs send an HTTP/2 reset frame with the specified error <code>code</code></li>
   *   <li>for HTTP/1.x, this closes the connection when the current request is inflight</li>
   * </ul>
   * <p/>
   * When the request has not yet been sent, the request will be aborted and false is returned as indicator.
   * <p/>
   * @param code the error code
   * @return <code>true</code> when reset has been performed
   */
  public boolean reset(long code) { 
    boolean ret = delegate.reset(code);
    return ret;
  }

  /**
   * Reset this request:
   * <p/>
   * <ul>
   *   <li>for HTTP/2, send an HTTP/2 reset frame with the specified error <code>code</code></li>
   *   <li>for HTTP/1.x, close the connection when the current request is inflight</li>
   * </ul>
   * <p/>
   * When the request has not yet been sent, the request will be aborted and <code>false</code> is returned as indicator.
   * <p/>
   * @param code the error code
   * @param cause an optional cause that can be attached to the error code
   * @return true when reset has been performed
   */
  public boolean reset(long code, java.lang.Throwable cause) { 
    boolean ret = delegate.reset(code, cause);
    return ret;
  }

  /**
   * @return the {@link io.vertx.rxjava.core.http.HttpConnection} associated with this request
   */
  public io.vertx.rxjava.core.http.HttpConnection connection() { 
    if (cached_1 != null) {
      return cached_1;
    }
    io.vertx.rxjava.core.http.HttpConnection ret = io.vertx.rxjava.core.http.HttpConnection.newInstance((io.vertx.core.http.HttpConnection)delegate.connection());
    cached_1 = ret;
    return ret;
  }

  /**
   * Write an HTTP/2 frame to the request, allowing to extend the HTTP/2 protocol.<p>
   *
   * The frame is sent immediatly and is not subject to flow control.<p>
   *
   * This method must be called after the request headers have been sent and only for the protocol HTTP/2.
   * The {@link io.vertx.rxjava.core.http.HttpClientRequest#sendHead} should be used for this purpose.
   * @param type the 8-bit frame type
   * @param flags the 8-bit frame flags
   * @param payload the frame payload
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.rxjava.core.http.HttpClientRequest writeCustomFrame(int type, int flags, io.vertx.rxjava.core.buffer.Buffer payload) { 
    delegate.writeCustomFrame(type, flags, payload.getDelegate());
    return this;
  }

  /**
   * @return the id of the stream of this response,  when it is not yet determined, i.e the request has not been yet sent or it is not supported HTTP/1.x
   */
  public int streamId() { 
    int ret = delegate.streamId();
    return ret;
  }

  /**
   * Like {@link io.vertx.rxjava.core.http.HttpClientRequest#writeCustomFrame} but with an {@link io.vertx.rxjava.core.http.HttpFrame}.
   * @param frame the frame to write
   * @return 
   */
  public io.vertx.rxjava.core.http.HttpClientRequest writeCustomFrame(io.vertx.rxjava.core.http.HttpFrame frame) { 
    delegate.writeCustomFrame(frame.getDelegate());
    return this;
  }

  /**
   * Sets the priority of the associated stream.
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param streamPriority the priority of this request's stream
   * @return 
   */
  public io.vertx.rxjava.core.http.HttpClientRequest setStreamPriority(io.vertx.core.http.StreamPriority streamPriority) { 
    delegate.setStreamPriority(streamPriority);
    return this;
  }

  /**
   * @return the priority of the associated HTTP/2 stream for HTTP/2 otherwise <code>null</code>
   */
  public io.vertx.core.http.StreamPriority getStreamPriority() { 
    io.vertx.core.http.StreamPriority ret = delegate.getStreamPriority();
    return ret;
  }

  /**
   * Like {@link io.vertx.rxjava.core.http.HttpClientRequest#putHeader} but using CharSequence
   * @param name 
   * @param value 
   * @return 
   */
  public io.vertx.rxjava.core.http.HttpClientRequest putHeader(java.lang.CharSequence name, java.lang.CharSequence value) { 
    delegate.putHeader(name, value);
    return this;
  }

  /**
   * Put an HTTP header with multiple values
   * @param name The header name
   * @param values The header values
   * @return @return a reference to this, so the API can be used fluently
   */
  public io.vertx.rxjava.core.http.HttpClientRequest putHeader(java.lang.String name, java.lang.Iterable<java.lang.String> values) { 
    delegate.putHeader(name, values);
    return this;
  }

  /**
   * Like {@link io.vertx.rxjava.core.http.HttpClientRequest#putHeader} but using CharSequence
   * @param name 
   * @param values 
   * @return 
   */
  public io.vertx.rxjava.core.http.HttpClientRequest putHeader(java.lang.CharSequence name, java.lang.Iterable<java.lang.CharSequence> values) { 
    delegate.putHeader(name, values);
    return this;
  }

  private io.vertx.rxjava.core.MultiMap cached_0;
  private io.vertx.rxjava.core.http.HttpConnection cached_1;
  public static HttpClientRequest newInstance(io.vertx.core.http.HttpClientRequest arg) {
    return arg != null ? new HttpClientRequest(arg) : null;
  }

}
