/*
 * Decompiled with CFR 0.152.
 */
package io.temporal.internal.replay;

import com.google.common.annotations.VisibleForTesting;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.util.Timestamps;
import com.uber.m3.tally.Scope;
import com.uber.m3.util.Duration;
import io.temporal.api.command.v1.ContinueAsNewWorkflowExecutionCommandAttributes;
import io.temporal.api.common.v1.Payloads;
import io.temporal.api.history.v1.HistoryEvent;
import io.temporal.api.history.v1.WorkflowExecutionCancelRequestedEventAttributes;
import io.temporal.api.history.v1.WorkflowExecutionSignaledEventAttributes;
import io.temporal.api.protocol.v1.Message;
import io.temporal.api.query.v1.WorkflowQuery;
import io.temporal.api.update.v1.Input;
import io.temporal.api.update.v1.Request;
import io.temporal.failure.CanceledFailure;
import io.temporal.internal.common.FailureUtils;
import io.temporal.internal.common.ProtobufTimeUtils;
import io.temporal.internal.common.UpdateMessage;
import io.temporal.internal.replay.ReplayWorkflow;
import io.temporal.internal.replay.ReplayWorkflowContextImpl;
import io.temporal.internal.statemachines.WorkflowStateMachines;
import io.temporal.internal.sync.SignalHandlerInfo;
import io.temporal.internal.sync.UpdateHandlerInfo;
import io.temporal.internal.worker.WorkflowExecutionException;
import io.temporal.worker.NonDeterministicException;
import io.temporal.workflow.HandlerUnfinishedPolicy;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

final class ReplayWorkflowExecutor {
    @VisibleForTesting
    public static final String unfinishedUpdateHandlesWarnMessage = "[TMPRL1102] Workflow finished while update handlers are still running. This may have interrupted work that the update handler was doing, and the client that sent the update will receive a 'workflow execution already completed' Exception instead of the update result. You can wait for all update and signal handlers to complete by using `await workflow.Await(() -> workflow.isEveryHandlerFinished())`. Alternatively, if both you and the clients sending the update are okay with interrupting running handlers when the workflow finishes, and causing clients to receive errors, then you can disable this warning via the update handler annotations: `@UpdateMethod(unfinishedPolicy = HandlerUnfinishedPolicy.ABANDON)`.";
    @VisibleForTesting
    public static final String unfinishedSignalHandlesWarnMessage = "[TMPRL1102] Workflow finished while signal handlers are still running. This may have interrupted work that the signal handler was doing. You can wait for all update and signal handlers to complete by using `await workflow.Await(() -> workflow.isEveryHandlerFinished())`. Alternatively, if both you and the clients sending the signal are okay with interrupting running handlers when the workflow finishes you can disable this warning via the signal handler annotations: `@SignalMethod(unfinishedPolicy = HandlerUnfinishedPolicy.ABANDON)`.";
    private static final Logger log = LoggerFactory.getLogger(ReplayWorkflowExecutor.class);
    private final ReplayWorkflow workflow;
    private final WorkflowStateMachines workflowStateMachines;
    private final ReplayWorkflowContextImpl context;
    private final Scope metricsScope;

    public ReplayWorkflowExecutor(ReplayWorkflow workflow, WorkflowStateMachines workflowStateMachines, ReplayWorkflowContextImpl context) {
        this.workflow = workflow;
        this.workflowStateMachines = workflowStateMachines;
        this.context = context;
        this.metricsScope = context.getMetricsScope();
    }

    public void eventLoop() {
        boolean completed = this.context.isWorkflowMethodCompleted();
        if (completed) {
            return;
        }
        WorkflowExecutionException failure = null;
        try {
            completed = this.workflow.eventLoop();
        }
        catch (WorkflowExecutionException e) {
            failure = e;
            completed = true;
        }
        catch (CanceledFailure e) {
            if (!this.context.isCancelRequested()) {
                failure = new WorkflowExecutionException(this.workflow.getWorkflowContext().mapWorkflowExceptionToFailure(e));
            }
            completed = true;
        }
        if (completed) {
            this.context.setWorkflowMethodCompleted();
            this.completeWorkflow(failure);
        }
    }

    private void completeWorkflow(@Nullable WorkflowExecutionException failure) {
        if (log.isWarnEnabled() && (failure == null || this.context.isCancelRequested())) {
            Map<String, UpdateHandlerInfo> runningUpdateHandlers;
            List unfinishedUpdateHandlers;
            Map<Long, SignalHandlerInfo> runningSignalHandlers = this.workflow.getWorkflowContext().getRunningSignalHandlers();
            List unfinishedSignalHandlers = runningSignalHandlers.values().stream().filter(a -> a.getPolicy() == HandlerUnfinishedPolicy.WARN_AND_ABANDON).collect(Collectors.toList());
            if (!unfinishedSignalHandlers.isEmpty()) {
                MDC.put((String)"Signals", (String)unfinishedSignalHandlers.toString());
                log.warn(unfinishedSignalHandlesWarnMessage);
                MDC.remove((String)"Signals");
            }
            if (!(unfinishedUpdateHandlers = (runningUpdateHandlers = this.workflow.getWorkflowContext().getRunningUpdateHandlers()).values().stream().filter(a -> a.getPolicy() == HandlerUnfinishedPolicy.WARN_AND_ABANDON).collect(Collectors.toList())).isEmpty()) {
                MDC.put((String)"Updates", (String)unfinishedUpdateHandlers.toString());
                log.warn(unfinishedUpdateHandlesWarnMessage);
                MDC.remove((String)"Updates");
            }
        }
        if (this.context.isCancelRequested()) {
            this.workflowStateMachines.cancelWorkflow();
            this.metricsScope.counter("temporal_workflow_canceled").inc(1L);
        } else if (failure != null) {
            this.workflowStateMachines.failWorkflow(failure.getFailure());
            if (!FailureUtils.isBenignApplicationFailure(failure.getFailure())) {
                this.metricsScope.counter("temporal_workflow_failed").inc(1L);
            }
        } else {
            ContinueAsNewWorkflowExecutionCommandAttributes attributes = this.context.getContinueAsNewOnCompletion();
            if (attributes != null) {
                this.workflowStateMachines.continueAsNewWorkflow(attributes);
                this.metricsScope.counter("temporal_workflow_continue_as_new").inc(1L);
            } else {
                Optional<Payloads> workflowOutput = this.workflow.getOutput();
                this.workflowStateMachines.completeWorkflow(workflowOutput);
                this.metricsScope.counter("temporal_workflow_completed").inc(1L);
            }
        }
        Duration d = ProtobufTimeUtils.toM3Duration(Timestamps.fromMillis((long)System.currentTimeMillis()), Timestamps.fromMillis((long)this.context.getRunStartedTimestampMillis()));
        this.metricsScope.timer("temporal_workflow_endtoend_latency").record(d);
    }

    public void handleWorkflowExecutionCancelRequested(HistoryEvent event) {
        WorkflowExecutionCancelRequestedEventAttributes attributes = event.getWorkflowExecutionCancelRequestedEventAttributes();
        this.context.setCancelRequested();
        String cause = attributes.getCause();
        this.workflow.cancel(cause);
    }

    public void handleWorkflowExecutionSignaled(HistoryEvent event) {
        WorkflowExecutionSignaledEventAttributes signalAttributes = event.getWorkflowExecutionSignaledEventAttributes();
        if (this.context.isWorkflowMethodCompleted()) {
            throw new NonDeterministicException("Signal received after workflow is completed. Typically this is caused by a nondeterministic code change in a workflow or a change is what payloads data converters can handle");
        }
        Optional<Payloads> input = signalAttributes.hasInput() ? Optional.of(signalAttributes.getInput()) : Optional.empty();
        this.workflow.handleSignal(signalAttributes.getSignalName(), input, event.getEventId(), signalAttributes.getHeader());
    }

    public void handleWorkflowExecutionUpdated(UpdateMessage updateMessage) {
        if (this.context.isWorkflowMethodCompleted()) {
            throw new NonDeterministicException("Update received after workflow is completed.");
        }
        try {
            Message protocolMessage = updateMessage.getMessage();
            Request update = (Request)protocolMessage.getBody().unpack(Request.class);
            Input input = update.getInput();
            Optional<Payloads> args = Optional.of(input.getArgs());
            this.workflow.handleUpdate(input.getName(), update.getMeta().getUpdateId(), args, protocolMessage.getEventId(), input.getHeader(), updateMessage.getCallbacks());
        }
        catch (InvalidProtocolBufferException e) {
            throw new IllegalStateException("Message is not an update.");
        }
    }

    public Optional<Payloads> query(WorkflowQuery query) {
        return this.workflow.query(query);
    }

    public void close() {
        this.workflow.close();
    }

    public void start(HistoryEvent startWorkflowEvent) {
        this.workflow.start(startWorkflowEvent, this.context);
    }
}

