/*
 * Decompiled with CFR 0.152.
 */
package reactor.test.scheduler;

import java.time.Duration;
import java.time.Instant;
import java.util.Queue;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicIntegerFieldUpdater;
import java.util.concurrent.atomic.AtomicLongFieldUpdater;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import reactor.core.Disposable;
import reactor.core.Disposables;
import reactor.core.Exceptions;
import reactor.core.publisher.Operators;
import reactor.core.scheduler.Scheduler;
import reactor.core.scheduler.Schedulers;
import reactor.util.annotation.Nullable;
import reactor.util.concurrent.Queues;

public class VirtualTimeScheduler
implements Scheduler {
    final Queue<TimedRunnable> queue = new PriorityBlockingQueue<TimedRunnable>(Queues.XS_BUFFER_SIZE);
    volatile long counter;
    volatile long nanoTime;
    volatile long deferredNanoTime;
    static final AtomicLongFieldUpdater<VirtualTimeScheduler> DEFERRED_NANO_TIME = AtomicLongFieldUpdater.newUpdater(VirtualTimeScheduler.class, "deferredNanoTime");
    volatile int advanceTimeWip;
    static final AtomicIntegerFieldUpdater<VirtualTimeScheduler> ADVANCE_TIME_WIP = AtomicIntegerFieldUpdater.newUpdater(VirtualTimeScheduler.class, "advanceTimeWip");
    volatile boolean shutdown;
    final boolean defer;
    final VirtualTimeWorker directWorker;
    private Schedulers.Snapshot schedulersSnapshot;
    static final Disposable CANCELLED = Disposables.disposed();
    static final Disposable EMPTY = Disposables.never();
    static final AtomicReference<VirtualTimeScheduler> CURRENT = new AtomicReference();
    static final AtomicLongFieldUpdater<VirtualTimeScheduler> COUNTER = AtomicLongFieldUpdater.newUpdater(VirtualTimeScheduler.class, "counter");
    static final long CLOCK_DRIFT_TOLERANCE_NANOSECONDS = TimeUnit.MINUTES.toNanos(Long.getLong("reactor.scheduler.drift-tolerance", 15L));

    public static VirtualTimeScheduler create() {
        return VirtualTimeScheduler.create(false);
    }

    public static VirtualTimeScheduler create(boolean defer) {
        VirtualTimeScheduler instance = new VirtualTimeScheduler(defer);
        instance.start();
        return instance;
    }

    public static VirtualTimeScheduler getOrSet() {
        return VirtualTimeScheduler.enable(VirtualTimeScheduler::create, false);
    }

    public static VirtualTimeScheduler getOrSet(boolean defer) {
        return VirtualTimeScheduler.enable(() -> VirtualTimeScheduler.create(defer), false);
    }

    public static VirtualTimeScheduler getOrSet(VirtualTimeScheduler scheduler) {
        return VirtualTimeScheduler.enable(() -> scheduler, false);
    }

    public static VirtualTimeScheduler set(VirtualTimeScheduler scheduler) {
        return VirtualTimeScheduler.enable(() -> scheduler, true);
    }

    static VirtualTimeScheduler enable(Supplier<VirtualTimeScheduler> schedulerSupplier) {
        return VirtualTimeScheduler.enable(schedulerSupplier, false);
    }

    static VirtualTimeScheduler enable(Supplier<VirtualTimeScheduler> schedulerSupplier, boolean exact) {
        VirtualTimeScheduler newS;
        while (true) {
            VirtualTimeScheduler s;
            if ((s = CURRENT.get()) != null && !exact) {
                return s;
            }
            newS = schedulerSupplier.get();
            if (newS == CURRENT.get()) {
                return newS;
            }
            if (!CURRENT.compareAndSet(s, newS)) continue;
            if (s != null) {
                newS.schedulersSnapshot = s.schedulersSnapshot;
                Schedulers.setFactory((Schedulers.Factory)new AllFactory(newS));
            } else {
                newS.schedulersSnapshot = Schedulers.setFactoryWithSnapshot((Schedulers.Factory)new AllFactory(newS));
            }
            if (CURRENT.get() == newS) break;
        }
        return newS;
    }

    public static VirtualTimeScheduler get() {
        VirtualTimeScheduler s = CURRENT.get();
        if (s == null) {
            throw new IllegalStateException("Check if VirtualTimeScheduler#enable has been invoked first");
        }
        return s;
    }

    public static boolean isFactoryEnabled() {
        return CURRENT.get() != null;
    }

    public static void reset() {
        VirtualTimeScheduler s = CURRENT.get();
        if (s != null && CURRENT.compareAndSet(s, null)) {
            Schedulers.resetFrom((Schedulers.Snapshot)s.schedulersSnapshot);
        }
    }

    protected VirtualTimeScheduler(boolean defer) {
        this.defer = defer;
        this.directWorker = this.createWorker();
    }

    public void advanceTime() {
        this.advanceTimeBy(Duration.ZERO);
    }

    public void advanceTimeBy(Duration delayTime) {
        this.advanceTime(delayTime.toNanos());
    }

    public void advanceTimeTo(Instant instant) {
        long targetTime = TimeUnit.NANOSECONDS.convert(instant.toEpochMilli(), TimeUnit.MILLISECONDS);
        this.advanceTime(targetTime - this.nanoTime);
    }

    public long getScheduledTaskCount() {
        return this.counter;
    }

    public VirtualTimeWorker createWorker() {
        if (this.shutdown) {
            throw new IllegalStateException("VirtualTimeScheduler is shutdown");
        }
        return new VirtualTimeWorker();
    }

    public long now(TimeUnit unit) {
        return unit.convert(this.nanoTime + this.deferredNanoTime, TimeUnit.NANOSECONDS);
    }

    public Disposable schedule(Runnable task) {
        if (this.shutdown) {
            throw Exceptions.failWithRejected();
        }
        return this.directWorker.schedule(task);
    }

    public Disposable schedule(Runnable task, long delay, TimeUnit unit) {
        if (this.shutdown) {
            throw Exceptions.failWithRejected();
        }
        return this.directWorker.schedule(task, delay, unit);
    }

    public boolean isDisposed() {
        return this.shutdown;
    }

    public void dispose() {
        if (this.shutdown) {
            return;
        }
        this.queue.clear();
        this.shutdown = true;
        this.directWorker.dispose();
        VirtualTimeScheduler s = CURRENT.get();
        if (s == this && CURRENT.compareAndSet(s, null)) {
            Schedulers.resetFrom((Schedulers.Snapshot)this.schedulersSnapshot);
        }
    }

    public Disposable schedulePeriodically(Runnable task, long initialDelay, long period, TimeUnit unit) {
        if (this.shutdown) {
            throw Exceptions.failWithRejected();
        }
        PeriodicDirectTask periodicTask = new PeriodicDirectTask(task);
        this.directWorker.schedulePeriodically(periodicTask, initialDelay, period, unit);
        return periodicTask;
    }

    final void advanceTime(long timeShiftInNanoseconds) {
        Operators.addCap(DEFERRED_NANO_TIME, (Object)this, (long)timeShiftInNanoseconds);
        this.drain();
    }

    final void drain() {
        int remainingWork = ADVANCE_TIME_WIP.incrementAndGet(this);
        if (remainingWork != 1) {
            return;
        }
        do {
            TimedRunnable current;
            if (this.defer && this.queue.isEmpty()) continue;
            long targetNanoTime = this.nanoTime + DEFERRED_NANO_TIME.getAndSet(this, 0L);
            while (!this.queue.isEmpty() && (current = this.queue.peek()) != null && current.time <= targetNanoTime) {
                this.nanoTime = current.time == 0L ? this.nanoTime : current.time;
                this.queue.remove();
                if (current.scheduler.shutdown) continue;
                current.run.run();
            }
            this.nanoTime = targetNanoTime;
        } while ((remainingWork = ADVANCE_TIME_WIP.addAndGet(this, -remainingWork)) != 0);
    }

    static boolean replace(AtomicReference<Disposable> ref, @Nullable Disposable c) {
        Disposable current;
        do {
            if ((current = ref.get()) != CANCELLED) continue;
            if (c != null) {
                c.dispose();
            }
            return false;
        } while (!ref.compareAndSet(current, c));
        return true;
    }

    static class PeriodicDirectTask
    implements Runnable,
    Disposable {
        final Runnable run;
        volatile boolean disposed;

        PeriodicDirectTask(Runnable run) {
            this.run = run;
        }

        @Override
        public void run() {
            if (!this.disposed) {
                try {
                    this.run.run();
                }
                catch (Throwable ex) {
                    Exceptions.throwIfFatal((Throwable)ex);
                    throw Exceptions.propagate((Throwable)ex);
                }
            }
        }

        public void dispose() {
            this.disposed = true;
        }
    }

    final class VirtualTimeWorker
    implements Scheduler.Worker {
        volatile boolean shutdown;

        VirtualTimeWorker() {
        }

        public Disposable schedule(Runnable run) {
            if (this.shutdown) {
                throw Exceptions.failWithRejected();
            }
            TimedRunnable timedTask = new TimedRunnable(this, 0L, run, COUNTER.getAndIncrement(VirtualTimeScheduler.this));
            VirtualTimeScheduler.this.queue.add(timedTask);
            VirtualTimeScheduler.this.drain();
            return () -> {
                VirtualTimeScheduler.this.queue.remove(timedTask);
                VirtualTimeScheduler.this.drain();
            };
        }

        public Disposable schedule(Runnable run, long delayTime, TimeUnit unit) {
            if (this.shutdown) {
                throw Exceptions.failWithRejected();
            }
            TimedRunnable timedTask = new TimedRunnable(this, VirtualTimeScheduler.this.nanoTime + unit.toNanos(delayTime), run, COUNTER.getAndIncrement(VirtualTimeScheduler.this));
            VirtualTimeScheduler.this.queue.add(timedTask);
            VirtualTimeScheduler.this.drain();
            return () -> {
                VirtualTimeScheduler.this.queue.remove(timedTask);
                VirtualTimeScheduler.this.drain();
            };
        }

        public Disposable schedulePeriodically(Runnable task, long initialDelay, long period, TimeUnit unit) {
            long periodInNanoseconds = unit.toNanos(period);
            long firstNowNanoseconds = VirtualTimeScheduler.this.nanoTime;
            long firstStartInNanoseconds = firstNowNanoseconds + unit.toNanos(initialDelay);
            PeriodicTask periodicTask = new PeriodicTask(firstStartInNanoseconds, task, firstNowNanoseconds, periodInNanoseconds);
            VirtualTimeScheduler.replace(periodicTask, this.schedule(periodicTask, initialDelay, unit));
            return periodicTask;
        }

        public void dispose() {
            this.shutdown = true;
        }

        public boolean isDisposed() {
            return this.shutdown;
        }

        final class PeriodicTask
        extends AtomicReference<Disposable>
        implements Runnable,
        Disposable {
            final Runnable decoratedRun;
            final long periodInNanoseconds;
            long count;
            long lastNowNanoseconds;
            long startInNanoseconds;

            PeriodicTask(long firstStartInNanoseconds, Runnable decoratedRun, long firstNowNanoseconds, long periodInNanoseconds) {
                this.decoratedRun = decoratedRun;
                this.periodInNanoseconds = periodInNanoseconds;
                this.lastNowNanoseconds = firstNowNanoseconds;
                this.startInNanoseconds = firstStartInNanoseconds;
                this.lazySet(EMPTY);
            }

            @Override
            public void run() {
                this.decoratedRun.run();
                if (this.get() != CANCELLED && !VirtualTimeWorker.this.shutdown) {
                    long nextTick;
                    long nowNanoseconds = VirtualTimeScheduler.this.nanoTime;
                    if (nowNanoseconds + CLOCK_DRIFT_TOLERANCE_NANOSECONDS < this.lastNowNanoseconds || nowNanoseconds >= this.lastNowNanoseconds + this.periodInNanoseconds + CLOCK_DRIFT_TOLERANCE_NANOSECONDS) {
                        nextTick = nowNanoseconds + this.periodInNanoseconds;
                        this.startInNanoseconds = nextTick - this.periodInNanoseconds * ++this.count;
                    } else {
                        nextTick = this.startInNanoseconds + ++this.count * this.periodInNanoseconds;
                    }
                    this.lastNowNanoseconds = nowNanoseconds;
                    long delay = nextTick - nowNanoseconds;
                    VirtualTimeScheduler.replace(this, VirtualTimeWorker.this.schedule(this, delay, TimeUnit.NANOSECONDS));
                }
            }

            public void dispose() {
                this.getAndSet(CANCELLED).dispose();
            }
        }
    }

    static final class AllFactory
    implements Schedulers.Factory {
        final VirtualTimeScheduler s;

        AllFactory(VirtualTimeScheduler s) {
            this.s = s;
        }

        public Scheduler newElastic(int ttlSeconds, ThreadFactory threadFactory) {
            return this.s;
        }

        public Scheduler newBoundedElastic(int threadCap, int taskCap, ThreadFactory threadFactory, int ttlSeconds) {
            return this.s;
        }

        public Scheduler newParallel(int parallelism, ThreadFactory threadFactory) {
            return this.s;
        }

        public Scheduler newSingle(ThreadFactory threadFactory) {
            return this.s;
        }
    }

    static final class TimedRunnable
    implements Comparable<TimedRunnable> {
        final long time;
        final Runnable run;
        final VirtualTimeWorker scheduler;
        final long count;

        TimedRunnable(VirtualTimeWorker scheduler, long time, Runnable run, long count) {
            this.time = time;
            this.run = run;
            this.scheduler = scheduler;
            this.count = count;
        }

        @Override
        public int compareTo(TimedRunnable o) {
            if (this.time == o.time) {
                return TimedRunnable.compare(this.count, o.count);
            }
            return TimedRunnable.compare(this.time, o.time);
        }

        static int compare(long a, long b) {
            return a < b ? -1 : (a > b ? 1 : 0);
        }
    }
}

