/*
Copyright 2020 The Kubernetes Authors.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/
package io.kubernetes.client.openapi;

import java.util.Map;

/** Representing a Server configuration. */
public class ServerConfiguration {
  public String URL;
  public String description;
  public Map<String, ServerVariable> variables;

  /**
   * @param URL A URL to the target host.
   * @param description A describtion of the host designated by the URL.
   * @param variables A map between a variable name and its value. The value is used for
   *     substitution in the server's URL template.
   */
  public ServerConfiguration(
      String URL, String description, Map<String, ServerVariable> variables) {
    this.URL = URL;
    this.description = description;
    this.variables = variables;
  }

  /**
   * Format URL template using given variables.
   *
   * @param variables A map between a variable name and its value.
   * @return Formatted URL.
   */
  public String URL(Map<String, String> variables) {
    String url = this.URL;

    // go through variables and replace placeholders
    for (Map.Entry<String, ServerVariable> variable : this.variables.entrySet()) {
      String name = variable.getKey();
      ServerVariable serverVariable = variable.getValue();
      String value = serverVariable.defaultValue;

      if (variables != null && variables.containsKey(name)) {
        value = variables.get(name);
        if (serverVariable.enumValues.size() > 0 && !serverVariable.enumValues.contains(value)) {
          throw new RuntimeException(
              "The variable " + name + " in the server URL has invalid value " + value + ".");
        }
      }
      url = url.replaceAll("\\{" + name + "\\}", value);
    }
    return url;
  }

  /**
   * Format URL template using default server variables.
   *
   * @return Formatted URL.
   */
  public String URL() {
    return URL(null);
  }
}
