/*
 * Decompiled with CFR 0.152.
 */
package com.wavefront.agent.queueing;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterators;
import com.google.errorprone.annotations.CanIgnoreReturnValue;
import com.wavefront.agent.queueing.QueueFile;
import com.wavefront.agent.queueing.QueueFileFactory;
import com.wavefront.common.Utils;
import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.ConcurrentModificationException;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.ObjectUtils;

public class ConcurrentShardedQueueFile
implements QueueFile {
    private static final int HEADER_SIZE_BYTES = 36;
    private static final int TASK_HEADER_SIZE_BYTES = 4;
    private static final int SUFFIX_DIGITS = 4;
    private final String fileNamePrefix;
    private final String fileNameSuffix;
    private final int shardSizeBytes;
    private final QueueFileFactory queueFileFactory;
    @VisibleForTesting
    final Deque<Shard> shards = new ConcurrentLinkedDeque<Shard>();
    private final ReentrantLock globalLock = new ReentrantLock(true);
    private final ReentrantLock tailLock = new ReentrantLock(true);
    private final ReentrantLock headLock = new ReentrantLock(true);
    private volatile boolean closed = false;
    private volatile byte[] head;
    private final AtomicLong modCount = new AtomicLong();

    public ConcurrentShardedQueueFile(String fileNamePrefix, String fileNameSuffix, int shardSizeBytes, QueueFileFactory queueFileFactory) throws IOException {
        this.fileNamePrefix = fileNamePrefix;
        this.fileNameSuffix = fileNameSuffix;
        this.shardSizeBytes = shardSizeBytes;
        this.queueFileFactory = queueFileFactory;
        for (String filename : (List)ObjectUtils.firstNonNull((Object[])new List[]{ConcurrentShardedQueueFile.listFiles(fileNamePrefix, fileNameSuffix), ImmutableList.of((Object)this.getInitialFilename())})) {
            Shard shard = new Shard(filename);
            shard.close();
            this.shards.add(shard);
        }
    }

    @Override
    @Nullable
    public byte[] peek() throws IOException {
        this.checkForClosedState();
        this.headLock.lock();
        try {
            if (this.head == null) {
                this.globalLock.lock();
                Shard shard = this.shards.getFirst().updateStats();
                if (this.shards.size() > 1) {
                    this.globalLock.unlock();
                }
                this.head = Objects.requireNonNull(shard.queueFile).peek();
            }
            byte[] byArray = this.head;
            return byArray;
        }
        finally {
            this.headLock.unlock();
            if (this.globalLock.isHeldByCurrentThread()) {
                this.globalLock.unlock();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void add(byte[] data, int offset, int count) throws IOException {
        this.checkForClosedState();
        this.tailLock.lock();
        try {
            this.globalLock.lock();
            Shard shard = this.shards.getLast();
            if (shard.newShardRequired(count)) {
                if (this.shards.size() > 1) {
                    shard.close();
                }
                String newFileName = ConcurrentShardedQueueFile.incrementFileName(shard.shardFileName, this.fileNameSuffix);
                shard = new Shard(newFileName);
                this.shards.addLast(shard);
            }
            shard.updateStats();
            this.modCount.incrementAndGet();
            if (this.shards.size() > 2) {
                this.globalLock.unlock();
            }
            Objects.requireNonNull(shard.queueFile).add(data, offset, count);
            shard.updateStats();
        }
        finally {
            this.tailLock.unlock();
            if (this.globalLock.isHeldByCurrentThread()) {
                this.globalLock.unlock();
            }
        }
    }

    @Override
    public void remove() throws IOException {
        this.checkForClosedState();
        this.headLock.lock();
        try {
            this.head = null;
            Shard shard = this.shards.getFirst().updateStats();
            if (this.shards.size() == 1) {
                this.globalLock.lock();
            }
            this.modCount.incrementAndGet();
            Objects.requireNonNull(shard.queueFile).remove();
            shard.updateStats();
            if (this.shards.size() > 1 && shard.numTasks == 0) {
                shard.close();
                this.shards.removeFirst();
                new File(shard.shardFileName).delete();
            }
        }
        finally {
            this.headLock.unlock();
            if (this.globalLock.isHeldByCurrentThread()) {
                this.globalLock.unlock();
            }
        }
    }

    @Override
    public int size() {
        return this.shards.stream().mapToInt(shard -> ((Shard)shard).numTasks).sum();
    }

    @Override
    public long storageBytes() {
        return this.shards.stream().mapToLong(shard -> ((Shard)shard).fileLength).sum();
    }

    @Override
    public long usedBytes() {
        return this.shards.stream().mapToLong(shard -> ((Shard)shard).usedBytes).sum();
    }

    @Override
    public long availableBytes() {
        Shard shard = this.shards.getLast();
        return shard.fileLength - shard.usedBytes;
    }

    @Override
    public void close() throws IOException {
        this.closed = true;
        for (Shard shard : this.shards) {
            shard.close();
        }
    }

    @Override
    public void clear() throws IOException {
        this.headLock.lock();
        this.tailLock.lock();
        try {
            this.head = null;
            for (Shard shard : this.shards) {
                shard.close();
                new File(shard.shardFileName).delete();
            }
            this.shards.clear();
            this.shards.add(new Shard(this.getInitialFilename()));
            this.modCount.incrementAndGet();
        }
        finally {
            this.headLock.unlock();
            this.tailLock.unlock();
        }
    }

    @Override
    @Nonnull
    public Iterator<byte[]> iterator() {
        this.checkForClosedState();
        return new ShardedIterator();
    }

    private void checkForClosedState() {
        if (this.closed) {
            throw new IllegalStateException("closed");
        }
    }

    private String getInitialFilename() {
        return new File(this.fileNamePrefix).exists() ? this.fileNamePrefix : ConcurrentShardedQueueFile.incrementFileName(this.fileNamePrefix, this.fileNameSuffix);
    }

    @Nullable
    @VisibleForTesting
    static List<String> listFiles(String path, String suffix) {
        String fnPrefix = (String)Iterators.getLast(Splitter.on((char)'/').split((CharSequence)path).iterator());
        Pattern pattern = ConcurrentShardedQueueFile.getSuffixMatchingPattern(suffix);
        File bufferFilePath = new File(path);
        File[] files = bufferFilePath.getParentFile().listFiles((dir, fileName) -> (fileName.endsWith(suffix) || pattern.matcher(fileName).matches()) && fileName.startsWith(fnPrefix));
        return files == null || files.length == 0 ? null : Arrays.stream(files).map(File::getAbsolutePath).sorted().collect(Collectors.toList());
    }

    @VisibleForTesting
    static String incrementFileName(String fileName, String suffix) {
        Pattern pattern = ConcurrentShardedQueueFile.getSuffixMatchingPattern(suffix);
        String zeroes = StringUtils.repeat((String)"0", (int)4);
        if (pattern.matcher(fileName).matches()) {
            int nextId = Integer.parseInt(StringUtils.right((String)fileName, (int)4), 16) + 1;
            String newHex = StringUtils.right((String)(zeroes + Long.toHexString(nextId)), (int)4);
            return StringUtils.left((String)fileName, (int)(fileName.length() - 4)) + newHex;
        }
        return fileName + "_" + zeroes;
    }

    private static Pattern getSuffixMatchingPattern(String suffix) {
        return Pattern.compile("^.*" + Pattern.quote(suffix) + "_[0-9a-f]{" + 4 + "}$");
    }

    private final class Shard {
        private final String shardFileName;
        @Nullable
        private QueueFile queueFile;
        private long fileLength;
        private Long usedBytes;
        private int numTasks;

        private Shard(String shardFileName) throws IOException {
            this.shardFileName = shardFileName;
            this.updateStats();
        }

        @CanIgnoreReturnValue
        private Shard updateStats() throws IOException {
            if (this.queueFile == null) {
                this.queueFile = ConcurrentShardedQueueFile.this.queueFileFactory.get(this.shardFileName);
            }
            if (this.queueFile != null) {
                this.fileLength = this.queueFile.storageBytes();
                this.numTasks = this.queueFile.size();
                this.usedBytes = this.queueFile.usedBytes();
            }
            return this;
        }

        private void close() throws IOException {
            if (this.queueFile != null) {
                this.queueFile.close();
                this.queueFile = null;
            }
        }

        private boolean newShardRequired(int taskSize) {
            return (long)taskSize > (long)ConcurrentShardedQueueFile.this.shardSizeBytes - this.usedBytes - 4L && (taskSize <= ConcurrentShardedQueueFile.this.shardSizeBytes - 36 || this.numTasks > 0);
        }
    }

    private final class ShardedIterator
    implements Iterator<byte[]> {
        long expectedModCount;
        Iterator<byte[]> currentIterator;
        Shard currentShard;
        Iterator<Shard> shardIterator;
        int nextElementIndex;

        ShardedIterator() {
            this.expectedModCount = ConcurrentShardedQueueFile.this.modCount.get();
            this.currentIterator = Collections.emptyIterator();
            this.currentShard = null;
            this.shardIterator = ConcurrentShardedQueueFile.this.shards.iterator();
            this.nextElementIndex = 0;
        }

        private void checkForComodification() {
            ConcurrentShardedQueueFile.this.checkForClosedState();
            if (ConcurrentShardedQueueFile.this.modCount.get() != this.expectedModCount) {
                throw new ConcurrentModificationException();
            }
        }

        @Override
        public boolean hasNext() {
            this.checkForComodification();
            try {
                while (!((Iterator)Preconditions.checkNotNull(this.currentIterator)).hasNext()) {
                    if (!this.shardIterator.hasNext()) {
                        return false;
                    }
                    this.currentShard = this.shardIterator.next().updateStats();
                    this.currentIterator = Objects.requireNonNull(this.currentShard.queueFile).iterator();
                }
            }
            catch (IOException e) {
                throw (Error)Utils.throwAny(e);
            }
            return true;
        }

        @Override
        public byte[] next() {
            this.checkForComodification();
            if (this.hasNext()) {
                ++this.nextElementIndex;
                return this.currentIterator.next();
            }
            throw new NoSuchElementException();
        }

        @Override
        public void remove() {
            this.checkForComodification();
            if (this.nextElementIndex > 1) {
                throw new UnsupportedOperationException("Removal is only permitted from the head.");
            }
            try {
                this.currentIterator.remove();
                this.currentShard.updateStats();
                --this.nextElementIndex;
            }
            catch (IOException e) {
                throw (Error)Utils.throwAny(e);
            }
        }
    }
}

