/*
 * Decompiled with CFR 0.152.
 */
package com.singlestore.jdbc.pool;

import com.singlestore.jdbc.Configuration;
import com.singlestore.jdbc.Connection;
import com.singlestore.jdbc.Driver;
import com.singlestore.jdbc.Statement;
import com.singlestore.jdbc.pool.InternalPoolConnection;
import com.singlestore.jdbc.pool.PoolMBean;
import com.singlestore.jdbc.pool.PoolThreadFactory;
import com.singlestore.jdbc.pool.Pools;
import com.singlestore.jdbc.util.log.Logger;
import com.singlestore.jdbc.util.log.Loggers;
import java.lang.management.ManagementFactory;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.sql.ConnectionEvent;
import javax.sql.ConnectionEventListener;

public class Pool
implements AutoCloseable,
PoolMBean {
    private final Logger logger;
    private static final int POOL_STATE_OK = 0;
    private static final int POOL_STATE_CLOSING = 1;
    private final AtomicInteger poolState = new AtomicInteger();
    private final Configuration conf;
    private final AtomicInteger pendingRequestNumber = new AtomicInteger();
    private final AtomicInteger totalConnection = new AtomicInteger();
    private final LinkedBlockingDeque<InternalPoolConnection> idleConnections;
    private final ThreadPoolExecutor connectionAppender;
    private final BlockingQueue<Runnable> connectionAppenderQueue;
    private final String poolTag;
    private final ScheduledThreadPoolExecutor poolExecutor;
    private final ScheduledFuture<?> scheduledFuture;
    private int waitTimeout;

    public Pool(Configuration conf, int poolIndex, ScheduledThreadPoolExecutor poolExecutor) {
        this.logger = Loggers.getLogger(Pool.class);
        this.conf = conf;
        this.poolTag = this.generatePoolTag(poolIndex);
        this.connectionAppenderQueue = new ArrayBlockingQueue<Runnable>(conf.maxPoolSize());
        this.connectionAppender = new ThreadPoolExecutor(1, 1, 10L, TimeUnit.SECONDS, this.connectionAppenderQueue, new PoolThreadFactory(this.poolTag + "-appender"));
        this.connectionAppender.allowCoreThreadTimeOut(true);
        this.connectionAppender.prestartCoreThread();
        this.idleConnections = new LinkedBlockingDeque();
        int minDelay = Integer.parseInt(conf.nonMappedOptions().getProperty("testMaxRemovalDelay", "30"));
        int scheduleDelay = Math.min(minDelay, conf.maxIdleTime() / 2);
        this.poolExecutor = poolExecutor;
        this.scheduledFuture = poolExecutor.scheduleAtFixedRate(this::removeIdleTimeoutConnection, scheduleDelay, scheduleDelay, TimeUnit.SECONDS);
        if (conf.registerJmxPool()) {
            try {
                this.registerJmx();
            }
            catch (Exception ex) {
                this.logger.error("pool " + this.poolTag + " not registered due to exception : " + ex.getMessage());
            }
        }
        try {
            Statement stmt;
            ResultSet rs;
            for (int i = 0; i < Math.max(1, conf.minPoolSize()); ++i) {
                this.addConnection();
            }
            this.waitTimeout = 28800;
            if (!this.idleConnections.isEmpty() && (rs = (stmt = this.idleConnections.getFirst().getConnection().createStatement()).executeQuery("SELECT @@wait_timeout")).next()) {
                this.waitTimeout = rs.getInt(1);
            }
        }
        catch (SQLException sqle) {
            this.logger.error("error initializing pool connection", sqle);
        }
    }

    private void addConnectionRequest() {
        if (this.totalConnection.get() < this.conf.maxPoolSize() && this.poolState.get() == 0) {
            this.connectionAppender.prestartCoreThread();
            boolean bl = this.connectionAppenderQueue.offer(() -> {
                if ((this.totalConnection.get() < this.conf.minPoolSize() || this.pendingRequestNumber.get() > 0) && this.totalConnection.get() < this.conf.maxPoolSize()) {
                    try {
                        this.addConnection();
                    }
                    catch (SQLException sqle) {
                        this.logger.error("error adding connection to pool", sqle);
                    }
                }
            });
        }
    }

    private void removeIdleTimeoutConnection() {
        Iterator<InternalPoolConnection> iterator = this.idleConnections.descendingIterator();
        while (iterator.hasNext()) {
            InternalPoolConnection item = iterator.next();
            long idleTime = System.nanoTime() - item.getLastUsed().get();
            boolean timedOut = idleTime > TimeUnit.SECONDS.toNanos(this.conf.maxIdleTime());
            boolean shouldBeReleased = false;
            Connection con = item.getConnection();
            if (this.waitTimeout > 0) {
                if (idleTime > TimeUnit.SECONDS.toNanos(this.waitTimeout - 45)) {
                    shouldBeReleased = true;
                }
                if (timedOut && this.totalConnection.get() > this.conf.minPoolSize()) {
                    shouldBeReleased = true;
                }
            } else if (timedOut) {
                shouldBeReleased = true;
            }
            if (!shouldBeReleased || !this.idleConnections.remove(item)) continue;
            this.totalConnection.decrementAndGet();
            this.silentCloseConnection(con);
            this.addConnectionRequest();
            if (!this.logger.isDebugEnabled()) continue;
            this.logger.debug("pool {} connection {} removed due to inactivity (total:{}, active:{}, pending:{})", this.poolTag, con.getThreadId(), this.totalConnection.get(), this.getActiveConnections(), this.pendingRequestNumber.get());
        }
    }

    private void addConnection() throws SQLException {
        Connection connection = Driver.connect(this.conf);
        InternalPoolConnection item = new InternalPoolConnection(connection);
        item.addConnectionEventListener(new ConnectionEventListener(){

            @Override
            public void connectionClosed(ConnectionEvent event) {
                InternalPoolConnection item = (InternalPoolConnection)event.getSource();
                if (Pool.this.poolState.get() == 0) {
                    try {
                        if (!Pool.this.idleConnections.contains(item)) {
                            item.getConnection().reset();
                            item.lastUsedToNow();
                            if (item.isFailed()) {
                                item.setFailed(false);
                                Pool.this.totalConnection.incrementAndGet();
                            }
                            Pool.this.idleConnections.addFirst(item);
                        }
                    }
                    catch (SQLException sqle) {
                        Pool.this.totalConnection.decrementAndGet();
                        Pool.this.silentCloseConnection(item.getConnection());
                        Pool.this.logger.debug("connection removed from pool {} due to error during reset", Pool.this.poolTag);
                    }
                } else {
                    try {
                        item.getConnection().close();
                    }
                    catch (SQLException sQLException) {
                        // empty catch block
                    }
                    Pool.this.totalConnection.decrementAndGet();
                }
            }

            @Override
            public void connectionErrorOccurred(ConnectionEvent event) {
                InternalPoolConnection item = (InternalPoolConnection)event.getSource();
                item.setFailed(true);
                Pool.this.totalConnection.decrementAndGet();
                boolean unused = Pool.this.idleConnections.remove(item);
                Pool.this.idleConnections.forEach(InternalPoolConnection::ensureValidation);
                Pool.this.silentCloseConnection(item.getConnection());
                Pool.this.addConnectionRequest();
                Pool.this.logger.debug("connection {} removed from pool {} due to having throw a Connection exception (total:{}, active:{}, pending:{})", item.getConnection().getThreadId(), Pool.this.poolTag, Pool.this.totalConnection.get(), Pool.this.getActiveConnections(), Pool.this.pendingRequestNumber.get());
            }
        });
        if (this.poolState.get() == 0 && this.totalConnection.incrementAndGet() <= this.conf.maxPoolSize()) {
            this.idleConnections.addFirst(item);
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("pool {} new physical connection created (total:{}, active:{}, pending:{})", this.poolTag, this.totalConnection.get(), this.getActiveConnections(), this.pendingRequestNumber.get());
            }
            return;
        }
        this.silentCloseConnection(connection);
    }

    private InternalPoolConnection getIdleConnection(long timeout, TimeUnit timeUnit) throws InterruptedException {
        while (true) {
            InternalPoolConnection item;
            block5: {
                InternalPoolConnection internalPoolConnection = item = timeout == 0L ? this.idleConnections.pollFirst() : this.idleConnections.pollFirst(timeout, timeUnit);
                if (item == null) break;
                try {
                    if (TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - item.getLastUsed().get()) > (long)this.conf.poolValidMinDelay()) {
                        if (item.getConnection().isValid(10)) {
                            return item;
                        }
                        break block5;
                    }
                    return item;
                }
                catch (SQLException sQLException) {
                    // empty catch block
                }
            }
            this.silentAbortConnection(item.getConnection());
            this.addConnectionRequest();
            if (!this.logger.isDebugEnabled()) continue;
            this.logger.debug("pool {} connection {} removed from pool due to failed validation (total:{}, active:{}, pending:{})", this.poolTag, item.getConnection().getThreadId(), this.totalConnection.get(), this.getActiveConnections(), this.pendingRequestNumber.get());
        }
        return null;
    }

    private void silentCloseConnection(Connection con) {
        con.setPoolConnection(null);
        try {
            con.close();
        }
        catch (SQLException sQLException) {
            // empty catch block
        }
    }

    private void silentAbortConnection(Connection con) {
        con.setPoolConnection(null);
        try {
            con.abort(this.poolExecutor);
        }
        catch (SQLException sQLException) {
            // empty catch block
        }
    }

    public InternalPoolConnection getPoolConnection() throws SQLException {
        this.pendingRequestNumber.incrementAndGet();
        try {
            InternalPoolConnection poolConnection = this.getIdleConnection(this.totalConnection.get() > 4 ? 0L : 50L, TimeUnit.MICROSECONDS);
            if (poolConnection != null) {
                InternalPoolConnection internalPoolConnection = poolConnection;
                return internalPoolConnection;
            }
            this.addConnectionRequest();
            poolConnection = this.getIdleConnection(TimeUnit.MILLISECONDS.toNanos(this.conf.connectTimeout()), TimeUnit.NANOSECONDS);
            if (poolConnection != null) {
                InternalPoolConnection internalPoolConnection = poolConnection;
                return internalPoolConnection;
            }
            try {
                throw new SQLException(String.format("No connection available within the specified time (option 'connectTimeout': %s ms)", NumberFormat.getInstance().format(this.conf.connectTimeout())));
            }
            catch (InterruptedException interrupted) {
                throw new SQLException("Thread was interrupted", "70100", interrupted);
            }
        }
        finally {
            this.pendingRequestNumber.decrementAndGet();
        }
    }

    public InternalPoolConnection getPoolConnection(String username, String password) throws SQLException {
        if (username == null ? this.conf.user() == null : (username.equals(this.conf.user()) && (password == null || password.isEmpty()) ? this.conf.password() == null : password.equals(this.conf.password()))) {
            return this.getPoolConnection();
        }
        Configuration tmpConf = this.conf.clone(username, password);
        return new InternalPoolConnection(Driver.connect(tmpConf));
    }

    private String generatePoolTag(int poolIndex) {
        if (this.conf.poolName() == null) {
            return "SingleStore-pool";
        }
        return this.conf.poolName() + "-" + poolIndex;
    }

    public Configuration getConf() {
        return this.conf;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        try {
            Pool pool = this;
            synchronized (pool) {
                Pools.remove(this);
                this.poolState.set(1);
                this.pendingRequestNumber.set(0);
                this.scheduledFuture.cancel(false);
                this.connectionAppender.shutdown();
                try {
                    boolean bl = this.connectionAppender.awaitTermination(10L, TimeUnit.SECONDS);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
                if (this.logger.isInfoEnabled()) {
                    this.logger.debug("closing pool {} (total:{}, active:{}, pending:{})", this.poolTag, this.totalConnection.get(), this.getActiveConnections(), this.pendingRequestNumber.get());
                }
                ThreadPoolExecutor connectionRemover = new ThreadPoolExecutor(this.totalConnection.get(), this.conf.maxPoolSize(), 10L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(this.conf.maxPoolSize()), new PoolThreadFactory(this.poolTag + "-destroyer"));
                long start = System.nanoTime();
                do {
                    this.closeAll(this.idleConnections);
                    if (this.totalConnection.get() <= 0) continue;
                    Thread.sleep(0L, 1000);
                } while (this.totalConnection.get() > 0 && TimeUnit.NANOSECONDS.toSeconds(System.nanoTime() - start) < 10L);
                if (this.totalConnection.get() > 0 || this.idleConnections.isEmpty()) {
                    this.closeAll(this.idleConnections);
                }
                connectionRemover.shutdown();
                try {
                    this.unRegisterJmx();
                }
                catch (Exception exception) {
                    // empty catch block
                }
                boolean bl = connectionRemover.awaitTermination(10L, TimeUnit.SECONDS);
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void closeAll(Collection<InternalPoolConnection> collection) {
        Collection<InternalPoolConnection> collection2 = collection;
        synchronized (collection2) {
            for (InternalPoolConnection item : collection) {
                collection.remove(item);
                this.totalConnection.decrementAndGet();
                this.silentAbortConnection(item.getConnection());
            }
        }
    }

    public String getPoolTag() {
        return this.poolTag;
    }

    @Override
    public long getActiveConnections() {
        return this.totalConnection.get() - this.idleConnections.size();
    }

    @Override
    public long getTotalConnections() {
        return this.totalConnection.get();
    }

    @Override
    public long getIdleConnections() {
        return this.idleConnections.size();
    }

    @Override
    public long getConnectionRequests() {
        return this.pendingRequestNumber.get();
    }

    private void registerJmx() throws Exception {
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        String jmxName = this.poolTag.replace(":", "_");
        ObjectName name = new ObjectName("com.singlestore.jdbc.pool:type=" + jmxName);
        if (!mbs.isRegistered(name)) {
            mbs.registerMBean(this, name);
        }
    }

    private void unRegisterJmx() throws Exception {
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        String jmxName = this.poolTag.replace(":", "_");
        ObjectName name = new ObjectName("com.singlestore.jdbc.pool:type=" + jmxName);
        if (mbs.isRegistered(name)) {
            mbs.unregisterMBean(name);
        }
    }

    public List<Long> testGetConnectionIdleThreadIds() {
        ArrayList<Long> threadIds = new ArrayList<Long>();
        for (InternalPoolConnection pooledConnection : this.idleConnections) {
            threadIds.add(pooledConnection.getConnection().getThreadId());
        }
        return threadIds;
    }
}

