/*
 * Decompiled with CFR 0.152.
 */
package com.palantir.conjure.java.config.ssl;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.io.BaseEncoding;
import com.palantir.conjure.java.config.ssl.PemX509Certificate;
import com.palantir.conjure.java.config.ssl.pkcs1.Pkcs1PrivateKeyReader;
import com.palantir.logsafe.Arg;
import com.palantir.logsafe.exceptions.SafeRuntimeException;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.KeySpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.RSAPrivateKeySpec;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

final class KeyStores {
    private static final Cache<EqualByteArray, X509Certificate> certCache = Caffeine.newBuilder().maximumSize(1024L).softValues().build();
    private static final Pattern KEY_PATTERN = Pattern.compile("-----BEGIN (RSA)? ?PRIVATE KEY-----\n?(.+?)\n?-----END (RSA)? ?PRIVATE KEY-----", 32);
    private static final Pattern CERT_PATTERN = Pattern.compile("-----BEGIN CERTIFICATE-----\n?(.+?)\n?-----END CERTIFICATE-----", 32);
    private static final FileFilter VISIBLE_FILE_FILTER = new FileFilter(){

        @Override
        public boolean accept(File pathname) {
            return !pathname.isHidden();
        }
    };

    private KeyStores() {
    }

    static KeyStore createTrustStoreFromCertificates(Path path) {
        KeyStore keyStore = KeyStores.createKeyStore();
        for (File currFile : KeyStores.getFilesForPath(path)) {
            try (BufferedInputStream in = new BufferedInputStream(Files.newInputStream(currFile.toPath(), new OpenOption[0]));){
                KeyStores.addCertificatesToKeystore(keyStore, currFile.getName(), KeyStores.readX509Certificates(in));
            }
            catch (IOException e) {
                throw new UncheckedIOException(String.format("IOException encountered when opening '%s'", currFile.toPath()), e);
            }
            catch (KeyStoreException | CertificateException e) {
                throw new RuntimeException(String.format("Could not read file at \"%s\" as an X.509 certificate", currFile.toPath()), e);
            }
        }
        return keyStore;
    }

    static KeyStore createTrustStoreFromCertificates(Map<String, PemX509Certificate> certificatesByAlias) {
        KeyStore keyStore = KeyStores.createKeyStore();
        for (Map.Entry<String, PemX509Certificate> entry : certificatesByAlias.entrySet()) {
            try (ByteArrayInputStream certIn = new ByteArrayInputStream(entry.getValue().pemCertificate().getBytes(StandardCharsets.UTF_8));){
                KeyStores.addCertificatesToKeystore(keyStore, entry.getKey(), KeyStores.readX509Certificates(certIn));
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
            catch (GeneralSecurityException e) {
                throw new RuntimeException(String.format("Could not read certificate alias \"%s\" as an X.509 certificate", entry.getKey()), e);
            }
        }
        return keyStore;
    }

    private static void addCertificatesToKeystore(KeyStore keyStore, String certificateEntryNamePrefix, List<Certificate> certificates) throws KeyStoreException {
        int certIndex = 0;
        for (Certificate cert : certificates) {
            keyStore.setCertificateEntry(certificateEntryNamePrefix + "-" + certIndex, cert);
            ++certIndex;
        }
    }

    static KeyStore createKeyStoreFromCombinedPems(Path filePathOrDirectory) {
        try {
            KeyStore keyStore = KeyStore.getInstance("pkcs12");
            keyStore.load(null, null);
            for (File currFile : KeyStores.getFilesForPath(filePathOrDirectory)) {
                KeyStore.PrivateKeyEntry privateKeyEntry = KeyStores.readKeyEntryFromPems(currFile.toPath(), currFile.toPath());
                keyStore.setKeyEntry(currFile.getName(), privateKeyEntry.getPrivateKey(), null, privateKeyEntry.getCertificateChain());
            }
            return keyStore;
        }
        catch (IOException | GeneralSecurityException e) {
            throw new RuntimeException(e);
        }
    }

    static KeyStore createKeyStoreFromPemDirectories(Path keyDirPath, String keyExtension, Path certDirPath, String certExtension) {
        if (!keyDirPath.toFile().isDirectory()) {
            throw new IllegalStateException(String.format("keyDirPath is not a directory: \"%s\"", keyDirPath));
        }
        if (!certDirPath.toFile().isDirectory()) {
            throw new IllegalStateException(String.format("certDirPath is not a directory: \"%s\"", certDirPath));
        }
        try {
            File[] keyFiles;
            KeyStore keyStore = KeyStore.getInstance("pkcs12");
            keyStore.load(null, null);
            for (File currKeyFile : keyFiles = KeyStores.getFilesForPath(keyDirPath)) {
                String currKeyFileName = currKeyFile.getName();
                if (!currKeyFileName.endsWith(keyExtension)) continue;
                String baseName = currKeyFileName.substring(0, currKeyFileName.length() - keyExtension.length());
                Path currCertPath = certDirPath.resolve(baseName + certExtension);
                KeyStore.PrivateKeyEntry privateKeyEntry = KeyStores.readKeyEntryFromPems(currKeyFile.toPath(), currCertPath);
                keyStore.setKeyEntry(baseName, privateKeyEntry.getPrivateKey(), null, privateKeyEntry.getCertificateChain());
            }
            return keyStore;
        }
        catch (IOException | GeneralSecurityException e) {
            throw new SafeRuntimeException("Failed to create key store from PEM directories", (Throwable)e, new Arg[0]);
        }
    }

    private static File[] getFilesForPath(Path path) {
        File[] files;
        File pathFile = path.toFile();
        if (pathFile.isDirectory()) {
            files = pathFile.listFiles(VISIBLE_FILE_FILTER);
            if (files == null) {
                throw new IllegalStateException(String.format("failed to list visible files in directory %s", path));
            }
        } else {
            files = new File[]{pathFile};
        }
        return files;
    }

    static KeyStore loadKeyStore(String storeType, Path path, Optional<String> password) {
        try {
            KeyStore keyStore = KeyStore.getInstance(storeType);
            try (InputStream stream = Files.newInputStream(path, new OpenOption[0]);){
                char[] passwordChars = password.map(String::toCharArray).orElse(null);
                keyStore.load(stream, passwordChars);
            }
            return keyStore;
        }
        catch (IOException | GeneralSecurityException e) {
            throw new RuntimeException(e);
        }
    }

    static KeyStore newKeyStoreWithEntry(KeyStore original, Optional<String> password, String alias) {
        try {
            KeyStore newKeyStore = KeyStore.getInstance(original.getType());
            char[] passwordChar = password.map(String::toCharArray).orElse(null);
            newKeyStore.load(null, passwordChar);
            Key aliasKey = original.getKey(alias, passwordChar);
            if (aliasKey == null) {
                throw new IllegalStateException(String.format("Could not find key with alias \"%s\" in key store", alias));
            }
            Certificate[] certificateChain = original.getCertificateChain(alias);
            if (certificateChain == null) {
                throw new IllegalStateException(String.format("Could not find certificate chain with alias \"%s\" in key store", alias));
            }
            newKeyStore.setKeyEntry(alias, aliasKey, passwordChar, certificateChain);
            return newKeyStore;
        }
        catch (IOException | GeneralSecurityException e) {
            throw new RuntimeException(e);
        }
    }

    private static KeyStore createKeyStore() {
        KeyStore keyStore;
        try {
            keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
            keyStore.load(null, null);
        }
        catch (IOException | GeneralSecurityException e) {
            throw new RuntimeException(e);
        }
        return keyStore;
    }

    static List<Certificate> readX509Certificates(InputStream certificateIn) throws CertificateException {
        return CertificateFactory.getInstance("X.509").generateCertificates(certificateIn).stream().map(cert -> KeyStores.getCertFromCache((X509Certificate)cert)).collect(Collectors.toList());
    }

    private static KeyStore.PrivateKeyEntry readKeyEntryFromPems(Path privateKeyFilePath, Path certFilePath) {
        Certificate[] certificates;
        PrivateKey privateKey;
        String keyPemFileString;
        try {
            keyPemFileString = KeyStores.readFileAsString(privateKeyFilePath);
            privateKey = KeyStores.getPrivateKeyFromString(keyPemFileString);
        }
        catch (IOException | GeneralSecurityException e) {
            throw new RuntimeException(String.format("Failed to read private key from file at \"%s\"", privateKeyFilePath), e);
        }
        try {
            String certPemFileString = privateKeyFilePath.equals(certFilePath) ? keyPemFileString : KeyStores.readFileAsString(certFilePath);
            certificates = KeyStores.getCertificatesFromString(certPemFileString).toArray(new Certificate[0]);
        }
        catch (IOException | GeneralSecurityException e) {
            throw new RuntimeException(String.format("Failed to read certificates from file at \"%s\"", certFilePath), e);
        }
        return new KeyStore.PrivateKeyEntry(privateKey, certificates);
    }

    private static String readFileAsString(Path path) throws IOException {
        byte[] fileBytes = Files.readAllBytes(path);
        return new String(fileBytes, StandardCharsets.UTF_8);
    }

    static PrivateKey getPrivateKeyFromString(String pemFileString) throws GeneralSecurityException {
        Matcher matcher = KEY_PATTERN.matcher(pemFileString);
        if (!matcher.find() || !Objects.equals(matcher.group(1), matcher.group(3))) {
            throw new GeneralSecurityException(String.format("unable to find valid RSA key in the provided string: %s", pemFileString));
        }
        String privateKeyString = matcher.group(2).replace("\n", "");
        byte[] privateKeyDerBytes = BaseEncoding.base64().decode((CharSequence)privateKeyString);
        KeySpec rsaPrivKeySpec = "RSA".equals(matcher.group(1)) ? KeyStores.parsePkcs1PrivateKey(privateKeyDerBytes) : KeyStores.parsePkcs8PrivateKey(privateKeyDerBytes);
        return KeyFactory.getInstance("RSA").generatePrivate(rsaPrivKeySpec);
    }

    static RSAPrivateKeySpec parsePkcs1PrivateKey(byte[] pkcs1DerBytes) {
        return new Pkcs1PrivateKeyReader(pkcs1DerBytes).readRsaKey();
    }

    static PKCS8EncodedKeySpec parsePkcs8PrivateKey(byte[] pkcs8DerBytes) {
        return new PKCS8EncodedKeySpec(pkcs8DerBytes);
    }

    private static List<Certificate> getCertificatesFromString(String pemFileString) throws IOException, CertificateException {
        Matcher matcher = CERT_PATTERN.matcher(pemFileString);
        CertificateFactory certFactory = CertificateFactory.getInstance("X.509");
        ArrayList<Certificate> certList = new ArrayList<Certificate>();
        while (matcher.find()) {
            try (ByteArrayInputStream stream = new ByteArrayInputStream(matcher.group().getBytes(StandardCharsets.UTF_8));){
                certList.add(KeyStores.getCertFromCache((X509Certificate)certFactory.generateCertificate(stream)));
            }
        }
        return certList;
    }

    private static X509Certificate getCertFromCache(X509Certificate certificate) {
        try {
            return (X509Certificate)certCache.get((Object)new EqualByteArray(certificate.getEncoded()), _input -> certificate);
        }
        catch (CertificateEncodingException e) {
            throw new SafeRuntimeException("Unable to get certificate bytes", (Throwable)e, new Arg[0]);
        }
    }

    private static class EqualByteArray {
        private final byte[] bytes;
        private int hash;

        EqualByteArray(byte[] bytes) {
            this.bytes = bytes;
        }

        public int hashCode() {
            if (this.hash == 0 && this.bytes.length > 0) {
                this.hash = Arrays.hashCode(this.bytes);
            }
            return this.hash;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof EqualByteArray)) {
                return false;
            }
            EqualByteArray other = (EqualByteArray)obj;
            return Arrays.equals(this.bytes, other.bytes);
        }
    }
}

