/*
 * Decompiled with CFR 0.152.
 */
package com.ovea.jetty.session.redis;

import com.ovea.jetty.session.Serializer;
import com.ovea.jetty.session.SessionManagerSkeleton;
import com.ovea.jetty.session.redis.JedisCallback;
import com.ovea.jetty.session.redis.JedisExecutor;
import com.ovea.jetty.session.redis.PooledJedisExecutor;
import com.ovea.jetty.session.serializer.XStreamSerializer;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import javax.naming.InitialContext;
import javax.servlet.http.HttpServletRequest;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.TransactionBlock;
import redis.clients.jedis.exceptions.JedisException;

public final class RedisSessionManager
extends SessionManagerSkeleton<RedisSession> {
    static final Logger LOG = Log.getLogger((String)"com.ovea.jetty.session");
    private static final String[] FIELDS = new String[]{"id", "created", "accessed", "lastNode", "expiryTime", "lastSaved", "lastAccessed", "maxIdle", "cookieSet", "attributes"};
    private final JedisExecutor jedisExecutor;
    private final Serializer serializer;
    private long saveIntervalSec = 20L;

    public RedisSessionManager(JedisPool jedisPool) {
        this(jedisPool, (Serializer)new XStreamSerializer());
    }

    public RedisSessionManager(String jndiName) {
        this(jndiName, (Serializer)new XStreamSerializer());
    }

    public RedisSessionManager(JedisPool jedisPool, Serializer serializer) {
        this.serializer = serializer;
        this.jedisExecutor = new PooledJedisExecutor(jedisPool);
    }

    public RedisSessionManager(final String jndiName, Serializer serializer) {
        this.serializer = serializer;
        this.jedisExecutor = new JedisExecutor(){
            JedisExecutor delegate;

            @Override
            public <V> V execute(JedisCallback<V> cb) {
                if (this.delegate == null) {
                    try {
                        InitialContext ic = new InitialContext();
                        JedisPool jedisPool = (JedisPool)ic.lookup(jndiName);
                        this.delegate = new PooledJedisExecutor(jedisPool);
                    }
                    catch (Exception e) {
                        throw new IllegalStateException("Unable to find instance of " + JedisExecutor.class.getName() + " in JNDI location " + jndiName + " : " + e.getMessage(), e);
                    }
                }
                return this.delegate.execute(cb);
            }
        };
    }

    public void setSaveInterval(long sec) {
        this.saveIntervalSec = sec;
    }

    @Override
    public void doStart() throws Exception {
        this.serializer.start();
        super.doStart();
    }

    @Override
    public void doStop() throws Exception {
        super.doStop();
        this.serializer.stop();
    }

    @Override
    protected RedisSession loadSession(String clusterId, RedisSession current) {
        RedisSession loaded;
        long now = System.currentTimeMillis();
        if (current == null) {
            LOG.debug("[RedisSessionManager] loadSession - No session found in cache, loading id={}", new Object[]{clusterId});
            loaded = this.loadFromStore(clusterId, current);
        } else if (current.requestStarted()) {
            LOG.debug("[RedisSessionManager] loadSession - Existing session found in cache, loading id={}", new Object[]{clusterId});
            loaded = this.loadFromStore(clusterId, current);
        } else {
            loaded = current;
        }
        if (loaded == null) {
            LOG.debug("[RedisSessionManager] loadSession - No session found in Redis for id={}", new Object[]{clusterId});
            if (current != null) {
                current.invalidate();
            }
        } else {
            if (loaded == current) {
                LOG.debug("[RedisSessionManager] loadSession - No change found in Redis for session id={}", new Object[]{clusterId});
                return loaded;
            }
            if (!loaded.lastNode.equals(this.getSessionIdManager().getWorkerName()) || current == null) {
                if (loaded.expiryTime * 1000L > now) {
                    loaded.changeLastNode(this.getSessionIdManager().getWorkerName());
                } else {
                    LOG.debug("[RedisSessionManager] loadSession - Loaded session has expired, id={}", new Object[]{clusterId});
                    loaded = null;
                }
            }
        }
        return loaded;
    }

    private RedisSession loadFromStore(final String clusterId, final RedisSession current) {
        List<String> redisData = this.jedisExecutor.execute(new JedisCallback<List<String>>(){

            @Override
            public List<String> execute(Jedis jedis) {
                String key = "jetty-session-" + clusterId;
                if (current == null) {
                    return jedis.exists(key) != false ? jedis.hmget(key, FIELDS) : null;
                }
                String val = jedis.hget(key, "lastSaved");
                if (val == null) {
                    return Collections.emptyList();
                }
                if (current.lastSaved != Long.parseLong(val)) {
                    return jedis.hmget(key, FIELDS);
                }
                return null;
            }
        });
        if (redisData == null) {
            return current;
        }
        if (redisData.isEmpty() || redisData.get(0) == null) {
            return null;
        }
        HashMap<String, String> data = new HashMap<String, String>();
        for (int i = 0; i < FIELDS.length; ++i) {
            data.put(FIELDS[i], redisData.get(i));
        }
        String attrs = (String)data.get("attributes");
        return new RedisSession(data, attrs == null ? new HashMap<String, Object>() : this.serializer.deserialize(attrs, Map.class));
    }

    @Override
    protected void storeSession(final RedisSession session) {
        if (!session.redisMap.isEmpty()) {
            Map toStore;
            Map map = toStore = session.redisMap.containsKey("attributes") ? session.redisMap : new TreeMap(session.redisMap);
            if (toStore.containsKey("attributes")) {
                toStore.put("attributes", this.serializer.serialize(session.getSessionAttributes()));
            }
            LOG.debug("[RedisSessionManager] storeSession - Storing session id={}", new Object[]{session.getClusterId()});
            this.jedisExecutor.execute(new JedisCallback<Object>(){

                @Override
                public Object execute(Jedis jedis) {
                    session.lastSaved = System.currentTimeMillis();
                    toStore.put("lastSaved", "" + session.lastSaved);
                    return jedis.multi(new TransactionBlock(){

                        public void execute() throws JedisException {
                            String key = "jetty-session-" + session.getClusterId();
                            super.hmset(key, toStore);
                            super.expireAt(key, session.expiryTime);
                        }
                    });
                }
            });
            session.redisMap.clear();
        }
    }

    protected RedisSession newSession(HttpServletRequest request) {
        return new RedisSession(request);
    }

    @Override
    protected void deleteSession(final RedisSession session) {
        LOG.debug("[RedisSessionManager] deleteSession - Deleting from Redis session id={}", new Object[]{session.getClusterId()});
        this.jedisExecutor.execute(new JedisCallback<Object>(){

            @Override
            public Object execute(Jedis jedis) {
                return jedis.del(new String[]{"jetty-session-" + session.getClusterId()});
            }
        });
    }

    final class RedisSession
    extends SessionManagerSkeleton.SessionSkeleton {
        private final Map<String, String> redisMap;
        private long expiryTime;
        private long lastSaved;
        private String lastNode;
        private final ThreadLocal<Boolean> firstAccess;

        private RedisSession(HttpServletRequest request) {
            super(RedisSessionManager.this, request);
            this.redisMap = new TreeMap<String, String>();
            this.firstAccess = new ThreadLocal<Boolean>(){

                @Override
                protected Boolean initialValue() {
                    return true;
                }
            };
            this.lastNode = RedisSessionManager.this.getSessionIdManager().getWorkerName();
            long maxidle = this.getMaxInactiveInterval();
            this.expiryTime = maxidle < 0L ? 0L : System.currentTimeMillis() / 1000L + maxidle;
            this.redisMap.put("id", this.getClusterId());
            this.redisMap.put("context", RedisSessionManager.this.getCanonicalizedContext());
            this.redisMap.put("virtualHost", RedisSessionManager.this.getVirtualHost());
            this.redisMap.put("created", "" + this.getCreationTime());
            this.redisMap.put("lastNode", this.lastNode);
            this.redisMap.put("lastAccessed", "" + this.getLastAccessedTime());
            this.redisMap.put("accessed", "" + this.getAccessed());
            this.redisMap.put("expiryTime", "" + this.expiryTime);
            this.redisMap.put("maxIdle", "" + maxidle);
            this.redisMap.put("cookieSet", "" + this.getCookieSetTime());
            this.redisMap.put("attributes", "");
        }

        RedisSession(Map<String, String> redisData, Map<String, Object> attributes) {
            super(RedisSessionManager.this, Long.parseLong(redisData.get("created")), Long.parseLong(redisData.get("accessed")), redisData.get("id"));
            this.redisMap = new TreeMap<String, String>();
            this.firstAccess = new /* invalid duplicate definition of identical inner class */;
            this.lastNode = redisData.get("lastNode");
            this.expiryTime = Long.parseLong(redisData.get("expiryTime"));
            this.lastSaved = Long.parseLong(redisData.get("lastSaved"));
            super.setMaxInactiveInterval(Integer.parseInt(redisData.get("maxIdle")));
            this.setCookieSetTime(Long.parseLong(redisData.get("cookieSet")));
            for (Map.Entry<String, Object> entry : attributes.entrySet()) {
                super.doPutOrRemove(entry.getKey(), entry.getValue());
            }
            super.access(Long.parseLong(redisData.get("lastAccessed")));
        }

        public void changeLastNode(String lastNode) {
            this.lastNode = lastNode;
            this.redisMap.put("lastNode", lastNode);
        }

        public void setAttribute(String name, Object value) {
            super.setAttribute(name, value);
            this.redisMap.put("attributes", "");
        }

        public void removeAttribute(String name) {
            super.removeAttribute(name);
            this.redisMap.put("attributes", "");
        }

        public final Map<String, Object> getSessionAttributes() {
            LinkedHashMap<String, Object> attrs = new LinkedHashMap<String, Object>();
            for (String key : super.getNames()) {
                attrs.put(key, super.doGet(key));
            }
            return attrs;
        }

        protected boolean access(long time) {
            boolean ret = super.access(time);
            this.firstAccess.remove();
            int maxidle = this.getMaxInactiveInterval();
            this.expiryTime = maxidle < 0 ? 0L : time / 1000L + (long)maxidle;
            this.redisMap.put("lastAccessed", "" + this.getLastAccessedTime());
            this.redisMap.put("accessed", "" + this.getAccessed());
            this.redisMap.put("expiryTime", "" + this.expiryTime);
            return ret;
        }

        public void setMaxInactiveInterval(int secs) {
            super.setMaxInactiveInterval(secs);
            this.redisMap.put("maxIdle", "" + secs);
        }

        protected void cookieSet() {
            super.cookieSet();
            this.redisMap.put("cookieSet", "" + this.getCookieSetTime());
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        protected void complete() {
            super.complete();
            if (!(this.redisMap.isEmpty() || this.redisMap.size() == 3 && this.redisMap.containsKey("lastAccessed") && this.redisMap.containsKey("accessed") && this.redisMap.containsKey("expiryTime") && this.getAccessed() - this.lastSaved < RedisSessionManager.this.saveIntervalSec * 1000L)) {
                try {
                    this.willPassivate();
                    RedisSessionManager.this.storeSession(this);
                    this.didActivate();
                }
                catch (Exception e) {
                    LOG.warn("[RedisSessionManager] complete - Problem persisting changed session data id=" + this.getId(), (Throwable)e);
                }
                finally {
                    this.redisMap.clear();
                }
            }
        }

        public boolean requestStarted() {
            boolean first = this.firstAccess.get();
            if (first) {
                this.firstAccess.set(false);
            }
            return first;
        }
    }
}

